% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rand_lefko_set.R
\name{rand_lefko_set}
\alias{rand_lefko_set}
\title{Generate lists of Lefkovitch matrix population models (MPMs) based on life
history archetypes}
\usage{
rand_lefko_set(
  n_models = 5,
  n_stages = 3,
  archetype = 1,
  fecundity = 1.5,
  output = "Type1",
  max_surv = 0.99,
  constraint = NULL,
  attempts = 1000
)
}
\arguments{
\item{n_models}{An integer indicating the number of MPMs to generate.}

\item{n_stages}{The number of stages for the MPMs. Default is \code{3}.}

\item{archetype}{The archetype of the MPMs. Default is \code{1}.}

\item{fecundity}{The average number of offspring produced (fecundity).
Values can be provided in 4 ways:
\itemize{
\item An numeric vector of length 1 providing a single measure of fecundity
to the top right corner of the matrix model only.
\item A numeric vector of integers of length equal to \code{n_stages} to provide
fecundity estimates for the whole top row of the matrix model. Use 0 for
cases with no fecundity.
\item A matrix of numeric values of the same dimension as \code{n_stages} to provide
fecundity estimates for the entire matrix model. Use 0 for cases with no
fecundity.
\item A list of two matrices of numeric values, both with the same dimension as
\code{n_stages}, to provide lower and upper limits of mean fecundity for the
entire matrix model.
}

In the latter case, a fecundity value will be drawn from a uniform
distribution for the defined range. If there is no fecundity in a
particular age class, use a value of 0 for both the lower and upper limit.}

\item{output}{Character string indicating the type of output.
\itemize{
\item \code{Type1}: A \code{compadreDB} Object containing MPMs split into the submatrices
(i.e. A, U, F and C).
\item \code{Type2}: A \code{compadreDB} Object containing MPMs that are not split into
submatrices (i.e. only the A matrix is included).
\item \code{Type3}: A \code{list} of MPMs arranged so that each element of the list
contains a model and associated submatrices (i.e. the nth element contains
the nth A matrix alongside the nth U and F matrices).
\item \code{Type4}: A \code{list} of MPMs arranged so that the list contains 3 lists for
the A  matrix and the U and F submatrices respectively.
\item \code{Type5}: A \code{list} of MPMs, including only the A matrix.
}}

\item{max_surv}{The maximum acceptable survival value, calculated across all
transitions from a stage. Defaults to \code{0.99}. This is only used the output
splits a matrix into the submatrices.}

\item{constraint}{An optional data frame with 4 columns named \code{fun}, \code{arg},
\code{lower} and \code{upper}. These columns specify (1) a function that outputs a
metric derived from an A matrix and (2) an argument for the function (\code{NA},
if no argument supplied) (3) the lower acceptable bound for the metric and
(4) upper acceptable bound for the metric.}

\item{attempts}{An integer indicating the number of attempts To be made when
simulating matrix model. The default is 1000. If it takes more than 1000
attempts to make a matrix that satisfies the conditions set by the other
arguments, then a warning is produced.}
}
\value{
A \code{compadreDB} object or list of MPMs that meet the specified
criteria.
}
\description{
This function generates a list of \code{n} MPMs according to the specified
criteria. Criteria include the \code{archetype}, and the acceptable constraining
criteria, which could include lambda, generation time or any other metric
derived from an A matrix. The function attempts to find matrices that fulfil
the criteria, discarding unacceptable matrices. By default, if it takes more
than 1000 attempts to find a suitable matrix model, then an error is
produced. However, the number of attempts can be altered with the \code{attempts}
parameter.
}
\examples{
set.seed(42) # set seed for repeatability

# Basic operation, without splitting matrices and with no constraints
rand_lefko_set(
  n_models = 3, n_stages = 5, fecundity = c(0, 0, 4, 8, 10),
  archetype = 4, output = "Type5"
)

# Constrain outputs to A matrices with lambda between 0.9 and 1.1
library(popbio)
constrain_df <- data.frame(
  fun = "lambda", arg = NA, lower = 0.9, upper =
    1.1
)
rand_lefko_set(
  n_models = 10, n_stages = 5, fecundity = c(0, 0, 4, 8, 10),
  archetype = 4, constraint = constrain_df, output = "Type5"
)

# As above, but using popdemo::eigs function instead of popbio::lambda
# to illustrate use of argument
library(popdemo)
constrain_df <- data.frame(
  fun = "eigs", arg = "lambda", lower = 0.9, upper = 1.1
)

rand_lefko_set(
  n_models = 10, n_stages = 5, fecundity = c(0, 0, 4, 8, 10),
  archetype = 4, constraint = constrain_df, output = "Type5"
)

# Multiple constraints
# Constrain outputs to A matrices with lambda between 0.9 and 1.1, generation
# time between 3 and 5 and damping ratio between 1 and 7.
library(popbio)
constrain_df <- data.frame(
  fun = c("lambda", "generation.time", "damping.ratio"),
  arg = c(NA, NA, NA),
  lower = c(0.9, 3.0, 1.0),
  upper = c(1.1, 5.0, 7.0)
)
rand_lefko_set(
  n_models = 10, n_stages = 5, fecundity = c(0, 0, 4, 8, 10),
  archetype = 4, constraint = constrain_df, output = "Type5"
)

}
\references{
Caswell, H. (2001). Matrix Population Models: Construction, Analysis, and
Interpretation. Sinauer.

Lefkovitch, L. P. (1965). The study of population growth in organisms grouped
by stages. Biometrics, 21(1), 1.

Takada, T., Kawai, Y., & Salguero-Gómez, R. (2018). A cautionary note on
elasticity analyses in a ternary plot using randomly generated population
matrices. Population Ecology, 60(1), 37–47.
}
\seealso{
\code{\link[=rand_lefko_mpm]{rand_lefko_mpm()}} which this function is essentially a wrapper for.

Other Lefkovitch matrices: 
\code{\link{generate_mpm_set}()},
\code{\link{rand_lefko_mpm}()},
\code{\link{random_mpm}()}
}
\author{
Owen Jones \href{mailto:jones@biology.sdu.dk}{jones@biology.sdu.dk}
}
\concept{Lefkovitch matrices}
