#' Compute 95% confidence intervals for derived estimates from the U submatrix
#' of a matrix population model
#'
#' This function computes the 95% confidence interval for measures derived from
#' the U submatrix of a matrix population model using parametric bootstrapping.
#' In this approach a sampling distribution of the U submatrix is generated by
#' taking a large number of random independent draws using the sampling
#' distribution of each underlying transition rate. The approach rests on our
#' assumption that survival-related processes are binomial (see the function
#' `add_mpm_error()` for details).
#'
#' The main inputs is the U matrix, which describes the survival-related
#' processes. The underlying assumption is that the U matrix is the average of a
#' binomial process. The confidence interval will depend largely on the sample
#' size used.
#'
#' @param mat_U A matrix that describes the growth and survival process.
#' @param sample_size either (1) a single matrix of sample sizes for each
#'   element of the U matrix, (2) a single value applied to the whole matrix
#' @param FUN A function to apply to each simulated matrix population model.
#'   This function must take, as input, a single U submatrix of a matrix
#'   population model (i.e., the U matrix). For functions that require the A
#'   matrix, use `compute_ci`.
#' @param ... Additional arguments to be passed to `FUN`.
#' @param n_sim An integer indicating the number of simulations to run. Default
#'   is 1000.
#' @param dist.out Logical. If TRUE, returns a list with both the quantiles and
#'   the simulated estimates. Default is FALSE.
#'
#' @return If dist.out is FALSE, a numeric vector of the 2.5th and 97.5th
#'   quantiles of the estimated measures. If `dist.out = TRUE`, a list with two
#'   elements: `quantiles` and `estimates`. `quantiles` is a numeric vector of
#'   the 2.5th and 97.5th quantiles of the estimated measures, and `estimates`
#'   is a numeric vector of the estimated measures.
#' @references Chapter 12 in Caswell, H. (2001). Matrix Population Models.
#'   Sinauer Associates Incorporated.
#' @family errors
#'
#' @examples
#' set.seed(42) # set seed for repeatability
#'
#' # Data for use in example
#' matU <- matrix(c(
#'   0.1, 0.0,
#'   0.2, 0.4
#' ), byrow = TRUE, nrow = 2)
#'
#'
#' # Example of use to calculate 95% CI of life expectancy
#' compute_ci_U(
#'   mat_U = matU, sample_size = 10, FUN =
#'     Rage::life_expect_mean
#' )
#'
#' # Example of use to calculate 95% CI of generation time and show the
#' # distribution of those bootstrapped estimates
#' xx <- compute_ci_U(
#'   mat_U = matU, sample_size = 100, FUN =
#'     Rage::life_expect_mean, dist.out = TRUE
#' )
#'
#' summary(xx$quantiles)
#' hist(xx$estimates)
#'
#' @importFrom stats quantile
#' @importFrom Rage life_expect_mean
#'
#' @author Owen Jones <jones@biology.sdu.dk>
#'
#' @export
#'
# Validation of inputs
# Check input matrix
compute_ci_U <- function(mat_U, sample_size, FUN, ...,
                         n_sim = 1000, dist.out = FALSE) {
  if (!is.matrix(mat_U)) {
    stop("mat_U must be a matrix.")
  }

  if (!dim(mat_U)[1] == dim(mat_U)[2]) {
    stop("mat_U must be a square matrix.")
  }

  # Sample size validation
  if (!(inherits(sample_size, "matrix") || length(sample_size) == 1)) {
    stop("sample_size needs to be a matrix or an integer with length 1")
  }

  if (inherits(sample_size, "matrix")) {
    if (nrow(sample_size) != nrow(mat_U)) {
      stop("if sample_size is a matrix, it must have same dimension as mat_U")
    }
  }

  unlisted_sample_size <- unlist(sample_size)

  if (!min(abs(c(unlisted_sample_size %% 1, unlisted_sample_size %% 1 - 1))) <
    .Machine$double.eps^0.5) {
    stop("sample_size must be integer value(s)")
  }

  if (min(unlisted_sample_size) < 0) {
    stop("sample_size must be >= 0.")
  }

  # Check FUN argument
  if (!is.function(FUN)) {
    stop("FUN must be a function.")
  }

  # replicate the simulation of MPMs

  sim_out <- replicate(n_sim, add_mpm_error(mat_U,
    mat_F = matrix(0,
      nrow = nrow(mat_U),
      ncol = ncol(mat_U)
    ),
    sample_size,
    split = FALSE,
    by_type = FALSE
  ),
  simplify = FALSE
  )

  # apply the function FUN to each matrix
  estimates <- sapply(sim_out, FUN, ...)

  # Check the estimates and use warnings if necessary
  if (any(is.infinite(estimates))) {
    warning("Some estimates are Inf. \n
            Try running with argument `dist.out = TRUE`
            and examine the estimates.")
  }

  emp_quantiles <- quantile(estimates, c(0.025, 0.975), na.rm = TRUE)

  if (dist.out == FALSE) {
    return(emp_quantiles)
  }
  if (dist.out == TRUE) {
    out <- list(
      "quantiles" = emp_quantiles, "estimates" = estimates,
      "matrices" = sim_out
    )
    return(out)
  }
}
