% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ensemble_model_spec.R
\name{ensemble_model_spec}
\alias{ensemble_model_spec}
\title{Creates a Stacked Ensemble Model from a Model Spec}
\usage{
ensemble_model_spec(
  object,
  resamples,
  model_spec,
  kfolds = 5,
  param_info = NULL,
  grid = 6,
  control = control_grid()
)
}
\arguments{
\item{object}{A Modeltime Table. Used for ensemble sub-models.}

\item{resamples}{An \code{rset} resample object.
Used to generate sub-model predictions for the meta-learner.
See \code{\link[timetk:time_series_cv]{timetk::time_series_cv()}} or \code{\link[rsample:vfold_cv]{rsample::vfold_cv()}} for making resamples.}

\item{model_spec}{A \code{model_spec} object defining the
meta-learner stacking model specification to be used.

Can be either:
\enumerate{
\item \strong{A non-tunable \code{model_spec}:} Parameters are specified and are not optimized via tuning.
\item \strong{A tunable \code{model_spec}:} Contains parameters identified for tuning with
\code{tune::tune()}
}}

\item{kfolds}{K-Fold Cross Validation for tuning the Meta-Learner.
Controls the number of folds used in the meta-learner's cross-validation.
Gets passed to \code{rsample::vfold_cv()}.}

\item{param_info}{A \code{dials::parameters()} object or \code{NULL}. If none is given, a
parameters set is derived from other arguments. Passing this argument
can be useful when parameter ranges need to be customized.}

\item{grid}{Grid specification or grid size for tuning the Meta Learner.
Gets passed to \code{tune::tune_grid()}.}

\item{control}{An object used to modify the tuning process.
Uses \code{tune::control_grid()} by default.
Use \code{control_grid(verbose = TRUE)} to follow the training process.}
}
\description{
A 3-stage stacking regressor that follows:
\enumerate{
\item Stage 1: Sub-Model's are Trained & Predicted using \code{resamples}
\item Stage 2: A Meta-learner (\code{model_spec}) is trained on Out-of-Sample Sub-Model Predictions
\item Stage 3: The Best Meta-Learner Model is Selected (if tuning is used)
}
}
\details{
\strong{Important Details:}

Results will vary considerably if poor sub-model candidates are used,
a poor sub-model resampling strategy is selected,
a poor meta-learner is selected, if the metalearner is not tuned.
\itemize{
\item Use \code{object} (a Modeltime Table) to define your sub-models
\item Use \code{resamples} to define the submodel resampling procedure.
Results will vary considerably if a poor resampling strategy is selected.
\item Use \code{model_spec} to define the meta-learner. Use \code{tune::tune()} to define
meta-learner parameters for tuning.
}

\strong{Ensemble Process}

The Meta-Learner Ensembling Process uses the following basic steps:
\enumerate{
\item \strong{Make cross-validation predictions for each sub-model.}
The user provides the sub-models as a Modeltime Table (\code{object}) and
the cross validation set as \code{resamples}
(using a function like \code{\link[timetk:time_series_cv]{timetk::time_series_cv()}} or \code{\link[rsample:vfold_cv]{rsample::vfold_cv()}}.
Each model in the Modeltime Table is trained & predicted on the \code{resamples}.
The out-of-sample sub-model predictions are used as the input to the
meta-learner.
\item \strong{Train a Stacked Regressor (Meta-Learner).}
The sub-model out-of-sample cross validation predictions are then
modeled using a \code{model_spec} with options:
\itemize{
\item \strong{Tuning:} If the \code{model_spec} does include tuning parameters via \code{tune::tune()}
then the meta-learner will be hypeparameter tuned using K-Fold Cross Validation. The
parameters and grid can adjusted using \code{kfolds}, \code{grid}, and \code{param_info}.
\item \strong{No-Tuning:} If the \code{model_spec} does \emph{not} include tuning parameters via \code{tune::tune()}
then the meta-learner will not be hypeparameter tuned and will have the model
fitted to the sub-model predictions.
}
\item \strong{Final Model Selection}
\itemize{
\item \strong{If tuned}, the final model is selected based on RMSE, then
retrained on the full set of out of sample predictions.
\item \strong{If not-tuned}, the fitted model from Stage 2 is used.
}
}

\strong{Progress}

The best way to follow the training process and watch progress is to use
\code{control = control_grid(verbose = TRUE)} to see progress.

\strong{Parallelize}

Portions of the process can be parallelized. To parallelize, set
up parallelization using \code{tune} via one of the backends such as
\code{doFuture}. Then set \code{control = control_grid(allow_par = TRUE)}
}
\examples{
library(tidymodels)
library(modeltime)
library(modeltime.ensemble)
library(tidyverse)
library(timetk)

\donttest{
resamples_tscv <- training(m750_splits) \%>\%
    time_series_cv(
        assess  = "2 years",
        initial = "5 years",
        skip    = "2 years",
        slice_limit = 1
    )

# No Metalearner Tuning ----
ensemble_fit_lm <- m750_models \%>\%
    ensemble_model_spec(
        resamples  = resamples_tscv,
        model_spec = linear_reg() \%>\% set_engine("lm"),
        control    = control_grid(verbose = TRUE)
    )

ensemble_fit_lm

# With Metalearner Tuning ----
ensemble_fit_glmnet <- m750_models \%>\%
    ensemble_model_spec(
        resamples  = resamples_tscv,
        model_spec = linear_reg(
                penalty = tune(),
                mixture = tune()
            ) \%>\%
            set_engine("glmnet"),
        grid       = 2,
        control    = control_grid(verbose = TRUE)
    )

ensemble_fit_glmnet

}

}
