% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vlmc_likelihood.R, R/vlmc_likelihood_cpp.R
\name{loglikelihood}
\alias{loglikelihood}
\alias{loglikelihood.vlmc}
\alias{loglikelihood.vlmc_cpp}
\title{Log-Likelihood of a VLMC}
\usage{
loglikelihood(
  vlmc,
  newdata,
  initial = c("truncated", "specific", "extended"),
  ignore,
  ...
)

\method{loglikelihood}{vlmc}(
  vlmc,
  newdata,
  initial = c("truncated", "specific", "extended"),
  ignore,
  ...
)

\method{loglikelihood}{vlmc_cpp}(
  vlmc,
  newdata,
  initial = c("truncated", "specific", "extended"),
  ignore,
  ...
)
}
\arguments{
\item{vlmc}{the vlmc representation.}

\item{newdata}{an optional discrete time series.}

\item{initial}{specifies the likelihood function, more precisely the way the
first few observations for which contexts cannot be calculated are integrated
in the likelihood. Defaults to \code{"truncated"}. See below for details.}

\item{ignore}{specifies the number of initial values for which the loglikelihood
will not be computed. The minimal number depends on the likelihood function as
detailed below.}

\item{...}{additional parameters for loglikelihood.}
}
\value{
an object of class \code{logLikMixVLMC} and \code{logLik}. This is a number,
the log-likelihood of the (CO)VLMC with the following attributes:
\itemize{
\item \code{df}: the number of parameters used by the VLMC for this likelihood calculation
\item \code{nobs}: the number of observations included in this likelihood calculation
\item \code{initial}: the value of the \code{initial} parameter used to compute this likelihood
}
}
\description{
This function evaluates the log-likelihood of a VLMC fitted on a discrete time series.
When the optional argument \code{newdata} is provided, the function evaluates instead the
log-likelihood for this (new) discrete time series.
}
\details{
The definition of the likelihood function depends on the value of the
\code{initial} parameters, see the section below as well as the dedicated
vignette: \code{vignette("likelihood", package = "mixvlmc")}.

For VLMC objects, the method \code{loglikelihood.vlmc} will be used. For VLMC with covariables, \code{loglikelihood.covlmc}
will instead be called. For more informations on \code{loglikelihood} methods, use \code{methods(loglikelihood)} and their associated documentation.
}
\section{likelihood calculation}{


In a (CO)VLMC of \code{\link[=depth]{depth()}}=k, we need k past values in order to compute the
context of a given observation. As a consequence, in a time series \code{x}, the
contexts of \code{x[1]} to \code{x[k]} are unknown. Depending on the value of \code{initial}
different likelihood functions are used to tackle this difficulty:
\itemize{
\item \code{initial=="truncated"}: the likelihood is computed using only
\code{x[(k+1):length(x)]}
\item \code{initial=="specific"}: the likelihood is computed on the full time series
using a specific context for the initial values, \code{x[1]} to \code{x[k]}. Each of
the specific context is unique, leading to a perfect likelihood of 1 (0 in
log scale). Thus the numerical value of the likelihood is identical as the
one obtained with \code{initial=="truncated"} but it is computed on \code{length(x)}
with a model with more parameters than in this previous case.
\item \code{initial=="extended"} (default): the likelihood is computed on the full time series
using an extended context matching for the initial values, \code{x[1]} to \code{x[k]}.
This can be seen as a compromised between the two other possibilities:
the relaxed context matching needs in general to turn internal nodes
of the context tree into actual context, increasing the number of parameters,
but not as much as with "specific". However, the likelihood of say \code{x[1]}
with an empty context is generally not 1 and thus the full likelihood is
smaller than the one computed with "specific".
}

In all cases, the \code{ignore} first values of the time series are not included
in the computed likelihood, but still used to compute contexts. If \code{ignore}
is not specified, it is set to the minimal possible value, that is k for the
\code{truncated} likelihood and 0 for the other ones. If it is specified, it must
be larger or equal to k for \code{truncated}.

See the dedicated vignette for a more mathematically oriented discussion:
\code{vignette("likelihood", package = "mixvlmc")}.
}

\examples{
## Likelihood for a fitted VLMC.
pc <- powerconsumption[powerconsumption$week == 5, ]
breaks <- c(
  0,
  median(powerconsumption$active_power, na.rm = TRUE),
  max(powerconsumption$active_power, na.rm = TRUE)
)
labels <- c(0, 1)
dts <- cut(pc$active_power, breaks = breaks, labels = labels)
m_nocovariate <- vlmc(dts)
ll <- loglikelihood(m_nocovariate)
ll
attr(ll, "nobs")
attr(ll, "df")

## Likelihood for a new time series with previously fitted VLMC.
pc_new <- powerconsumption[powerconsumption$week == 11, ]
dts_new <- cut(pc_new$active_power, breaks = breaks, labels = labels)
ll_new <- loglikelihood(m_nocovariate, newdata = dts_new)
ll_new
attributes(ll_new)
ll_new_specific <- loglikelihood(m_nocovariate, initial = "specific", newdata = dts_new)
ll_new_specific
attributes(ll_new_specific)
ll_new_extended <- loglikelihood(m_nocovariate, initial = "extended", newdata = dts_new)
ll_new_extended
attributes(ll_new_extended)

}
\seealso{
\code{\link[stats:logLik]{stats::logLik()}}
}
