# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Calculate (estimate) environmental suitability
#'
#' Calculate / estimate the environmental suitability for a given environmental value,
#' based on a beta distribution, using the three "cardinal" values of the species for that environmental niche.
#'
#' @param vmax `<numeric>` upper (i.e. maximum) tolerable value
#' @param vopt `<numeric>` optimal (i.e. preferred) value
#' @param vmin `<numeric>` lower (i.e. minimum) tolerable value
#' @param venv `<numeric>` environmental value for which to calculate the suitability
#' @return `<numeric>` environmental suitability
#' @details The environmental suitability is calculated based on a beta distribution
#' after a formula provided by Yin et al. (1995) and simplified by Yan and Hunt (1999) (see references paragraph)
#' \deqn{suitability = (\frac{V_{max} - V_{env}}{V_{max} - V_{opt}}) * (\frac{V_{env} - V_{min}}{V_{opt} - V_{min}})^{\frac{V_{opt} - V_{min}}{V_{max} - V_{opt}}}}{suitability = ((V_max - V_env) / (V_max - V_opt)) * ((V_env - V_min) / (V_opt - V_min))^((V_opt - V_min) / (V_max - V_opt)))}
#' @note The original formula by Yin et al. was only intended to calculate
#' the relative daily growth rate of plants in relation to temperature. The abstraction to
#' use this to A) calculate a niche suitability; and B) use it on other
#' environmental values than temperature might not be valid. However, the assumption that the
#' environmental suitability for one niche dimension is highest at one optimal value and
#' decreases towards the tolerable minimum and maximum values in a nonlinear fashion seems reasonable.
#' @references
#' Yin, X., Kropff, M.J., McLaren, G., Visperas, R.M., (1995)
#' A nonlinear model for crop development as a function of temperature,
#' *Agricultural and Forest Meteorology*,
#' Volume **77**, Issues 1–2,
#' Pages 1--16,
#' \doi{10.1016/0168-1923(95)02236-Q}
#'
#' Also, see equation 4 in:
#' Weikai Yan, L.A. Hunt, (1999)
#' An Equation for Modelling the Temperature Response of Plants using only the Cardinal Temperatures,
#' *Annals of Botany*,
#' Volume **84**, Issue 5,
#' Pages 607--614,
#' ISSN 0305-7364,
#' \doi{10.1006/anbo.1999.0955}
#' @examples
#' calculate_suitability(
#'     vmax = 30,
#'     vopt = 25,
#'     vmin = 10,
#'     venv = 1:40
#' )
#' calculate_suitability(
#'     vmax = seq(30, 32, length.out = 40),
#'     vopt = seq(20, 23, length.out = 40),
#'     vmin = seq(9, 11, length.out = 40),
#'     venv = 1:40
#' )
#'
#' try(calculate_suitability(
#'     vmax = 1,
#'     vopt = seq(20, 23, length.out = 40),
#'     vmin = seq(9, 11, length.out = 40),
#'     venv = 1:40
#' ))
#' @export
calculate_suitability <- function(vmax, vopt, vmin, venv) {
    .Call('_metaRange_calculate_suitability', PACKAGE = 'metaRange', vmax, vopt, vmin, venv)
}

#' Unweighted and fixed sized dispersal
#'
#' Dispersal function that uses a fixed sized kernel that isn't influenced by
#' external factors. The individuals in each cell are redistributed to the
#' surrounding cells according to probability defined in the dispersal kernel.
#' Useful for e.g. wind dispersed plants.
#'
#' @param abundance `<numeric matrix>` Values need to be non-negative.
#' @param dispersal_kernel `<numeric matrix>` Dispersal kernel. Needs to have an odd size.
#' @return `<numeric matrix>` The new abundance matrix.
#' @keywords internal
dispersal_fixed_unweighted <- function(abundance, dispersal_kernel) {
    .Call('_metaRange_dispersal_fixed_unweighted', PACKAGE = 'metaRange', abundance, dispersal_kernel)
}

#' Weighted and fixed sized dispersal
#'
#' Dispersal function that uses a fixed sized kernel and weighted dispersal towards areas that have a higher weight.
#' Use case are e.g. animals that can sense their surroundings.
#'
#' @param abundance `<numeric matrix>` Values need to be non-negative.
#' @param weights `<numeric matrix>` Values need to non-missing and between 0 and 1 for the result to make sense.
#' Needs to have same size as abundance.
#' @param dispersal_kernel `<numeric matrix>` Dispersal kernel. Needs to have an odd size.
#' @return `<numeric matrix>` The new abundance matrix.
#' @keywords internal
dispersal_fixed_weighted <- function(abundance, weights, dispersal_kernel) {
    .Call('_metaRange_dispersal_fixed_weighted', PACKAGE = 'metaRange', abundance, weights, dispersal_kernel)
}

#' Metabolic scaling
#'
#' A function to calculate the metabolic scaling of a parameter, based on the
#' metabolic theory of ecology (Brown et al. 2004).
#'
#' @param normalization_constant `<numeric>` normalization constant.
#' @param scaling_exponent `<numeric>` allometric scaling exponent of the mass.
#' @param mass `<numeric matrix>`  mean (individual) mass.
#' @param temperature `<numeric matrix>` temperature in kelvin (K).
#' @param E `<numeric>` activation energy in electronvolts (eV).
#' @param k `<numeric>` Boltzmann's constant (eV / K).
#' @details
#' ## Equation:
#' The function uses the equation in the form of:
#' \deqn{parameter = normalization\_constant \cdot mass^{scaling\_exponent} \cdot e^{\frac{Activation\_energy}{k \cdot temperature}}}{parameter = normalization_constant * mass^scaling_exponent * e^(Activation_energy/ (k * temperature))}
#'
#' ## Parameter:
#' Note the different scaling values for different parameter.
#' The following is a summary from table 4 in Brown, Sibly and Kodric-Brown (2012)
#' (see references).
#' | Parameter  | Scaling exponent | Activation energy |
#' | :------------ | :-----------: | -------------------: |
#' | resource usage | 3/4 | -0.65 |
#' | reproduction, mortality | -1/4 | -0.65 |
#' | carrying capacity | -3/4 | 0.65 |
#'
#' ## Units:
#'
#' \deqn{1 \ electronvolt = 1.602176634 \cdot 10^{-19} Joule}{1 electronvolt = 1.602176634 * 10^-19 Joule}
#'
#' \deqn{Boltzmann \ constant = 1.380649 \cdot 10^{-23} \frac{Joule}{Kelvin}}{Boltzmann constant = 1.380649 * 10^-23 Joule/Kelvin}
#'
#' \deqn{Boltzmann \ constant \ in \frac{eV}{K} = 8.617333e-05 = \frac{1.380649 \cdot 10^{-23}}{1.602176634 \cdot 10^{-19}}}{Boltzmann constant in eV/K = 8.617333e-05 = (1.380649 * 10^-23) / (1.602176634 * 10^-19)}
#' @references
#' Brown, J.H., Gillooly, J.F., Allen, A.P., Savage, V.M. and West, G.B. (2004)
#' Toward a Metabolic Theory of Ecology. *Ecology*, **85** 1771--1789.
#' \doi{10.1890/03-9000}
#'
#' Brown, J.H., Sibly, R.M. and Kodric-Brown, A. (2012)
#' Introduction: Metabolism as the Basis for a Theoretical Unification of Ecology.
#' In *Metabolic Ecology* (eds R.M. Sibly, J.H. Brown and A. Kodric-Brown)
#' \doi{10.1002/9781119968535.ch}
#' @seealso
#' `calculate_normalization_constant()`
#' @return `<numeric>` The scaled parameter.
#' @examples
#' reproduction_rate <- 0.25
#' E_reproduction_rate <- -0.65
#' estimated_normalization_constant <-
#'     calculate_normalization_constant(
#'         parameter_value = reproduction_rate,
#'         scaling_exponent = -1/4,
#'         mass = 100,
#'         reference_temperature = 273.15 + 10,
#'         E = E_reproduction_rate
#'     )
#' metabolic_scaling(
#'     normalization_constant = estimated_normalization_constant,
#'     scaling_exponent = -1/4,
#'     mass = 100,
#'     temperature = 273.15 + 20,
#'     E = E_reproduction_rate
#' )
#'
#' carrying_capacity <- 100
#' E_carrying_capacity <- 0.65
#' estimated_normalization_constant <-
#'     calculate_normalization_constant(
#'         parameter_value = carrying_capacity,
#'         scaling_exponent = -3/4,
#'         mass = 100,
#'         reference_temperature = 273.15 + 10,
#'         E = E_carrying_capacity
#'     )
#' metabolic_scaling(
#'     normalization_constant = estimated_normalization_constant,
#'     scaling_exponent = -3/4,
#'     mass = 100,
#'     temperature = 273.15 + 20,
#'     E = E_carrying_capacity
#' )
#' @export
metabolic_scaling <- function(normalization_constant, scaling_exponent, mass, temperature, E, k = 8.617333e-05) {
    .Call('_metaRange_metabolic_scaling', PACKAGE = 'metaRange', normalization_constant, scaling_exponent, mass, temperature, E, k)
}

#' Ricker reproduction model
#'
#' An implementation of the "classic" Ricker reproduction model (Ricker, 1954).
#'
#' @param abundance `<numeric>` vector (or matrix) of abundances.
#' @param reproduction_rate `<numeric>` vector (or matrix) of reproduction rates.
#' @param carrying_capacity `<numeric>` vector (or matrix) of carrying capacities.
#' @details
#' ## Equation:
#' \deqn{abundance_{t+1} = abundance_t \cdot e^{reproduction\_rate \cdot (1 - \frac{abundance_t}{carrying\_capacity})}}{abundance_t1 = abundance_t0 * e^(reproduction_rate * (1 - abundance_t0 / carrying_capacity))}
#'
#' Note that the input should have an equal size and that the input abundance
#' should be positive for the results to make sense.
#' @return `<numeric>` vector (or matrix) of abundances.
#' @examples
#' ricker_reproduction_model(
#'     abundance = 10,
#'     reproduction_rate = 0.25,
#'     carrying_capacity = 100
#' )
#' ricker_reproduction_model(
#'     abundance = matrix(10, 10, 5),
#'     reproduction_rate =  0.25,
#'     carrying_capacity =  100
#' )
#' ricker_reproduction_model(
#'     abundance = matrix(10, 10, 5),
#'     reproduction_rate =  matrix(seq(-0.5, 0.5, length.out = 25), 10, 5),
#'     carrying_capacity =  matrix(100, 10, 5)
#' )
#' @references
#' Cabral, J.S. and Schurr, F.M. (2010)
#' Estimating demographic models for the range dynamics of plant species.
#' *Global Ecology and Biogeography*, **19**, 85--97.
#' \doi{10.1111/j.1466-8238.2009.00492.x}
#'
#' Original model:
#' Ricker, W.E. (1954) Stock and recruitment.
#' *Journal of the Fisheries Research Board of Canada*, **11**, 559--623.
#' \doi{10.1139/f54-039}
#' @export
ricker_reproduction_model <- function(abundance, reproduction_rate, carrying_capacity) {
    .Call('_metaRange_ricker_reproduction_model', PACKAGE = 'metaRange', abundance, reproduction_rate, carrying_capacity)
}

