% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matsindf_apply.R
\name{matsindf_apply}
\alias{matsindf_apply}
\title{Apply a function to a \code{matsindf} data frame (and more)}
\usage{
matsindf_apply(.dat = NULL, FUN, ..., .warn_missing_FUN_args = TRUE)
}
\arguments{
\item{.dat}{A list of named items or a data frame.}

\item{FUN}{The function to be applied to \code{.dat}.}

\item{...}{Named arguments to be passed by name to \code{FUN}.}

\item{.warn_missing_FUN_args}{A boolean that tells
whether to warn of missing arguments to \code{FUN}.
Default is \code{TRUE}.}
}
\value{
A named list or a data frame. (See details.)
}
\description{
Applies \code{FUN} to \code{.dat} or
performs the calculation specified by \code{FUN}
on numbers or matrices.
\code{FUN} must return a named list.
The values of the list returned \code{FUN} become
entries in columns in a returned data frame
or entries in the sub-lists of a returned list.
The names of the items in the list returned by \code{FUN} become
names of the columns in a returned data frame or
names of the list items in the returned list.
}
\details{
If \code{is.null(.dat)} and \code{...} are all named numbers or matrices
of the form \code{argname = m},
\code{m}s are passed to \code{FUN} by \code{argname}s.
The return value is a named list provided by \code{FUN}.
The arguments in \code{...} are not included in the output.

If \code{is.null(.dat)} and \code{...} are all lists of numbers or matrices
of the form \code{argname = l},
\code{FUN} is \code{Map}ped across the various \code{l}s
to obtain a list of named lists returned from \code{FUN}.
The return value is a list
whose top-level names are the names of the returned items from \code{FUN}
\code{.dat} is not included in the return value.

If \code{!is.null(.dat)} and \code{...} are all named, \code{length == 1} character strings
of the form \code{argname = string},
\code{argname}s are expected to be names of arguments to \code{FUN}, and
\code{string}s are expected to be column names in \code{.dat}.
The return value is \code{.dat} with additional columns (at right)
whose names are the names of list items returned from \code{FUN}.
When \code{.dat} contains columns whose names are same as columns added at the right,
a warning is emitted.

\code{.dat} can be a list of named items in which case a list will be returned
instead of a data frame.

If items in \code{.dat} have same names as arguments to \code{FUN},
it is not necessary to specify any arguments in \code{...}.
\code{matsindf_apply} assumes that the appropriately-named items in \code{.dat} are
intended to be arguments to \code{FUN}.
When an item name appears in both \code{...} and \code{.dat},
\code{...} takes precedence.

if \code{.dat} is a data frame,
the items in its columns (possibly matrices)
are \code{unname()}d before calling \code{FUN}.

\code{NULL} arguments in \code{...} are ignored for the purposes of deciding whether
all arguments are numbers, matrices, lists of numbers of matrices, or named character strings.
However, all \code{NULL} arguments are passed to \code{FUN},
so \code{FUN} should be able to deal with \code{NULL} arguments appropriately.

If \code{.dat} is present, \code{...} contains \code{length == 1} strings, and one of the \code{...} strings is not the name
of a column in \code{.dat},
\code{FUN} is called WITHOUT the argument whose column is missing.
I.e., that argument is treated as missing.
If \code{FUN} works despite the missing argument, execution proceeds.
If \code{FUN} cannot handle the missing argument, an error will occur in \code{FUN}.

It is suggested that \code{FUN} is able to handle empty data gracefully,
returning an empty result with the same names as when
non-empty data are fed to \code{FUN}.
Attempts are made to handle zero-row data (in \code{.dat} or \code{...})
gracefully.
First, \code{FUN} is called with the empty (but named) data.
If \code{FUN} can handle empty data without error,
the result is returned.
If \code{FUN} errors when fed empty data, \code{FUN} is called with an empty
argument list in the hopes that \code{FUN} has reasonable default values.
If that fails,
\code{.dat} is returned unmodified (if not \code{NULL})
or the data in \code{...} is returned.
}
\examples{
library(matsbyname)
example_fun <- function(a, b){
  return(list(c = sum_byname(a, b),
              d = difference_byname(a, b)))
}
# Single values for arguments
matsindf_apply(FUN = example_fun, a = 2, b = 2)
# Matrices for arguments
a <- 2 * matrix(c(1,2,3,4), nrow = 2, ncol = 2, byrow = TRUE,
              dimnames = list(c("r1", "r2"), c("c1", "c2")))
b <- 0.5 * a
matsindf_apply(FUN = example_fun, a = a, b = b)
# Single values in lists are treated like columns of a data frame
matsindf_apply(FUN = example_fun, a = list(2, 2), b = list(1, 2))
# Matrices in lists are treated like columns of a data frame
matsindf_apply(FUN = example_fun, a = list(a, a), b = list(b, b))
# Single numbers in a data frame
DF <- data.frame(a = c(4, 4, 5), b = c(4, 4, 4))
matsindf_apply(DF, FUN = example_fun, a = "a", b = "b")
# By default, arguments to FUN come from DF
matsindf_apply(DF, FUN = example_fun)
# Now put some matrices in a data frame.
DF2 <- data.frame(a = I(list(a, a)), b = I(list(b,b)))
matsindf_apply(DF2, FUN = example_fun, a = "a", b = "b")
# All arguments to FUN are supplied by named items in .dat
matsindf_apply(list(a = 1, b = 2), FUN = example_fun)
# All arguments are supplied by named arguments in ..., but mix them up.
# Note that the named arguments override the items in .dat
matsindf_apply(list(a = 1, b = 2, z = 10), FUN = example_fun, a = "z", b = "b")
# A warning is issued when an output item has same name as an input item.
matsindf_apply(list(a = 1, b = 2, c = 10), FUN = example_fun, a = "c", b = "b")
# When a zero-row data frame supplied to .dat,
# .dat is returned unmodified, unless FUN can handle empty data.
DF3 <- DF2[0, ]
DF3
matsindf_apply(DF3, FUN = example_fun, a = "a", b = "b")
}
