% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/track_sensor.R
\name{track_sensor}
\alias{track_sensor}
\title{Reconstruct the trajectory of the LiDAR sensor using multiple returns}
\usage{
track_sensor(
  las,
  algorithm,
  extra_check = TRUE,
  thin_pulse_with_time = 0.001,
  multi_pulse = FALSE
)
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{algorithm}{function. An algorithm to compute sensor tracking. \code{lidR} implements
\link{Roussel2020} and  \link{Gatziolis2019} (see respective documentation and examples).}

\item{extra_check}{boolean. Datasets are rarely perfectly populated, leading to unexpected errors.
Time-consuming checks of data integrity are performed. These checks can be skipped as they account
for an significant proportion of the computation time. See also section 'Tests of data integrity'.}

\item{thin_pulse_with_time}{numeric. In practice, it is not useful to compute the position using all
multiple returns. It is more computationally demanding but not necessarily more accurate. This keeps
only one pulse every x seconds. Set to 0 to use all multiple returns. Use 0 if the file has already
been read with \code{filter = "-thin_pulses_with_time 0.001"}.}

\item{multi_pulse}{logical. TRUE only for systems with multiple pulses. Pulse ID must be recorded
in the UserData attribute.}
}
\value{
An sf object with POINT Z geometries. Information about the time interval and the score of
the positioning (according to the method used) are also in the table of attributes.
}
\description{
Use multiple returns to estimate the positioning of the sensor by computing the intersection in
space of the line passing through the first and last returns. To work, this function requires a
dataset where the 'gpstime', 'ReturnNumber', 'NumberOfReturns' and 'PointSourceID' attributes are
properly populated, otherwise the output may be incorrect or weird. For LAScatalog processing
it is recommended to use large chunks and large buffers (e.g. a swath width). The point cloud must
not be normalized.
}
\section{Non-supported LAScatalog options}{

The option `select` is not supported and not respected  because it is internally known what is the
best to select\cr
The option `output_files` is not supported and not respected because the output must be post-processed
as a whole
}

\section{Test of data integrity}{

In theory, sensor tracking is a simple problem to solve as long as each pulse is properly
identified from a well-populated dataset. In practice, many problems may arise from datasets that are populated
incorrectly. Here is a list of problems that may happen. Those with a * denote problems already encountered and
internally checked to remove weird points:
\itemize{
\item 'gpstime' does not record the time at which pulses were emitted and thus pulses are not identifiable
\item *A pulse (two or more points that share the same gpstime) is made of points from different
flightlines (different PointSourceID). This is impossible and denotes an improperly populated PointSourceID
attribute.
\item 'ReturnNumber' and 'NumberOfReturns' are wrongly populated with either some ReturnNumber > NumberOfReturn
 or several first returns by pulses
}
For a given time interval, when weird points are not filtered, the position is not computed for this
interval.
}

\examples{
# A valid file properly populated
LASfile <- system.file("extdata", "Topography.laz", package="lidR")
las = readLAS(LASfile)
#plot(las)

# pmin = 15 because it is an extremely small file
# strongly decimated to reduce its size. There are
# actually few multiple returns
flightlines <- track_sensor(las, Roussel2020(pmin = 15))

plot(las@header)
plot(sf::st_geometry(flightlines), add = TRUE)

#plot(las) |> add_flightlines3d(flightlines, radius = 10)

\dontrun{
# With a LAScatalog "-drop_single" and "-thin_pulses_with_time"
# are used by default
ctg = readLAScatalog("folder/")
flightlines <- track_sensor(ctg,  Roussel2020(pmin = 15))
plot(flightlines)
}
}
\author{
Jean-Francois Bourdon & Jean-Romain Roussel
}
\concept{range}
