#' Readability: Bj\"ornsson's L\"asbarhetsindex (LIX)
#' 
#' This is just a convenient wrapper function for \code{\link[koRpus:readability]{readability}}.
#'
#' This function calculates the readability index ("läsbarhetsindex") by Björnsson. In contrast to \code{\link[koRpus:readability]{readability}},
#' which by default calculates all possible indices, this function will only calculate the index value.
#'
#' This formula doesn't need syllable count.
#'
#' @param txt.file Either an object of class \code{\link[koRpus]{kRp.tagged-class}}, a character vector which must be be
#'		a valid path to a file containing the text to be analyzed, or a list of text features. If the latter, calculation
#'		is done by \code{\link[koRpus:readability.num]{readability.num}}. 
#' @param parameters A numeric vector with named magic numbers, defining the relevant parameters for the index.
#' @param ... Further valid options for the main function, see \code{\link[koRpus:readability]{readability}} for details.
#' @return An object of class \code{\link[koRpus]{kRp.readability-class}}.
#' @author m.eik michalke \email{meik.michalke@@hhu.de}
#' @references
#'		Anderson, J. (1981). Analysing the readability of english and non-english texts in the classroom with Lix. In
#'			\emph{Annual Meeting of the Australian Reading Association}, Darwin, Australia.
#'
#'		Anderson, J. (1983). Lix and Rix: Variations on a little-known readability index. \emph{Journal of Reading}, 26(6), 490--496.
#' @keywords readability
#' @export
#' @examples
#' \dontrun{
#' 	LIX(tagged.text)
#' }

LIX <- function(txt.file, parameters=c(char=6, const=100), ...){
	if(is.list(txt.file)){
		results <- readability.num(txt.features=txt.file, index="LIX", parameters=list(LIX=parameters), ...)
	} else {
		results <- readability(txt.file=txt.file, index="LIX", parameters=list(LIX=parameters), ...)
	}
	return(results)
}
