#' Biplot of a PCA object
#'
#' @description This function create a biplot from a pca object generated by
#' prcomp function.
#'
#' @param PC A pca object generated by prcomp function.
#' @param x X axis.
#' @param y Y axis.
#' @param var.line A logical input, if variable lines are plotted.
#' @param colobj A vector for giving colours to the objects.
#'
#'
#' @details This is a function to produce a pca biplot from prcomp
#' function.
#'
#' @return Function returns a pca biplot.
#'
#' @author Weksi Budiaji \cr Contact: \email{budiaji@untirta.ac.id}
#'
#' @examples
#' pcadat <- prcomp(iris[,1:4], scale. = TRUE)
#' pcabiplot(pcadat)
#'
#' @export

pcabiplot <- function (PC, x = "PC1", y = "PC2", var.line = TRUE, colobj = rep(1, nrow(PC$x))) {

  if (class(PC) == "princomp")
    stop("Use prcomp function instead!")

  if (class(PC) != "prcomp")
    stop("The object is not a principal component class, use prcomp function instead!")

  if (length(colobj) != nrow(PC$x))
    stop("The vector of colobj must have the same length with the number of objects!")
  
  varpca <- PC$sdev^2
  names(varpca) <- colnames(PC$x)
  prop.exp <- (varpca[x] + varpca[y])/sum(varpca)
  if (is.null(rownames(PC$x)) == FALSE) {
    data <- data.frame(obs = row.names(PC$x), PC$x)
    plot <- ggplot(data, aes_string(x = x, y = y)) +
      geom_text(alpha = 0.4, size = 3, aes_(label = ~obs),
                color = as.factor(colobj))
  }
  else {
    data <- data.frame(PC$x)
    plot <- ggplot(data, aes_string(x = x, y = y)) +
      geom_point(alpha = 0.4, size = 0.5, color = as.factor(colobj))
  }
  datapc <- data.frame(varnames = rownames(PC$rotation), PC$rotation)
  mult <- min((max(data[, y]) - min(data[, y])/(max(datapc[,y]) - min(datapc[, y]))),
              (max(data[, x]) - min(data[, x])/(max(datapc[, x]) - min(datapc[, x]))))
  datapc <- transform(datapc, v1 = 0.7 * mult * (get(x)), v2 = 0.7 * mult * (get(y)))
  plot <- plot + coord_equal() +
    xlab(paste(names(varpca[x]), "(Variance explained =", round(varpca[x]/sum(varpca) * 100, 1), "%)")) +
    ylab(paste(names(varpca[y]), "(Variance explained =", round(varpca[y]/sum(varpca) * 100, 1), "%)")) +
    theme(panel.grid.major = element_blank(),
          panel.border = element_blank(), legend.position = "none") +
    labs(caption = paste("Total variance =", round(prop.exp * 100, 1), "%"))
  if (var.line != TRUE) {
    plot <- plot
  }
  else {
    plot <- plot +
      geom_text(data = datapc, aes_(x = ~v1, y = ~v2, label = ~varnames), size = 3, vjust = 1, color = "red") +
      geom_segment(data = datapc, aes_(x = 0, y = 0, xend = ~v1, yend = ~v2),
                   arrow = arrow(length = unit(0.2, "cm")), alpha = 0.75, color = "red")
  }
  return(plot)
}
