#' Document a decision
#'
#' Used to programmatically document decisions - note that you have to
#' store them to a file to not lose them (i.e. if used interactively).
#'
#' @param label A human-readable label for the `decision`,
#' @param description A human-readable description.
#' @param alternatives The alternatives between which was chosen.
#' @param date The date of the decision.
#' @param id Optionally, a manually specified id (otherwise, randomly
#' generated).
#' @param justification A justification specified using [justifier::jstf()], or
#' more than one, combined with the `c` operator.
#' @param silent Whether to print messages.
#' @param ... Any additional options will be stored in the decision.
#'
#' @return Invisibly, the decision as a `justifier` object (generated by
#' [justifier::dcsn()]).
#' @export
#'
#' @examples clean_workspace(force = TRUE, silent=FALSE);
#' log_decision("First we start using `justifier`.",
#'              silent=FALSE);
#' log_decision(paste0("Then we start documenting our ",
#'                     "decisions and justifications."),
#'              silent=FALSE);
#' log_decision("Then we start learning from ourselves.",
#'              silent=FALSE);
#' workspace();
log_decision <- function(label,
                         description = "",
                         alternatives = "",
                         date = as.character(Sys.Date()),
                         id = NULL,
                         justification = "",
                         silent=justifier::opts$get('silent'),
                         ...) {

  args <-
    c(as.list(environment()), list(...));
  args <- args[setdiff(names(args), "silent")];

  newDecision <-
    do.call(dcsn,
            args);

  oldWorkspace <-
    getOption(justifier::opts$get('workspace'),
              NULL);

  if (is.null(oldWorkspace)) {
    do.call(options,
            stats::setNames(list(newDecision),
                            nm = justifier::opts$get('workspace')));
  } else {

    do.call(options,
            stats::setNames(list(c(oldWorkspace,
                                   newDecision)),
                            nm = justifier::opts$get('workspace')));
  }

  if ("singleJustifierElement" %in% class(oldWorkspace)) {
    oldWorkspaceSize <- 1;
  } else {
    oldWorkspaceSize <- length(oldWorkspace);
  }

  if (!silent) {
    message("Added new decision with identifier '",
            newDecision$id,
            "' to a justifier workspace that already contained ",
            oldWorkspaceSize, " justifications.");
  }

  return(invisible(newDecision));

}
