//
// Copyright 2019 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

/*
                                 Apache License
                           Version 2.0, January 2004
                        http://www.apache.org/licenses/

   TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

   1. Definitions.

      "License" shall mean the terms and conditions for use, reproduction,
      and distribution as defined by Sections 1 through 9 of this document.

      "Licensor" shall mean the copyright owner or entity authorized by
      the copyright owner that is granting the License.

      "Legal Entity" shall mean the union of the acting entity and all
      other entities that control, are controlled by, or are under common
      control with that entity. For the purposes of this definition,
      "control" means (i) the power, direct or indirect, to cause the
      direction or management of such entity, whether by contract or
      otherwise, or (ii) ownership of fifty percent (50%) or more of the
      outstanding shares, or (iii) beneficial ownership of such entity.

      "You" (or "Your") shall mean an individual or Legal Entity
      exercising permissions granted by this License.

      "Source" form shall mean the preferred form for making modifications,
      including but not limited to software source code, documentation
      source, and configuration files.

      "Object" form shall mean any form resulting from mechanical
      transformation or translation of a Source form, including but
      not limited to compiled object code, generated documentation,
      and conversions to other media types.

      "Work" shall mean the work of authorship, whether in Source or
      Object form, made available under the License, as indicated by a
      copyright notice that is included in or attached to the work
      (an example is provided in the Appendix below).

      "Derivative Works" shall mean any work, whether in Source or Object
      form, that is based on (or derived from) the Work and for which the
      editorial revisions, annotations, elaborations, or other modifications
      represent, as a whole, an original work of authorship. For the purposes
      of this License, Derivative Works shall not include works that remain
      separable from, or merely link (or bind by name) to the interfaces of,
      the Work and Derivative Works thereof.

      "Contribution" shall mean any work of authorship, including
      the original version of the Work and any modifications or additions
      to that Work or Derivative Works thereof, that is intentionally
      submitted to Licensor for inclusion in the Work by the copyright owner
      or by an individual or Legal Entity authorized to submit on behalf of
      the copyright owner. For the purposes of this definition, "submitted"
      means any form of electronic, verbal, or written communication sent
      to the Licensor or its representatives, including but not limited to
      communication on electronic mailing lists, source code control systems,
      and issue tracking systems that are managed by, or on behalf of, the
      Licensor for the purpose of discussing and improving the Work, but
      excluding communication that is conspicuously marked or otherwise
      designated in writing by the copyright owner as "Not a Contribution."

      "Contributor" shall mean Licensor and any individual or Legal Entity
      on behalf of whom a Contribution has been received by Licensor and
      subsequently incorporated within the Work.

   2. Grant of Copyright License. Subject to the terms and conditions of
      this License, each Contributor hereby grants to You a perpetual,
      worldwide, non-exclusive, no-charge, royalty-free, irrevocable
      copyright license to reproduce, prepare Derivative Works of,
      publicly display, publicly perform, sublicense, and distribute the
      Work and such Derivative Works in Source or Object form.

   3. Grant of Patent License. Subject to the terms and conditions of
      this License, each Contributor hereby grants to You a perpetual,
      worldwide, non-exclusive, no-charge, royalty-free, irrevocable
      (except as stated in this section) patent license to make, have made,
      use, offer to sell, sell, import, and otherwise transfer the Work,
      where such license applies only to those patent claims licensable
      by such Contributor that are necessarily infringed by their
      Contribution(s) alone or by combination of their Contribution(s)
      with the Work to which such Contribution(s) was submitted. If You
      institute patent litigation against any entity (including a
      cross-claim or counterclaim in a lawsuit) alleging that the Work
      or a Contribution incorporated within the Work constitutes direct
      or contributory patent infringement, then any patent licenses
      granted to You under this License for that Work shall terminate
      as of the date such litigation is filed.

   4. Redistribution. You may reproduce and distribute copies of the
      Work or Derivative Works thereof in any medium, with or without
      modifications, and in Source or Object form, provided that You
      meet the following conditions:

      (a) You must give any other recipients of the Work or
          Derivative Works a copy of this License; and

      (b) You must cause any modified files to carry prominent notices
          stating that You changed the files; and

      (c) You must retain, in the Source form of any Derivative Works
          that You distribute, all copyright, patent, trademark, and
          attribution notices from the Source form of the Work,
          excluding those notices that do not pertain to any part of
          the Derivative Works; and

      (d) If the Work includes a "NOTICE" text file as part of its
          distribution, then any Derivative Works that You distribute must
          include a readable copy of the attribution notices contained
          within such NOTICE file, excluding those notices that do not
          pertain to any part of the Derivative Works, in at least one
          of the following places: within a NOTICE text file distributed
          as part of the Derivative Works; within the Source form or
          documentation, if provided along with the Derivative Works; or,
          within a display generated by the Derivative Works, if and
          wherever such third-party notices normally appear. The contents
          of the NOTICE file are for informational purposes only and
          do not modify the License. You may add Your own attribution
          notices within Derivative Works that You distribute, alongside
          or as an addendum to the NOTICE text from the Work, provided
          that such additional attribution notices cannot be construed
          as modifying the License.

      You may add Your own copyright statement to Your modifications and
      may provide additional or different license terms and conditions
      for use, reproduction, or distribution of Your modifications, or
      for any such Derivative Works as a whole, provided Your use,
      reproduction, and distribution of the Work otherwise complies with
      the conditions stated in this License.

   5. Submission of Contributions. Unless You explicitly state otherwise,
      any Contribution intentionally submitted for inclusion in the Work
      by You to the Licensor shall be under the terms and conditions of
      this License, without any additional terms or conditions.
      Notwithstanding the above, nothing herein shall supersede or modify
      the terms of any separate license agreement you may have executed
      with Licensor regarding such Contributions.

   6. Trademarks. This License does not grant permission to use the trade
      names, trademarks, service marks, or product names of the Licensor,
      except as required for reasonable and customary use in describing the
      origin of the Work and reproducing the content of the NOTICE file.

   7. Disclaimer of Warranty. Unless required by applicable law or
      agreed to in writing, Licensor provides the Work (and each
      Contributor provides its Contributions) on an "AS IS" BASIS,
      WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
      implied, including, without limitation, any warranties or conditions
      of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
      PARTICULAR PURPOSE. You are solely responsible for determining the
      appropriateness of using or redistributing the Work and assume any
      risks associated with Your exercise of permissions under this License.

   8. Limitation of Liability. In no event and under no legal theory,
      whether in tort (including negligence), contract, or otherwise,
      unless required by applicable law (such as deliberate and grossly
      negligent acts) or agreed to in writing, shall any Contributor be
      liable to You for damages, including any direct, indirect, special,
      incidental, or consequential damages of any character arising as a
      result of this License or out of the use or inability to use the
      Work (including but not limited to damages for loss of goodwill,
      work stoppage, computer failure or malfunction, or any and all
      other commercial damages or losses), even if such Contributor
      has been advised of the possibility of such damages.

   9. Accepting Warranty or Additional Liability. While redistributing
      the Work or Derivative Works thereof, You may choose to offer,
      and charge a fee for, acceptance of support, warranty, indemnity,
      or other liability obligations and/or rights consistent with this
      License. However, in accepting such obligations, You may act only
      on Your own behalf and on Your sole responsibility, not on behalf
      of any other Contributor, and only if You agree to indemnify,
      defend, and hold each Contributor harmless for any liability
      incurred by, or claims asserted against, such Contributor by reason
      of your accepting any such warranty or additional liability.

   END OF TERMS AND CONDITIONS

   APPENDIX: How to apply the Apache License to your work.

      To apply the Apache License to your work, attach the following
      boilerplate notice, with the fields enclosed by brackets "[]"
      replaced with your own identifying information. (Don't include
      the brackets!)  The text should be enclosed in the appropriate
      comment syntax for the file format. We also recommend that a
      file or class name and description of purpose be included on the
      same "printed page" as the copyright notice for easier
      identification within third-party archives.

   Copyright [yyyy] [name of copyright owner]

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

// Most of the contents in this file come from (unless otherwise indicated):
//   https://github.com/google/differential-privacy/blob/main/cc/algorithms/distributions.h
//   https://github.com/google/differential-privacy/blob/main/cc/algorithms/distributions.cc

#ifndef GAUSSIAN_DISTRIBUTION_HPP
#define GAUSSIAN_DISTRIBUTION_HPP

#include <inttypes.h> // uint32_t, uint_fast64_t
#include <stddef.h> // size_t, ptrdiff_t
#include <random>
#include <string.h> // memcpy
#include <float.h> // DBL_MANT_DIG

#include "logging.h" // EBM_ASSERT
#include "zones.h"

namespace DEFINED_ZONE_NAME {
#ifndef DEFINED_ZONE_NAME
#error DEFINED_ZONE_NAME must be defined
#endif // DEFINED_ZONE_NAME

// The square root of the maximum number n of Bernoulli trials from which a
// binomial sample is drawn. Larger values result in more fine grained noise,
// but increase the chance of sampling inaccuracies due to overflows. The
// probability of such an event will be roughly 2^-45 or less, if the square
// root is set to 2^57.
static constexpr double kBinomialBound = static_cast<double>(1LL << 57);

// We usually expect DBL_MANT_DIG to be 53.
static_assert(DBL_MANT_DIG < 64, "Double mantissa must have less than 64 bits.");
static_assert(sizeof(double) == sizeof(uint64_t) && std::numeric_limits<double>::is_iec559 &&
   std::numeric_limits<double>::radix == 2, "double representation is not IEEE 754 binary64.");
static constexpr int kMantDigits = DBL_MANT_DIG - 1;
static constexpr uint64_t kMantissaMask = (uint64_t { 1 } << kMantDigits) - 1ULL;

static constexpr double kPi = 3.14159265358979323846;

inline static double GetNextPowerOfTwo(double n) { 
   return std::pow(2.0, ceil(log2(n))); 
}

template<typename TRng>
inline static bool CoinFlip(TRng & rng) {
   // Google's code uses a Bernoulli with probability 0.5, which is the same as a coin flip
   return static_cast<bool>(rng.Next(uint64_t { 1 }));
}

inline static double ApproximateBinomialProbability(double sqrt_n, int64_t m) {
   // Approximates the probability of a random sample m + n / 2 drawn from a
   // binomial distribution of n Bernoulli trials that have a success probability
   // of 1 / 2 each. The approximation is taken from Lemma 7 of the noise
   // generation documentation available in
   // https://github.com/google/differential-privacy/blob/main/common_docs/Secure_Noise_Generation.pdf

   const double n = sqrt_n * sqrt_n;
   if(std::abs(m) > sqrt_n * std::sqrt(std::log(n)) / 2) {
      return 0;
   }
   return std::sqrt(2 / kPi) / sqrt_n * std::exp(-2.0 * m * m / n) * (1 - (0.4 * std::pow(std::log(n), 1.5) / sqrt_n));
}

inline static int CountLeadingZeroes64(uint64_t x) {
   // FROM: https://github.com/abseil/abseil-cpp/blob/628a2825f8dc0219964886e7cc3f7f519e3bd950/absl/numeric/internal/bits.h

   int zeroes = 60;
   if(x >> 32) {
      zeroes -= 32;
      x >>= 32;
   }
   if(x >> 16) {
      zeroes -= 16;
      x >>= 16;
   }
   if(x >> 8) {
      zeroes -= 8;
      x >>= 8;
   }
   if(x >> 4) {
      zeroes -= 4;
      x >>= 4;
   }
   return "\4\3\2\2\1\1\1\1\0\0\0\0\0\0\0"[x] + zeroes;
}

template<typename TRng>
inline static uint64_t Geometric(TRng & rng) {
   // FROM: https://github.com/google/differential-privacy/blob/327972c1ae710e8cd0a4754fffdd78c3500272ee/cc/algorithms/rand.cc#L73

   uint64_t result = 1;
   uint64_t r = 0;
   while(r == 0 && result < 1023) {
      r = rng.Next(std::numeric_limits<uint64_t>::max());
      result += CountLeadingZeroes64(r);
   }
   return result;
}

template<typename TRng>
inline static double UniformDouble(TRng & rng) {
   // FROM: https://github.com/google/differential-privacy/blob/327972c1ae710e8cd0a4754fffdd78c3500272ee/cc/algorithms/rand.cc

   uint64_t uint_64_number = rng.Next(std::numeric_limits<uint64_t>::max());
   // A random integer of Uniform[0, 2^kMantDigits).
   uint64_t i = uint_64_number & kMantissaMask;

   // Instead of throwing the leading 12 bits away, we use them to create
   // geometric random number.
   uint64_t j = uint_64_number >> kMantDigits;

   // exponent is the number of leading zeros in the first 11 bits plus one.
   uint64_t exponent = static_cast<uint64_t>(CountLeadingZeroes64(j)) - static_cast<uint64_t>(kMantDigits) + 1;

   // Extra geometric sampling is needed only when the leading 11 bits are all 0.
   if(j == 0) {
      exponent += Geometric(rng) - 1;
   }

   j = (uint64_t { 1023 } - exponent) << kMantDigits;
   if(exponent >= 1023) {
      // Denormalized value. Extremely improbable.
      j = 0;
   }
   // Addition instead of bitwise or since the carry overflow increments the
   // floating point exponent, which is exactly what we want.
   i += j;
   double r;
   memcpy(&r, &i, sizeof(r));
   return r == 0 ? 1.0 : r;
}


class GaussianDistribution final {
   double stddev_;

public:

   template<typename TRng>
   inline int SampleGeometric(TRng & rng) {
      int geom_sample = 0;
      while(CoinFlip(rng)) {
         ++geom_sample;
      }
      return geom_sample;
   }

   inline double GetGranularity(double scale) const {
      double sigma = scale * stddev_;
      return GetNextPowerOfTwo(2 * sigma / kBinomialBound);
   }

   template<typename TRng>
   inline double Sample(TRng & rng, double scale) {
      EBM_ASSERT(0 < scale);

      double sigma = scale * stddev_;
      // Use at least the lowest positive floating point number as granularity when
      // sigma is very small.
      double granularity = std::max(GetGranularity(scale), std::numeric_limits<double>::min());

      // The square root of n is chosen in a way that ensures that the respective
      // binomial distribution approximates a Gaussian distribution close enough.
      // The sqrt(n) is taken instead of n, to ensure that all results of arithmetic
      // operations fit in 64 bit integer range.
      double sqrt_n = 2.0 * sigma / granularity;
      return SampleBinomial(rng, sqrt_n) * granularity;
   }

   template<typename TRng>
   inline double SampleBinomial(TRng & rng, double sqrt_n) {
      // Returns a random sample m where {@code m + n / 2} is drawn from a binomial
      // distribution of n Bernoulli trials that have a success probability of 1 / 2
      // each. The sampling technique is based on Bringmann et al.'s rejection
      // sampling approach proposed in "Internal DLA: Efficient Simulation of a
      // Physical Growth Model", available
      // https://people.mpi-inf.mpg.de/~kbringma/paper/2014ICALP.pdf. The square root
      // of n must be at least 10^6. This is to ensure an accurate approximation of a
      // Gaussian distribution.

      int64_t step_size = static_cast<int64_t>(std::round(std::sqrt(2.0) * sqrt_n + 1));

      while(true) {
         int geom_sample = SampleGeometric(rng);
         int two_sided_geom = CoinFlip(rng) ? geom_sample : (-geom_sample - 1);
         int64_t uniform_sample = static_cast<int64_t>(rng.Next(static_cast<uint64_t>(step_size)));
         int64_t result = step_size * two_sided_geom + uniform_sample;

         double result_prob = ApproximateBinomialProbability(sqrt_n, result);
         double reject_prob = UniformDouble(rng);

         if(result_prob > 0 && reject_prob > 0 &&
            reject_prob < result_prob * step_size * std::pow(2.0, geom_sample - 2)) {
            return static_cast<double>(result);
         }
      }
   }

   inline GaussianDistribution(double stddev) : stddev_(stddev) {
      EBM_ASSERT(0 <= stddev);
   }
};

} // DEFINED_ZONE_NAME

#endif // GAUSSIAN_DISTRIBUTION_HPP
