##' Invert the Information Matrix
##' 
##' Using the eigenvalue decomposition method to invert the information matrix.
##' 
##' 
##' @param C a matrix of block projector for a single stratum.
##' @param N a matrix representation the smallest unit of block or treatment
##' effects generated by \code{\link{makeOverDesMat}}.
##' @param T a list of contrast matrices from \code{\link{makeContrMat}}.
##' @return This function returns a matrix.
##' @author Kevin Chang
##' @references Nelder JA (1965b). "The Analysis of Randomized Experiments with
##' Orthogonal Block Structure. II. Treatment Structure and the General
##' Analysis of Variance." \emph{Proceedings of the Royal Society of London.
##' Series A, Mathematical and Physical Sciences}, 283(1393), 163-178.
##' @examples
##' 
##' 
##' m <- matrix(rnorm(10), 10, 10)
##' 
##' invInfMat(m, identityMat(10), identityMat(10))
##'      
##' 
##' 
##' @export invInfMat
invInfMat <- function(C, N, T) {
    
    ei <- eigen(t(T) %*% t(N) %*% C %*% N %*% T)
    nn <- length(ei$values)
    L <- matrix(0, nrow = nn, ncol = nn)
    for (i in 1:(nn)) {
        if (Re(ei$values[i]) < 1e-07) 
            next
        
        L <- L + (1/Re(ei$values[i])) * Re(ei$vectors[, i]) %*% t(Re(ei$vectors[, i]))
    }
    return(L)
} 
