% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/role_analysis.R
\name{role_analysis}
\alias{role_analysis}
\title{Positional (Role) Analysis in Networks (\code{role_analysis})}
\usage{
role_analysis(
  graph,
  nodes,
  directed = NA,
  method = "cluster",
  min_partitions = NA,
  max_partitions = NA,
  min_partition_size = NA,
  backbone = 0.9,
  viz = FALSE,
  fast_triad = NULL,
  retain_variables = FALSE,
  cluster_summaries = FALSE,
  dendro_names = FALSE,
  self_ties = FALSE,
  cutoff = 0.999,
  max_iter = 50
)
}
\arguments{
\item{graph}{An igraph object or a list of igraph objects produced as output from \code{\link{netwrite}}.}

\item{nodes}{A data frame containing individual-level network measures for each node in the network. Ideally, the \code{node_measures} data frame produced by \code{\link{netwrite}} should be assigned to this argument.}

\item{directed}{A logical value indicating whether network edges should be treated as directed.}

\item{method}{A character value indicating the method used for positional analysis. Valid arguments are currently \code{"cluster"} for hierarchical clustering and \code{"concor"} for CONCOR.}

\item{min_partitions}{A numeric value indicating the number of minimum number of clusters or partitions to assign to nodes in the network. When using hierarchical clustering, this value reflects the minimum number of clusters produced by analysis. When using CONCOR, this value reflects the minimum number of partitions produced in analysis, such that a value of 1 results in a partitioning of two groups, a value of 2 results in four groups, and so on.}

\item{max_partitions}{A numeric value indicating the number of maximum number of clusters or partitions to assign to nodes in the network. When using hierarchical clustering, this value reflects the maximum number of clusters produced by analysis. When using CONCOR, this value reflects the maximum number of partitions produced in analysis, such that a value of 1 results in a partitioning of two groups, a value of 2 results in four groups, and so on.}

\item{min_partition_size}{A numeric value indicating the minimum number of nodes required for inclusion in a cluster. If an inferred cluster or partition contains fewer nodes than the number assigned to \code{min_partition_size}, nodes in this cluster/partition will be labeled as members of a parent cluster/partition.}

\item{backbone}{A numeric value ranging from 0-1 indicating which edges in the similarity/correlation matrix should be kept when calculating modularity of cluster/partition assignments. When calculating optimal modularity, it helps to backbone the similarity/correlation matrix according to the nth percentile. Larger networks benefit from higher \code{backbone} values, while lower values generally benefit smaller networks.}

\item{viz}{A logical value indicating whether to produce summary visualizations of the positional analysis.}

\item{fast_triad}{(Hierarchical clustering method only.) A logical value indicating whether to use a faster method for counting individual nodes' positions in different types of triads. This faster method may lead to memory issues and should be avoided when working with larger networks.}

\item{retain_variables}{(Hierarchical clustering method only.) A logical value indicating whether output should include a data frame of all node-level measures used in hierarchical clustering.}

\item{cluster_summaries}{(Hierarchical clustering method only.) A logical value indicating whether output should includde a data frame containing by-cluster mean values of variables used in hierarchical clustering.}

\item{dendro_names}{(Hierarchical clustering method only.) A logical value indicating whether the cluster dendrogram visualization should display node labels rather than ID numbers.}

\item{self_ties}{(CONCOR only.) A logical value indicting whether to include self-loops (ties directed toward oneself) in CONCOR calculation.}

\item{cutoff}{(CONCOR only.) A numeric value ranging from 0 to 1 that indicates the correlation cutoff for detecting convergence in CONCOR calculation.}

\item{max_iter}{(CONCOR only.) A numeric value indicating the maximum number of iteractions allowed for CONCOR calculattion.}
}
\value{
The \code{role_analysis} returns a list of outputs that users can access to help interpret results. This contents of this list varies somewhat depending on the method being used for positional analysis.

When hierarchical clustering is used, the list contains the following:
\code{cluster_assignments} is a data frame indicating each node's membership within inferred clusters at each level of partitioning.
\code{cluster_sociogram} contains a visualization of the network wherein nodes are colored by their membership within clusters at the optimal level of partitioning.
\code{cluster_dendrogram} is a visualization of the dendrogram produced from clustering nodes. Red boxes on the visualization indicate nodes' cluster memberships at the optimal level of partitioning.
\code{cluster_modularity} is a visualization of the modularity scores of the matrix of similarity scores between nodes for each level of partitioning. This visualization helps identify the optimal level of partitioning inferred by the \code{role_analysis} function.
\code{cluster_summaries_cent} contains one or more visualization representing how clusters inferred at the optimal level of partitioning differ from one another on several important node-level measures.
\code{cluster_summaries_triad} contains one or more visualization representing how clusters inferred at the optimal level of partitioning differ from one another on in terms of their positions within certain kinds of triads in the network.
\code{cluster_relations_heatmaps} is a list object containing several heatmap visualizations representing the extent to which nodes in one inferred cluster are connected to nodes in another cluster.
\code{cluster_relations_sociogram} contains a network visualization representing the extent to which nodes in clusters inferred at the optimal level of partitioning are tied to one another. Nodes in this visualization represent inferred clusters in the aggregate.

When CONCOR is used, this list contains the following:
\code{concor_assignments} is a data frame indicating each node's membership within inferred blocks at each level of partitioning.
\code{concor_sociogram} contains a visualization of the network wherein nodes are colored by their membership within blocks at the optimal level of partitioning.
\code{concor_block_tree} is a visualization representing how smaller blocks are derived from larger blocks at each level of partitioning using CONCOR.
\code{concor_modularity} is a visualization of the modularity scores of the matrix of similarity scores between nodes for each level of partitioning. This visualization helps identify the optimal level of partitioning inferred by the \code{role_analysis} function.
\code{concor_relations_heatmaps} is a list object containing several heatmap visualizations representing the extent to which nodes in one inferred block are connected to nodes in another block.
\code{concor_relations_sociogram} contains a network visualization representing the extent to which nodes in blocks inferred at the optimal level of partitioning are tied to one another. Nodes in this visualization represent inferred blocks in the aggregate.
}
\description{
The \code{role_analysis} function takes networks processed by the \code{\link{netwrite}} function and performs positional analysis on them. Positional analysis methods allows users to infer distinct "roles" in networks from patterns in network activity. \code{role_analysis} currently supports the identification of roles using two methods: hierarchical clustering (cite) and convergence of correlations (CONCOR, Breiger 1975).
}
\examples{
flor <- netwrite(nodelist = florentine_nodes,
                 node_id = "id",
                 i_elements = florentine_edges$source,
                 j_elements = florentine_edges$target,
                 type = florentine_edges$type,
                 directed = FALSE,
                 net_name = "florentine")

# Clustering method
flor_cluster <- role_analysis(graph = flor$igraph_list,
                              nodes = flor$node_measures,
                              directed = FALSE,
                              method = "cluster",
                              min_partitions = 2,
                              max_partitions = 8,
                              viz = TRUE)

### View cluster dendrogram
flor_cluster$cluster_dendrogram

### View modularity summary plot
flor_cluster$cluster_modularity

### View cluster assignments
head(flor_cluster$cluster_assignments)

### View centrality summary plot for aggregate network
flor_cluster$cluster_summaries_cent$summary_graph
### View cenrality summary plot for network of relation `business`
flor_cluster$cluster_summaries_cent$business

### View triad position summary plot for network of relation `marriage`
flor_cluster$cluster_summaries_triad$marriage


# CONCOR method
flor_concor <- role_analysis(graph = flor$igraph_list,
                             nodes = flor$node_measures,
                             directed = FALSE,
                             method = "concor",
                             min_partitions = 1,
                             max_partitions = 4,
                             viz = TRUE)

### View CONCOR tree
flor_concor$concor_block_tree

### View modularity summary plot
flor_concor$concor_modularity

### View cluster assignments
head(flor_concor$concor_assignments)

### View chi-squared heatmaps of relations between blocks
flor_concor$concor_relations_heatmaps$chisq
}
