% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/psm.R
\name{Psm}
\alias{Psm}
\title{N-state partitioned survival model}
\format{
An \link[R6:R6Class]{R6::R6Class} object.
}
\description{
Simulate outcomes from an N-state partitioned survival model.
}
\examples{
library("flexsurv")
library("ggplot2")
theme_set(theme_bw())

# Simulation data
strategies <- data.frame(strategy_id = c(1, 2, 3), 
                         strategy_name = paste0("Strategy ", 1:3))
patients <- data.frame(patient_id = seq(1, 3),
                       age = c(45, 50, 60),
                       female = c(0, 0, 1))
states <- data.frame(state_id =  seq(1, 3),
                     state_name = paste0("State ", seq(1, 3)))
hesim_dat <- hesim_data(strategies = strategies,
                        patients = patients,
                        states = states)
labs <- c(
  get_labels(hesim_dat),
  list(curve = c("Endpoint 1" = 1,
                "Endpoint 2" = 2,
                 "Endpoint 3" = 3))
)
n_samples <- 2

# Survival models
surv_est_data <- psm4_exdata$survival
fit1 <- flexsurv::flexsurvreg(Surv(endpoint1_time, endpoint1_status) ~ factor(strategy_id),
                              data = surv_est_data, dist = "exp")
fit2 <- flexsurv::flexsurvreg(Surv(endpoint2_time, endpoint2_status) ~ factor(strategy_id),
                              data = surv_est_data, dist = "exp")
fit3 <- flexsurv::flexsurvreg(Surv(endpoint3_time, endpoint3_status) ~ factor(strategy_id),
                              data = surv_est_data, dist = "exp")
fits <- flexsurvreg_list(fit1, fit2, fit3)

surv_input_data <- expand(hesim_dat, by = c("strategies", "patients"))
psm_curves <- create_PsmCurves(fits, input_data = surv_input_data,
                               uncertainty = "bootstrap", est_data = surv_est_data, 
                               n = n_samples)

# Cost model(s)
cost_input_data <- expand(hesim_dat, by = c("strategies", "patients", "states"))
fit_costs_medical <- stats::lm(costs ~ female + state_name, 
                               data = psm4_exdata$costs$medical)
psm_costs_medical <- create_StateVals(fit_costs_medical, 
                                      input_data = cost_input_data, 
                                      n = n_samples)

# Utility model
utility_tbl <- stateval_tbl(tbl = data.frame(state_id = states$state_id,
                                             min = psm4_exdata$utility$lower,
                                             max = psm4_exdata$utility$upper),
                            dist = "unif")
psm_utility <- create_StateVals(utility_tbl, n = n_samples,
                                hesim_data = hesim_dat)

# Partitioned survival decision model
psm <- Psm$new(survival_models = psm_curves,
               utility_model = psm_utility,
               cost_models = list(medical = psm_costs_medical))
psm$sim_survival(t = seq(0, 5, 1/12))
autoplot(psm$survival_, labels = labs, ci = FALSE, ci_style = "ribbon")
psm$sim_stateprobs()
autoplot(psm$stateprobs_, labels = labs)
psm$sim_costs(dr = .03)
head(psm$costs_)
head(psm$sim_qalys(dr = .03)$qalys_)

}
\seealso{
\code{\link{PsmCurves}}, \code{\link[=create_PsmCurves]{create_PsmCurves()}}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{survival_models}}{The survival models used to predict survival curves. Must be
an object of class \code{\link{PsmCurves}}.}

\item{\code{utility_model}}{The model for health state utility. Must be an object of
class \code{\link{StateVals}}.}

\item{\code{cost_models}}{The models used to predict costs by health state.
Must be a list of objects of class \code{\link{StateVals}}, where each element of the
list represents a different cost category.}

\item{\code{n_states}}{Number of states in the partitioned survival model.}

\item{\code{t_}}{A numeric vector of times at which survival curves were predicted. Determined
by the argument \code{t} in \verb{$sim_curves()}.}

\item{\code{survival_}}{An object of class \link{survival} simulated using \code{sim_survival()}.}

\item{\code{stateprobs_}}{An object of class \link{stateprobs} simulated using \verb{$sim_stateprobs()}.}

\item{\code{qalys_}}{An object of class \link{qalys} simulated using \verb{$sim_qalys()}.}

\item{\code{costs_}}{An object of class \link{costs} simulated using \verb{$sim_costs()}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{Psm$new()}}
\item \href{#method-sim_survival}{\code{Psm$sim_survival()}}
\item \href{#method-sim_stateprobs}{\code{Psm$sim_stateprobs()}}
\item \href{#method-sim_qalys}{\code{Psm$sim_qalys()}}
\item \href{#method-sim_costs}{\code{Psm$sim_costs()}}
\item \href{#method-summarize}{\code{Psm$summarize()}}
\item \href{#method-clone}{\code{Psm$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{Psm} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Psm$new(survival_models, utility_model = NULL, cost_models = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{survival_models}}{The \code{survival_models} field.}

\item{\code{utility_model}}{The \code{utility_model} field.}

\item{\code{cost_models}}{The \code{cost_models} field.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\code{n_states} is set equal to the number of survival models plus one.
}

\subsection{Returns}{
A new \code{Psm} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sim_survival"></a>}}
\if{latex}{\out{\hypertarget{method-sim_survival}{}}}
\subsection{Method \code{sim_survival()}}{
Simulate survival curves as a function of time using \code{PsmCurves$survival()}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Psm$sim_survival(t)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{t}}{A numeric vector of times. The first element must be \code{0}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output from \code{PsmCurves$survival()}
stored in \code{survival_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sim_stateprobs"></a>}}
\if{latex}{\out{\hypertarget{method-sim_stateprobs}{}}}
\subsection{Method \code{sim_stateprobs()}}{
Simulate health state probabilities from \code{survival_} using a partitioned
survival analysis.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Psm$sim_stateprobs()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{stateprobs}
stored in \code{stateprobs_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sim_qalys"></a>}}
\if{latex}{\out{\hypertarget{method-sim_qalys}{}}}
\subsection{Method \code{sim_qalys()}}{
Simulate quality-adjusted life-years (QALYs) as a function of \code{stateprobs_} and
\code{utility_model}. See \code{vignette("expected-values")} for details.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Psm$sim_qalys(
  dr = 0.03,
  integrate_method = c("trapz", "riemann_left", "riemann_right"),
  lys = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dr}}{Discount rate.}

\item{\code{integrate_method}}{Method used to integrate state values when computing (QALYs).}

\item{\code{lys}}{If \code{TRUE}, then life-years are simulated in addition to QALYs.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{qalys} stored
in \code{qalys_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sim_costs"></a>}}
\if{latex}{\out{\hypertarget{method-sim_costs}{}}}
\subsection{Method \code{sim_costs()}}{
Simulate costs as a function of \code{stateprobs_} and \code{cost_models}.
See \code{vignette("expected-values")} for details.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Psm$sim_costs(
  dr = 0.03,
  integrate_method = c("trapz", "riemann_left", "riemann_right")
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dr}}{Discount rate.}

\item{\code{integrate_method}}{Method used to integrate state values when computing costs.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{costs} stored
in \code{costs_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-summarize"></a>}}
\if{latex}{\out{\hypertarget{method-summarize}{}}}
\subsection{Method \code{summarize()}}{
Summarize costs and QALYs so that cost-effectiveness analysis can be performed.
See \code{\link[=summarize_ce]{summarize_ce()}}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Psm$summarize(by_grp = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{by_grp}}{If \code{TRUE}, then costs and QALYs are computed by subgroup. If
\code{FALSE}, then costs and QALYs are aggregated across all patients (and subgroups).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Psm$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
