% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hbam.R
\name{hbam}
\alias{hbam}
\title{Fit an HBAM model}
\usage{
hbam(
  self = NULL,
  stimuli = NULL,
  model = "HBAM",
  allow_miss = 2,
  req_valid = NA,
  req_unique = 2,
  prefs = NULL,
  prep_data = TRUE,
  data = NULL,
  chains = 4,
  cores = parallel::detectCores(logical = FALSE),
  warmup = 1000,
  iter = 4000,
  thin = 3,
  control = list(adapt_delta = 0.6),
  seed = sample.int(.Machine$integer.max, 1),
  ...
)
}
\arguments{
\item{self}{A numerical vector of N ideological self-placements. Any missing data must be coded as NA. This argument will not be used if the data have been prepared in advance via the \code{prep_data} function.}

\item{stimuli}{An N × J matrix of numerical stimulus placements, where J is the number of stimuli. Any missing data must be coded as NA. This argument will not be used if the data have been prepared in advance via the \code{prep_data} function.}

\item{model}{Character: Name of the model to be used. One of: \code{"HBAM"}, \code{"HBAM_2"}, \code{"HBAM_NE"}, \code{"HBAM_HM"}, \code{"HBAM_MINI"}, \code{"HBAM_0"}, \code{"HBAM_R"}, \code{"HBAM_R_MINI"}, \code{"FBAM_MINI"}, or \code{"BAM"}. Defaults to \code{"HBAM"}.}

\item{allow_miss}{Integer specifying how many missing stimulus positions to be accepted for an individual still to be included in the analysis. This argument will not be used if the data have been prepared in advance via the \code{prep_data} function. Defaults to 2.}

\item{req_valid}{Integer specifying how many valid observations to require for a respondent to be included in the analysis. The default is \code{req_valid = J - allow_miss}, but if specified, \code{req_valid} takes precedence. This argument will not be used if the data have been prepared in advance via the \code{prep_data} function.}

\item{req_unique}{Integer specifying how may unique positions on the ideological scale each respondent is required to have used when placing the stimuli in order to be included in the analysis. The default is \code{req_unique = 2}. This argument will not be used if the data have been prepared in advance via the \code{prep_data} function.}

\item{prefs}{An N × J matrix of numerical stimulus ratings or preference scores. These data are only required by the \code{"HBAM_R"} and \code{"HBAM_R_MINI"} models and will be ignored when fitting other models.}

\item{prep_data}{Logical: Should the data be prepared before fitting the model? (Or have the data been prepared in advance via the \code{prep_data} function? If so, set \code{prep_data = FALSE}.)}

\item{data}{List of data that have been prepared in advance via the \code{prep_data} function. Only applicable when \code{prep_data = TRUE}.}

\item{chains}{A positive integer specifying the number of Markov chains. Defaults to 4.}

\item{cores}{The number of cores to use when executing the Markov chains in parallel. By default, all detected physical cores will be used if \code{chains} is equal to or higher than the number of cores.}

\item{warmup}{A positive integer specifying the number of warmup (aka burn-in) iterations per chain. If step-size adaptation is on (which it is by default), this also controls the number of iterations for which adaptation is run (and hence these warmup samples should not be used for inference). The number of warmup iterations should be smaller than \code{iter}.}

\item{iter}{A positive integer specifying the number of iterations for each chain (including warmup).}

\item{thin}{A positive integer specifying the period for saving samples.}

\item{control}{A named list of parameters to control the sampler's behavior. See the details in the documentation for the control argument in the \code{stan} function in the \code{rstan} package.}

\item{seed}{A positive integer specifying an optional seed for reproducibility. If this argument is not supplied, a random seed will be generated and the function will produce slightly different results on each run.}

\item{...}{Arguments passed to \code{rstan::sampling}.}
}
\value{
An object of S4 class \code{stanfit}.
}
\description{
Fit a Hierarchical Bayesian Aldrich-McKelvey model using automatically tuned Hamiltonian Monte Carlo (NUTS) sampling via \code{rstan}.
}
\details{
This package provides several alternative models, which can be specified using the names below. Users who are unsure which model to use are adviced to use the default HBAM model. If speed or sampling diagnostics are an issue, HBAM_MINI may provide a useful alternative.

\strong{HBAM} is the default model, which allows for scale flipping and employs hierarchical priors on all individual level parameters. It also models heteroskedastic errors that vary by both individual and stimuli. This model was introduced by Bølstad (2023).

\strong{HBAM_2} uses different hyperpriors for the shifting parameters of respondents with different self-placements. In particular, the model estimates a separate mean hyperparameter for each self-placement. This model avoids shrinking the shifting parameters toward a common population mean, and may therefore fit better than HBAM if there are clear differences in average shifting across the scale of self-placements. However, this model also tends to run slower than the standard model.

\strong{HBAM_NE} models the self-placements as if they contain no error. The latent respondent positions are not treated as parameters, but rather calculated as function of the self-placements and other individual level parameters. The respondents positions are not given a prior, which means the model relies on the likelihood function and the prior on beta to yield meaningful results for these positions. By assuming no error in the self-placements, the model may underestimate the uncertainty in estimated respondents positions, while otherwise yielding very similar results to the standard HBAM model. In contrast to the standard model, the estimated respondent positions from this model will not exhibit any shrinkage, which for some purposes may be desirable, as the results may better represent the true distances between respondents and stimuli. This model also runs somewhat faster than the standard HBAM model.

\strong{HBAM_HM} assumes the prediction errors in the stimuli placements to be homoskedastic. This simplified model should normally produce very similar results to the HBAM model, and it runs somewhat faster.

\strong{HBAM_MINI} combines the characteristics of HBAM_NE and HBAM_HM: It models the self-placements as if they contain no error and assumes the prediction errors in the stimuli placements to be homoskedastic. This is the simplest model provided in this package that still retains all key features of the HBAM model. It is also the fastest HBAM variant in this package -- sampling about twice as fast as the standard HBAM model for the dataset analyzed here (while yielding very similar point estimates). For large datasets, this model may provide a reasonable compromise between model complexity and estimation speed.

\strong{HBAM_0} does not allow for scale flipping. This may be useful if there are truly zero cases of scale flipping in the data. Such scenarios can be created artificially, but may also arise in real data. For example, expert surveys appear unlikely to contain many instances of scale flipping.

\strong{HBAM_R} incorporates the rationalization component of the ISR model by Bølstad (2020). This model requires additional data to be supplied to the \code{prep_data()} function: An N × J matrix of stimuli ratings from the respondents, supplied as the argument \code{pref}. The rationalization part of the model is simplified relative to the original ISR model: The direction in which respondents move disfavored stimuli is estimated as a common expectation for each possible self-placement on the scale.

\strong{HBAM_R_MINI} combines the features of the HBAM_R model with the light-weight features of the HBAM_MINI model to achieve faster sampling compared to HBAM_R.

\strong{FBAM_MINI} is a version of the HBAM_MINI model with fixed hyperparameters to allow fitting via optimization rather than MCMC -- which can be useful for large data sets. The hyperparameters have been set to realistic values based on analyses of ANES data. As with the other models, scale-dependent priors are automatically adjusted to the length of the survey scale.

\strong{BAM} is an unpooled model, similar to the JAGS version introduced by Hare et al. (2015). This model is mainly provided to offer a baseline for model comparisons. While it is simple and fast, this model tends to overfit the data and produce invalid posterior distributions for some respondent positions (Bølstad 2023).

Some of these models can also be used in situations where self-placements are not available and the only goal is to estimate stimulus positions. This can be achieved by supplying a vector of zeros (or random data) instead of real self-placements: \code{self = rep(0, nrow(stimuli))}. The HBAM_NE and HBAM_MINI models are then the relevant alternatives, as the other HBAM variants will include superfluous parameters (and will not sample properly with zero variance in the supplied self-placement data).
}
\examples{
\donttest{
# Loading and re-coding ANES 1980 data:
data(LC1980)
dat <- LC1980
dat[dat == 0 | dat == 8 | dat == 9] <- NA

# Making a small subset of the data for illustration:
self <- dat[1:100, 1]
stimuli <- dat[1:100, -1]

# Fitting the HBAM_MINI model, obtaining 1000 draws:
fit_hbam_mini <- hbam(self, stimuli, model = "HBAM_MINI",
                 warmup = 500, iter = 1000, chains = 2, thin = 1)

# Preparing the data before fitting, requiring complete responses:
dat <- prep_data(self, stimuli, allow_miss = 0)
fit_hbam_mini <- hbam(data = dat, prep_data = FALSE, model = "HBAM_MINI",
                 warmup = 500, iter = 1000, chains = 2, thin = 1)

# Obtaining posterior summaries for the latent stimulus positions:
theta_est <- get_est(fit_hbam_mini, par = "theta")
}
}
\references{
\itemize{
\item Bølstad, Jørgen (forthcoming). Hierarchical Bayesian Aldrich-McKelvey Scaling. \if{html}{\out{<i>}}Political Analysis\if{html}{\out{</i>}}.
\item Bølstad, Jørgen (2020). Capturing Rationalization Bias and Differential Item Functioning: A Unified Bayesian Scaling Approach. \if{html}{\out{<i>}}Political Analysis\if{html}{\out{</i>}} 28(3): 340–355.
\item Hare, Christopher et al. (2015). Using Bayesian Aldrich-McKelvey Scaling to Study Citizens' Ideological Preferences and Perceptions. \if{html}{\out{<i>}}American Journal of Political Science\if{html}{\out{</i>}} 59(3): 759–774.
}
}
