% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cols_units.R
\name{cols_units}
\alias{cols_units}
\title{Define units for one or more columns}
\usage{
cols_units(.data, ..., .list = list2(...), .units_pattern = NULL)
}
\arguments{
\item{.data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{...}{\emph{Column units definitions}

\verb{<multiple expressions>} // \strong{required} (or, use \code{.list})

Expressions for the assignment of column units for the table columns in
\code{.data}. Two-sided formulas (e.g., \verb{<LHS> ~ <RHS>}) can be used, where the
left-hand side corresponds to selections of columns and the right-hand side
evaluates to single-length values for the units to apply. Column names
should be enclosed in \code{c()}. Select helpers like \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, and \code{\link[=everything]{everything()}} can be used in
the LHS. Named arguments are also valid as input for simple mappings of
column name to the \strong{gt} units syntax; they should be of the form
\verb{<column name> = <units text>}. Subsequent expressions that operate on the
columns assigned previously will result in overwriting column units
definition values.}

\item{.list}{\emph{Alternative to \code{...}}

\verb{<list of multiple expressions>} // \strong{required} (or, use \code{...})

Allows for the use of a list as an input alternative to \code{...}.}

\item{.units_pattern}{\emph{Pattern to combine column labels and units}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional pattern to be used for combining column labels with the defined
units. The default pattern is \code{"{1}, {2}"}, where \code{"{1}"} refers to the
column label text and \code{"{2}"} is the text related to the associated units.
This default can be modified through the \code{column_labels.units_pattern}
option found in \code{\link[=tab_options]{tab_options()}}. Setting a value here will provide an
override to the \code{column_labels.units_pattern} default (only for the
resolved columns in the invocation of \code{cols_units()}).}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Column labels can sometimes contain measurement units, and these might range
from easy to define and typeset (e.g., \code{"m/s"}) to very difficult. Such
difficulty can arise from the need to include subscripts or superscripts,
non-ASCII symbols, etc. The \code{cols_units()} function tries to make this task
easier by letting you apply text pertaining to units to various columns. This
takes advantage of \strong{gt}'s specialized units notation (e.g.,
\code{"J Hz^-1 mol^-1"} can be used to generate units for the
\emph{molar Planck constant}). The notation here provides several conveniences for
defining units, letting you produce the correct formatting no matter what the
table output format might be (i.e., HTML, LaTeX, RTF, etc.). Details
pertaining to the units notation can be found in the section entitled
\emph{How to use \strong{gt}'s units notation}.
}
\section{How to use \strong{gt}'s units notation}{


The units notation involves a shorthand of writing units that feels familiar
and is fine-tuned for the task at hand. Each unit is treated as a separate
entity (parentheses and other symbols included) and the addition of subscript
text and exponents is flexible and relatively easy to formulate. This is all
best shown with examples:
\itemize{
\item \code{"m/s"} and \code{"m / s"} both render as \code{"m/s"}
\item \code{"m s^-1"} will appear with the \code{"-1"} exponent intact
\item \code{"m /s"} gives the same result, as \code{"/<unit>"} is equivalent to
\code{"<unit>^-1"}
\item \code{"E_h"} will render an \code{"E"} with the \code{"h"} subscript
\item \code{"t_i^2.5"} provides a \code{t} with an \code{"i"} subscript and a \code{"2.5"} exponent
\item \code{"m[_0^2]"} will use overstriking to set both scripts vertically
\item \code{"g/L \%C6H12O6\%"} uses a chemical formula (enclosed in a pair of \code{"\%"}
characters) as a unit partial, and the formula will render correctly with
subscripted numbers
\item Common units that are difficult to write using ASCII text may be implicitly
converted to the correct characters (e.g., the \code{"u"} in \code{"ug"}, \code{"um"},
\code{"uL"}, and \code{"umol"} will be converted to the Greek \emph{mu} symbol; \code{"degC"}
and \code{"degF"} will render a degree sign before the temperature unit)
\item We can transform shorthand symbol/unit names enclosed in \code{":"} (e.g.,
\code{":angstrom:"}, \code{":ohm:"}, etc.) into proper symbols
\item Greek letters can added by enclosing the letter name in \code{":"}; you can
use lowercase letters (e.g., \code{":beta:"}, \code{":sigma:"}, etc.) and uppercase
letters too (e.g., \code{":Alpha:"}, \code{":Zeta:"}, etc.)
\item The components of a unit (unit name, subscript, and exponent) can be
fully or partially italicized/emboldened by surrounding text with \code{"*"} or
\code{"**"}
}
}

\section{Examples}{


Let's analyze some \code{\link{pizzaplace}} data with \strong{dplyr} and then make a \strong{gt}
table. Here we are separately defining new column labels with \code{\link[=cols_label]{cols_label()}}
and then defining the units (to combine to those labels) through
\code{cols_units()}. The default pattern for combination is \code{"{1}, {2}"} which
is acceptable here.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace |>
  dplyr::mutate(month = lubridate::month(date, label = TRUE, abbr = TRUE)) |>
  dplyr::group_by(month) |>
  dplyr::summarize(
    n_sold = dplyr::n(),
    rev = sum(price)
  ) |>
  dplyr::mutate(chg = (rev - dplyr::lag(rev)) / dplyr::lag(rev)) |>
  dplyr::mutate(month = as.character(month)) |>
  gt(rowname_col = "month") |>
  fmt_integer(columns = n_sold) |>
  fmt_currency(columns = rev, use_subunits = FALSE) |>
  fmt_percent(columns = chg) |>
  sub_missing() |>
  cols_label(
    n_sold = "Number of Pizzas Sold",
    rev = "Revenue Generated",
    chg = "Monthly Changes in Revenue"
  ) |>
  cols_units(
    n_sold = "units month^-1",
    rev = "USD month^-1",
    chg = "\% change *m*/*m*"
  ) |>
  cols_width(
    stub() ~ px(40),
    everything() ~ px(200)
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_units_1.png" alt="This image of a table was generated from the first code example in the `cols_units()` help file." style="width:100\%;">
}}

The \code{\link{sza}} dataset has a wealth of information and here we'll generate
a smaller table that contains the average solar zenith angles at noon for
different months and at different northern latitudes. The column labels are
numbers representing the latitudes and it's convenient to apply units
of 'degrees north' to each of them with \code{cols_units()}. The extra thing we
wanted to do here was to ensure that the units are placed directly after
the column labels, and we do that with \code{.units_pattern = "{1}{2}"}. This
append the units (\code{"{2}"}) right to the column label (\code{"{1}"}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sza |>
  dplyr::filter(tst == "1200") |>
  dplyr::select(-tst) |>
  dplyr::arrange(desc(latitude)) |>
  tidyr::pivot_wider(
    names_from = latitude,
    values_from = sza
  ) |>
  gt(rowname_col = "month") |>
  cols_units(
    everything() ~ ":degree:N",
    .units_pattern = "\{1\}\{2\}"
  ) |>
  tab_spanner(
    label = "Solar Zenith Angle",
    columns = everything()
  ) |>
  text_transform(
    fn = toupper,
    locations = cells_stub()
  ) |>
  tab_style(
    style = cell_text(align = "right"),
    locations = cells_stub()
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_units_2.png" alt="This image of a table was generated from the second code example in the `cols_units()` help file." style="width:100\%;">
}}

Taking a portion of the \code{\link{towny}} dataset, let's use spanners to describe
what's in the columns and use only measurement units for the column labels.
The columns labels that have to do with population and density information
will be replaced with units defined in \code{cols_units()}. We'll use a
\code{.units_pattern} value of \code{"{2}"}, which means that only the units will
be present (the \code{"{1}"}, representing the column label text, is omitted).
Spanners added through several invocations of \code{\link[=tab_spanner]{tab_spanner()}} will declare
what the last four columns contain.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::select(
    name, land_area_km2,
    ends_with("2016"), ends_with("2021")
  ) |>
  dplyr::slice_max(population_2021, n = 10) |>
  gt(rowname_col = "name") |>
  tab_stubhead(label = "City") |>
  fmt_integer() |>
  cols_label(
    land_area_km2 ~ "Area, \{\{km^2\}\}",
    starts_with("population") ~ "",
    starts_with("density") ~ ""
  ) |>
  cols_units(
    starts_with("population") ~ "*ppl*",
    starts_with("density") ~ "*ppl* km^-2",
    .units_pattern = "\{2\}"
  ) |>
  tab_spanner(
    label = "Population",
    columns = starts_with("population"),
    gather = FALSE
  ) |>
  tab_spanner(
    label = "Density",
    columns = starts_with("density"),
    gather = FALSE
  ) |>
  tab_spanner(
    label = "2016",
    columns = ends_with("2016"),
    gather = FALSE
  ) |>
  tab_spanner(
    label = "2021",
    columns = ends_with("2021"),
    gather = FALSE
  ) |>
  tab_style(
    style = cell_text(align = "center"),
    locations = cells_column_labels(
      c(starts_with("population"), starts_with("density"))
    )
  ) |>
  cols_width(everything() ~ px(120)) |>
  opt_horizontal_padding(scale = 3)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_units_3.png" alt="This image of a table was generated from the third code example in the `cols_units()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

5-6
}

\section{Function Introduced}{

\code{v0.10.0} (October 7, 2023)
}

\seealso{
Other column modification functions: 
\code{\link{cols_add}()},
\code{\link{cols_align}()},
\code{\link{cols_align_decimal}()},
\code{\link{cols_hide}()},
\code{\link{cols_label}()},
\code{\link{cols_label_with}()},
\code{\link{cols_merge}()},
\code{\link{cols_merge_n_pct}()},
\code{\link{cols_merge_range}()},
\code{\link{cols_merge_uncert}()},
\code{\link{cols_move}()},
\code{\link{cols_move_to_end}()},
\code{\link{cols_move_to_start}()},
\code{\link{cols_nanoplot}()},
\code{\link{cols_unhide}()},
\code{\link{cols_width}()}
}
\concept{column modification functions}
