% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_url}
\alias{fmt_url}
\title{Format URLs to generate links}
\usage{
fmt_url(
  data,
  columns = everything(),
  rows = everything(),
  label = NULL,
  as_button = FALSE,
  color = "auto",
  show_underline = "auto",
  button_fill = "auto",
  button_width = "auto",
  button_outline = "auto"
)
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{columns}{The columns to format. Can either be a series of column names
provided in \code{\link[=c]{c()}}, a vector of column indices, or a helper function
focused on selections. The select helper functions are: \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and
\code{\link[=everything]{everything()}}.}

\item{rows}{Optional rows to format. Providing \code{\link[=everything]{everything()}} (the
default) results in all rows in \code{columns} being formatted. Alternatively,
we can supply a vector of row captions within \code{\link[=c]{c()}}, a vector of row
indices, or a helper function focused on selections. The select helper
functions are: \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}. We can also use expressions
to filter down to the rows we need (e.g.,
\verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{label}{The visible 'label' to use for the link. If \code{NULL} (the default)
the URL will serve as the label. There are two non-\code{NULL} options: (1) a
static text can be used for the label by providing a string, and (2) a
function can be provided to fashion a label from every URL.}

\item{as_button}{An option to style the link as a button. By default, this is
\code{FALSE}. If this option is chosen then the \code{button_fill} argument becomes
usable.}

\item{color}{The color used for the resulting link and its underline. This is
\code{"auto"} by default; this allows \strong{gt} to choose an appropriate color
based on various factors (e.g., background \code{button_fill} when \code{as_button}
is \code{TRUE}).}

\item{show_underline}{Should the link be decorated with an underline? By
default this is \code{"auto"} which means that \strong{gt} will choose \code{TRUE} when
\code{as_button = FALSE} and \code{FALSE} in the other case. The link underline will
be the same color as that set in the \code{color} option.}

\item{button_fill, button_width, button_outline}{Options for styling a
link-as-button (and only applies if \code{as_button = TRUE}). All of these
options are by default set to \code{"auto"}, allowing \strong{gt} to choose
appropriate fill, width, and outline values.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Should cells contain URLs, the \code{fmt_url()} function can be used to make them
navigable links. This should be expressly used on columns that contain \emph{only}
URL text (i.e., no URLs as part of a larger block of text). Should you have
such a column of data, there are options for how the links should be styled.
They can be of the conventional style (with underlines and text coloring that
sets it apart from other text), or, they can appear to be button-like (with
a surrounding box that can be filled with a color of your choosing).

URLs in data cells are detected in two ways. The first is using the simple
Markdown notation for URLs of the form: \verb{[label](URL)}. The second assumes
that the text is the URL. In the latter case the URL is also used as the
label but there is the option to use the \code{label} argument to modify that
text.
}
\section{Compatibility of formatting function with data values}{


The \code{fmt_url()} formatting function is compatible with body cells that are
of the \code{"character"} or \code{"factor"} types. Any other types of body cells are
ignored during formatting. This is to say that cells of incompatible data
types may be targeted, but there will be no attempt to format them.
}

\section{Targeting cells with \code{columns} and \code{rows}}{


Targeting of values is done through \code{columns} and additionally by \code{rows} (if
nothing is provided for \code{rows} then entire columns are selected). The
\code{columns} argument allows us to target a subset of cells contained in the
resolved columns. We say resolved because aside from declaring column names
in \code{c()} (with bare column names or names in quotes) we can use
\strong{tidyselect}-style expressions. This can be as basic as supplying a select
helper like \code{starts_with()}, or, providing a more complex incantation like

\code{where(~ is.numeric(.x) && max(.x, na.rm = TRUE) > 1E6)}

which targets numeric columns that have a maximum value greater than
1,000,000 (excluding any \code{NA}s from consideration).

By default all columns and rows are selected (with the \code{everything()}
defaults). Cell values that are incompatible with a given formatting function
will be skipped over, like \code{character} values and numeric \verb{fmt_*()}
functions. So it's safe to select all columns with a particular formatting
function (only those values that can be formatted will be formatted), but,
you may not want that. One strategy is to format the bulk of cell values with
one formatting function and then constrain the columns for later passes with
other types of formatting (the last formatting done to a cell is what you get
in the final output).

Once the columns are targeted, we may also target the \code{rows} within those
columns. This can be done in a variety of ways. If a stub is present, then we
potentially have row identifiers. Those can be used much like column names in
the \code{columns}-targeting scenario. We can use simpler \strong{tidyselect}-style
expressions (the select helpers should work well here) and we can use quoted
row identifiers in \code{c()}. It's also possible to use row indices (e.g.,
\code{c(3, 5, 6)}) though these index values must correspond to the row numbers of
the input data (the indices won't necessarily match those of rearranged rows
if row groups are present). One more type of expression is possible, an
expression that takes column values (can involve any of the available columns
in the table) and returns a logical vector. This is nice if you want to base
formatting on values in the column or another column, or, you'd like to use a
more complex predicate expression.
}

\section{Examples}{


Use the \code{\link{towny}} dataset to create a \strong{gt} table. After some major
\strong{dplyr}ing to get a nicely formatted data table, we use the \code{fmt_url()}
function on the \code{website} column to generate navigable links to websites.
By default the links are underlined and the color will be chosen for you
(it's dark cyan).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::filter(csd_type == "city") |>
  dplyr::arrange(desc(population_2021)) |>
  dplyr::select(name, website, population_2021) |>
  dplyr::slice_head(n = 10) |>
  gt() |>
  tab_header(
    title = md("The 10 Largest Municipalities in `towny`"),
    subtitle = "Population values taken from the 2021 census."
  ) |>
  fmt_integer() |>
  fmt_url(columns = website) |>
  cols_label(
    name = "Name",
    website = "Site",
    population_2021 = "Population"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_url_1.png" alt="This image of a table was generated from the first code example in the `fmt_url()` help file." style="width:100\%;">
}}

Let's try something else. We can set a static text label for the link with
the \code{label} argument (and we'll use the word \code{"site"} for this). The link
underline is removable with \code{show_underline = FALSE}. With this change, it
seems sensible to merge the link to the \code{"name"} column and enclose the link
text in parentheses (the \code{\link[=cols_merge]{cols_merge()}} function handles all that).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::filter(csd_type == "city") |>
  dplyr::arrange(desc(population_2021)) |>
  dplyr::select(name, website, population_2021) |>
  dplyr::slice_head(n = 10) |>
  gt() |>
  tab_header(
    title = md("The 10 Largest Municipalities in `towny`"),
    subtitle = "Population values taken from the 2021 census."
  ) |>
  fmt_integer() |>
  fmt_url(
    columns = website,
    label = "site",
    show_underline = FALSE
  ) |>
  cols_merge(
    columns = c(name, website),
    pattern = "\{1\} (\{2\})"
  ) |>
  cols_label(
    name = "Name",
    population_2021 = "Population"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_url_2.png" alt="This image of a table was generated from the second code example in the `fmt_url()` help file." style="width:100\%;">
}}

The \code{fmt_url()} function allows for the styling of links as 'buttons'. This
is as easy as setting \code{as_button = TRUE}. Doing that unlocks the ability to
set a \code{button_fill} color. This color can automatically selected by \strong{gt}
(this is the default) but here we're using \code{"steelblue"}. The \code{label}
argument also accepts a function! We can choose to adapt the label text from
the URLs by eliminating any leading \code{"https://"} or \code{"www."} parts.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::filter(csd_type == "city") |>
  dplyr::arrange(desc(population_2021)) |>
  dplyr::select(name, website, population_2021) |>
  dplyr::slice_head(n = 10) |>
  dplyr::mutate(ranking = dplyr::row_number()) |>
  gt(rowname_col = "ranking") |>
  tab_header(
    title = md("The 10 Largest Municipalities in `towny`"),
    subtitle = "Population values taken from the 2021 census."
  ) |>
  fmt_integer() |>
  fmt_url(
    columns = website,
    label = function(x) gsub("https://|www.", "", x),
    as_button = TRUE,
    button_fill = "steelblue",
    button_width = px(150)
  ) |>
  cols_move_to_end(columns = website) |>
  cols_align(align = "center", columns = website) |>
  cols_width(
    ranking ~ px(40),
    website ~ px(200)
  ) |>
  tab_options(column_labels.hidden = TRUE) |>
  tab_style(
    style = cell_text(weight = "bold"),
    locations = cells_stub()
  ) \%>\%
  opt_vertical_padding(scale = 0.75)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_url_3.png" alt="This image of a table was generated from the third code example in the `fmt_url()` help file." style="width:100\%;">
}}

It's perhaps inevitable that you'll come across missing values in your column
of URLs. The \code{fmt_url()} function will preserve input \code{NA} values, allowing
you to handle them with \code{\link[=sub_missing]{sub_missing()}}. Here's an example of that.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::arrange(population_2021) |>
  dplyr::select(name, website, population_2021) |>
  dplyr::slice_head(n = 10) |>
  gt() |>
  tab_header(
    title = md("The 10 Smallest Municipalities in `towny`"),
    subtitle = "Population values taken from the 2021 census."
  ) |>
  fmt_integer() |>
  fmt_url(columns = website) |>
  cols_label(
    name = "Name",
    website = "Site",
    population_2021 = "Population"
  ) |>
  sub_missing()
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_url_4.png" alt="This image of a table was generated from the fourth code example in the `fmt_url()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-18
}

\section{Function Introduced}{

\emph{In Development}
}

\seealso{
Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_date}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_time}()},
\code{\link{fmt}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
