% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gpboost.R
\name{gpboost}
\alias{gpboost}
\title{Train a GPBoost model}
\usage{
gpboost(data, label = NULL, weight = NULL, params = list(),
  nrounds = 100L, gp_model = NULL, use_gp_model_for_validation = TRUE,
  train_gp_model_cov_pars = TRUE, valids = list(), obj = NULL,
  eval = NULL, verbose = 1L, record = TRUE, eval_freq = 1L,
  early_stopping_rounds = NULL, init_model = NULL, colnames = NULL,
  categorical_feature = NULL, callbacks = list(), ...)
}
\arguments{
\item{data}{a \code{gpb.Dataset} object, used for training. Some functions, such as \code{\link{gpb.cv}},
may allow you to pass other types of data like \code{matrix} and then separately supply
\code{label} as a keyword argument.}

\item{label}{Vector of labels, used if \code{data} is not an \code{\link{gpb.Dataset}}}

\item{weight}{vector of response values. If not NULL, will set to dataset}

\item{params}{List of parameters (many of them tuning paramters), see Parameters.rst for more information. A few key parameters:
\itemize{
    \item{\code{learning_rate}}{ The learning rate, also called shrinkage or damping parameter 
    (default = 0.1). An important tuning parameter for boosting. Lower values usually 
    lead to higher predictive accuracy but more boosting iterations are needed }
    \item{\code{num_leaves}}{ Number of leaves in a tree. Tuning parameter for 
    tree-boosting (default = 31)}
    \item{\code{min_data_in_leaf}}{ Minimal number of samples per leaf. Tuning parameter for 
    tree-boosting (default = 20)}
    \item{\code{max_depth}}{ Maximal depth of a tree. Tuning parameter for tree-boosting (default = no limit)}
    \item{\code{leaves_newton_update}}{ Set this to TRUE to do a Newton update step for the tree leaves 
    after the gradient step. Applies only to Gaussian process boosting (GPBoost algorithm)}
    \item{\code{train_gp_model_cov_pars}}{ If TRUE, the covariance parameters of the Gaussian process 
    are stimated in every boosting iterations, 
    otherwise the gp_model parameters are not estimated. In the latter case, you need to 
    either esimate them beforehand or provide the values via 
    the 'init_cov_pars' parameter when creating the gp_model (default = TRUE).}
    \item{\code{use_gp_model_for_validation}}{ If TRUE, the Gaussian process is also used 
    (in addition to the tree model) for calculating predictions on the validation data 
    (default = TRUE)}
    \item{\code{use_nesterov_acc}}{ Set this to TRUE to do boosting with Nesterov acceleration (default = FALSE). 
    Can currently only be used for tree_learner = "serial" (default option)}
    \item{\code{nesterov_acc_rate}}{ Acceleration rate for momentum step in case Nesterov accelerated 
    boosting is used (default = 0.5)}
    \item{\code{oosting}}{ Boosting type. \code{"gbdt"}, \code{"rf"}, \code{"dart"} or \code{"goss"}.
    Only \code{"gbdt"} allows for doing Gaussian process boosting.}
    \item{num_threads}{ Number of threads. For the best speed, set this to
                       the number of real CPU cores, not the number of threads (most
                       CPU using hyper-threading to generate 2 threads per CPU core).}
}}

\item{nrounds}{nNumber of boosting iterations (= number of trees). This is the most important tuning parameter for boosting. Default = 100}

\item{gp_model}{A \code{GPModel} object that contains the random effects (Gaussian process and / or grouped random effects) model}

\item{use_gp_model_for_validation}{Boolean (default = TRUE). If TRUE, the Gaussian process is also used (in addition to the tree model) for calculating predictions on the validation data}

\item{train_gp_model_cov_pars}{Boolean (default = TRUE). If TRUE, the covariance parameters of the Gaussian process are estimated in every boosting iterations, 
otherwise the gp_model parameters are not estimated. In the latter case, you need to either esimate them beforehand or provide the values via 
the 'init_cov_pars' parameter when creating the gp_model}

\item{valids}{a list of \code{gpb.Dataset} objects, used for validation}

\item{obj}{objective function, can be character or custom objective function. Examples include
\code{regression}, \code{regression_l1}, \code{huber},
\code{binary}, \code{lambdarank}, \code{multiclass}, \code{multiclass}}

\item{eval}{evaluation function(s). This can be a character vector, function, or list with a mixture of
            strings and functions.

            \itemize{
                \item{\bold{a. character vector}:
                    If you provide a character vector to this argument, it should contain strings with valid
                    evaluation metrics.
                    See \href{https://lightgbm.readthedocs.io/en/latest/Parameters.html#metric}{
                    The "metric" section of the documentation}
                    for a list of valid metrics.
                }
                \item{\bold{b. function}:
                     You can provide a custom evaluation function. This
                     should accept the keyword arguments \code{preds} and \code{dtrain} and should return a named
                     list with three elements:
                     \itemize{
                         \item{\code{name}: A string with the name of the metric, used for printing
                             and storing results.
                         }
                         \item{\code{value}: A single number indicating the value of the metric for the
                             given predictions and true values
                         }
                         \item{
                             \code{higher_better}: A boolean indicating whether higher values indicate a better fit.
                             For example, this would be \code{FALSE} for metrics like MAE or RMSE.
                         }
                     }
                }
                \item{\bold{c. list}:
                    If a list is given, it should only contain character vectors and functions.
                    These should follow the requirements from the descriptions above.
                }
            }}

\item{verbose}{verbosity for output, if <= 0, also will disable the print of evaluation during training}

\item{record}{Boolean, TRUE will record iteration message to \code{booster$record_evals}}

\item{eval_freq}{evaluation output frequency, only effect when verbose > 0}

\item{early_stopping_rounds}{int. Activates early stopping. Requires at least one validation data
and one metric. If there's more than one, will check all of them
except the training data. Returns the model with (best_iter + early_stopping_rounds).
If early stopping occurs, the model will have 'best_iter' field.}

\item{init_model}{path of model file of \code{gpb.Booster} object, will continue training from this model}

\item{colnames}{feature names, if not null, will use this to overwrite the names in dataset}

\item{categorical_feature}{list of str or int
type int represents index,
type str represents feature names}

\item{callbacks}{List of callback functions that are applied at each iteration.}

\item{...}{Additional arguments passed to \code{\link{gpb.train}}. For example
\itemize{
   \item{\code{valids}: a list of \code{gpb.Dataset} objects, used for validation}
   \item{\code{obj}: objective function, can be character or custom objective function. Examples include
              \code{regression}, \code{regression_l1}, \code{huber},
               \code{binary}, \code{lambdarank}, \code{multiclass}, \code{multiclass}}
   \item{\code{eval}: evaluation function, can be (a list of) character or custom eval function}
   \item{\code{record}: Boolean, TRUE will record iteration message to \code{booster$record_evals}}
   \item{\code{colnames}: feature names, if not null, will use this to overwrite the names in dataset}
   \item{\code{categorical_feature}: categorical features. This can either be a character vector of feature
                       names or an integer vector with the indices of the features (e.g. \code{c(1L, 10L)} to
                       say "the first and tenth columns").}
   \item{\code{reset_data}: Boolean, setting it to TRUE (not the default value) will transform the booster model
                     into a predictor model which frees up memory and the original datasets}
}}
}
\value{
a trained \code{gpb.Booster}
}
\description{
Simple interface for training a GPBoost model.
}
\section{Early Stopping}{


         "early stopping" refers to stopping the training process if the model's performance on a given
         validation set does not improve for several consecutive iterations.

         If multiple arguments are given to \code{eval}, their order will be preserved. If you enable
         early stopping by setting \code{early_stopping_rounds} in \code{params}, by default all
         metrics will be considered for early stopping.

         If you want to only consider the first metric for early stopping, pass
         \code{first_metric_only = TRUE} in \code{params}. Note that if you also specify \code{metric}
         in \code{params}, that metric will be considered the "first" one. If you omit \code{metric},
         a default metric will be used based on your choice for the parameter \code{obj} (keyword argument)
         or \code{objective} (passed into \code{params}).
}

\examples{
# See https://github.com/fabsig/GPBoost/tree/master/R-package for more examples

library(gpboost)
data(GPBoost_data, package = "gpboost")

#--------------------Combine tree-boosting and grouped random effects model----------------
# Create random effects model
gp_model <- GPModel(group_data = group_data[,1], likelihood = "gaussian")
# The default optimizer for covariance parameters for Gaussian data is Fisher scoring.
# For non-Gaussian data, gradient descent is used.
# Optimizer properties can be changed as follows:
# re_params <- list(optimizer_cov = "gradient_descent", use_nesterov_acc = TRUE)
# gp_model$set_optim_params(params=re_params)
# Use trace = TRUE to monitor convergence:
# re_params <- list(trace = TRUE)
# gp_model$set_optim_params(params=re_params)

# Train model
bst <- gpboost(data = X,
               label = y,
               gp_model = gp_model,
               nrounds = 16,
               learning_rate = 0.05,
               max_depth = 6,
               min_data_in_leaf = 5,
               objective = "regression_l2",
               verbose = 0)
# Estimated random effects model
summary(gp_model)

# Make predictions
pred <- predict(bst, data = X_test, group_data_pred = group_data_test[,1],
                predict_var= TRUE)
pred$random_effect_mean # Predicted mean
pred$random_effect_cov # Predicted variances
pred$fixed_effect # Predicted fixed effect from tree ensemble
# Sum them up to otbain a single prediction
pred$random_effect_mean + pred$fixed_effect

\donttest{
#--------------------Combine tree-boosting and Gaussian process model----------------
# Create Gaussian process model
gp_model <- GPModel(gp_coords = coords, cov_function = "exponential",
                    likelihood = "gaussian")
# Train model
bst <- gpboost(data = X,
               label = y,
               gp_model = gp_model,
               nrounds = 8,
               learning_rate = 0.1,
               max_depth = 6,
               min_data_in_leaf = 5,
               objective = "regression_l2",
               verbose = 0)
# Estimated random effects model
summary(gp_model)
# Make predictions
pred <- predict(bst, data = X_test, gp_coords_pred = coords_test,
                predict_cov_mat =TRUE)
pred$random_effect_mean # Predicted (posterior) mean of GP
pred$random_effect_cov # Predicted (posterior) covariance matrix of GP
pred$fixed_effect # Predicted fixed effect from tree ensemble
# Sum them up to otbain a single prediction
pred$random_effect_mean + pred$fixed_effect
}
}
\author{
Authors of the LightGBM R package, Fabio Sigrist
}
