% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glamlassoS.R
\name{glamlassoS}
\alias{glamlassoS}
\title{Penalization in Large Scale Generalized Linear Array Models}
\usage{
glamlassoS(X, 
           Y,
           V, 
           Z = NULL,
           family = "gaussian",
           penalty = "lasso",
           intercept = FALSE,
           weights = NULL,
           thetainit = NULL,
           alphainit = NULL,
           nlambda = 100,
           lambdaminratio = 1e-04,
           lambda = NULL,
           penaltyfactor = NULL,
           penaltyfactoralpha = NULL,
           reltolinner = 1e-07,
           reltolouter = 1e-04,
           maxiter = 15000,
           steps = 1,
           maxiterinner = 3000,
           maxiterouter = 25,
           btinnermax = 100,
           btoutermax = 100,
           iwls = "exact",
           nu = 1)
}
\arguments{
\item{X}{A list containing the tensor components (2 or 3) of the tensor design matrix.
These are  matrices of sizes \eqn{n_i   \times p_i}.}

\item{Y}{The response values, an array of size \eqn{n_1 \times\cdots\times n_d}. For option 
\code{family = "binomial"} this array must contain the proportion of successes and the 
number of trials is then specified as \code{weights} (see below).}

\item{V}{The weight values, an array of size \eqn{n_1 \times\cdots\times n_d}.}

\item{Z}{The non tensor structrured part of the design matrix. A matrix of size \eqn{n_1 \cdots n_d\times q}. 
Is set to \code{NULL} as default.}

\item{family}{A string specifying the model family (essentially the response distribution). Possible values 
are \code{"gaussian", "binomial", "poisson", "gamma"}.}

\item{penalty}{A string specifying the penalty. Possible values 
are \code{"lasso", "scad"}.}

\item{intercept}{Logical variable indicating if the model includes an intercept. 
When \code{intercept = TRUE} the first coulmn in the non-tensor design component \code{Z} is all 1s.
 Default is \code{FALSE}.}

\item{weights}{Observation weights, an array of size \eqn{n_1 \times \cdots \times n_d}. For option 
\code{family = "binomial"} this array must contain the number of trials and must be provided.}

\item{thetainit}{The initial parameter values. Default is NULL in which case all parameters are initialized at zero.}

\item{alphainit}{A \eqn{q\times 1} vector containing the initial parameter values for the non-tensor parameter. 
Default is NULL in which case all parameters are initialized at 0.}

\item{nlambda}{The number of \code{lambda} values.}

\item{lambdaminratio}{The smallest value for \code{lambda}, given as a fraction of 
\eqn{\lambda_{max}}; the (data derived) smallest value for which all coefficients are zero.}

\item{lambda}{The sequence of penalty parameters for the regularization path.}

\item{penaltyfactor}{An array of size \eqn{p_1 \times \cdots \times p_d}. Is multiplied 
with each element in \code{lambda} to allow differential shrinkage on the coefficients.}

\item{penaltyfactoralpha}{A \eqn{q \times 1} vector multiplied with each element in \code{lambda} to allow differential 
shrinkage on the non-tensor coefficients.}

\item{reltolinner}{The convergence tolerance for the inner loop}

\item{reltolouter}{The convergence tolerance for the outer loop.}

\item{maxiter}{The maximum number of inner iterations allowed for each \code{lambda}
value, when  summing over all outer iterations for said \code{lambda}.}

\item{steps}{The number of steps used in the multi-step adaptive lasso algorithm for non-convex penalties. Automatically set to 1 when \code{penalty = "lasso"}.}

\item{maxiterinner}{The maximum number of inner iterations allowed for each outer iteration.}

\item{maxiterouter}{The maximum number of outer iterations allowed for each lambda.}

\item{btinnermax}{Maximum number of backtracking steps allowed in each inner iteration. Default is \code{btinnermax = 100}.}

\item{btoutermax}{Maximum number of backtracking steps allowed in each outer iteration. Default is \code{btoutermax = 100}.}

\item{iwls}{A string indicating whether to use the exact iwls weight matrix or use a kronecker structured approximation to it.}

\item{nu}{A number between 0 and 1 that controls the step size \eqn{\delta} in the proximal algorithm (inner loop) by 
scaling the upper bound \eqn{\hat{L}_h} on the Lipschitz constant \eqn{L_h} (see \cite{Lund et al., 2017}). 
For \code{nu = 1} backtracking never occurs and the proximal step size is always \eqn{\delta = 1 / \hat{L}_h}. 
For \code{nu = 0} backtracking always occurs and the proximal step size is initially \eqn{\delta = 1}. 
For \code{0 < nu < 1} the proximal step size is initially \eqn{\delta = 1/(\nu\hat{L}_h)} and backtracking 
is only employed if the objective function does not decrease. A \code{nu} close  to 0 gives large step 
sizes and presumably more backtracking in the inner loop. The default is \code{nu = 1} and the option is only 
used if \code{iwls = "exact"}.}
}
\description{
Efficient design matrix free procedure for fitting   a special case of  a generalized linear  model 
with  array structured response and partially tensor structured covariates.  See \cite{Lund et al., 2017} for an application of this special purpose function.
}
\details{
Given the setting from \code{\link{glamlasso}} we consider a model
 where the tensor design component is only partially tensor structured as
 \deqn{X = [V_1X_2^\top\otimes X_1^\top,\ldots,V_{n_3}X_2^\top\otimes X_1^\top]^\top.} 
 Here  \eqn{X_i} is a \eqn{n_i\times p_i} matrix  for \eqn{i=1,2} and  \eqn{V_i} is a \eqn{n_1n_2\times n_1n_2} diagonal matrix for \eqn{i=1,\ldots,n_3}.
  
Letting \eqn{Y} denote the  \eqn{n_1\times n_2\times n_3} response array and \eqn{V} the  \eqn{n_1\times n_2\times n_3}  weight array containing the diagonals of the \eqn{V_i}s, 
the function \code{glamlassoS} solves the PMLE problem using  \eqn{Y, V, X_1, X_2} and the non-tensor component \eqn{Z} as input.
}
\examples{
\dontrun{

##size of example
n1 <- 65; n2 <- 26; n3 <- 13; p1 <- 13; p2 <- 5; 

##marginal design matrices (tensor components)
X1 <- matrix(rnorm(n1 * p1), n1, p1)
X2 <- matrix(rnorm(n2 * p2), n2, p2)
X <- list(X1, X2)
V <- array(rnorm(n3 * n2 * n1), c(n1, n2, n3))

##gaussian example
Beta <- array(rnorm(p1 * p2) * rbinom(p1 * p2, 1, 0.1), c(p1 , p2))
Mu <- V * array(RH(X2, RH(X1, Beta)), c(n1, n2, n3))
Y <- array(rnorm(n1 * n2 * n3, Mu), c(n1, n2, n3))
system.time(fit <- glamlassoS(X, Y, V))

modelno <- length(fit$lambda)
plot(c(Beta), ylim = range(Beta, fit$coef[, modelno]), type = "h")
points(c(Beta))
lines(c(fit$coef[, modelno]), col = "red", type = "h")

###with non tensor design component Z
q <- 5
alpha <- matrix(rnorm(q)) * rbinom(q, 1, 0.5)
Z <- matrix(rnorm(n1 * n2 * n3 * q), n1 * n2 *n3, q) 
Y <- array(rnorm(n1 * n2 * n3, Mu + array(Z \%*\% alpha, c(n1, n2, n3))), c(n1, n2, n3))
system.time(fit <- glamlassoS(X, Y, V , Z))

modelno <- length(fit$lambda)
par(mfrow = c(1, 2))
plot(c(Beta), type="h", ylim = range(Beta, fit$coef[, modelno]))
points(c(Beta))
lines(fit$coef[ , modelno], col = "red", type = "h")
plot(c(alpha), type = "h", ylim = range(alpha, fit$alpha[, modelno]))
points(c(alpha))
lines(fit$alpha[ , modelno], col = "red", type = "h")

################ poisson example
Beta <- matrix(rnorm(p1 * p2, 0, 0.1) * rbinom(p1 * p2, 1, 0.1), p1 , p2)
Mu <- V * array(RH(X2, RH(X1, Beta)), c(n1, n2, n3))
Y <- array(rpois(n1 * n2 * n3, exp(Mu)), dim = c(n1, n2, n3))
system.time(fit <- glamlassoS(X, Y, V, family = "poisson", nu = 0.1))

modelno <- length(fit$lambda)
plot(c(Beta), type = "h", ylim = range(Beta, fit$coef[, modelno]))
points(c(Beta))
lines(fit$coef[ , modelno], col = "red", type = "h")

}
}
\references{
Lund, A. and N. R. Hansen (2017). Sparse Network  Estimation for  Dynamical Spatio-temporal Array Models. 
 \emph{ArXiv}.
}
\author{
Adam Lund

Maintainer: Adam Lund, \email{adam.lund@math.ku.dk}
}
\keyword{package}
