% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/angle_correction.r
\name{angle_correction}
\alias{angle_correction}
\title{Calculate correction for angle in the plot coordinate system}
\usage{
angle_correction(data, panel_params, coord)
}
\arguments{
\item{data}{fortified data used in a \code{\link[=geom_fields]{geom_fields()}}. Should at least contain \code{numeric} columns \code{x}, \code{y} and
\code{angle}.}

\item{panel_params}{panel parameters as returned by \link[=GeomFields]{GeomFields$setup_params()}}

\item{coord}{A \link[ggplot2:coord_cartesian]{coord} object.}
}
\value{
A \code{data.frame} with an additional \code{angle_correction} column. The corrected angle is given
by \code{angle_correction + angle}.
}
\description{
The angle of a vector may be distorted when your plot uses a different
coordinate system than the one for which the angle is specified. If \code{data}
is a simple feature object (\link[sf:sf]{sf}), the angle will be corrected
for the displayed coordinate reference system (\link[sf:st_crs]{crs}). When
the crs is missing, an aspect ratio of 1 is assumed. For any other data,
the angle is corrected for the aspect ratio in the plot.
}
\details{
This function is used by default by \code{\link[=geom_fields]{geom_fields()}}. For more details on
why this correction is required and how to customize corrections please see
\code{vignette("angle_correction")}.
}
\examples{
## Create a data.frame with some xy-coordinates and all angles pointing North (0 degrees)
d <-
  data.frame(
    x = seq(1, 2, 0.1),
    y = seq(50, 51, 0.1),
    angle = 0
  ) |>
  sf::st_as_sf(coords = c("x", "y"), crs = 4326, remove = FALSE)

## Create a mockup of ggplot params. Normally this is handled automatically by ggplot2
params_mockup <-
  c(
    ggplot2::ggplot() + geom_fields(),
    list(
      x_range = range(d$x),
      y_range = range(d$y),
      crs = sf::st_crs(4326),
      default_crs = 4326
    )
  )

## When plotting as lon-lat, the angle correction will be zero
angle_correction(d, params_mockup, ggplot2::coord_sf(default_crs = 4326))

## Transform to UTM zone 31N in meters
d2 <- d |> sf::st_transform(32631)

## Again get parameter mockup values
params_mockup2 <-
  c(
    ggplot2::ggplot() + geom_fields(),
      list(
        x_range = range(sf::st_coordinates(d2)[,1]),
        y_range = range(sf::st_coordinates(d2)[,1]),
        crs = sf::st_crs(32631),
        default_crs = 4326
      )
    )
## in UTM projection in this area (which is slightly tilted) the correction is
## larger than zero
angle_correction(d2, params_mockup2,
                 ggplot2::coord_sf(crs = 32631, default_crs = 4326))
}
\author{
Pepijn de Vries
}
