% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cut_cdf_qi.R
\name{cut_cdf_qi}
\alias{cut_cdf_qi}
\title{Categorize values from a CDF into quantile intervals}
\usage{
cut_cdf_qi(p, .width = c(0.66, 0.95, 1), labels = NULL)
}
\arguments{
\item{p}{A numeric vector of values from a cumulative distribution function,
such as values returned by \code{p}-prefixed distribution functions in base R (e.g. \code{\link[=pnorm]{pnorm()}}),
the \code{\link[=cdf]{cdf()}} function, or values of the \code{cdf} computed aesthetic from the
\code{\link[=stat_slabinterval]{stat_slabinterval()}} family of stats.}

\item{.width}{vector of probabilities to use that determine the widths of the resulting intervals.}

\item{labels}{One of:
\itemize{
\item \code{NULL} to use the default labels (\code{.width} converted to a character vector).
\item A character vector giving labels (must be same length as \code{.width})
\item A function that takes numeric probabilities as input and returns labels as output
(a good candidate might be \code{\link[scales:percent_format]{scales::percent_format()}}).
}}
}
\value{
An \link{ordered} factor of the same length as \code{p} giving the quantile interval to
which each value of \code{p} belongs.
}
\description{
Given a vector of probabilities from a cumulative distribution function (CDF)
and a list of desired quantile intervals, return a vector categorizing each
element of the input vector according to which quantile interval it falls into.
\strong{NOTE:} While this function can be used for (and was originally designed for)
drawing slabs with intervals overlaid on the density, this is can now be
done more easily by mapping the \code{.width} or \code{level} computed variable to
slab fill or color. See \strong{Examples}.
}
\examples{

library(ggplot2)
library(dplyr)
library(scales)
library(distributional)

theme_set(theme_ggdist())

# NOTE: cut_cdf_qi() used to be the recommended way to do intervals overlaid
# on densities, like this...
tibble(x = dist_normal(0, 1)) \%>\%
  ggplot(aes(xdist = x)) +
  stat_slab(
    aes(fill = stat(cut_cdf_qi(cdf)))
  ) +
  scale_fill_brewer(direction = -1)

# ... however this is now more easily and flexibly accomplished by directly
# mapping .width or level onto fill:
tibble(x = dist_normal(0, 1)) \%>\%
  ggplot(aes(xdist = x)) +
  stat_slab(
    aes(fill = stat(level)),
    .width = c(.66, .95, 1)
  ) +
  scale_fill_brewer()

# See vignette("slabinterval") for more examples. The remaining examples
# below using cut_cdf_qi() are kept for posterity.

# With a halfeye (or other geom with slab and interval), NA values will
# show up in the fill scale from the CDF function applied to the internal
# interval geometry data and can be ignored, hence na.translate = FALSE
tibble(x = dist_normal(0, 1)) \%>\%
  ggplot(aes(xdist = x)) +
  stat_halfeye(aes(
    fill = stat(cut_cdf_qi(cdf, .width = c(.5, .8, .95, 1)))
  )) +
  scale_fill_brewer(direction = -1, na.translate = FALSE)

# we could also use the labels parameter to apply nicer formatting
# and provide a better name for the legend, and omit the 100\% interval
# if desired
tibble(x = dist_normal(0, 1)) \%>\%
  ggplot(aes(xdist = x)) +
  stat_halfeye(aes(
    fill = stat(cut_cdf_qi(cdf, .width = c(.5, .8, .95), labels = percent_format(accuracy = 1)))
  )) +
  labs(fill = "Interval") +
  scale_fill_brewer(direction = -1, na.translate = FALSE)

}
\seealso{
See \code{\link[=stat_slabinterval]{stat_slabinterval()}} and
its shortcut stats, which generate \code{cdf} aesthetics that can be used with
\code{\link[=cut_cdf_qi]{cut_cdf_qi()}} to draw slabs colored by their intervals.
}
