#' Download official data of municipal seats (sede dos municipios) in Brazil as an sf object.
#'
#' This function reads the official data on the municipal seats (sede dos municipios) of Brazil.
#' The data brings the spatial coordinates (lat lon) of of municipal seats for various years
#' between 1872 and 2010. Orignal data were generated by Brazilian Institute of Geography
#' and Statistics (IBGE).
#'
#'
#' @param year A year number in YYYY format (Defaults to 2010)
#' @export
#' @examples \donttest{
#'
#' library(geobr)
#'
#' # Read municipal seats in an specific year
#'   m <- read_municipal_seat(year=1991)
#'
#' }
#'
#'
read_municipal_seat <- function(year=NULL){

  # Get metadata with data addresses
  metadata <- download_metadata()

  # Select geo
  temp_meta <- subset(metadata, geo=="municipal_seat")

  # Verify year input
  if (is.null(year)){ message("Using latest data from year 2010\n")
    year <- 2010
    temp_meta <- subset(temp_meta, year==2010)

  } else if (year %in% temp_meta$year){ temp_meta <- temp_meta[temp_meta[,2] == year, ]

  } else { stop(paste0("Error: Invalid Value to argument 'year'. It must be one of the following: ",
                       paste(unique(temp_meta$year),collapse = " ")))
  }


  # list paths of files to download
  filesD <- as.character(temp_meta$download_path)

  # download files
  temps <- download_gpkg(filesD)


  # read sf
  temp_sf <- sf::st_read(temps, quiet=T)
  return(temp_sf)
}
