#' Compute the Synthesis Operator for Transform Coefficients
#'
#' \code{synthesis} computes the graph signal synthesis from its transform coefficients using the provided frame coefficients.
#'
#' @export synthesis
#' @param coeff Numeric vector/matrix. Transformed coefficients of the graph signal.
#' @param tf Numeric matrix. Frame coefficients.
#' @return \code{y} Numeric vector/matrix. Synthesized graph signal.
#'
#' @details
#' The \code{synthesis} operator uses the frame coefficients to retrieve the graph signal from its representation in the transform domain. It is the adjoint of the analysis operator \eqn{T_{\mathfrak F}}{T_F} and is defined by the linear map \eqn{T_{\mathfrak F}^\ast : \mathbb R^I \rightarrow \mathbb R^V}{T_F* : R^I -> R^V}. For a vector of coefficients \eqn{(c_i)_{i \in I}}{(c_i) for all i in I}, the synthesis operation is defined as:
#' \deqn{T^\ast_{\mathfrak F}(c_i)_{i \in I}=\sum_{i \in I} c_i r_i}{T*_F(c_i) for all i in I = sum of c_i * r_i for all i in I}
#'
#' The synthesis is computed as:
#' \deqn{\code{y} = \code{coeff}^T\code{tf}}{y = t(coeff) \%*\% tf}
#'
#' @examples
#' \dontrun{
#' # Extract the adjacency matrix from the grid1 and compute the Laplacian
#' L <- laplacian_mat(grid1$sA)
#'
#' # Compute the spectral decomposition of L
#' decomp <- eigensort(L)
#'
#' # Generate the tight frame coefficients using the tight_frame function
#' tf <- tight_frame(decomp$evalues, decomp$evectors)
#'
#' # Create a random graph signal.
#' f <- rnorm(nrow(L))
#'
#' # Compute the transform coefficients using the analysis operator
#' coef <- analysis(f, tf)
#'
#' # Retrieve the graph signal using the synthesis operator
#' f_rec <- synthesis(coef, tf)
#' }
#' @seealso \code{\link{analysis}}, \code{\link{tight_frame}}

synthesis <- function(coeff, tf) {
  y <- t(coeff) %*% tf
  return(as.vector(y))
}

