#' @title Creation of an empty fuzzy spatial inference model
#'
#' @description This function builds a fuzzy spatial inference (FSI) model without elements of the data source component (i.e., spatial plateau objects, fuzzy rules set, and fuzzy sets).
#'
#' @usage
#'
#' fsi_create(name, and_method = "min", or_method = "max",
#'            imp_method = "min", agg_method = "max", 
#'            defuzz_method = "centroid", default_conseq = NULL)
#'
#' @param name A character value that specifies the name of the FSI model.
#' @param and_method A character value that defines the operator name for the logical connective AND. Default value is `"min"`.
#' @param or_method A character value that defines the operator for the logical connective OR. Default value is `"max"`.
#' @param imp_method A character value that defines the operator for the implication operator. Default value is `"min"`.
#' @param agg_method A character value that defines the operator for the aggregation operator. Default value is `"max"`.
#' @param defuzz_method A character value that determines the defuzzification technique. Default value is the centroid technique.
#' @param default_conseq This parameter is a membership function generated by the function `genmf` of the FuzzyR package.
#' 
#' @details
#' 
#' The FSI model created with the function `fsi_create` and its default parameter values will implement a model using Mamdani's method.
#' The possible values for the parameters `and_method` and `imp_method` are: `"min"`, `"prod"`. Other t-norms from the FuzzyR package are also conceivable.
#' The possible value for the parameters `or_method` and `agg_method` is: `"max"`. Other t-conorms from the FuzzyR package are also conceivable.
#' The possible values for the parameter `defuzz_method` include other defuzzification techniques from the FuzzyR package.
#' The parameter `default_conseq` defines the default behavior of the FSI model when there is no fuzzy rule with a degree of fulfillment greater than 0 returned by the FSI model.
#' 
#' After creating an empty FSI model, you have to call the functions `fsi_add_fsa`, `fsi_add_cs`, and `fsi_add_rules` to fulfill the FSI model.
#' 
#' @return
#'
#' An empty named FSI model that is ready to be populated with fuzzy rules representing the antecedents and the consequent.
#'
#' @examples
#'
#' library(FuzzyR)
#' # Creating the FSI model
#' fsi <- fsi_create("To visit or not to visit, that is the question", 
#'                   default_conseq = genmf("trimf", c(10, 30, 60))) 
#'
#' @export
fsi_create <- function(name, and_method = "min",
                    or_method = "max", imp_method = "min", agg_method = "max",
                    defuzz_method = "centroid", default_conseq = NULL) {
  fsi <- list(name = name, type = "mamdani",
              and_method = and_method, or_method = or_method, imp_method = imp_method,
              agg_method = agg_method, defuzz_method = defuzz_method,
              fsa = NULL, cs = NULL, rule = NULL, default_conseq = default_conseq)
  fsi
}



#' @title Adding an antecedent to an FSI model
#' 
#' @description This function adds a fuzzy spatial antecedent to a fuzzy spatial inference (FSI) model. 
#' A fuzzy spatial antecedent corresponds to a layer of fuzzy spatial objects that describe the different characteristics of the problem.
#' The antecedent has a linguistic variable and its fuzzy spatial objects have linguistic values so that they are used in the IF part of fuzzy rules.
#' 
#' @usage 
#' 
#' fsi_add_fsa(fsi, lvar, tbl)
#' 
#' @param fsi The FSI model instantiated with the `fsi_create` function.
#' @param lvar A character value that represents a linguistic variable of the antecedent.
#' @param tbl  A tibble with spatial plateau objects annotated with linguistic values of the linguistic variable specified by the above `lvar` parameter.
#' 
#' @details 
#' 
#' The fuzzy spatial antecedent added by the `fsi_add_fsa` function is composed of a linguistic variable and its corresponding `pgeometry` objects annotated by linguistic values. 
#' The format of the `tbl` parameter is the same as the output of the function `spa_creator`, allowing the user to directly provides plateau region objects as input when designing FSI models.
#' 
#' @returns 
#' 
#' An FSI model populated with a fuzzy spatial antecedent.
#' 
#' @examples
#' 
#' library(FuzzyR)
#' library(tibble)
#' 
#' # Create spatial plateau objects for the linguistic variable accomodation_price
#' lvals_accom_price <- c("cut-rate", "affordable", "expensive")
#' cut_rate_mf <- genmf("trapmf", c(0, 0, 10, 48))
#' affordable_mf <- genmf("trapmf", c(10, 48, 80, 115))
#' expensive_mf <- genmf("trapmf", c(80, 115, 10000, 10000))
#' 
#' # Example of dataset
#' accom_price <- tibble(
#'                       `longitude` = c(-74.0, -74.0, -74.0), 
#'                       `latitude` = c(40.8, 40.7, 40.7),
#'                       `price` = c(150, 76, 60)
#')
#'  
#' accom_price_layer <- spa_creator(accom_price, classes = lvals_accom_price, 
#'                          mfs = c(cut_rate_mf, affordable_mf, expensive_mf))
#'                          
#' # Create the fsi_model:
#' fsi <- fsi_create("To visit or not to visit, that is the question", 
#'                   default_conseq = genmf("trimf", c(10, 30, 60)))
#' 
#' # Add the fuzzy spatial antecedent to the fsi_model:
#' fsi <- fsi_add_fsa(fsi, "accommodation price", accom_price_layer)
#' 
#' @export
fsi_add_fsa <- function(fsi, lvar, tbl) {
  if(nrow(tbl) <= 0) {
    stop("The tibble with spatial plateau objects should contain at least one line", call. = FALSE)
  }

  # TODO validate the format of tbl
  pgeometry_layer <- tbl[c(1, 2)]
  colnames(pgeometry_layer) <- c("lval", "pgeometry")

  fsi$fsa <- append(fsi$fsa, list(list(name = lvar, layer = pgeometry_layer)))

  fsi
}



#' @title Adding the consequent to an FSI model
#' 
#' @description This function adds the consequent to a fuzzy spatial inference (FSI) model. It consists of a set of membership functions labeled with linguistic values.
#' 
#' @usage 
#' 
#' fsi_add_cs(fsi, lvar, lvals, mfs, bounds)
#' 
#' @param fsi The FSI model instantiated with the `fsi_create` function.
#' @param lvar A character value that represents a linguistic variable of the consequent.
#' @param lvals A character vector that represents linguistic values of the linguistic variable of the consequent.
#' @param mfs A vector of functions created by the `genmf` of the FuzzyR package.
#' @param bounds A numeric vector that represents the lower and upper bounds of the consequent domain. 
#' 
#' @details 
#' 
#' Each linguistic value defined at the `lvals` parameter has a membership function defined at the `mfs` parameter.
#' `lvals` is a character vector containing the names of linguistic values and `mfs` is vector containing its corresponding membership functions.
#' Thus, the vectors defined for these two parameters must have the same length.
#' For instance, the first value of `lvals` is the linguistic value for the first membership function in `mfs`.
#' In `bounds`, the lower and upper values correspond to the first and second parameter, respectively.
#' 
#' @returns 
#' 
#' An FSI model populated with a consequent.
#' 
#' @examples 
#' 
#' library(FuzzyR)
#' 
#' # Create the fsi_model:
#' fsi <- fsi_create("To visit or not to visit, that is the question", 
#'                    default_conseq = genmf("trimf", c(10, 30, 60)))
#' 
#' # Create the vector with the linguistic values of the linguistic variable "visiting experience":
#' lvals_visiting_exp <- c("awful", "average", "great")
#' 
#' # Define the membership function for each linguistic value:
#' awful_mf <- genmf("trimf", c(0, 0, 20))
#' average_mf <- genmf("trimf", c(10, 30, 60))
#' great_mf <- genmf("trapmf", c(40, 80, 100, 100))
#' 
#' # Add the consequent to the FSI model:
#' fsi <- fsi_add_cs(fsi, "visiting experience", lvals_visiting_exp,
#'                   c(awful_mf, average_mf, great_mf), c(0, 100))
#' 
#' @export
fsi_add_cs <- function(fsi, lvar, lvals, mfs, bounds) {
  if(length(lvals) != length(mfs)) {
    stop("The length of the linguistic values (lvals) and membership functions (mfs) should be equal", call. = FALSE)
  }
  fsi$cs <- append(fsi$cs, list(list(name = lvar, lvals = lvals, range = bounds, mfs = mfs)))

  fsi
}

#' Get each part of the antecedent from a fuzzy rule specified by the user.
#'
#' This function extracts the linguistic variable and its value from an input rule.
#' The rule must be in the following pattern:
#' IF linguistic variable is linguistic value LOGICAL OPERATOR linguistic variable is linguistic value THEN linguistic variable is linguistic value
#' Example:
#' IF hotel is affordable AND attraction is free THEN visiting is accessible
#' Pay attention that there is no punctuation in the rule.
#' The rule can use only one type of logical operator at a time (e.g. parts of the antecedents connected either by AND or by OR).
#' A rule can have one or more parts of antecedents.
#'
#' @param user_rule Fuzzy rule specified by the user
#' @param logical_op_ant Logical operator of the antecedent. Default value is NULL (there is only one antecedent)
#' @return A list of linguistic variables and their values from the antecedent of the rule
#' 
#' @import stringr
#' @noRd
get_antecedents <- function(user_rule) {
  us_rule <- str_to_lower(user_rule)
  us_rule <- str_replace_all(us_rule, "[\r\n]" , " ")
  antecedents <- str_extract(us_rule, "(?<=if )(.*\n?)(?=then)")

  if (str_count(antecedents, pattern = " and ") > 0) {
    lant <- "AND"

    qtd_and <- str_count(antecedents, pattern = " and ") + 1
    ant <- vector("list", length = qtd_and)
    tmpres <- str_split_fixed(antecedents, " and ", qtd_and)

    for (i in 1:qtd_and) {
      tmp_res <- str_split_fixed(tmpres[i], " is ", 2)
      a1 <- c(tmp_res[1][1], tmp_res[2][1])
      ant[[i]] <- str_trim(a1)
    }
  } else if (str_count(antecedents, pattern = " or ") > 0) {
    lant <- "OR"
    qtd_or <- str_count(antecedents, pattern = " or ") + 1
    ant <- vector("list", length = qtd_or)
    tmpres <- str_split_fixed(antecedents, " or ", qtd_or)

    for (i in 1:qtd_or) {
      tmp_res <- str_split_fixed(tmpres[i], " is ", 2)
      a1 <- c(tmp_res[1][1], tmp_res[2][1])
      ant[[i]] <- str_trim(a1)
    }
  } else {
    lant <- "OR"
    ant <- vector("list", length = 1)
    tmp_res <- str_split_fixed(antecedents, " is ", 2)
    a1 <- c(tmp_res[1][1], tmp_res[2][1])
    ant[[1]] <- str_trim(a1)
  }
  list(ants=ant, op=lant)
}

#' @import stringr
#' @noRd
get_consequent <- function(user_rule) {
  us_rule <- str_to_lower(user_rule)
  us_rule <- str_replace_all(us_rule, "[\r\n]" , " ")
  consequent <- str_extract(us_rule, "(?<=then )(.*\n?)")
  conseq <- vector("list", length = 1)
  tmp_res <- str_split_fixed(consequent, " is ", 2)
  cons1 <- c(tmp_res[1][1], tmp_res[2][1])
  conseq[[1]] <- str_trim(cons1)
  return(conseq)
}

#' @title Adding fuzzy rules to an FSI model
#' 
#' @description This function adds the fuzzy rules set to a fuzzy spatial inference (FSI) model. A fuzzy rule must contain only linguistic variables and values employed by the added antecedent parts and consequent.
#' 
#' @usage 
#' 
#' fsi_add_rules(fsi, rules, weights = rep(1, length(rules)))
#' 
#' @param fsi An FSI model instantiated with the function `fsi_create`.
#' @param rules A character vector containing the rules defined by the user. It follows a specific format, as detailed below.
#' @param weights A numeric vector of weight values for each rule. Default values are 1.
#' 
#' @details 
#' 
#' The definition of a fuzzy rule is user-friendly since users can write it by using the _linguistic variables_ and _linguistic values_ previously defined and added to the FSI model. 
#'  A fuzzy rule has the format `IF A THEN B`, where `A` is called the antecedent and `B` the consequent of the rule such that `A` implies `B`.  Further, `A` and `B` are statements that combine fuzzy propositions by using logical connectives like `AND` or `OR`. Each fuzzy proposition has the format `LVar is LVal` where `LVal` is a linguistic value in the scope of the linguistic variable `LVar`.
#' To avoid possible contradictions keep in mind the following items when specifying the rules:
#'  - the order of the statements in the antecedent is not relevant;
#'  - each linguistic variable has to appear at most one time in each fuzzy rule;
#’ - only one kind of logical connective (i.e., `AND` or `OR`) must be used in the statements of the antecedent.
#' 
#' @return 
#' 
#' An FSI model populated with fuzzy rules set.
#' 
#' @examples 
#' 
#' # Creating the FSI model from an example implemented with the visitation function:
#' fsi <- visitation()
#'
#' # Creating a vector of fuzzy rules; 
#' ## note that we make use of the linguistic variables and linguistic values previously defined:
#' rules <- c(
#'   "IF accommodation review is reasonable AND food safety is low 
#'   THEN visiting experience is awful",
#'  "IF accommodation price is expensive AND accommodation review is reasonable 
#'    THEN visiting experience is awful",
#'  "IF accommodation price is affordable AND accommodation review is good AND food safety is medium 
#'    THEN visiting experience is average",
#'  "IF accommodation price is affordable AND accommodation review is excellent 
#'                                                                 AND food safety is high 
#'    THEN visiting experience is great",
#'  "IF accommodation price is cut-rate AND accommodation review is excellent AND food safety is high 
#'    THEN visiting experience is great")
#' 
#' # Adding these rules to the FSI model previously instantiated:
#' fsi <- fsi_add_rules(fsi, rules)
#' 
#' @export
fsi_add_rules <- function(fsi, rules, weights = rep(1, length(rules))) {
  if (length(rules) != length(weights)) {
    stop("The length of parameters for rules and weights have to be equal", call. = FALSE)
  }
  i <- 1
  for(ur in rules) {
    #TODO improve performance
    antecedents <- get_antecedents(ur)
    consequents <- get_consequent(ur)
    fsi$rule <- append(fsi$rule,
                       list(list(ant = antecedents$ants, cons = consequents, w = weights[i], connective = antecedents$op)))
    i <- i + 1
  }
  fsi
}

#' @title Evaluating an FSI model for a given point location
#' 
#' @description This function executes the reasoning process of a fuzzy spatial inference (FSI) model 
#' for a given point location (i.e., `sfg` object of the type `POINT`).
#' 
#' @usage 
#' 
#' fsi_eval(fsi, point, ...)
#' 
#' @param fsi An FSI model built with the function `fsi_create` and populated by the functions `fsi_add_fsa`, `fsi_add_cs`, and `fsi_add_rules`.
#' @param point An `sfg` object of geometry type `point`, which is created through the function `st_point` of the sf package.
#' @param ... <[`dynamic-dots`][rlang::dyn-dots]> Informs the `fsi_eval` how the elements of the resulting fuzzy set should be discretized if the user does not want the default configuration (see below). Default values: `discret_by` is 0.5 and `discret_length` is NULL.
#'
#' @details 
#' 
#' This function evaluates an FSI model populated with its fuzzy spatial antecedent, consequent, and fuzzy rules set on a specific point location. This evaluation is based on the algorithm specified by [FIFUS](https://ieeexplore.ieee.org/document/8015707). 
#'
#' The default behavior of the function `fsi_eval` in the parameter `...` is to consider a discrete interval of values with an increment of 0.5 between lower and upper values for the consequent domain (i.e., defined at `fsi_add_cs` function with the parameter `bounds`).
#' 
#' The user can modify the default behavior by using one of the following two ways:
#' - define a value for the parameter `discret_by` by changing the incremental value.
#' - define a desired length for the sequence of values domain of the consequent `discret_length`.
#’ This means that it has the same behavior as the function `seq` of the R base.
#' 
#' @return 
#' 
#' A numeric value that belongs to the domain of the consequent (i.e., as specified by `fsi_add_cs`) and represents the result of the reasoning process in a particular point location.
#' 
#' @references
#'
#' [Carniel, A. C.; Schneider, M. Fuzzy inference on fuzzy spatial objects (FIFUS) for spatial decision support systems. In Proceedings of the 2017 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2017), pp. 1-6, 2017.](https://ieeexplore.ieee.org/document/8015707)
#'
#' @examples 
#'
#' library(sf)
#' # Creating the FSI model from an example implemented with the visitation function:
#' fsi <- visitation()
#'
#' # Creating a vector of fuzzy rules; 
#' ## note that we make use of the linguistic variables and linguistic values previously defined:
#' rules <- c(
#'   "IF accommodation review is reasonable AND food safety is low 
#'   THEN visiting experience is awful",
#'  "IF accommodation price is expensive AND accommodation review is reasonable 
#'    THEN visiting experience is awful",
#'  "IF accommodation price is affordable AND accommodation review is good AND food safety is medium 
#'    THEN visiting experience is average",
#'  "IF accommodation price is affordable AND accommodation review is excellent 
#'                                                                 AND food safety is high 
#'    THEN visiting experience is great",
#'  "IF accommodation price is cut-rate AND accommodation review is excellent AND food safety is high 
#'    THEN visiting experience is great")
#' 
#' # Adding these rules to the FSI model previously instantiated:
#' fsi <- fsi_add_rules(fsi, rules)
#' 
#' # Using the default configuration:
#' res <- fsi_eval(fsi, st_point(c(-74.0, 40.7)))
#' 
#' # Change the default discretization by modifying the default step value:
#' res <- fsi_eval(fsi, st_point(c(-74.0, 40.7)), discret_by=0.8)
#' 
#' # Change the default discretization by choosing the quantity of values 
#' ## between the lower and upper values for the consequent domain:
#' res <- fsi_eval(fsi, st_point(c(-74.0, 40.7)), discret_length=200)
#' 
#' @import tibble FuzzyR
#' @export
fsi_eval <- function(fsi, point, ...) {
  discret_by <- 0.5
  discret_length <- NULL

  if(any(is.na(point))){
    stop("'point' should not be NA", call. = FALSE)
  }
  if(class(point)[[2]] != "POINT"){
    stop("'point' must be a simple point object", call. = FALSE)
  }

  args_function <- list(...)
  if(length(args_function) > 0){
    if(!is.null(args_function[["discret_by"]])){
      discret_by <- args_function[["discret_by"]]
    }
    if(!is.null(args_function[["discret_length"]])){
      discret_length <- args_function[["discret_length"]]
    }
  }

  # First step: store the degree to which point belongs to each part of the FSA
  tbl_eval <- tibble(lvar = character(), lval = character(), degree = numeric())
  for(input in fsi$fsa) {
    input_layer <- input$layer
    for(i in 1:nrow(input_layer)) {
      row <- input_layer[i, ]
      tbl_eval <- add_row(tbl_eval, lvar = input$name, lval = row$lval, degree = spa_eval(row$pgeometry[[1]], point))
    }
  }

  fire_rules <- tibble(rule_index = integer(), degree = numeric(), consequent = list())

  # Second step: for each rule, we compute its firing rule strength
  i <- 1
  for(rule in fsi$rule) {
    degrees <- numeric()

    for(ant_part in rule$ant) {
      #the first position in ant_part is the lvar, the second position is its lval
      degrees <- append(degrees,
                        tbl_eval[tbl_eval$lvar == ant_part[1] & tbl_eval$lval == ant_part[2], ]$degree)
    }
    
    if(length(degrees) == 0) {
      degrees <- 0  
    }

    if(rule$connective == "AND") {
      connective_method <- match.fun(fsi$and_method)
    } else {
      connective_method <- match.fun(fsi$or_method)
    }
    fire_rules <- add_row(fire_rules, rule_index = i, 
                          degree = (connective_method(degrees) * rule$w), 
                          consequent = rule$cons)
    i <- i + 1
  }
  # Third step: compute the implication of each fired rule
  imp_method <- match.fun(fsi$imp_method)

  results_imp <- list()

  min_conseq <- fsi$cs[[1]]$range[1]
  max_conseq <- fsi$cs[[1]]$range[2]

  for(j in 1:nrow(fire_rules)) {
    row <- fire_rules[j, ]

    if(row$degree > 0) {
      consequent <- row$consequent[[1]]
      mf_pos <- match(consequent[2], fsi$cs[[1]]$lvals)

      if(!is.na(mf_pos) && mf_pos >= 1) {
        mf_conseq <- fsi$cs[[1]]$mfs[[mf_pos]]

        #implication here
        # TODO improve a little bit, we can discretize the values here too, instead of calling the tnorm
        mf_cut <- genmf("trapmf", c(min_conseq, min_conseq, max_conseq, max_conseq, row$degree))
        res_imp <- fuzzy.tnorm(imp_method, mf_conseq, mf_cut)
        results_imp <- append(results_imp, res_imp)
      }
    }
  }

  # Fourth step: compute the aggregation

  conseq_values <- NULL
  if(!is.null(discret_by)) {
    conseq_values <- seq(min_conseq, max_conseq, by = discret_by)
  } else if(!is.null(discret_length)) {
    conseq_values <- seq(min_conseq, max_conseq, length.out = discret_length)
  } else {
    conseq_values <- seq(min_conseq, max_conseq)
  }

  agg_method <- NULL
  if(fsi$agg_method == "max") {
    agg_method <- match.fun("pmax")
  } else {
    agg_method <- match.fun(fsi$agg_method)
  }

  if(length(results_imp) == 0) {
    db <- fsi$default_conseq
    if(is.null(db)) {
      warning("No default rule defined")
      return(NA)
    }
    result_fsi <- evalmf(conseq_values, db)
  } else {
    result_fsi <- evalmf(conseq_values, results_imp[[1]])
    if(length(results_imp) >= 2) {
      for(i in 2:length(results_imp)) {
        result_fsi <- agg_method(evalmf(conseq_values, results_imp[[i]]), result_fsi)
      }
      ## TODO improve this evaluation in terms of performance
    }
  }

  defuzz(conseq_values, result_fsi, fsi$defuzz_method)
}

#' @import sf tibble
#' @noRd
fsi_qwi_discretization <- function(fsi, qw, k, n_col = NULL, n_row = NULL, ...) {
  if(!(is.null(n_col) && is.null(n_row))) {
    regular_grid_points <- st_make_grid(qw, n = c(n_row, n_col), what = "centers")
  } else {
    n <- as.integer(sqrt(k))
    regular_grid_points <- st_make_grid(qw, n = c(n, n), what = "centers")
  }
  qw_inference_grid_output <- numeric(length = length(regular_grid_points))

  i <- 1
  for(point in regular_grid_points) {
    qw_inference_grid_output[i] <- fsi_eval(fsi, point)
    i <- i + 1
  }

  tibble(points = regular_grid_points, inferred_values = qw_inference_grid_output)
}

#' @import sf pso tibble
#' @noRd
fsi_qwi_pso <- function(fsi, qw, target_mf, max_depth = 2, maxit = 50, population = 10, convergence = Inf,
                        what = "max", stats_output = FALSE) {
  pso.env <- new.env()
  pso.env$count_fitness_function <- 0
  pso.env$count_iteration <- 0

  fitness <- function(p){
    x <- p[1]
    y <- p[2]
    fsi_eval(fsi, st_point(c(x, y)))
  }

  fsi_quadrants_pso <- function(fsi, quadrant, maxit, population, convergence, target_mf,
                                current_depth, max_depth, result, what = "max") {
    if(max_depth == current_depth) {
      return(result)
    }

    subquadrants <- st_make_grid(quadrant, n = c(2, 2))

    if(what == "max") {
      scale = -1
    } else {
      scale = 1
    }

    for(q in subquadrants) {
      bbox <- st_bbox(q)

      pso_result <- psoptim(rep(NA,2), fn = fitness, lower = c(bbox$xmin, bbox$ymin), upper = c(bbox$xmax, bbox$ymax),
                            control = list(maxit = maxit, s = population, fnscale = scale, abstol = convergence * scale))
      pso.env$count_fitness_function <- pso.env$count_fitness_function + as.numeric(pso_result$counts[1])
      pso.env$count_iteration <- pso.env$count_iteration + as.numeric(pso_result$counts[2])

      if(target_mf(pso_result$value * scale) > 0) {
        result <- add_row(result, points = list(st_point(pso_result$par)), inferred_values = pso_result$value * scale)

        result <- fsi_quadrants_pso(fsi, q, maxit, population, convergence, target_mf,
                                    current_depth + 1, max_depth, result, what)
      }
    }
    result
  }

  pso_result_tbl <- tibble(points = list(), inferred_values = numeric())

  pso_result_tbl <- fsi_quadrants_pso(fsi, qw, maxit, population, convergence, target_mf,
                                      0, max_depth, pso_result_tbl, what = "max")
  pso_result_tbl$points <- st_as_sfc(pso_result_tbl$points)

  if(stats_output) {
    list(iterations = pso.env$count_iteration, fitness_calls = pso.env$count_fitness_function, result = pso_result_tbl)
  } else {
    pso_result_tbl
  }
}



#' @title Evaluating a query window inference
#' 
#' @description This function implements two approaches for evaluating the query window inference on a fuzzy spatial inference (FSI) model.
#' Given a query window (i.e., a rectangular object), it returns a set of inferred points inside this window
#' that satisfy a specific condition (e.g., target linguistic value, or maximum/minimum inferred values).
#' 
#' @usage
#'
#' fsi_qw_eval(fsi, qw, approach = "discretization", ...) 
#'
#' @param fsi An FSI model built with the `fsi_create` function that is populated by the following functions `fsi_add_fsa`, `fsi_add_cs`, and `fsi_add_rules`.
#' @param qw An `sfg` object storing the query window that is supposed to be used as input for the inference. It has to be an axis-aligned rectangle represented by a simple polygon object of 5 points (since the last coordinate pair closes the external ring of the rectangle).
#' @param approach Defines which approach is employed to perform the query window inference: `“discretization”` or `“pso”`. Default value is `"discretization"``
#' @param ... <[`dynamic-dots`][rlang::dyn-dots]> Different set of parameters required depending on the chosen approach (see more in details below).
#'
#' @details 
#'
#' For the _discretization_ approach, two additional parameters are needed and they have to be informed by using the three-dots parameter `...`: 
#' - `target_lval`: A character value that indicates the target linguistic value from the linguistic variable of the consequent.
#' - `k`: A numeric value that defines the number of points that will be captured from the query window and evaluated by the function `fsi_eval`. Its square root has to an integer value.   Alternatively, you can inform the number of columns and rows of the regular grid to be performed on the query window by informing numeric values for `n_col` and `n_row`, respectively. Thus, these parameters can be given instead of the number `k`.
#' 
#' For the _pso_ approach, it is necessary to set the following parameters:
#' - `what`: A character value that defines the user's goal, which can be either **maximize** or **minimize** inferred values. Thus, this parameter can be `“max”` and `“min”`, respectively. The default value is `“max”`.
#' - `max_depth`: A numeric value that refers to the number of times the user wants to split the query window. The default value is equal to 2. For instance, a `max_depth` = 2 results in the query window split into four sub quadrants, where the particle swarm optimization (PSO) algorithm will be applied to each one as its search space. 
#' In addition, the PSO algorithm has its own set of parameters:
#' - `maxit`: A numeric value that defines the maximum number of iterations. Default value is 50.
#' - `population`: A numeric value that defines the number of particles. Default value is 10.
#'
#' @return
#'
#' A tibble in the format `(points, inferred_values)`, where `points` is an `sfc` object (i.e., a list of `sfg` objects of geometry type POINT) and `inferred_values` are inferred values in the domain of the consequent of the FSI model.
#'
#' @examples 
#'
#' library(sf)
#' # Creating the FSI model from an example implemented with the visitation function:
#' fsi <- visitation()
#'
#' # Creating a vector of fuzzy rules; 
#' ## note that we make use of the linguistic variables and linguistic values previously defined:
#' rules <- c(
#'   "IF accommodation review is reasonable AND food safety is low 
#'   THEN visiting experience is awful",
#'  "IF accommodation price is expensive AND accommodation review is reasonable 
#'    THEN visiting experience is awful",
#'  "IF accommodation price is affordable AND accommodation review is good AND food safety is medium 
#'    THEN visiting experience is average",
#'  "IF accommodation price is affordable AND accommodation review is excellent 
#'                                                                 AND food safety is high 
#'    THEN visiting experience is great",
#'  "IF accommodation price is cut-rate AND accommodation review is excellent AND food safety is high 
#'    THEN visiting experience is great")
#' 
#' # Adding these rules to the FSI model previously instantiated:
#' fsi <- fsi_add_rules(fsi, rules)
#' 
#' # Defining the query window that is defined over an application domain
#' pts_qw1 <- rbind(c(-73.92, 40.68527), c(-73.75, 40.68527), 
#'                  c(-73.75, 40.75), c(-73.92, 40.75), c(-73.92, 40.68527))
#' qw1 <- st_polygon(list(pts_qw1))
#' 
#' # Example using the discretization approach:
#' dis_res <- fsi_qw_eval(fsi, qw1, approach = "discretization", target_lval = "great", k = 25)
#'
#' ## Example using the pso approach in two levels:
#' \dontrun{
#' pso_res <- fsi_qw_eval(fsi, qw1, approach = "pso", max_depth = 2)
#' }
#' 
#' @import utils dplyr
#' @importFrom rlang .data
#' @export
fsi_qw_eval <- function(fsi, qw, approach = "discretization", ...) {
  params <- list(...)
  result_qwi <- switch(approach,
                       discretization = {
                         qwi_discret <- do.call(fsi_qwi_discretization, c(list(fsi, qw), params))
                         target_lval <- params$target_lval
                         target_mf <- NULL
                         mf_pos <- match(target_lval, fsi$cs[[1]]$lvals)
                         if(!is.na(mf_pos) && mf_pos >= 1) {
                           target_mf <- fsi$cs[[1]]$mfs[[mf_pos]]
                         } else {
                           stop("Invalid target linguistic value.", call. = FALSE)
                         }
                         qwi_discret %>% filter(target_mf(.data$inferred_values) > 0)
                       },
                       pso = {
                         target <- params$what
                         target_mf <- NULL
                         if(is.null(target) || target == "max"){
                           target_mf <- tail(fsi$cs[[1]]$mfs, n=1)[[1]]
                         } else if (target == "min") {
                           target_mf <- head(fsi$cs[[1]]$mfs, n=1)[[1]]
                         } else {
                           stop("Invalid value for the what parameter.", call. = FALSE)
                         }
                         do.call(fsi_qwi_pso, c(list(fsi, qw, target_mf), params))
                       },
                       stop("This query window inference approach is not valid.", call. = FALSE)
  )

  result_qwi
}


#' @title Visitation: An example of FSI model
#' 
#' @description This function provides an example, without rules, of a fuzzy spatial inference (FSI) model.
#' 
#' @usage 
#' visitation()
#'
#' @details 
#' 
#' The FSI model provided by this function represents an FSI model to estimate the visiting experience based on prices and overall ratings of accommodations as well as sanitary conditions of restaurants. The output of such a model infers a value between 0 and 100 that indicates how attractive it is to visit a specific location. For this, the experience can be classified as _awful_, _average_, and _great_. 
#' 
#' The linguistic variables and their linguistic values of this FSI model are listed below:
#' - _accommodation price_, with  _cut-rate_, _affordable_, and _expensive_ as linguistic values;
#' - _accommodation review_ with _bad_, _good_, and _excellent_ as linguistic values;
#' - _food safety_ with _low_, _medium_, and _high_ as linguistic values, which represent levels of sanitary conditions.
#' 
#' The usage of FSI models is subdivided into a _preparation phase_ and an _evaluation phase_. The preparation phase is responsible for instantiating a new FSI model with the elements of the data source component of FIFUS. For this, the fsr package provides the following functions: `fsi_create`, `fsi_add_fsa`, and `fsi_add_cs`. These functions are employed by `visitation` so that users can add their own fuzzy set rules (by using `fsi_add_rules`) and perform the evaluation phase (by using the functions `fsi_eval` and/or `fsi_qw_eval`).
#' In this sense, `visitation` performs the following internal actions to return an FSI model:
#' 1. specify the linguistic variables and their corresponding linguistic values, which are in turn represented by membership functions generated by the function `genmf` of the FuzzyR package. These items are specified according to the context of the running example.
#' 2. define small point datasets that represent each linguistic variable. Such datasets are `tibble` objects.
#' 3. build spatial plateau objects by using the function `spa_creator` on the datasets. As a result, we get spatial plateau objects that represent each linguistic value.
#' 4. create an FSI model with `fsi_create` function.
#' 5. add fuzzy spatial antecedents with the `fsi_add_fsa` function. Recall that the antecedents are spatial plateau objects previously built.
#' 6. define the linguistic variable and its linguistic values with membership functions for the consequent.
#' 7. add the consequent to the FSI model by using the function `fsi_add_cs`.
#'
#' @return
#' 
#' An example of an FSI model implemented without fuzzy rules set.
#'
#' @examples 
#'
#' fsi <- visitation()
#'
#' @import FuzzyR sf tibble
#' @export
visitation <- function() {
  # Define linguistic values and membership functions for each linguistic variable
  lvals_accom_price <- c("cut-rate", "affordable", "expensive")
  cut_rate_mf <- genmf("trapmf", c(0, 0, 10, 48))
  affordable_mf <- genmf("trapmf", c(10, 48, 80, 115))
  expensive_mf <- genmf("trapmf", c(80, 115, 10000, 10000))
  
  lvals_accom_review <- c("cut-rate", "affordable", "expensive")
  reasonable_mf <- genmf("trapmf", c(0, 0, 40, 65))
  good_mf <- genmf("trapmf", c(40, 65, 80, 85))
  excellent_mf <- genmf("trapmf", c(80, 85, 100, 100))
  
  lvals_food_safety <- c("cut-rate", "affordable", "expensive")
  low_mf <- genmf("trapmf", c(24, 28, 115, 115))
  medium_mf <- genmf("trapmf", c(10, 14, 24, 28))
  high_mf <- genmf("trapmf", c(0, 0, 10, 14))
  
  # Example of datasets
  accom_price <- tibble(
    `longitude` = c(-74.0, -74.0, -74.0, -74.0, -74.0, -74.0, -74.0, -74.0, -73.9, -74.0), 
    `latitude` = c(40.8, 40.7, 40.7, 40.7, 40.8, 40.8, 40.7, 40.7, 40.8, 40.7),
    `price` = c(150, 76, 60, 175, 79, 75, 92, 109, 62, 196)
  )
  
  accom_review <- tibble(
    `longitude` = c(-74.0, -74.0, -74.0, -74.0, -74.0, -74.0, -74.0, -74.0, -74.0, -74.0), 
    `latitude` = c(40.8, 40.7, 40.7, 40.7, 40.8, 40.8, 40.7, 40.7, 40.8, 40.7),
    `review_scores_rating` = c(94, 89, 90, 97, 84, 98, 94, 91, 98, 97)
  )
  
  food_safety <- tibble(
    `longitude` = c( -74.01291, -73.99912, -73.84201, -73.98959, -73.84348, -73.98704, -73.93084, -73.91315, -73.94999, -73.96237), 
    `latitude` = c(40.70626, 40.72316, 40.75804, 40.76259, 40.84978, 40.75079, 40.67171, 40.70068, 40.77965, 40.77600),
    `score` = c(4, 9, 7, 9, 9, 24, 11, 11, 11, 13)
  )
  
  # Building the spatial plateau objects by using the previous datasets
  accom_price_layer <- spa_creator(accom_price, classes = lvals_accom_price, 
                                   mfs = c(cut_rate_mf, affordable_mf, expensive_mf))
  accom_review_layer <- spa_creator(accom_review, classes = lvals_accom_review, 
                                    mfs = c(reasonable_mf, good_mf, excellent_mf))
  food_safety_layer <- spa_creator(food_safety, classes = lvals_food_safety, 
                                   mfs = c(low_mf, medium_mf, high_mf))
  
  # Creating the FSI model with a name and a default consequent:
  fsi <- fsi_create("To visit or not to visit, that is the question", default_conseq = genmf("trimf", c(10, 30, 60)))
  
  # Adding fuzzy spatial antecedents:
  fsi <- fsi_add_fsa(fsi, "accommodation price", accom_price_layer)
  fsi <- fsi_add_fsa(fsi, "accommodation review", accom_review_layer)
  fsi <- fsi_add_fsa(fsi, "food safety", food_safety_layer)
  
  # Defining the linguistic values for the linguistic variable `"visiting experience"`
  lvals_visiting_exp <- c("awful", "average", "great")
  
  # Specifying the membership functions for the linguistic values created on the previous step:
  awful_mf <- genmf("trimf", c(0, 0, 20))
  average_mf <- genmf("trimf", c(10, 30, 60))
  great_mf <- genmf("trapmf", c(40, 80, 100, 100))
  
  # Adding the consequent:
  fsi <- fsi_add_cs(fsi, "visiting experience", lvals_visiting_exp, c(awful_mf, average_mf, great_mf), c(0, 100))
  
  fsi
}