\name{frscore}
\alias{frscore}
\title{frscore}
\usage{
frscore(
  sols,
  scoretype = c("full", "supermodel", "submodel"),
  normalize = c("truemax", "idealmax", "none"),
  maxsols = 50,
  verbose = FALSE,
  print.all = FALSE
)
}
\arguments{
\item{sols}{Character vector of class "stdAtomic" or "stdComplex" (as generated by \code{\link[cna:cna]{cna()}}) that contains the solutions/models to be scored.}

\item{scoretype}{Character vector specifying the scoring method: \code{"full"}
(default; scoring is based on counting both sub- and supermodels), \code{"supermodel"}
(count supermodels only), \code{"submodel"} (count submodels only).}

\item{normalize}{Character vector that determines the method used in
normalizing the scores. \code{"truemax"} (default) normalizes by the highest score among the elements of \code{sols},
such that the highest scoring solution types get score 1. \code{"idealmax"}
normalizes by a theoretical maximum score (see Details). }

\item{maxsols}{Integer determining the maximum number of unique solution
types found in \code{sols} to be included in the scoring (see Details).}

\item{verbose}{Logical; if \code{TRUE}, additional
information about submodel relations among the unique solution types found
in \code{sols} are calculated and printed. Defaults to \code{FALSE}, which makes execution slightly
faster.}

\item{print.all}{Logical, controls the number of entries printed when
printing the results. If \code{TRUE}, results are printed as when using the defaults of
\code{print.data.frame}. If \code{FALSE}, 20 highest scoring
solutions/models are printed.}
}
\value{
A named list where the first element is a data frame containing
the unique solution/model types and their scores. Rest of the elements
contain additional information about the submodel relations among
the unique solutions types (if \code{verbose = TRUE}) and about how
the function was called.
}
\description{
Calculate fit-robustness scores for a set of \code{cna}
solutions/models
}
\details{
\code{frscore()} implements fit-robustness scoring as introduced in Parkkinen and Baumgartner (2021). The function calculates the
fit-robustness scores of Boolean solutions/models output by the \code{\link[cna:cna]{cna()}} function of the \pkg{cna} package. The solutions are given to \code{frscore()} as a character vector \code{sols} obtained by reanalyzing
a data set repeatedly, e.g. with \code{rean_cna()}, using different consistency and coverage
thresholds in each analysis.

When the argument \code{scoretype} is set to its default value \code{"full"}, the score of each \code{sols[i]} is calculated by counting both the sub- and supermodel relations \code{sols[i]} has to the other elements of \code{sols}. Setting \code{scoretype} to \code{"supermodel"} or \code{"submodel"} forces the scoring to be based on, respectively, supermodel and submodel relations only. In the former setting, less complex models will tend to get higher fit-robustness scores because they tend to have more supermodels in \code{sols}, while the latter setting gives preference to more complex models, which have more submodels in \code{sols} on average.

The fit-robustness scores can be normalized in two ways. In the default setting \code{normalize = "truemax"}, the score of each \code{sols[i]} is divided by the maximum score obtained by an element of \code{sols}. In case of \code{normalize = "idealmax"}, the score is normalized not by an actually obtained maximum but by an idealized maximum, which is calculated by assuming that all solutions of equal complexity in \code{sols} are identical and that for every \code{sols[i]} of a given complexity, all less complex elements of \code{sols} are its
submodels and all more complex elements of \code{sols} are its supermodels.

If the size of the consistency and coverage interval scanned in the reanalysis series generating \code{sols} is large or there are many model ambiguities, \code{sols} may contain so many different types of solutions/models that robustness cannot be calculated for all of them in reasonable time. In that case, the argument \code{maxsols} allows for capping the number of solution types to be included in the scoring (defaults to 50). \code{frscore()} then selects the most frequent solutions/models in \code{sols} of each complexity level until  \code{maxsols} is reached and only scores the thus selected elements of \code{sols}.

If the argument \code{verbose} is set to \code{TRUE}, \code{frscore()} not only returns an ordered score table but also a list indicating for each \code{sols[i]} how many raw score points it receives from which elements of \code{sols}. The verbose list is ordered with decreasing fit robustness scores.

}
\references{
V.P. Parkkinen and M. Baumgartner (2021), \dQuote{Robustness and Model Selection in Configurational Causal Modeling,} \emph{Sociological Methods and Research}, doi:10.1177/0049124120986200.

Basurto, Xavier. 2013. \dQuote{Linking Multi-Level Governance to Local Common-Pool
Resource Theory using Fuzzy-Set Qualitative Comparative Analysis: Insights from
Twenty Years of Biodiversity Conservation in Costa Rica.} \emph{Global Environmental Change} \strong{23} (3):573-87.

}
\seealso{\code{\link{rean_cna}}, \code{\link[cna:cna]{cna}}}
\examples{
# Artificial data from Parkkinen and Baumgartner (2021)
sols1 <- rean_cna(d.error, attempt = seq(1, 0.8, -0.1))
sols1 <- do.call(rbind, sols1)
frscore(sols1$condition)

# Real fuzzy-set data from Basurto (2013)
\donttest{
sols2 <- rean_cna(d.autonomy, type="fs", ordering = list("EM", "SP"),
         strict = TRUE, maxstep = c(3,3,9))
sols2 <- do.call(rbind, sols2)$condition  # there are 217 solutions
# At the default maxsols only 50 of those solutions are scored.
frscore(sols2)
# By increasing maxsols the number of solutions to be scored can be controlled.
frscore(sols2, maxsols = 100)

# Changing the normalization
frscore(sols2, normalize = "none")
frscore(sols2, normalize = "truemax")
frscore(sols2, normalize = "idealmax")

# Changing the scoring
frscore(sols2, scoretype = "supermodel")
frscore(sols2, scoretype = "submodel")

frscore(sols2, maxsols = 20, verbose = TRUE)
}

}
