#' desvio UI Function
#'
#' @description A shiny Module.
#'
#' @param id,input,output,session Internal parameters for {shiny}.
#'
#' @noRd 
#'
#' @importFrom shiny NS tagList 
#' @importFrom forecast forecast rwf
mod_desvio_ui <- function(id){
  ns <- NS(id)
  
  opc_drift <- div(
    conditionalPanel(
      condition = "input.BoxDrift == 'tabText' | input.BoxDrift == 'tabPlot'", ns = ns,
      tabsOptions(list(icon("gear")), 100, 70, tabs.content = list(
        list(
          conditionalPanel(
            condition = "input.BoxDrift == 'tabText'", ns = ns,
            options.run(ns("run_drift")), tags$hr(style = "margin-top: 0px;"),
          ),
          conditionalPanel(
            condition = "input.BoxDrift == 'tabPlot'", ns = ns,
            options.run(NULL), tags$hr(style = "margin-top: 0px;"),
            fluidRow(
              col_4(
                colourpicker::colourInput(
                  ns("col_train_drift"), labelInput("coltrain"), "#5470c6", 
                  allowTransparent = T)
              ),
              col_4(
                colourpicker::colourInput(
                  ns("col_test_drift"), labelInput("coltest"), "#91cc75", 
                  allowTransparent = T)
              ),
              col_4(
                colourpicker::colourInput(
                  ns("col_p_drift"), labelInput("colpred"), "#fac858", 
                  allowTransparent = T)
              )
            )
          )
        )
      ))
    )
  )
  
  tagList(
    tabBoxPrmdt(
      id = ns("BoxDrift"), opciones = opc_drift, #title = titulo_disp,
      tabPanel(
        title = labelInput("text_m"), value = "tabText",
        div(style = "height: 70vh; overflow: scroll;", 
            withLoader(verbatimTextOutput(ns("text_drift")),
                       type = "html", loader = "loader4"))
      ),
      tabPanel(
        title = labelInput("table_m"), value = "tabTable",
        withLoader(DT::dataTableOutput(ns('table_drift'), height = "70vh"),
                   type = "html", loader = "loader4")
      ),
      tabPanel(
        title = labelInput("plot_m"), value = "tabPlot",
        echarts4rOutput(ns("plot_drift"), height = "70vh")
      ),
      tabPanel(
        title = labelInput("error_m"), value = "tabError",
        uiOutput(ns("error_drift"))
      )
    )
  )
}
    
#' desvio Server Function
#'
#' @noRd 
mod_desvio_server <- function(input, output, session, updateData, rvmodelo) {
  ns <- session$ns
  
  observeEvent(input$BoxDrift, {
    if(input$BoxDrift == "tabText") {
      shinyjs::show('run_drift')
    } else {
      shinyjs::hide('run_drift')
    }
  })
  
  observeEvent(c(updateData$train, updateData$test), {
    updateTabsetPanel(session, "BoxDrift", selected = "tabText")
  })
  
  output$text_drift <- renderPrint({
    input$run_drift
    train <- updateData$train
    test  <- updateData$test
    
    tryCatch({
      modelo <- rwf(train, h = length(test), drift = T)
      pred   <- modelo$mean
      rvmodelo$drif$model <- modelo
      rvmodelo$drif$pred  <- pred
      rvmodelo$drif$error <- tabla.errores(list(pred), test, c("drif"))
      
      cod <- paste0(
        "modelo.drif <- rwf(train, h = length(test), drift = T)\n", 
        "pred.drif   <- modelo.drif$mean\n",
        "error.drif  <- tabla.errores(list(pred.drif), test, 'Drift')")
      isolate(updateData$code[['drif']] <- list(docdrifm = cod))
      
      modelo
    }, error = function(e) {
      showNotification(paste0("ERROR 0000: ", e), type = "error")
      return(NULL)
    })
  })
  
  output$table_drift <- DT::renderDataTable({
    lg <- updateData$idioma
    test <- isolate(updateData$test)
    seriedf <- tail(isolate(updateData$seriedf), length(test))
    seriedf[[1]] <- format(seriedf[[1]], '%Y-%m-%d %H:%M:%S')
    
    tryCatch({
    res <- data.frame(seriedf[[1]], seriedf[[2]], rvmodelo$drif$pred, 
                      abs(seriedf[[2]] - rvmodelo$drif$pred))
    colnames(res) <- tr(c('date', 'Real', 'table_m', 'diff'), lg)
    res[, 2:4] <- round(res[, 2:4], 3)
    
    cod <- paste0(
      "s <- tail(seriedf, length(test))\n",
      "res <- data.frame(s[[1]], s[[2]], pred.drif, abs(s[[2]] - pred.drif))\n",
      "colnames(res) <- c('", paste(colnames(res), collapse = "','"), "')\n",
      "res[, 2:4] <- round(res[, 2:4], 3)\nres")
    isolate(updateData$code[['drif']][['docdrift']] <- cod)
    
    DT::datatable(res, selection = 'none', editable = F, rownames = F,
                  options = list(dom = 'frtp', scrollY = "50vh"))
    }, error = function(e) {
      showNotification(paste0("ERROR 0000: ", e), type = "error")
      return(NULL)
    })
  })
  
  output$plot_drift <- renderEcharts4r({
    train <- isolate(updateData$train)
    test  <- isolate(updateData$test)
    lg    <- updateData$idioma
    pred  <- rvmodelo$drif$pred
    serie <- data.frame(ts.union(train, test, pred))
    serie$date <- isolate(updateData$seriedf)[[1]]
    colnames(serie) <- c("train", "test", "pred", "date")
    colors <- c(input$col_train_drift, input$col_test_drift, input$col_p_drift)
    
    tryCatch({
      noms <- c(tr(c('train', 'test', 'table_m'), lg), 'pred.drif')
      isolate(updateData$code[['drif']][['docdrifp']] <- code.plots(noms, colors))
      
      opts <- list(
        xAxis = list(
          type = "category", data = format(serie$date, "%Y-%m-%d %H:%M:%S")),
        yAxis = list(show = TRUE, scale = T),
        series = list(
          list(type = "line", data = serie$train, name = noms[1]),
          list(type = "line", data = serie$test,  name = noms[2]),
          list(type = "line", data = serie$pred,  name = noms[3])
        )
      )
      
      e_charts() |> e_list(opts) |> e_legend() |> e_datazoom() |> 
        e_tooltip(trigger = 'axis') |> e_show_loading() |> e_color(colors)
    }, error = function(e) {
      showNotification(paste0("ERROR 0000: ", e), type = "error")
      return(NULL)
    })
  })
  
  output$error_drift <- renderUI({
    lg <- updateData$idioma
    
    tryCatch({
      res <- div(
        style = "display: table; width: 100%; height: 70vh; overflow: scroll;",
        infoBox2(tr("mse", lg), rvmodelo$drif$error$MSE, NULL, 
                 tags$img(src = 'img/ECM.svg', style = "max-width: 90%;"), "red", 6, fill = T),
        infoBox2(tr("rmse", lg), rvmodelo$drif$error$RMSE, NULL, 
                 tags$img(src = 'img/RECM.svg', style = "max-width: 90%;"), "yellow", 6, fill = T),
        infoBox2(tr("re", lg), rvmodelo$drif$error$RE, NULL, 
                 tags$img(src = 'img/ER.svg', style = "max-width: 90%;"), "green", 6, fill = T),
        infoBox2(tr("cor", lg), rvmodelo$drif$error$CORR, NULL, 
                 tags$img(src = 'img/correlacion.svg', style = "max-width: 90%;"), "navy", 6, fill = T)
      )
      isolate(updateData$code[['drif']][['docdrife']] <- "error.drif")
      
      res
    }, error = function(e) {
      showNotification(paste0("ERROR 0000: ", e), type = "error")
      return(NULL)
    })
  })
}
    
## To be copied in the UI
# mod_desvio_ui("desvio_ui_1")
    
## To be copied in the server
# callModule(mod_desvio_server, "desvio_ui_1")
 
