#' Individual Conditional Expectation (ICE)
#'
#' Generates Individual Conditional Expectation (ICE) profiles. An ICE profile shows how the prediction of an observation changes if one or multiple variables are systematically changed across its ranges, holding all other values fixed [1]. The curves can be centered in order to increase visibility of interaction effects. Centering is done within subgroups specified by "by".
#'
#' There are two ways to specify the variable(s) to be profiled. The first option is to pass the variable name via \code{v} and an optional vector with evaluation points \code{evaluate_at} (or \code{breaks}). This works for dependence on a single variable. The second option is much more general: You can specify any \code{grid} as a \code{data.frame} with one or more columns. It can e.g. be generated by a call to \code{expand.grid}. Currently, there is no option to pass more than one variable name without such grid. The minimum required elements in the (multi-)flashlight are "predict_function", "model", "linkinv" and "data", where the latest can be passed on the fly. Which rows in \code{data} are profiled? This is specified by \code{indices}. If not given and \code{n_max} is smaller than the number of rows in \code{data}, then row indices will be sampled randomly from \code{data}. If the same rows should be used for all flashlights in a multiflashlight, there are two options: Either pass a \code{seed} (with potentially undesired consequences for subsequent code) or a vector of indices used to select rows. In both cases, \code{data} should be the same for all flashlights considered.
#'
#' @importFrom stats setNames
#' @importFrom tidyr crossing
#' @importFrom dplyr inner_join
#' @param x An object of class \code{flashlight} or \code{multiflashlight}.
#' @param v The variable to be profiled.
#' @param data An optional \code{data.frame}.
#' @param by An optional vector of column names used to additionally group the results.
#' @param evaluate_at Vector with values of \code{v} used to evaluate the profile.
#' @param breaks Instead of \code{evaluate_at} (and \code{grid}), cut points for \code{x} can be provided. From them, \code{evaluate_at} values are calculates as averages.
#' @param grid A \code{data.frame} with grid values as those generated by \code{expand.grid}.
#' @param n_bins Maximum number of unique values to evaluate for numeric \code{v}. Only used in neither \code{grid} nor \code{evaluate_at} is specified.
#' @param cut_type For the default "equal", bins of equal width are created for \code{v} by \code{pretty}. Choose "quantile" to create quantile bins. Only used in neither \code{grid} nor \code{evaluate_at} is specified.
#' @param indices A vector of row numbers to consider.
#' @param n_max If \code{indices} is not given, maximum number of rows to consider. Will be randomly picked from \code{data} if necessary.
#' @param seed An integer random seed.
#' @param use_linkinv Should retransformation function be applied? Default is TRUE.
#' @param center Should curves be centered? Default is FALSE. Note that centering will be done at the first evaluation point and within "by" group. It will work also for a \code{grid} with multiple columns.
#' @param value_name Column name in resulting \code{data} containing the profile value. Defaults to "value".
#' @param label_name Column name in resulting \code{data} containing the label of the flashlight. Defaults to "label".
#' @param id_name Column name in resulting \code{data} containing the row id of the profile. Defaults to "id_name".
#' @param ... Further arguments passed to or from other methods.
#' @return An object of class \code{light_ice}, \code{light} (and a list) with the following elements.
#' \itemize{
#'   \item \code{data} A tibble containing the results. Can be used to build fully customized visualizations. Its column names are specified by all other items in this list.
#'   \item \code{by} Same as input \code{by}.
#'   \item \code{v} The variable(s) evaluated.
#'   @item \code{center} Flag if ICE curves are centered.
#'   \item \code{value_name} Same as input \code{value_name}.
#'   \item \code{label_name} Same as input \code{label_name}.
#'   \item \code{id_name} Same as input \code{id_name}.
#' }
#' @export
#' @references [1] Goldstein, A. et al. (2015). Peeking inside the black box: Visualizing statistical learning with plots of individual conditional expectation. Journal of Computational and Graphical Statistics, 24:1 <doi.org/10.1080/10618600.2014.907095>.
#' @examples
#' fit_full <- lm(Sepal.Length ~ ., data = iris)
#' fit_part <- lm(Sepal.Length ~ Petal.Length, data = iris)
#' mod_full <- flashlight(model = fit_full, label = "full", data = iris, y = "Sepal.Length")
#' mod_part <- flashlight(model = fit_part, label = "part", data = iris, y = "Sepal.Length")
#' mods <- multiflashlight(list(mod_full, mod_part))
#' grid <- expand.grid(Species = levels(iris$Species), Petal.Length = 2:4)
#' light_ice(mod_full, v = "Species")
#' light_ice(mod_full, v = "Species", indices = (1:15) * 10)
#' light_ice(mod_full, v = "Species", evaluate_at = levels(iris$Species))
#' light_ice(mod_full, grid = grid, data = iris[1,])$data
#' light_ice(mods, v = "Species", indices = (1:15) * 10)
#' light_ice(mods, v = "Species", indices = (1:15) * 10, center = TRUE)
#' light_ice(mods, v = "Petal.Width", n_bins = 5)
#' light_ice(mods, v = "Petal.Width", by = "Species", n_bins = 5)
#' light_ice(mods, v = "Petal.Width", by = "Species",
#'   id_name = "profile", value_name = "val", label_name = "lab")
#' @seealso \code{\link{light_profile}}, \code{\link{plot.light_ice}}.
light_ice <- function(x, ...) {
  UseMethod("light_ice")
}

#' @describeIn light_ice Default method not implemented yet.
#' @export
light_ice.default <- function(x, ...) {
  stop("light_ice method is only available for objects of class flashlight or multiflashlight.")
}

#' @describeIn light_ice ICE profiles for a flashlight object.
#' @export
light_ice.flashlight <- function(x, v = NULL, data = x$data, by = x$by,
                                 evaluate_at = NULL, breaks = NULL, grid = NULL,
                                 n_bins = 27, cut_type = c("equal", "quantile"),
                                 indices = NULL, n_max = 20,
                                 seed = NULL, use_linkinv = TRUE, center = FALSE,
                                 value_name = "value",
                                 label_name = "label", id_name = "id", ...) {
  cut_type <- match.arg(cut_type)

  stopifnot((n <- nrow(data)) >= 1L,
            !is.null(grid) || !is.null(v),
            !anyDuplicated(c(by, union(v, names(grid)),
                             value_name, label_name, id_name)))

  # Complete/evaluate grid
  if (is.null(grid)) {
    if (is.null(evaluate_at)) {
      evaluate_at <- if (!is.null(breaks)) midpoints(breaks) else
        auto_cut(data[[v]], n_bins = n_bins, cut_type = cut_type, ...)$bin_means
    }
    grid <- setNames(data.frame(evaluate_at), v)
  } else {
    v <- colnames(grid)
  }

  # Pick ids
  if (is.null(indices)) {
    if (n_max < n) {
      if (!is.null(seed)) {
        set.seed(seed)
      }
      indices <- sample(n, n_max)
    } else {
      indices <- seq_len(n)
    }
  }
  data <- data[indices, , drop = FALSE]

  # Full outer join of data and grid
  cols <- colnames(data)
  data[, v] <- NULL
  data[[id_name]] <- indices
  data <- crossing(data, grid)

  # Update flashlight
  x <- flashlight(x, data = data[, cols, drop = FALSE], by = by,
                  linkinv = if (use_linkinv) x$linkinv else function(z) z)

  # Add predictions and organize output
  data <- data[, c(id_name, by, v, x$w), drop = FALSE]
  data[[value_name]] <- predict(x)

  # c-ICE curves by centering at first evaluation point
  if (center) {
    central_data <- inner_join(data, grid[1, , drop = FALSE], by = v)
    group_means <- grouped_stats(central_data, x = value_name, w = x$w,
                                 by = x$by, counts = FALSE,
                                 value_name = "global_mean", na.rm = TRUE)
    stopifnot(!("global_mean" %in% colnames(central_data)))
    central_data <- merge(central_data, group_means, by = x$by, all.x = TRUE)
    local_shift <- central_data[["global_mean"]] - central_data[[value_name]]
    data[[value_name]] <- data[[value_name]] +
      local_shift[match(data[[id_name]], central_data[[id_name]])]
  }
  data[[label_name]] <- x$label

  # Collect results
  out <- list(data = data, by = by, v = colnames(grid),
              center = center, value_name = value_name,
              label_name = label_name, id_name = id_name)
  class(out) <- c("light_ice", "light", "list")
  out
}

#' @describeIn light_ice ICE profiles for a multiflashlight object.
#' @export
light_ice.multiflashlight <- function(x, ...) {
  light_combine(lapply(x, light_ice, ...), new_class = "light_ice_multi")
}
