\name{MRbasis}
\alias{MRbasis}
\alias{MRbasis.setup}
\alias{MRprecision}
\alias{MR.cov}
\alias{MR.sim}
\title{
Functions for generating a multiresolution, compactly supported basis,  
multiresolution covariance functions and simulating form these processes.
}
\description{
 Creates radial basis functions of varying support on a nested set of regular grids. 
The Gaussian process is an expansion in these basis functions with a covariance matrix on the basis coefficients using a Markov random field model for each resolution level.  
}
\usage{
MRbasis(x1, MRinfo, spam.format = TRUE, verbose = FALSE)
MRbasis.setup(grid.info, nlevel = 1, eflag = FALSE, overlap = 2.5,INFLATE = 1.5) 
MRprecision(MRinfo, alpha = 1, beta = -0.2, spam.format = TRUE) 
MR.cov(x1, x2 = NULL, grid.info, alpha = 1, nlevel = 1, beta = -0.2, 
    C = NA, marginal = FALSE, eflag = FALSE, INFLATE = 1.5) 
MR.sim(x1, grid.info, alpha = 1, nlevel = 1, beta = -0.2, 
    overlap = 2.5, eflag = FALSE, INFLATE = 1.5, M = 1) 
}

\arguments{
  \item{x1}{ Matrix of 2-d locations to evaluate basis functions or the first 
set of locations to evaluate the covariance function or the locations for the simulated process.}
  \item{MRinfo}{ List describing the multiresoluion, usually the output from 
   \code{MRbasis.setup} }
  \item{spam.format}{ If TRUE matrix is returned in sparse matrix format.}
  \item{grid.info}{A list with components \code{xmin, xmax, ymin, ymax, delta} that 
 specifies the  range and spacing for the coarsest level grid.}
  \item{nlevel}{Number of levels in multiresolution. Note that each subsequent level 
increases in size by a factor of 4= 2X2. }
   \item{eflag}{If true the last level will use tapered exponential covariance functions
in place of the Wendland.}
   \item{overlap}{The amount of overlap between basis functions. If the grid spaciing
in x and y is \code{delta} then the support of the basis functions will be \code{overlap*delta}}
   \item{INFLATE}{If eflag is true then the Wendland taper will extend to a range
of \code{INFLATE*overlap*delta}}
   \item{alpha}{A scalar or vector with the relative weights to give the different
precision matrices for each level of coefficients.}
    \item{beta}{A scalar or vector giving the lattice connection strengths used to 
construct the precision matrix for each resolution level.}
\item{x2}{Second set of locations to evaluate covariance function.}
\item{C}{If passed the covariance matrix will be multiplied by this vector or matrix.}
\item{marginal}{If TRUE returns the marginal variance. Currently not implemented!}
\item{M}{Number of simulated fields.}
\item{verbose}{If TRUE intermediate steps and information is printed.}
}
\details{
The basis functions are two-dimensional radial basis functions based on the Wendland covariance
function, centered on the grid points and with the scaling tied to the spacing of the grid points.
For a basis at a the coarest level, the grid centers are generated by expanding the
sequences  \code{seq(grid.info$xmin, grid.info$xmax,grid.info$delta)} \code{seq(grid.info$ymin, grid.info$ymax,grid.info$delta)} into a matrix of centers. The basis functions are evaluated at 
locations \code{x1} by finding the pairwise, radial  distances among \code{centers} and \code{x1}, 
scaling by \code{grid.info$delta * overlap} and then evaluating at the 2-d Wendland covariance with 
order 2. In R code:

\preformatted{
 centers<- make.surface.grid(list(
                          x=seq(grid.info$xmin, grid.info$xmax,grid.info$delta),
                          y= seq(grid.info$ymin, grid.info$ymax,grid.info$delta)))
 bigD<- rdist( x1, centers)/(grid.info$delta*overlap)
 PHI<- Wendland( bigD)
} 

The actual code uses the function \code{wendland.cov} to take advantage of the routines fro 
creating sparse formats. 

The precision matrix for the coefficients at each is computed by \code{MRprecision} and these
are assembled as the block diagonals of a larger precision matrix for the entire vector of
coefficients. Note these matrices are also created in a sparse format. 

}
\value{

\code{MRbasis}
A matrix with rows equal to the rows of \code{x1} and columns equal to the number of 
basis functions ( MRinfo\$Ntotal). Attached to the matrix is the info attribute that contains the 
list passed as \code{MRinfo}. 

\code{MRbasis.setup}
A list with components giving the numbers of basis functions at each level and the information to 
reconstruct the grid for the basis centers. 

\code{MRprecision} A matrix with dimensions of the number of basis functions.

\code{MR.cov} If \code{C=NA} a cross covariance matrix with dimensions \code{nrow(x1)} and 
\code{nrow(x2)}. 
If \code{C} is passed the result of multiplying the cross covariance matrix times \code{C}.

\code{MR.sim} An  matrix with dimensions of \code{nrow(x1)} by \code{M} of simulated values at the locations \code{x1}. 

}
\author{
Doug Nychka
}

\seealso{
LatticeKrig, wendland.cov, mKrig, Krig, fastTps, Wendland
}

\examples{
# generating a basis on the domain [-1,1]X[-1,1] with 3 levels
  ginfo<- list( xmin=-1, xmax=1, ymin=-1, ymax=1, delta= 2/3)
  MRinfo<- MRbasis.setup(ginfo, nlevel=3)
# evaluate the basis functions on a grid to look at them
  xg<- make.surface.grid( list(x=seq(-1,1,,40), y= seq(-1,1,,40)))
  PHI<- MRbasis( xg,MRinfo)
# plot the 6th basis function
  image.plot(as.surface(xg, PHI[,6]))
# See a more elaborate example at the end that illustrates indexing and 
# shapes
#
# create four resolution levels of basis functions 
# take a look at the covariance function w/wo the tapered exponential level
  NL<-4
  beta<-  c( -.245, -.245, -.1, 0)
  alpha<- c(1.0)
  grid.info<- list( xmin=-1, xmax=1, ymin=-1, ymax=1, delta=.25)
# evaluate the covariance along x-axis to reduce number of points. 
  x0<-  cbind( 0,0)
  xg<- cbind( seq(0,1,,200),0)
  hold<- MR.cov(xg,x0,grid.info, alpha=alpha, beta=beta,nlevel=NL, eflag=TRUE )
  hold<- hold/max(hold)
  hold2<- MR.cov(xg,x0,grid.info, alpha=alpha, beta=beta,nlevel=NL, eflag=FALSE)
  hold2<- hold2/max( hold2)
  set.panel(2,1)
  matplot( xg[,1], (1-cbind( hold, hold2)) , type="l")
  matplot( xg[1:20,1], 1- cbind( hold, hold2)[1:20,], type="l")
#
# simulate 4 realizations of process. 
  xg2<- make.surface.grid( list( x= seq( -1,1,,80), y= seq( -1,1,,80)))
  set.seed(234)
  out<- MR.sim(xg2, grid.info, alpha=alpha, beta=beta,nlevel=NL, eflag=TRUE,M=4)
  set.panel(2,2)
  for( k in 1:4){
    image.plot( as.surface( xg2, out[,k]), axes=FALSE) }
#
# example of basis function indexing
#
# generating a basis on the domain [-1,1]X[-1,1] with 3 levels
  ginfo<- list( xmin=-1, xmax=1, ymin=-1, ymax=1, delta= 2/3)
  MRinfo<- MRbasis.setup(ginfo, nlevel=3)
# evaluate the basis functions on a grid to look at them
  xtemp<- seq(-1,1,,40)
  xg<- make.surface.grid( list(x=xtemp, y= xtemp) )
  PHI<- MRbasis( xg,MRinfo)
# first second, and last basis function in each resolution level
# basis functions centers are added
  set.panel(3,3)
  grid.info<- MRinfo$grid.info
  for(  j in 1:3){
    id1<- MRinfo$offset[j]+ 1
    id2<-  MRinfo$offset[j]+ 2
    idlast<- MRinfo$offset[j]+ MRinfo$N1[j]*MRinfo$N2[j]

    centers<-  make.surface.grid( list( x= seq(grid.info$xmin, grid.info$xmax, MRinfo$delta[j]),
                      y= seq(grid.info$xmin, grid.info$xmax, MRinfo$delta[j]) ) )
    image.plot(xtemp, xtemp, matrix( PHI[,id1],40,40))
    points( centers, cex=.2, col="grey")
    image.plot(xtemp, xtemp, matrix( PHI[,id2],40,40))
    points( centers, cex=.2, col="grey")
    image.plot(xtemp, xtemp, matrix( PHI[,idlast],40,40))
    points( centers, cex=.2, col="grey")}

}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{spatial}

