#' Plot Estimated Return Levels for S-FFA
#'
#' Generates a plot with return periods on the x-axis and return levels (annual 
#' maxima magnitudes) on the y-axis for S-FFA. The confidence bound is shown as a 
#' semi-transparent ribbon, and the point estimates are overlaid as a solid line.
#' Return periods are shown on a logarithmic scale.
#'
#' @param results A fitted flood frequency model generated by [fit_lmoments()], 
#' [fit_mle()] or [fit_gmle()] OR a fitted model with confidence intervals generated
#' by [uncertainty_bootstrap()], [uncertainty_rfpl()], or [uncertainty_rfgpl()].
#'
#' @inheritParams param-periods
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return `ggplot`; a plot showing:
#' - A solid black line for the point estimates produced by the model.
#' - A semi-transparent gray ribbon indicating the confidence interval, if given.
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' years <- seq(from = 1901, to = 2000)
#' results <- fit_lmoments(data, "WEI")
#' plot_sffa_estimates(results)
#'
#' @import ggplot2
#' @export
plot_sffa_estimates <- function(
	results,
	periods = c(2, 5, 10, 20, 50, 100),
	...
) {

	# Override the periods argument if contained in 'results'
	if ("ci" %in% names(results)) {
		periods <- results$ci$periods
	}

	# Compute the quantile estimates
	returns = 1 - (1 / periods)
	structure = list(location = FALSE, scale = FALSE)
	estimates = utils_quantiles(returns, results$distribution, results$params)

	# Capture optional arguments
	args <- list(...)

	# Define labels for the plot, overriding if args are provided
	title <- args$title %||% "S-FFA Results"
	xlabel <- args$xlabel %||% "Return Period (Years)" 
	ylabel <- args$ylabel %||% expression("Streamflow (" * m^3/s * ")")

	# Define labels and styles for the legend
	labels <- c("Estimates", "Confidence Bounds")
	colors <- c("black", NA)

	# Create a dataframe for the return levels
	df <- data.frame(periods = periods, estimates = estimates)

	# Add the lines 
	p1 <- ggplot(data = df, aes(x = .data$periods, y = .data$estimates)) +
		geom_line(aes(color = "1"), linewidth = 1) +
		geom_point(aes(color = "1"), size = 2.25)

	# Add the CI ribbon if possible
	if ("ci" %in% names(results)) {
		p1 <- p1 + geom_ribbon(
			data = results$ci,
			aes(ymin = .data$lower, ymax = .data$upper, color = "2"),
			linewidth = 0,
			alpha = 0.2
		)
	}

	# Add the labels and legend
	p1 <- p1 + scale_color_manual(labels = labels, values = colors) +
		labs(title = title, x = xlabel, y = ylabel, color = "Legend") + 
		scale_x_log10(breaks = df$periods)

	# Add the theme and return 
	add_theme(p1)

}
