% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/undirected_sbm.R
\name{sbm}
\alias{sbm}
\title{Create an undirected stochastic blockmodel object}
\usage{
sbm(
  n,
  k = NULL,
  B = NULL,
  ...,
  pi = rep(1/k, k),
  sort_nodes = TRUE,
  poisson_edges = TRUE,
  allow_self_loops = TRUE
)
}
\arguments{
\item{n}{The number of nodes in the network. Must be
a positive integer. This argument is required.}

\item{k}{(mixing matrix) The number of blocks in the blockmodel.
Use when you don't want to specify the mixing-matrix by hand.
When \code{k} is specified, the elements of \code{B} are drawn
randomly from a \code{Uniform(0, 1)} distribution.
This is subject to change, and may not be reproducible.
\code{k} defaults to \code{NULL}. You must specify either \code{k}
or \code{B}, but not both.}

\item{B}{(mixing matrix) A \code{k} by \code{k} matrix of block connection
probabilities. The probability that a node in block \code{i} connects
to a node in community \code{j} is \code{Poisson(B[i, j])}. Must be
a square matrix. \code{matrix} and \code{Matrix} objects are both
acceptable. If \code{B} is not symmetric, it will be
symmetrized via the update \code{B := B + t(B)}. Defaults to \code{NULL}.
You must specify either \code{k} or \code{B}, but not both.}

\item{...}{
  Arguments passed on to \code{\link[=undirected_factor_model]{undirected_factor_model}}
  \describe{
    \item{\code{expected_degree}}{If specified, the desired expected degree
of the graph. Specifying \code{expected_degree} simply rescales \code{S}
to achieve this. Defaults to \code{NULL}. Do not specify both
\code{expected_degree} and \code{expected_density} at the same time.}
    \item{\code{expected_density}}{If specified, the desired expected density
of the graph. Specifying \code{expected_density} simply rescales \code{S}
to achieve this. Defaults to \code{NULL}. Do not specify both
\code{expected_degree} and \code{expected_density} at the same time.}
  }}

\item{pi}{(relative block probabilities) Relative block
probabilities. Must be positive, but do not need to sum
to one, as they will be normalized internally.
Must match the dimensions of \code{B} or \code{k}. Defaults to
\code{rep(1 / k, k)}, or a balanced blocks.}

\item{sort_nodes}{Logical indicating whether or not to sort the nodes
so that they are grouped by block and by \code{theta}. Useful for plotting.
Defaults to \code{TRUE}. When \code{TRUE}, nodes are first sorted by block
membership, and then by degree-correction parameters within each block.
Additionally, \code{pi} is sorted in increasing order, and the columns
of the \code{B} matrix are permuted to match the new order of \code{pi}.}

\item{poisson_edges}{Logical indicating whether or not
multiple edges are allowed to form between a pair of
nodes. Defaults to \code{TRUE}. When \code{FALSE}, sampling proceeds
as usual, and duplicate edges are removed afterwards. Further,
when \code{FALSE}, we assume that \code{S} specifies a desired between-factor
connection probability, and back-transform this \code{S} to the
appropriate Poisson intensity parameter to approximate Bernoulli
factor connection probabilities. See Section 2.3 of Rohe et al. (2017)
for some additional details.}

\item{allow_self_loops}{Logical indicating whether or not
nodes should be allowed to form edges with themselves.
Defaults to \code{TRUE}. When \code{FALSE}, sampling proceeds allowing
self-loops, and these are then removed after the fact.}
}
\value{
An \code{undirected_sbm} S3 object, which is a subclass of the
\code{\link[=dcsbm]{dcsbm()}} object.
}
\description{
To specify a stochastic blockmodel, you must specify
the number of nodes (via \code{n}), the mixing matrix (via \code{k} or \code{B}),
and the relative block probabilites (optional, via \code{pi}).
We provide defaults for most of these options to enable
rapid exploration, or you can invest the effort
for more control over the model parameters. We \strong{strongly recommend}
setting the \code{expected_degree} or \code{expected_density} argument
to avoid large memory allocations associated with
sampling large, dense graphs.
}
\details{
A stochastic block is equivalent to a degree-corrected
stochastic blockmodel where the degree heterogeneity parameters
have all been set equal to 1.
}
\examples{

set.seed(27)

lazy_sbm <- sbm(n = 100, k = 5, expected_density = 0.01)
lazy_sbm

# by default we get a multigraph (i.e. multiple edges are
# allowed between the same two nodes). using bernoulli edges
# will with an adjacency matrix with only zeroes and ones

bernoulli_sbm <- sbm(
  n = 500,
  k = 30,
  poisson_edges = FALSE,
  expected_degree = 8
)

bernoulli_sbm

edgelist <- sample_edgelist(bernoulli_sbm)
edgelist

A <- sample_sparse(bernoulli_sbm)

# only zeroes and ones!
sign(A)

}
\seealso{
Other stochastic block models: 
\code{\link{dcsbm}()},
\code{\link{directed_dcsbm}()},
\code{\link{mmsbm}()},
\code{\link{overlapping_sbm}()},
\code{\link{planted_partition}()}

Other undirected graphs: 
\code{\link{chung_lu}()},
\code{\link{dcsbm}()},
\code{\link{erdos_renyi}()},
\code{\link{mmsbm}()},
\code{\link{overlapping_sbm}()},
\code{\link{planted_partition}()}
}
\concept{stochastic block models}
\concept{undirected graphs}
