
#' Read EXIF data
#'
#' Reads EXIF data into a \code{data.frame} by calling the ExifTool command-line
#' application, written by Phil Harvey. Depending on number of images and
#' command-line length requirements, the command may be called multiple times.
#'
#' From the \href{http://www.sno.phy.queensu.ca/~phil/exiftool/}{ExifTool
#' website}: ExifTool is a platform-independent Perl library plus a command-line
#' application for reading, writing and editing meta information in a wide
#' variety of files. ExifTool supports many different metadata formats including
#' EXIF, GPS, IPTC, XMP, JFIF, GeoTIFF, ICC Profile, Photoshop IRB, FlashPix,
#' AFCP and ID3, as well as the maker notes of many digital cameras by Canon,
#' Casio, FLIR, FujiFilm, GE, HP, JVC/Victor, Kodak, Leaf,
#' Minolta/Konica-Minolta, Motorola, Nikon, Nintendo, Olympus/Epson,
#' Panasonic/Leica, Pentax/Asahi, Phase One, Reconyx, Ricoh, Samsung, Sanyo,
#' Sigma/Foveon and Sony.
#'
#' For more information, see the \href{http://www.sno.phy.queensu.ca/~phil/exiftool/}{ExifTool
#' website}
#'
#' @param filename A vector of filenames (like that generated by
#'   \code{list.files(..., full.names=TRUE)})
#' @param recursive If directory is specified by \code{filename}, process files recursively.
#' @param quiet \code{FALSE} if status updates are desired, \code{TRUE}
#'   otherwise.
#' @param exiftoolargs a list of args to be passed to ExifTool (e.g. \code{c("-filename", "-imagesize")})
#'
#' @return A \code{data.frame} of photo EXIF information.
#'
#' @examples
#' files <- list.files(path.package("exifr"), recursive=TRUE, pattern="*.jpg", full.names=TRUE)
#' exifinfo <- exifr(files)
#' # is equivalent to
#' exifinfo <- exifr(path.package("exifr"), recursive=TRUE)
#'
#' exifr(files, exiftoolargs="-filename -imagesize")
#'
#' @export
#'
exifr <- function(filename, recursive=FALSE, quiet=TRUE, exiftoolargs=NULL) {
  fnames <- NULL ; rm(fnames); i <-NULL; rm(i); command <- NULL; rm(command)

  if(length(filename) == 0) {
    return(data.frame())
  }

  if(recursive) {
    exiftoolargs <- c(exiftoolargs, "-r")
  }

  foreach <- foreach::foreach
  `%do%` <- foreach::`%do%`

  if(.Platform$OS.type == "windows") {
    commandlength <- 8191 #according to https://support.microsoft.com/en-us/kb/830473
  } else {
    #according to the interweb this should be around 100kb, but 50kb should do
    #(is 'getconf ARG_MAX' on mac, equals 262144)
    commandlength <- 50*1024
  }

  message("Generating command line arguments...")

  filenameslist <- list(filename)
  nominallen <- length(filename)
  commands <- foreach(fnames=filenameslist, .combine=c) %do% {
    exiftool.command(args=c("-n", "-csv", "-q", exiftoolargs), fnames)
  }
  while(any(nchar(commands) >= commandlength) && commandlength != 0 && nominallen >= 2) {
    #subdivide files until all commands are less than commandlength
    nominallen <- as.integer(nominallen / 2)
    if((length(filename) %% nominallen) == 0) {
      ngroups = length(filename) / nominallen
    } else {
      ngroups <- as.integer(length(filename)/nominallen) + 1
    }
    filenameslist <- foreach(i=1:ngroups) %do% {
      minind <- ((i-1)*nominallen+1)
      maxind <- min(i*nominallen, length(filename))
      filename[minind:maxind]
    }
    commands <- foreach(fnames=filenameslist, .combine=c) %do% {
      exiftool.command(args=c("-n", "-csv", "-q", exiftoolargs), fnames)
    }
  }

  message("Running ", length(commands), " commands")
  return(foreach(command=commands, .combine=plyr::rbind.fill, .multicombine=TRUE) %do% {
    if(!quiet) message(command)
    if(quiet) {
      suppressMessages(suppressWarnings({
        utils::read.csv(textConnection(system(command, intern=TRUE, ignore.stderr = TRUE)), stringsAsFactors = FALSE)
      }))
    } else {
      utils::read.csv(textConnection(system(command, intern=TRUE, ignore.stderr = TRUE)), stringsAsFactors = FALSE)
    }
  })
}

#' Call exiftool from R
#'
#' Uses \code{system()} to run a basic call to \code{exiftool}.

#' @param args a list of non-shell quoted arguments (e.g. \code{-n -csv})
#' @param fnames a list of filenames (\code{shQuote()} will be applied to this vector)
#' @param intern \code{TRUE} if output should be returned as a character vector.
#' @param ... additional arguments to be passed to \code{system()}
#'
#' @return The exit code if \code{intern=FALSE}, or the standard output as a character vector
#'  if \code{intern=TRUE}.
#' @export
#'
#' @examples
#' exiftool.call("--help")
#'
exiftool.call <- function(args=c("--help"), fnames=NULL, intern=FALSE, ...) {
  system(exiftool.command(args, fnames), intern=intern, ...)
}


#private helper command
exiftool.command <- function(args, fnames) {
  exiftoolpath <- options("exifr.exiftoolcommand")$exifr.exiftoolcommand
  if(is.null(exiftoolpath)) stop("ExifTool not properly installed")
  if(length(fnames) > 0) {
    paste(exiftoolpath, paste(args, collapse=" "), paste(shQuote(fnames), collapse=" "))
  } else {
    paste(exiftoolpath, paste(args, collapse=" "))
  }
}
