## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 8,
  fig.height = 5  
)

## ----setup, message=FALSE-----------------------------------------------------
library(ethiodate)
library(ggplot2)
library(dplyr)
head(cpieth)

## -----------------------------------------------------------------------------
cpieth <- cpieth |>
  mutate(edate = eth_date(date),
         eyear = eth_year(edate),
         emonth = eth_month(edate),
         equarter = eth_quarter(edate))

## -----------------------------------------------------------------------------
p <- ggplot(cpieth, aes(x = edate, y = cpi)) +
  geom_line(color = "steelblue", linewidth = 1) +
  labs(title = "Consumer Price Index (CPI) Over Time",
       subtitle = "Based on Ethiopian Calendar",
       x = "Ethiopian Date", y = "CPI") +
  theme_bw()

p

## -----------------------------------------------------------------------------
p + 
  scale_x_ethdate(breaks = eth_breaks(7),
                  labels = eth_labels("%Y")) + 
  scale_y_continuous(labels = scales::label_percent(scale = 1)) + 
  geom_hline(yintercept = 0, linewidth = 0.3, linetype = 2, color = "tomato")

## -----------------------------------------------------------------------------
p1 <- cpieth |> 
  filter(eyear > 2005 & eyear < 2015) |> 
  ggplot(aes(x = emonth, y = cpi, group = eyear, color = factor(eyear))) +
  geom_line() +
  scale_x_continuous(breaks = 1:13, labels = eth_show(lang = "lat")) +
  scale_y_continuous(labels = scales::label_percent(scale = 1)) +
  labs(title = "CPI Seasonality by Ethiopian Year",
       x = "Month", y = "CPI", color = "Year") +
  theme_bw() + 
  theme(axis.text.x = element_text(angle = 45, vjust = 0.9, hjust = 1, colour = "black"))

p1

## -----------------------------------------------------------------------------
p2 <- cpieth |> 
  filter(eyear > 1993 & eyear < 2015) |> 
  ggplot(aes(x = factor(emonth), y = factor(eyear), fill = cpi)) +
  geom_tile(color = "white") +
  scale_fill_viridis_c(labels = scales::label_percent(scale = 1)) +
  scale_x_discrete(labels = eth_show("%b", "lat")) +
  labs(title = "Monthly CPI Heatmap",
       x = "Ethiopian Month", y = "Ethiopian Year", fill = "CPI") +
  theme_bw() + 
  theme(axis.text.x = element_text(colour = "black"))

p2

## -----------------------------------------------------------------------------
cpieth |>
  filter(eyear > 2010) |> 
  mutate(equarter = eth_quarter(edate)) |> 
  group_by(eyear, equarter) |>
  summarise(mean_cpi = mean(cpi), .groups = "drop") |> 
  ggplot(aes(x = interaction(eyear, equarter), y = mean_cpi)) +
  geom_col(fill = "darkorange") +
  labs(title = "Quarterly Average CPI after 2010",
       x = "Ethiopian Quarter", y = "Average CPI") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust=1))

