% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/retra_edr.R
\name{retra_edr}
\alias{retra_edr}
\title{Representative trajectories in Ecological Dynamic Regimes (RETRA-EDR)}
\usage{
retra_edr(
  d,
  trajectories,
  states,
  minSegs,
  dSegs = NULL,
  coordSegs = NULL,
  traj_Segs = NULL,
  state1_Segs = NULL,
  state2_Segs = NULL,
  Dim = NULL,
  eps = 0
)
}
\arguments{
\item{d}{Either a symmetric matrix or an object of class \code{\link{dist}} containing the
dissimilarities between each pair of states of all trajectories in the EDR.}

\item{trajectories}{Vector indicating the trajectory or site to which each
state in \code{d} belongs.}

\item{states}{Vector of integers indicating the order of the states in \code{d} for
each trajectory.}

\item{minSegs}{Integer indicating the minimum number of segments in a region
of the EDR represented by a segment of the representative trajectory.}

\item{dSegs}{Either a symmetric matrix or an object of class \code{\link{dist}} containing
the dissimilarities between every pair of trajectory segments (see Details).}

\item{coordSegs}{Matrix containing the coordinates of trajectory segments (rows)
in each axis (columns) of an ordination space (see Details).}

\item{traj_Segs}{Vector indicating the trajectory to which each segment in \code{dSeg}
and/or \code{coordSegs} belongs. Only required if \code{dSegs} or \code{coordSegs} are not \code{NULL}.}

\item{state1_Segs}{Vector indicating the initial state of each segment in \code{dSegs}
and/or \code{coordSegs} according to the values given in \code{states}. Only required if
\code{dSegs} or \code{coordSegs} are not \code{NULL}.}

\item{state2_Segs}{Vector indicating the final state of each segment in \code{dSegs}
and/or \code{coordSegs} according to the values given in \code{states}. Only required if
\code{dSegs} or \code{coordSegs} are not \code{NULL}.}

\item{Dim}{Optional integer indicating the number of axes considered to
partition the segment space and generate a k-d tree. By default (\code{Dim = NULL}),
all axes are considered.}

\item{eps}{Numeric value indicating the minimum length in the axes of the segment
space to be partitioned when the k-d tree is generated. If \code{eps = 0} (default),
partitions are made regardless of the size.}
}
\value{
The function \code{retra_edr()} returns an object of class \code{RETRA}, which is a list
of length equal to the number of representative trajectories identified. For
each trajectory, the following information is returned:
\describe{
\item{\code{minSegs}}{Value of the \code{minSegs} parameter.}
\item{\code{Segments}}{Vector of strings including the sequence of segments forming the
representative trajectory. Each segment is identified by a string of the form
\code{traj[st1-st2]}, where \code{traj} is the identifier of the original trajectory to
which the segment belongs and \code{st1} and \code{st2} are identifiers of the initial
and final states defining the segment.}
\item{\code{Size}}{Numeric value indicating the number of states forming the representative
trajectory.}
\item{\code{Length}}{Numeric value indicating the length of the representative trajectory,
calculated as the sum of the dissimilarities in \code{d} between every pair of
consecutive states.}
\item{\code{Link_distance}}{Data frame of two columns indicating artificial links between
representative segments (\code{Link}) and the dissimilarity between the connected
states (\code{Distance}). When two representative segments are linked by a common
state or by two consecutive states of the same trajectory, the link distance
is zero or equal to the length of a real segment, respectively. In both cases,
the link is not considered in the returned data frame.}
\item{\code{Seg_density}}{Data frame of two columns and one row for each representative
segment. \code{Density} contains the number of segments in the EDR that is represented
by each segment of the representative trajectory. \code{kdTree_depth} contains the
depth of the k-d tree for each leaf represented by the corresponding segment.
That is, the number of partitions of the ordination space until finding a region
with \code{minSegs} segments or less.}
}
}
\description{
\code{retra_edr()} applies the algorithm RETRA-EDR (Sánchez-Pinillos et al., 2023)
to identify representative trajectories summarizing the main dynamical patterns
of an Ecological Dynamic Regime (EDR).
}
\details{
The algorithm RETRA-EDR is based on a partition-and-group approach by which it
identifies regions densely crossed by ecological trajectories in an EDR, selects
a representative segment in each dense region, and joins the representative
segments by a set of artificial \code{Links} to generate a network of representative
trajectories. For that, RETRA-EDR splits the trajectories of the EDR into
segments and uses an ordination space generated from a matrix containing the
dissimilarities between trajectory segments. Dense regions are identified by
applying a k-d tree to the ordination space.

By default, RETRA-EDR calculates segment dissimilarities following the approach
by De Cáceres et al. (2019) and applies metric multidimensional scaling (mMDS,
Borg and Groenen, 2005) to generate the ordination space. It is possible to use
other dissimilarity metrics and/or ordination methods and reduce the computational
time by indicating the dissimilarity matrix and the coordinates of the segments
in the ordination space through the arguments \code{dSegs} and \code{coordSegs}, respectively.
\itemize{
\item If \code{!is.null(dSegs)} and \code{is.null(coordSegs)}, RETRA-EDR is computed by
applying mMDS to \code{dSegs}.
\item If \code{!is.null(dSegs)} and \code{!is.null(coordSegs)}, RETRA-EDR is directly computed
from the coordinates provided in \code{coordSegs} and representative segments are
identified using \code{dSegs}. \code{coordSegs} should be calculated by the user from
\code{dSegs}.
\item If \code{is.null(dSegs)} and \code{!is.null(coordSegs)} (not recommended), RETRA-EDR
is directly computed from the coordinates provided in \code{coordSegs}. As \code{dSegs}
is not provided, \code{retra_edr()} assumes that the ordination space is metric and
identifies representative segments using the Euclidean distance.
}
}
\examples{
# Example 1 -----------------------------------------------------------------
# Identify representative trajectories from state dissimilarities

# Calculate state dissimilarities (Bray-Curtis) from species abundances
abundance <- data.frame(EDR_data$EDR1$abundance)
d <- vegan::vegdist(abundance[, -c(1:3)], method = "bray")

# Identify the trajectory (or site) and states in d
trajectories <- abundance$traj
states <- as.integer(abundance$state)

# Compute RETRA-EDR
RT1 <- retra_edr(d = d, trajectories = trajectories, states = states,
                 minSegs = 5)

# Example 2 -----------------------------------------------------------------
# Identify representative trajectories from segment dissimilarities

# Calculate segment dissimilarities using the Hausdorff distance
dSegs <- ecotraj::segmentDistances(ecotraj::defineTrajectories(d = d, sites = trajectories,
                                                               surveys = states),
                                   distance.type = "Hausdorff")
dSegs <- dSegs$Dseg

# Identify the trajectory (or site) and states in dSegs:
# Split the labels of dSegs (traj[st1-st2]) into traj, st1, and st2
seg_components <- strsplit(gsub("\\\\]", "", gsub("\\\\[", "-", labels(dSegs))), "-")
traj_Segs <- sapply(seg_components, "[", 1)
state1_Segs <- as.integer(sapply(seg_components, "[", 2))
state2_Segs <- as.integer(sapply(seg_components, "[", 3))

# Compute RETRA-EDR
RT2 <- retra_edr(d = d, trajectories = trajectories, states = states, minSegs = 5,
                dSegs = dSegs, traj_Segs = traj_Segs,
                state1_Segs = state1_Segs, state2_Segs = state2_Segs)


}
\references{
Borg, I., & Groenen, P. J. F. (2005). Modern Multidimensional Scaling (2nd ed.).
Springer.

De Cáceres, M, Coll L, Legendre P, Allen RB, Wiser SK, Fortin MJ,
Condit R & Hubbell S. (2019). Trajectory analysis in community ecology. Ecological
Monographs.

Sánchez-Pinillos, M., Kéfi, S., De Cáceres, M., Dakos, V. 2023. Ecological Dynamic
Regimes: Identification, characterization, and comparison. \emph{Ecological Monographs}.
\url{doi:10.1002/ecm.1589}
}
\seealso{
\code{\link[=summary]{summary()}} for summarizing the characteristics of the representative
trajectories.

\code{\link[=plot]{plot()}} for plotting representative trajectories in an ordination space
representing the state space of the EDR.

\code{\link[=define_retra]{define_retra()}} for defining representative trajectories from a subset of
segments or trajectory features.
}
\author{
Martina Sánchez-Pinillos
}
