% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/diffeqr.R
\name{sde.solve}
\alias{sde.solve}
\title{Solve Stochastic Differential Equations (SDE)}
\usage{
sde.solve(f, g, u0, tspan, p = NULL, alg = "nothing", noise.dims = NULL,
  reltol = 0.01, abstol = 0.01, saveat = NULL)
}
\arguments{
\item{f}{the drift function.}

\item{g}{the diffusion function.}

\item{u0}{the initial condition. Can be a number or (arbitrary dimension) array.}

\item{tspan}{the timespan to solve over. Should be a list of two values: (initial time, end time).}

\item{p}{the parameters. Defaults to no parameters. Can be a number or an array.}

\item{alg}{the algorithm used to solve the differential equation. Defaults to an adaptive choice.
Algorithm choices are done through a string which matches the DifferentialEquations.jl form.}

\item{noise.dims}{list of the dimensions for the noise rate term. Defaults to NULL which gives diagonal noise.}

\item{reltol}{the relative tolerance of the ODE solver. Defaults to 1e-3.}

\item{abstol}{the absolute tolerance of the ODE solver. Defaults to 1e-6}

\item{saveat}{the time points to save values at. Should be an array of times. Defaults to automatic.}
}
\value{
sol. Has the sol$t for the time points and sol$u for the values.
}
\description{
Solves an SDE with du=f(u,p,t)dt + g(u,p,t)dW_t, for u(0)=u0 over the tspan
}
\examples{

\donttest{ ## diffeq_setup() is time-consuming and requires Julia+DifferentialEquations.jl

diffeqr::diffeq_setup()

# Scalar SDEs

f <- function(u,p,t) {
  return(1.01*u)
}
g <- function(u,p,t) {
  return(0.87*u)
}
u0 = 1/2
tspan <- list(0.0,1.0)
sol = diffeqr::sde.solve(f,g,u0,tspan)
#plotly::plot_ly(udf, x = sol$t, y = sol$u, type = 'scatter', mode = 'lines')

# Diagonal Noise SDEs

f <- JuliaCall::julia_eval("
function f(du,u,p,t)
  du[1] = 10.0*(u[2]-u[1])
  du[2] = u[1]*(28.0-u[3]) - u[2]
  du[3] = u[1]*u[2] - (8/3)*u[3]
end")

g <- JuliaCall::julia_eval("
function g(du,u,p,t)
  du[1] = 0.3*u[1]
  du[2] = 0.3*u[2]
  du[3] = 0.3*u[3]
end")
tspan <- list(0.0,100.0)
sol = diffeqr::sde.solve('f','g',u0,tspan,p=p,saveat=0.05)
udf = as.data.frame(sol$u)
#plotly::plot_ly(udf, x = ~V1, y = ~V2, z = ~V3, type = 'scatter3d', mode = 'lines')

# Non-Diagonal Noise SDEs

f <- JuliaCall::julia_eval("
function f(du,u,p,t)
  du[1] = 10.0*(u[2]-u[1])
  du[2] = u[1]*(28.0-u[3]) - u[2]
  du[3] = u[1]*u[2] - (8/3)*u[3]
end")
g <- JuliaCall::julia_eval("
function g(du,u,p,t)
  du[1,1] = 0.3u[1]
  du[2,1] = 0.6u[1]
  du[3,1] = 0.2u[1]
  du[1,2] = 1.2u[2]
  du[2,2] = 0.2u[2]
  du[3,2] = 0.3u[2]
end")
u0 = c(1.0,0.0,0.0)
tspan <- list(0.0,100.0)
noise.dims = list(3,2)
sol = diffeqr::sde.solve('f','g',u0,tspan,saveat=0.005,noise.dims=noise.dims)
udf = as.data.frame(sol$u)
#plotly::plot_ly(udf, x = ~V1, y = ~V2, z = ~V3, type = 'scatter3d', mode = 'lines')

}

}
