% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/in-line.R
\name{inline_desirability}
\alias{inline_desirability}
\alias{d_max}
\alias{d_min}
\alias{d_target}
\alias{d_box}
\alias{d_custom}
\alias{d_category}
\title{Desirability functions for in-line computations}
\usage{
d_max(x, low, high, scale = 1, missing = NA_real_, use_data = FALSE)

d_min(x, low, high, scale = 1, missing = NA_real_, use_data = FALSE)

d_target(
  x,
  low,
  target,
  high,
  scale_low = 1,
  scale_high = 1,
  missing = NA_real_,
  use_data = FALSE
)

d_box(x, low, high, missing = NA_real_, use_data = FALSE)

d_custom(x, x_vals, desirability, missing = NA_real_)

d_category(x, categories, missing = NA_real_)
}
\arguments{
\item{x}{A vector of data to compute the desirability function}

\item{low, high, target}{Single numeric values that define the active ranges of
desirability.}

\item{scale, scale_low, scale_high}{A single numeric value to rescale the
desirability function (each should be great than 0.0). Values >1.0 make the
desirability more difficult to satisfy while smaller values make it easier
(see the examples below). \code{scale_low} and  \code{scale_high} do the same for
target functions with \code{scale_low} affecting the range below the \code{target}
value and \code{scale_high} affecting values greater than \code{target}.}

\item{missing}{A single numeric value on \verb{[0, 1]} (or \code{NA_real_}) that
defines how missing values in \code{x} are mapped to the desirability score.}

\item{use_data}{Should the low, middle, and/or high values be derived from
the data (\code{x}) using the minimum, maximum, or median (respectively)?}

\item{x_vals, desirability}{Numeric vectors of the same length that define the
desirability results at specific values of \code{x}. Values below and above the
data in \code{x_vals} are given values of zero and one, respectively.}

\item{categories}{A named list of desirability values that match all
possible categories to specific desirability values. Data that are not
included in \code{categories} are given the value in \code{missing}.}
}
\value{
A numeric vector on \verb{[0, 1]} where larger values are more
desirable.
}
\description{
Desirability functions map some input to a \verb{[0, 1]} scale where zero is
unacceptable and one is most desirable. The mapping depends on the situation.
For example, \code{d_max()} increases desirability with the input while \code{d_min()}
does the opposite. See the plots in the examples to see more examples.

Currently, only the desirability functions defined by Derringer and Suich
(1980) are implemented.
}
\details{
Each function translates the values to desirability on \verb{[0, 1]}.
\subsection{Equations}{
\subsection{Maximization}{
\itemize{
\item \code{data > high}: d = 1.0
\item \code{data < low}: d = 0.0
\item \verb{low <= data <= high}: \eqn{d = \left(\frac{data-low}{high-low}\right)^{scale}}
}
}

\subsection{Minimization}{
\itemize{
\item \code{data > high}: d = 0.0
\item \code{data < low}: d = 1.0
\item \verb{low <= data <= high}: \eqn{d = \left(\frac{data = low}{low - high}\right)^{scale}}
}
}

\subsection{Target}{
\itemize{
\item \code{data > high}: d = 0.0
\item \code{data < low}: d = 0.0
\item \verb{low <= data <= target}: \eqn{d = \left(\frac{data - low}{target - low}\right)^{scale\_low}}
\item \verb{target <= data <= high}: \eqn{d = \left(\frac{data - high}{target - high}\right)^{scale\_high}}
}
}

\subsection{Minimization}{
\itemize{
\item \code{data > high}: d = 0.0
\item \code{data < low}: d = 0.0
\item \verb{low <= data <= high}:d = 1.0
}
}

\subsection{Categories}{
\itemize{
\item \code{data = level}: d = 1.0
\item \code{data != level}: d = 0.0
}
}

\subsection{Custom}{

For the sequence of values given to the function, \code{d_custom()} will return
the desirability values that correspond to data matching values in \code{x_vals}.
Otherwise, linear interpolation is used for values in-between.
}

}

\subsection{Data-Based Values}{

By default, most of the \verb{d_*()} functions require specific user inputs for
arguments such as \code{low}, \code{target} and \code{high}. When \code{use_data = TRUE}, the
functions can use the minimum, median, and maximum values of the existing
data to estimate those values (respectively) but \emph{only when users do not
specify them}.
}
}
\examples{
\dontshow{if (rlang::is_installed(c("dplyr", "ggplot2"))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(dplyr)
library(ggplot2)

set.seed(1)
dat <- tibble(x = sort(runif(30)), y = sort(runif(30)))
d_max(dat$x[1:10], 0.1, 0.75)

dat \%>\%
  mutate(d_x = d_max(x, 0.1, 0.75))

set.seed(2)
tibble(z = sort(runif(100))) \%>\%
  mutate(
    no_scale = d_max(z, 0.1, 0.75),
    easier   = d_max(z, 0.1, 0.75, scale = 1/2)
  ) \%>\%
  ggplot(aes(x = z)) +
  geom_point(aes(y = no_scale)) +
  geom_line(aes(y = no_scale), alpha = .5) +
  geom_point(aes(y = easier), col = "blue") +
  geom_line(aes(y = easier), col = "blue", alpha = .5) +
  lims(x = 0:1, y = 0:1) +
  coord_fixed() +
  ylab("Desirability")

# ------------------------------------------------------------------------------
# Target example

dat \%>\%
  mutate(
    triangle = d_target(x, 0.1, 0.5, 0.9, scale_low = 2, scale_high = 1/2)
  ) \%>\%
  ggplot(aes(x = x, y = triangle)) +
  geom_point() +
  geom_line(alpha = .5) +
  lims(x = 0:1, y = 0:1) +
  coord_fixed() +
  ylab("Desirability")

# ------------------------------------------------------------------------------
# Box constraints

dat \%>\%
  mutate(box = d_box(x, 1/4, 3/4)) \%>\%
  ggplot(aes(x = x, y = box)) +
  geom_point() +
  geom_line(alpha = .5) +
  lims(x = 0:1, y = 0:1) +
  coord_fixed() +
  ylab("Desirability")

# ------------------------------------------------------------------------------
# Custom function

v_x <- seq(0, 1, length.out = 20)
v_d <- 1 - exp(-10 * abs(v_x - .5))

dat \%>\%
  mutate(v = d_custom(x, v_x, v_d)) \%>\%
  ggplot(aes(x = x, y = v)) +
  geom_point() +
  geom_line(alpha = .5) +
  lims(x = 0:1, y = 0:1) +
  coord_fixed() +
  ylab("Desirability")

# ------------------------------------------------------------------------------
# Qualitative data

set.seed(3)
groups <- sort(runif(10))
names(groups) <- letters[1:10]

tibble(x = letters[1:7]) \%>\%
  mutate(d = d_category(x, groups)) \%>\%
  ggplot(aes(x = x, y = d)) +
  geom_bar(stat = "identity") +
  lims(y = 0:1) +
  ylab("Desirability")

# ------------------------------------------------------------------------------
# Apply the same function to many columns at once (dplyr > 1.0)

dat \%>\%
  mutate(across(c(everything()), ~ d_min(., .2, .6), .names = "d_{col}"))

# ------------------------------------------------------------------------------
# Using current data

set.seed(9015)
tibble(z = c(0, sort(runif(20)), 1)) \%>\%
  mutate(
    user_specified = d_max(z, 0.1, 0.75),
    data_driven   = d_max(z, use_data = TRUE)
  ) \%>\%
  ggplot(aes(x = z)) +
  geom_point(aes(y = user_specified)) +
  geom_line(aes(y = user_specified), alpha = .5) +
  geom_point(aes(y = data_driven), col = "blue") +
  geom_line(aes(y = data_driven), col = "blue", alpha = .5) +
  lims(x = 0:1, y = 0:1) +
  coord_fixed() +
  ylab("Desirability")
\dontshow{\}) # examplesIf}
}
\references{
Derringer, G. and Suich, R. (1980), Simultaneous Optimization of
Several Response Variables. \emph{Journal of Quality Technology}, 12, 214-219.
}
\seealso{
\code{\link[=d_overall]{d_overall()}}
}
