% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_models.R
\name{fit_one_layer}
\alias{fit_one_layer}
\title{MCMC sampling for one layer GP}
\usage{
fit_one_layer(
  x,
  y,
  nmcmc = 10000,
  trace = TRUE,
  g_0 = 0.01,
  theta_0 = 0.5,
  true_g = NULL
)
}
\arguments{
\item{x}{vector or matrix of input locations}

\item{y}{vector of response values}

\item{nmcmc}{number of MCMC iterations}

\item{trace}{logical indicating whether to print iteration progress}

\item{g_0}{initial value for \code{g}}

\item{theta_0}{initial value for \code{theta}}

\item{true_g}{if true nugget is known it may be specified here (set to a small
value to make fit deterministic)}
}
\value{
a list of the S3 class "\code{gp}" with elements:
\itemize{
  \item \code{x}: copy of input matrix
  \item \code{y}: copy of response vector
  \item \code{nmcmc}: number of MCMC iterations
  \item \code{g}: vector of MCMC samples for \code{g}
  \item \code{theta}: vector of MCMC samples for \code{theta}
  \item \code{time}: computation time in seconds
}
}
\description{
Conducts MCMC sampling of hyperparameters for a one layer 
    GP.  Covariance structure is based on inverse exponentiated squared 
    euclidean distance with length scale parameter "\code{theta}" governing 
    the strength of the correlation and nugget parameter "\code{g}" governing 
    noise.
}
\details{
Utilizes Metropolis Hastings sampling of the length scale and
    nugget parameters with a  uniform proposal function (ranging from half 
    to twice the previous iteration) and the following priors:
    \itemize{
        \item \code{prior(g) <- dgamma(g, shape = 1.5, rate = 3.9)}
        \item \code{prior(theta) <- dgamma(theta, shape = 1.5, rate = 3.9/4)}
    }
    These priors are designed for "\code{x}" scaled to [0,1] and "\code{y}" 
    scaled to have mean zero and variance 1.  The output object of class 
    "\code{gp}" is designed for use with \code{continue}, \code{trim}, and 
    \code{predict}.
}
\examples{
# Toy example (runs in less than 5 seconds) -----------------------------------
# This example uses a small number of MCMC iterations in order to run quickly
# More iterations are required to get appropriate fits
# Function defaults are recommended (see additional example below)

f <- function(x) {
  if (x <= 0.4) return(-1)
  if (x >= 0.6) return(1)
  if (x > 0.4 & x < 0.6) return(10*(x-0.5))
}
x <- seq(0.05, 0.95, length = 7)
y <- sapply(x, f)
x_new <- seq(0, 1, length = 100)

# Fit model and calculate EI
fit <- fit_one_layer(x, y, nmcmc = 500)
fit <- trim(fit, 400)
fit <- predict(fit, x_new, lite = FALSE)
ei <- EI(fit)

\donttest{
# One Layer and EI ------------------------------------------------------------

f <- function(x) {
  sin(5 * pi * x) / (2 * x) + (x - 1) ^ 4
}
  
# Training data
x <- seq(0.5, 2, length = 30)
y <- f(x) + rnorm(30, 0, 0.01)
  
# Testing data
xx <- seq(0.5, 2, length = 100)
yy <- f(xx)
  
# Standardize inputs and outputs
xx <- (xx - min(x)) / (max(x) - min(x))
x <- (x - min(x)) / (max(x) - min(x))
yy <- (yy - mean(y)) / sd(y)
y <- (y - mean(y)) / sd(y)
  
# Conduct MCMC
fit <- fit_one_layer(x, y, nmcmc = 10000)
plot(fit) # investigate trace plots
fit <- trim(fit, 8000, 2)
  
# Predict and calculate EI
fit <- predict(fit, xx, lite = FALSE)
ei <- EI(fit)
  
# Visualize Fit
plot(fit)
par(new = TRUE) # overlay EI
plot(xx, ei$value, type = 'l', lty = 2, axes = FALSE, xlab = '', ylab = '')

# Select next design point
x_new <- xx[which.max(ei$value)]

# Evaluate fit
rmse(yy, fit$mean) # lower is better
score(yy, fit$mean, fit$Sigma) # higher is better
}

}
\references{
Gramacy, RB. \emph{Surrogates: Gaussian Process Modeling, Design, and Optimization 
    for the Applied Sciences}. Chapman Hall, 2020.
}
