#' ctStanPlotPost
#'
#' Plots prior and posterior distributions of model parameters in a ctStanModel or ctStanFit object.
#' 
#' @param obj fit or model object as generated by \code{\link{ctStanFit}},
#' \code{\link{ctModel}}, or \code{\link{ctStanModel}}.
#' @param rows vector of integers denoting which rows of obj$setup$popsetup to plot priors for. 
#' Character string 'all' plots all rows with parameters to be estimated. 
#' @param npp Integer number of parameters to show per page.
#' @param priorwidth if TRUE, plots will be scaled to show bulk of both the prior 
#' and posterior distributions. If FALSE, scale is based only on the posterior.
#' @param smoothness Positive numeric -- multiplier to modify smoothness of density plots, higher is smoother but
#' can cause plots to exceed natural boundaries, such as standard deviations below zero.
#' @param wait If true, user is prompted to continue before plotting next graph.  
#' If false, graphs are plotted one after another without waiting.
#' @param plot Logical, if FALSE, ggplot objects are returned in a list instead of plotting.
#' @examples
#' ctStanPlotPost(ctstantestfit, rows=3:4)
#' @export

ctStanPlotPost<-function(obj, rows='all', npp=6,priorwidth=TRUE, smoothness=1,
  plot=TRUE,wait=FALSE){
  
  if(!priorwidth) message('priorwidth argument temporarily unavailable sorry...')
  if(!(class(obj) %in% c('ctStanFit','ctStanModel'))) stop('not a ctStanFit or ctStanModel object!')
  plots <- list()
  densiter <- 1e5
  popsetup <- obj$setup$popsetup
  popvalues<- obj$setup$matvalues[obj$setup$matsetup$when==0 & obj$setup$matsetup$param > 0,]
  
  # paroriginal<-graphics::par()[c('mfrow','mgp','mar')]
  
  # do.call(graphics::par,parcontrol)
  
  e<-extract(obj)
  
  if(rows[1]=='all') rows<-which(!duplicated(obj$setup$popsetup$parname))
  nplots<-ceiling(length(rows) /4)
  if(1==99) Par.Value <- type <- quantity <- Density <- NULL
  # if(all(mfrow=='auto')) {
  #   mfrow <- grDevices::n2mfrow( (length(rows)+sum(as.logical(obj$setup$popsetup$indvarying[rows]))*2))
  #   mfrow[mfrow > 3] <- 3
  # }
  # 
  # if(any(mfrow!=par()$mfrow)) graphics::par(mfrow=mfrow)
  
  # nsubjects<-obj$data$nsubjects 
  quantity <- c('Posterior','Prior')
  for(ploti in 1:nplots){
    dat <- data.table(quantity='',Par.Value=0, Density=0,type='',param='')
    for(ri in if(length(rows) > 1) rows[as.integer(cut_number(rows,nplots))==ploti] else rows){
      pname <- obj$setup$popsetup$parname[ri]
      pari <- obj$setup$popsetup[ri,'param']
      rawpopmeans<- e$rawpopmeans[,pari]
      param<-rawpopmeans
      popmeanspost<-tform(param,popsetup$transform[ri],popvalues$multiplier[ri], popvalues$meanscale[ri],popvalues$offset[ri], popvalues$inneroffset[ri])
      
      param<-stats::rnorm(densiter,0,1)
      meanprior <- tform(param,popsetup$transform[ri],popvalues$multiplier[ri], popvalues$meanscale[ri],popvalues$offset[ri], popvalues$inneroffset[ri])

      dens <- ctDensityList(list(popmeanspost, meanprior),probs=c(.05,.95),plot=FALSE)
      quantity <- c('Posterior','Prior')
      for(i in 1:length(dens$density)){
        dat <- rbind(dat,data.table(quantity=quantity[i],Par.Value=dens$density[[i]]$x,
          Density=dens$density[[i]]$y, type='Pop. Mean',param=pname))
      }
      
      
      if(obj$setup$popsetup[ri,'indvarying']>0){ #then also plot sd and subject level pars
        sdscale <- popvalues[ri,'sdscale']
        sdtform <- gsub('.*', '*',obj$ctstanmodel$rawpopsdtransform,fixed=TRUE)

        rawpopsd <- e$rawpopsd[,popsetup$indvarying[ri]] #c(eval(parse(text=sdtform)) * ifelse(!is.null(obj$standata$varreg),exp(e$varregbase),1))
        
        param<-stats::rnorm(densiter,rawpopmeans,rawpopsd)
        subjectprior<-tform(param,popsetup$transform[ri],popvalues$multiplier[ri], popvalues$meanscale[ri],popvalues$offset[ri], popvalues$inneroffset[ri])
        
        if(!obj$data$intoverpop) {
          rawindparams<-e$baseindparams[,popsetup$indvarying[ri],,drop=FALSE] *
            rawpopsd + rawpopmeans
          param<-rawindparams
          indparamspost<-tform(param,popsetup$transform[ri],popvalues$multiplier[ri], popvalues$meanscale[ri],popvalues$offset[ri], popvalues$inneroffset[ri])

          dens <- ctDensityList(list(indparamspost, subjectprior),probs=c(.01,.99),plot=FALSE)
          quantity <- c('Posterior','Prior')
          for(i in 1:length(dens$density)){
            dat <- rbind(dat,data.table(quantity=quantity[i],Par.Value=dens$density[[i]]$x,
              Density=dens$density[[i]]$y, type='Subject Params',param=pname))
          }
        }
        
        rawpopsdbase<-  stats::rnorm(densiter,0,1)
        rawpopsdprior <- c(eval(parse(text=sdtform))  * sdscale)#rawpopsd prior samples
        hsdpost <- e$popsd[,popsetup$param[ri]]
        
        param<-suppressWarnings(rawpopmeans+rawpopsdprior)
        high<-tform(param,popsetup$transform[ri],popvalues$multiplier[ri], popvalues$meanscale[ri],popvalues$offset[ri], popvalues$inneroffset[ri])
        param<-suppressWarnings(rawpopmeans-rawpopsdprior)
        low<-tform(param,popsetup$transform[ri],popvalues$multiplier[ri], popvalues$meanscale[ri],popvalues$offset[ri], popvalues$inneroffset[ri])
        hsdprior<-abs(high - low)/2
        
        dens <- ctDensityList(list(hsdpost, hsdprior),probs=c(.05,.95),plot=FALSE)
        for(i in 1:length(dens$density)){
          dat <- rbind(dat,data.table(quantity=quantity[i],Par.Value=dens$density[[i]]$x,
            Density=dens$density[[i]]$y, type='Pop. SD',param=pname))
        }
        
      }
    }
    
    
    dat <- dat[-1,]
    # dat[,xlow := quantile(Par.Value,.3),by=list(quantity,type,param)]
    # dat[,xhigh := quantile(Par.Value,.7),by=list(quantity,type,param)]
    # dat<-dat[Par.Value <= (xhigh+(xhigh-xlow)*2),]
    # dat<-dat[Par.Value >= (xlow-(xhigh-xlow)*2),]

      plots<-c(plots,list(
      ggplot(dat,aes(x=Par.Value,fill=quantity,ymax=Density,y=Density) )+
        geom_line(alpha=.3) +
        geom_ribbon(alpha=.4,ymin=0) +
        scale_fill_manual(values=c('red','blue')) +
        theme_minimal()+
        theme(legend.title = element_blank(),
          panel.grid.minor = element_line(size = 0.1), panel.grid.major = element_line(size = .2),
          strip.text.x = element_text(margin = margin(.01, 0, .01, 0, "cm"))) +
        facet_wrap(vars(type,param),scales='free')
    ))
  }
  
  if(plot) {
    firstplot=TRUE
    lapply(plots,function(x){
      if(wait && !firstplot) readline("Press [return] for next plot.")
      firstplot <<- FALSE
      suppressWarnings(print(x))
    })
    return(invisible(NULL))
  } else return(plots)
  # do.call(graphics::par,paroriginal)
  
}

