% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_cors.R
\name{calculate_cors}
\alias{calculate_cors}
\alias{calculate_cors.explainer}
\alias{calculate_cors.matrix}
\alias{calculate_cors.table}
\alias{calculate_cors.default}
\title{Calculate correlation coefficients}
\usage{
calculate_cors(
  x,
  num_num_f = NULL,
  num_cat_f = NULL,
  cat_cat_f = NULL,
  max_cor = NULL
)

\method{calculate_cors}{explainer}(
  x,
  num_num_f = NULL,
  num_cat_f = NULL,
  cat_cat_f = NULL,
  max_cor = NULL
)

\method{calculate_cors}{matrix}(
  x,
  num_num_f = NULL,
  num_cat_f = NULL,
  cat_cat_f = NULL,
  max_cor = NULL
)

\method{calculate_cors}{table}(
  x,
  num_num_f = NULL,
  num_cat_f = NULL,
  cat_cat_f = NULL,
  max_cor = NULL
)

\method{calculate_cors}{default}(
  x,
  num_num_f = NULL,
  num_cat_f = NULL,
  cat_cat_f = NULL,
  max_cor = NULL
)
}
\arguments{
\item{x}{object used to select method. See more below.}

\item{num_num_f}{A \code{function} used to determine correlation coefficient between a pair of numeric variables}

\item{num_cat_f}{A \code{function} used to determine correlation coefficient between a pair of numeric and categorical variable}

\item{cat_cat_f}{A \code{function} used to determine correlation coefficient between a pair of categorical variables}

\item{max_cor}{A number used to indicate absolute correlation (like 1 in \code{cor}). Must be supplied if any of \code{*_f} arguments is supplied.}
}
\value{
A symmetrical \code{matrix} A of size n x n, where n - amount of columns in \code{x} (or dimensions for \code{table}).
The value at A(i,j) is the correlation coefficient between ith and jth variable.
On the diagonal, values from \code{max_cor} are set.
}
\description{
Calculate correlation coefficients between variables in a \code{data.frame}, \code{matrix} or \code{table} 
using 3 different functions for 3 different possible pairs of vairables:
\itemize{
\item{numeric - numeric}
\item{numeric - categorical}
\item{categorical - categorical}}
}
\section{X argument}{


When \code{x} is a \code{data.frame}, all columns of \code{numeric} type are treated as numeric variables and all columns of \code{factor} type are treated as categorical variables. Columns of other types are ignored.

When \code{x} is a \code{matrix}, it is converted to \code{data.frame} using \code{\link{as.data.frame.matrix}}.

When \code{x} is a \code{explainer}, the tests are performed on its \code{data} element.

When \code{x} is a \code{table}, it is treated as contingency table. Its dimensions must be named, but none of them may be named \code{Frequency}.
}

\section{Default functions}{


By default, the function calculates p_value of statistical tests ( \code{\link[stats]{cor.test}} for 2 \code{numeric}, \code{\link[stats]{chisq.test}} for \code{factor} and \code{\link[stats]{kruskal.test}} for mixed).

Then, the correlation coefficients are calculated as \code{-log10(p_value)}. Any results above 100 are treated as absolute correlation and cut to 100. 

The results are then divided by 100 to fit inside [0,1].

If only \code{numeric} data was supplied, the function used is \code{\link[stats]{cor.test}}.
}

\section{Custom functions}{


Creating consistent measures for correlation coefficients, which are comparable for different kinds of variables, is a non-trivial task.
Therefore, if user wishes to use custom function for calculating correlation coefficients, he must provide \strong{all} necessary functions.
Using a custom function for one case and a default for the other is consciously not supported. 
Naturally, user may supply copies of default functions at his own responsibility.

Function \code{calculate_cors} chooses, which parameters of \code{*_f} are required based on data supported. 
For example, for a \code{matrix} with \code{numeric} data only \code{num_num_f} is required. 
On the other hand, for a \code{table} only \code{cat_cat_f} is required.

All \code{*_f} parameters must be functions, which accept 2 parameters (\code{numeric} or \code{factor} vectors respectively)
and return a single number from [0,max_num]. The \code{num_cat_f} must accept \code{numeric} argument as first and \code{factor} argument as second.
}

\examples{

data(mtcars)
# Make sure, that categorical variables are factors
mtcars$vs <- factor(mtcars$vs, labels = c('V-shaped', 'straight'))
mtcars$am <- factor(mtcars$am, labels = c('automatic', 'manual'))
calculate_cors(mtcars)

# For a table:
data(HairEyeColor)
calculate_cors(HairEyeColor)

# Custom functions:
num_mtcars <- mtcars[,-which(colnames(mtcars) \%in\% c('vs', 'am'))]
my_f <- function(x,y) cor.test(x, y, method = 'spearman', exact=FALSE)$estimate
calculate_cors(num_mtcars, num_num_f = my_f, max_cor = 1)

}
\seealso{
\code{\link[stats]{cor.test}}, \code{\link[stats]{chisq.test}}, \code{\link[stats]{kruskal.test}}
}
