% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scale_process_mirai.R
\name{par_grid_mirai}
\alias{par_grid_mirai}
\title{Parallelize spatial computation over the computational grids}
\usage{
par_grid_mirai(grids, fun_dist, ..., pad_y = FALSE, .debug = TRUE)
}
\arguments{
\item{grids}{List of two sf/SpatVector objects. Computational grids.
It takes a strict assumption that the grid input is
an output of \code{par_pad_grid}.}

\item{fun_dist}{\code{sf}, \code{terra} or \code{chopin} functions.
This function should have \code{x} and \code{y} arguments.}

\item{...}{Arguments passed to the argument \code{fun_dist}.}

\item{pad_y}{logical(1). Whether to filter y with the padded grid.
Should be TRUE when x is where the values are calculated.
Default is \code{FALSE}. In the reverse case, like \code{terra::extent} or
\code{exactextractr::exact_extract}, the raster (x) extent should be set
with the padded grid.}

\item{.debug}{logical(1). Default is \code{FALSE}. Otherwise,
if a unit computation fails, the error message and the \code{CGRIDID}
value where the error occurred will be included in the output.}
}
\value{
a data.frame object with computation results.
For entries of the results, consult the documentation of the function put
in \code{fun_dist} argument.
}
\description{
\link[mirai:daemons]{mirai::daemons} will set the parallel backend then \link[mirai:mirai_map]{mirai::mirai_map}
will parallelize the work in each grid. For details of the terminology
in \code{mirai} package, refer to \code{\link[mirai:mirai]{mirai::mirai}}. This function assumes that
users have one raster file and a sizable and spatially distributed
target locations. Each thread will process
the nearest integer of $|N_g| / |N_t|$ grids
where $|N_g|$ denotes the number of grids and $|N_t|$ denotes
the number of threads.
}
\note{
In dynamic dots (\code{...}), \code{fun_dist} arguments should include
x and y where sf/terra class objects or file paths are accepted.
Virtually any sf/terra functions that accept two arguments
can be put in \code{fun_dist}; however be advised that
some spatial operations do not necessarily give the
exact result from what would have been done with one thread.
For example, distance calculated through this function may return the
lower value than actual because the computational region was reduced.
This would be the case especially where the target features
are spatially sparsely distributed.
}
\examples{
\donttest{
lastpar <- par(mfrow = c(1, 1))
library(sf)
library(mirai)
options(sf_use_s2 = FALSE)
daemons(4, dispatcher = "process")
ncpath <- system.file("shape/nc.shp", package = "sf")
ncpoly <- sf::st_read(ncpath)
ncpoly <- sf::st_transform(ncpoly, "EPSG:5070")

# sf object
ncpnts <-
  sf::st_sample(ncpoly, 2000)
ncpnts <- sf::st_as_sf(ncpnts)
ncpnts$pid <- seq_len(nrow(ncpnts))

# file path
rrast <- terra::rast(ncpoly, nrow = 600, ncol = 1320)
terra::values(rrast) <- rgamma(7.92e5, 4, 2)
# Using raster path
rastpath <- file.path(tempdir(), "ncelev.tif")
terra::writeRaster(rrast, rastpath, overwrite = TRUE)

nccompreg <-
  chopin::par_pad_grid(
    input = ncpnts,
    mode = "grid",
    nx = 4L,
    ny = 2L,
    padding = 5e3L
  )
res <-
  par_grid_mirai(
    grids = nccompreg,
    fun_dist = extract_at,
    x = rastpath,
    y = ncpnts,
    qsegs = 90L,
    radius = 5e3L,
    id = "pid"
  )
mirai::daemons(0L)
par(lastpar)
}
}
\seealso{
\code{\link[mirai:daemons]{mirai::daemons}}, \code{\link[mirai:mirai_map]{mirai::mirai_map}}, \code{\link{par_convert_f}}

Other Parallelization: 
\code{\link{par_cut_coords}()},
\code{\link{par_grid}()},
\code{\link{par_hierarchy}()},
\code{\link{par_hierarchy_mirai}()},
\code{\link{par_make_grid}()},
\code{\link{par_merge_grid}()},
\code{\link{par_multirasters}()},
\code{\link{par_multirasters_mirai}()},
\code{\link{par_pad_balanced}()},
\code{\link{par_pad_grid}()},
\code{\link{par_split_list}()}
}
\author{
Insang Song \email{geoissong@gmail.com}
}
\concept{Parallelization}
