/*
This C++ code library includes routines for efficiently working with huge phylogenetic trees in R (using the Rcpp interface).

Most code supports multifurcating trees, as well as trees containing monofurcations (i.e. some nodes having only one child).
In most cases, the input tree must be rooted.
The library is meant for large (>100,000 tips) trees, structured in the conventional "phylo" format in R.
The computational complexity of most routines is O(Nedges).

In the R "phylo" format, the tree topology (ignoring branch lengths) is encoded in a 2D array edge[,] of size Nedges x 2, where tree$edge[e,:] encodes the e-th edge, tree$edge[e,1] --> tree$edge[e,2], with tree$edge[e,1] and tree$edge[e,2] being indices in 1:(Ntips+Nnodes).
A formal definition of the "phylo" format can be found here (section 4): http://ape-package.ird.fr/misc/FormatTreeR.pdf
Note that in C++ (including this code) indices are zero-based (in contrast to R). 
Hence, tips are indexed 0..(Ntips-1) and nodes are indexed 0..(Nnodes-1), and edge[,] has values in 0,..,(Ntips+Nnodes-1)
All arrays passed to this code must be flattened and stored in row-major-format.

General terminology and indexing conventions for trees, as used below:
	A 'node' is always an internal node.
	Any index called 'node' will be within 0:(Nnodes-1)
	Any index called 'tip' will be within 0:(Ntips-1)
	Any index called 'parent' and 'child' will be within 0:(Ntips+Nnodes-1)
	Any index called 'edge' will be within 0:(Nedges-1)
	Any index called 'root' will be within Ntips:(Ntips+Nnodes-1)
	tree_edge[] will always be of size Nedge x 2 (flattened in row-major-format), with entries in 0:(Ntips+Nnodes-1)
	edge_length[] will always be of length Nedges


Stilianos Louca
*/

#include <new>
#include <limits>
#include <cmath>
#include <string>
#include <iostream>
#include <sstream>
#include <fstream>
#include <cstdlib>
#include <complex>
#include <algorithm>
#include <Rcpp.h>
#include <time.h>
#include <ctype.h>
#include <sys/time.h>
#include "STMathExpression.h"

#ifndef INFTY_D
#define INFTY_D numeric_limits<double>::infinity()
#endif

#ifndef NAN_D
#define NAN_D std::numeric_limits<double>::quiet_NaN()
#endif

#ifndef RELATIVE_EPSILON
#define RELATIVE_EPSILON 1e-10
#endif

#ifndef STRANDOM_EPSILON 
#define STRANDOM_EPSILON 1e-30
#endif


#ifdef __MACH__ 
	// headers specific to Mac OS X to substitute for stuff available in Linux, for timing functions
	#include <sys/types.h> 
	#include <mach/mach_time.h> // needed for high-resolution monotonic timer
	
	#include <mach/mach_init.h>
	#include <mach/thread_act.h>
	#include <mach/mach_port.h>
#endif

#if defined(WIN32) || defined(_WIN32) || defined(__WIN32__) || defined(__TOS_WIN__)
	#ifndef IS_WINDOWS
		#define IS_WINDOWS
	#endif
#endif


typedef std::complex<double> cdouble;
typedef std::vector<double> dvector;
typedef std::vector<long> lvector;

using namespace Rcpp;
using namespace std;

// defines outcome of request for vector field during simulations
typedef enum {
	RequestedDynamicsRateOfChange,
	RequestedDynamicsForceJumpToState,
	RequestedDynamicsInvalidState
} RequestedDynamics;

// define answers to question whether simulation crossed the domain boundary
typedef enum {
	CrossedBoundaryYesButFixedByReducingTimeStep,
	CrossedBoundaryYesButFixedBruteForce,
	CrossedBoundaryNo
} CrossedBoundary;


typedef enum _MathError{
	MathErrorNone = 0,
	MathErrorNoData = -1,
	MathErrorInvalidData = -2,
	MathErrorUndefined = -2,
	MathErrorNoSolution = -3,
	MathErrorDivisionByZero = -4,
	MathErrorUnknownError = -5,
	MathErrorMaxIterationsReached = -6,
	MathErrorBadStartingPoint = -7,
	MathErrorBadBounds = -8
} MathError;


typedef enum {
	ExtrapolationTypeNaN,
	ExtrapolationTypeConst,
	ExtrapolationTypeLinear,
	ExtrapolationTypePolynomial,
	ExtrapolationTypePeriodic
} ExtrapolationType;



typedef enum {
	EdgeLengthUnitTime,
	EdgeLengthUnitGenerations,
	EdgeLengthUnitMutationsExpected,
	EdgeLengthUnitMutationsRandom
} EdgeLengthUnit;


typedef enum {
	SBMTransitionDensityAngular,
	SBMTransitionDensityAxial,
	SBMTransitionDensitySurface
} SBMTransitionDensity;


typedef enum {
	CladeTypeNode,
	CladeTypeTip,
	CladeTypeExtantTip,
	CladeTypeExtinctTip,
	CladeTypeSampledTip,
	CladeTypeSampledRemovedLineage,
	CladeTypeSampledRetainedLineage,
} CladeType;



// ****************************************************** //
// BASIC AUXILIARY FUNCTIONS



#pragma mark -
#pragma mark Functions for debugging
#pragma mark 

// print a 2D matrix (in row-major format) as a python list-of-lists
// Mainly used for debugging
template<class TYPE>
void print_as_python_matrix(const long NR, const long NC, const TYPE A[]){
	Rcout << "[";
	for(long r=0; r<NR; ++r){
		Rcout << (r>0 ? ", [" : "[");
		for(long c=0; c<NC; ++c){
			Rcout << (c>0 ? ", " : "") << A[r*NC+c];
		}
		Rcout << "]";
	}
	Rcout << "]\n";
}


template<class ARRAY_TYPE>
void print_as_python_matrix(const long NR, const long NC, const ARRAY_TYPE &A){
	Rcout << "[";
	for(long r=0; r<NR; ++r){
		Rcout << (r>0 ? ", [" : "[");
		for(long c=0; c<NC; ++c){
			Rcout << (c>0 ? ", " : "") << A[r*NC+c];
		}
		Rcout << "]";
	}
	Rcout << "]\n";
}

template<class ARRAY_TYPE>
void print_as_matrix(const long NR, const long NC, const ARRAY_TYPE &A){
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			Rcout << (c>0 ? ", " : "") << A[r*NC+c];
		}
		Rcout << "\n";
	}
	Rcout << "\n";
}


template<class TYPE>
void print_as_matrix(const long NR, const long NC, const TYPE A[]){
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			Rcout << (c>0 ? ", " : "") << A[r*NC+c];
		}
		Rcout << "\n";
	}
	Rcout << "\n";
}


template<class ARRAY_TYPE>
void print_as_matrix_column_major(const long NR, const long NC, const ARRAY_TYPE &A){
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			Rcout << (c>0 ? ", " : "") << A[r + c*NR];
		}
		Rcout << "\n";
	}
	Rcout << "\n";
}




#pragma mark -
#pragma mark Auxiliary functions
#pragma mark 



// returns the number of seconds since an arbitrary (but consistent) point in the past
// The timer is monotonic (i.e. not affected by manual changes in the system time), and is specific to the calling thread
// Note that for Windows this function is not thread-specific 
double get_thread_monotonic_walltime_seconds(){
	#if __MACH__ 
		mach_timebase_info_data_t info;
		int error_code = mach_timebase_info(&info);
		if (error_code != KERN_SUCCESS) return 0.0;
		return 1e-9 * mach_absolute_time() * double(info.numer)/double(info.denom);
	#elif defined(unix) || defined(__unix) || defined(__unix__) || defined(__linux__)
		// POSIX code
		// For details on clock_gettime() see: http://www.tin.org/bin/man.cgi?section=3&topic=clock_gettime
		struct timespec T;
		clock_gettime(CLOCK_MONOTONIC, &T); // Note that CLOCK_MONOTONIC_RAW is not available on all Linux distros
		return double(T.tv_sec) + 1e-9*T.tv_nsec;
	#elif defined(IS_WINDOWS)
		//return GetTickCount()*1e-6; // note that this is not thread-specific, but it's the best you can get on Windows. Update: It requires the windows.h library, which causes problems with Rcpp on CRAN.
		return clock()/double(CLOCKS_PER_SEC);
	#else
		return 0; // not implemented for other systems
	#endif
}




inline double string2Double(const string &number){
	return strtod(number.c_str(), NULL);
}

inline double string2Long(const string &number){
	return strtol(number.c_str(), NULL, 0);
}

template<class TYPE> 
string makeString(const TYPE &data){
	ostringstream stream;
	stream << data;
	return stream.str();
}


// Formated string creation
string vstringprintf(const char *format, va_list args){
	va_list temp;
	va_copy(temp, args);
	char *buffer = new char[vsnprintf(NULL, 0, format, temp) + 1];
	va_end(temp);
	
	vsprintf(buffer, format, args);
	string s(buffer);
	delete [] buffer;
	return s;
}

// Formated string creation
string stringprintf(const char *format, ...){
	string s;
	va_list args;
	va_start(args, format);
	s = vstringprintf(format, args);
	va_end(args);
	return s;
}



string trim_whitespace(const std::string &haystack){
	long right = haystack.length()-1;
	long left = 0;
	while(((haystack[right]==' ') || (haystack[right]=='\t') || (haystack[right]=='\n')) && (right>=0)){
		--right;
	}
	while(((haystack[left]==' ') || (haystack[left]=='\t') || (haystack[left]=='\n')) && (left<right)){
		++left;
	}
	return haystack.substr(left,right-left+1);
}

// count the number of occurrences of a char in a string, optionally restricting counts to those outside of quotes
long count_occurrences(	const std::string	&haystack,
						const char			needle,				// (INPUT) character to count the occurrences of
						const bool			non_quoted_only){	// (INPUT) if true, only occurrences not bracketed by single or double quotes are counted
	long count = 0;
	bool open_single = false, open_double = false;
	for(long i=0; i<haystack.size(); ++i){
		if(haystack[i]==needle){
			if((!non_quoted_only) || ((!open_single) && (!open_double))){
				++count;
			}
		}else if((!open_single) && (haystack[i]=='"')){
			open_double = !open_double;
		}else if((!open_double) && (haystack[i]=='\'')){
			open_single = !open_single;
		}
	}
	return count;		
}


inline bool XOR(bool a, bool b){
	return ((!a) && b) || (a && (!b));
}

inline double SQR(double x){
	return x*x;
}

template<class TYPE>
inline TYPE SQ(TYPE value){
	return value * value;
}

template<class TYPE>
inline TYPE Qube(TYPE value){
	return value * value * value;
}

template<class TYPE>
inline TYPE QTR(TYPE value){
	return SQ(SQ(value));
}

template<class TYPE>
inline int sgn(const TYPE value){
	return (value<0 ? -1 : 1);
}


// given an arithmetic series x1, x1+delta, x1+2*delta, ...
// calculate the sum of the first n elements
inline long sum_arithmetic_series(const long x1, const long delta, const long n){
	return (n*(2*x1+(n-1)*delta))/2;
}


// given a quadratic N x N matrix and some row & column not on the diagonal, return the corresponding flattened index i (where i only enumerates non-diagonal elements in row-major format)
inline long row_col_to_nondiagonal_index(	const long 	N,
											const long 	row,
											const long	col){
	if(row==col) return -1; // invalid request, since element is on the diagonal
	return sum_arithmetic_series(N-1,-1,row)+(col<row ? col : col-1);											
}


// calculate result = a*X + b*Y, for vectors X & Y and scalars a & b
template<class TYPE>
inline void linear_combination(const double a, const std::vector<TYPE> &X, const double b, const std::vector<TYPE> &Y, std::vector<TYPE> &result){
	result.resize(X.size());
	for(long i=0; i<X.size(); ++i) result[i] = a*X[i] + b*Y[i];
}

// calculate result = a*X + b*Y, for scalars X & Y and scalars a & b
// needed for use in some ODE solvers, where states could be simple scalars
inline void linear_combination(const double a, const double X, const double b, const double Y, double &result){
	result = a*X + b*Y;
}


void cast_ComplexVector_to_CPP(	const ComplexVector &A,		// (INPUT)
								vector<cdouble>		&B){	// (OUTPUT)
	B.resize(A.size());
	for(long i=0; i<A.size(); ++i){
		B[i] = cdouble(A[i].r, A[i].i);
	}
}


void set_array_to_value(long X[], long N, long value){
	for(long n=0; n<N; ++n) X[n] = value;
}

void set_array_to_value(double X[], long N, double value){
	for(long n=0; n<N; ++n) X[n] = value;
}


long find_first_non_nan(const std::vector<double> &haystack, const long start){
	for(long n=start; n<haystack.size(); ++n){
		if(!std::isnan(haystack[n])) return n;
	}
	return -1;
}

template<class TYPE1, class TYPE2>
long find_first_occurrence(const std::vector<TYPE1> &haystack, const TYPE2 needle){
	for(long n=0; n<haystack.size(); ++n){
		if(haystack[n]==needle) return n;
	}
	return -1;
}

template<class TYPE1, class TYPE2>
long find_first_non_occurrence(const std::vector<TYPE1> &haystack, const TYPE2 antineedle){
	for(long n=0; n<haystack.size(); ++n){
		if(haystack[n]!=antineedle) return n;
	}
	return -1;
}


// searches for needle in a list of ascending values
template<class TYPE1, class TYPE2>
long find_in_ascending_list(const std::vector<TYPE1> &haystack, const TYPE2 needle, const long start){
	for(long n=start; n<haystack.size(); ++n){
		if(haystack[n]>needle) return -1;
		if(haystack[n]==needle) return n;
	}
	return -1;
}


// find grid point on the immediate left (or equal) to the needle
// this function is most efficient when subsequent calls always ask for non-decreasing needle values, or always ask for non-increasing needle values
// values in grid[] are assumed to be non-decreasing
// Under the above assumptions, the returned index g will be the largest possible g satisfying grid[g]<=needle; if no such index exists (i.e. needle<grid[0]), this function returns -1.
// If you don't know where to start searching, you can set previous_g=-1
long find_next_left_grid_point(const std::vector<double> &grid, const double needle, long previous_g){
	const long N = grid.size();
	if(N==0) return -1;
	if(needle<grid[0]) return -1;
	if(previous_g<0){
		// no first guess provided, so pick one using linear interpolation
		previous_g = max(0l, min(N-1l, long((N-1)*(needle-grid[0])/(grid[N-1]-grid[0]))));
	}
	long g;
	if(grid[previous_g]<=needle){
		// search for grid point, towards the right
		for(g=previous_g; g<N-1; ++g){
			if(grid[g+1] > needle) return g;
		}
		return N-1;
	}else{
		// search for grid point, towards the left
		for(g=previous_g; g>=0; --g){
			if(grid[g] <= needle) return g;
		}
		return -1;
	}
}


long find_next_left_grid_point(const long N, const double grid[], const double needle, long previous_g){
	if(N==0) return -1;
	if(needle<grid[0]) return -1;
	if(previous_g<0){
		// no first guess provided, so pick one using linear interpolation
		previous_g = max(0l, min(N-1l, long((N-1)*(needle-grid[0])/(grid[N-1]-grid[0]))));
	}
	long g;
	if(grid[previous_g]<=needle){
		// search for grid point, towards the right
		for(g=previous_g; g<N-1; ++g){
			if(grid[g+1] > needle) return g;
		}
		return N-1;
	}else{
		// search for grid point, towards the left
		for(g=previous_g; g>=0; --g){
			if(grid[g] <= needle) return g;
		}
		return -1;
	}
}



// similar to find_next_left_grid_point(), but searching for the grid point on the immediate right (or equal) to the needle
// grid[] must be in increasing order
long find_next_right_grid_point(const std::vector<double> &grid, const double needle, long previous_g){
	const long N = grid.size();
	if(N==0) return -1;
	if(needle>grid.back()) return -1;
	if(previous_g<0){
		// no first guess provided, so pick one using linear interpolation
		previous_g = max(0l, min(N-1l, long((N-1)*(needle-grid[0])/(grid[N-1]-grid[0]))));
	}
	long g;
	if(grid[previous_g]<needle){
		// search for grid point, towards the right
		for(g=previous_g; g<N; ++g){
			if(grid[g] >= needle) return g;
		}
		return -1;
	}else{
		// search for grid point, towards the left
		for(g=previous_g; g>=1; --g){
			if(grid[g-1] < needle) return g;
		}
		return 0;
	}
}



template<class ARRAY_TYPE>
void multiply_array_with_scalar(ARRAY_TYPE &X, const double scalar){
	for(long i=0; i<X.size(); ++i) X[i] *= scalar;
}




template<class ARRAY_TYPE>
void reverse_array(ARRAY_TYPE &X){
	const long N = X.size();
	long scratch;
	for(long n=0; n<(N/2.0); ++n){
		scratch = X[n];
		X[n] = X[N-n-1];
		X[N-n-1] = scratch;
	}
}



template<class ARRAY_TYPE>
double array_min(const ARRAY_TYPE &X){
	const long N = X.size();
	if(N==0) return NAN_D;
	double minX = X[0];
	for(long n=0; n<N; ++n){
		if(X[n]<minX) minX = X[n];
	}
	return minX;
}


template<class ARRAY_TYPE>
double array_min(const ARRAY_TYPE &X, const long start, const long end){
	if(end<start) return NAN_D;
	double minX = X[start];
	for(long n=start; n<=end; ++n){
		if(X[n]<minX) minX = X[n];
	}
	return minX;
}


template<class ARRAY_TYPE>
inline double array_nonzero_min(const ARRAY_TYPE &X){
	const long N = X.size();
	double minX = NAN_D;
	for(long n=0; n<N; ++n){
		if((X[n]!=0) && (std::isnan(minX) || (X[n]<minX))) minX = X[n];
	}
	return minX;
}


template<class ARRAY_TYPE>
inline double array_max(const ARRAY_TYPE &X){
	const long N = X.size();
	if(N==0) return NAN_D;
	double maxX = X[0];
	for(long n=0; n<N; ++n){
		if(X[n]>maxX) maxX = X[n];
	}
	return maxX;
}



double array_min(const std::vector<double> &X, long start_index, long end_index){
	if(end_index<start_index) return NAN_D;
	double minX = X[start_index];
	for(long n=start_index; n<=end_index; ++n){
		if(X[n]<minX) minX = X[n];
	}
	return minX;
}


template<class ARRAY_TYPE>
inline long array_argmax(const ARRAY_TYPE &X){
	const long N = X.size();
	if(N==0) return -1;
	long max_n = 0;
	for(long n=1; n<N; ++n){
		if(X[n]>X[max_n]) max_n = n;
	}
	return max_n;
}


template<class TYPE>
inline long array_argmax(const long N, const TYPE X[]){
	if(N==0) return -1;
	long max_n = 0;
	for(long n=1; n<N; ++n){
		if(X[n]>X[max_n]) max_n = n;
	}
	return max_n;
}


template<class ARRAY_TYPE>
inline bool arrays_are_equal(const ARRAY_TYPE &A, const ARRAY_TYPE &B){
	if(A.size()!=B.size()) return false;
	for(long i=0; i<A.size(); ++i){
		if(A[i]!=B[i]) return false;
	}
	return true;
}


template<class TYPE>
inline TYPE vector_sum(const std::vector<TYPE> &values){
	TYPE S = 0;
	for(long i=0; i<values.size(); ++i) S += values[i];
	return S;
}


template<class TYPE>
inline TYPE vector_sum(const std::vector<TYPE> &values, const lvector &indices){
	TYPE S = 0;
	for(long i=0; i<indices.size(); ++i) S += values[indices[i]];
	return S;
}


inline long vector_sum(const std::vector<char> &values){
	long S = 0;
	for(long i=0; i<values.size(); ++i) S += values[i];
	return S;
}


template<class TYPE>
inline TYPE scalar_product(const std::vector<TYPE> &A, const std::vector<TYPE> &B){
	TYPE S = 0;
	for(long i=0; i<min(A.size(),B.size()); ++i) S += A[i] * B[i];
	return S;
}


double smallest_nonzero_step(const std::vector<double> &times){
	double S = INFTY_D;
	for(long i=0; i<times.size(); ++i){
		if(times[i+1]>times[i]){
			S = min(S, times[i+1]-times[i]);
		}
	}
	return S;
}


inline double vector_mean(const double values[], const long N){
	double S = 0;
	for(long i=0; i<N; ++i) S += values[i];
	return (S/N);
}


inline double vector_mean(const std::vector<double> &values){
	double S = 0;
	for(long i=0; i<values.size(); ++i) S += values[i];
	return (S/values.size());
}


inline double vector_abs_mean(const std::vector<double> &values){
	double S = 0;
	for(long i=0; i<values.size(); ++i) S += abs(values[i]);
	return (S/values.size());
}



inline double vector_mean(const std::vector<double> &values, const long first, const long last){
	double S = 0;
	for(long i=first; i<=last; ++i) S += values[i];
	return (S/(last-first+1.0));
}


template<class TYPE>
inline TYPE vector_min(const std::vector<TYPE> &values){
	TYPE A = values[0];
	for(long i=0; i<values.size(); ++i) A = (A>values[i] ? values[i] : A);
	return A;
}

template<class TYPE>
inline TYPE vector_max(const std::vector<TYPE> &values){
	TYPE A = values[0];
	for(long i=0; i<values.size(); ++i) A = (A<values[i] ? values[i] : A);
	return A;
}

inline double vector_max_abs(const std::vector<double> &values){
	double A = 0;
	for(long i=0; i<values.size(); ++i) A = max(A, abs(values[i]));
	return A;
}

// determine the maximum modulus of any non-diagonal entries in a 2D matrix
// The matrix should be of size NR x NC, in row-major format
inline double matrix_nondiagonal_max_abs(const std::vector<double> &matrix, const long NR, const long NC){
	double A = 0;
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			if(r==c) continue;
			A = max(A, abs(matrix[r*NC+c]));
		}
	}
	return A;
}


// calculate sum of a single row in a 2D matrix of size NR x NC
// matrix must be provided in row-major format, i.e. matrix[r*NC+c] is the entry in row r & column c
inline double row_sum(const std::vector<double> &matrix, const long NC, const long row){
	double S = 0;
	for(long c=0; c<NC; ++c){
		S += matrix[row*NC + c];
	}
	return S;
}


// make sure no entry is negative
void make_vector_positive(std::vector<double> &values){
	for(long i=0; i<values.size(); ++i) values[i] = max(0.0, values[i]);
}

// replace negative entries
void replace_negatives(std::vector<double> &values, const double replacement){
	for(long i=0; i<values.size(); ++i){
		if(values[i]<0) values[i] = replacement;
	}
}

// replace non-strictly positive entries
void replace_non_positives(std::vector<double> &values, const double replacement){
	for(long i=0; i<values.size(); ++i){
		if(values[i]<=0) values[i] = replacement;
	}
}


// make sure entries in a vector are within the specified limits [min_value:max_value]
void cap_values(const double 		min_value,
				const double 		max_value,
				std::vector<double> &values){ // (INPUT/OUTPUT) the vector to be modified in-situ
	for(long i=0; i<values.size(); ++i){
		values[i] = max(min_value, min(max_value, values[i]));
	}
}


// extract a specific row from a 2D matrix in row-major format
template<class TYPE>
void extract_row(const std::vector<TYPE> &matrix, const long NC, const long row, std::vector<TYPE> &extracted_row){
	extracted_row.resize(NC);
	for(long c=0; c<NC; ++c){
		extracted_row[c] = matrix[row*NC+c];
	}
}


template<class TYPE>
void flatten_matrix(const std::vector<std::vector<TYPE> > 	&matrix,		// (INPUT) 2D matrix, where matrix[r][c] is the element in row r and column c. The number of rows is assumed to be matrix.size(). The number of columns is determined assumed to be matrix[0].size().
					std::vector<TYPE> 						&flattened){	// (OUTPUT) Flattened 2D matrix in row-major format, where flattened[r*NC + c] is the element in row r and column c.
	const long NR = matrix.size();
	if(NR==0){
		flattened.resize(0);
		return;
	}
	const long NC = matrix[0].size();
	flattened.resize(NR*NC);
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			flattened[r*NC + c] = matrix[r][c];
		}
	}		
}


template<class TYPE>
inline bool contains_nan(const std::vector<TYPE> &values){
	for(long i=0; i<values.size(); ++i){
		if(std::isnan(values[i])) return true;
	}
	return false;
}


template<class TYPE>
inline bool contains_inf(const std::vector<TYPE> &values){
	for(long i=0; i<values.size(); ++i){
		if(std::isinf(values[i])) return true;
	}
	return false;
}

inline long vector_count_zeros(const std::vector<long> &values){
	long S = 0;
	for(long i=0; i<values.size(); ++i) S += (values[i]==0 ? 1 : 0);
	return S;
}


template<class ARRAY_TYPE>
inline long count_values_below_threshold(const ARRAY_TYPE &values, double threshold){
	long N = 0;
	for(long i=0; i<values.size(); ++i){
		if(values[i]<=threshold) ++N;
	}
	return N;
}


// remove an item from a vector, by replacing it with the last item in the list and then removing the last item
// if the order of items in the vector do not matter, then this is more efficient than removing an item from within a vector
// index is assumed to be a valid location in the vector
template<class TYPE>
inline void remove_item_from_vector(std::vector<TYPE> &list, long index){
	if(index==list.size()-1){
		// the item is the last one in the list, or nothing else left in the list, so just remove item
		list.pop_back();
	}else{
		list[index] = list.back();
		list.pop_back();
	}
}

// remove list of items from a vector, by replacing them with the last items in the list and then removing the last items
// if the order of items in the vector do not matter, then this is more efficient than removing the items from within a vector
template<class TYPE>
inline void remove_items_from_vector(std::vector<TYPE> &list, const lvector &indices){
	for(long i=0, index; i<indices.size(); ++i){
		index = indices[i];
		if(index==list.size()-1){
			// the item is the last one in the list, or nothing else left in the list, so just remove item
			list.pop_back();
		}else{
			list[index] = list.back();
			list.pop_back();
		}
	}
}

// remove an item from a list, which is part of a reciprocal mapping list <--> pool
// the item will be removed from list2pool, and the mapping pool2list will be set to -1 for the removed item
// this function should only be used if the order of entries in list2pool does not matter
// this function guarantees that the only entries affected in pool2list[] will be those pointed to by list2pool[] and pool_index
inline void remove_item_from_mapped_list(lvector &list2pool, lvector &pool2list, long pool_index){
	if(pool2list[pool_index]==list2pool.size()-1){
		// the item is the last one in the list, or nothing else left in the list, so just remove item
		list2pool.pop_back();
		pool2list[pool_index] = -1;
	}else{
		// first replace the entry to be removed from list2pool with the tail of the list2pool, then pop the tail
		const long list_index = pool2list[pool_index];
		pool2list[list2pool.back()] = list_index;	
		list2pool[list_index] = list2pool.back();
		list2pool.pop_back();
		pool2list[pool_index] = -1;
	}
}



// get index of item in haystack[] closest to needle
// assumes that haystack is in ascending order
template<class TYPE>
long get_nearest_index(const std::vector<TYPE> &haystack, const TYPE needle){
	for(long i=0; i<(haystack.size()-1); ++i){
		if(abs(needle-haystack[i+1])>abs(needle-haystack[i])) return i;
	}
	return (haystack.size()-1);
}


long get_nearest_index(const std::vector<long> &haystack, const long needle){
	for(long i=0; i<(haystack.size()-1); ++i){
		if(abs(needle-haystack[i+1])>abs(needle-haystack[i])) return i;
	}
	return (haystack.size()-1);
}



// Convert a dense binary matrix from row-major format (i.e. provided as a list of lists of non-zero column entries)
// to column-major format, i.e. return a list of lists of non-zero row entries
// Note that since the matrix is assumed to be binary (0/1), the actual values don't need to be provided and neither are they returned; we only need to keep track of which cells are non-zero
// [[Rcpp::export]]
Rcpp::List dense_binary_matrix_row2column_major_CPP(const long 			NR,
													const long 			NC,
													const Rcpp::List	&dense_rows, 	// (INPUT) list of size NR, with the r-th entry being an integer std::vector of arbitrary size containing the columns with non-zero entries in row r
													const bool			Rindexing){		// (INPUT) if true, then indices listed in dense_rows are 1-based (as opposed to 0-based, as expected in C++), and the indices listed in the returned dense_columns should also be 1-based. If false, then 0-based indexing is used in input and output
	const long index_shift = (Rindexing ? -1 : 0);
	std::vector< std::vector<long> > dense_columns(NC); // dense_columns[c][] will be a list of row indices, so that dense_columns[c][i] is the i-th non-zero entry in column c (matrix[i,c]!=0)
	vector<long> dense_row;
	for(long r=0; r<NR; ++r){
		dense_row = Rcpp::as<vector<long> >(dense_rows[r]);
		for(long i=0; i<dense_row.size(); ++i){
			dense_columns[dense_row[i]+index_shift].push_back(r-index_shift);
		}
	}
	return Rcpp::wrap(dense_columns);
}


template<typename TYPE>
inline void aux_qsortIndices_swap(std::vector<TYPE> &X, long A, long B, TYPE &temp){
	temp = X[A];
	X[A] = X[B];
	X[B] = temp;
}


template<typename TYPE>
long aux_qsortIndices_partition(const std::vector<TYPE> &values, std::vector<long> &indices, long start, long end){
	long temp;
	
	// choose pivot at middle
	const long pi = start+(end-start)/2;
	
	// Alternative: choose pivot via median-of-three rule
	/*
	long pi = start+(end-start)/2;
	if(values[indices[end]] < values[indices[start]]){
		aux_qsortIndices_swap(indices, start, end, temp);
	}
	if(values[indices[pi]] < values[indices[start]]){
		aux_qsortIndices_swap(indices, start, pi, temp);
	}
	if(values[indices[end]] < values[indices[pi]]){
		aux_qsortIndices_swap(indices, end, pi, temp);
	}
	*/
	const TYPE pv = values[indices[pi]];
	
	// swap (put pivot temporarily at end)
	aux_qsortIndices_swap(indices, pi, end, temp);
	
	long si = start;
	for(long i=start; i<end; ++i){
		if((values[indices[i]] < pv) || ((values[indices[i]]==pv) && (i%2==0))){ // modified condition of incrementing partition point, to avoid worst-case-scenarios when all (or many) values are equal
			// swap
			aux_qsortIndices_swap(indices, i, si, temp);
			++si;
		}
	}
	// swap (place pivot onto si)
	aux_qsortIndices_swap(indices, si, end, temp);
	
	return si;
}




template<typename TYPE>
void aux_qsortIndices(const std::vector<TYPE> &values, std::vector<long> &indices, long start, long end){
	if(start<end){
		long p = aux_qsortIndices_partition(values, indices, start, end);
		aux_qsortIndices(values, indices, start, p-1);
		aux_qsortIndices(values, indices, p+1, end);
	}
}


//quick-sort (average order n*log(n)) of indices pointing to values, in ascending order
//sortedIndices[] will contain the original positions of the sorted values in values[], that is
//  values[sortedIndices[0]] <= values[sortedIndices[1]] <= ... 
template<typename TYPE>
void qsortIndices(const std::vector<TYPE> &values, std::vector<long> &sortedIndices){
	sortedIndices.resize(values.size());
	for(long n=0; n<sortedIndices.size(); ++n) sortedIndices[n] = n;
	aux_qsortIndices(values, sortedIndices, 0, sortedIndices.size()-1);
}

// similar to qsortIndices(..) but sorting in descending order
template<typename TYPE>
void qsortIndices_reverse(const std::vector<TYPE> &values, std::vector<long> &sortedIndices){
	// first sort in ascending order
	const long N = values.size();
	sortedIndices.resize(N);
	for(long n=0; n<N; ++n) sortedIndices[n] = n;
	aux_qsortIndices(values, sortedIndices, 0, sortedIndices.size()-1);
	
	// reverse order
	for(long n=0, m, temp; n<1+(N/2); ++n){
		m = N-n-1;
		if(m<=n) break;
		// swap indices n & m
		temp = sortedIndices[n];
		sortedIndices[n] = sortedIndices[m];
		sortedIndices[m] = temp;
	}
}

// same as qsortIndices(..), but only considering a subset of the values[]
template<typename TYPE>
void qsortIndices(const std::vector<TYPE> &values, const std::vector<long> &onlyIndices, std::vector<long> &sortedIndices){
	sortedIndices = onlyIndices;
	aux_qsortIndices(values, sortedIndices, 0, sortedIndices.size()-1);
}


// same as qsortIndices(..), but only considering a subset of the values[], namely those for which includeIndex[i]==true
template<typename TYPE>
void qsortIndices(const std::vector<TYPE> &values, const std::vector<char> &includeIndex, std::vector<long> &sortedIndices){
	sortedIndices.clear();
	sortedIndices.reserve(values.size());
	for(long i=0; i<values.size(); ++i){ if(includeIndex[i]) sortedIndices.push_back(i); }
	aux_qsortIndices(values, sortedIndices, 0, sortedIndices.size()-1);
}


template<typename TYPE, typename TYPE2>
long aux_qsortIndices_partition(const std::vector<TYPE> &values, const std::vector<TYPE2> &values2, std::vector<long> &indices, long start, long end){
	long temp;
	
	// choose pivot at middle
	const long pi 	= start+(end-start)/2;	
	const TYPE pv 	= values[indices[pi]];
	const TYPE2 pv2 = values2[indices[pi]];
	
	// swap (put pivot temporarily at end)
	aux_qsortIndices_swap(indices, pi, end, temp);
	
	long si = start;
	for(long i=start; i<end; ++i){
		if((values[indices[i]]<pv) || ((values[indices[i]]==pv) && (values2[indices[i]]<pv2)) || ((values[indices[i]]==pv) && (values2[indices[i]]==pv2) && (i%2==0))){ // modified condition of incrementing partition point, to avoid worst-case-scenarios when all (or many) values are equal
			// swap
			aux_qsortIndices_swap(indices, i, si, temp);
			++si;
		}
	}
	// swap (place pivot onto si)
	aux_qsortIndices_swap(indices, si, end, temp);
	
	return si;
}

template<typename TYPE, typename TYPE2>
void aux_qsortIndices(const std::vector<TYPE> &values, const std::vector<TYPE2> &values2, std::vector<long> &indices, long start, long end){
	if(start<end){
		long p = aux_qsortIndices_partition(values, values2, indices, start, end);
		aux_qsortIndices(values, values2, indices, start, p-1);
		aux_qsortIndices(values, values2, indices, p+1, end);
	}
}


// same as qsortIndices(..), but including a 2nd set of values as a tie-breaker
// specifically, if values[sortedIndices[i]]==values[sortedIndices[j]] and i<j, then it is guaranteed that values2[sortedIndices[i]]<=values2[sortedIndices[j]]
// values2[] must be the same length as values[]
template<typename TYPE, typename TYPE2>
void qsortIndices(const std::vector<TYPE> &values, const std::vector<TYPE2> &values2, std::vector<long> &sortedIndices){
	sortedIndices.resize(values.size());
	for(long n=0; n<sortedIndices.size(); ++n) sortedIndices[n] = n;
	aux_qsortIndices(values, values2, sortedIndices, 0, sortedIndices.size()-1);
}




//returns the positive (negative) modulo M of division numerator/denominator if denominator>0 (denominator<0)
//that is: 	numerator = D * denominator + M, 
//			whereas M in [0,denominator) if denominator>0 or M in (denominator, 0] if denominator<0
//			and D integer
//For example: 	-5 mod +3 = +1 
//				-5 mod -3 = -2
//				+5 mod -3 = -1
double modulo(double numerator, double denominator){
	if((numerator>=0) && (denominator >= 0)){
		return numerator - denominator * floor(numerator/denominator);
	}else if((numerator < 0) && (denominator < 0)){
		return numerator - denominator * floor(numerator/denominator);
	}else if((numerator < 0) && (denominator > 0)){
		return numerator + denominator * ceil(abs(numerator/denominator));
	}else if((numerator >0) && (denominator < 0)){
		return numerator + denominator * ceil(numerator/abs(denominator));
	}
	return 0; //should not have arrived here
}


//Calculates equivalence class of value modulo L=(intervalMax - intervalMin)
//Returns equivalence class using representants from within the interval [intervalMin, intervalMax]
//Otherwise said: Wrap a number x onto a periodic interval [intervalMin,intervalMax]
//Use for example as moduloInterval(angle, 0, 2*PI) or moduloInterval(angle, -PI, +PI)
inline double moduloInterval(double value, double intervalMin, double intervalMax){
	return modulo(value - intervalMin, intervalMax - intervalMin) + intervalMin;
}

// wrap a number x onto a periodic interval [left,right]
// inline double wrap_to_interval(double x, double left, double right){
// 	if(x>=left){
// 		return left + fmod(x-left,(right-left));
// 	}else{
// 		return right - fmod(right-x,(right-left));
// 	}
// }




// inverse cumulative distribution function of the Student's t distribution with n degrees of freedom
// Returns t such that P(x<=t) = p
// Approximation according to:
//    Abramowitz and Stegun (1970). Handbook of mathematical functions. Page 949
//    Voutier (2010). A New Approximation to the Normal Distribution Quantile Function. arXiv:1002.0567
double quantile_Students_t(double p, long n){
	if(p<0.5) return -quantile_Students_t(1-p,n); // the approximation formula below is only valid for p>0.5 (i.e. q<0.5)
	double q = 1-p;
	const double A = sqrt(-2*log(q));
	//const double xq = A - (2.515517 + 0.802853*A + 0.010328*SQ(A))/(1 + 1.432788*A + 0.189269*SQ(A) + 0.001308*Qube(A));// [Abramowitz and Stegun (1970). Page 933, formula 26.2.23]
	const double xq = A - (2.653962002601684482 + 1.561533700212080345*A + 0.061146735765196993*SQ(A))/(1 + 1.904875182836498708*A + 0.454055536444233510*SQ(A) + 0.009547745327068945*Qube(A));// [Voutier (2010). A New Approximation to the Normal Distribution Quantile Function. arXiv:1002.0567. Pages 5-6]
	const double g1 = 0.25 * (Qube(xq) + xq);
	const double g2 = (1.0/96) * (5*pow(xq,5) + 16*Qube(xq) + 3*xq);
	const double g3 = (1.0/384) * (3*pow(xq,7) + 19*pow(xq,5) + 17*pow(xq,3) - 15*xq);
	const double g4 = (1.0/92160) * (79*pow(xq,9) + 776*pow(xq,7) + 1482*pow(xq,5) - 1920*Qube(xq) - 945*xq);
	double tq = xq + g1/n + g2/SQ(n) + g3/Qube(n) + g4/QTR(n); // [Abramowitz and Stegun (1970). Page 949]
	return tq;
}




// Starting from a mapping pool-->group, calculate the reverse mapping group-->member_list
// [[Rcpp::export]]
Rcpp::List get_member_lists_from_group_assignments_CPP(	const long				Ngroups,
														const std::vector<long> &pool2group){	// (INPUT) 1D array mapping each item in the pool to a group index. Values must be between 0 and Ngroups-1. Negative values mean that an item is not affiliated with any group.

	const long N = pool2group.size();
	lvector group2member_count(Ngroups,0);
	for(long i=0; i<N; ++i){
		if(pool2group[i]>=0) ++group2member_count[pool2group[i]];
	}
	std::vector<lvector> group2members(Ngroups);
	for(long g=0; g<Ngroups; ++g){
		group2members[g].reserve(group2member_count[g]);
	}
	for(long i=0; i<N; ++i){
		if(pool2group[i]>=0) group2members[pool2group[i]].push_back(i);
	}
	return Rcpp::List::create(Rcpp::Named("group2members") = group2members);
}




// place a sorted list of values into a sorted list of bins (i.e. a list of closed intervals)
// items[], bin_mins[] and bin_maxs[] must be sorted in ascending order
// while bins may in principle be overlapping, it must be guaranteed that the lists bin_mins[] and bin_maxs[] are in ascending order
// It is guaranteed that every value will be placed in at most one bin (e.g., if bins are overlapping).
// Values that don't fall within any bin will be assigned the bin index -1
// [[Rcpp::export]]
Rcpp::List place_sorted_values_into_bins_CPP(	const std::vector<double> &items,		// (INPUT) 1D vector listing values to be binned, in ascending order
												const std::vector<double> &bin_mins,	// (INPUT) 1D vector of size NB, listing the lower bounds (inclusive) of the bins, in ascending order
												const std::vector<double> &bin_maxs){	// (INPUT) 1D vector of size NB, listing the upper bounds (inclusive) of the bins, in ascending order

	const long N  = items.size();
	const long NB = bin_mins.size();
	std::vector<long> item2bin(N,-1l);
	lvector bin2count(NB,0l);
	for(long n=0, bin=-1; n<N; ++n){
		bin = find_next_right_grid_point(bin_maxs, items[n], bin);
		if(bin<0) break; // all remaining values are outside (to the right) of all bins
		if(bin_mins[bin]<=items[n]){
			// item n is in this bin
			item2bin[n] = bin;
			++bin2count[bin];
		}
	}
	std::vector< std::vector<long> > bin2items(NB);
	for(long b=0; b<NB; ++b){
		bin2items[b].reserve(bin2count[b]);
	}
	for(long n=0; n<N; ++n){
		if(item2bin[n]>=0) bin2items[item2bin[n]].push_back(n);
	}
	return Rcpp::List::create(	Rcpp::Named("item2bin")  = Rcpp::wrap(item2bin),
								Rcpp::Named("bin2items") = Rcpp::wrap(bin2items));
}



// calculate the 1st Wasserstein distance between two sets of numbers (X & Y), each interpreted as a probability distribution composed of a sum of Dirac functions
// alternatively, this distance can be interpreted as an estimate of the 1st Wasserstein metric between two probability distributions, having generated X and Y, respectively
// As shown by Ramdas et al. (2017), the p-Wasserstein distance between two probability distributions can be written as:
//	  D(X,Y) = \int_0^1 |g(u) - f(u)| du
// where g and f are the quantile functions of the two distributions. For the special case where p=1 (first Wasserstein distance), this is equivalent to:
//	  D(X,Y) = \int_{-\infty}^\infty |G(t) - F(t)| dt
// where G and F are the cumulative distribution functions (CDFs). 
// For finite sets of numbers X and Y, generated by two probability distributions, G and F are simply the respective empirical CDFs.
// This approach is used in python's function scipy.stats.wasserstein_distance.
// Notice the similarity to the Kolmogorov-Smirnov metric (i.e. the test statistic used in the Kolmogorov-Smirnov test): D_KS(X,Y) = max_t |G(t) - F(t)|
// References:
//   Ramdas et al. (2017). On Wasserstein two-sample testing and related families of nonparametric tests. Entropy. 19(2):47. Proposition 1.
// [[Rcpp::export]]
double first_Wasserstein_distance_CPP(	const std::vector<double> &X, 	// 1D numeric vector of arbitrary size, in ascending order
										const std::vector<double> &Y){	// 1D numeric vector of arbitrary size, in ascending order
	const long NX = X.size();
	const long NY = Y.size();
	if((NX==0) || (NY==0)) return NAN_D;
	if(X[0]>X[NX-1]) return NAN_D; // values must be sorted
	if(Y[0]>Y[NY-1]) return NAN_D; // values must be sorted
	long i=-1, j=-1;
	double CDFX=0, CDFY=0, t1, t2;
	double W = 0;
	while(TRUE){
		if((j==NY-1) || ((i<NX-1) && (X[i+1]<=Y[j+1]))){
			// advance X
			++i;
			t1 = X[i];
			CDFX = (i+1.0)/NX;
		}else if((i==NX-1) || ((j<NY-1) && (Y[j+1]<=X[i+1]))){
			// advance Y
			++j;
			t1 = Y[j];
			CDFY = (j+1.0)/NY;
		}
		if((i==NX-1) && (j==NY-1)) break;
		t2 = min((i<NX-1 ? X[i+1] : INFTY_D), (j<NY-1 ? Y[j+1] : INFTY_D));
		W += (t2-t1) * abs(CDFY-CDFX);
	}
	return(W);
}





#pragma mark -
#pragma mark Graphs
#pragma mark 




// Given a graph (node,edges), create lookup tables listing incoming & outgoing edges for each node
// Requirements:
//    The graph may be directed or undirected
//    The graph may be cyclic or acyclic and need not be fully connected 
//    Any two nodes may be connected by zero, one or multiple edges in either direction
void get_graph_edge_mapping(const long 		Nnodes,				// (INPUT) number of nodes in the graph
							const long		Nedges,				// (INPUT) number of edges in the graph
							const lvector	&edges,				// (INPUT) 2D array of size Nedges x 2, in row-major format, listing the source & target node of each edge
							lvector			&node2first_edge,	// (OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their first incoming or outgoing edge.
							lvector			&node2last_edge,	// (OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their last incoming or outgoing edge.
							lvector			&edge_mapping){		// (OUTPUT) 1D array of size 2*Nedges, with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in node2first_edge[] and node2last_edge[]) to original edge indices. Note that each edge is listed twice in this array (once as an outgoing and once as an incoming edge).
	edge_mapping.resize(2*Nedges);
	node2first_edge.resize(Nnodes);
	node2last_edge.resize(Nnodes);

	// determine number of edges per node
	// edge_count_per_node[n] will be the number of edges connected to node n (regardless of direction)
	lvector edge_count_per_node(Nnodes, 0);
	for(long e=0; e<Nedges; ++e){
		edge_count_per_node[edges[e*2+0]] += 1;	// count edge towards source node
		edge_count_per_node[edges[e*2+1]] += 1;	// count edge towards target node
	}
	// collect edges per node
	node2first_edge[0] = 0;
	node2last_edge[0]  = node2first_edge[0]+edge_count_per_node[0] - 1;
	if(Nnodes>1){
		for(long n=1; n<Nnodes; ++n){
			node2first_edge[n] = node2last_edge[n-1]+1;
			node2last_edge[n]  = node2first_edge[n]+edge_count_per_node[n] - 1;
		}
	}
	for(long e=0, node; e<Nedges; ++e){
		// add edge to parent
		node = edges[e*2+0];
		edge_mapping[node2first_edge[node]+edge_count_per_node[node]-1] = e;
		edge_count_per_node[node] -= 1;

		// add edge to child
		node = edges[e*2+1];
		edge_mapping[node2first_edge[node]+edge_count_per_node[node]-1] = e;
		edge_count_per_node[node] -= 1;
	}
}



// given an undirected graph (nodes,edges), find its maximal connected subgraphs
// any two nodes may be connected by zero, one or multiple edges
// [[Rcpp::export]]
Rcpp::List split_undirected_graph_CPP(	const long 				Nnodes,		// (INPUT) the toal number of nodes in the graph
										const long				Nedges,		// (INPUT) number of edges in the graph
										const std::vector<long>	&edges){	// (INPUT) 2D array of size Nedges x 2, in row-major format, listing the source & target node of each edge. Note that swapping source & target does not matter since the graph is treated as undirected.
	if(Nnodes==0) return Rcpp::List::create(Rcpp::Named("Nsubgraphs") = 0);
	
	// determine edges connected to each node
	lvector node2first_edge, node2last_edge, edge_mapping;
	get_graph_edge_mapping(Nnodes, Nedges, edges, node2first_edge, node2last_edge, edge_mapping);
	
	// traverse graph in depth-first-search
	// keep track of non-visited nodes
	lvector nonvisited_nodes(Nnodes); 	// current list of non-visited nodes. Will shrink as we traverse the graph
	lvector node2nonvisited(Nnodes);	// lookup table mapping each node to its index in nonvisited_nodes[]. If a node has already been visited, the value will be -1.
	for(long node=0; node<Nnodes; ++node){ 
		nonvisited_nodes[node] = node;
		node2nonvisited[node] = node;
	}
	lvector node_stack; // current stack of nodes in the currently traversed subgraph
	node_stack.reserve(Nnodes);
	lvector node2subgraph(Nnodes,-1);
	lvector edge2subgraph(Nedges,-1);
	long current_subgraph = -1; // subgraph currently traversed
	long parent, child, edge;
	while((!node_stack.empty()) || (!nonvisited_nodes.empty())){
		if(node_stack.empty()){
			// finished traversing this subgraph
			// pick a non-visited node and assign it to a new subgraph
			parent = nonvisited_nodes.back();
			node2nonvisited[parent] = -1;
			nonvisited_nodes.pop_back();
			++current_subgraph;
			node2subgraph[parent] = current_subgraph;
		}else{
			// pick the next node in the current subgraph's stack
			parent = node_stack.back();
			node_stack.pop_back();
		}
					
		// traverse through all nodes directly connected to this parent node
		// we treat the other node as "child", regardless of who is actually the source & target node of this edge (since the graph is treated as undirected)
		for(long e=node2first_edge[parent]; e<=node2last_edge[parent]; ++e){
			edge  = edge_mapping[e];
			edge2subgraph[edge] = current_subgraph;
			child = (edges[2*edge+0]==parent ? edges[2*edge+1] : edges[2*edge+0]);
			if(node2subgraph[child]>=0){
				// this child node was already visited before, so skip
				continue;
			}
			node2subgraph[child] = current_subgraph;
			node_stack.push_back(child);
			// mark child node as visited (remove from nonvisited list)
			remove_item_from_mapped_list(nonvisited_nodes, node2nonvisited, child);
		}
	}
	const long NS = current_subgraph + 1;
	
	// create lists of nodes per subgraph
	lvector subgraph2node_count(NS,0);
	for(long node=0; node<Nnodes; ++node){
		++subgraph2node_count[node2subgraph[node]];
	}
	std::vector<lvector> subgraph2nodes(NS);
	for(long s=0; s<NS; ++s){
		subgraph2nodes[s].reserve(subgraph2node_count[s]);
	}
	for(long node=0; node<Nnodes; ++node){
		subgraph2nodes[node2subgraph[node]].push_back(node);
	}
	
	// create lists of edges per subgraph
	lvector subgraph2edge_count(NS,0);
	for(long edge=0; edge<Nedges; ++edge){
		++subgraph2edge_count[edge2subgraph[edge]];
	}
	std::vector<lvector> subgraph2edges(NS);
	for(long s=0; s<NS; ++s){
		subgraph2edges[s].reserve(subgraph2edge_count[s]);
	}
	for(long edge=0; edge<Nedges; ++edge){
		subgraph2edges[edge2subgraph[edge]].push_back(edge);
	}
	
	return Rcpp::List::create(	Rcpp::Named("Nsubgraphs") 		= NS,				// number of subgraphs found
								Rcpp::Named("subgraph2nodes") 	= subgraph2nodes, 	// list of length NS, with each entry being a 1D array of node indices belonging to the particular subgraph
								Rcpp::Named("subgraph2edges") 	= subgraph2edges, 	// list of length NS, with each entry being a 1D array of edge indices belonging to the particular subgraph
								Rcpp::Named("node2subgraph")	= node2subgraph,	// 1D array of size Nnodes, mapping each node to its subgraph
								Rcpp::Named("edge2subgraph")	= edge2subgraph);	// 1D array of size Nedges, mapping each edge to its subgraph
}



#pragma mark -
#pragma mark Polynomials
#pragma mark 


// compute the value of a polynomial at a specific x
template<class VALUE_TYPE>
VALUE_TYPE polynomial_value(const long			P,			// (INPUT) polynomial degree
							const VALUE_TYPE	coeff[],	// (INPUT) 1D array of size P+1, listing polynomial coefficients
							const double		x){			// (INPUT) x-value at which to evaluate the polynomial
	VALUE_TYPE y(0);
	for(long p=0; p<=P; ++p){
		y += pow(x,p) * coeff[p];
	}
	return y;
}


// compute the value of a piecewise polynomial curve at a specific x, based on the coefficients in a specific grid cell
inline double piecewise_polynomial_value(const dvector 		&X,			// (INPUT) 1D array of size N, listing x-values in ascending order
										const dvector	 	&Ycoeff,	// (INPUT) 2D array of size N x (Ydegree+1) in row-major format, listing the polynomial coefficients (f0,f1,..,fP) of the function Y in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has Y(x) = Ycoeff[i*(P+1)+0] + Ycoeff[i*(P+1)+1]*x + .. + Ycoeff[i*(P+1)+P]*x^P.
										const long			Ydegree,	// (INPUT) polynomial degree P of the input function Y=Y(x) in each interval. For example, if Y is piecewise linear then P=1.
										const bool			slideX,		// (INPUT) whether Y is defined as a function of the sliding-X, i.e. in the interval X[i]:X[i+1] Y is defined as a polynomial of (X-X[i]) rather than X. Note that the antiderivative will also be defined w.r.t. a sliding-X if Y is.
										const long			grid_point,	// (INPUT) grid point in which to evaluate the polynomial
										const double		xout){		// (INPUT) X-value at which to evaluate the polynomial
	return polynomial_value(Ydegree, &Ycoeff[grid_point*(Ydegree+1)], xout-(slideX ? X[grid_point] : 0.0));
}


// given a list of NC piecewise polynomials Y_i and weights w_i, compute the value sum_i w_i*Y_i(x) for some requested x
double sum_piecewise_polynomials(	const std::vector<dvector> 	&Ycoeff,		// (INPUT) 1D list of size NC, listing polynomial curves. The i-th element is a 2D array of size NX x (Ydegrees[i]+1), in row-major format, listing polynomial coefficients of Y_i at each grid point
									const lvector 				&Ydegrees,		// (INPUT) 1D array of size NC, listing the polynomial degrees of the curves Y_i
									const dvector				&weights,		// (INPUT) 1D array of size NC, listing the weight w_i of each polynomial curve
									const double				x,				// (INPUT) the x-value at which to evaluate the sum of the polynomials
									const long					grid_point){	// (INPUT) grid point from which to obtain the polynomial coefficients
	double V = 0;
	for(long i=0; i<Ydegrees.size(); ++i){
		V += weights[i] * polynomial_value(Ydegrees[i], &Ycoeff[i][grid_point*(Ydegrees[i]+1)+0], x);
	}
	return V;
}

// compute the derivative of a polynomial at a specific x
template<class VALUE_TYPE>
VALUE_TYPE polynomial_derivative(	const long			P,			// (INPUT) polynomial degree
									const VALUE_TYPE	coeff[],	// (INPUT) 1D array of size P+1, listing polynomial coefficients
									const double		x){			// (INPUT) x-value at which to evaluate the derivative
	VALUE_TYPE derivative(0);
	for(long p=1; p<=P; ++p){
		derivative += p * pow(x,p-1) * coeff[p];
	}
	return derivative;
}


// compute the derivative of a piecewise polynomial curve at a specific x, based on the coefficients in a specific grid cell
inline double piecewise_polynomial_derivative(	const dvector 		&X,			// (INPUT) 1D array of size N, listing x-values in ascending order
												const dvector	 	&Ycoeff,	// (INPUT) 2D array of size N x (Ydegree+1) in row-major format, listing the polynomial coefficients (f0,f1,..,fP) of the function Y in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has Y(x) = Ycoeff[i*(P+1)+0] + Ycoeff[i*(P+1)+1]*x + .. + Ycoeff[i*(P+1)+P]*x^P.
												const long			Ydegree,	// (INPUT) polynomial degree P of the input function Y=Y(x) in each interval. For example, if Y is piecewise linear then P=1.
												const bool			slideX,		// (INPUT) whether Y is defined as a function of the sliding-X, i.e. in the interval X[i]:X[i+1] Y is defined as a polynomial of (X-X[i]) rather than X. Note that the antiderivative will also be defined w.r.t. a sliding-X if Y is.
												const long			grid_point,	// (INPUT) grid point in which to evaluate the polynomial
												const double		xout){		// (INPUT) X-value at which to evaluate the polynomial
	return polynomial_derivative(Ydegree, &Ycoeff[grid_point*(Ydegree+1)], xout-(slideX ? X[grid_point] : 0.0));
}


// compute the 2nd derivative of a polynomial at a specific x
template<class VALUE_TYPE>
VALUE_TYPE polynomial_second_derivative(const long			P,			// (INPUT) polynomial degree
										const VALUE_TYPE	coeff[],	// (INPUT) 1D array of size P+1, listing polynomial coefficients
										const double		x){			// (INPUT) x-value at which to evaluate the derivative
	VALUE_TYPE derivative(0);
	for(long p=2; p<=P; ++p){
		derivative += p * (p-1) * pow(x,p-2) * coeff[p];
	}
	return derivative;
}


// given a polynomial function, Y=Y(X), modify its coefficients so that it becomes a function of X' := X-Xshift
// Hence, this will yield a new polynomial Z such that Z(X-Xshift) = Y(X)
template<class VALUE_TYPE>
void get_Xshifted_polynomial(	const long 			P,
								const VALUE_TYPE	Ycoeff[],	// (INPUT) 1D array of size P+1, listing the original polynomial coefficients
								const double		Xshift,		// (INPUT) the X-shift to apply, i.e. redefine the polynomial to be w.r.t. (X-Xshift) rather than X
								VALUE_TYPE			Zcoeff[]){	// (OUTPUT) 1D array of size P+1, listing the new polynomial coefficients w.r.t. the shifted X-axis. Must be pre-allocated.
	for(long p=0; p<=P; ++p) Zcoeff[p] = 0; // initialize
	double pfactorial = 1; // factorial of p, i.e. p!, will be adjusted on the fly
	double kfactorial, pkfactorial; // factorials k! and (p-k)!, will be modified on the fly
	double pkbinom; // binomial coefficient (p k)
	for(long p=0, k; p<=P; ++p){
		// distribute the various coefficients and corrections emerging from the expansion of (X' + Xshift)^p
		// for efficiency, we compute the binomial coefficients binom(p,k) gradually as we go
		if(p>0) pfactorial *= p;
		kfactorial  = 1;
		pkfactorial = pfactorial;
		for(k=0; k<=p; ++k){
			if(k>0){
				kfactorial *= k;
				pkfactorial /= (p-k+1);
			}
			pkbinom = pfactorial/(kfactorial * pkfactorial);
			Zcoeff[k] += Ycoeff[p] * pkbinom * pow(Xshift,p-k);
		}
	}
}


// get the location and value of a quadratic function f(x) = Ax^2 + Bx + C at its local extremum
template<class VALUE_TYPE>
void get_quadratic_extremum(const VALUE_TYPE 	&A,
							const VALUE_TYPE 	&B,
							const VALUE_TYPE 	&C,
							double				&x,		// (OUTPUT) location of the extremum. Will be NAN_D if it does not exist (e.g. if A==0).
							VALUE_TYPE			&y){	// (OUTPUT) value at the extremum. Will have unpredictable value if it does not exist (i.e. you should first check x).
	if(A==0){
		// this function is only linear, so does not have an extremum
		x = NAN_D;
	}else{
		x = -B/(2*A);
		y = A*SQ(x) + B*x + C;
	}
}



// get the locations and values of a cubic function f(x) = Ax^3 + Bx^2 + Cx + D at its two local extrema
template<class VALUE_TYPE>
void get_cubic_extrema(	const VALUE_TYPE 	&A,
						const VALUE_TYPE 	&B,
						const VALUE_TYPE 	&C,
						const VALUE_TYPE 	&D,
						double				&x1,	// (OUTPUT) location of the 1st extremum. Will be NAN_D if it does not exist or is a saddle.
						double				&x2, 	// (OUTPUT) location of the 2nd extremum. Will be NAN_D if it does not exist or is a saddle.
						VALUE_TYPE			&y1,	// (OUTPUT) value at the 1st extremum. Will have unpredictable value if it does not exist or is a saddle (i.e. you should first check x1).
						VALUE_TYPE			&y2){	// (OUTPUT) value at the 1st extremum. Will have unpredictable value if it does not exist or is a saddle (i.e. you should first check x2)
	if(A==0){
		// this function is at most quadratic, not cubic, so it only has at most a single extremum
		get_quadratic_extremum(B,C,D,x1,y1);
		x2 = NAN_D;
	}else{
		const double Delta = SQ(B) - 3*A*C;
		if(Delta<=0){
			// no local extrema, or both local extrema coincide and are thus a mere saddle
			x1 = NAN_D;
			x2 = NAN_D;
		}else{
			x1 = (-B + sqrt(Delta))/(3*A);
			x2 = (-B - sqrt(Delta))/(3*A);
			y1 = A*Qube(x1) + B*SQ(x1) + C*x1 + D;
			y2 = A*Qube(x2) + B*SQ(x2) + C*x2 + D;
		}
	}
}


// compute an upper bound for the absolute value of a polynomial within a specific interval
// for polynomials of degree <=3 this method yields the exact maximum; for higher-degree polynomials this method only yields an upper bound (i.e. not the minimal upper bound)
template<class VALUE_TYPE>
VALUE_TYPE polynomial_bound_abs(const long			P,			// (INPUT) polynomial degree
								const VALUE_TYPE	coeff[],	// (INPUT) 1D array of size P, listing polynomial coefficients
								const double		xmin,		// (INPUT)
								const double		xmax){		// (INPUT)
	if(P==0){
		return abs(coeff[0]);
	}else if(P==1){
		return max(abs(coeff[0] + coeff[1]*xmin), abs(coeff[0] + coeff[1]*xmax));
	}else if(P==2){
		// f(x) = a*x^2 + b*x + c
		const VALUE_TYPE a = coeff[2];
		const VALUE_TYPE b = coeff[1];
		const VALUE_TYPE c = coeff[0];
		double Xvertex;
		VALUE_TYPE Yvertex;
		get_quadratic_extremum(a,b,c,Xvertex,Yvertex);
		return max(abs(std::isnan(Xvertex) || (Xvertex<xmin) || (Xvertex>xmax) ? 0.0 : Yvertex), max(abs(c + b*xmin + a*SQ(xmin)), abs(c + b*xmax + a*SQ(xmax))));
	}else if(P==3){
		// f(x) = a*x^3 + b*x^2 + c*x + d
		const VALUE_TYPE a = coeff[3];
		const VALUE_TYPE b = coeff[2];
		const VALUE_TYPE c = coeff[1];
		const VALUE_TYPE d = coeff[0];
		double Xvertex1, Xvertex2;
		VALUE_TYPE Yvertex1, Yvertex2;
		get_cubic_extrema(a,b,c,d,Xvertex1,Xvertex2,Yvertex1,Yvertex2);
		return max(max(abs(std::isnan(Xvertex1) || (Xvertex1<xmin) || (Xvertex1>xmax) ? 0.0 : Yvertex1), abs(std::isnan(Xvertex2) || (Xvertex2<xmin) || (Xvertex2>xmax) ? 0.0 : Yvertex2)), max(abs(d + c*xmin + b*SQ(xmin) + a*Qube(xmin)), abs(d + c*xmin + b*SQ(xmax) + a*Qube(xmax))));
	}else{
		VALUE_TYPE Ymax = max(abs(coeff[0] + coeff[1]*xmin), abs(coeff[0] + coeff[1]*xmax));
		for(long p=2; p<=P; ++p){
			Ymax = Ymax + max(abs(coeff[p]*pow(xmin,p)), abs(coeff[p]*pow(xmax,p)));
		}
		return Ymax;
	}
}


// compute an upper bound for the absolute derivative of a polynomial within a specific interval
// for polynomials of degre <=4 this method yields the exact maximum of |derivative|; for higher-degree polynomials this method only yields an upper bound (i.e. not the minimal upper bound)
template<class VALUE_TYPE>
VALUE_TYPE polynomial_bound_abs_derivative(	const long			P,			// (INPUT) polynomial degree
											const VALUE_TYPE	coeff[],	// (INPUT) 1D array of size P, listing polynomial coefficients
											const double		xmin,		// (INPUT)
											const double		xmax){		// (INPUT)
	if(P==0) return 0;
	std::vector<VALUE_TYPE> Dcoeff(P);
	for(long p=1; p<=P; ++p){
		Dcoeff[p-1] = p*coeff[p];
	}
	return polynomial_bound_abs(P-1, &Dcoeff[0], xmin, xmax);
}


// compute an upper bound for the value of a polynomial within a specific interval
// for constant, linear and quadratic polynomials this method yields the exact maximum; for higher-degree polynomials this method only yields an upper bound (i.e. not the minimal upper bound)
template<class VALUE_TYPE>
VALUE_TYPE polynomial_upper_bound(	const long			P,			// (INPUT) polynomial degree
									const VALUE_TYPE	coeff[],	// (INPUT) 1D array of size P, listing polynomial coefficients
									const double		xmin,		// (INPUT)
									const double		xmax){		// (INPUT)
	if(P==0){
		return coeff[0];
	}else if(P==1){
		return max(coeff[0] + coeff[1]*xmin, coeff[0] + coeff[1]*xmax);
	}else if(P==2){
		// f(x) = a*x^2 + b*x + c
		const VALUE_TYPE a = coeff[2];
		const VALUE_TYPE b = coeff[1];
		const VALUE_TYPE c = coeff[0];
		double Xvertex;
		VALUE_TYPE Yvertex;
		get_quadratic_extremum(a,b,c,Xvertex,Yvertex);
		return max((std::isnan(Xvertex) || (Xvertex<xmin) || (Xvertex>xmax) ? -INFTY_D : Yvertex), max(c + b*xmin + a*SQ(xmin), c + b*xmax + a*SQ(xmax)));
	}else if(P==3){
		// f(x) = a*x^3 + b*x^2 + c*x + d
		const VALUE_TYPE a = coeff[3];
		const VALUE_TYPE b = coeff[2];
		const VALUE_TYPE c = coeff[1];
		const VALUE_TYPE d = coeff[0];
		double Xvertex1, Xvertex2;
		VALUE_TYPE Yvertex1, Yvertex2;
		get_cubic_extrema(a,b,c,d,Xvertex1,Xvertex2,Yvertex1,Yvertex2);
		return max(max((std::isnan(Xvertex1) || (Xvertex1<xmin) || (Xvertex1>xmax) ? -INFTY_D : Yvertex1), (std::isnan(Xvertex2) || (Xvertex2<xmin) || (Xvertex2>xmax) ? -INFTY_D : Yvertex2)), max((d + c*xmin + b*SQ(xmin) + a*Qube(xmin)), (d + c*xmin + b*SQ(xmax) + a*Qube(xmax))));	
	}else{
		VALUE_TYPE Ymax = max(coeff[0] + coeff[1]*xmin, coeff[0] + coeff[1]*xmax);
		for(long p=2; p<=P; ++p){
			Ymax = Ymax + max(coeff[p]*pow(xmin,p), coeff[p]*pow(xmax,p));
		}
		return Ymax;
	}
}


// compute a lower bound for the value of a polynomial within a specific interval
// for constant, linear and quadratic polynomials this method yields the exact minimum; for higher-degree polynomials this method only yields a lower bound (i.e. not the maximal lower bound)
template<class VALUE_TYPE>
VALUE_TYPE polynomial_lower_bound(	const long			P,			// (INPUT) polynomial degree
									const VALUE_TYPE	coeff[],	// (INPUT) 1D array of size P, listing polynomial coefficients
									const double		xmin,		// (INPUT)
									const double		xmax){		// (INPUT)
	if(P==0){
		return coeff[0];
	}else if(P==1){
		return min(coeff[0] + coeff[1]*xmin, coeff[0] + coeff[1]*xmax);
	}else if(P==2){
		// f(x) = a*x^2 + b*x + c
		const VALUE_TYPE a = coeff[2];
		const VALUE_TYPE b = coeff[1];
		const VALUE_TYPE c = coeff[0];
		double Xvertex;
		VALUE_TYPE Yvertex;
		get_quadratic_extremum(a,b,c,Xvertex,Yvertex);
		return min((std::isnan(Xvertex) || (Xvertex<xmin) || (Xvertex>xmax) ? +INFTY_D : Yvertex), min(c + b*xmin + a*SQ(xmin), c + b*xmax + a*SQ(xmax)));
	}else if(P==3){
		// f(x) = a*x^3 + b*x^2 + c*x + d
		const VALUE_TYPE a = coeff[3];
		const VALUE_TYPE b = coeff[2];
		const VALUE_TYPE c = coeff[1];
		const VALUE_TYPE d = coeff[0];
		double Xvertex1, Xvertex2;
		VALUE_TYPE Yvertex1, Yvertex2;
		get_cubic_extrema(a,b,c,d,Xvertex1,Xvertex2,Yvertex1,Yvertex2);
		return min(min((std::isnan(Xvertex1) || (Xvertex1<xmin) || (Xvertex1>xmax) ? +INFTY_D : Yvertex1), (std::isnan(Xvertex2) || (Xvertex2<xmin) || (Xvertex2>xmax) ? +INFTY_D : Yvertex2)), min((d + c*xmin + b*SQ(xmin) + a*Qube(xmin)), (d + c*xmin + b*SQ(xmax) + a*Qube(xmax))));	
	}else{
		VALUE_TYPE Ymin = min(coeff[0] + coeff[1]*xmin, coeff[0] + coeff[1]*xmax);
		for(long p=2; p<=P; ++p){
			Ymin = Ymin + min(coeff[p]*pow(xmin,p), coeff[p]*pow(xmax,p));
		}
		return Ymin;
	}
}


// Given 3 points (x1,y1), (x2,y2), (x3,y3), determine the coefficients of a quadratic polynomial E(x) = A + B*x + C*x^2 that passess through these points
inline void quadratic_interpolation_3points(double	x1,			// (INPUT)
											double 	x2,			// (INPUT)
											double	x3,			// (INPUT)
											double	y1,			// (INPUT)
											double 	y2,			// (INPUT)
											double	y3,			// (INPUT)
											double	Ecoeff[]){	// (OUTPUT) 1D array of size 3, listing the polynomial coefficients (E0,E1,E2) of the quadratic approximation of Y(x) ~ E0 + E1*x + E2*x^2. Must be preallocated.
	// shift Y-axis temporarily for numerical stability when Y values are very close to each other
	const double dy = y1;
	y1 -= dy; 
	y2 -= dy; 
	y3 -= dy;
	
	// shift X-values for numerical stability when X values are very close to each other
	const double dx = x1;
	x1 -= dx;
	x2 -= dx;
	x3 -= dx;
	
	double A, B, C; // polynomial coefficients for the shifted X & Y, i.e. such that (Y-dy) = A + B*(X-dx) + C*(X-dx)^2
	if((y1==y2) && (y2==y3)){
		// constant function
		A = y1;
		B = 0;
		C = 0;
	}else{
		// compute the polynomial coefficients for E
		A = (x1*x3*(-x1 + SQ(x3))*y2 + SQ(x2)*(x3*y1 - x1*y3) + x2*(-(Qube(x3)*y1) + SQ(x1)*y3))/((x1 - x2)*(x1*(x2 - x3) - x2*x3 + Qube(x3)));
		B = (Qube(x3)*(y1 - y2) + SQ(x1)*(y2 - y3) + SQ(x2)*(-y1 + y3))/((x1 - x2)*(x1*(x2 - x3) - x2*x3 + Qube(x3)));
		C = (x3*(-y1 + y2) + x2*(y1 - y3) + x1*(-y2 + y3))/(SQ(x1)*(x2 - x3) + x2*x3*(x2 - SQ(x3)) + x1*(-SQ(x2) + Qube(x3)));
	}
	// reverse shifts
	Ecoeff[0] = dy+A-dx*B+SQ(dx)*C;
	Ecoeff[1] = B-2*dx*C;
	Ecoeff[2] = C;
}


// given some polynomial function:
//	 Y(x) = f0 + f1*x + f2*x^2 + .. + fP*x^P
// and three points x1,x2,x3, compute coefficients E0,E1,E2 such that the quadratic function:
//	 E(x) := E0 + E1*x + E2*x^2
// equals exp(Y(x)) at the points x1,x2,x3.
// Interpretation: The quadratic function E(x) defined by the returned coefficients E0,E1,E2 is an approximation for exp(Y(x)), and is most accurate at (or near) the points x1,x2,x3.
void quadratic_approximation_of_exp_polynomial(	const long		P,			// (INPUT) polynomial degree of Y(x)
												const double 	Ycoeff[],	// (INPUT) 1D array of size P, listing the polynomial coefficients (f0,f1,f2) of the function Y(x) = f0 + f1*x + f2*x^2 + .. + fP*x^P
												double			x1,			// (INPUT)
												double 			x2,			// (INPUT)
												double			x3,			// (INPUT)
												double			Ecoeff[]){	// (OUTPUT) 1D array of size 3, listing the polynomial coefficients (E0,E1,E2) of the quadratic approximation of exp(Y(x)) ~ E0 + E1*x + E2*x^2. Must be preallocated.
	// evaluate exp(Y) on x1,x2,x3
	double y1=0, y2=0, y3=0;
	for(long p=0; p<=P; ++p){
		y1 += Ycoeff[p]*pow(x1,p);
		y2 += Ycoeff[p]*pow(x2,p);
		y3 += Ycoeff[p]*pow(x3,p);
	}
	y1 = exp(y1); y2 = exp(y2); y3 = exp(y3);
	quadratic_interpolation_3points(x1, x2, x3, y1, y2, y3, Ecoeff);
}


// given some polynomial function:
//	 Y(x) = f0 + f1*x + f2*x^2 + .. + fP*x^P
// and three points x1,x2,x3, compute coefficients E0,E1,E2 such that the quadratic function:
//	 E(x) := E0 + E1*x + E2*x^2
// equals 1/Y(x) at the points x1,x2,x3.
// Interpretation: The quadratic function E(x) defined by the returned coefficients E0,E1,E2 is an approximation for 1/Y(x), and is most accurate at (or near) the points x1,x2,x3.
void quadratic_approximation_of_inv_polynomial(	const long		P,			// (INPUT) polynomial degree of Y(x)
												const double 	Ycoeff[],	// (INPUT) 1D array of size P, listing the polynomial coefficients (f0,f1,f2) of the function Y(x) = f0 + f1*x + f2*x^2 + .. + fP*x^P
												const double	x1,			// (INPUT)
												const double 	x2,			// (INPUT)
												const double	x3,			// (INPUT)
												double			Ecoeff[]){	// (OUTPUT) 1D array of size 3, listing the polynomial coefficients (E0,E1,E2) of the quadratic approximation of exp(Y(x)) ~ E0 + E1*x + E2*x^2. Must be preallocated.
	// evaluate exp(Y) on x1,x2,x3
	double y1=0, y2=0, y3=0;
	for(long p=0; p<=P; ++p){
		y1 += Ycoeff[p]*pow(x1,p);
		y2 += Ycoeff[p]*pow(x2,p);
		y3 += Ycoeff[p]*pow(x3,p);
	}
	y1 = 1/y1; y2 = 1/y2; y3 = 1/y3;
	quadratic_interpolation_3points(x1, x2, x3, y1, y2, y3, Ecoeff);
}


// given a piecewise polynomial function:
//	 Y(x) = f0 + f1*x + f2*x^2 + ... fn*x^n,
// with polynomial coefficients (f0,f1,f2,..) defined on an X-grid, define a piecewise quadratic function:
//	 E(x) := E0 + E1*x + E2*x^2
// on the same grid that approximates exp(Y(x)).
// If slideX==true, then both Y and E are defined w.r.t. the sliding-X, i.e. in the i-th grid interval Y and E are polynomials of (X-X[i]) rather than X
void quadratic_approximation_of_piecewise_exp_polynomial(	const dvector	&X,			// (INPUT) 1D array of size N, listing x-values in ascending degree
															const long		Ydegree,	// (INPUT) polynomial degree of Y
															const dvector	&Ycoeff,	// (INPUT) 2D array of size N x 3, listing the polynomial coefficients (f0,f1,f2) of the function Y(x) on each grid point
															const bool		slideX,		// (INPUT) whether Y is defined as a function of the sliding-X, i.e. in the interval X[i]:X[i+1] Y is defined as a polynomial of (X-X[i]) rather than X. Note that the output Ecoeff will also be defined w.r.t. the sliding-X.
															dvector			&Ecoeff){	// (OUTPUT) 2D array of size N x 3, listing the polynomial coefficients (E0,E1,E2) of the quadratic approximation of exp(Y(x)) ~ E0 + E1*x + E2*x^2 at each grid point
	const long N = X.size();
	const long Edegree = 2;
	double Xshift;
	Ecoeff.resize(N*(Edegree+1));
	for(long g=0, g2; g<N; ++g){
		if(g==N-1){
			g2 = N-2;
		}else{
			g2 = g+1;
		}
		Xshift = (slideX ? X[g] : 0.0);
		// determine coefficients (E0,E1,E2) on grid point g
		quadratic_approximation_of_exp_polynomial(Ydegree,&Ycoeff[g*(Ydegree+1)],X[g]-Xshift,0.5*(X[g]+X[g2])-Xshift,X[g2]-Xshift,&Ecoeff[g*(Edegree+1)]);
	}
}


template<class VALUE_TYPE>
void multiply_piecewise_polynomials(const long						NG,			// (INPUT) number of grid points on which the two functions A & B are defined in a piece-wise manner
									const long 						Adegree,	// (INPUT) polynomial degree of A
									const std::vector<VALUE_TYPE>	&Acoeff,	// (INPUT) 2D array of size NG x (Adegree+1) in row-major format, specifying the polynomial coefficients of A on the grid
									const long 						Bdegree,	// (INPUT) polynomial degree of B
									const std::vector<VALUE_TYPE>	&Bcoeff,	// (INPUT) 2D array of size NG x (Bdegree+1) in row-major format, specifying the polynomial coefficients of B on the grid
									long							&ABdegree,	// (OUTPUT) polynomial degree of A*B
									std::vector<VALUE_TYPE>			&ABcoeff){	// (OUTPUT) 2D array of size NG x (ABdegree+1) in row-major format, specifying the polynomial coefficients of A*B on the grid
	ABdegree = Adegree + Bdegree;
	ABcoeff.assign(NG*(ABdegree+1),0);
	for(long g=0, pa,pb; g<NG; ++g){
		for(pa=0; pa<=Adegree; ++pa){
			for(pb=0; pb<=Bdegree; ++pb){
				ABcoeff[g*(ABdegree+1)+pa+pb] += Acoeff[g*(Adegree+1)+pa] * Bcoeff[g*(Bdegree+1)+pb];	
			}
		}
	}
}


// add a piecewise polynomial B to another piecewise polynomial A, defined on the same grid (but not necessarily with the same degree)
// addition is done in-situ, i.e. by modifying the coefficients of A; the resulting piecewise polynomial A+B will thus have the same degree as A
// ATTENTION: The added polynomial B must have equal or smaller degree than A
template<class VALUE_TYPE>
void add_piecewise_polynomials_insitu(	const long						Adegree,	// (INPUT) the degree of A
										std::vector<VALUE_TYPE>			&Acoeff,	// (INPUT/OUTPUT) 2D array of size NG x (Adegree+1) in row-major format, specifying the polynomial coefficients of A on the grid. Will be modified in situ to store the resulting A+B
										const long						Bdegree,	// (INPUT) the degree of B. Must be equal or smaller than Adegree
										const std::vector<VALUE_TYPE>	&Bcoeff,	// (INPUT) 2D array of size NG x (Bdegree+1) in row-major format, specifying the polynomial coefficients of B on the grid
										const double					factorA,	// (INPUT) optional multiplicative factor for A. Set to 1 to use the original A.
										const double					factorB){	// (INPUT) optional multiplicative factor for B. Set to 1 to use the original B.
	const long NG = Acoeff.size()/(Adegree+1);
	for(long g=0, p; g<NG; ++g){
		for(p=0; p<=Bdegree; ++p){
			Acoeff[g*(Adegree+1)+p] = factorA * Acoeff[g*(Adegree+1)+p] + factorB * Bcoeff[g*(Bdegree+1)+p];	
		}
	}
}


/* OLD CODE
// subtract a piecewise polynomial B from another piecewise polynomial A, defined on the same grid (but not necessarily with the same degree)
// subtraction is done in-situ, i.e. by modifying the coefficients of A; the resulting piecewise polynomial A-B will thus have the same degree as A
// ATTENTION: The subtracted polynomial B must have equal or smaller degree than A
template<class VALUE_TYPE>
void subtract_piecewise_polynomials_insitu(	const long						Adegree,	// (INPUT) the degree of A
											std::vector<VALUE_TYPE>			&Acoeff,	// (INPUT/OUTPUT) 2D array of size NG x (Adegree+1) in row-major format, specifying the polynomial coefficients of A on the grid. Will be modified in situ to store the resulting A-B
											const long						Bdegree,	// (INPUT) the degree of B. Must be equal or smaller than Adegree
											const std::vector<VALUE_TYPE>	&Bcoeff){	// (INPUT) 2D array of size NG x (Bdegree+1) in row-major format, specifying the polynomial coefficients of B on the grid
	const long NG = Acoeff.size()/(Adegree+1);
	for(long g=0, p; g<NG; ++g){
		for(p=0; p<=Bdegree; ++p){
			Acoeff[g*(Adegree+1)+p] -= Bcoeff[g*(Bdegree+1)+p];	
		}
	}
}
*/

template<class VALUE_TYPE>
void piecewise_linear_to_polynomial(const long				N,			// (INPUT) grid size
									const double			X[],		// (INPUT) 1D array of size N, listing X-values in ascending order
									const double			Y[],		// (INPUT) 1D array of size N, listing Y-values on the grid
									const bool				slideX,		// (INPUT) whether to define Y as a function of the sliding-X, i.e. in the interval X[i]:X[i+1] Y will be defined as a polynomial of (X-X[i]) rather than X
									std::vector<VALUE_TYPE>	&Ycoeff){	// (OUTPUT) 2D array of size N x 2 in row-major format, listing the polynomial coefficients (y0,y1) for the piecewise-linear function Y at each grid point
	Ycoeff.resize(N*2);
	VALUE_TYPE alpha,beta;
	for(long g=0, g2; g<N; ++g){
		if(g==N-1){
			g2 = N-2;
		}else{
			g2 = g+1;
		}
		beta  = (Y[g]-Y[g2])/(X[g]-X[g2]);
		alpha = Y[g2]-beta*(X[g2]-(slideX ? X[g] : 0.0));
		Ycoeff[g*2+0] = alpha;
		Ycoeff[g*2+1] = beta;
	}
}


// wrapper for piecewise_linear_to_polynomial() accepting std::vectors as inputs
template<class VALUE_TYPE>
void piecewise_linear_to_polynomial(const dvector			&X,			// (INPUT) 1D array of size N, listing X-values in ascending order
									const dvector			&Y,			// (INPUT) 1D array of size N, listing Y-values on the grid
									std::vector<VALUE_TYPE>	&Ycoeff){	// (OUTPUT) 2D array of size N x 2 in row-major format, listing the polynomial coefficients (y0,y1) for the piecewise-linear function Y at each grid point
	piecewise_linear_to_polynomial(X.size(), &X[0], &Y[0], Ycoeff);
}




// given a curve defined at discrete points (X1,Y1), (X2,Y2),..., construct a spline interpolation of a given degree for the curve
// In other words, for each interval X[i]:X[i+1] find polynomial coefficients S0,..,S_degree such that the polynomial S0+S1*x+..+S_degree*x^degree is an interpolation for the curve in that interval
// the splines will satisfy continuity at the breakpoints, as well as continuity of the derivatives 1,...,degree-1
void get_spline(const long		N,		// (INPUT) grid size
				const double	X[],	// (INPUT) 1D array of size N, listing x-values for the curve to be interpolated
				const double	Y[],	// (INPUT) 1D array of size N, listing y-values for the curve to be interpolated
				const long		degree,	// (INPUT) degree of splines, i.e. the maximum exponent in the piecewise polynomials. Currently only degrees 0, 1, 2 and 3 are supported.
				const bool		slideX,	// (INPUT) whether to define Y as a function of the sliding-X, i.e. in the interval X[i]:X[i+1] Y will be defined as a polynomial of (X-X[i]) rather than X
				dvector			&S){	// (OUTPUT) 2D matrix of size N x (degree+1), in row-major format, listing the polynomial coefficients of the splines curve at each interval. Hence, S[i,0..degree] lists the polynomial coefficients S_0,..,S_degree for the interval X[i]:X[i+1]
	S.assign(N*(degree+1),0.0);
	if(N==0) return; // nothing to do   
	if(degree==0){
		// piecewise constant curve (not really a spline)
		for(long i=0; i<N; ++i){
			S[i] = Y[i];
		}

	}else if(N==1){
		// grid only contains a single point, so there's no way to figure out all coefficients. So set to constant.
		S[0] = Y[0];

	}else if(degree==1){
		// piecewise linear spline
		piecewise_linear_to_polynomial(N, X, Y, slideX, S);
		
	}else if(N==2){
		// grid only contains two points, so there's no way to figure out all coefficients. So set to linear.
		dvector linS;
		piecewise_linear_to_polynomial(N, X, Y, slideX, linS);
		S[0*(degree+1)+0] = linS[0];
		S[0*(degree+1)+1] = linS[1];
		S[1*(degree+1)+0] = linS[2];
		S[1*(degree+1)+1] = linS[3];

	}else if(degree==2){
		// piecewise quadratic spline
		dvector Z(N);
		Z[0] = 0;
		for(long i=0; i<(N-1); ++i){
			Z[i+1] = -Z[i] + 2*(Y[i+1]-Y[i])/(X[i+1]-X[i]);
		}
		
		double q;
		for(long i=0; i<N-1; ++i){
			q = (Z[i+1]-Z[i])/(2*(X[i+1]-X[i]));
			S[i*(degree+1)+0] = Y[i] - Z[i]*X[i] + SQ(X[i])*q;
			S[i*(degree+1)+1] = Z[i] - 2*X[i]*q;
			S[i*(degree+1)+2] = q;
		}
		
		// set coefficients at last point equal to previous point
		for(long d=0; d<=degree; ++d){
			S[(N-1)*(degree+1)+d] = S[(N-2)*(degree+1)+d];
		}
		
		if(slideX){
			// redefine polynomial coefficients so that in the i-th grid cell the polynomial is defined w.r.t. the shifted X-X[i]
			double A,B,C;
			for(long i=0; i<N; ++i){
				A = S[i*(degree+1)+0];
				B = S[i*(degree+1)+1];
				C = S[i*(degree+1)+2];
				S[i*(degree+1)+0] = A+B*X[i]+C*SQ(X[i]);
				S[i*(degree+1)+1] = B+2*C*X[i];
				S[i*(degree+1)+2] = C;
			}
		}

	}else if(degree==3){
		// piecewise cubic spline
    	dvector W(N-1), H(N-1), FTT(N);

		for(long i=0; i<(N-1); i++){
			W[i] = (X[i+1]-X[i]);
			H[i] = (Y[i+1]-Y[i])/W[i];
		}

		FTT[0] = 0;
		for(long i=0; i<N-2; i++){
			FTT[i+1] = 3*(H[i+1]-H[i])/(W[i+1]+W[i]);
		}
		FTT[N-1] = 0;

		double A,B,C,D;
		for(long i=0; i<N-1; i++){
			// first find representation f(x) = A*(x-xi)^3 + B*(x-xi)^2 + C*(x-xi) + D
			D = Y[i];
			C = H[i]-W[i]*(FTT[i+1]+2*FTT[i])/6;
			B = FTT[i]/2;
			A = (FTT[i+1]-FTT[i])/(6*W[i]);
			// translate to representation f(x) = S0 + S1*x + S2*x^2 + S3*x^3
			S[i*(degree+1)+0] = D - C*X[i] + B*SQ(X[i]) - A*Qube(X[i]);
			S[i*(degree+1)+1] = C - 2*B*X[i] + 3*A*SQ(X[i]);
			S[i*(degree+1)+2] = B - 3*A*X[i];
			S[i*(degree+1)+3] = A;
		}
		
		// set coefficients at last point equal to previous point
		for(long d=0; d<=degree; ++d){
			S[(N-1)*(degree+1)+d] = S[(N-2)*(degree+1)+d];
		}

		if(slideX){
			// redefine polynomial coefficients so that in the i-th grid cell the polynomial is defined w.r.t. the shifted X-X[i]
			double A,B,C,D;
			for(long i=0; i<N; ++i){
				D = S[i*(degree+1)+0];
				C = S[i*(degree+1)+1];
				B = S[i*(degree+1)+2];
				A = S[i*(degree+1)+3];
				S[i*(degree+1)+0] = D+C*X[i]+B*SQ(X[i])+A*Qube(X[i]);
				S[i*(degree+1)+1] = C+2*B*X[i]+3*A*SQ(X[i]);
				S[i*(degree+1)+2] = B+3*A*X[i];
				S[i*(degree+1)+3] = A;
			}
		}
	}
}



// wrapper for get_spline(..) accepting std::vectors as inputs
void get_spline(const dvector	&X,		// (INPUT) 1D array of size N, listing x-values for the curve to be interpolated
				const dvector	&Y,		// (INPUT) 1D array of size N, listing y-values for the curve to be interpolated
				const long		degree,	// (INPUT) degree of splines, i.e. the maximum exponent in the piecewise polynomials. Currently only degrees 0, 1, 2 and 3 are supported.
				const bool		slideX,	// (INPUT) whether to define Y as a function of the sliding-X, i.e. in the interval X[i]:X[i+1] Y will be defined as a polynomial of (X-X[i]) rather than X
				dvector			&S){	// (OUTPUT) 2D matrix of size N x (degree+1), in row-major format, listing the polynomial coefficients of the splines curve at each interval. Hence, S[i,0..degree] lists the polynomial coefficients S_0,..,S_degree for the interval X[i]:X[i+1]
	const long N = X.size();
	get_spline(	N, &X[0], &Y[0], degree, slideX, S);
}




// given some polynomial curve Y(X)=A + B*(X-Xshift) + C*(X-Xshift)^2 + ... and some target value V, solve the equation Y(x)=V via the bisection method (or algebraically, if possible)
// this method assumes that a solution does exist within the specified interval [Xmin,Xmax]
// if an error occurs, NAN is returned
double solve_polynomial_via_bisection(	const double 	Ycoeff[],			// (INPUT) 1D array of size Ydegree+1, listing polynomial coefficients for Y
										const long 		Ydegree,			// (INPUT) the polynomial degree of Y
										const double	Xshift,				// (INPUT) optional offset of X, i.e. the polynomial is defined w.r.t. (X-Xshift) rather than X
										const double	Xmin,				// (INPUT) the left boundary of the interval within which to search. Assumes that sign(Y(xmin)-V) != sign(Y(xmax)-V)
										const double	Xmax,				// (INPUT) the right boundary of the interval within which to search. Assumes that sign(Y(xmin)-V) != sign(Y(xmax)-V)
										const double	V,					// (INPUT) the target value
										const double	xepsilon,			// (INPUT) absolute error tolerance on an x-scale. If the x-step is within this threshold, the search is halted
										const double	yepsilon,			// (INPUT) absolute error tolerance on a y-scale. If |y-V| is within this threshold, the search is halted
										const long		max_iterations){	// (INPUT) maximum number of iterations allowed, i.e. prior to halting the search
	double a = Xmin, b = Xmax, c;
	double fa, fb, fc;
	long iterations = 0;

	fa  = polynomial_value(Ydegree,&Ycoeff[0],a-Xshift);
	fb  = polynomial_value(Ydegree,&Ycoeff[0],b-Xshift);
	
	if(fa==V) return a;
	if(fb==V) return b;
	if(sgn(fa-V)==sgn(fb-V)) return NAN_D;

	if(Ydegree==1){
		// Y is linear, so we can directly calculate the solution's location
		return (Xshift+(V-Ycoeff[0])/Ycoeff[1]);
	}else if(Ydegree==2){
		// Y is quadratic, so we can directly calculate the solution's location
		const double alpha = Ycoeff[2];
		const double beta  = Ycoeff[1];
		const double gamma = Ycoeff[0];
		if(alpha==0){
			// Y is actually only linear in this interval
			return (Xshift + (V-Ycoeff[0])/Ycoeff[1]);
		}
		const double Delta = SQ(beta) - 4*alpha*(gamma-V);
		if(Delta<0) return NAN_D; // something went wrong, this quadratic function does not have a solution for Y(x) = V
		const double x1 = (-beta+sqrt(Delta))/(2*alpha);
		const double x2 = (-beta-sqrt(Delta))/(2*alpha);
		if((Xmin-Xshift<=x1) && (x1<=Xmax-Xshift)){
			return Xshift+x1;
		}else if((Xmin-Xshift<=x2) && (x2<=Xmax-Xshift)){
			return Xshift+x2;
		}else{
			// something went wrong, as none of the solutions are within the expected bracket [a,b]
			return NAN_D;
		}
	}
	
	// Y is of of greater degree than 2, so use bisection method to get solution
	while(iterations<max_iterations){
		c   = 0.5*(a+b); // standard bisection choice
		//c = a + (V-fa) * (b-a)/(fb-fa); // guess next c based on linear interpolation between (a,fa) and (b,fb)
		fc  = polynomial_value(Ydegree,&Ycoeff[0],c-Xshift);
		if((fc==V) || (abs(b-a)<=xepsilon) || (abs(fc-V)<=yepsilon)){
			return c;
		}
		if(sgn(fc-V)==sgn(fa-V)){
			a  = c;
			fa = fc;
		}else{
			b  = c;
			fb = fc;
		}
		++iterations;
	}
	return c; // did not converge, but the last c is the best estimate
}



// given some piecewise polynomial curve (Y=Y(x)) and some target value V, solve the equation Y(x)=V via the bisection method
// this method assumes that a solution does exist within the specified interval [Xmin,Xmax]
// if an error occurs, NAN is returned
double solve_piecewise_polynomial_bisection(const dvector 	&X,					// (INPUT) 1D array of size NX, listing x-values on the grid, in ascending order
											const dvector 	&Ycoeff,			// (INPUT) 2D array of size NX x (Ydegree+1), in row-major format, listing polynomial coefficients of Y at each grid point
											const long 		Ydegree,			// (INPUT) the polynomial degree of Y
											const bool		slideX,				// (INPUT) whether the input polynomial is defined w.r.t. a sliding-X, i.e. in the interval X[i]:X[i+1] Y is defined as a polynomial of (X-X[i]) rather than X
											const double	Xmin,
											const double	Xmax,
											const double	V,
											const bool		monotonic,			// (INPUT) if true, the function Y is guaranteed to be monotonic in x. In this case, the provided Xmin & Xmax don't necessarily need to bracket the solution and may be automatically searched for if needed
											const double	xepsilon,			// (INPUT) absolute error tolerance on an x-scale. If the x-step is within this threshold, the search is halted
											const double	yepsilon,			// (INPUT) absolute error tolerance on a y-scale. If |y-V| is within this threshold, the search is halted
											const long		max_iterations,		// (INPUT) maximum number of iterations allowed, i.e. prior to halting the search
											long			&Xmin_grid_guess,	// (INPUT/OUTPUT) a guess for which grid point Xmin may be located in. For efficiency reasons only. Set to -1 if not available.
											long			&Xmax_grid_guess){	// (INPUT/OUTPUT) a guess for which grid point Xmax may be located in. For efficiency reasons only. Set to -1 if not available.
	double a = Xmin, b = Xmax, c;
	double fa, fb, fc;
	long iterations = 0, ga, gb, gc = -1;

	ga  = max(0l,find_next_left_grid_point(X, a, Xmin_grid_guess));
	fa  = piecewise_polynomial_value(X,Ycoeff,Ydegree,slideX,ga,a);
	Xmin_grid_guess = ga;

	gb  = max(0l,find_next_left_grid_point(X, b, Xmax_grid_guess));
	fb  = piecewise_polynomial_value(X,Ycoeff,Ydegree,slideX,gb,b);
	Xmax_grid_guess = gb;
	
	if(fa==V) return a;
	if(fb==V) return b;
	if(sgn(fa-V)==sgn(fb-V)){
		if(!monotonic) return NAN_D; // [Xmin,Xmax] does not bracket the solution x and the query function Y(x) is not monotonic, so it's hard to find a suitable bracket
		// search for a suitable bracket, i.e. extend [Xmin, Xmax] until V is within [f(Xmin),f(Xmax)]
		double L;
		while(true){
			// double the bracket span
			L  = b - a;
			a -= L/2;
			b += L/2;
			ga = max(0l,find_next_left_grid_point(X, a, ga));
			fa = piecewise_polynomial_value(X,Ycoeff,Ydegree,slideX,ga,a);
			gb = max(0l,find_next_left_grid_point(X, b, gb));
			fb = piecewise_polynomial_value(X,Ycoeff,Ydegree,slideX,gb,b);
			++iterations;
			if(sgn(fa-V)!=sgn(fb-V)) break;
			if(iterations>=max_iterations) return NAN_D;
		}
		Xmin_grid_guess = ga;
		Xmax_grid_guess = gb;
	}
	
	gc = long(0.5*(ga+gb)); // first guess for gc

	while(iterations<max_iterations){
		if(ga==gb){
			// root is bracketted within a single grid-interval
			if(Ydegree==1){
				// Y is linear in this interval, so we can directly calculate the solution's location
				return((slideX ? X[ga] : 0.0) + (V-Ycoeff[ga*(Ydegree+1)+0])/Ycoeff[ga*(Ydegree+1)+1]);
			}
		}
		c   = 0.5*(a+b);
		gc  = max(0l,find_next_left_grid_point(X, c, gc));
		fc  = piecewise_polynomial_value(X,Ycoeff,Ydegree,slideX,gc,c);
		if((fc==V) || (abs(b-a)<=xepsilon) || (abs(fc-V)<=yepsilon)){
			return c;
		}
		if(sgn(fc-V)==sgn(fa-V)){
			a  = c;
			fa = fc;
			ga = gc;
		}else{
			b  = c;
			fb = fc;
			gb = gc;
		}
		++iterations;
	}
	return c; // did not converge, but the last c is the best estimate
}



// given NC piecewise polynomial curves (Y_i=Y_i(x), where i=1,..,N) and some target value V, and weights w_1,..,w_NC, solve the equation sum_{i=1}^NC w_i*Y_i(x) = V via the bisection method
// this method assumes that a solution does exist within the specified bracket [Xmin,Xmax]; if this is not the case, the bracket may be automatically extended (if monotonic==true) or NAN_D will be returned (if monotonic==false)
// if an error occurs, NAN is returned
double solve_piecewise_polynomial_sum_bisection(const dvector 				&X,			// (INPUT) 1D array of size NX, listing x-values on the grid, in ascending order
												const std::vector<dvector> 	&Ycoeff,	// (INPUT) 1D list of size NC, listing polynomial curves. The i-th element is a 2D array of size NX x (Ydegrees[i]+1), in row-major format, listing polynomial coefficients of Y_i at each grid point
												const lvector 				&Ydegrees,	// (INPUT) 1D array of size NC, listing the polynomial degrees of the curves Y_i
												const bool					slideX,		// (INPUT) whether the input polynomials are defined w.r.t. a sliding-X, i.e. in the interval X[i]:X[i+1] each Yi is defined as a polynomial of (X-X[i]) rather than X
												const dvector				&weights,	// (INPUT) 1D array of size NC, listing the weight w_i of each polynomial curve
												const double				Xmin,		// (INPUT)
												const double				Xmax,		// (INPUT)
												const double				V,			// (INPUT) the target value
												const bool					monotonic,	// (INPUT) if true, the function Y(x):=sum_{i=1}^NC w_i*Y_i(x) is guaranteed to be monotonic in x. In this case, the provided Xmin & Xmax don't necessarily need to bracket the solution and may be automatically searched for if needed
												const double				xepsilon,			// (INPUT) absolute error tolerance on an x-scale. If the x-step is within this threshold, the search is halted
												const double				yepsilon,			// (INPUT) absolute error tolerance on a y-scale. If |y-V| is within this threshold, the search is halted
												const long					max_iterations,		// (INPUT) maximum number of iterations allowed, i.e. prior to halting the search
												long						&Xmin_grid_guess,	// (INPUT/OUTPUT) a guess for which grid point Xmin may be located in. For efficiency reasons only. Set to -1 if not available.
												long						&Xmax_grid_guess){	// (INPUT/OUTPUT) a guess for which grid point Xmax may be located in. For efficiency reasons only. Set to -1 if not available.
	const long NC = Ydegrees.size();
	if(NC==0) return NAN_D;
	if(NC==1){
		if(weights[0]==0) return NAN_D;
		return solve_piecewise_polynomial_bisection(X, Ycoeff[0], Ydegrees[0], slideX, Xmin, Xmax, V/weights[0], monotonic, xepsilon, yepsilon, max_iterations, Xmin_grid_guess, Xmax_grid_guess);
	}
	const long max_degree = array_max(Ydegrees);
	
	double a = Xmin, b = Xmax, c;
	double fa, fb, fc;
	long iterations = 0, ga, gb, gc = -1;

	ga  = max(0l,find_next_left_grid_point(X, a, Xmin_grid_guess));
	fa  = sum_piecewise_polynomials(Ycoeff, Ydegrees, weights, a-(slideX ? X[ga] : 0.0), ga);
	Xmin_grid_guess = ga;

	gb  = max(0l,find_next_left_grid_point(X, b, Xmax_grid_guess));
	fb  = sum_piecewise_polynomials(Ycoeff, Ydegrees, weights, b-(slideX ? X[gb] : 0.0), gb);
	Xmax_grid_guess = gb;
			
	if(fa==V) return a;
	if(fb==V) return b;
	if(sgn(fa-V)==sgn(fb-V)){
		if(!monotonic) return NAN_D; // [Xmin,Xmax] does not bracket the solution x and the query function Y(x) is not monotonic, so it's hard to find a suitable bracket
		// search for a suitable bracket, i.e. extend [Xmin, Xmax] until V is within [f(Xmin),f(Xmax)]
		double L;
		while(true){
			// double the bracket span
			L  = b - a;
			a -= L/2;
			b += L/2;
			ga = max(0l,find_next_left_grid_point(X, a, ga));
			fa = sum_piecewise_polynomials(Ycoeff, Ydegrees, weights, a-(slideX ? X[ga] : 0.0), ga);
			gb = max(0l,find_next_left_grid_point(X, b, gb));
			fb = sum_piecewise_polynomials(Ycoeff, Ydegrees, weights, b-(slideX ? X[gb] : 0.0), gb);
			++iterations;
			if(sgn(fa-V)!=sgn(fb-V)) break;
			if(iterations>=max_iterations) return NAN_D;
		}
		Xmin_grid_guess = ga;
		Xmax_grid_guess = gb;
	}
	
	gc = long(0.5*(ga+gb)); // first guess for gc
	while(iterations<max_iterations){
		if(ga==gb){
			// root is bracketted within a single grid-interval
			if(max_degree==1){
				// all Y_i are at most linear in this interval, so we can directly calculate the solution's location
				// Y = alpha*x + beta
				double alpha=0, beta=0;
				for(long i=0; i<NC; ++i){
					alpha += weights[i] * ((Ydegrees[i]>=1) ? Ycoeff[i][ga*(Ydegrees[i]+1)+1] : 0.0);
					beta  += weights[i] * ((Ydegrees[i]>=0) ? Ycoeff[i][ga*(Ydegrees[i]+1)+0] : 0.0);
				}
				return X[ga] + (V-beta)/alpha;
			}
		}
		c   = 0.5*(a+b);
		gc  = max(0l,find_next_left_grid_point(X, c, gc));
		fc  = sum_piecewise_polynomials(Ycoeff, Ydegrees, weights, c-(slideX ? X[gc] : 0.0), gc);
		if((fc==V) || (abs(b-a)<=xepsilon) || (abs(fc-V)<=yepsilon)){
			return c;
		}
		if(sgn(fc-V)==sgn(fa-V)){
			a  = c;
			fa = fc;
			ga = gc;
		}else{
			b  = c;
			fb = fc;
			gb = gc;
		}
		++iterations;
	}
	return c; // did not converge, but the last c is the best estimate
}



// class for constructing, storing and evaluating piecewise polynomials
// evaluating a polynomial at some specific x-value is fastest when consecutively requested x values are close to each other (e.g. as usual for ODE solvers)
template<class VALUE_TYPE>
class PiecewisePolynomial{
protected:	
	ExtrapolationType extrapolation_type_left, extrapolation_type_right;
	VALUE_TYPE	extrapolation_value_left, extrapolation_value_right; // extrapolation values to left and right, in case the respective extraplation type is ExtrapolationTypeConst
	
	mutable long last_requested_index;
	
	// get Y-value at the requested x, using polynomial coefficients in cell i
	VALUE_TYPE get_value_in_cell(const long i, double x) const{
		double V = 0;
		if(slideX) x -= X[i];
		for(long p=0; p<=degree; ++p) V += coeff[i*(degree+1)+p] * pow(x,p);
		return V;
	}
	
	// get 1st derivative at the requested x, using polynomial coefficients in cell i
	VALUE_TYPE get_derivative_in_cell(const long i, double x) const{
		double D = 0;
		if(slideX) x -= X[i];
		for(long p=1; p<=degree; ++p) D += p*coeff[i*(degree+1)+p] * pow(x,p-1);
		return D;
	}
	
public:
	
	// these variables are public for efficient reading, but should only be modified from within this class
	std::vector<double> X;
	std::vector<VALUE_TYPE> coeff; // 2D array of size NX x (degree+1)
	long degree; // polynomial degree in each grid interval
	bool slideX;

	// constructors
	PiecewisePolynomial(): last_requested_index(-1), degree(0), slideX(false) {}

	PiecewisePolynomial(const std::vector<double> 		&X_, 		// (INPUT) 1D array of size NX, listing x-values in ascending order
						const std::vector<VALUE_TYPE> 	&coeff_, 	// (INPUT) 2D array of size NX x (degree+1), listing polynomial coefficients in each grid cell
						const long 						degree_,	// (INPUT) polynomial degree
						const bool						slideX_,	// (INPUT) whether the polynomial is defined w.r.t. the sliding-X, i.e. as a function of X-X[g] in the g-th grid cell
						const ExtrapolationType			extrapolation_type_left_,		// (INPUT) how to extrapolate outside of the X-domain, towards the left (i.e. <X[0])
						const ExtrapolationType			extrapolation_type_right_,		// (INPUT) how to extrapolate outside of the X-domain, towards the right (i.e. >X.back())
						const VALUE_TYPE				&extrapolation_value_left_,		// (INPUT) value to return for requests outside (leftwards) of the x-grid range. Only relevant if extrapolation_type_left_==ExtrapolationTypeConst
						const VALUE_TYPE				&extrapolation_value_right_){	// (INPUT) value to return for requests outside (rightwards) of the x-grid range. Only relevant if extrapolation_type_right_==ExtrapolationTypeConst
		degree 	= degree_;
		X 		= X_;
		coeff 	= coeff_;
		slideX	= slideX_;
		last_requested_index 		= -1;
		extrapolation_type_left		= extrapolation_type_left_;
		extrapolation_type_right	= extrapolation_type_right_;
		extrapolation_value_left 	= extrapolation_value_left_;
		extrapolation_value_right 	= extrapolation_value_right_;
	}


	VALUE_TYPE operator()(const double x) const{
		if(X.empty() && (coeff.size()==1)) return coeff[0]; // defined as constant functor
		if(x<X[0]){
			if(extrapolation_type_left==ExtrapolationTypeNaN){
				return VALUE_TYPE(NAN_D);
			}else if(extrapolation_type_left==ExtrapolationTypeConst){
				return extrapolation_value_left;
			}else if(extrapolation_type_left==ExtrapolationTypeLinear){
				if(X.size()<2) return VALUE_TYPE(NAN_D);
				const VALUE_TYPE y0 = get_value_in_cell(0, X[0]);
				const VALUE_TYPE y1 = get_value_in_cell(1, X[1]);
				return y0 + (x-X[0]) * (y1-y0)/(X[1]-X[0]);
			}else if(extrapolation_type_left==ExtrapolationTypePolynomial){
				return get_value_in_cell(0,x);
			}else{
				return VALUE_TYPE(NAN_D);
			}
		}
		if(x>X.back()){
			if(extrapolation_type_right==ExtrapolationTypeNaN){
				return VALUE_TYPE(NAN_D);
			}else if(extrapolation_type_right==ExtrapolationTypeConst){
				return extrapolation_value_right;
			}else if(extrapolation_type_right==ExtrapolationTypeLinear){
				if(X.size()<2) return VALUE_TYPE(NAN_D);
				const long N = X.size();
				const double x0 = X[N-1];
				const double x1 = X[N-2];
				const VALUE_TYPE y0 = get_value_in_cell(N-1, x0);
				const VALUE_TYPE y1 = get_value_in_cell(N-2, x1);
				return y0 + (x-x0) * (y1-y0)/(x1-x0);
			}else if(extrapolation_type_right==ExtrapolationTypePolynomial){
				return get_value_in_cell(X.size()-1,x);
			}else{
				return VALUE_TYPE(NAN_D);
			}
		}
		last_requested_index = find_next_left_grid_point(X, x, last_requested_index);
		return get_value_in_cell(last_requested_index,x);
	}
	
	
	VALUE_TYPE value_at_grid_point(const long i) const{
		return get_value_in_cell(i, X[i]);
	}
	
	
	VALUE_TYPE derivative(const double x) const{
		if((x<X[0]) || (x>X.back())) return NAN_D;
		last_requested_index = find_next_left_grid_point(X, x, last_requested_index);
		return get_derivative_in_cell(last_requested_index,x);
	}
	

	VALUE_TYPE derivative_at_grid_point(const long i) const{
		return get_derivative_in_cell(i,X[i]);
	}
	
	
	bool set_to_piecewise_linear(	const std::vector<double> 		&X_, 	// (INPUT) 1D array of size NX, listing x-grid values in ascending order
									const std::vector<VALUE_TYPE> 	&Y,		// (INPUT) 1D array of size NX, listing y values on the grid
									const bool						slideX_,						// (INPUT) whether to define the polynomial w.r.t. the sliding-X
									const ExtrapolationType			extrapolation_type_left_,		// (INPUT) how to extrapolate outside of the X-domain, towards the left (i.e. <X[0])
									const ExtrapolationType			extrapolation_type_right_,		// (INPUT) how to extrapolate outside of the X-domain, towards the right (i.e. >X.back())
									const VALUE_TYPE				&extrapolation_value_left_,		// (INPUT) value to return for requests outside (leftwards) of the x-grid range. Only relevant if extrapolation_type_left_==ExtrapolationTypeConst
									const VALUE_TYPE				&extrapolation_value_right_){	// (INPUT) value to return for requests outside (rightwards) of the x-grid range. Only relevant if extrapolation_type_right_==ExtrapolationTypeConst
		last_requested_index 		= -1;
		extrapolation_type_left		= extrapolation_type_left_;
		extrapolation_type_right	= extrapolation_type_right_;
		extrapolation_value_left	= extrapolation_value_left_;
		extrapolation_value_right 	= extrapolation_value_right_;
		X 	   = X_;
		degree = 1;
		slideX = slideX_;
		const long N = X.size();
		if(N==0) return false;
		coeff.resize(N*(degree+1));
		VALUE_TYPE alpha,beta;
		for(long g=0, g2; g<N; ++g){
			if(g==N-1){
				g2 = N-2;
			}else{
				g2 = g+1;
			}
			beta  = (Y[g]-Y[g2])/(X[g]-X[g2]);
			alpha = Y[g2]-beta*(X[g2]-(slideX ? X[g] : 0.0));
			coeff[g*2+0] = alpha;
			coeff[g*2+1] = beta;
		}
		return true;
	}
	
	
	// initialize as splines of a given degree
	// Note: Currently only degrees 0, 1, 2 or 3 are supported
	bool set_to_spline(	const std::vector<double> 		&X_, 		// (INPUT) 1D array of size NX, listing x-grid values in ascending order
						const std::vector<VALUE_TYPE> 	&Y, 		// (INPUT) 1D array of size NX, listing y values on the grid
						const long 						degree_,	// (INPUT) desired polynomial degree for the constructed splines. Currently only values 0, 1, 2 and 3 are supported.
						const bool						slideX_,						// (INPUT) whether to define the polynomial w.r.t. the sliding-X
						const ExtrapolationType			extrapolation_type_left_,		// (INPUT) how to extrapolate outside of the X-domain, towards the left (i.e. <X[0])
						const ExtrapolationType			extrapolation_type_right_,		// (INPUT) how to extrapolate outside of the X-domain, towards the right (i.e. >X.back())
						const VALUE_TYPE				&extrapolation_value_left_,		// (INPUT) value to return for requests outside (leftwards) of the x-grid range. Only relevant if extrapolation_type_left_==ExtrapolationTypeConst
						const VALUE_TYPE				&extrapolation_value_right_){	// (INPUT) value to return for requests outside (rightwards) of the x-grid range. Only relevant if extrapolation_type_right_==ExtrapolationTypeConst
		if((degree<0) || (degree>3)) return false;
		last_requested_index 		= -1;
		extrapolation_type_left		= extrapolation_type_left_;
		extrapolation_type_right	= extrapolation_type_right_;
		extrapolation_value_left	= extrapolation_value_left_;
		extrapolation_value_right 	= extrapolation_value_right_;
		X		= X_;
		degree	= degree_;
		slideX 	= slideX_;
		if(X.empty()) return false;
		get_spline(X, Y, degree, slideX, coeff);
		return true;
	}
	
	
	void set_to_constant(const VALUE_TYPE &value){
		X.clear();
		coeff.assign(1,value);
		degree = 0;
		slideX = false;
		extrapolation_type_left   = ExtrapolationTypeConst;
		extrapolation_type_right  = ExtrapolationTypeConst;
		extrapolation_value_left  = value;
		extrapolation_value_right = value;
	}
	
	
	// add a constant increment to the polynomial
	// Note that this does not affect the extrapolation types/values
	void add(const VALUE_TYPE &added_value){
		const long NG = X.size();
		for(long g=0; g<NG; ++g){
			coeff[g*(degree+1) + 0] += added_value;
		}
	}
	
	
	// estimate maximum value within the domain covered by X
	VALUE_TYPE getMaxAbs() const{
		double maxAbs = 0;
		double Xshift;
		for(long i=0; i<X.size()-1; ++i){
			Xshift = (slideX ? X[i] : 0.0);
			maxAbs = max(maxAbs,polynomial_bound_abs(degree, &coeff[i*(degree+1)+0],X[i]-Xshift,X[i+1]-Xshift));
		}
		return maxAbs;
	}
	
	// estimate maximum value within the domain covered by X, restricted to some sub-interval
	VALUE_TYPE getMaxAbs(const double minX, const double maxX) const{
		double maxAbs = 0;
		double Xshift;
		for(long i=0; i<X.size()-1; ++i){
			if(X[i]<minX) continue;
			if(X[i]>maxX) break;
			Xshift = (slideX ? X[i] : 0.0);
			maxAbs = max(maxAbs,polynomial_bound_abs(degree, &coeff[i*(degree+1)+0],X[i]-Xshift,X[i+1]-Xshift));
		}
		return maxAbs;
	}
	
	
	// modify the way extrapolation is performed beyond the domain covered by the X grid
	void set_extrapolation_type(const ExtrapolationType			extrapolation_type_left_,		// (INPUT) how to extrapolate outside of the X-domain, towards the left (i.e. <X[0])
								const ExtrapolationType			extrapolation_type_right_,		// (INPUT) how to extrapolate outside of the X-domain, towards the right (i.e. >X.back())
								const VALUE_TYPE				&extrapolation_value_left_,		// (INPUT) value to return for requests outside (leftwards) of the x-grid range. Only relevant if extrapolation_type_left_==ExtrapolationTypeConst
								const VALUE_TYPE				&extrapolation_value_right_){	// (INPUT) value to return for requests outside (rightwards) of the x-grid range. Only relevant if extrapolation_type_right_==ExtrapolationTypeConst
		extrapolation_type_left		= extrapolation_type_left_;
		extrapolation_type_right	= extrapolation_type_right_;
		extrapolation_value_left	= extrapolation_value_left_;
		extrapolation_value_right 	= extrapolation_value_right_;
	}
	
	
	// calculate the antiderivative of the function over x, i.e. A(x):=int_Xstart^x Y(u) du
	// the result will be a piecewise polynomial of higher degree, stored in A
	// returns true on success
	bool get_antiderivative(const double Xstart, 						// (INPUT) the x-value where the antiderivative should be zero
							PiecewisePolynomial<VALUE_TYPE> &A) const{	// (OUTPUT) the calculated antiderivative, again stored as a piecewise polynomial
		const long NG = X.size();
		if(NG==0) return false;
		A.degree = max(degree+1, max((extrapolation_type_left==ExtrapolationTypeLinear ? 2l : 0l), (extrapolation_type_right==ExtrapolationTypeLinear ? 2l : 0l))); // depending on the extrapolation type for A, A may need to have at least quadratic degree
		A.last_requested_index = -1;
		A.slideX = slideX;
		
		// define X grid for A
		// you may need to include a dummy grid point on the left and/or right to accommodate special extrapolation needs for A
		// in any case, A.X[g+A_grid_offset_left] corresponds to X[g]
		const int A_grid_offset_left = (extrapolation_type_left==ExtrapolationTypeConst || extrapolation_type_left==ExtrapolationTypeLinear ? 1 : 0);
		const int A_grid_offset_right = (extrapolation_type_right==ExtrapolationTypeConst || extrapolation_type_right==ExtrapolationTypeLinear ? 1 : 0);
		const long NAG = NG + A_grid_offset_left + A_grid_offset_right;
		A.X.resize(NAG);
		if(A_grid_offset_left>0) A.X[0] = X[0]-(NG>1 ? X[1]-X[0] : (X[0]==0 ? 1.0 : abs(X[0])));
		if(A_grid_offset_right>0) A.X[A.X.size()-1] = X[NG-1]+(NG>1 ? X[NG-1]-X[NG-2] : (X[NG-1]==0 ? 1.0 : abs(X[NG-1])));
		for(long i=0; i<NG; ++i){
			A.X[i+A_grid_offset_left] = X[i];
		}
		
		// calculate polynomial coefficients of A
		// first calculate antiderivative A such that A(X[0]) = 0, shift afterwads
		// recall that A.X[g+A_grid_offset_left] corresponds to X[g]
		A.coeff.assign(NAG*(A.degree+1),0.0);
		VALUE_TYPE cumulativeA(0);
		double alpha;
		for(long g=0; g<NG; ++g){
			A.coeff[(A_grid_offset_left+g)*(A.degree+1) + 0] = cumulativeA;
			for(long p=0; p<=degree; ++p){
				// term of degree p in Y gives rise to a term of degree (p+1) in the antiderivative, as well as part of degree 0
				alpha = coeff[g*(degree+1) + p]/(p+1);
				A.coeff[(A_grid_offset_left+g)*(A.degree+1) + (p+1)] = alpha;
				if(!slideX) A.coeff[(A_grid_offset_left+g)*(A.degree+1) + 0] -= alpha * pow(X[g],p+1);
				if(g<NG-1) cumulativeA += (slideX ? alpha*pow(X[g+1]-X[g],p+1) : alpha*(pow(X[g+1],p+1) - pow(X[g],p+1)));
			}
		}
		
		// decide how to extrapolate A beyond edges and define polynomial coefficients on dummy grid points if needed
		if(extrapolation_type_left==ExtrapolationTypeNaN){
			A.extrapolation_type_left=ExtrapolationTypeNaN;
		}else if(extrapolation_type_left==ExtrapolationTypeConst){
			// A is extrapolated towards the left as if one integrated the constant function Y(x) = extrapolation_type_left, with A(X[0]) = 0.
			// define coefficients on left-end dummy grid point
			A.extrapolation_type_left = ExtrapolationTypePolynomial;
			A.coeff[0*(A.degree+1)+0] = (slideX ? 0.0 : -X[0]*extrapolation_value_left);
			A.coeff[0*(A.degree+1)+1] = extrapolation_value_left;
		}else if(extrapolation_type_left==ExtrapolationTypeLinear){
			A.extrapolation_type_left=ExtrapolationTypePolynomial;
			const double x0 = X[0];
			const double x1 = X[1];
			const VALUE_TYPE y0 = get_value_in_cell(0, X[0]);
			const VALUE_TYPE y1 = get_value_in_cell(1, X[1]);
			const double beta  = (y1-y0)/(x1-x0);
			const double alpha = y0 - x0 * beta;
			// A is extrapolated towards the left as if one integrated the linear function Y(x) = alpha + beta*x, with A(X[0]) = 0.
			// Note that if slidex=true, the coefficients should define a polynomial in (x-X[0]) rather than x
			A.coeff[0*(A.degree+1)+0] = 0 - (slideX ? 0.0 : alpha*X[0]+0.5*beta*SQ(X[0]));
			A.coeff[0*(A.degree+1)+1] = alpha + (slideX ? beta*X[0] : 0.0);
			A.coeff[0*(A.degree+1)+2] = 0.5*beta;
		}else if(extrapolation_type_left==ExtrapolationTypePolynomial){
			// use the same polynomial coefficients as at X[0] to extrapolate leftwards, so need need to include a dummy grid point
			A.extrapolation_type_left=ExtrapolationTypePolynomial;
		}else{
			A.extrapolation_type_left=ExtrapolationTypeNaN;
		}
		if(extrapolation_type_right==ExtrapolationTypeNaN){
			A.extrapolation_type_right=ExtrapolationTypeNaN;
		}else if(extrapolation_type_right==ExtrapolationTypeConst){
			// A is extrapolated towards the right as if one integrated the constant function Y(x) = extrapolation_type_right, with A(X[NG-1])=Aend as calculated already above
			// store coefficients on right-end dummy grid point
			A.extrapolation_type_right=ExtrapolationTypePolynomial;
			const VALUE_TYPE Aend = polynomial_value(A.degree,&A.coeff[(NAG-2)*(A.degree+1)],(slideX ? 0.0 : X[NG-1]));
			A.coeff[(NAG-1)*(A.degree+1)+0] = Aend + (slideX ? 0.0 : - X[NG-1]*extrapolation_value_right);
			A.coeff[(NAG-1)*(A.degree+1)+1] = extrapolation_value_right;
		}else if(extrapolation_type_right==ExtrapolationTypeLinear){
			A.extrapolation_type_right=ExtrapolationTypePolynomial;
			const double x0 = X[NG-1];
			const double x1 = X[NG-2];
			const VALUE_TYPE y0 = get_value_in_cell(NG-1, X[NG-1]);
			const VALUE_TYPE y1 = get_value_in_cell(NG-2, X[NG-2]);
			const double beta  = (y1-y0)/(x1-x0);
			const double alpha = y0 - x0 * beta;
			// A is extrapolated towards the right as if one integrated the linear function Y(x) = alpha + beta*x, with A(X[NG-1])=Aend as calculated already above
			// store coefficients on right-end dummy grid point
			// Note that if slidex=true, the coefficients should define a polynomial in (x-X[NG-1]) rather than x
			const VALUE_TYPE Aend = polynomial_value(A.degree,&A.coeff[(NAG-2)*(A.degree+1)],(slideX ? 0.0 : X[NG-1]));
			A.coeff[(NAG-1)*(A.degree+1)+0] = Aend - (slideX ? 0.0 : alpha*X[NG-1]+0.5*beta*SQ(X[NG-1]));
			A.coeff[(NAG-1)*(A.degree+1)+1] = alpha + (slideX ? beta*X[NG-1] : 0.0);
			A.coeff[(NAG-1)*(A.degree+1)+2] = 0.5*beta;
		}else if(extrapolation_type_right==ExtrapolationTypePolynomial){
			// use the same polynomial coefficients as at X[NG-1] to extrapolate rightwards, so need need to include a dummy grid point
			A.extrapolation_type_right=ExtrapolationTypePolynomial;
		}else{
			A.extrapolation_type_right=ExtrapolationTypeNaN;
		}
		
		// shift antiderivative if needed, i.e. if Xstart is different from the convention used above (where A=0 at X[0]).
		if(Xstart!=X[0]){
			const long g_start = max(0l,find_next_left_grid_point(A.X, Xstart, -1));
			const VALUE_TYPE Astart = polynomial_value(A.degree,&A.coeff[g_start*(A.degree+1)],Xstart-(slideX ? A.X[g_start] : 0.0));
			for(long g=0; g<NAG; ++g) A.coeff[g*(A.degree+1)+0] -= Astart;
		}
				
		return true;
	}
	
	
	// get an approximation for E(x):=exp(f(x)) as a new piecewise quadratic polynomial
	void get_quadratic_approximation_of_exp(PiecewisePolynomial<VALUE_TYPE> &E) const{	// (OUTPUT) the approximation for exp(f(x)), again stored as a piecewise polynomial
		E.degree 				= 2;
		E.last_requested_index 	= -1;
		E.slideX 				= slideX;
		E.X 					= X;
		quadratic_approximation_of_piecewise_exp_polynomial(X, degree, coeff, slideX, E.coeff);
	}
};




#pragma mark -
#pragma mark Integration, antiderivatives, derivatives
#pragma mark 



// integrate a scalar function over a finite 1D interval using the trapezoid rule, with adaptively refined x-steps
template<class FUNCTOR>
double integrate1D(	const FUNCTOR &f, 			// (INPUT) functor to be integrated. Must be a function f:[xmin,xmax]->R
					const double xmin,			// (INPUT) left end of the x-interval over which to integrate
					const double xmax,			// (INPUT) right end of the x-interval over which to integrate
					const double default_dx,	// (INPUT) default x-step. Note that locally x-steps may be refined to achieve a y-change smaller than max_dy
					const double min_dx,		// (INPUT) minimum allowed x-step, i.e. refinements are not allowed to lead to a dx smaller than min_dx
					const double max_dy,		// (INPUT) max allowed y-change during a single x-step. This constraint is weaker than min_dx, i.e. min_dx will be obeyed even if max_dy is violated. Set this to INFTY_D to disable.
					const double max_rel_dy){	// (INPUT) max allowed relative y-change during a single x-step. This constraint is weaker than min_dx, i.e. min_dx will be obeyed even if max_dy is violated. Set this to INFTY_D to disable.
	if(xmin==xmax) return 0;
	double S  = 0;
	double x  = xmin;
	double dx = default_dx;
	double y  = f(x);
	double x2, y2;
	while(x<xmax){
		x2 = min(xmax, x+dx);
		dx = x2-x;
		y2 = f(x2);
		while(((abs(y2-y)>max_dy) || (abs(y2-y)>max_rel_dy*0.5*(abs(y)+abs(y2)))) && (dx*0.9>min_dx)){
			// refine this x-step
			dx = max(dx/2, min_dx);
			x2 = min(xmax,x + dx);
			dx = x2-x;
			y2 = f(x2);
		}
		// accept current x-step (either because we reached min_dx or because we satisfied max_dy)
		S += dx*0.5*(y+y2);
		x  = x2;
		y  = y2;
		if(dx<default_dx){
			dx = min(dx*2, default_dx); // carefully increase dx step again
		}
	}
	return S;
}


// integrate a piecewise linear time series between two time points
// this function can skip over NAN values
template<class TIME_ARRAY>
double integrate1D(const TIME_ARRAY &times, const std::vector<double> &values, const long start, const long end, const bool ignore_inf){
	double S = 0;
	long last_valid_t = -1;
	for(long t=max(0l,start); t<=end; ++t){
		if(std::isnan(values[t]) || (ignore_inf && abs(values[t])==INFTY_D)){
			continue;
		}else if(last_valid_t<0){ 
			last_valid_t = t;
			continue;
		}
		S += (times[t]-times[last_valid_t])*0.5*(values[t]+values[last_valid_t]);
		last_valid_t = t;
	}
	return S;
}


// integrate a piecewise linear curve on a 1D interval: \int_{Xstart}^{Xend} Y(x) dx
double integrate_piecewise_linear(	const dvector	&X,			// (INPUT) 1D array of size N, listing x-values in ascending order. If this grid does not fully cover the integration interval [Xstart,Xend], Y is extrapolated as a constant where necessary
									const dvector 	&Y,			// (INPUT) !D array of size N, listing the the values of Y on the grid
									double			Xstart,		// (INPUT) lower end of the integration interval
									double			Xend){		// (INPUT) upper end of the integration interval
	const long N = X.size();
	// if integration ends are in negative direction, switch to positive direction, then correct the sign of the calculated integral at the end
	const double factor = (Xstart>Xend ? -1.0 : 1.0);
	if(Xstart>Xend) std::swap(Xstart, Xend);
	
	double S = 0;
	if(Xend<X[0]){
		// the entire integration interval is outside (leftwards) of the grid X[]
		S = Y[0] * (Xend-Xstart);
	}else if(Xstart>X[N-1]){
		// the entire integration interval is outside (rightwards) of the grid X[]
		S = Y[N-1] * (Xend-Xstart);
	}else{
		// if integration interval is not entirely covered by the X-grid, extrapolate Y as constant
		if(Xstart<X[0]){
			S += Y[0] * (X[0]-Xstart);
		}
		if(Xend>X[N-1]){
			S += Y[N-1]* (Xend-X[N-1]);
		}
		// integrate over the part that's covered by the X-grid
		long g = (Xstart<X[0] ? 0 : find_next_left_grid_point(X, Xstart, -1));
		double x1, x2;
		while((g<N-1) && (X[g]<=Xend)){
			x1 = max(Xstart, X[g]);
			x2 = min(Xend, X[g+1]);
			S += (x2-x1)*0.5*(Y[g+1]+Y[g]);
			++g;
		}
	}
	
	return factor*S;
}



// WARNING: THIS FUNCTION SHOULD WORK, BUT HAS NOT YET BEEN THOROUGHLY DEBUGGED
// integrate a piecewise polynomial curve on a 1D interval
double integrate_piecewise_polynomial(	const dvector	&X,			// (INPUT) 1D array of size N, listing x-values in ascending order
										const long		P,			// (INPUT) polynomial order of the input function Y=Y(x) in each interval. For example, if Y is piecewise linear then P=1.
										const dvector 	&Ycoeff,	// (INPUT) 2D array of size N x (P+1) in row-major format, listing the polynomial coefficients (f0,f1,..,fP) of the function Y in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has Y(x) = Ycoeff[i*(P+1)+0] + Ycoeff[i*(P+1)+1]*x + .. + Ycoeff[i*(P+1)+P]*x^P.
										double			Xstart,		// (INPUT) lower end of the integration interval
										double			Xend){		// (INPUT) upper end of the integration interval
	const long N = X.size();
	// if integration ends are in negative direction, switch to positive direction, then correct the sign of the calculated integral at the end
	const double factor = (Xstart>Xend ? -1.0 : 1.0);
	if(Xstart>Xend) std::swap(Xstart, Xend);
	
	double S = 0;
	double alpha;
	if(Xend<X[0]){
		// the entire integration interval is outside (leftwards) of the grid X[]
		// use polynomial coefficients at X[0]
		for(long p=0; p<=P; ++p){
			alpha = Ycoeff[0*(P+1)+p]/(p+1);
			S += alpha*(pow(Xend,p+1) - pow(Xstart,p+1));
		}
		return S;
	}	
	// at this point it is guaranteed that Xend>=X[0]
	
	long g = (Xstart<X[0] ? 0 : find_next_left_grid_point(X, Xstart, -1));
	double xnext, xlast=Xstart;
	long p;
	while((g<N) && (X[g]<=Xend)){
		xnext = (g<N-1 ? min(Xend, X[g+1]) : Xend);
		for(p=0; p<=P; ++p){
			alpha = Ycoeff[g*(P+1)+p]/(p+1);
			S += alpha*(pow(xnext,p+1) - pow(xlast,p+1));
		}
		++g;
		xlast = xnext;
	}
	
	return factor*S;
}


// calculate the antiderivative of a piecewise linear function y=y(x) over a 1D grid, i.e. calculate:
//   A(x_i):=int_{Xstart}^{x_i} y(x)dx
// for all grid points x_1,x_2,.., 
// Assuming: y is piecewise linear between grid points. Beyond the range covered by X[], Y is extrapolated linearly if needed (i.e. if Xstart is outside the range of X[])
template<class VALUE_TYPE>
void get_antiderivative(const std::vector<double> 		&X,			// (INPUT) 1D array of size N, listing x-values in ascending order
						const std::vector<VALUE_TYPE> 	&Y,			// (INPUT) 1D array of size N, listing y-values in ascending order
						const double					Xstart,		// (INPUT) X value where antiderivative is zero
						std::vector<VALUE_TYPE>			&A){		// (OUTPUT) 1D array of size N, listing the computed antiderivative on the same x-grid
	const long N = X.size();
	A.resize(N);
	if(N==0) return;
	if(N==1){
		A[0] = (Xstart==X[0] ? 0.0 : NAN_D);
		return;
	}

	// first calculate the antiderivative assuming Xstart=X[0], correct afterwards if needed
	A[0] = 0;
	for(long i=1; i<N; ++i){
		A[i] = A[i-1] + (X[i]-X[i-1])*0.5*(Y[i]+Y[i-1]);
	}
	
	// shift antiderivative if needed, i.e. if Xstart is different from the convention used above (where A=0 at Xgrid[0]).
	if(Xstart!=X[0]){
		long g_start = max(0l, find_next_left_grid_point(X, Xstart, -1));
		if(g_start==N-1) g_start = N-2;
		const double Ystart = Y[g_start] + (Xstart-X[g_start])*(Y[g_start+1]-Y[g_start])/(X[g_start+1]-X[g_start]);
		const VALUE_TYPE Astart = A[g_start] + (Xstart-X[g_start])*0.5*(Ystart+Y[g_start]);
		for(long g=0; g<N; ++g){
			A[g] -= Astart;	
		}
	}

}




// calculate the antiderivative of a piecewise linear curve Y=Y(x) over a 1D grid, i.e. calculate:
//   A(x_i):=int_Xstart^{x_i} Y(x)dx
// for all grid points x_1,x_2,.., 
// Assuming: Y is piecewise linear between grid points
// Since Y is assumed to be linear within each grid segment (i.e. between adjacent x-grid points), 
//   the antiderivative will be piecewise quadratic, i.e. with polynomial coefficients a0, a1, a2 such that:
//   A(x) = a0 + a1*x + a2*x^2
// This function thus not only returns the values of A on the original grid, but also the polynomial coefficients a0,a1,a2 for each grid interval
template<class VALUE_TYPE>
void get_antiderivative(const std::vector<double> 		&X,				// (INPUT) 1D array of size N, listing x-values in ascending order
						const double					&Xstart,		// (INPUT) lower end of the integration, i.e. x-value where antiderivative is set to zero
						const std::vector<VALUE_TYPE> 	&Y,				// (INPUT) 1D array of size N, listing y-values defined on the x-grid
						std::vector<VALUE_TYPE>			&A,				// (OUTPUT) 1D array of size N, listing the computed antiderivative on the same x-grid
						std::vector<VALUE_TYPE>			&Acoeff){		// (OUTPUT) 2D array of size N x 3 in row-major format, listing the polynomial coefficients (a0,a1,a2) of the antiderivative in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has A(x) = Acoeff[i*3+0] + Acoeff[i*3+1]*x + Acoeff[i*3+2]*x^2.
	const long N = X.size();
	A.resize(N);
	Acoeff.resize(N*3);
	if(N==0) return;

	// first calculate the antiderivative assuming Xstart=X[0], correct afterwards if needed
	A[0] = 0;
	double alpha, beta; // polynomial coefficients for the piecewise-linear Y at the focal grid interval
	for(long i=1; i<N; ++i){
		beta  = (Y[i]-Y[i-1])/(X[i]-X[i-1]);
		alpha = Y[i-1]-X[i-1]*beta;
		A[i]  = A[i-1] + (X[i]-X[i-1])*0.5*(Y[i]+Y[i-1]);
		Acoeff[(i-1)*3+0] = A[i-1] - alpha*X[i-1] - (beta/2)*SQ(X[i-1]); // intercept
		Acoeff[(i-1)*3+1] = alpha;  // linear coefficient
		Acoeff[(i-1)*3+2] = beta/2; // quadratic coefficient
	}
	// set the coefficient at the last grid point equal to the coefficients at the second-last grid point
	Acoeff[(N-1)*3+0] = Acoeff[(N-2)*3+0];
	Acoeff[(N-1)*3+1] = Acoeff[(N-2)*3+1];
	Acoeff[(N-1)*3+2] = Acoeff[(N-2)*3+2];
	
	// shift antiderivative if needed, i.e. if Xstart is different from the convention used above (where A=0 at Xgrid[0]).
	if(Xstart!=X[0]){
		const long g_start = max(0l, find_next_left_grid_point(X, Xstart, -1));
		const VALUE_TYPE Astart = polynomial_value(3,&Acoeff[g_start*3],Xstart);
		for(long g=0; g<N; ++g){
			A[g] -= Astart;	
			Acoeff[g*3+0] -= Astart;
		}
	}
}



// calculate the antiderivative of a piecewise polynomial curve Y=Y(x) over a 1D grid, i.e. calculate:
//   A(x_i):=int_Xstart^{x_i} Y(x)dx
// for all grid points x_1,x_2,.., 
// Assuming: Y is piecewise polynomial of degree P between grid points, i.e.:
//   Y(x) = f0 + f1*x + .. + f_P*x^P
// Since Y is assumed to be polynomial within each grid segment (i.e. between adjacent x-grid points), 
//   the antiderivative will be piecewise polynomial of order P+1, i.e. with polynomial coefficients a0, a1, .., a_{P+1} such that:
//   A(x) = a0 + a1*x + .. + a_{P+1}*x^(P+1)
// This function thus not only returns the values of A on the original grid, but also the polynomial coefficients a0,a1,..,a_{P+1} for each grid interval
template<class VALUE_TYPE>
void get_antiderivative(const std::vector<double> 		&X,				// (INPUT) 1D array of size N, listing x-values in ascending order
						const double					&Xstart,		// (INPUT) lower end of the integration, i.e. x-value where antiderivative is set to zero
						const long						P,				// (INPUT) polynomial degree of the input function Y=Y(x) in each interval. For example, if Y is piecewise linear then P=1.
						const std::vector<VALUE_TYPE> 	&Ycoeff,		// (INPUT) 2D array of size N x (P+1) in row-major format, listing the polynomial coefficients (f0,f1,..,fP) of the function Y in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has Y(x) = Ycoeff[i*(P+1)+0] + Ycoeff[i*(P+1)+1]*x + .. + Ycoeff[i*(P+1)+P]*x^P.
						const bool						slideX,			// (INPUT) whether Y is defined as a function of the sliding-X, i.e. in the interval X[i]:X[i+1] Y is defined as a polynomial of (X-X[i]) rather than X. Note that the antiderivative will also be defined w.r.t. a sliding-X if Y is.
						std::vector<VALUE_TYPE>			&A,				// (OUTPUT) 1D array of size N, listing the computed antiderivative on the same x-grid
						std::vector<VALUE_TYPE>			&Acoeff){		// (OUTPUT) 2D array of size N x (P+2) in row-major format, listing the polynomial coefficients (a0,a1,..,a_P,a_{P+1}) of the antiderivative in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has A(x) = Acoeff[i*(P+2)+0] + Acoeff[i*(P+2)+1]*x + .. + Acoeff[i*(P+2)+P+1]*x^(P+1).
	const long N = X.size();
	A.resize(N);
	Acoeff.resize(N*(P+2));
	if(N==0) return;

	// first calculate the antiderivative assuming Xstart=X[0], correct afterwards if needed
	A[0] = 0;
	double alpha;
	for(long i=0; i<N; ++i){
		if(i<N-1) A[i+1] = A[i];
		Acoeff[i*(P+2) + 0] = A[i];
		for(long p=0; p<=P; ++p){
			// term of order p in Y gives rise to a term of order (p+1) in the antiderivative, as well as part of order 0
			alpha = Ycoeff[i*(P+1) + p]/(p+1);
			Acoeff[i*(P+2) + (p+1)]  = alpha;
			if(!slideX) Acoeff[i*(P+2) + 0] -= alpha * pow(X[i],p+1);
			if(i<N-1) A[i+1] += (slideX ? alpha*pow(X[i+1]-X[i],p+1) : alpha*(pow(X[i+1],p+1) - pow(X[i],p+1)));
		}
	}
	
	// shift antiderivative if needed, i.e. if Xstart is different from the convention used above (where A=0 at Xgrid[0]).
	if(Xstart!=X[0]){
		const long g_start = max(0l, find_next_left_grid_point(X, Xstart, -1));
		const VALUE_TYPE Astart = piecewise_polynomial_value(X,Acoeff,P+1,slideX,g_start,Xstart);
		for(long g=0; g<N; ++g){
			A[g] -= Astart;	
			Acoeff[g*(P+2)+0] -= Astart;
		}
	}
}



// calculate the antiderivative of a curve F=F(x) over a 1D grid, i.e. calculate:
//   A(x):=int_{Xstart}^{x} F(s) ds
// Assuming: F is the sum of two functions, F = Y + D:
//	 A piecewise polynomial between grid points, i.e. Y(x) = f0 + f1*x + .. + f_degree*x^degree
//	 A Dirac comb consisting of discrete Dirac distributions at specific x-locations, i.e. D(x) = delta(x-x0)*weight0 + delta(x-x1)*weight1 + ..
//     A requirement is that the Dirac comb's x-grid points are part of the X-grid used for the polynomial part, i.e. x0=X[DiracPoints[0]], x1=X[DiracPoints[1]] and so on.
// The returned antiderivative will be piecewise polynomial of order degree+1, i.e. with polynomial coefficients a0, a1, .., a_{degree+1} such that:
//   A(x) = a0 + a1*x + .. + a_{degree+1}*x^(degree+1)
// Note that if the Dirac comb D is non-zero, then the antiderivative will have discontinuities at the Dirac locations. A Dirac peak at a specific grid point X[i] will result in the antiderivative A being incremented by the Dirac peak's weight right at that grid point and beyond.
template<class VALUE_TYPE>
void get_antiderivative(const std::vector<double> 		&X,				// (INPUT) 1D array of size N, listing x-values in ascending order
						const double					&Xstart,		// (INPUT) lower end of the integration, i.e. x-value where antiderivative is set to zero
						const long						Ydegree,		// (INPUT) degree of the input polynomial Y=Y(x) in each interval. For example, if Y is piecewise linear then degree=1.
						const std::vector<VALUE_TYPE> 	&Ycoeff,		// (INPUT) 2D array of size N x (P+1) in row-major format, listing the polynomial coefficients (f0,f1,..,fP) of the function Y in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has Y(x) = Ycoeff[i*(degree+1)+0] + Ycoeff[i*(degree+1)+1]*x + .. + Ycoeff[i*(degree+1)+degree]*x^degree.
						const bool						slideX,			// (INPUT) whether Y is defined as a function of the sliding-X, i.e. in the interval X[i]:X[i+1] Y is defined as a polynomial of (X-X[i]) rather than X. Note that the antiderivative will also be defined w.r.t. a sliding-X if Y is.
						const std::vector<long>	 		&DiracPoints,	// (INPUT) 1D array of size ND, listing the X-grid indices where the Dirac peaks are located, in ascending order
						const std::vector<VALUE_TYPE> 	&DiracWeights,	// (INPUT) 1D array of size ND, listing the weights of the Dirac peaks
						std::vector<VALUE_TYPE>			&Acoeff){		// (OUTPUT) 2D array of size N x (P+2) in row-major format, listing the polynomial coefficients (a0,a1,..,a_degree,a_{degree+1}) of the antiderivative in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has A(x) = Acoeff[i*(degree+2)+0] + Acoeff[i*(degree+2)+1]*x + .. + Acoeff[i*(degree+2)+degree+1]*x^(degree+1).
	const long NX = X.size();
	const long ND = DiracPoints.size();
	Acoeff.resize(NX*(Ydegree+2));
	if(NX==0) return;

	// first calculate the antiderivative assuming Xstart=X[0], correct afterwards if needed
	VALUE_TYPE A = 0; // this will be the antiderivative at the current grid point
	VALUE_TYPE alpha;
	long next_Dirac = 0;
	for(long i=0; i<NX; ++i){
		if((next_Dirac<ND) && (i==DiracPoints[next_Dirac])){
			// passing through a Dirac peak, so need to increment antiderivative here
			A += DiracWeights[next_Dirac];
			++next_Dirac;
		}
		Acoeff[i*(Ydegree+2) + 0] = A;
		for(long p=0; p<=Ydegree; ++p){
			// term of order p in Y gives rise to a term of order (p+1) in the antiderivative, as well as part of order 0
			alpha = Ycoeff[i*(Ydegree+1) + p]/(p+1);
			Acoeff[i*(Ydegree+2) + (p+1)]  = alpha;
			if(!slideX) Acoeff[i*(Ydegree+2) + 0] -= alpha * pow(X[i],p+1);
			A += (slideX ? alpha*pow(X[i+1]-X[i],p+1) : alpha*(pow(X[i+1],p+1) - pow(X[i],p+1)));
		}
	}
	
	// shift antiderivative if needed, i.e. if Xstart is different from the convention used above (where A=0 at Xgrid[0]).
	if(Xstart!=X[0]){
		const long g_start = max(0l, find_next_left_grid_point(X, Xstart, -1));
		const VALUE_TYPE Astart = piecewise_polynomial_value(X,Acoeff,Ydegree+1,slideX,g_start,Xstart);
		for(long g=0; g<NX; ++g){
			Acoeff[g*(Ydegree+2)+0] -= Astart;
		}
	}
}


// given a scalar function defined as a splines on a 1D x-grid, calculate its values at some target x-values
// [[Rcpp::export]]
NumericVector evaluate_spline_CPP(	const std::vector<double> 	&Xgrid,			// (INPUT) 1D array of size NG, listing X values in ascending order
									const std::vector<double> 	&Ygrid,			// (INPUT) 1D array of size NG, listing Y values on the X grid
									const long			 		splines_degree,	// (INPUT) integer, either 0,1,2 or 3, specifying the polynomial degree of the function Y=Y(X) at each grid interval
									const std::vector<double> 	&Xtarget,		// (INPUT) 1D array of size NT, listing target x-values on which to evaluate the function Y
									const std::string			&extrapolate,	// (INPUT) string, specifying how to extrapolate Y beyond Xgrid if needed. Available options are "const" or "splines" (i.e. use the polynomial coefficients from the nearest grid point)
									const long					derivative){	// (INPUT) which derivative to evaluate. Options are 0, 1 and 2. Set to 0 to get the value.
	const long NG = Xgrid.size();
	if(NG==0) return(Rcpp::wrap(std::vector<double>(NAN_D,Xtarget.size())));
	if(derivative>2) return(Rcpp::wrap(std::vector<double>(NAN_D,Xtarget.size()))); // only derivatives 0,1,2 are supported right now
	const bool extrapolate_const = (extrapolate=="const");
	
	// get splines representation Y
	dvector Ycoeff;
	get_spline(Xgrid, Ygrid, splines_degree, false, Ycoeff);
	
	dvector Ytarget(Xtarget.size());
	for(long t=0, g=-1; t<Xtarget.size(); ++t){
		if((Xtarget[t]<Xgrid[0]) && extrapolate_const){
			Ytarget[t] = (derivative==0 ? Ygrid[0] : 0.0);
		}else if((Xtarget[t]>Xgrid[NG-1]) && extrapolate_const){
			Ytarget[t] = (derivative==0 ? Ygrid[NG-1] : 0.0);
		}else{
			g = (Xtarget[t]<=Xgrid[0] ? 0 : find_next_left_grid_point(Xgrid, Xtarget[t], g)); // determine grid point to the immediate left of target x
			if(derivative==0){
				Ytarget[t] = polynomial_value(splines_degree,&Ycoeff[g*(splines_degree+1)],Xtarget[t]);
			}else if(derivative==1){
				Ytarget[t] = polynomial_derivative(splines_degree,&Ycoeff[g*(splines_degree+1)],Xtarget[t]);
			}else if(derivative==2){
				Ytarget[t] = polynomial_second_derivative(splines_degree,&Ycoeff[g*(splines_degree+1)],Xtarget[t]);
			}	
		}
	}
	return Rcpp::wrap(Ytarget);
}


// given a scalar function defined as a splines on a 1D x-grid, determine its piecewise polynomial coefficients
// [[Rcpp::export]]
NumericVector get_spline_CPP(	const std::vector<double> 	&Xgrid,				// (INPUT) 1D array of size NG, listing X values in ascending order
								const std::vector<double> 	&Ygrid,				// (INPUT) 1D array of size NG, listing Y values on the X grid
								const long			 		splines_degree){	// (INPUT) integer, either 0,1,2 or 3, specifying the polynomial degree of the function Y=Y(X) at each grid interval
	const long NG = Xgrid.size();
	if(NG==0) return(Rcpp::wrap(std::vector<double>()));
	dvector Ycoeff;
	get_spline(Xgrid, Ygrid, splines_degree, false, Ycoeff);	
	return Rcpp::wrap(Ycoeff);
}


// given a scalar piecewise linear function defined on a 1D x-grid, calculate its derivatives with respect to Xgrid[] and Ygrid[], evaluated (almost) at the same grid points (slightly to the right of the grid points)
// this function returns a 3D array D of size 2*NG x NG x 2, in layer-row-major format
//   with D[a,g,p] corresponding to the p-th polynomial coefficient of dY/dXgrid[a] at Xgrid[g]+0
//   and D[NG+a,g,p] corresponding to the p-th polynomial coefficient of dY/dYgrid[a] at Xgrid[g]+0
//   (for a=0,..,NG-1, and p=0,1)
// note that D[a,:,:] should be interpreted as a piecewise polynomial (and discontinuous) function on the grid Xgrid[]
// In the special case where NG=1, the derivative dY/dXgrid[0] will be zero and the derivative dY/dYgrid[0] will be 1
// [[Rcpp::export]]
NumericVector derivatives_of_grid_curve_CPP(const std::vector<double> 	&Xgrid,		// (INPUT) 1D array of size NG, listing X values in ascending order
											const std::vector<double> 	&Ygrid){	// (INPUT) 1D array of size NG, listing Y values on the X grid
	const long NG = Xgrid.size();
	const long NA = 2*NG;
	const long Ddegree = 1;
	if(NG==0) return(Rcpp::wrap(std::vector<double>(NAN_D,NA*NG*(Ddegree+1))));
	const long NperA = NG*(Ddegree+1); // number of values per derivative
	dvector D(NA*NG*(Ddegree+1),0);

	if(NG==1){
		const long g=0;
		D[(NG+g)*NperA + g*(Ddegree+1) + 0] = 1;
		return Rcpp::wrap(D);
	}
	
	for(long g=0; g<NG-1; ++g){
		// calculate derivative dY(x)/dXgrid[g] for x in (Xgrid[g],Xgrid[g+1])
		D[g*NperA + g*(Ddegree+1) + 0] = -Xgrid[g+1]*(Ygrid[g+1]-Ygrid[g])/SQ(Xgrid[g+1]-Xgrid[g]);
		D[g*NperA + g*(Ddegree+1) + 1] = (Ygrid[g+1]-Ygrid[g])/SQ(Xgrid[g+1]-Xgrid[g]);
		// calculate derivative dY(x)/dXgrid[g+1] for x in (Xgrid[g],Xgrid[g+1])
		D[(g+1)*NperA + g*(Ddegree+1) + 0] = Xgrid[g]*(Ygrid[g+1]-Ygrid[g])/SQ(Xgrid[g+1]-Xgrid[g]);
		D[(g+1)*NperA + g*(Ddegree+1) + 1] = -(Ygrid[g+1]-Ygrid[g])/SQ(Xgrid[g+1]-Xgrid[g]);
		// calculate derivative dY(x)/dYgrid[g] for x in (Xgrid[g],Xgrid[g+1])
		D[(NG+g)*NperA + g*(Ddegree+1) + 0] = 1+Xgrid[g]/(Xgrid[g+1]-Xgrid[g]);
		D[(NG+g)*NperA + g*(Ddegree+1) + 1] = -1/(Xgrid[g+1]-Xgrid[g]);
		// calculate derivative dY(x)/dYgrid[g+1] for x in (Xgrid[g],Xgrid[g+1])
		D[(NG+g+1)*NperA + g*(Ddegree+1) + 0] = -Xgrid[g]/(Xgrid[g+1]-Xgrid[g]);
		D[(NG+g+1)*NperA + g*(Ddegree+1) + 1] = 1/(Xgrid[g+1]-Xgrid[g]);
	}
	
	// set D[:,NG-1,:] equal to D[:,NG-2,:]
	for(long k=0; k<NA; ++k){
		D[k*NperA + (NG-1)*(Ddegree+1) + 0] = D[k*NperA + (NG-2)*(Ddegree+1) + 0];
		D[k*NperA + (NG-1)*(Ddegree+1) + 1] = D[k*NperA + (NG-2)*(Ddegree+1) + 1];
	}
	
	return Rcpp::wrap(D);
}



// given a scalar function defined as a splines on a 1D x-grid, calculate its antiderivative at some target x-values
// [[Rcpp::export]]
NumericVector get_antiderivative_CPP(	const std::vector<double> 	&Xgrid,			// (INPUT) 1D array of size NG, listing X values in ascending order
										const double  		 		Xstart,			// (INPUT) X value at which the antiderivative should be zero
										const std::vector<double> 	&Ygrid,			// (INPUT) 1D array of size NG, listing Y values on the X grid
										const long			 		splines_degree,	// (INPUT) integer, either 0,1,2 or 3, specifying the polynomial degree of the function Y=Y(X) at each grid interval
										const std::vector<double> 	&Xtarget){		// (INPUT) 1D array of size NT, listing target x-values on which to evaluate the antiderivativative
	// get splines representation Y
	dvector Ycoeff;
	const bool splines_slideX = true;
	get_spline(Xgrid, Ygrid, splines_degree, splines_slideX, Ycoeff);

	// calculate antiderivative as a piecewise polynomial function
	dvector Agrid, Acoeff;
	const long Adegree = splines_degree+1;
	get_antiderivative(Xgrid, Xstart, splines_degree, Ycoeff, splines_slideX, Agrid, Acoeff);
	
	dvector Atarget(Xtarget.size());
	for(long t=0, g=0; t<Xtarget.size(); ++t){
		g = (Xtarget[t]<=Xgrid[0] ? 0 : find_next_left_grid_point(Xgrid, Xtarget[t], g)); // determine grid point to the immediate left of target x
		Atarget[t] = polynomial_value(Adegree,&Acoeff[g*(Adegree+1)],Xtarget[t]-(splines_slideX ? Xgrid[g] : 0.0));
	}
	
	return Rcpp::wrap(Atarget);
}



// given a scalar function defined as a splines on a 1D x-grid, calculate its first derivative at some target x-values
// [[Rcpp::export]]
NumericVector get_derivative_CPP(	const std::vector<double> 	&Xgrid,			// (INPUT) 1D array of size NG, listing X values in ascending order
									const std::vector<double> 	&Ygrid,			// (INPUT) 1D array of size NG, listing Y values on the X grid
									const long			 		splines_degree,	// (INPUT) integer, either 0,1,2 or 3, specifying the polynomial degree of the function Y=Y(X) at each grid interval
									const std::vector<double> 	&Xtarget){		// (INPUT) 1D array of size NT, listing target x-values on which to evaluate the derivativative
	dvector Dtarget(Xtarget.size(),0);
	if(splines_degree==0){
		// Y is piecewise constant, so its derivative is zero almost everywhere
		return Rcpp::wrap(Dtarget);
	}
	
	// get splines representation Y
	dvector Ycoeff;
	get_spline(Xgrid, Ygrid, splines_degree, false, Ycoeff);

	// calculate derivative based on splines (polynomial) coefficients
	for(long t=0, g=0, p; t<Xtarget.size(); ++t){
		g = (Xtarget[t]<=Xgrid[0] ? 0 : find_next_left_grid_point(Xgrid, Xtarget[t], g)); // determine grid point to the immediate left of target x
		for(p=1; p<=splines_degree; ++p){
			Dtarget[t] += Ycoeff[g*(splines_degree+1) + p] * p * pow(Xtarget[t],p-1.0);
		}
	}
	
	return Rcpp::wrap(Dtarget);
}



// Given some polynomial function Y, calculate the integral \int_Xstart^Xend 1/Y(x) dx
template<class VALUE_TYPE>
VALUE_TYPE integrate_inverse_polynomial(const double		Xstart,			// (INPUT) lower end of the integration
										const double		Xend,			// (INPUT) upper end of the integration
										const long			P,				// (INPUT) polynomial order of the input function Y=Y(x). For example, if Y is linear then P=1.
										const VALUE_TYPE 	Ycoeff[],		// (INPUT) 1D array of size (P+1), listing the polynomial coefficients (Y0,Y1,..,YP) of the function Y. Hence one has Y(x) = Ycoeff[0] + Ycoeff[1]*x + .. + Ycoeff[P]*x^P.
										dvector				&scratch){		// (SCRATCH) scratch vector, will be allocated as needed

	VALUE_TYPE A = 0;
	double a,b,c;
	if(P==0){
		// Y is piecewise constant: Y=a
		a = Ycoeff[0];
		A = (1/a) * (Xend-Xstart);
	}else if(P==1){
		// Y is piecewise linear: Y=a*x + b
		a = Ycoeff[1];
		b = Ycoeff[0];
		if(a==0){
			// actually constant
			A = (1/b) * (Xend-Xstart);
		}else{
			A = (1/a) * (log(a*Xend+b) - log(a*Xstart+b));
		}
	}else if(P==2){
		// Y is piecewise quadratic: Y=a*x^2 + b*x + c
		a = Ycoeff[2];
		b = Ycoeff[1];
		c = Ycoeff[0];
		if(a==0){
			// actually linear
			if(b==0){
				// actually constant
				A = (1/c) * (Xend-Xstart);
			}else{
				A = (1/b) * (log(b*Xend+c) - log(b*Xstart+c));
			}
		}else{
			const double m2 	= (4*a*c-SQ(b))/(4*SQ(a));
			const double tstart = Xstart + b/(2*a);
			const double tend 	= Xend + b/(2*a);
			if(m2==0){
				A = -(1/a) * ((1/tend) - (1/tstart));
			}else if(m2>0){
				A = (1/sqrt(m2)) * (atan(tend/sqrt(m2))-atan(tstart/sqrt(m2)));
			}else{
				A = - (1/sqrt(-m2)) * (atanh(tend/sqrt(-m2))-atanh(tstart/sqrt(-m2)));
			}
		}
	}else{
		// Y is of order 3 or higher
		// closed formula for integral is messy, so approximate 1/Y by a quadratic formula and then calculate the integral
		// Y ~ scratch[0] + scratch[1] * x + scratch[2] * x^2
		scratch.resize(3);
		double alpha;
		quadratic_approximation_of_inv_polynomial(	P,
													Ycoeff,
													Xstart,
													0.5*(Xstart+Xend),
													Xend,
													&scratch[0]);
		for(long p=0; p<=2; ++p){
			alpha	= scratch[p]/(p+1);
			A		+= alpha*(pow(Xend,p+1) - pow(Xstart,p+1));
		}
	}
	return A;
}




// calculate the antiderivative of a 1/Y(x) over a 1D grid, where Y is piecewise polynomial, i.e. calculate:
//   A(x_i):=int_Xstart^{x_i} (1/Y(x)) dx
// for all grid points x_1,x_2,.., 
// Assuming: Y is piecewise polynomial of order P between grid points, i.e.:
//   Y(x) = f0 + f1*x + .. + f_P*x^P
// This function is only exact if Y is piecewise constant, piecewise linear or piecewise quadratic; for higher orders an approximation is returned
template<class VALUE_TYPE>
void get_antiderivative_of_inverse_polynomial(	const std::vector<double> 		&X,				// (INPUT) 1D array of size N, listing x-values in ascending order
												const double					&Xstart,		// (INPUT) lower end of the integration, i.e. x-value where antiderivative is set to zero
												const long						P,				// (INPUT) polynomial order of the input function Y=Y(x) in each interval. For example, if Y is piecewise linear then P=1.
												const std::vector<VALUE_TYPE> 	&Ycoeff,		// (INPUT) 2D array of size N x (P+1) in row-major format, listing the polynomial coefficients (f0,f1,..,fP) of the function Y in each grid-interval. Hence, for X[i]<=x<=X[i+1] one has Y(x) = Ycoeff[i*(P+1)+0] + Ycoeff[i*(P+1)+1]*x + .. + Ycoeff[i*(P+1)+P]*x^P.
												std::vector<VALUE_TYPE>			&A){			// (OUTPUT) 1D array of size N, listing the computed antiderivative on the same x-grid
	const long N = X.size();
	A.resize(N);
	if(N==0) return;

	// first calculate the antiderivative assuming Xstart=X[0], correct afterwards if needed
	A[0] = 0;
	dvector scratch;
	for(long i=1; i<N; ++i){
		A[i] = A[i-1] + integrate_inverse_polynomial(X[i-1], X[i], P, &Ycoeff[(i-1)*(P+1) + 0], scratch);
	}
	
	// shift antiderivative if needed, i.e. if Xstart is different from the convention used above (where A=0 at Xgrid[0]).
	if(Xstart!=X[0]){
		const long g_start = max(0l, find_next_left_grid_point(X, Xstart, -1));
		const VALUE_TYPE Astart = A[g_start] + integrate_inverse_polynomial(X[g_start], Xstart, P, &Ycoeff[g_start*(P+1) + 0], scratch);
		for(long g=0; g<N; ++g){
			A[g] -= Astart;	
		}
	}
}




#pragma mark -
#pragma mark Vectorized basic arithmetics
#pragma mark


template<class TYPE>
inline vector<TYPE> operator*(vector<TYPE> x, const vector<TYPE> &y){
	for(long i=0; i<x.size(); ++i){
		x[i] *= y[i];
	}
	return x;
}


template<class TYPE>
inline vector<TYPE>& operator*=(vector<TYPE> &x, const vector<TYPE> &y){
	for(long i=0; i<x.size(); ++i){
		x[i] *= y[i];
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> operator*(vector<TYPE> x, double scalar){
	for(long i=0; i<x.size(); ++i){
		x[i] *= scalar;
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> operator*(double scalar, vector<TYPE> x){
	for(long i=0; i<x.size(); ++i){
		x[i] *= scalar;
	}
	return x;
}


template<class TYPE>
vector<TYPE>& operator*=(vector<TYPE> &x, double scalar) {
	for(long i=0; i<x.size(); ++i){
		x[i] *= scalar;
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> operator+(vector<TYPE> x, const vector<TYPE> &y){
	for(long i=0; i<x.size(); ++i){
		x[i] += y[i];
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> &operator+=(vector<TYPE> &x, const vector<TYPE> &y){
	for(long i=0; i<x.size(); ++i){
		x[i] += y[i];
	}
	return x;
}


template<class TYPE>
vector<TYPE>& operator+=(vector<TYPE> &x, const TYPE &scalar) {
	for(long i=0; i<x.size(); ++i){
		x[i] += scalar;
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> operator-(vector<TYPE> x, const vector<TYPE> &y){
	for(long i=0; i<x.size(); ++i){
		x[i] -= y[i];
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> operator-(vector<TYPE> x, const TYPE &scalar){
	for(long i=0; i<x.size(); ++i){
		x[i] -= scalar;
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> &operator-=(vector<TYPE> &x, const vector<TYPE> &y){
	for(long i=0; i<x.size(); ++i){
		x[i] -= y[i];
	}
	return x;
}


template<class TYPE>
vector<TYPE>& operator-=(vector<TYPE> &x, const TYPE &scalar) {
	for(long i=0; i<x.size(); ++i){
		x[i] -= scalar;
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> operator/(vector<TYPE> x, const vector<TYPE> &y){
	for(long i=0; i<x.size(); ++i){
		x[i] /= y[i];
	}
	return x;
}



template<class TYPE>
inline vector<TYPE> operator/(vector<TYPE> x, double scalar){
	for(long i=0; i<x.size(); ++i){
		x[i] /= scalar;
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> &operator/=(vector<TYPE> &x, double scalar){
	for(long i=0; i<x.size(); ++i){
		x[i] /= scalar;
	}
	return x;
}


template<class TYPE>
inline vector<TYPE> abs(vector<TYPE> x){
	for(long i=0; i<x.size(); ++i){
		x[i] = abs(x[i]);
	}
	return x;
}



#pragma mark -
#pragma mark Random numbers
#pragma mark 


inline double uniformWithinInclusiveRight(double minimum, double maximum){
	return minimum + STRANDOM_EPSILON + (maximum - minimum - STRANDOM_EPSILON)*R::runif(0.0,1.0);
}

inline double uniformWithinInclusiveLeft(double minimum, double maximum){
	return minimum + (maximum - minimum - STRANDOM_EPSILON)*R::runif(0.0,1.0);
}


inline long uniformIntWithin(long minimum, long maximum){
	//return min(maximum, (long) floor(minimum + (maximum-minimum+1)*(double(rand())/RAND_MAX))); // rand() is discouraged by R package builders
	return min(maximum, (long) floor(minimum + (maximum-minimum+1) * R::runif(0.0,1.0)));
}


// draw a standard-normal random variable
inline double random_standard_normal(){
	return sqrt(-2.0*log(uniformWithinInclusiveRight(0, 1)))*cos(2.0*M_PI*uniformWithinInclusiveRight(0,1));
}


// generate the order statistic of N uniformly distributed random variables
// That is, return N uniform and independent random variables after sorting in ascending order
// Based on the pseucode 3.2 described here: http://www.math.kent.edu/~kazim/StatSeminar/paper-review.pdf
void uniform_order_statistic(	double 		minimum, 
								double 		maximum,
								const long 	N, 					// (INPUT) number of random variables to generate
								dvector 	&random_values){	// (OUTPUT) D array of size N, listing uniforly distributed random numbers in [min:max] and in ascending order
	// first generate order statistic in [0,1]
	random_values.resize(N);
	double temp=0, U, V;
	for(long i=0; i<N; ++i){
		V = R::runif(0.0,1.0);
		U = 1 - (1-temp)*pow(V,1.0/(N-i));
		temp = U;
		random_values[i] = U;
	}
	
	// shift & scale random variables into interval [min:max]
	if((minimum!=0.0) || (maximum!=1.0)){
		const long L = abs(maximum-minimum);
		if(minimum>maximum) std::swap(minimum,maximum);
		for(long i=0; i<N; ++i){
			random_values[i] = minimum + L*random_values[i];
		}
	}
}



// draw a standard-normal random variable
inline double random_normal(double mean, double std){
	return mean + std * sqrt(-2.0*log(uniformWithinInclusiveRight(0, 1)))*cos(2.0*M_PI*uniformWithinInclusiveRight(0,1));
}

template<class TYPE>
long random_int_from_distribution(const TYPE probabilities[], const long N){
	double p = R::runif(0.0,1.0);
	for(long i=0; i<N; ++i){
		if(p<=probabilities[i]) return i;
		p -= probabilities[i];
	}
	return N-1;
}


// pick an index within 0:(N-1) at probability proportional to weights[i]
// for efficiency, the caller guarantees that total_weight = sum_i weights[i]
long random_int_from_distribution(const std::vector<double> &weights, const double total_weight){
	const long N = weights.size();
	double p = R::runif(0.0,1.0);
	for(long i=0; i<N; ++i){
		if(p<=weights[i]/total_weight) return i;
		p -= weights[i]/total_weight;
	}
	return N-1;
}




// pick an index within 0:(N-1) at probability proportional to weights[index_pool[i]]
// for efficiency, the caller guarantees that total_weight = sum_i weights[index_pool[i]]
long random_int_from_distribution(const std::vector<long> &index_pool, const std::vector<double> &weights, const double total_weight){
	const long N = index_pool.size();
	double p = R::runif(0.0,1.0);
	for(long i=0; i<N; ++i){
		if(p<=weights[index_pool[i]]/total_weight) return i;
		p -= weights[index_pool[i]]/total_weight;
	}
	return N-1;
}


// pick an index within 0:(N-1) at probability proportional to weights[weight_indices[index_pool[i]]]
// for efficiency, the caller guarantees that total_weight = sum_i weights[weight_indices[index_pool[i]]]
long random_int_from_distribution(const std::vector<long> &index_pool, const std::vector<double> &weights, const std::vector<long> &weight_indices, const double total_weight){
	const long N = index_pool.size();
	double probability = R::runif(0.0,1.0);
	double dp;
	for(long i=0; i<N; ++i){
		dp = weights[weight_indices[index_pool[i]]]/total_weight;
		if(probability<=dp) return i;
		probability -= dp;
	}
	return N-1;
}



// given a list of index pools pool_1, pool_2, ..., pool_NP, and a probability weight associated with each pool (e.g. each item in pool_i has weight weight_i), pick a random item among all pools
// for efficiency, the caller provides total_weight := sum_p weights[p] * pools[p].size()
// The time complexity of this function is O(NP)
// if the number of items in each pool is much larger than the number of distinct pools, this function is much more efficient than considering all items within a single pool
// this function guarantees that only non-empty pools are picked, or returns false if all pools are actually empty
// returns true upon success (guaranteed, provided that at least one pooll is non-empty)
bool random_int_from_pools(	const std::vector<lvector> 	&pools, 		// (INPUT) array of size NP, each element of which is a pool of integers (indices)
							const std::vector<double> 	&weights, 		// (INPUT) array of size NP, listing probability weights associated with each pool. Hence, weights[p] is the weight assigned to each element in index_pools[p], and hence weights[p]*pools[p].size() is the probability weight of landing in pool p.
							const double 				total_weight,	// (INPUT) the total weight across all items in all pools. Provided by the caller, for efficiency
							long						&p,				// (OUTPUT) the random pool chosen, i.e. a random integer between 0,..,NP-1
							long						&i){			// (OUTPUT) the random item in the chosen pool, i.e. a random integer between 0,...,pools[p].size()-1
	const long NP = pools.size();
	// step 1: choose a random pool based on their total probabilities
	double probability = R::runif(0.0,1.0);
	double dp;
	p = 0;
	long last_valid_p = -1; // keep track of the last non-empty pool
	while(p<NP){
		if(!pools[p].empty()) last_valid_p = p;
		dp = weights[p]*pools[p].size()/total_weight;
		if((probability<=dp) && (!pools[p].empty())) break; // pick this pool
		probability -= dp;
		++p; // try the next pool
	}
	if(last_valid_p<0){
		// all pools were empty, so return failure code
		p = -1; i = -1;
		return false;
	}
	if(p>=NP) p = last_valid_p; // probability overflow, likely due to numerical rounding errors. So pick the last non-empty pool

	// step 2: choose a random item in the chosen pool
	// this step is efficient, because all items in this pool have the same probability weight
	i = uniformIntWithin(0,pools[p].size()-1);
	return true;
}




// given a list of index pools pool_1, pool_2, ..., pool_NP, and a probability weight associated with each pool (e.g. each item in pool_i has weight weight_i), and a subset of considered pools, pick a random item among the items in all considered pools
// for efficiency, the caller provides total_weight := sum_i weights[only_pools[i]] * pools[only_pools[i]].size()
// The time complexity of this function is O(|only_pools|)
// if the number of items in each pool is much larger than the number of distinct pools, this function is much more efficient than considering all items within a single pool
// this function guarantees that only non-empty pools are picked, or returns false if all pools are actually empty
// returns true upon success (guaranteed, provided that at least one pooll is non-empty)
bool random_int_from_pools(	const std::vector<lvector> 	&pools, 		// (INPUT) array of size NP, each element of which is a pool of integers (indices)
							const dvector 				&weights, 		// (INPUT) array of size NP, listing probability weights associated with each pool. Hence, weights[p] is the weight assigned to each element in index_pools[p], and hence weights[p]*pools[p].size() is the probability weight of landing in pool p.
							const lvector				&only_pools,	// (INPUT) array listing pool indices to consider (ignoring all other pools)
							const double 				total_weight,	// (INPUT) the total weight across all items in all pools. Provided by the caller, for efficiency
							long						&c,				// (OUTPUT) the considered pool chosen, i.e. random integer between 0,..,only_pools.size()-1
							long						&i){			// (OUTPUT) the random item in the chosen pool, i.e. a random integer between 0,...,pools[only_pools[c]].size()-1
	const long NC = only_pools.size();
	// step 1: choose a random pool based on their total probabilities
	double probability = R::runif(0.0,1.0);
	double dp;
	long p;
	c = 0;
	long last_valid_c = -1; // keep track of the last non-empty pool
	while(c<NC){
		p = only_pools[c];
		if(!pools[p].empty()) last_valid_c = c;
		dp = weights[p]*pools[p].size()/total_weight;
		if((probability<=dp) && (!pools[p].empty())) break; // pick this pool
		probability -= dp;
		++c; // try the next allowed pool
	}
	if(last_valid_c<0){
		// all allowed pools were empty, so return failure code
		c = -1; i = -1;
		return false;
	}
	if(c>=NC) c = last_valid_c; // probability overflow, likely due to numerical rounding errors. So pick the last non-empty allowed pool
	p = only_pools[c];

	// step 2: choose a random item in the chosen pool
	// this step is efficient, because all items in this pool have the same probability weight
	i = uniformIntWithin(0,pools[p].size()-1);
	return true;
}





// generate exponentially distributed random variable, with PDF f(x) = lambda*exp(-lambda*x)
double random_exponential_distribution(double lambda){
	// alternative: return R::rexp(1/lambda); // note that the argument to R::rexp is the scale, not the rate
	return -log(R::runif(0.0,1.0))/lambda;
}


inline bool random_bernoulli(double p){
	//return ((double(rand())/RAND_MAX) <= p); // rand() is discouraged by R package builders
	return (R::runif(0.0,1.0)<=p);
}


// generate a Binomially distributed random variable
//   n: number of trials
//   prob: probability of success per trial
// modified from ranlib: https://people.sc.fsu.edu/~jburkardt/cpp_src/ranlib/ranlib.html
long random_binomial(long n, double prob){
	if(prob<=0) return 0;
	if(prob>=1) return n;
	double al, alv, amaxp, c, f, f1, f2, ffm, fm, g;
	double minp, p1, p2, p3, p4, q, qn, r, t, u, v;
	double w, w2, x, x1, x2, xl, xll, xlr, xm, xnp, xnpq, xr, ynorm, z, z2;
	long i, ix, ix1, k, m, mp, value;
	
	if((n*prob>10) && (n*(1-prob)>10)){
		// normal aprobroximation
		return max(0l,min(n,long(round(random_normal(n*prob, sqrt(n*prob*(1-prob)))))));
	}

	minp = min(prob, 1.0-prob);
	q = 1.0 - minp;
	xnp =(double)(n) * minp;

	if(xnp < 30.0){
		qn = pow(q, n);
		r = minp / q;
		g = r *(double)(n + 1);
		while(true){
			ix = 0;
			f = qn;
			u = R::runif(0.0,1.0);
			while(true){
				if(u < f){
					if(0.5 < prob) ix = n - ix;
					value = ix;
					return value;
				}
				if(110 < ix) break;
				u 	= u - f;
				ix 	= ix + 1;
				f 	= f *(g /(double)(ix) - r);
			}
		}
	}
	ffm 	= xnp + minp;
	m 		= ffm;
	fm 		= m;
	xnpq 	= xnp * q;
	p1 		= (long)(2.195 * sqrt(xnpq) - 4.6 * q) + 0.5;
	xm 		= fm + 0.5;
	xl 		= xm - p1;
	xr 		= xm + p1;
	c 		= 0.134 + 20.5 /(15.3 + fm);
	al 		= (ffm - xl) /(ffm - xl * minp);
	xll 	= al *(1.0 + 0.5 * al);
	al 		= (xr - ffm) /(xr * q);
	xlr 	= al *(1.0 + 0.5 * al);
	p2 		= p1 *(1.0 + c + c);
	p3 		= p2 + c / xll;
	p4 		= p3 + c / xlr;

	//  Generate a variate.
	while(true){
		u = R::runif(0.0,1.0) * p4;
		v = R::runif(0.0,1.0);
		if(u < p1){
			ix = xm - p1 * v + u;
			if(0.5 < prob) ix = n - ix;
			value = ix;
			return value;
		}
		if(u <= p2){
			x = xl +(u - p1) / c;
			v = v * c + 1.0 - fabs(xm - x) / p1;
			if(v <= 0.0 || 1.0 < v) continue;
			ix = x;
		}else if(u <= p3){
			ix = xl + log(v) / xll;
			if(ix < 0) continue;
			v = v *(u - p2) * xll;
		}else{
			ix = xr - log(v) / xlr;
			if(n < ix) continue;
			v = v *(u - p3) * xlr;
		}
		k = abs(ix - m);

		if(k <= 20 || xnpq / 2.0 - 1.0 <= k){
			f = 1.0;
			r = minp / q;
			g =(n + 1) * r;
			if(m < ix){
				mp = m + 1;
				for(i = mp; i <= ix; i++){
					f = f *(g / i - r);
				}
			}else if(ix < m){
				ix1 = ix + 1;
				for(i = ix1; i <= m; i++){
					f = f /(g / i - r);
				}
			}
			if(v <= f){
				if(0.5 < prob) ix = n - ix;
				value = ix;
				return value;
			}
		}else{
			amaxp 	= (k / xnpq) *(( k *(k / 3.0 + 0.625) + 0.1666666666666) / xnpq + 0.5);
			ynorm 	= -(double)(k * k) /(2.0 * xnpq);
			alv 	= log(v);
			if(alv < ynorm - amaxp){
				if(0.5 < prob) ix = n - ix;
				value = ix;
				return value;
			}
			if(ynorm + amaxp < alv) continue;
			x1 	= (double)(ix + 1);
			f1 	= fm + 1.0;
			z 	= (double)(n + 1) - fm;
			w 	= (double)(n - ix + 1);
			z2 	= z * z;
			x2 	= x1 * x1;
			f2 	= f1 * f1;
			w2 	= w * w;

			t = xm * log(f1 / x1) +(n - m + 0.5) * log(z / w) + (double)(ix - m) * log(w * minp /(x1 * q)) + (13860.0 -(462.0 -(132.0 -(99.0 - 140.0 / f2) / f2) / f2) / f2) / f1 / 166320.0 + (13860.0 -(462.0 -(132.0 -(99.0 - 140.0 / z2) / z2) / z2) / z2) / z / 166320.0 + (13860.0 -(462.0 -(132.0 -(99.0 - 140.0 / x2) / x2) / x2) / x2) / x1 / 166320.0 + (13860.0 -(462.0 -(132.0 -(99.0 - 140.0 / w2) / w2) / w2) / w2) / w / 166320.0;
			if(alv <= t){
				if(0.5 < prob) ix = n - ix;
				value = ix;
				return value;
			}
		}
	}
	return value;
}



// randomly shuffle the elements in a vector, between indices start & end
// Based on the Fisher–Yates shuffling algorithm
template<class TYPE>
void random_shuffle_insitu(	std::vector<TYPE> 	&values, 	// (INPUT/OUTPUT) the array whose entries are to be shuffled in situ
							const long 			start, 		// (INPUT) the lowest index of shuffled elements
							const long 			end){		// (INPUT) the highest index of shuffled elements
	TYPE temp;
	for(long i=end, j; i>start; --i){
		j = uniformIntWithin(start, i);
		// swap values i & j
		temp 		= values[i];
		values[i] 	= values[j];
		values[j] 	= temp;
	}
}



// ##########################################################
// MATRIX ALGEBRA

#pragma mark -
#pragma mark Matrix algebra
#pragma mark 


void get_identity_matrix(	const long 			NR,
							std::vector<double>	&A){	// (INPUT/OUTPUT) will be the identity matrix of size NR x NR, in row-major format
	A.assign(NR*NR,0);
	for(long r=0; r<NR; ++r){
		A[r*NR + r] = 1;
	}
}


// calculate the dot-product between two vectors, as needed for QR decomposition
double QR_dot_product(	const long 		N, 			 // (INPUT) the number of entries in the vectors
						const double	X[], 		 // (INPUT) vector of size N
						const long 		xincrement,  // (INPUT) the increment between successive entries in X
						const double 	Y[], 		 // (INPUT) vector of size N
						const long 		yincrement){ // (INPUT) the increment between successive entries in Y
	long i, ix, iy, m;
	if (N<=0) return 0;

	double S = 0.0;
	if(xincrement != 1 || yincrement != 1){
		if(0 <= xincrement){
			ix = 0;
		}else{
			ix = (-N + 1) * xincrement;
		}

		if(0 <= yincrement){
			iy = 0;
		}else{
			iy = (-N + 1) * yincrement;
		}

		for(i = 0; i<N; i++){
			S = S + X[ix] * Y[iy];
			ix = ix + xincrement;
			iy = iy + yincrement;
		}
	}else{
		m = N % 5;
		for (i = 0; i<m; i++){
			S = S + X[i] * Y[i];
		}
		for(i = m; i<N; i = i + 5){
			S = S + X[i] * Y[i] + X[i+1] * Y[i+1] + X[i+2] * Y[i+2] + X[i+3] * Y[i+3] + X[i+4] * Y[i+4];
		}
	}
	return S;
}


// swap two vectors in-situ, as needed for QR decomposition
void QR_swap_vectors(	const long 	N, 				// (INPUT) the length of the vectors
						double 		x[],			// (INPUT/OUTPUT) one of the vectors to swap
						const long 	xincrement, 	// (INPUT) increment between successive elements in Y
						double 		y[], 			// (INPUT/OUTPUT) one of the vectors to swap
						const long 	yincrement){	// (INPUT) increment between successive elements in Y
	long i, ix, iy, m;
	double temp;
	if(N<=0) return;
	
	if((xincrement == 1) && (yincrement == 1)){
		m = N % 3;
		for(i = 0; i<m; i++){
			temp = x[i];
			x[i] = y[i];
			y[i] = temp;
		}

		for(i = m; i<N; i = i + 3){
			temp 	= x[i];
			x[i] 	= y[i];
			y[i] 	= temp;

			temp 	= x[i+1];
			x[i+1] 	= y[i+1];
			y[i+1] 	= temp;

			temp 	= x[i+2];
			x[i+2] 	= y[i+2];
			y[i+2] 	= temp;
		}
	}else{
		if(0 <= xincrement){
			ix = 0;
		}else{
			ix = (-N + 1) * xincrement;
		}
		if(0 <= yincrement){
			iy = 0;
		}else{
			iy = (-N + 1) * yincrement;
		}
		for(i = 0; i<N; i++){
			temp 	= x[ix];
			x[ix] 	= y[iy];
			y[iy] 	= temp;
			ix 		= ix + xincrement;
			iy 		= iy + yincrement;
		}
	}
}


// calculate euclidean norm of vector X
double euclidean_norm(	const long 		N,				// (INPUT) the number of elements in the vector
						const double	X[], 			// (INPUT) 1D vector of size N
						const long 		xincrement){	// (INPUT) the increment between successive entries of X
	long i, ix;
	double absxi, norm, scale, sum_squares;

	if((N<1) || (xincrement<1)){
		norm = 0;
	}else if(N== 1){
		norm = abs(X[0]);
	}else{
		scale 	= 0.0;
		sum_squares 	= 1.0;
		ix 		= 0;
		for(i = 0; i < N; i++){
			if(X[ix] != 0.0){
				absxi = abs(X[ix]);
				if(scale < absxi){
					sum_squares = 1.0 + sum_squares * (scale/absxi) * (scale/absxi);
					scale = absxi;
				}else{
					sum_squares = sum_squares + (absxi/scale) * (absxi/scale);
				}
			}
			ix = ix + xincrement;
		}
		norm  = scale * sqrt(sum_squares);
	}
	return norm;
}



void QR_scale_vector(	const long 	N, 				// (INPUT) the number of elements in the vector
						double 		scaling_factor, // (INPUT) scaling factor
						double 		X[], 			// (INPUT/OUTPUT) 1D vector of size N
						const long	xincrement){			// (INPUT) the increment between successive entries of X.
	long i, ix, m;
	if(N<=0) return;

	if(xincrement == 1){
		m = N % 5;
		for(i = 0; i < m; i++){
			X[i] = scaling_factor * X[i];
		}
		for(i = m; i<N; i = i + 5){
			X[i]   = scaling_factor * X[i];
			X[i+1] = scaling_factor * X[i+1];
			X[i+2] = scaling_factor * X[i+2];
			X[i+3] = scaling_factor * X[i+3];
			X[i+4] = scaling_factor * X[i+4];
		}
	}else{
		if(0 <= xincrement){
			ix = 0;
		}else{
			ix = (-N + 1) * xincrement;
		}
		for(i = 0; i < N; i++){
			X[ix] = scaling_factor * X[ix];
			ix = ix + xincrement;
		}
	}
	return;
}



// add two vectors, as used for QR decomposition
void QR_add_vectors(const long 		N, 				// (INPUT) the number of elements in the vectors
					const double 	alpha, 			// (INPUT) an optional multiplicative factor for X
					const double	X[], 			// (INPUT) 1D vector of size N
					const long 		xincrement, 	// (INPUT) the increment between successive entries of X
					double 			Y[], 			// (INPUT/OUTPUT) 1D vector of size N. Upon return, this will store the result alpha*X+Y
					const long		yincrement){	// (INPUT) the increment between successive entries of Y
  long i, ix, iy, m;
  if(N<=0) return;
  if(alpha==0) return;

	if((xincrement != 1) || (yincrement != 1)){
		if(0 <= xincrement){
			ix = 0;
		}else{
			ix = (-N + 1) * xincrement;
		}

		if(0 <= yincrement){
			iy = 0;
		}else{
			iy = (-N + 1) * yincrement;
		}

		for(i = 0; i<N; i++){
			Y[iy] = Y[iy] + alpha * X[ix];
			ix = ix + xincrement;
			iy = iy + yincrement;
		}
	}else{
		m = N % 4;

		for(i = 0; i<m; i++){
			Y[i] = Y[i] + alpha * X[i];
		}

		for(i = m; i<N; i = i + 4){
			Y[i]   = Y[i]   + alpha * X[i];
			Y[i+1] = Y[i+1] + alpha * X[i+1];
			Y[i+2] = Y[i+2] + alpha * X[i+2];
			Y[i+3] = Y[i+3] + alpha * X[i+3];
		}
	}
	return;
}


// QR decomposition of a real 2D matrix of size NR x NC
// code adjusted from: https://people.sc.fsu.edu/~jburkardt/c_src/qr_solve/qr_solve.html
void QR_decomposition (	double 			A[], 			// (INPUT/OUTPUT) on input, the matrix to be decomposed, in column-major format. On output, A contains in its upper triangle the upper triangular matrix R of the QR factorization.  Below its diagonal A contains information from which the orthogonal part of the decomposition can be recovered.
						const long 		LDA, 			// (INPUT) the leading dimension of A, i.e. linear index periodicity between subsequent columns. Typically this will be equal to NR, but may also be larger (e.g. when processing a sub-matrix)
						const long 		NR,				// (INPUT) the number of rows of A
						const long 		NC,				// (INPUT) the number of columns of A
						const bool 		pivoting,		// (INPUT) whether to perform pivoting
						double 			scratch[], 		// (SCRATCH) pre-allocated scratch space of size >=NC. Only needed if pivoting=true.
						double 			QRaux[], 		// (OUTPUT) 1D array of size NC, must be pre-allocated. Will contain information required to recover the orthogonal part of the decomposition.
						long 			pivots[],		// (OUTPUT) 1D array of size NC, must be pre-allocated. pivots[k] specifies the index of the column of A that has been interchanged into the K-th column, if pivoting was requested.
						long			&rank){			// (OUTPUT) the estimated rank of the matrix A
	long j, jp, l, lup, maxj;
	double maxnrm, nrmxl;
	long pl, pu, swapj;
	double t, tt;

	// initialize pivots
	if(pivoting){
		for(long j = 0; j<NC; j++) pivots[j] = 0;
	}

	pl = 1;
	pu = 0;
	if(pivoting){	
		for(j = 1; j <= NC; j++){
			swapj = (0 < pivots[j-1]);
			if(pivots[j-1] < 0){
				pivots[j-1] = -j;
			}else{
				pivots[j-1] = j;
			}

			if(swapj){
				if(j != pl) QR_swap_vectors(NR, A+0+(pl-1)*LDA, 1, A+0+(j-1), 1);
				pivots[j-1] = pivots[pl-1];
				pivots[pl-1] = j;
				pl = pl + 1;
			}
		}
		pu = NC;
		
		for(j = NC; 1<=j; j--){
			if(pivots[j-1] < 0){
				pivots[j-1] = -pivots[j-1];
				if(j != pu){
					QR_swap_vectors(NR, A+0+(pu-1)*LDA, 1, A+0+(j-1)*LDA, 1);
					jp = pivots[pu-1];
					pivots[pu-1] = pivots[j-1];
					pivots[j-1] = jp;
				}
				pu = pu - 1;
			}
		}
	}

	// Compute column norms
	for(j = pl; j<=pu; j++){
		QRaux[j-1] = euclidean_norm(NR, A+0+(j-1)*LDA, 1);
	}

	for(j = pl; j<=pu; j++){
		scratch[j-1] = QRaux[j-1];
	}

	// Householder reduction of A
	lup = min(NR, NC);
	for(l = 1; l<=lup; l++){
		if(pl<=l && l<pu){
			maxnrm = 0.0;
			maxj = l;
			for(j = l; j<=pu; j++){
				if(maxnrm < QRaux[j-1]){
					maxnrm 	= QRaux[j-1];
					maxj 	= j;
				}
			}

			if(maxj != l){
				QR_swap_vectors(NR, A+0+(l-1)*LDA, 1, A+0+(maxj-1)*LDA, 1);
				QRaux[maxj-1] 	= QRaux[l-1];
				scratch[maxj-1] = scratch[l-1];
				jp 				= pivots[maxj-1];
				pivots[maxj-1] 	= pivots[l-1];
				pivots[l-1] 	= jp;
			}
		}

		// Householder transformation for column L.
		QRaux[l-1] = 0.0;
		if(l != NR){
			nrmxl = euclidean_norm(NR-l+1, A+l-1+(l-1)*LDA, 1);
			if(nrmxl != 0.0){
				if(A[l-1+(l-1)*LDA] != 0.0){
					nrmxl = nrmxl * (A[l-1+(l-1)*LDA]<0 ? -1 : +1);
				}
				QR_scale_vector(NR-l+1, 1.0 / nrmxl, A+l-1+(l-1)*LDA, 1);
				A[l-1+(l-1)*LDA] = 1.0 + A[l-1+(l-1)*LDA];
				for(j = l + 1; j <= NC; j++){
					t = -QR_dot_product(NR-l+1, A+l-1+(l-1)*LDA, 1, A+l-1+(j-1)*LDA, 1)/A[l-1+(l-1)*LDA];
					QR_add_vectors(NR-l+1, t, A+l-1+(l-1)*LDA, 1, A+l-1+(j-1)*LDA, 1);

					if(pl <= j && j <= pu){
						if(QRaux[j-1] != 0.0){
							tt = 1.0 - pow(abs(A[l-1+(j-1)*LDA])/QRaux[j-1], 2);
							tt = max(tt, 0.0);
							t = tt;
							tt = 1.0 + 0.05 * tt * pow(QRaux[j-1]/scratch[j-1], 2);

							if(tt != 1.0){
								QRaux[j-1] = QRaux[j-1] * sqrt(t);
							}else{
								QRaux[j-1] = euclidean_norm(NR-l, A+l+(j-1)*LDA, 1);
								scratch[j-1] = QRaux[j-1];
							}
						}
					}
				}
				// save transformation
				QRaux[l-1] = A[l-1+(l-1)*LDA];
				A[l-1+(l-1)*LDA] = -nrmxl;
			}
		}
	}
	
	// compute the rank
	const double tol = 1e-6;
	rank = 0;
  	long k = min(NR,NC);
	for(j = 0; j < k; j++){
		if(abs(A[j+j*LDA]) <= tol * abs(A[0+0*LDA])){
			break;
		}
		rank = j + 1;
	}	
}



// perform various operations using the QR-decomposition of some real 2D matrix A
// this function does not actually use the matrix A, but instead its pre-computed QR-decomposition
// code adjusted from: https://people.sc.fsu.edu/~jburkardt/c_src/qr_solve/qr_solve.html
long QR_operation(	double			QRA[], 		// (INPUT/OUTPUT) matrix of size LDA*NC, storing the QR-decomposition of A. Not actually modified permanently by this function, but used temporarily internally. So its output is the same as input.
					const long		LDA, 		// (INPUT) the leading dimension of the matrix A
					const long		NR, 		// (INPUT) number of rows in A
					const long 		NK, 		// (INPUT) number of columns in the AK matrix, formed during QR decomposition. NK will always be <=min(NR,NC)
					const double	QRaux[], 	// (INPUT) 1D array of size NC, containing information regarding the QR decomposition. Must be as calculated using QR_decomposition(..).
					const double	Y[], 		// (INPUT) 1D vector of size NR, to be operated upon by the QR-decomposed A
					double 			QY[], 		// (OUTPUT) the product Q*Y, if requested. In that case, must be preallocated of size NR.
					double 			QTY[], 		// (OUTPUT) the product Q^T*Y, if requested. In that case, must be preallocated of size NR.
					double 			X[], 		// (OUTPUT) the solution to the least squares problem: minimize ||AK*X - Y||_2, if requested. In that case, it must be preallocated of size NK
					double 			residuals[], 	// (OUTPUT) the least-squares residuals Y - AK*X, if requested. In that case, must be preallocated of size NR.
					double 			AX[], 			// (OUTPUT) the least-squares approximation AX:=AK*X, if requested. In that case, it must be preallocated of size NR.
					const string 	&job){			// (INPUT) string of size 5 and of format ABCDE, where each character is either 1 or 0, specifying the various jobs to perform. For example, "00110" performs jobs C and D.
	long i, info, j, jj, ju;
	double t, temp;
	info = 0;

	// determine jobs
	bool cQY 	= (job[0]!='0'); // compute QY
	bool cQTY	= (job[1]!='0'); // compute QTY
	bool cx 	= (job[2]!='0'); // compute QTY and X
	bool cr 	= (job[3]!='0'); // compute QTY and RSD
	bool cax 	= (job[4]!='0'); // compute QTY and AX
	
	cQTY = cQTY || cx || cr || cax; // always compute QTY if one of cx, cr, cax is requested

	ju = min(NK, NR-1);
	if(ju == 0){
		if(cQY) QY[0] = Y[0];
		if(cQTY) QTY[0] = Y[0];
		if(cax) AX[0] = Y[0];
		if(cx){
			if(QRA[0+0*LDA] == 0.0){
				info = 1;
			}else{
				X[0] = Y[0] / QRA[0+0*LDA];
			}
		}
		if(cr) residuals[0] = 0.0;
		return info;
	}

	// prepare computation of QY or QTY
	if(cQY){
		for(i = 1; i <= NR; i++){
			QY[i-1] = Y[i-1];
		}
	}
	if(cQTY){
		for(i = 1; i <= NR; i++){
			QTY[i-1] = Y[i-1];
		}
	}
	
	// compute QY
	if(cQY){
		for(jj = 1; jj <= ju; jj++){
			j = ju - jj + 1;
			if(QRaux[j-1] != 0.0){
				temp = QRA[j-1+(j-1)*LDA];
				QRA[j-1+(j-1)*LDA] = QRaux[j-1];
				t = -QR_dot_product(NR-j+1, QRA+j-1+(j-1)*LDA, 1, QY+j-1, 1 ) / QRA[j-1+(j-1)*LDA];
				QR_add_vectors(NR-j+1, t, QRA+j-1+(j-1)*LDA, 1, QY+j-1, 1 );
				QRA[j-1+(j-1)*LDA] = temp;
			}
		}
	}
	
	// compute Q'*Y.
	if(cQTY){
		for(j = 1; j <= ju; j++){
			if(QRaux[j-1] != 0.0){
				temp = QRA[j-1+(j-1)*LDA];
				QRA[j-1+(j-1)*LDA] = QRaux[j-1];
				t = -QR_dot_product(NR-j+1, QRA+j-1+(j-1)*LDA, 1, QTY+j-1, 1 ) / QRA[j-1+(j-1)*LDA];
				QR_add_vectors(NR-j+1, t, QRA+j-1+(j-1)*LDA, 1, QTY+j-1, 1 );
				QRA[j-1+(j-1)*LDA] = temp;
			}
		}
	}

	// prepare computation of X, RSD, or AX.
	if(cx){
		for(i = 1; i <= NK; i++){
			X[i-1] = QTY[i-1];
		}
	}
	if(cax){
		for(i = 1; i <= NK; i++){
			AX[i-1] = QTY[i-1];
		}
	}
	if(cr && NK < NR){
		for(i = NK+1; i <= NR; i++){
			residuals[i-1] = QTY[i-1];
		}
	}
	if(cax && NK+1 <= NR){
		for(i = NK+1; i <= NR; i++){
			AX[i-1] = 0.0;
		}
	}
	if(cr){
		for(i = 1; i <= NK; i++){
			residuals[i-1] = 0.0;
		}
	}

	// compute X
	if(cx){
		for(jj = 1; jj <= NK; jj++){
			j = NK - jj + 1;
			if(QRA[j-1+(j-1)*LDA] == 0.0){
				info = j;
				break;
			}
			X[j-1] = X[j-1] / QRA[j-1+(j-1)*LDA];
			if(j != 1){
				t = -X[j-1];
				QR_add_vectors(j-1, t, QRA+0+(j-1)*LDA, 1, X, 1);
			}
		}
	}

	// compute residuals and AX, if needed
	if(cr || cax){
		for(jj = 1; jj <= ju; jj++){
			j = ju - jj + 1;
			if(QRaux[j-1] != 0.0){
				temp = QRA[j-1+(j-1)*LDA];
				QRA[j-1+(j-1)*LDA] = QRaux[j-1];
				if(cr){
					t = -QR_dot_product(NR-j+1, QRA+j-1+(j-1)*LDA, 1, residuals+j-1, 1) / QRA[j-1+(j-1)*LDA];
					QR_add_vectors(NR-j+1, t, QRA+j-1+(j-1)*LDA, 1, residuals+j-1, 1);
				}
				if(cax){
					t = -QR_dot_product(NR-j+1, QRA+j-1+(j-1)*LDA, 1, AX+j-1, 1) / QRA[j-1+(j-1)*LDA];
					QR_add_vectors(NR-j+1, t, QRA+j-1+(j-1)*LDA, 1, AX+j-1, 1 );
				}
				QRA[j-1+(j-1)*LDA] = temp;
			}
		}
	}

	return info;
}



// solve a linear system of equations of the format:
//	  A*X = B
// in a least squares sence, i.e. minimize ||A*X - B||_2
// Here, A is a real matrix of size NR x NC, and X is a 1D vector of size NC.
// Uses QR-decomposition, which must be performed beforehand using the function QR_decomposition(..)
// Hence, this function does not actually use the matrix A, but its QR decomposition QRA (and some other auxiliary variables).
// Code adjusted from: https://people.sc.fsu.edu/~jburkardt/c_src/qr_solve/qr_solve.html
void QR_linear_least_squares( 	double			QRA[], 			// (INPUT/OUTPUT) array of size LDA*NC, containing the QR factorization, as computed using QR_decomposition(..).  Not actually modified permanently by this function, but used temporarily internally. So its output is the same as input.
								const long 		LDA,			// (INPUT) the leading dimension of A, i.e. linear index periodicity between subsequent columns. Typically this will be equal to NR, but may also be larger (e.g. when processing a sub-matrix)
								const long 		NR, 			// (INPUT) the number of rows of A
								const long 		NC,				// (INPUT) the number of columns of A
								const long		rank, 			// (INPUT) the rank of the matrix, for example estimated via QR_decomposition
								const double	B[], 			// (INPUT) 1D vector of size NR
								long 			pivots[],		// (INPUT) 1D array of size NC, specifying the index of the column of A that has been interchanged into the K-th column, if pivoting was requested during QR. Must be as calculated using QR_decomposition(..).
								double 			QRaux[],		// (INPUT) 1D array of size NC, containing information regarding the QR decomposition. Must be as calculated using QR_decomposition(..).
								double 			X[], 			// (OUTPUT) 1D vector of size NC, containing a least-squares solution. Must be preallocated.
								double 			residuals[]){	// (OUTPUT) residuals, B - A*X
	long i, info, j, k;
	double t;
	double *dummyAB=NULL, *dummyQY=NULL;

	if(rank != 0){
		info = QR_operation(QRA, LDA, NR, rank, QRaux, B, dummyQY, residuals, X, residuals, dummyAB, "00110");
	}
	for(i = 0; i<NC; i++){
		pivots[i] = - pivots[i];
	}
	for(i = rank; i<NC; i++){
		X[i] = 0.0;
	}
	for(j = 1; j<=NC; j++){
		if(pivots[j-1] <= 0){
			k = - pivots[j-1];
			pivots[j-1] = k;

			while(k != j){
				t = X[j-1];
				X[j-1] = X[k-1];
				X[k-1] = t;
				pivots[k-1] = -pivots[k-1];
				k = pivots[k-1];
			}
		}
	}
	return;
}



// Solver the linear least squares problem:
//	minimize ||A*X-B||_2
// for some matrix A and some vector or matrix X, using QR-decomposition of A
// If B (and thus X) has NCb columns, this corresponds to NCb independent least-squares problems, which are solved separately (but profiting from the same QR decomposition)
// Most of the computation time goes into the QR-decomposition of the matrix A
void QR_linear_least_squares(	const long		NRa,		// (INPUT) number of rows in A
								const long 		NCa,		// (INPUT) number of columns in A, and number of rows in X
								const long		NCb,		// (INPUT) number of columns in B. This is the number of independent problems to solve.
								const std::vector<double> 	&A,			// (INPUT) 2D matrix of size NRa x NCa, in row-major or column-major format
								const std::vector<double> 	&B,			// (INPUT) 1D column vector of length NRa = NRb, or a 2D matrix of size NRa x NCb, in row-major or column-major format
								const bool					row_major,	// (INPUT) indicating whether A and B are stored in row-major format (instead of column-major). The same applies to the output X.
								std::vector<double>			&QRA,		// (SCRATCH) scratch space for internal computations and for storing the QR-decomposition of A. Will be resized as needed up to size NRa x NCa
								std::vector<double>			&scratch,	// (SCRATCH) scratch space for internal computations. Will be resized as needed up to size max(NCa,NRa)
								std::vector<double>			&X,			// (OUTPUT) vector or matrix of size NCa x NCb, storing the solutitions to the least-squares problems. Will be in row-major format if row_major=true, otherwise it will be in column-major format.
								long						&rank){		// (OUTPUT) the estimated rank of the matrix A
	const long NRx = NCa;
	const long NCx = NCb;
	long r,c;

	// prepare and work on a copy of A, in column-major format
	const long LDA = NRa; // the leading dimension of A, i.e. linear index periodicity between subsequent columns. Here we assume that A is stored compactly in memory, i.e. LDA=NRa (after A is turned into column-major format, if needed).
	std::vector<double> BCM(NRa*NCb); // store a copy of B in column-major format
	if(row_major){
		// transform A into column-major format and store result in QRA
		QRA.resize(NRa*NCa);
		for(r=0; r<NRa; ++r){
			for(c=0; c<NCa; ++c){
				QRA[c*NRa + r] = A[r*NCa + c];
			}
		}
		// transform B into column-major format and store result in BCM
		for(r=0; r<NRa; ++r){
			for(c=0; c<NCb; ++c){
				BCM[c*NRa + r] = B[r*NCb + c];
			}
		}
	}else{
		// A and B are already in column-major format
		QRA = A;
		BCM = B;
	}
						
	// compute QR decomposition of A, and store results directly in QRA
	scratch.resize(NCa);
	std::vector<double> QRaux(NCa);
	std::vector<long> pivots(NCa);
	QR_decomposition (	&QRA[0],
						LDA,
						NRa,
						NCa,
						true, // pivoting
						&scratch[0],
						&QRaux[0],
						&pivots[0],
						rank);
	
	// use QR decomposition to minimize ||A*X - B||_2, separately for each column in X
	// QRA, BCM, XCM are all treated in column-major format at this point
	std::vector<double> XCM(NCa*NCb);
	scratch.resize(NRa); // use to store residuals
	for(long j=0; j<NCb; ++j){
		QR_linear_least_squares(&QRA[0], LDA, NRa, NCa, rank, &BCM[j*NRa], &pivots[0], &QRaux[0], &XCM[j*NCa], &scratch[0]);
	}
		
	// transform XCM to row-major format and store in X, if needed
	if(row_major){
		X.resize(NRx*NCx);
		for(r=0; r<NRx; ++r){
			for(c=0; c<NCx; ++c){
				X[r*NCx + c] = XCM[c*NRx + r];
			}
		}
	}else{
		X = XCM;
	}
}



// calculate the inverse of a square matrix A, using QR decomposition
// A can be in row-major or column-major format; the internal computation is the same, because confusing majority is equivalernt to temporary transposing
void QR_matrix_inverse(	const long					N,			// (INPUT) number of rows & columns in A
						const std::vector<double> 	&A,			// (INPUT) 2D matrix of size N x N, in row-major or column-major format (which, does not matter)
						std::vector<double>			&QRA,		// (SCRATCH) scratch space for internal computations and for storing the QR-decomposition of A. Will be resized as needed up to size N^2
						std::vector<double>			&Ainv,		// (OUTPUT) 2D matrix of size N x N, storing the inverse of A (or an approximation thereof, if A is non-invertible). Ainv will be in row-major format iff A was in row-major format.
						long						&rank){		// (OUTPUT) an estimate of the rank of A. If A is invertible, this will be equal to N.
	dvector identity, scratch;
	get_identity_matrix(N,identity);
	// solve linear system A*Ainv = identity, in the least squares sense
	// note that we pretent as if A was in column-major format for efficiency (QR would otherwise transform everything temporarily)
	//   If A is actually in row-major format, then pretending as if it's column-major is equivalent to transposing A, 
	//   hence the obtained Ainv will just be the inverse of A^T in column-major format, or equivalently, the inverse of A in row-major format.
	QR_linear_least_squares(N,N,N,A,identity,false,QRA,scratch,Ainv,rank);
}




// LU decomposition of square matrix
// Crout’s method with partial pivoting
// matrix[] (input/output) will store the LU decomposition as a return value (in row-major format). This output can be used as input for solving linear equations (Ax=b) or inverting the original matrix
// pivoting_indices[] (output) will store the pivoting indices, i.e. the row permutation effected by the partial pivoting. 
// pivoting_indices[] should already be allocated before calling (size at least N).
// Psign (output) returns the signature of the permutation pivoting_indices, i.e. will be +1 (-1) if the number of row permutations is even (uneven)
template<class TYPE>
bool LUDecomposition(	TYPE			matrix[], 	// (INPUT/OUTPUT) in row major format
						unsigned long 	N, 			// (INPUT) matrix row count = column count
						unsigned long 	pivoting_indices[], 
						int 			&Psign){
	if(N==0) return false;
	long i,imax,j,k;
	double big,dummy,temp;
	TYPE sum,dummyT;
	double *W = new double[N];
	Psign = 1;
	
	for(i=0; i<N; ++i){
		for(j=0, big=0; j<N; ++j){
			if((temp = abs(matrix[i*N+j])) > big){ big = temp; }
		}
		if(big == 0){
			delete[] W;
			return false; //singular matrix
		}
		W[i] = 1.0/big;
	}
	for(j=0; j<N; ++j){
		for(i=0; i<j; ++i){
			sum = matrix[i*N+j];
			for(k=0; k<i; ++k){ sum -= matrix[i*N+k] * matrix[k*N+j]; }
			matrix[i*N+j] = sum;
		}
		big=0.0;
		for(i=j; i<N; ++i){
			sum = matrix[i*N+j];
			for (k=0; k<j; ++k){ sum -= matrix[i*N+k] * matrix[k*N+j]; }
			matrix[i*N+j] = sum;
			if((dummy=W[i]*abs(sum)) >= big){
				big	 = dummy;
				imax = i;
			}
		}
		if(j != imax){
			for(k=0; k<N; ++k){
				dummyT 			= matrix[imax*N+k];
				matrix[imax*N+k] = matrix[j*N+k]; 
				matrix[j*N+k] 	= dummyT;
			}
			Psign = -Psign;
			W[imax] = W[j];
		}
		pivoting_indices[j] = imax;
		if(abs(matrix[j*N+j]) == 0.0){
			matrix[j*N+j] = RELATIVE_EPSILON; //If the pivot element is zero the matrix is singular (at least to the precision of the algorithm). For some applications on singular matrices, it is desirable to substitute EPSILON for zero. 
		}
		if(j < N-1){
			dummyT = 1.0/(matrix[j*N+j]);
			for(i=j+1; i<N; ++i){ matrix[i*N+j] *= dummyT; }
		}
	}
	delete[] W;
	return true;
}


// Solve the set of n linear equations Ax=b for the vector x and a rectangular (non-singular) matrix A.
// Uses forward substitution and back substitution.
// Matrix needs to be in LU decomposition row-major format, as returned by the routine LUDecomposition(..) defined above.
// pivoting_indices[] should store the pivoting indices from the LU decomposition, as returned by LUDecomposition(..)
// b[] (input/output) will return the solution vector x upon completion.
// This routine takes into account the possibility that b[] will begin with many zero elements, so it is efficient for use in matrix inversion.
template<class TYPE>
void LUSolveLinearSystem(	const TYPE			LUmatrix[],  // (INPUT)
							unsigned long 		N, 
							const unsigned long pivoting_indices[], 
							TYPE				b[]){
	long i, ii=-1, ip, j; 
	TYPE sum;
	for(i=0; i<N; ++i){
		ip = pivoting_indices[i];
		sum = b[ip];
		b[ip] = b[i];
		if(ii>=0){
			for(j=ii; j<=i-1; ++j){ sum -= LUmatrix[i*N+j] * b[j]; }
		}else if(abs(sum)>RELATIVE_EPSILON){
			ii = i; 
		}
		b[i] = sum;
	}
	for(i=N-1; i>=0; --i){
		sum = b[i];
		for(j=i+1; j<N; ++j){ sum -= LUmatrix[i*N+j]*b[j]; }
		b[i] = sum/LUmatrix[i*N+i];
	}
}



template<class TYPE>
double errorInLinearSolution(	const TYPE			matrix[], // (INPUT) in row-major format
								unsigned long 		N, 
								const TYPE 			b[], 
								const TYPE 			x[]){
	double error = 0;
	TYPE Ax;
	unsigned long i,j;
	for(i=0; i<N; ++i){
		for(j=0, Ax=0; j<N; ++j){ Ax += matrix[i*N+j]*x[j]; }
		error += SQ(abs(Ax-b[i]));
	}
	return std::sqrt(error);
}



// Iteratively improve solution x[] to linear system Ax=b
// Uses LU-decomposed matrix as well as original matrix
// Only iterates once, so repeat this step as much as needed to achieve the wanted accuracy
// x[] (input/output) should initially store the approximate (but inaccurate) solution. Upon return, it will store the improved solution.
template<class TYPE>
void LUImproveSolutionToLinearSystem(	const TYPE			matrix[], 	//original square matrix (in row-major format)
										const TYPE			LUmatrix[], //LU decomposed matrix in row-major format, as returned by LUDecomposition()
										unsigned long 		N, 
										const unsigned long pivoting_indices[], 	//as returned by LUDecomposition()
										const TYPE 			b[], 
										TYPE 				x[]){ 		//approximate solution we want to improve
	long i,j;
	TYPE s;
	TYPE *r = new TYPE[N];
	for(i=0; i<N; ++i){
		s = -b[i];
		for(j=0; j<N; ++j){ s += matrix[i*N+j] * x[j]; }
		r[i] = s;
	}
	LUSolveLinearSystem(LUmatrix, N, pivoting_indices, r);
	for(i=0; i<N; ++i){ x[i] -= r[i]; }
	delete[] r;
}



// Solve linear system Ax=b for vector x, using LU decomposition of the matrix A
// scratchSpace[] is only needed for the internal calculations, and should be allocated prior to calling (size at least N*N).
// maxError defines the maximum acceptable L2 norm E(X):=|AX-b|, where X is the approximate solution.
// As long as E(X)>maxError, the solution X is improved iteratively.
// If maxError<=0 or maxImprovements==0, no iterative improvement is performed, i.e. X is set to the solution obtained from regular LU decomposition.
// x[] will contain the solution vector upon return.
// x[] should be allocated prior to calling (size at least N).
// Returns false on error (e.g. when matrix is singular).
template<class TYPE>
bool LUsolveLinearSystem(	const TYPE 			matrix[], 			// (INPUT) in row major format
							TYPE				scratchSpace[],		// (SCRATCH) pre-allocated of size N*N
							unsigned long 		N,
							const TYPE 			b[],
							double 				maxError, 			//max allowed L2 error |Ax-b|
							unsigned int 		maxImprovements, 	//max allowed number of iterative improvements of solution. Use this to cap computing time if your matrices are unpredictably *pathological*
							TYPE 				x[]){				// (OUTPUT) the solution vector x. Must be pre-allocated of size N.
	if(N==0) return false;
	int Psign;
	long i,j;
	unsigned long *pivoting_indices = new unsigned long[N];
	for(i=0; i<N; ++i){
		for(j=0; j<N; ++j){
			scratchSpace[i*N+j] = matrix[i*N+j];
		}
		x[i] = b[i];
	}
	
	if(!LUDecomposition(scratchSpace, N, pivoting_indices, Psign)){
		delete[] pivoting_indices;
		return false; //error on LU decomposition
	}
	
	LUSolveLinearSystem(scratchSpace, N, pivoting_indices, x);
	
	if((maxError>0) && (maxImprovements>0)){
		int impCount = 0;
		while((errorInLinearSolution(matrix, N, b, x) > maxError) && (impCount<maxImprovements)){
			LUImproveSolutionToLinearSystem(matrix, scratchSpace, N, pivoting_indices, b, x);
			++impCount;
		}
	}
	
	
	delete[] pivoting_indices;
	return true;
}


// Calculate the inverse of a rectangular non-singular matrix.
// Matrix needs to be in LU decomposition format (LUmatrix), as returned by the routine LUDecomposition(..) defined above.
// IPIV[] should store the pivoting indices from the LU decomposition, as returned by LUDecomposition(..)
// inverse[] (output) will contain the inverse matrix in row-major format, and should already be allocated before calling (size at least N*N).
template<class TYPE>
void LUInverse(	const TYPE			LUmatrix[], 
				unsigned long 		N, 
				const unsigned long IPIV[], 
				TYPE				inverse[]){
	TYPE *col = new TYPE[N];
	long i,j;
	for(i=0; i<N; ++i){
		for(j=0; j<N; ++j){ col[j] = 0; }
		col[i] = 1;
		LUSolveLinearSystem(LUmatrix, N, IPIV, col);
		for(j=0; j<N; ++j){ inverse[j*N+i] = col[j]; }
	}
	delete[] col;
}



// Calculate the inverse of a rectangular, non-singular matrix.
// Uses LU decomposition, forward substitution and back substitution.
// Merely a wrapper for the routines LUDecomposition() and LUInverse() defined above.
// Returns false on error (e.g. if matrix is singular).
// matrix[] (input) will be modified after this call (so back it up if you still need it).
// inverse[] (output) should already be allocated before calling (at least size N*N).
template<class TYPE>
bool inverseMatrix(	TYPE 			matrix[], 
					unsigned long 	N, 
					TYPE			inverse[]){
	if(N==0) return false;
	unsigned long *IPIV = new unsigned long[N];
	int Psign;
	if(!LUDecomposition(matrix, N, IPIV, Psign)){
		delete[] IPIV;
		return false; // error on LU decomposition
	}
	LUInverse(matrix, N, IPIV, inverse);
	delete[] IPIV;
	return true;
}



// same as above, but LU decomposition method is supplemented by iterative improvement of solution
template<class TYPE>
bool inverseMatrix(	const TYPE			matrix[], // (input)
					unsigned long 		N, 
					TYPE		 		inverse[], // (output) should be allocated to size at least N*N
					TYPE				scratchSpace[], // should be allocated to size at least N*N	
					double 				maxError, //max allowed L2 error |Ax - b|, where x is any column of the inverse and b is the corresponding unit vector (0,..,1,..,0)
					unsigned int 		maxImprovements){ //max allowed number of iterative improvements of inverse. Use this to cap computing time if your matrices are unpredictably *pathological*. Set to 0 for unlimited number of improvements.
	if(N==0) return false;
	unsigned long *IPIV = new unsigned long[N];
	int Psign;
	long i,j;
	int impCount;	
	
	//scratch with a copy of the matrix
	for(i=0; i<N*N; ++i){
		scratchSpace[i] = matrix[i];
	}
	
	if(!LUDecomposition(scratchSpace, N, IPIV, Psign)){
		delete[] IPIV;
		return false; // error on LU decomposition
	}
	
	TYPE *b = new TYPE[N];
	TYPE *x = new TYPE[N];
	for(j=0; j<N; ++j){
		for(i=0; i<N; ++i){ x[i] = b[i] = 0; }
		x[j] = b[j] = 1;
		LUSolveLinearSystem(scratchSpace, N, IPIV, x);
		if((maxError>0) && (maxImprovements>0)){
			impCount = 0;
			while((errorInLinearSolution(matrix, N, b, x) > maxError) && ((impCount<maxImprovements) || (maxImprovements==0))){
				LUImproveSolutionToLinearSystem(matrix, scratchSpace, N, IPIV, b, x);
				++impCount;
			}
		}
		for(i=0; i<N; ++i){ inverse[i*N+j] = x[i]; }
	}
	
	
	delete[] b;
	delete[] x;
	delete[] IPIV;
	return true;
}



double get_matrix_trace(const long NR, const std::vector<double> &matrix){
	double T = 0;
	for(long r=0; r<NR; ++r){
		T +=matrix[r*NR+r];
	}
	return T;
}




// calculate the Hilbert-Schmidt norm (L2-norm) of a matrix: ||A||_2 = SQRT(sum_{r,c}|A_rc|^2)
double get_matrix_norm_L2(	const long 	 NR,	// (INPUT) number of rows & columns of the matrix A
							const double A[]){ 	// (INPUT) matrix of size NR x NR, in row-major format
	double S = 0;
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NR; ++c){
			S += SQR(A[r*NR+c]);
		}
	}
	return sqrt(S);
}


template<class ARRAY_TYPE>
double get_norm_L2_of_vector(const ARRAY_TYPE &X){
	double S=0;
	for(long i=0; i<X.size(); ++i) S += SQR(X[i]);
	return sqrt(S);
}


template<class ARRAY_TYPE>
double get_norm_L2_of_inverted_vector(const ARRAY_TYPE &X){
	double S=0;
	for(long i=0; i<X.size(); ++i) S += SQR(1.0/X[i]);
	return sqrt(S);
}


double get_column_norm_L2(	const long 					NR,		// (INPUT) number of rows & columns in the matrix A
							const long 					c,		// (INPUT) column index for which to calculate the L2 norm
							const std::vector<double>	&A){	// (INPUT) quadratic matrix of size NR x NR, in row-major format
	double S = 0;
	for(long r=0; r<NR; ++r) S += SQR(A[r*NR + c]);
	return sqrt(S);
}


double get_row_norm_L2(	const long 					NR,		// (INPUT) number of rows & columns in the matrix A
						const long 					r,		// (INPUT) row index for which to calculate the L2 norm
						const std::vector<double>	&A){	// (INPUT) quadratic matrix of size NR x NR, in row-major format
	double S = 0;
	for(long c=0; c<NR; ++c) S += SQR(A[r*NR + c]);
	return sqrt(S);
}


double sum_of_column(	const long 					NR,			// (INPUT) number of rows in the matrix A
						const long 					NC,			// (INPUT) number of columns in the matrix A
						const std::vector<double>	&A,			// (INPUT) matrix of size NR x NC, in row-major format
						const long 					column){	// (INPUT) focal column, the entries of which are to be summed
	double sum = 0;
	for(long r=0; r<NR; ++r){
		sum += A[r*NC+column];
	}
	return sum;
}

double sum_of_row(	const long 					NR,			// (INPUT) number of rows in the matrix A
					const long 					NC,			// (INPUT) number of columns in the matrix A
					const std::vector<double>	&A,			// (INPUT) matrix of size NR x NC, in row-major format
					const long 					row){	// (INPUT) focal row, the entries of which are to be summed
	double sum = 0;
	for(long c=0; c<NC; ++c){
		sum += A[row*NC+c];
	}
	return sum;
}


// transform a quadratic matrix A using another diagonal matrix D: A --> D*A*D^{-1}  or  D^{-1}*A*D
void diagonally_transform_matrix(	const long	 				NR,				// number of rows & columns in A & D
									const std::vector<double> 	&D,				// array of size NR, diagonal elements of the diagonal transformation matrix D
									const bool					inverse_right,	// (INPUT) if true, then the transformed matrix will be D*A*D^{-1}, otherwise it will be D^{-1}*A*D.
									double 						A[]){			// (INPUT/OUTPUT) 2D array of size NR x NR, in row-major format. Matrix to be transformed in-situ.
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NR; ++c){
			A[r*NR + c] *= (inverse_right ? D[r]/D[c] : D[c]/D[r]);
		}
	}
}


// replace a quadratic matrix A in-situ by its square, A*A
void get_square_matrix(	const long		NR,				// (INPUT) number of rows & columns in the matrix A
						const double	A[],			// (INPUT) 2D matrix of size NR x NR, in row-major format. Matrix to be squared.
						double			Asquared[]){	// (OUTPUT) 2D matrix of size NR x NR, in row-major format, equal to A*A. Must be pre-allocated.
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NR; ++c){
			Asquared[r*NR + c] = 0;
			for(long k=0; k<NR; ++k){
				Asquared[r*NR + c] += A[r*NR + k] * A[k*NR + c];
			}
		}
	}
}






// Calculate the product Y = A * X
template<class TYPE1,class TYPE2,class TYPE3>
void multiply_matrix_with_vector(	const long			NR,
									const long			NC,
									TYPE1				A[],	// (INPUT) array of size NR*NC, in row-major format
									TYPE2				X[],	// (INPUT) pre-allocated array of size NC or greater
									std::vector<TYPE3>	&Y){	// (OUTPUT) product A*X, of size NR
	if((NR==2) && (NC==2)){
		// 2 x 2 matrix, treat as special case for computational efficiency
		// this is useful for example in BiSSE models, where a matrix is multiplied with a vector numerous times
		Y.resize(NR);
		Y[0] = A[0]*X[0] + A[1]*X[1];
		Y[1] = A[2]*X[0] + A[3]*X[1];
	}else if((NR==3) && (NC==3)){
		// 3 x 3 matrix, treat as special case for computational efficiency
		// this is useful for example in spherical diffusion models, where a matrix is multiplied with a vector numerous times
		Y.resize(NR);
		Y[0] = A[0]*X[0] + A[1]*X[1] + A[2]*X[2];
		Y[1] = A[3]*X[0] + A[4]*X[1] + A[5]*X[2];
		Y[2] = A[6]*X[0] + A[7]*X[1] + A[8]*X[2];
	}else{
		Y.assign(NR,0);
		for(long r=0; r<NR; ++r){
			for(long c=0; c<NC; ++c){
				Y[r] += A[r*NC+c] * X[c];
			}
		}
	}
}


// multiply matrix A with column-vector X
// matrix is assumed in row-major format
template<class TYPE1, class TYPE2, class TYPE3>
inline void multiply_matrix_with_vector(const long					NR,
										const long					NC,
										const std::vector<TYPE1>	&A,		// (INPUT) matrix of size NR*NC, in row-major format
										const std::vector<TYPE2>	&X,		// (INPUT) std::vector of size NC
										std::vector<TYPE3>			&Y){	// (OUTPUT) product A*X, of size NR
	if((NR==2) && (NC==2)){
		// 2 x 2 matrix, treat as special case for computational efficiency
		// this is useful for example in BiSSE models, where a matrix is multiplied with a vector numerous times
		Y.resize(NR);
		Y[0] = A[0]*X[0] + A[1]*X[1];
		Y[1] = A[2]*X[0] + A[3]*X[1];
	}else if((NR==3) && (NC==3)){
		// 3 x 3 matrix, treat as special case for computational efficiency
		// this is useful for example in spherical diffusion models, where a matrix is multiplied with a vector numerous times
		Y.resize(NR);
		Y[0] = A[0]*X[0] + A[1]*X[1] + A[2]*X[2];
		Y[1] = A[3]*X[0] + A[4]*X[1] + A[5]*X[2];
		Y[2] = A[6]*X[0] + A[7]*X[1] + A[8]*X[2];
	}else{
		Y.assign(NR,0);
		long r, c;
		for(r=0; r<NR; ++r){
			for(c=0; c<NC; ++c){
				Y[r] += A[r*NC+c] * X[c];
			}
		}
	}
}


// Calculate the product Y = A * X, where X and Y are passed in log-transformed format
template<class TYPE1,class TYPE2,class TYPE3>
void multiply_matrix_with_log_vector(	const long			NR,
										const long			NC,
										TYPE1				A[],	// (INPUT) array of size NR*NC, in row-major format
										TYPE2				logX[],	// (INPUT) array of size NC or greater, log-transformed X
										std::vector<TYPE3>	&logY){	// (OUTPUT) log-transformed product A*X, of size NR
	if((NR==2) && (NC==2)){
		// 2 x 2 matrix, treat as special case for computational efficiency
		// this is useful for example in BiSSE models, where a matrix is multiplied with a vector numerous times
		logY.resize(NR);
		const double X0 = exp(logX[0]);
		const double X1 = exp(logX[1]);
		logY[0] = log(A[0]*X0 + A[1]*X1);
		logY[1] = log(A[2]*X0 + A[3]*X1);
	}else if((NR==3) && (NC==3)){
		// 3 x 3 matrix, treat as special case for computational efficiency
		// this is useful for example in spherical diffusion models, where a matrix is multiplied with a vector numerous times
		logY.resize(NR);
		const double X0 = exp(logX[0]);
		const double X1 = exp(logX[1]);
		const double X2 = exp(logX[2]);
		logY[0] = log(A[0]*X0 + A[1]*X1 + A[2]*X2);
		logY[1] = log(A[3]*X0 + A[4]*X1 + A[5]*X2);
		logY[2] = log(A[6]*X0 + A[7]*X1 + A[8]*X2);
	}else{
		logY.assign(NR,0);
		double Xc;
		for(long c=0; c<NC; ++c){
			Xc = exp(logX[c]);
			for(long r=0; r<NR; ++r){
				logY[r] += A[r*NC+c] * Xc;
			}
		}
		for(long r=0; r<NR; ++r){
			logY[r] = log(logY[r]);
		}
	}
}



// Calculate the product Y = A^T * X
template<class TYPE1,class TYPE2,class TYPE3>
void multiply_Tmatrix_with_vector(	const long			NR,		// (INPUT) number of rows in A, must be equal to the length of X
									const long			NC,		// (INPUT) number of columns in A, will be equal to the length of Y
									TYPE1				A[],	// (INPUT) array of size NR*NC, in row-major format
									TYPE2				X[],	// (INPUT) pre-allocated array of size NC or greater
									std::vector<TYPE3>	&Y){	// (OUTPUT) product A*X, of size NR
	if((NR==2) && (NC==2)){
		// 2 x 2 matrix, treat as special case for computational efficiency
		// this is useful for example in BiSSE models, where a matrix is multiplied with a vector numerous times
		Y.resize(NC);
		Y[0] = A[0]*X[0] + A[2]*X[1];
		Y[1] = A[1]*X[0] + A[3]*X[1];
	}else if((NR==3) && (NC==3)){
		// 3 x 3 matrix, treat as special case for computational efficiency
		// this is useful for example in spherical diffusion models, where a matrix is multiplied with a vector numerous times
		Y.resize(NC);
		Y[0] = A[0]*X[0] + A[3]*X[1] + A[6]*X[2];
		Y[1] = A[1]*X[0] + A[4]*X[1] + A[7]*X[2];
		Y[2] = A[2]*X[0] + A[5]*X[1] + A[8]*X[2];
	}else{
		Y.assign(NC,0);
		for(long r=0; r<NR; ++r){
			for(long c=0; c<NC; ++c){
				Y[c] += A[r*NC+c] * X[r];
			}
		}
	}
}


// Calculate the product Y = A^T * X = (X^T * A)^T, where X and Y are passed in log-transformed format
template<class TYPE1,class TYPE2,class TYPE3>
void multiply_Tmatrix_with_log_vector(	const long			NR,		// (INPUT) number of rows in A, must be equal to the length of X
										const long			NC,		// (INPUT) number of columns in A, will be equal to the length of Y
										TYPE1				A[],	// (INPUT) array of size NR*NC, in row-major format. A[0]=A[1,1], A[1]=A[1,2], A[2]=A[1,3], A[3]=A[2,1], A[4]=A[2,2], A[5]=A[2,3], A[6]=A[3,1], A[7]=A[3,2], A[8]=A[3,3]
										TYPE2				logX[],	// (INPUT) array of size NC or greater, log-transformed X
										std::vector<TYPE3>	&logY){	// (OUTPUT) log-transformed product A*X, of size NR
	if((NR==2) && (NC==2)){
		// 2 x 2 matrix, treat as special case for computational efficiency
		// this is useful for example in BiSSE models, where a matrix is multiplied with a vector numerous times
		logY.resize(NC);
		const double X0 = exp(logX[0]);
		const double X1 = exp(logX[1]);
		logY[0] = log(A[0]*X0 + A[2]*X1);
		logY[1] = log(A[1]*X0 + A[3]*X1);
	}else if((NR==3) && (NC==3)){
		// 3 x 3 matrix, treat as special case for computational efficiency
		// this is useful for example in spherical diffusion models, where a matrix is multiplied with a vector numerous times
		logY.resize(NC);
		const double X0 = exp(logX[0]);
		const double X1 = exp(logX[1]);
		const double X2 = exp(logX[2]);
		logY[0] = log(A[0]*X0 + A[3]*X1 + A[6]*X2);
		logY[1] = log(A[1]*X0 + A[4]*X1 + A[7]*X2);
		logY[2] = log(A[2]*X0 + A[5]*X1 + A[8]*X2);
	}else{
		logY.assign(NC,0);
		double Xr;
		for(long r=0; r<NR; ++r){
			Xr = exp(logX[r]);
			for(long c=0; c<NC; ++c){
				logY[c] += A[r*NC+c] * Xr;
			}
		}
		for(long c=0; c<NC; ++c){
			logY[c] = log(logY[c]);
		}
	}
}



// Calculate the product Y = X^T * A
template<class TYPE1,class TYPE2,class TYPE3>
void multiply_vector_with_matrix(	const long			NR,
									const long			NC,
									TYPE2				X[],	// (INPUT) pre-allocated array of size NR or greater
									TYPE1				A[],	// (INPUT) array of size NR*NC, in row-major format
									std::vector<TYPE3>	&Y){	// (OUTPUT) product X^T*A, of size NC
	Y.assign(NC,0);
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			Y[c] += X[r] * A[r*NC+c];
		}
	}
}




// Calculate the product Y = X^T * A
// matrix is assumed in row-major format
template<class TYPE1, class TYPE2, class TYPE3>
void multiply_vector_with_matrix(	const long					NR,
									const long					NC,
									const std::vector<TYPE2>	&X,		// (INPUT) std::vector of size NR
									const std::vector<TYPE1>	&A,		// (INPUT) matrix of size NR*NC, in row-major format
									std::vector<TYPE3>			&Y){	// (OUTPUT) product X^T*A, of size NC
	Y.assign(NC,0);
	long r,c;
	for(r=0; r<NR; ++r){
		for(c=0; c<NC; ++c){
			Y[c] += X[r] * A[r*NC+c];
		}
	}
}



// multiply matrix A with another matrix B
// matrices are assumed to be in row-major format
template<class ARRAY_TYPE1, class ARRAY_TYPE2, class ARRAY_TYPE3>
void multiply_matrices(	const long			NRa,	// (INPUT) number of rows in A
						const long			NCa,	// (INPUT) number of columns in A. Must be equal to the number of rows in B.
						const long			NCb,	// (INPUT) number of columns in B
						const ARRAY_TYPE1	&A,		// (INPUT) 2D matrix of size NRa x NCa, in row-major format
						const ARRAY_TYPE2	&B,		// (INPUT) 2D matrix of size NCa x NCb, in row-major format
						ARRAY_TYPE3			&AB){	// (OUTPUT) 2D matrix of size NRa x NCb, in row-major format, containing the product A*B.
	const long NR = NRa;
	const long NC = NCb;
	AB.assign(NR*NC,0);
	long r,c,k;
	for(r=0; r<NR; ++r){
		for(c=0; c<NC; ++c){
			for(k=0; k<NCa; ++k){
				AB[r*NC + c] += A[r*NCa + k] * B[k*NCb + c];
			}
		}
	}
}



// multiply matrix A with another matrix B
// matrices are assumed to be in row-major format
// Special case: cast the output to real valued, by omitting the imaginary part
template<>
void multiply_matrices(	const long					NRa,	// (INPUT) number of rows in A
						const long					NCa,	// (INPUT) number of columns in A. Must be equal to the number of rows in B.
						const long					NCb,	// (INPUT) number of columns in B
						const std::vector<cdouble>	&A,		// (INPUT) 2D matrix of size NRa x NCa, in row-major format
						const std::vector<cdouble>	&B,		// (INPUT) 2D matrix of size NCa x NCb, in row-major format
						std::vector<double>			&AB){	// (OUTPUT) 2D matrix of size NRa x NCb, in row-major format, containing the product A*B.
	const long NR = NRa;
	const long NC = NCb;
	AB.resize(NR*NC);
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			cdouble S = 0;
			for(long k=0; k<NCa; ++k){
				S += A[r*NCa + k] * B[k*NCb + c];
			}
			AB[r*NC + c] = S.real();			
		}
	}
}



// multiply matrix A with another diagonal matrix D
// this is the same as multiplying the c-th column in A with the c-th diagonal entry in D
template<class ARRAY_TYPE1, class ARRAY_TYPE2, class ARRAY_TYPE3>
void multiply_matrix_with_diagonal_matrix(	const long			NR,		// (INPUT) number of rows in A
											const long			NC,		// (INPUT) number of columns in A. Must be equal to the number of rows & columns in D.
											const ARRAY_TYPE1	&A,		// (INPUT) 2D matrix of size NR x NC, in row-major format
											const ARRAY_TYPE2	&D,		// (INPUT) 1D array of size NC, containing the diagonal elements of the diagonal matrix D.
											ARRAY_TYPE3			&AD){	// (OUTPUT) matrix of size NR x NC, in row-major format, containing the product A*D.
	AD.resize(NR*NC);
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			AD[r*NC + c] = A[r*NC + c] * D[c];
		}
	}
}




template<class ARRAY_TYPE>
void transpose_matrix(	const long			NR,		// (INPUT) number of rows in A
						const long			NC,		// (INPUT) number of columns in A.
						const ARRAY_TYPE	&A,		// (INPUT) 2D matrix of size NR x NC, in row-major format
						ARRAY_TYPE			&At){	// (OUTPUT) 2D matrix of size NC x NR, in row-major format, containing the transpose of A
	At.resize(NR*NC);
	for(long r=0; r<NR; ++r){
		for(long c=0; c<NC; ++c){
			At[c*NR+r] = A[r*NC+c];
		}
	}
}


// calculate the appropriate base-2 scaling power for the scaling-and-squaring in matrix exponentiation
// and automatically rescale the matrix
void get_scaling_power_for_matrix_exponentiation(	const long 			&NR,				// (INPUT) number of rows & column in the matrix A
													std::vector<double> &A,					// (INPUT/OUTPUT) 2D matrix of size NR x NR, in row-major format. Matrix to be rescaled and replaced by A/2^scaling_power.
													long				&scaling_power,		// (OUTPUT) base-2 power of the scaling applied, i.e. the rescaled matrix is A/2^scaling_power
													double				&rescaled_norm){	// (OUTPUT) norm of the rescaled matrix, i.e. ||A/2^scaling_power||_2
	scaling_power = 0;
	const double norm = get_matrix_norm_L2(NR, &A[0]);
	if(norm<=1.0){
		scaling_power = 0.0; // no rescaling needed
		rescaled_norm = norm;
	}else{
		scaling_power = (long)ceil(log(norm)/log(2.0));
		double factor = pow(0.5, scaling_power);
		multiply_array_with_scalar(A,factor);
		rescaled_norm = norm * factor;
	}
}



// balance a quadratic matrix A using a diagonal transformation matrix D ("balances matrix"): A --> A'=D^{-1}*A*D
// so that the L2-norms of the columns of the transformed A' are roughly the same as the L2 norms of the corresponding rows
// Uses the balancing algorithm proposed by: Rodney James et al (2014). On matrix balancing and eigenvector computation. Algorithm 3.
void balance_matrix_with_diagonal_transformation(	const long			NR,		// (INPUT) number of rows & columns in the matrix A
													std::vector<double>	&A,		// (INPUT/OUTPUT) 2D matrix of size NR x NR, in row-major format. Input matrix to be balanced, i.e. replaced by D^{-1}*A*D
													std::vector<double>	&D){	// (OUTPUT) 1D array of size NR, storing the diagonal elements of the balances matrix D.
	D.assign(NR,1.0);
	const double beta = 2;
	bool converged = false;
	double C,R,S,f;
	while(!converged){
		converged = true;
		for(long i=0; i<NR; ++i){
			C = get_column_norm_L2(NR,i,A);
			R = get_row_norm_L2(NR,i,A);
			if((C<=1e-32*R) || (R<=1e-32*C)){
				// row and/or column is practically zero, so no point in rescaling. Move on to next row & column.
				D[i] = 1;
				continue;
			}
			S = SQR(C) + SQR(R);
			f = 1;
			while(C<R/beta){
				C *= beta;
				R /= beta;
				f *= beta;
			}
			while(C>=R*beta){
				C /= beta;
				R *= beta;
				f /= beta;
			}
			if((SQR(C)+SQR(R))<0.95*S){
				converged = false;
				D[i] *= f;
				for(long j=0; j<NR; ++j){
					A[j*NR + i] *= f;
					A[i*NR + j] /= f;
				}
			}
		}
	}
}



// estimate the minimum number of polynomials A^p/p! to include for exponentiating an irreducible matrix A via Taylor expansion, such that the approximated exp(A) does not contain any zeros.
// For an irreducible matrix A, exp(A) cannot include zeros.
// In Mk-model ancestral state reconstruction algorithms with transition matrix Q, false zeros in exp(Q) can mess up the algorithm
// This function determines the minimum number P (P<=NR+1) such that sum_{p=0}^{P-1} abs(A^p) has positive entries everywhere
// Note that if the matrix is reducible, the returned value would still be <=NR+1.
template<class ARRAY_TYPE>
long min_polynomials_for_positive_exponential_of_irreducible_matrix(const long			NR,		// (INPUT) number of rows & columns of the matrix A
																	const ARRAY_TYPE	&A){	// (INPUT) 2D array of size NR x NR, in row-major format
	std::vector<double> scratch_power1, scratch_power2;
	get_identity_matrix(NR,scratch_power1);
	std::vector<double> sum_of_powers = scratch_power1;
	std::vector<double> &last_power = scratch_power1;
	for(long p=1; p<=NR; ++p){
		if((p%2)==1){
			multiply_matrices(NR, NR, NR, scratch_power1, A, scratch_power2);
			last_power = scratch_power2;
		}else{
			multiply_matrices(NR, NR, NR, scratch_power2, A, scratch_power1);
			last_power = scratch_power1;
		}
		for(long i=0; i<NR*NR; ++i) sum_of_powers[i] += abs(last_power[i]);
		if(find_first_occurrence(sum_of_powers, 0.0)<0) return (p+1);
	}
	return (NR+1); // this may be returned e.g. if the matrix is reducible
}



// calculate polynomials C_p=A^p/p! of some quadratic matrix A and for p=0,..,NP-1
// The number of polynomials NP is determined based on the requested accuracy, i.e. such that ||exp(A)-sum_{p=0}^{NP-1}C_p||<epsilon
// Optionally, a scalar scaling factor T may be provided, so that T*A is considered in all calculations instead of A.
// This function is meant to be used in conjunction with the function get_matrix_exponential_using_polynomials(..).
void calculate_matrix_polynomials(	const long					NR,					// (INPUT) number of rows & columns of the matrix A
									const std::vector<double>	&A,					// (INPUT) 2D array of size NR x NR, in row-major format
									const double				T,					// (INPUT) scalar scaling factor for input matrix (i.e. use T*A instead of A in all calculations). Set to 1.0 for no rescaling.
									const double				epsilon,			// (INPUT) norm threshold for calculated polynomials C_p=A^p/p!, i.e. stop calculating polynomials as soon as ||exp(A)-sum_{p=0}^{NP-1}C_p||<epsilon. Norm refers to the Hilbert-Schmidt L2 norm.
									const long					NPmin,				// (INPUT) minimum number of polynomials to calculate if possible (including A^0), regardless of the pursued accuracy epsilon. For sparse Markov transition matrix it is recommended to set this to NR+1, so that the matrix exponential does not contain zeros that it shouldn't contain (assuming A is irreducible). The presence of false zeros in exp(A) can mess up ancestral state reconstruction algorithms.
									const long					NPmax,				// (INPUT) maximum possible number of polynomials to calculate (including A^0), regardless of the pursued accuracy epsilon. Used as safety vault, but may break the guaranteed accuracy.
									std::vector<double>			&polynomials,		// (OUTPUT) array of size NP x NR x NR, containing the polynomials C_0,C_1,..,C_{NP-1} in layer-row-major format, i.e. such that polynomials[p*NR*NR + r*NR + c] is the (r,c)-th element of the p-th polynomial
									std::vector<double>			&polynomial_norms,	// (OUTPUT) Hilbert-Schmidt (L2) norms ||C_p||_2 of the polynomials
									long						&NP){				// (OUTPUT) number of polynomials calculated
	const double base_norm = abs(T)*get_matrix_norm_L2(NR,&A[0]);
	
	// set first polynomial to identity matrix
	get_identity_matrix(NR,polynomials);
	polynomial_norms.assign(1, get_matrix_norm_L2(NR,&polynomials[0]));
	NP = 1;
	double suzuki_error = base_norm*exp(base_norm);
	for(long p=1; p<NPmax; ++p){
		NP = p+1;
		// calculate polynomial C_p, resize storage array if needed
		polynomials.resize(NP*NR*NR,0);
		for(long r=0; r<NR; ++r){
			for(long c=0; c<NR; ++c){
				const long pentry = p*NR*NR + r*NR + c;
				polynomials[pentry] = 0;
				for(long k=0; k<NR; ++k){
					polynomials[pentry] += polynomials[(p-1)*NR*NR + r*NR + k] * (T/p) * A[k*NR + c];
				}
			}
		}
		// calculate norm
		polynomial_norms.resize(NP);
		polynomial_norms[p] = get_matrix_norm_L2(NR, &polynomials[p*NR*NR]);
		// check accuracy
		suzuki_error *= base_norm/NP; // suzuki_error = ||A||^NP * exp(||A||)/(NP!)
		if(p>=NPmin-1){
			const double R = base_norm/NP; // it is guaranteed that the norm of any polynomial p>=NP is not greater than polynomial_norms[NP-1] * R^(p-NP+1)
			if(R>=0.999) continue; // it is guaranteed that eventually R will fall below 1 (i.e. for sufficiently large NP)
			const double max_global_error = polynomial_norms[NP-1] * R/(1-R); // it is guaranteed that sum_{p=NP}^INF polynomial_norm[p] is less than max_global_error
			if(min(suzuki_error,max_global_error)<epsilon) break;
		}
	}
}




// calculate exp(tau*A) for some quadratic matrix A and some scalar tau
// provided that the polynomials C0=A^0/0!, C1=A^1/1!, C2=A^2/2!, ... have been pre-computed (up to sufficient power)
// these polynomials can be calculated e.g. using the function calculate_matrix_polynomials(..)
// For example, if the polynomials were calculated using calculate_matrix_polynomials(T*A,epsilon), 
//		then get_matrix_exponential_using_polynomials(tau,epsilon) will yield exp(tau*T*A) with an error of less than epsilon (in terms of the Hilbert-Schmidt L2-norm as well as in terms of the entrywise maximum norm).
// It is recommended to only call this function with |tau|<=1 and prepare the polynomials for some rescaled T*A such that tau*T covers the original desired range of values.
// This way the accuracy threshold (epsilon) can be guaranteed.
// Note that in principle you can provide any epsilon here (i.e. regardless of the epsilon used to calculate the polynomials), but then there may not be enough polynomials available to achieve this accuracy.
void get_matrix_exponential_using_polynomials(	const long 					NR,					// (INPUT) number of rows & columns of the matrix to be exponentiated
												const long					NP,					// (INPUT) number of available pre-computed polynomials of the matrix				
												const std::vector<double> 	&polynomials,		// (INPUT) array of size NP x NR x NR, containing the pre-computed polynomials of matrix: Cp:=A^p/p! for p=0,..,NP-1. Polynomials are stored in layer-row-major format, polynomials[p*NR*NR + r*NR + c] is (r,c)-th-element of A^p/p!
												const std::vector<double>	&polynomial_norms,	// (INPUT) array of size NP, containing the Hilbert-Schmidt L2 norm of each polynomial Cp, ||Cp||_2. Used to reduce the number of incorporated polynomials to the only the necessary number (for optimization reasons).
												const double 				tau,				// (INPUT) scaling factor in exponent
												const double				epsilon,			// (INPUT) maximum allowed error in terms of the L2 norm. Inclusion of polynomials stops as soon as the error (per component) can be guaranteed to be below epsilon.
												const long					NPmin,				// (INPUT) minimum number of polynomials to include if possible (including A^0), regardless of the pursued accuracy epsilon. For sparse Markov transition matrix it is recommended to set this to NR+1, so that the matrix exponential does not contain zeros that it shouldn't contain (assuming A is irreducible). The presence of false zeros in exp(A) can mess up ancestral state reconstruction algorithms.
												std::vector<double>			&exponential){		// (OUTPUT) array of size NR x NR, containing the exponentiated matrix exp(tau*A), in row-major format.
	const double base_norm = abs(tau)*polynomial_norms[1];

	exponential.assign(NR*NR,0);
	double ptau = 1.0;
	double suzuki_error = base_norm*exp(base_norm);
	for(long p=0; p<NP; ++p, ptau*=tau){
		for(long r=0; r<NR; ++r){
			for(long c=0; c<NR; ++c){
				exponential[r*NR + c] += ptau * polynomials[p*NR*NR + r*NR + c];
			}
		}

		// check accuracy
		suzuki_error *= base_norm/(p+1.0); // suzuki_error = ||A||^NP * exp(||A||)/(NP!)
		if(p>=NPmin-1){
			const double R = base_norm/(p+1.0);
			if(R>=0.999) continue; // it is guaranteed that eventually R will fall below 1 (i.e. for sufficiently large p)
			const double max_global_error = abs(ptau)*polynomial_norms[p] * R/(1-R); // it is guaranteed that sum_{k=p+1}^INF ||tau^k*polynomial[k]|| is less than max_global_error
			if(min(suzuki_error,max_global_error)<epsilon) break;
		}
	}
}




// Calculate matrix polynomials similarly to above, but after applying a "balancing transformation" A-->D^{-1}*A*D with some appropriately chosen diagonal matrix D ("balances matrix")
// The matrix D is chosen so that the column L2-norms in D^{-1}*A*D are roughly equal to its row L2-norms
// The returned polynomials are then actually the polynomials of D^{-1}*(T*A)*D
// The error threshold epsilon is automatically adjusted internally to account for the transformation D
// This function is meant to be used in conjunction with the function get_matrix_exponential_using_balanced_polynomials(..)
// For an example of its use see the routine exponentiate_matrix_for_multiple_scalings_CPP(..) below
void calculate_balanced_matrix_polynomials(	const long			NR,					// (INPUT) number of rows & columns of the matrix A
											std::vector<double>	A,					// (INPUT) 2D array of size NR x NR, in row-major format
											const double		T,					// (INPUT) scalar scaling factor for input matrix (i.e. use T*A instead of A in all calculations). Set to 1.0 for no rescaling.
											double				epsilon,			// (INPUT) norm threshold for calculated polynomials C_p=A^p/p!, i.e. stop calculating polynomials as soon as ||exp(A)-sum_{p=0}^{NP-1}C_p||<epsilon. Norm refers to the Hilbert-Schmidt L2 norm.
											const long			NPmin,				// (INPUT) minimum number of polynomials to calculate if possible (including A^0), regardless of the pursued accuracy epsilon. For sparse Markov transition matrix it is recommended to set this to NR+1, so that the matrix exponential does not contain zeros that it shouldn't contain (assuming A is irreducible). The presence of false zeros in exp(A) can mess up ancestral state reconstruction algorithms.
											const long			NPmax,				// (INPUT) maximum possible number of polynomials to calculate, regardless of the pursued accuracy epsilon. Used as safety vault, but may break the guaranteed accuracy.
											std::vector<double>	&polynomials,		// (OUTPUT) array of size NP x NR x NR, containing the polynomials C_0,C_1,..,C_{NP-1} in layer-row-major format, i.e. such that polynomials[p*NR*NR + r*NR + c] is the (r,c)-th element of the p-th polynomial
											std::vector<double>	&polynomial_norms,	// (OUTPUT) Hilbert-Schmidt (L2) norms ||C_p||_2 of the polynomials
											long				&NP,				// (OUTPUT) number of polynomials calculated
											std::vector<double>	&balances,			// (OUTPUT) 1D array of size NR, storing the diagonal elements of a calculated diagonal matrix D that was applied for balancing A prior to exponentiation. That is, the returned polynomials will be those for D^{-1}*A*D instead of A.	
											long				&scaling_power){	// (OUTPUT) base-2 scaling power used to rescale the matrix prior to calculating the polynomials (according to the scaling-and-squaring method). This rescaling will need to be corrected for (via repeated squaring) when calculating exponentials with the polynomials
	multiply_array_with_scalar(A, T);	
	balance_matrix_with_diagonal_transformation(NR, A, balances);
	
	// adjust error threshold (make more strict) to account for balancing (A -> D^{-1}*A*D)
	// in the worst-case scenario the error can be emplified by a factor ||D||_2 * ||D^{-1}||_2
	epsilon /= get_norm_L2_of_vector(balances) * get_norm_L2_of_inverted_vector(balances);
	
	double rescaled_norm;
	get_scaling_power_for_matrix_exponentiation(NR, A, scaling_power, rescaled_norm);
	
	// adjust error threshold (make more strict) to account for rescaling
	// in the worst-case scenario, the error can be amplified by (roughly) 2^scaling_power * ||exp(rescaled_A)||_2 <= 2^scaling_power * exp(rescaled_norm)
	epsilon /= pow(2.0,scaling_power) * exp(rescaled_norm);

	calculate_matrix_polynomials(	NR,
									A,
									1.0,
									epsilon,
									NPmin,
									NPmax,
									polynomials,
									polynomial_norms,
									NP);
}




// This function is meant to be used in conjunction with the function calculate_balanced_matrix_polynomials(..)
// For an example of its use see the routine exponentiate_matrix_for_multiple_scalings_CPP(..) below
void get_matrix_exponential_using_balanced_polynomials(	const long 					NR,					// (INPUT) number of rows & columns of the matrix to be exponentiated
														const long					NP,					// (INPUT) number of available pre-computed polynomials of the matrix				
														const std::vector<double> 	&polynomials,		// (INPUT) array of size NP x NR x NR, containing the pre-computed polynomials of matrix: Cp:=A^p/p! for p=0,..,NP-1. Polynomials are stored in layer-row-major format, polynomials[p*NR*NR + r*NR + c] is (r,c)-th-element of A^p/p!
														const std::vector<double>	&polynomial_norms,	// (INPUT) array of size NP, containing the Hilbert-Schmidt L2 norm of each polynomial Cp, ||Cp||_2. Used to reduce the number of incorporated polynomials to the only the necessary number (for optimization reasons).
														double		 				tau,				// (INPUT) scaling factor in exponent
														double						epsilon,			// (INPUT) maximum allowed error in terms of the L2 norm. Inclusion of polynomials stops as soon as the error (per component) can be guaranteed to be below epsilon.
														const long					NPmin,				// (INPUT) minimum number of polynomials to include if possible (including A^0), regardless of the pursued accuracy epsilon. For sparse Markov transition matrix it is recommended to set this to NR+1, so that the matrix exponential does not contain zeros that it shouldn't contain (assuming A is irreducible). The presence of false zeros in exp(A) can mess up ancestral state reconstruction algorithms.
														const std::vector<double>	&balances,			// (INPUT) 1D array of size NR, storing the diagonal elements of a diagonal matrix D that was applied for balancing A prior to polynomial calculation. This transformation will now be reversed after exponentiation.
														long						scaling_power,		// (INPUT) base-2 scaling power that was applied to the matrix prior to calculating the polynomials. This scaling will now be reversed after exponentiation, via repeated squaring.
														std::vector<double>			&exponential){		// (OUTPUT) array of size NR x NR, containing the exponentiated matrix exp(tau*A), in row-major format.
	
	// reverse some of the scaling beforehand as permitted (this operation is faster than repeated squaring afterwards)
	// make sure tau stays below 1.0 and we only rescale by at most 2^scaling_power
	if((tau<1.0) && (scaling_power>0)){
		const long power_decrease = min(scaling_power,(long)floor(log(1.0/tau)/log(2.0)));
		scaling_power -= power_decrease;
		tau *= pow(2.0, power_decrease);
	}
	
	// correct error threshold to account for balancing and rescaling
	// in the worst-case scenario the error can be emplified due to balancing by a factor ||D||_2 * ||D^{-1}||_2
	// in the worst-case scenario, the error can be amplified due to scaling by (roughly) 2^scaling_power * ||exp(rescaled_A)||_2 <= 2^scaling_power * exp(rescaled_norm)
	epsilon /= (get_norm_L2_of_vector(balances) * get_norm_L2_of_inverted_vector(balances));
	epsilon /= pow(2.0,scaling_power) * exp(tau*polynomial_norms[1]);
	
	// calculate exponential of balanced & rescaled matrix
	get_matrix_exponential_using_polynomials(	NR,
												NP,
												polynomials,
												polynomial_norms,
												tau,
												epsilon,
												NPmin,
												exponential);
												
	// reverse scaling by repeated squaring (move last version between exponential[] and scratch[], since squaring can't be done in-situ).
	std::vector<double> scratch(NR*NR);
	for(long i=0; i<scaling_power; ++i){
		if((i%2)==0) get_square_matrix(NR, &exponential[0], &scratch[0]);
		else get_square_matrix(NR, &scratch[0], &exponential[0]);
	}
	if((scaling_power%2)==1) exponential = scratch;
	
	// reverse balancing in-situ
	diagonally_transform_matrix(NR, balances, true,	&exponential[0]);
}






// Calculate the exponential of a real-valued matrix A using its eigendecomposition.
// EVmatrix contains eigenvectors of A in its columns, hence inverse_EVmatrix * A * EVmatrix is a diagonal matrix D with eigenvalues of A in the diagonal
void get_matrix_exponential_using_eigendecomposition(	const long 					NR,					// (INPUT) number of rows & columns of A
														const std::vector<cdouble>	&eigenvalues,		// (INPUT) 1D array of size NR, listing the complex eigenvalues of A
														const std::vector<cdouble>	&EVmatrix,			// (INPUT) 2D array of size NR x NR, in row-major format, whose columns are the eigenvectors of A
														const std::vector<cdouble>	&inverse_EVmatrix,	// (INPUT) 2D array of size NR x NR, in row-major format, the inverse of EVmatrix
														double		 				tau,				// (INPUT) scaling factor in exponent, i.e. calculate exp(tau*A).
														std::vector<cdouble> 		&scratch,			// (SCRATCH SPACE)
														vector<double>				&exponential){		// (OUTPUT) 2D array of size NR x NR, containing the exponentiated matrix exp(tau*A), in row-major format.
	// Note that inverse_EVmatrix * A * EVmatrix is a diagonal matrix D with entries eigenvalues in the diagonal
	// Hence, first exponentiate D and then reverse diagonalization
	std::vector<cdouble> exponentiated_eigenvalues(NR);
	for(long i=0; i<NR; ++i) exponentiated_eigenvalues[i] = exp(tau * eigenvalues[i]);

	// reverse diagonalization
	scratch.resize(NR*NR);
	exponential.resize(NR*NR);
	multiply_matrix_with_diagonal_matrix(NR, NR, EVmatrix,exponentiated_eigenvalues, scratch);
	multiply_matrices(NR, NR, NR, scratch, inverse_EVmatrix, exponential);

}





// calculate the exponential exp(A) of some quadratic matrix A
// Returns the exponential as a flattened array of size NR x NR in row-major format, i.e. with exponential[r*NR + c] being the (r,c)-th entry of exp(A)
void exponentiate_matrix(	const long	 				NR,				// (INPUT) number of rows & columns of the matrix A
							const std::vector<double>	&A,				// (INPUT) 2D array of size NR x NR, in row-major format
							const double				epsilon,		// (INPUT) Absolute error threashold for the approximation of exp(T*A), in terms of the Hilbert-Schmidt L2 norm.
							const long					NPmin,			// (INPUT) minimum number of polynomials to include (including A^0), regardless of the pursued accuracy epsilon. For sparse Markov transition matrix it is recommended to set this to NR+1, so that the matrix exponential does not contain zeros that it shouldn't contain (assuming A is irreducible). The presence of false zeros in exp(A) can mess up ancestral state reconstruction algorithms.
							const long					NPmax,			// (INPUT) maximum possible number of polynomials to calculate (RAM required will scale linearly with NPmax * NR^2). Used as safety vault, but may break the guaranteed accuracy.
							const bool					enforce_probability_matrix, // (INPUT) if true, then the sum along each column is enforced to be 1
							std::vector<double>			&exponential){	// (OUTPUT) exponentiated matrix exp(A); 2D array of size NR x NR, in row-major format	
	
	// prepare data structures for exponentiations of transition matrix
	std::vector<double> polynomials, polynomial_norms, balances;
	long Npolynomials, scaling_power;
	calculate_balanced_matrix_polynomials(	NR,
											std::vector<double>(A.begin(), A.end()),
											1.0,
											epsilon,
											NPmin,
											NPmax,
											polynomials,
											polynomial_norms,
											Npolynomials,
											balances,
											scaling_power);
																
	// calculate exponential using the pre-computed polynomials
	get_matrix_exponential_using_balanced_polynomials(	NR,
														Npolynomials,
														polynomials,
														polynomial_norms,
														1.0,
														epsilon,
														NPmin,
														balances,
														scaling_power,
														exponential);
	
	if(enforce_probability_matrix){
		for(long c=0; c<NR; ++c){
			double col_sum = 0;
			for(long r=0; r<NR; ++r){
				exponential[r*NR + c] = max(0.0, exponential[r*NR + c]);
				if(r!=c) col_sum += exponential[r*NR + c];
			}
			exponential[c*NR + c] = 1 - col_sum;
		}
	}
}


// calculate the exponential exp(T*A) for some quadratic matrix A and for a large number of scalar scaling factors T
// Returns the exponentials (one per scaling) as a flattened array of size NS x NR x NR in layer-row-major format, i.e. with exponentials[s*NR*NR + r*NR + c] being the (r,c)-th entry of exp(scalings[s]*A)
// This routine is most efficient when T is very large.
// The approach resembles the one proposed by [Al-Mohy and Higham (2011). Computing the action of the matrix exponential, with an application to exponential integrators. SIAM Journal on Scientific Computing 33:488-511]
// [[Rcpp::export]]
NumericVector exponentiate_matrix_for_multiple_scalings_CPP(const long	 				NR,			// (INPUT) number of rows & columns of the matrix A
															const std::vector<double>	&A,			// (INPUT) 2D array of size NR x NR, in row-major format
															const std::vector<double>	&scalings,	// (INPUT) 1D numeric array of size NS containing scalar scaling factors
															const double				epsilon,	// (INPUT) Absolute error threashold for the approximation of exp(T*A), in terms of the Hilbert-Schmidt L2 norm.
															const long					NPmin,		// (INPUT) minimum number of polynomials to include (including A^0), regardless of the pursued accuracy epsilon. For sparse Markov transition matrix it is recommended to set this to NR+1, so that the matrix exponential does not contain zeros that it shouldn't contain (assuming A is irreducible). The presence of false zeros in exp(A) can mess up ancestral state reconstruction algorithms.
															const long					NPmax,		// (INPUT) maximum possible number of polynomials to calculate (RAM required will scale linearly with NPmax * NR^2). Used as safety vault, but may break the guaranteed accuracy.
															const bool					enforce_probability_matrix){ // (INPUT) if true, then the sum along each column is enforced to be 1
	const double max_scaling = array_max(scalings);
	const long NS = scalings.size();
	
	// prepare data structures for exponentiations of transition matrix
	std::vector<double> polynomials, polynomial_norms, balances;
	long Npolynomials, scaling_power;
	calculate_balanced_matrix_polynomials(	NR,
											std::vector<double>(A.begin(), A.end()),
											max_scaling,
											epsilon,
											NPmin,
											NPmax,
											polynomials,
											polynomial_norms,
											Npolynomials,
											balances,
											scaling_power);
																
	// calculate exponentials using the pre-computed polynomials
	dvector exponentials(NS*NR*NR), scratch_exponential;
	for(long s=0; s<NS; ++s){
		get_matrix_exponential_using_balanced_polynomials(	NR,
															Npolynomials,
															polynomials,
															polynomial_norms,
															scalings[s],
															epsilon,
															NPmin,
															balances,
															scaling_power,
															scratch_exponential);
		for(long r=0; r<NR; ++r){
			for(long c=0; c<NR; ++c){
				exponentials[s*NR*NR + r*NR + c] = scratch_exponential[r*NR + c];
			}
		}
		if(enforce_probability_matrix){
			for(long c=0; c<NR; ++c){
				double col_sum = 0;
				for(long r=0; r<NR; ++r){
					exponentials[s*NR*NR + r*NR + c] = max(0.0, exponentials[s*NR*NR + r*NR + c]);
					if(r!=c) col_sum += exponentials[s*NR*NR + r*NR + c];
				}
				exponentials[s*NR*NR + c*NR + c] = 1 - col_sum;
			}
		}
	}
	return Rcpp::wrap(exponentials);
}



// wrapper class for preparing exponentation of a matrix A and exponentiating A using various scalar factors
class matrix_exponentiator{
private:
	// variables for exponentiation using matrix polynomials
	long 				NP;					// number of polynomials available for calculating the exponential of the input matrix
	std::vector<double> polynomials;		// array of size NP x NR x NR, containing the pre-computed polynomials of matrix: Cp:=A^p/p! for p=0,..,NP-1. Polynomials are stored in layer-row-major format, polynomials[p*NR*NR + r*NR + c] is (r,c)-th-element of A^p/p!
	std::vector<double>	polynomial_norms;	// array of size NP, containing the Hilbert-Schmidt L2 norm of each polynomial Cp, ||Cp||_2. Used to reduce the number of incorporated polynomials to the only the necessary number (for optimization reasons).
	long				NPmin;
	double				epsilon;
	bool				balanced;			// whether the input matrix was balanced prior to calculating the polynomials
	std::vector<double>	balances;			// 1D array of size NR, storing the diagonal elements of a diagonal matrix D that was applied for balancing A prior to polynomial calculation. This transformation will be reversed after exponentiation. Only relevant if balanced==true.
	long				scaling_power;		// base-2 scaling power that was applied to the matrix prior to calculating the polynomials. This scaling will be reversed after exponentiation, via repeated squaring. Only relevant if balanced==true.
	
	// variables for exponentiation using eigendecomposition
	bool					use_eigendecomposition;	// whether exponentiation should be performed using eigendecomposition
	mutable std::vector<cdouble>	exponentiation_scratch;
	std::vector<cdouble> 	eigenvalues;
	std::vector<cdouble> 	EVmatrix;
	std::vector<cdouble> 	inverse_EVmatrix;
public:
	bool				initialized;
	long 				NR;					// number of rows & columns in the input matrix
	
	// Constructors
	matrix_exponentiator(){ initialized = false; }
	matrix_exponentiator(	const long 			NR_,
							std::vector<double>	A,					// (INPUT) 2D array of size NR x NR, in row-major format
							const double		rescaling,			// (INPUT) optional scalar scaling factor for input matrix (i.e. use rescaling*A instead of A in all calculations). Set to 1.0 for no rescaling.
							double				epsilon_,			// (INPUT) norm threshold for calculated polynomials C_p=A^p/p!, i.e. stop calculating polynomials as soon as ||exp(A)-sum_{p=0}^{NP-1}C_p||<epsilon. Norm refers to the Hilbert-Schmidt L2 norm.
							const long			NPmin_,				// (INPUT) minimum number of polynomials to calculate if possible (including A^0), regardless of the pursued accuracy epsilon. For sparse Markov transition matrix it is recommended to set this to NR+1, so that the matrix exponential does not contain zeros that it shouldn't contain (assuming A is irreducible). The presence of false zeros in exp(A) can mess up ancestral state reconstruction algorithms.
							const long			NPmax_,				// (INPUT) maximum possible number of polynomials to calculate, regardless of the pursued accuracy epsilon. Used as safety vault, but may break the guaranteed accuracy.
							bool				balanced_){
		initialize(NR_, A, rescaling, epsilon_, NPmin_, NPmax_, balanced_);
	}
	
	matrix_exponentiator(	const long 					NR_,
							const std::vector<cdouble>	&eigenvalues_,
							const std::vector<cdouble>	&EVmatrix_,
							const std::vector<cdouble>	&inverse_EVmatrix_,
							const double				rescaling){			// (INPUT) optional scalar scaling factor for input matrix (i.e. use rescaling*A instead of A in all calculations). Set to 1.0 for no rescaling.
		initialize(NR_, eigenvalues_, EVmatrix_, inverse_EVmatrix_, rescaling);
	}
	
	
	// prepare exponentiation of matrix A, by pre-calculating polynomials of A
	// if balanced==true then preps include balancing the matrix. This may be needed for some weird matrixes
	void initialize(const long 			NR_,
					std::vector<double>	A,					// (INPUT) 2D array of size NR x NR, in row-major format
					const double		rescaling,			// (INPUT) optional scalar scaling factor for input matrix (i.e. use rescaling*A instead of A in all calculations). Set to 1.0 for no rescaling.
					double				epsilon_,			// (INPUT) norm threshold for calculated polynomials C_p=A^p/p!, i.e. stop calculating polynomials as soon as ||exp(A)-sum_{p=0}^{NP-1}C_p||<epsilon. Norm refers to the Hilbert-Schmidt L2 norm.
					const long			NPmin_,				// (INPUT) minimum number of polynomials to calculate if possible (including A^0), regardless of the pursued accuracy epsilon. For sparse Markov transition matrix it is recommended to set this to NR+1, so that the matrix exponential does not contain zeros that it shouldn't contain (assuming A is irreducible). The presence of false zeros in exp(A) can mess up ancestral state reconstruction algorithms.
					const long			NPmax,				// (INPUT) maximum possible number of polynomials to calculate, regardless of the pursued accuracy epsilon. Used as safety vault, but may break the guaranteed accuracy.
					bool				balanced_){
		balanced 	= balanced_;
		NR 			= NR_;
		NPmin 		= NPmin_;
		epsilon 	= epsilon_;
		initialized	= true;
		use_eigendecomposition = false;
		if(balanced){
			calculate_balanced_matrix_polynomials(	NR,
													A,
													rescaling,
													epsilon,
													NPmin,
													NPmax,
													polynomials,
													polynomial_norms,
													NP,
													balances,
													scaling_power);
		}else{
			calculate_matrix_polynomials(	NR,
											A,
											rescaling,
											epsilon,
											NPmin,
											NPmax,
											polynomials,
											polynomial_norms,
											NP);
		}
	}
	
	// prepare exponentiation of matrix A based on eigendecomposition
	void initialize(const long 					NR_,
					const std::vector<cdouble>	&eigenvalues_,
					const std::vector<cdouble>	&EVmatrix_,
					const std::vector<cdouble>	&inverse_EVmatrix_,
					const double				rescaling){			// (INPUT) optional scalar scaling factor for input matrix (i.e. use rescaling*A instead of A in all calculations). Set to 1.0 for no rescaling.
		NR 						= NR_;
		initialized				= true;
		use_eigendecomposition 	= true;
		eigenvalues 			= eigenvalues_;
		EVmatrix 				= EVmatrix_;
		inverse_EVmatrix 		= inverse_EVmatrix_;
		if(rescaling!=1.0){
			for(long r=0; r<eigenvalues.size(); ++r) eigenvalues[r] *= rescaling;
		}
	}
	
	
	// calculate exp(tau*A)
	void get_exponential(	double				tau,					// (INPUT) scaling factor in exponent
							std::vector<double>	&exponential) const{	// (OUTPUT) array of size NR x NR, containing the exponentiated matrix exp(tau*A), in row-major format.
		if(use_eigendecomposition){
			get_matrix_exponential_using_eigendecomposition(NR, eigenvalues, EVmatrix, inverse_EVmatrix, tau, exponentiation_scratch, exponential);
		}else if(balanced){
			get_matrix_exponential_using_balanced_polynomials(NR, NP, polynomials, polynomial_norms, tau, epsilon, NPmin, balances, scaling_power, exponential);
		}else{
			get_matrix_exponential_using_polynomials(NR, NP, polynomials, polynomial_norms, tau, epsilon, NPmin, exponential);
		}
	}
};



// get an approximation for the expression exp(scaling*A)*X, where A is a quadratic matrix of size NR x NR, and X is a vector or matrix of size NR x NC
// the approximation is obtained using the expression sum_n=0^order (scaling*A)^n/n! * X
void apply_approximate_matrix_exponential(	const long					NR, 		// (INPUT) number of rows and columns in A, also equal to the number of rows in X
											const long					NC, 		// (INPUT) number of columns in X
											const std::vector<double> 	&A, 		// (INPUT) 2D matrix of size NR x NR, in row-major format. The matrix to be exponentially applied.
											const double				&scaling, 	// (INPUT) an optional multiplicative factor for A, i.e. compute exp(scaling*A) instead of exp(A)*X. Set this to 1 for no scaling.
											const std::vector<double>	&X, 		// (INPUT) the vector or matrix to which exp(A) should be applied
											const long					&order,		// (INPUT) how many polynomials to include in the expansion of exp(A), i.e. 1 + A + A^2/2! + A^3/3! ... A^order/order!
											std::vector<double>			&scratch1,	// (SCRATCH) scratch space needed for internal computation
											std::vector<double>			&scratch2,	// (SCRATCH) scratch space needed for internal computation
											std::vector<double>			&Y){		// (OUTPUT) the resulting vector or matrix of size NR x NC, an approximation to Y=exp(scaling*A)*X
	long n, r, c, k;
	scratch1 = X;
	scratch2.resize(NR*NC);
	Y = X;
	std::vector<double> *source, *target;
	for(n=1; n<=order; ++n){
		source = (n%2==1 ? &scratch1 : &scratch2);
		target = (n%2==0 ? &scratch1 : &scratch2);
		// compute the polynomial term of order n, i.e. ((scaling*A)^n/n!)*X
		// store results in target matrix
		target->assign(NR*NC,0);
		for(r=0; r<NR; ++r){
			for(c=0; c<NC; ++c){
				// compute target := (scaling*A)/n * source
				for(k=0; k<NR; ++k){
					(*target)[r*NC+c] += (scaling/n)*A[r*NR+k]*(*source)[k*NC+c];
				}
			}
		}
		// add target to Y
		for(r=0; r<NR; ++r){
			for(c=0; c<NC; ++c){
				Y[r*NC+c] += (*target)[r*NC+c];
			}
		}
	}
}




// get an approximation for the expression [exp(scaling*A) - Id]*A^{-1}*X, where A is a quadratic matrix of size NR x NR, and X is a vector or matrix of size NR x NC
// the approximation is obtained using the expression sum_n=0^order (scaling/(n+1)) * (scaling*A)^n/n! * X
// This routine is mainly used for the Rosenbrock-Euler ODE solver
void apply_approximate_RosenbrockEuler_exponential(	const long					NR, 		// (INPUT) number of rows and columns in A, also equal to the number of rows in X
													const long					NC, 		// (INPUT) number of columns in X
													const std::vector<double> 	&A, 		// (INPUT) 2D matrix of size NR x NR, in row-major format. The matrix to be exponentially applied.
													const double				&scaling, 	// (INPUT) an optional multiplicative factor for A, i.e. compute exp(scaling*A) instead of exp(A)*X. Set this to 1 for no scaling.
													const std::vector<double>	&X, 		// (INPUT) the vector or matrix to which exp(A) should be applied
													const long					&order,		// (INPUT) how many polynomials to include in the expansion of exp(A), i.e. 1 + A + A^2/2! + A^3/3! ... A^order/order!
													std::vector<double>			&scratch1,	// (SCRATCH) scratch space needed for internal computation
													std::vector<double>			&scratch2,	// (SCRATCH) scratch space needed for internal computation
													std::vector<double>			&Y){		// (OUTPUT) the resulting vector or matrix of size NR x NC, an approximation to Y=exp(scaling*A)*X
	long n, r, c, k;
	scratch1 = X;
	scratch2.resize(NR*NC);
	
	// zeroth-order term
	Y.resize(X.size());
	for(k=0; k<Y.size(); ++k) Y[k] = scaling*X[k];
	
	double REfactor;
	std::vector<double> *source, *target;
	for(n=1; n<=order; ++n){
		source = (n%2==1 ? &scratch1 : &scratch2);
		target = (n%2==0 ? &scratch1 : &scratch2);
		// compute the polynomial term of order n, i.e. ((scaling*A)^n/n!)*X
		// store results in target matrix
		target->assign(NR*NC,0);
		for(r=0; r<NR; ++r){
			for(c=0; c<NC; ++c){
				// compute target := (scaling*A)/n * source
				for(k=0; k<NR; ++k){
					(*target)[r*NC+c] += (scaling/n)*A[r*NR+k]*(*source)[k*NC+c];
				}
			}
		}
		// add target to Y
		REfactor =  (scaling/(n+1.0)); // modifying factor to get Rosenbrock-Euler form, instead of classical exponential
		for(r=0; r<NR; ++r){
			for(c=0; c<NC; ++c){
				Y[r*NC+c] += REfactor * (*target)[r*NC+c];
			}
		}
	}
}



inline double dot_product(const dvector &X, const dvector &Y){
	double S = 0;
	for(long i=0; i<X.size(); ++i) S += X[i]*Y[i];
	return S;
}



// estimate the dominant eigenvalue (i.e. with largest modulus) and corresponding eigenvector of a square matrix, using power iteration
// returns true if convergence was successful after two trials
bool get_dominant_eigenvalue(	const long		N,				// (INPUT) the number of rows & columns in A
								const dvector 	&A,				// (INPUT) square 2D matrix of size N x N, in row-major format
								const long		max_iterations, // (INPUT) maximum number of iterations (matrix multiplications) to perform
								const double	tolerance,		// (INPUT) relative error tolerance for detecting convergence of X (in terms of direction). This is the sine of the angle between two successive iterations of X; if the angle is small, it means the direction of X changes little. A tolerance = 1e-3 is usually sufficient.
								dvector			&X,				// (OUTPUT) an estimate for the dominant eigenvector, normalized to norm 1. This vector is only uniquely defined in terms of direction and norm, but not sign.
								double			&lambda){		// (OUTPUT) an estimate for the dominant eigenvalue
	double XAX, error, best_lambda=0, best_error=INFTY_D;
	dvector AX, Xnew(N), best_X;
	X.resize(N);
	for(int trial=0; trial<min(3L,N); ++trial){ // perform multiple trials, in case the first happens to start with a "bad" direction X
		if(trial==0){
			// generate a random start vector X
			for(long i=0; i<N; ++i) X[i] = R::runif(0.0,1.0);
		}else{
			// pick a new sart vector X that is perpendicular to the previous X
			// step 1: find largest (in magnitude) element in X
			long largest = 0;
			for(long i=0; i<N; ++i){
				if(abs(X[i])>abs(X[largest])) largest = i;
			}
			// step 2: generate N-1 random entries for Xnew, except at the position [largest]
			for(long i=0; i<N; ++i) Xnew[i] = (i==largest ? 0.0 : R::runif(0.0,1.0));
			// step 3: choose Xnew[largest] such that the dot-product X^T*Xnew is zero
			Xnew[largest] = (-1/X[largest]) * dot_product(X,Xnew);
			// step 4: adopt Xnew, and repeat power iteration below
			X = Xnew;
		}

		// normalize X
		double S = sqrt(dot_product(X,X));
		for(long i=0; i<N; ++i) X[i] /= S;
	
		// iterate over X(i+1) = A*X(i)/lambda(i), where lambda(i):=X(i)^T*A*X(i)/||X(i)||_2
		for(long n=0; n<max_iterations; ++n){
			// compute A*X
			AX.assign(N,0);
			for(long r=0; r<N; ++r){
				for(long c=0; c<N; ++c){
					AX[r] += A[r*N+c]*X[c];	
				}
			}
			// compute X^TX
			S = dot_product(X,X);
			// compute X^T*(A*X)
			XAX = 0;
			for(long i=0; i<N; ++i) XAX += X[i]*AX[i];
			if(XAX==0){
				// landed on the zero-eigenvalue, so we're stuck (converged) on this eigenspace
				lambda = 0;
				error  = 0;
				break;
			}
			// compute lambda = X^T*A*X/(X^T*X), the current estimate of the dominant eigenvalue
			lambda = XAX/S;
			// compute next iteration, Xnext = A*X/lambda
			for(long i=0; i<N; ++i) Xnew[i] = AX[i]/lambda;
			// normalize Xnew, just in case
			S = sqrt(dot_product(Xnew,Xnew));
			for(long i=0; i<N; ++i) Xnew[i] /= S;
			// compute error := sin(angle between X and Xnew)
			// an error << 1 implies convergence
			error = sqrt(1 - min(1.0,SQ(dot_product(X,Xnew)))); // this formula relies on the fact that X and Xnew are normalized
			// adopt new X
			X = Xnew;
			if(error<tolerance) break; // reached convergence within relative tolerance
		}
		if((trial==0) || (abs(lambda)>abs(best_lambda))){
			// record the outcome of this trial, regardless of convergence, if lambda is better than the previous ones
			best_lambda = lambda;
			best_error  = error;
			best_X		= X;
		}
	}
	lambda  = best_lambda;
	X		= best_X;
	return (best_error<tolerance);
}




// find the smallest eigenvalue (by magnitude) of a square matrix A
// this method first inverts A using QR decomposition and then uses the power method to compute the dominant eigenvalue of A
// if the matrix is non-invertible, its weakest eigenvalue is estimated to be 0
bool get_weakest_eigenvalue(const long		N,				// (INPUT) the number of rows & columns in A
							const dvector 	&A,				// (INPUT) square 2D matrix of size N x N, in row-major format
							const long		max_iterations, // (INPUT) maximum number of iterations (matrix multiplications) to perform during the power method
							const double	tolerance,		// (INPUT) relative error tolerance for detecting convergence during the power method
							double			&lambda){		// (OUTPUT) smallest eigenvalue (by magnitude)
	long rank;
	dvector QRA,Ainv,X;
	QR_matrix_inverse(N,A,QRA,Ainv,rank);
	if(rank<N){
		lambda = 0;
		return true;
	}
	// get dominant eigenvalue of Ainv; this will be the inverse of the weakest eigenvalue of A
	const bool converged = get_dominant_eigenvalue(N,Ainv,max_iterations, tolerance, X, lambda);
	lambda = 1/lambda;
	return converged;
}





// balance a square real matrix prior to eigendecomposition
void EIG_balance_matrix(const long 	N, 			// (INPUT) the number of rows & columns in A
						double 		A[], 		// (INPUT/OUTPUT) 2D matrix of size N x N, in column-major format. On input, the matrix to be balanced. On output, the matrix is modified (balanced) in situ.
						long 		&low, 		// (OUTPUT) information on the location of zeros in A
						long 		&igh,		// (OUTPUT) information on the location of zeros in A
						double 		scale[]){	// (OUTPUT) technical information on the permutations and scalings used. Must be preallocated to size N
	double b2, c, f, g, r, radix, s, t;
	bool done, noconv, swap;
	long i, j, k, l, m;

	radix 	= 16.0;
	b2 		= radix * radix;
	j 		= -1;
	m 		= -1;
	k 		= 0;
	l 		= N - 1;

	done = false;
	while(!done){
		for(j = l; 0 <= j; j--){
			swap = true;
			for(i = 0; i <= l; i++){
				if(i != j){
					if(A[j+i*N] != 0.0){
						swap = false;
						break;
					}
				}
			}
			if(swap){
				m = l;
				scale[m] = (double) j;

				if(j != m){
					for(i=0; i <= l; i++){
						t        = A[i+j*N];
						A[i+j*N] = A[i+m*N];
						A[i+m*N] = t;
					}
					for(i=k; i < N; i++){
						t        = A[j+i*N];
						A[j+i*N] = A[m+i*N];
						A[m+i*N] = t;
					}
				}
				if(l == 0){
					low = k;
					igh = l;
					return;
				}

				l = l - 1;
				if(l < 0){
					done = true;
				}
				break;
			}else if(j == 0){
				done = true;
				break;
			}
		}
	}

	done = false;
	while(!done){
		for(j = k; j <= l; j++){
			swap = true;
			for(i = k; i <= l; i++){
				if(i != j){
					if(A[i+j*N] != 0.0){
						swap = false;
						break;
					}
				}
			}
			if(swap){
				m = k;
				scale[m] = (double) j;
				if(j != m){
					for(i=0; i <= l; i++){
						t        = A[i+j*N];
						A[i+j*N] = A[i+m*N];
						A[i+m*N] = t;
					}
					for(i=k; i < N; i++){
						t        = A[j+i*N];
						A[j+i*N] = A[m+i*N];
						A[m+i*N] = t;
					}
				}
				k = k + 1;
				if(l < k){
					done = true;
				}
				break;
			}else{
				if(j == l){
					done = true;
					break;
				}
			}
		}
	}

	// balance submatrix in rows K to L.
	for(i = k; i <= l; i++){
		scale[i] = 1.0;
	}

	// norm reduction.
	noconv = true;
	while(noconv){
		noconv = false;
		for(i = k; i <= l; i++){
			c = 0.0;
			r = 0.0;
			for(j = k; j <= l; j++){
				if(j != i){
					c = c + abs(A[j+i*N]);
					r = r + abs(A[i+j*N]);
				}
			}
	
			// deal with zero C or R due to underflow.
			if(c != 0.0 && r != 0.0){
				g = r / radix;
				f = 1.0;
				s = c + r;
				while(c < g){
					f = f * radix;
					c = c * b2;
				}
				g = r * radix;
				while(g <= c){
					f = f / radix;
					c = c / b2;
				}
				if(( c + r ) / f < 0.95 * s){
					g = 1.0 / f;
					scale[i] = scale[i] * f;
					noconv = true;

					for(j = k; j < N; j++){
						A[i+j*N] = A[i+j*N] * g;
					}
					for(j = 0; j <= l; j++){
						A[j+i*N] = A[j+i*N] * f;
					}
				}
			}
		}
	}

	low = k;
	igh = l;
}




void EIG_reverse_balancing(	const long 		N, 			// (INPUT) the number of rows & columns in the original matrix A
							const long 		low, 		// (INPUT) the low index, as returned by EIG_balance_matrix
							const long 		igh, 		// (INPUT) the igh index, as returned by EIG_balance_matrix
							const double 	scale[], 	// (INPUT) the scale vector of size N, as returned by EIG_balance_matrix
							const long 		M,			// (INPUT) the number of columns of Z to be back-transformed
							double 			Z[]){		// (INPUT/OUTPUT) 2D matrix of size N x M, in column-major format, containing the real & imaginary parts of eigenvectors. Upon return, these will have been back-transformed
	long i, ii, j, k;
	double t;
	if (M <= 0) return;

	if(igh != low){
		for(i = low; i <= igh; i++){
			for(j = 0; j < M; j++){
				Z[i+j*N] = Z[i+j*N] * scale[i];
			}
		}
	}

	for(ii = 0; ii < N; ii++){
		i = ii;
		if(i<low || igh<i){
			if(i < low){
				i = low - ii;
			}
			k = long(scale[i]);
			if(k != i){
				for(j=0; j < M; j++){
					t        = Z[i+j*N];
					Z[i+j*N] = Z[k+j*N];
					Z[k+j*N] = t;
				}
			}
		}
	}
}




// auxiliary function used for EIG_eigendecomposition
void EIG_accumulate_similarity(	const long 	N, 		// (INPUT) the number of rows & column in the original matrix
								const long 	low, 	// (INPUT) The low index, as returned by EIG_balance_matrix. If balancing was not performed, set low = 0, igh = N - 1.
								const long 	igh, 	// (INPUT) The igh index, as returned by EIG_balance_matrix. If balancing was not performed, set low = 0, igh = N - 1.
								double 		H[], 	// (INPUT) 2D matrix of size N x N, in column-major format, containing HELMES multiplies
								const long	ind[], 	// (INPUT) technical information on the ELMHES reduction
								double 		Z[]){	// (OUTPUT) 2D matrix of size N x N, in column-major format, containing the transformation matrix produced by ELMHES. Must be preallocated to size N*N.
	long i,j, mp;
	
	// initialize to the identity matrix
	for(i=0; i<N; ++i){
		for(long j=0; j<N; ++j){
			Z[i + j*N] = 0;	
		}
		Z[i + i*N] = 1;
	}

	if(igh < low + 2){
		return;
	}
	for(mp = igh - 1; low + 1 <= mp; mp--){
		for(i = mp + 1; i <= igh; i++){
			Z[i+mp*N] = H[i+(mp-1)*N];
		}
		i = ind[mp];

		if(i != mp){
			for(j = mp; j <= igh; j++){
				Z[mp+j*N] = Z[i+j*N];
			}
			Z[i+mp*N] = 1.0;
			for(j = mp + 1; j <= igh; j++){
				Z[i+j*N] = 0.0;
			}
		}
	}
}




// transform a real square matrix to upper Hessenberg form, e.g. for subsequent eigendecomposition
// Code adjusted from: https://people.sc.fsu.edu/~jburkardt/c_src/eispack/eispack.html
void EIG_ELMHES(const long 	N, 		// (INPUT) the number of rows & column in the original matrix
				const long 	low, 	// (INPUT) The low index, as returned by EIG_balance_matrix. If balancing was not performed, set low = 1, igh = N.
				const long 	igh, 	// (INPUT) The igh index, as returned by EIG_balance_matrix. If balancing was not performed, set low = 1, igh = N.
				double 		A[], 	// (INPUT/OUTPUT) 2D matrix of size N x N, in column-major format. On input, the matrix to be reduced. On output, the Hessenberg matrix.
				long 		ind[]){	// (OUTPUT) 1D array of size N, containing informationon the rows & columns interchanged. Must be preallocated to size N.
	long i, j, m;
	double t, x, y;

	for(m = low + 1; m <= igh - 1; m++){
		x = 0.0;
		i = m;
		for(j = m; j <= igh; j++){
			if(abs(x) < abs(A[j+(m-1)*N] )){
				x = A[j+(m-1)*N];
				i = j;
			}
		}

		ind[m] = i;
		if(i != m){
			for(j = m - 1; j < N; j++){
				t        = A[i+j*N];
				A[i+j*N] = A[m+j*N];
				A[m+j*N] = t;
			}
			for(j = 0; j <= igh; j++){
				t        = A[j+i*N];
				A[j+i*N] = A[j+m*N];
				A[j+m*N] = t;
			}
		}

		if(x != 0.0){
			for(i = m + 1; i <= igh; i++){
				y = A[i+(m-1)*N];

				if(y != 0.0){
					y = y / x;
					A[i+(m-1)*N] = y;
					for(j = m; j < N; j++){
						A[i+j*N] = A[i+j*N] - y * A[m+j*N];
					}
					for(j = 0; j <= igh; j++){
						A[j+m*N] = A[j+m*N] + y * A[j+i*N];
					}
				}
			}
		}
	}
}




// compute the eigenvalues of a a real square upper Hessenberg matrix, using the QR method
// Returns 0 upon success, or J>0 if convergence was not reached at the J-th eigenvalue
// Code adjusted from: https://people.sc.fsu.edu/~jburkardt/c_src/eispack/eispack.html
long EIG_eigenvalues_RUH(	const long 	N, 			// (INPUT) the number of rows & columns in the matrix
							const long 	low, 		// (INPUT) The low index, as returned by EIG_balance_matrix. If balancing was not performed, set low = 0, igh = N-1
							const long 	igh, 		// (INPUT) The igh index, as returned by EIG_balance_matrix. If balancing was not performed, set low = 0, igh = N-1
							double 		H[],		// (INPUT/OUTPUT) 2D matrix of size N x N, in column-major format, storing the upper Hessenberg matrix. On output, this matrix will be modified, to store information about internal transformations
							double 		lambdaR[], 	// (OUTPUT) 1D array of size N, storing the real part of the eigenvalues. Conjugate pairs are listed consequitively, with the eigenvalue having positive imaginary part listed first. Must be preallocated to size N.
							double		lambdaI[]){	// (OUTPUT) 1D array of size N, storing the imaginary part of the eigenvalues. Must be preallocated to size N.
	long en, enm2, i, ierr, itn, its, j, k, l, m, na;
	double norm, p=0, q=0, r=0, s, t, tst1, tst2, w, x, y, zz;
	bool notlas;

	ierr = 0;
	norm = 0.0;
	k 	 = 0;
	for(i = 0; i < N; i++){
		for(j = k; j < N; j++){
			norm = norm + abs(H[i+j*N]);
		}
		k = i;
		if((i<low) || (igh<i)){
			lambdaR[i] = H[i+i*N];
			lambdaI[i] = 0.0;
		}
	}

	en 	= igh;
	t 	= 0.0;
	itn = 30 * N;

	if(igh < low) return ierr;

	its 	= 0;
	na 		= igh - 1;
	enm2 	= igh - 2;
	while(true){
		for(l = en; low <= l; l--){
			if(l == low) break;
			s = abs(H[l-1+(l-1)*N]) + abs(H[l+l*N]);
			if(s == 0.0) s = norm;
			tst1 = s;
			tst2 = tst1 + abs(H[l+(l-1)*N]);
			if(tst2 == tst1) break;
		}
		x = H[en+en*N];
		if(l == en){
			lambdaR[en] = x + t;
			lambdaI[en] = 0.0;
			en = na;
			if(en < low){
				return ierr;
			}
			its  = 0;
			na   = en - 1;
			enm2 = na - 1;
			continue;
		}
		y = H[na+na*N];
		w = H[en+na*N] * H[na+en*N];
		if(l == na){
			p = (y - x) / 2.0;
			q = p * p + w;
			zz = sqrt(abs(q));
			x = x + t;
			if(0.0 <= q){
				zz = p + abs(zz ) * (p<0 ? -1 : 1);
				lambdaR[na] = x + zz;
				if(zz == 0.0){
					lambdaR[en] = lambdaR[na];
				}else{
					lambdaR[en] = x - w / zz;
				}
				lambdaI[na] = 0.0;
				lambdaI[en] = 0.0;
			}else{
				lambdaR[na] = x + p;
				lambdaR[en] = x + p;
				lambdaI[na] = zz;
				lambdaI[en] = - zz;
			}
			en = enm2;
			if(en < low) return ierr;
			its = 0;
			na = en - 1;
			enm2 = na - 1;
			continue;
		}

		if(itn == 0){
			ierr = en;
			return ierr;
		}
		// exceptional shift.
		if(its == 10 || its == 20){
			t = t + x;
			for(i = low; i <= en; i++){
				H[i+i*N] = H[i+i*N] - x;
			}
			s = abs(H[en+na*N] ) + abs(H[na+enm2*N] );
			x = 0.75 * s;
			y = x;
			w = - 0.4375 * s * s;
		}

		its = its + 1;
		itn = itn - 1;
		for(m = enm2; l <= m; m--){
			zz = H[m+m*N];
			r = x - zz;
			s = y - zz;
			p = ( r * s - w ) / H[m+1+m*N] + H[m+(m+1)*N];
			q = H[m+1+(m+1)*N] - zz - r - s;
			r = H[m+2+(m+1)*N];
			s = abs(p) + abs(q) + abs(r);
			p = p / s;
			q = q / s;
			r = r / s;
			if(m == l) break;
			tst1 = abs(p) * (abs(H[m-1+(m-1)*N]) + abs(zz) + abs(H[m+1+(m+1)*N]));
			tst2 = tst1 + abs(H[m+(m-1)*N]) * (abs(q) + abs(r));
			if(tst2 == tst1) break;
		}

		for(i = m + 2; i <= en; i++){
			H[i+(i-2)*N] = 0.0;
			if(i != m + 2){
				H[i+(i-3)*N] = 0.0;
			}
		}

		// double QR step
		for(k = m; k <= na; k++){
			notlas = (k != na);
			if(k != m){
				p = H[k+(k-1)*N];
				q = H[k+1+(k-1)*N];
				if(notlas){
					r = H[k+2+(k-1)*N];
				}else{
					r = 0.0;
				}
				x = abs(p) + abs(q) + abs(r);
				if(x == 0.0) continue;
				p = p / x;
				q = q / x;
				r = r / x;
			}
			s = sqrt( p * p + q * q + r * r ) * (p<0 ? -1 : 1);
			if(k != m){
				H[k+(k-1)*N] = - s * x;
			}else if(l != m){
				H[k+(k-1)*N] = - H[k+(k-1)*N];
			}
			p  = p + s;
			x  = p / s;
			y  = q / s;
			zz = r / s;
			q  = q / p;
			r  = r / p;
		
			// row modification
			if(! notlas){
				for(j = k; j < N; j++){
					p = H[k+j*N] + q * H[k+1+j*N];
					H[k+j*N] = H[k+j*N] - p * x;
					H[k+1+j*N] = H[k+1+j*N] - p * y;
				}
				j = min(en, k + 3);
				// column modification
				for(i = 0; i <= j; i++){
					p = x * H[i+k*N] + y * H[i+(k+1)*N];
					H[i+k*N] = H[i+k*N] - p;
					H[i+(k+1)*N] = H[i+(k+1)*N] - p * q;
				}
			}else{
				for(j = k; j < N; j++){
					p = H[k+j*N] + q * H[k+1+j*N] + r * H[k+2+j*N];
					H[k+j*N]   = H[k+j*N] - p * x;
					H[k+1+j*N] = H[k+1+j*N] - p * y;
					H[k+2+j*N] = H[k+2+j*N] - p * zz;
				}
				j = min(en, k + 3);
				// column modification
				for(i = 0; i <= j; i++){
					p = x * H[i+k*N] + y * H[i+(k+1)*N] + zz * H[i+(k+2)*N];
					H[i+k*N] 	 = H[i+k*N] - p;
					H[i+(k+1)*N] = H[i+(k+1)*N] - p * q;
					H[i+(k+2)*N] = H[i+(k+2)*N] - p * r;
				}
			}
		}
	}
	return ierr;
}



// divide two complex numbers a and b, each represented as a real tuples (real & imaginary part)
// the result, c:=a/b is also returned as a real tuple
void divide_complex(const double ar, const double ai, const double br, const double bi, double &cr, double &ci){
	double ais, ars, bis, brs, s;
	s = abs(br) + abs(bi);

	ars = ar / s;
	ais = ai / s;
	brs = br / s;
	bis = bi / s;

	s = brs * brs + bis * bis;
	cr = (ars * brs + ais * bis) / s;
	ci = (ais * brs - ars * bis) / s;
}



// calculate eigenvalues & eigenvectors of a square real upper Hessenberg matrix, using the QR method
// returns 0 upon success, or an integer J>0 if convergence failed for the J-th eigenvalue.
// Code adjusted from: https://people.sc.fsu.edu/~jburkardt/c_src/eispack/eispack.html
long EIG_eigenvalues_RUH2(	const long 	N, 			// (INPUT) the number of rows & columns in the matrix
							const long 	low, 		// (INPUT) The low index, as returned by EIG_balance_matrix. If balancing was not performed, set low = 0, igh = N-1
							const long 	igh, 		// (INPUT) The igh index, as returned by EIG_balance_matrix. If balancing was not performed, set low = 0, igh = N-1
							double 		H[],		// (INPUT/OUTPUT) 2D matrix of size N x N, in column-major format, storing the upper Hessenberg matrix. On output, this matrix will be modified, to store information about internal transformations
							double 		lambdaR[], 	// (OUTPUT) 1D array of size N, storing the real part of the eigenvalues. Conjugate pairs are listed consequitively, with the eigenvalue having positive imaginary part listed first. Must be preallocated to size N. If an error occurred (ierr>0), the eigenvalues will be correct for indices ierr+1, .., N.
							double		lambdaI[],	// (OUTPUT) 1D array of size N, storing the imaginary part of the eigenvalues. Must be preallocated to size N.
							double		Z[]){		// (INPUT/OUTPUT) 2D matrix of size N x N, in column-major format. On input, contains the ELTRAN or ORTRAN trasformation, if performed. If the eigenvectors of H are requested, this must be the identity matrix. On output, Z contains the real & imaginary part of the eigenvectors. If the function returns with an error code, none of the returned eigenvectors are valid.
	long en, enm2, i, ierr, itn, its, j, k, l, m, na;
	double norm, p=0, q=0, r=0, ra, s=0, sa, t, ti=0, tr, tst1, tst2, vi, vr, w, x, y, zz=0;
	bool notlas;

	ierr = 0;
	norm = 0.0;
	k = 0;

	for(i = 0; i < N; i++){
		for(j = k; j < N; j++){
			norm = norm + abs(H[i+j*N]);
		}
		k = i;
		if(i < low || igh < i){
			lambdaR[i] = H[i+i*N];
			lambdaI[i] = 0.0;
		}
	}

	en = igh;
	t = 0.0;
	itn = 30 * N;
	while(low <= en){
		its = 0;
		na = en - 1;
		enm2 = na - 1;
		while(true){
			for(l = en; low <= l; l--){
				if(l == low) break;
				s = abs(H[l-1+(l-1)*N]) + abs(H[l+l*N]);
				if(s == 0.0) s = norm;
				tst1 = s;
				tst2 = tst1 + abs(H[l+(l-1)*N] );
				if(tst2 == tst1) break;
			}

			x = H[en+en*N];
			// found one root
			if(l == en){
				H[en+en*N] = x + t;
				lambdaR[en] = H[en+en*N];
				lambdaI[en] = 0.0;
				en = na;
				break;
			}
			y = H[na+na*N];
			w = H[en+na*N] * H[na+en*N];

			// found 2 roots
			if(l == na){
				p = (y - x) / 2.0;
				q = p * p + w;
				zz = sqrt(abs(q ) );
				H[en+en*N] = x + t;
				x = H[en+en*N];
				H[na+na*N] = y + t;

				if(q < 0.0){
					lambdaR[na] = x + p;
					lambdaR[en] = x + p;
					lambdaI[na] = zz;
					lambdaI[en] = - zz;
				}else{
					zz = p + abs(zz ) * (p<0 ? -1 : 1);
					lambdaR[na] = x + zz;
					lambdaR[en] = lambdaR[na];
					if(zz != 0.0 ){
						lambdaR[en] = x - w / zz;
					}
					lambdaI[na] = 0.0;
					lambdaI[en] = 0.0;
					x = H[en+na*N];
					s = abs(x ) + abs(zz );
					p = x / s;
					q = zz / s;
					r = sqrt(p * p + q * q );
					p = p / r;
					q = q / r;
					// row modification.
					for(j = na; j < N; j++){
						zz = H[na+j*N];
						H[na+j*N] = q * zz + p * H[en+j*N];
						H[en+j*N] = q * H[en+j*N] - p * zz;
					}
					// column modification.
					for(i = 0; i <= en; i++){
						zz = H[i+na*N];
						H[i+na*N] = q * zz + p * H[i+en*N];
						H[i+en*N] = q * H[i+en*N] - p * zz;
					}
					// accumulate transformations.
					for(i = low; i <= igh; i++){
						zz = Z[i+na*N];
						Z[i+na*N] = q * zz + p * Z[i+en*N];
						Z[i+en*N] = q * Z[i+en*N] - p * zz;
					}
				}
				en = enm2;
				break;
			}

			if(itn == 0 ){
				ierr = en;
				return ierr;
			}

			// exceptional shift.
			if(its == 10 || its == 20){
				t = t + x;
				for(i = low; i <= en; i++){
					H[i+i*N] = H[i+i*N] - x;
				}
				s = abs(H[en+na*N] ) + abs(H[na+enm2*N] );
				x = 0.75 * s;
				y = x;
				w = - 0.4375 * s * s;
			}

			its = its + 1;
			itn = itn - 1;
			for(m = enm2; l <= m; m--){
				zz = H[m+m*N];
				r = x - zz;
				s = y - zz;
				p = ( r * s - w ) / H[m+1+m*N] + H[m+(m+1)*N];
				q = H[m+1+(m+1)*N] - zz - r - s;
				r = H[m+2+(m+1)*N];
				s = abs(p ) + abs(q ) + abs(r );
				p = p / s;
				q = q / s;
				r = r / s;
				if(m == l) break;
				tst1 = abs(p ) * ( abs(H[m-1+(m-1)*N] ) + abs(zz ) 
				+ abs(H[m+1+(m+1)*N] ) );
				tst2 = tst1 + abs(H[m+(m-1)*N] ) * ( abs(q ) + abs(r ) );
				if(tst2 == tst1) break;
			}

			for(i = m + 2; i <= en; i++){
				H[i+(i-2)*N] = 0.0;
				if(i != m + 2 ){
					H[i+(i-3)*N] = 0.0;
				}
			}

			// double QR step
			for(k = m; k <= na; k++){
				notlas = (k != na);
				if(k != m){
					p = H[k+(k-1)*N];
					q = H[k+1+(k-1)*N];
					r = 0.0;
					if(notlas) r = H[k+2+(k-1)*N];
					x = abs(p ) + abs(q ) + abs(r );
					if(x == 0.0) continue;
					p = p / x;
					q = q / x;
					r = r / x;
				}
				s = sqrt(p * p + q * q + r * r) * (p<0 ? -1 : 1);
				if(k != m){
					H[k+(k-1)*N] = - s * x;
				}else if(l != m){
					H[k+(k-1)*N] = - H[k+(k-1)*N];
				}

				p = p + s;
				x = p / s;
				y = q / s;
				zz = r / s;
				q = q / p;
				r = r / p;

				if(!notlas){
					// row modification
					for(j = k; j < N; j++){
						p = H[k+j*N] + q * H[k+1+j*N];
						H[k+j*N] = H[k+j*N] - p * x;
						H[k+1+j*N] = H[k+1+j*N] - p * y;
					}
					j = min(en, k + 3);
					// column modification
					for(i = 0; i <= j; i++){
						p = x * H[i+k*N] + y * H[i+(k+1)*N];
						H[i+k*N] = H[i+k*N] - p;
						H[i+(k+1)*N] = H[i+(k+1)*N] - p * q;
					}
					// accumulate transformations
					for(i = low; i <= igh; i++){
						p = x * Z[i+k*N] + y * Z[i+(k+1)*N];
						Z[i+k*N] = Z[i+k*N] - p;
						Z[i+(k+1)*N] = Z[i+(k+1)*N] - p * q;
					}
				}else{
					// row modification
					for(j = k; j < N; j++){
						p = H[k+j*N] + q * H[k+1+j*N] + r * H[k+2+j*N];
						H[k+j*N] = H[k+j*N] - p * x;
						H[k+1+j*N] = H[k+1+j*N] - p * y;
						H[k+2+j*N] = H[k+2+j*N] - p * zz;
					}
					j = min(en, k + 3);
					// column modification
					for(i = 0; i <= j; i++){
						p = x * H[i+k*N] + y * H[i+(k+1)*N] + zz * H[i+(k+2)*N];
						H[i+k*N] = H[i+k*N] - p;
						H[i+(k+1)*N] = H[i+(k+1)*N] - p * q;
						H[i+(k+2)*N] = H[i+(k+2)*N] - p * r;
					}
					// accumulate transformations
					for(i = low; i <= igh; i++){
						p = x * Z[i+k*N] + y * Z[i+(k+1)*N] + zz * Z[i+(k+2)*N];
						Z[i+k*N] = Z[i+k*N] - p;
						Z[i+(k+1)*N] = Z[i+(k+1)*N] - p * q;
						Z[i+(k+2)*N] = Z[i+(k+2)*N] - p * r;
					}
				}
			}
		}
	}

	// all roots found, now backsubstitute
	if(norm == 0.0) return ierr;
	for(en = N - 1; 0 <= en; en--){
		p = lambdaR[en];
		q = lambdaI[en];
		na = en - 1;

		if(0.0 < q ){
			continue;
		}else if(q == 0.0){
			m = en;
			H[en+en*N] = 1.0;

			for(i = en - 1; en - na - 1 <= i; i--){
				w = H[i+i*N] - p;
				r = 0.0;
				for(j = m; j <= en; j++){
					r = r + H[i+j*N] * H[j+en*N];
				}
				if(lambdaI[i] < 0.0 ){
					zz = w;
					s = r;
					continue;
				}
				m = i;
				if(lambdaI[i] == 0.0){
					t = w;
					if(t == 0.0 ){
						tst1 = norm;
						t = tst1;
						while(true){
							t = 0.01 * t;
							tst2 = norm + t;
							if(tst2 <= tst1) break;
						}
					}
					H[i+en*N] = - r / t;
				}else{
					x = H[i+(i+1)*N];
					y = H[i+1+i*N];
					q = (lambdaR[i] - p) * (lambdaR[i] - p) + lambdaI[i] * lambdaI[i];
					t = (x * s - zz * r) / q;
					H[i+en*N] = t;
					if(abs(zz) < abs(x)){
						H[i+1+en*N] = ( - r - w * t ) / x;
					}else{
						H[i+1+en*N] = ( - s - y * t ) / zz;
					}
				}
				// overflow control
				t = abs(H[i+en*N]);
				if(t != 0.0 ){
					tst1 = t;
					tst2 = tst1 + 1.0 / tst1;
					if(tst2 <= tst1){
						for(j = i; j <= en; j++){
							H[j+en*N] = H[j+en*N] / t;
						}
					}
				}
			}
		}else if(q < 0.0){
			// complex vector
			m = na;
			if(abs(H[na+en*N]) < abs(H[en+na*N])){
				H[na+na*N] = q / H[en+na*N];
				H[na+en*N] = - ( H[en+en*N] - p ) / H[en+na*N];
			}else{
				divide_complex(0.0, -H[na+en*N], H[na+na*N] - p, q, tr, ti);
				H[na+na*N] = tr;
				H[na+en*N] = ti;
			}
			H[en+na*N] = 0.0;
			H[en+en*N] = 1.0;
			enm2 = na - 1;
			for(i = na - 1; na - enm2 <= i; i--){
				w = H[i+i*N] - p;
				ra = 0.0;
				sa = 0.0;
				for(j = m; j <= en; j++){
					ra = ra + H[i+j*N] * H[j+na*N];
					sa = sa + H[i+j*N] * H[j+en*N];
				}
				if(lambdaI[i] < 0.0 ){
					zz = w;
					r = ra;
					s = sa;
				}
				m = i;
				if(lambdaI[i] == 0.0 ){
					divide_complex(-ra, -sa, w, q, tr, ti);
					H[i+na*N] = tr;
					H[i+en*N] = ti;
				}else{
					x = H[i+(i+1)*N];
					y = H[i+1+i*N];
					vr = (lambdaR[i] - p) * (lambdaR[i] - p) + lambdaI[i] * lambdaI[i] - q * q;
					vi = (lambdaR[i] - p) * 2.0 * q;

					if(vr == 0.0 && vi == 0.0){
						tst1 = norm * (abs(w) + abs(q) + abs(x) + abs(y) + abs(zz));
						vr = tst1;
						while(true ){
							vr = 0.01 * vr;
							tst2 = tst1 + vr;
							if(tst2 <= tst1) break;
						}
					}

					divide_complex(x * r - zz * ra + q * sa, x * s - zz * sa - q * ra, vr, vi, tr, tr);
					H[i+na*N] = tr;
					H[i+en*N] = ti;
					if(abs(zz) + abs(q) < abs(x)){
						H[i+1+na*N] = (- ra - w * H[i+na*N] + q * H[i+en*N]) / x;
						H[i+1+en*N] = (- sa - w * H[i+en*N] - q * H[i+na*N]) / x;
					}else{
						divide_complex(- r - y * H[i+na*N], - s - y * H[i+en*N], zz, q, tr, ti);
						H[i+1+na*N] = tr;
						H[i+1+en*N] = ti;
					}
				}
				// overflow control.
				t = max(abs(H[i+na*N]), abs(H[i+en*N]));

				if(t != 0.0){
					tst1 = t;
					tst2 = tst1 + 1.0 / tst1;
					if(tst2 <= tst1){
						for(j = i; j <= en; j++){
							H[j+na*N] = H[j+na*N] / t;
							H[j+en*N] = H[j+en*N] / t;
						}
					}
				}
			}
		}
	}
	
	// end back substitution
	for(i = 0; i < N; i++){
		if(i < low || igh < i){
			for(j = i; j < N; j++){
				Z[i+j*N] = H[i+j*N];
			}
		}
	}
	
	// multiply by transformation matrix to obtain vectors of original matrix
	for(j = N - 1; low <= j; j--){
		m = min(j, igh);
		for(i = low; i <= igh; i++){
			zz = 0.0;
			for(k = low; k <= m; k++){
				zz = zz + Z[i+k*N] * H[k+j*N];
			}
			Z[i+j*N] = zz;
		}
	}

	return ierr;
}






// calculate eigenvalues and (optionally) eigenvectors of a real square matrix A
// Internally, A is first transformed into upper Hessenberg form, and then eigenvalues & eigenvectors are computed using the QR method
// returns 0 upon success, otherwise the error code is as returned by EIG_eigenvalues_RUH() or EIG_eigenvalues_RUH2()
// Code adjusted from: https://people.sc.fsu.edu/~jburkardt/c_src/eispack/eispack.html
long EIG_eigendecomposition(const long				N,						// (INPUT) the number of rows & columns in A
							const dvector 			&A,						// (INPUT) 2D matrix of size N x N, in row-major or column-major format
							const bool				row_major,				// (INPUT) whether the matrix A is in row-major or column-major format. The same formatting will also apply to the output. Note that internally this function transforms everything into column-major format if needed.
							const bool				include_eigenvectors,	// (INPUT) whether to also compute eigenvectors
							dvector					&scratchA,				// (SCRATCH) scratch space, will be resized to N x N
							dvector					&scratchZ,				// (SCRATCH) scratch space for eigenvector computation, will be resized to N x N. Only relevant if include_eigenvectors==true.
							dvector					&eigenvaluesR,			// (OUTPUT) the real part of the eigenvalues
							dvector					&eigenvaluesI,			// (OUTPUT) the imaginary part of the eigenvalues
							std::vector<cdouble>	&eigenvectors){			// (OUTPUT) 2D matrix of size N x N, containing the eigenvectors (one eigenvector per column). Only relevant if include_eigenvectors==true. Will be in row-major format iff row_major=true, otherwise it will be in column-major format.
	if(row_major){
		// transform A into column-major format and store result in scratchA
		scratchA.resize(N*N);
		for(long r=0; r<N; ++r){
			for(long c=0; c<N; ++c){
				scratchA[r + c*N] = A[r*N + c];
			}
		}
	}else{
		// A is already in column-major format
		scratchA = A;
	}
	
	int ierr;
	long is1, is2;
	dvector fv1(N);
	lvector iv1(N);
	eigenvaluesR.resize(N);
	eigenvaluesI.resize(N);
	EIG_balance_matrix(N, &scratchA[0], is1, is2, &fv1[0]);
	EIG_ELMHES(N, is1, is2, &scratchA[0], &iv1[0]);

	if(!include_eigenvectors){
		ierr = EIG_eigenvalues_RUH(N, is1, is2, &scratchA[0], &eigenvaluesR[0], &eigenvaluesI[0]);
		if(ierr!=0) return ierr;  // an error occurred during HQR
	}else{
		scratchZ.resize(N*N); // extra scratch space for eigenvectors
		EIG_accumulate_similarity(N, is1, is2, &scratchA[0], &iv1[0], &scratchZ[0]);
		ierr = EIG_eigenvalues_RUH2(N, is1, is2, &scratchA[0], &eigenvaluesR[0], &eigenvaluesI[0], &scratchZ[0]);
		if(ierr!=0) return ierr; // an error occurred during HQR2
		EIG_reverse_balancing(N, is1, is2, &fv1[0], N, &scratchZ[0]);
		
		// compactify eigenvectors to complex numbers
		// note that scratchZ is structured in column-major format, and each column (or pair of columns) corresponds to an eigenvector
		eigenvectors.resize(N*N);
		for(long j=0; j<N; ++j){
			if(eigenvaluesI[j]==0){
				// this eigenvalue is real, so the eigenvector is stored in the j-th column of Z
				for(long r=0; r<N; ++r){
					eigenvectors[r+j*N] = complex<double>(scratchZ[r+j*N],0);
				}
			}else{
				// this eigenvalue is complex, so the eigenvector is stored in the j-th and (j+1)-th columns of Z (real & imaginary part, respectively). 
				// the conjugate of this eigenvector is the eigenvector for the conjugate eigenvalue
				for(long r=0; r<N; ++r){
					eigenvectors[r+j*N]		= complex<double>(scratchZ[r+j*N],+scratchZ[r+(j+1)*N]);
					eigenvectors[r+(j+1)*N]	= complex<double>(scratchZ[r+j*N],-scratchZ[r+(j+1)*N]);
				}				
				++j; // skip next eigenvalue, since it is the complex-conjugate of the j-th eigenvalue
			}
		}
	}
	
	// transform eigenvectors from column-major to row-major format if needed
	// this basically means transposing the eigenvector matrix
	if(include_eigenvectors && row_major){
		cdouble temp;
		for(long r=1; r<N; ++r){
			for(long c=0; c<r; ++c){
				temp = eigenvectors[r*N + c];
				eigenvectors[r*N + c] = eigenvectors[r + c*N];
				eigenvectors[r + c*N] = temp;
			}
		}
	}
	return ierr;
}



// get spectral range of a real square general matrix, i.e. the difference between the largest (most positive) real part of an eigenvalue and the smallest (most negative) real part of an eigenvalue
// That is, if (R1,I1),...,(RN,IN) are the eigenvalues of the matrix (where Ri and Ii is the real & imaginary part of the i-th eigenvalue), calculate max_{i,j} (Ri-Rj).
// The input matrix A can be in row-major or column-major format; the result is the same regardless (since confusing majority simply corresponds to transposing A)
// returns NAN_D upon failure
double get_spectral_range(	const long 		N,		// (INPUT) the number of rows & columns in A
							const dvector	&A){	// (INPUT) 2D matrix of size N x N, in row-major or column-major format (it does not matter which)
	// get all eigenvalues of A
	std::vector<cdouble> eigenvectors;
	dvector scratchA, scratchZ, eigenvaluesR, eigenvaluesI;
	const long error = EIG_eigendecomposition(N, A, false, false, scratchA, scratchZ, eigenvaluesR, eigenvaluesI, eigenvectors);
	if(error!=0) return NAN_D;
	
	// find most positive and most negative part of any eigenvalue
	double maxR = eigenvaluesR[0], minR = eigenvaluesR[0];
	for(long i=0; i<N; ++i){
		maxR = max(maxR, eigenvaluesR[i]);
		minR = min(minR, eigenvaluesR[i]);
	}
	
	return (maxR-minR);
}


// calculate the largest singular value (sigma1) of a real matrix A
// this implementation is not the most efficient, since it just calculates the square-root of the largest eigenvalue of A*A^T
double get_largest_singular_value(	const long 		NR,		// (INPUT) the number of rows in A
									const long		NC,		// (INPUT) the number of columns in A
									const dvector	&A,		// (INPUT) 2D matrix of size N x N, in row-major or column-major format (it does not matter which)
									const long		max_iterations,	// (INPUT) max number of iterations for power method. Typical values are 1e3 - 1e6.
									const double	tolerance){		// (INPUT) relative error tolerance for detecting convergence during the power method. Typical values are 1e-3 - 1e-6.
	// calculate A*A^T or A^T*A, whichever has smaller dimensions
	const long N = (NR<NC ? NR : NC);
	dvector AA(N*N);
	if(NR<NC){
		// calculate A*A^T
		for(long r=0; r<N; ++r){
			for(long c=0; c<N; ++c){
				AA[r*N + c] = 0;
				for(long k=0; k<NC; ++k){
					AA[r*N + c] += A[r*NC + k] * A[c*NC+k];
				}
			}
		}
	}else{
		// calculate A^T*A
		for(long r=0; r<N; ++r){
			for(long c=0; c<N; ++c){
				AA[r*N + c] = 0;
				for(long k=0; k<NR; ++k){
					AA[r*N + c] += A[k*NC+r] * A[k*NC+c];
				}
			}
		}		
	}
	
	// calculate largest eigenvalue of AA via the power method (note that all eigenvalues of AA are real and non-negative)
	dvector dummyX;
	double lambda;
	const bool converged = get_dominant_eigenvalue(N, AA, max_iterations, tolerance, dummyX, lambda);
	if(converged) return sqrt(lambda);

	// power method failed, so use full eigendcomposition
	std::vector<cdouble> eigenvectors;
	dvector scratchA, scratchZ, eigenvaluesR, eigenvaluesI;
	const long error = EIG_eigendecomposition(N, AA, false, false, scratchA, scratchZ, eigenvaluesR, eigenvaluesI, eigenvectors);
	if(error!=0) return NAN_D;
	lambda = eigenvaluesR[0];
	for(long i=0; i<N; ++i){
		lambda = max(lambda, eigenvaluesR[i]);
	}
	return sqrt(lambda);
}


// construct SO(3) rotation matrix about the axis (ux,uy,uz) (which must be a unit vector in R^3)
// the convention is that if the axis vector u points towards the observer and angle>0, then the rotation proceeds counter-clockwise
void get_3D_rotation_matrix(const double ux, 	// (INPUT)
							const double uy, 	// (INPUT)
							const double uz, 	// (INPUT)
							const double angle, // (INPUT) rotation angle in radians
							dvector &matrix){	// (OUTPUT) 2D matrix of size 3 x 3, in row-major format
	matrix.resize(3*3);
	const double C = cos(angle);
	const double S = sin(angle);
	
	matrix[0*3+0] = C + ux*ux*(1-C);
	matrix[0*3+1] = ux*uy*(1-C) - uz*S;
	matrix[0*3+2] = ux*uz*(1-C) + uy*S;
	
	matrix[1*3+0] = uy*ux*(1-C) + uz*S;
	matrix[1*3+1] = C + uy*uy*(1-C);
	matrix[1*3+2] = uy*uz*(1-C) - ux*S;
	
	matrix[2*3+0] = uz*ux*(1-C) - uy*S;
	matrix[2*3+1] = uz*uy*(1-C) + ux*S;
	matrix[2*3+2] = C + uz*uz*(1-C);
}


// construct rotation matrix that takes north pole to the requested (latitude=theta, longitude=phi) (measured in radians)
inline void get_3D_rotation_matrix_from_north_pole(	const double 	theta,	// (INPUT) latitude, from -pi/2 to pi/2
													const double 	phi,	// (INPUT) longitude, from -pi to pi
													dvector 		&rotation_matrix){	// (OUTPUT) 2D matrix of size 3 x 3, in row-major format
	get_3D_rotation_matrix(sin(phi),-cos(phi),0,theta-0.5*M_PI,rotation_matrix);
}




#pragma mark -
#pragma mark Numerical solvers
#pragma mark



// RungeKutta2 integrator, handling:
//    1. Suggestions by the model for temporarily refined time steps
//    2. Situations where the domain boundary is crossed during an iteration (in which case the time step is temporarily decreased as needed in order to not overshoot)
//    3. Requests by the model to jump discontinuously to another state
// requested times might reverse (i.e. times at which model dynamics are requested might not be strictly increasing), but recorded time series will be strictly forward in time
template<class COORDINATE, class MODEL, class PROGRESS_REPORTER>
bool RungeKutta2(	double						startTime,						// (INPUT) simulation start time
					double 						endTime, 						// (INPUT) simulation end time
					double 						dt, 							// (INPUT) default integration time step
					MODEL 						&model,							// (INPUT/OUTPUT) object defining model dynamics, also handling time series storage
					long						maxRecordedPoints, 				// (INPUT) if small, some intermediate trajectory points are skipped (i.e. not recorded). This might be useful if accurate integration requires a small time step, but would produce a needlesly large time series. If maxRecordedPoints==2, then only the start and end points are recorded. If maxRecordedPoints==1, then only the end point is recorded.
					long						maxTimeStepRefinements,			// (INPUT) max allowed number of refinements of local time step when encountering invalid states. Only relevant if abortOnInvalidState==false.
					const double				refinement_factor,				// (INPUT) factor by which to refine time steps. Typical values are 2-10.
					const PROGRESS_REPORTER		&reporter,						// (INPUT) callback functor to handle progress reporting during simulation
					const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
					string						&warningMessage){				// (OUTPUT) will be non-empty in case of error, or if non-fatal problems occurred
	COORDINATE currentPoint, candidatePoint, point2, jumpPoint;
	COORDINATE k1, k2, kConsensus;
	double t=startTime, t2, current_dt1, current_dt2;
	long recorded, iterations;
	RequestedDynamics dynamics;
	CrossedBoundary crossedBoundary;
	const double simulationTime 	= endTime-startTime; // target duration of simulation
	const double start_runtime  	= get_thread_monotonic_walltime_seconds();
	const double min_dt 			= dt/pow(refinement_factor,maxTimeStepRefinements);
	
	// keep track of warnings
	bool crossedBoundaryButFixedByReducingTimeStep = false;
	bool crossedBoundaryYesButFixedBruteForce = false;
	bool crossedBoundaryYesButFixedUsingEuler = false;

	//preliminary error checking
	warningMessage = "";
	if(dt<simulationTime*RELATIVE_EPSILON){
		warningMessage = "Time step too small";
		return false;
	}
	if(simulationTime < dt){
		warningMessage = "Time step exceeds simulation time";
		return false;
	}
	if(maxRecordedPoints < 1){
		warningMessage = "Requested zero recorded points";
		return false;
	}
	
	const double recordingTimeStep = simulationTime/max(1L,maxRecordedPoints-1);
	model.reserveSpaceForTimeSeries(maxRecordedPoints);
	bool k1AlreadyCalculated = false;
	
	//initialization
	if(!model.getInitialState(t, currentPoint)){
		warningMessage = "Failed to get initial state";
		return false;
	}
	double lastRecordedTime = -INFTY_D;
	if(maxRecordedPoints>1){
		model.registerState(t, currentPoint);
		lastRecordedTime = t;
	}
	
	//run simulation
	//default time step is dt, but temporarily dt might be reduced to a smaller value
	for(recorded=1, current_dt1=current_dt2=dt, iterations=0; t<endTime; /* increment of t handled in loop */ ){
		// at this stage currentPoint is guaranteed to be a valid state
		// t should always correspond to currentPoint
		// t2 should always correspond to point2
		// current_dt1 is used for moving from currentPoint-->point2
		// current_dt2 is used for moving from currentPoint-->candidatePoint (= potential next currentPoint)
		++iterations;
		
		// check runtime-out
		if((runtime_out_seconds>0) && (iterations%100==0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation was aborted prematurely after "+makeString(iterations)+" iterations because it reached the maximum allowed processing time.";
			return true;
		}
		
		// reduce time step to not exceed end time, if needed
		current_dt1 = min(min(current_dt1, current_dt2), endTime-t);
				
		// get dynamics (k1) at currentPoint if needed
		if(!k1AlreadyCalculated){
			dynamics = model.getRateOfChangeAtState(t, currentPoint, k1, jumpPoint);
			if(dynamics==RequestedDynamicsForceJumpToState){
				currentPoint = jumpPoint;
				t = t + current_dt1;
				goto RK2_REGISTER_STATE; 
			}
			k1AlreadyCalculated = true;
		}
		
		// get point2 (in forward k1-direction)
		point2 = currentPoint + k1*current_dt1;
		t2 = t + current_dt1; // t2 should always correspond to point2
		
		// check if point2 crossed the boundary and move point2 backward as much as needed
		// point2 and current_dt1 will be adjusted if this routine returns true
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentPoint, current_dt1, point2, true);
		if(crossedBoundary == CrossedBoundaryYesButFixedBruteForce){ crossedBoundaryYesButFixedBruteForce = true; }
		else if(crossedBoundary == CrossedBoundaryYesButFixedByReducingTimeStep){ crossedBoundaryButFixedByReducingTimeStep = true; }
		
		// check if the time step should be reduced (as suggested by the model)
		while((current_dt1>min_dt) && model.checkShouldRefineTimeStep(t, currentPoint, current_dt1, point2)){
			current_dt1 /= refinement_factor;
			point2 = currentPoint + k1*current_dt1;
			t2 = t + current_dt1; // t2 should always correspond to point2
		}
		
		// stage-2 time-step should not be larger than the stage-1 time-step, but also not below min_dt
		current_dt2 = max(min(min_dt,endTime-t), min(current_dt1,current_dt2));
		
		// get dynamics (k2) at point2
		dynamics = model.getRateOfChangeAtState(t2, point2, k2, jumpPoint);
		if(dynamics==RequestedDynamicsForceJumpToState){
			currentPoint = jumpPoint;
			t = t2;
			goto RK2_REGISTER_STATE; 
		}

		// use k1 & k2 to move the currentPoint forward, potentially after refining the time step multiple times
		kConsensus 		= (k1+k2)*0.5;
		candidatePoint 	= currentPoint + kConsensus*current_dt2;
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentPoint, current_dt2, candidatePoint, false);
		if(crossedBoundary==CrossedBoundaryYesButFixedBruteForce){
			if(current_dt2>min_dt){
				current_dt2 /= refinement_factor;
				crossedBoundaryButFixedByReducingTimeStep = true;
				continue; // repeat the whole iteration with a smaller time step
			}else{
				// use simple Euler scheme for this time step, since kConsensus throws us out of the boundary at even arbitrarily small time steps
				currentPoint = point2;
				t 			 = t2;
				crossedBoundaryYesButFixedUsingEuler = true;
				goto RK2_REGISTER_STATE;
			}
		}
				
		// check if the time step should be reduced (as suggested by the model)
		if((current_dt2>min_dt) && model.checkShouldRefineTimeStep(t, currentPoint, current_dt2, candidatePoint)){
			current_dt2 /= refinement_factor; 
			continue; // repeat the last part with a smaller time step
		}
		
		// seems everything worked out as normal, so set the new currentPoint
		currentPoint = candidatePoint;
		t = t + current_dt2; // t should always correspond to currentPoint
		goto RK2_REGISTER_STATE;
		
		// check and register new state if needed	
		RK2_REGISTER_STATE:
		// check sanity (some ODEs may explode)		
		if(std::isnan(t) || model.stateIsNaN(currentPoint)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation reached NaN.";
			return (recorded>1);
		}
		k1AlreadyCalculated = false;
		if((t-lastRecordedTime > recordingTimeStep) && (recorded<maxRecordedPoints-1)){ // don't record if maxRecordedPoints has been almost exhausted (i.e. 1 remaining), since the final state will be recorded outside of the loop
			model.registerState(t, currentPoint);
			++recorded;
			lastRecordedTime = t;
			reporter(recorded, maxRecordedPoints, 1-(endTime-t)/simulationTime);
		}
		
		// initialize counters for next iteration (note: this step may be skipped by a 'continue' statement)
		// only gradually increase the time step, to avoid repeatedly wasting time in refinements
		current_dt1  = min(refinement_factor*current_dt1,dt);
		current_dt2  = min(refinement_factor*current_dt2,dt);
	}
	
	// register final state if needed
	if(t>lastRecordedTime){
		model.registerState(t, currentPoint);
	}
	
	// construct warnings message if needed
	if(crossedBoundaryYesButFixedUsingEuler) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally using forward Euler.";
	if(crossedBoundaryYesButFixedBruteForce) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by brute-force adjusting the trajectory.";
	if(crossedBoundaryButFixedByReducingTimeStep) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally reducing the time step.";

	return true;
}







// RungeKutta2 integrator, handling:
//    1. Suggestions by the model for temporarily refined time steps
//    2. Situations where the domain boundary is crossed during an iteration (in which case the time step is temporarily decreased as needed in order to not overshoot)
//    3. Requests by the model to jump discontinuously to another state
// requested times might reverse (i.e. times at which model dynamics are requested might not be strictly increasing), but recorded time series will be strictly forward in time
// This is similar to RungeKutta2 above, the difference being that here points are recorded when deemed appropriate based on the difference from the previous time point (or equivalently, the local rate of change)
// Hence, instead of specifying the max number of recorded time points, the caller specifies the minRecordingTimeStep and the recordingRelValueStep
template<class COORDINATE, class MODEL, class PROGRESS_REPORTER>
bool RungeKutta2(	double						startTime,						// (INPUT) simulation start time
					double 						endTime, 						// (INPUT) simulation end time
					double 						dt, 							// (INPUT) default integration time step
					MODEL 						&model,							// (INPUT/OUTPUT) object defining model dynamics, also handling time series storage
					double						minRecordingTimeStep, 			// (INPUT) the minimum time difference between subsequent recordings. Used to prevent excessive recordings.
					double						maxRecordingTimeStep,			// (INPUT) the maximum time difference recommended between subsequent recordings. The returned time series will have time steps (typically, not guaranteed) below this guiding threshold. This should be larger than minRecordingTimeStep.
					double						guessNrecordings,				// (INPUT) a guesstimate for the number of recorded time steps, used to pre-allocated space for the recorded time series
					double						recordingRelValueStep, 			// (INPUT) the minimum relative value difference to the previous recording, before triggering a new recording. This is sub-ordinate to minRecordingTimeStep, i.e. minRecordingTimeStep will never be violated (except perhaps at the end, because the last time point is always recorded). The "relative difference" between two states is defined by the model, not the solver. Typically this will be the maximum relative difference between components.
					long						maxTimeStepRefinements,			// (INPUT) max allowed number of refinements of local time step when encountering invalid states. Only relevant if abortOnInvalidState==false.
					const double				refinement_factor,				// (INPUT) factor by which to refine time steps. Typical values are 2-10.
					const PROGRESS_REPORTER		&reporter,						// (INPUT) callback functor to handle progress reporting during simulation
					const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
					string						&warningMessage){				// (OUTPUT) will be non-empty in case of error, or if non-fatal problems occurred
	COORDINATE currentPoint, candidatePoint, point2, jumpPoint;
	COORDINATE k1, k2, kConsensus;
	double t, t2, current_dt1, current_dt2;
	long recorded, iterations;
	RequestedDynamics dynamics;
	CrossedBoundary crossedBoundary;
	const double simulationTime 	= endTime-startTime; // target duration of simulation
	const double start_runtime  	= get_thread_monotonic_walltime_seconds();
	const double min_dt 			= dt/pow(refinement_factor,maxTimeStepRefinements);
	
	// keep track of warnings
	bool crossedBoundaryButFixedByReducingTimeStep = false;
	bool crossedBoundaryYesButFixedBruteForce = false;
	bool crossedBoundaryYesButFixedUsingEuler = false;

	//preliminary error checking
	warningMessage = "";
	if(dt<simulationTime*RELATIVE_EPSILON){
		warningMessage = "Time step too small";
		return false;
	}
	if(simulationTime < dt){
		warningMessage = "Time step exceeds simulation time";
		return false;
	}
	if(recordingRelValueStep < 0){
		warningMessage = "recordingRelValueStep is negative";
		return false;
	}
	if(minRecordingTimeStep>maxRecordingTimeStep){
		warningMessage = "minRecordingTimeStep is greater than maxRecordingTimeStep";
		return false;
	}
	
	model.reserveSpaceForTimeSeries(guessNrecordings);
	bool k1AlreadyCalculated = false;
	
	//initialization
	t = startTime;
	if(!model.getInitialState(t, currentPoint)){
		warningMessage = "Failed to get initial state";
		return false;
	}
	model.registerState(t, currentPoint);
	COORDINATE lastRecordedPoint = currentPoint;
	double lastRecordedTime = t;
	
	//run simulation
	//default time step is dt, but temporarily dt might be reduced to a smaller value
	for(recorded=1, current_dt1=current_dt2=dt, iterations=0; t<endTime; /* increment of t handled in loop */ ){
		// at this stage currentPoint is guaranteed to be a valid state
		// t should always correspond to currentPoint
		// t2 should always correspond to point2
		// lastRecordedTime should always correspond to lastRecordedPoint
		// current_dt1 is used for moving from currentPoint-->point2
		// current_dt2 is used for moving from currentPoint-->candidatePoint (= potential next currentPoint)
		++iterations;
		
		// check runtime-out
		if((runtime_out_seconds>0) && (iterations%100==0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation was aborted prematurely after "+makeString(iterations)+" iterations because it reached the maximum allowed processing time.";
			return true;
		}
		
		// reduce time step to not exceed end time, if needed
		current_dt1 = min(min(current_dt1, current_dt2), endTime-t);
				
		// get dynamics (k1) at currentPoint if needed
		if(!k1AlreadyCalculated){
			dynamics = model.getRateOfChangeAtState(t, currentPoint, k1, jumpPoint);
			if(dynamics==RequestedDynamicsForceJumpToState){
				currentPoint = jumpPoint;
				t 			 = t + current_dt1;
				goto ARK2_REGISTER_STATE; 
			}
			k1AlreadyCalculated = true;
		}
		
		// get point2 (in forward k1-direction)
		linear_combination(1.0,currentPoint,current_dt1,k1,point2); // point2 = currentPoint + k1*current_dt1;
		t2 = t + current_dt1; // t2 should always correspond to point2
		
		// check if point2 crossed the boundary and move point2 backward as much as needed
		// point2 and current_dt1 will be adjusted if this routine returns true
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentPoint, current_dt1, point2, true);
		if(crossedBoundary == CrossedBoundaryYesButFixedBruteForce){ crossedBoundaryYesButFixedBruteForce = true; }
		else if(crossedBoundary == CrossedBoundaryYesButFixedByReducingTimeStep){ crossedBoundaryButFixedByReducingTimeStep = true; }
		
		// check if the time step should be reduced (as suggested by the model)
		while((current_dt1>min_dt) && model.checkShouldRefineTimeStep(t, currentPoint, current_dt1, point2)){
			current_dt1 /= refinement_factor;
			linear_combination(1.0,currentPoint,current_dt1,k1,point2); // point2 = currentPoint + k1*current_dt1;
			t2 = t + current_dt1; // t2 should always correspond to point2
		}
		
		// stage-2 time-step should not be larger than the stage-1 time-step, but also not below min_dt
		current_dt2 = max(min(min_dt,endTime-t), min(current_dt1,current_dt2));
		
		// get dynamics (k2) at point2
		dynamics = model.getRateOfChangeAtState(t2, point2, k2, jumpPoint);
		if(dynamics==RequestedDynamicsForceJumpToState){
			currentPoint = jumpPoint;
			t			 = t2;
			goto ARK2_REGISTER_STATE; 
		}

		// use k1 & k2 to move the currentPoint forward, potentially after refining the time step multiple times
		linear_combination(0.5, k1, 0.5, k2, kConsensus); // kConsensus = 0.5*(k1+k2)
		linear_combination(1.0, currentPoint, current_dt2, kConsensus, candidatePoint); // candidatePoint 	= currentPoint + kConsensus*current_dt2
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentPoint, current_dt2, candidatePoint, false);
		if(crossedBoundary==CrossedBoundaryYesButFixedBruteForce){
			if(current_dt2>min_dt){
				current_dt2 /= refinement_factor;
				crossedBoundaryButFixedByReducingTimeStep = true;
				continue; // repeat the whole iteration with a smaller time step
			}else{
				// use simple Euler scheme for this time step, since kConsensus throws us out of the boundary at even arbitrarily small time steps
				currentPoint = point2;
				t	 		 = t2;
				crossedBoundaryYesButFixedUsingEuler = true;
				goto ARK2_REGISTER_STATE;
			}
		}
				
		// check if the time step should be reduced (as suggested by the model)
		if((current_dt2>min_dt) && model.checkShouldRefineTimeStep(t, currentPoint, current_dt2, candidatePoint)){
			current_dt2 /= refinement_factor; 
			continue; // repeat the last part with a smaller time step
		}

		// seems everything worked out as normal, so set the new currentPoint
		currentPoint = candidatePoint;
		t 			 = t + current_dt2; // t should always correspond to currentPoint
				
		// register new state if needed	
		ARK2_REGISTER_STATE:
				
		// check sanity (some ODEs may explode)		
		if(std::isnan(t) || model.stateIsNaN(currentPoint)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation reached NaN.";
			return (recorded>1);
		}
		k1AlreadyCalculated = false;
		// check if we should record this new point, based on the time and value step compared to the previous recording
		// don't record if maxRecordedPoints has been almost exhausted (i.e. 1 remaining), since the final state will be recorded outside of the loop			
		if((t-lastRecordedTime >= minRecordingTimeStep) && ((t-lastRecordedTime>=maxRecordingTimeStep) || (model.getRelativeChange(lastRecordedPoint,currentPoint)>recordingRelValueStep))){
			model.registerState(t, currentPoint);
			++recorded;
			lastRecordedTime  = t;
			lastRecordedPoint = currentPoint;
			reporter((t-startTime), simulationTime);
		}
		
		// initialize counters for next iteration (note: this step may be skipped by a 'continue' statement)
		// only gradually increase the time step, to avoid repeatedly wasting time in refinements
		current_dt1  = min(refinement_factor*current_dt1,dt);
		current_dt2  = min(refinement_factor*current_dt2,dt);
	}
	
	// register final state if needed
	if(t>lastRecordedTime){
		model.registerState(t, currentPoint);
	}
	
	// construct warnings message if needed
	if(crossedBoundaryYesButFixedUsingEuler) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally using forward Euler.";
	if(crossedBoundaryYesButFixedBruteForce) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by brute-force adjusting the trajectory.";
	if(crossedBoundaryButFixedByReducingTimeStep) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally reducing the time step.";

	return true;
}






// Two-step Runge-Kutta solver for ODEs where the "scale" and the "shape" (rescaled-version) of the trajectory should be simulated instead.
// Consider the following ODE:
//    dX/dt = f(t,X)
// Here, instead of directly solving the ODE for X(t), we solve two auxiliary ODEs for S(t) and Y(t), where S is the "scale" (in some abstract sense) of X, and Y is a rescaled version of X (i.e. the "shape" of X)
// For example, if X is a classical vector, then S may be defined as log(mean(X)) and Y = X/S.
// Hence, any current state X is represented by a tuple (S,Y), where S is scalar and Y is of similar nature as X (the exact format of Y is handled transparently by the model, not the solver).
// A separation between "shape" Y and "scale" S may be especially useful if X(t) decays to zero exponentially, resulting in numerical underflow, or explodes to infinity exponentially, resulting in numerical overflow.
//
// The solver can handle:
//    1. Suggestions by the model for temporarily refined time steps
//    2. Situations where the domain boundary is crossed during an iteration (in which case the time step is temporarily decreased as needed in order to not overshoot)
//    3. Requests by the model to jump discontinuously to another state
//
// The shape Y must be of type COORDINATE, and the scale S must be of type double.
// The model must be able to provide initial conditions and the dynamics for S and Y at each time point, as well as storing of the computed trajectory (defined by S and Y).
// Of course, the model may internally store the trajectory X instead of S and Y, but the integrator will always compute and provide S and Y.
//
// Requested times might reverse (i.e. times at which model dynamics are requested might not be strictly increasing), but recorded time series will be strictly forward in time
template<class COORDINATE, class MODEL, class PROGRESS_REPORTER>
bool ScaledRungeKutta(	const double				startTime,						// (INPUT) simulation start time
						const double				endTime, 						// (INPUT) simulation end time
						double 						dt, 							// (INPUT) default integration time step
						MODEL 						&model,							// (INPUT/OUTPUT) object defining model dynamics (rates of change of S & Y), and handling time series storage
						long						maxRecordedPoints, 				// (INPUT) if small, some intermediate trajectory points are skipped (i.e. not recorded). This might be useful if accurate integration requires a small time step, but would produce a needlesly large time series. If maxRecordedPoints==2, then only the start and end points are recorded. If maxRecordedPoints==1, then only the end point is recorded.
						long						maxTimeStepRefinements,			// (INPUT) max allowed number of refinements of local time step when encountering invalid states. Only relevant if abortOnInvalidState==false.
						const PROGRESS_REPORTER		&reporter,						// (INPUT) callback functor to handle progress reporting during simulation
						const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
						string						&warningMessage){				// (OUTPUT) will be non-empty in case of error, or if non-fatal problems occurred
	COORDINATE currentY, candidateY, Y2, jumpY;
	double currentS, candidateS, S2, jumpS;
	COORDINATE kY1, kY2, kYConsensus;
	double kS1, kS2, kSConsensus;
	double t=startTime, t2, current_dt1, current_dt2;
	long recorded, refinements, iterations;
	RequestedDynamics dynamics;
	CrossedBoundary crossedBoundary;
	const double simulationTime = endTime-startTime; // target duration of simulation
	const double start_runtime  = get_thread_monotonic_walltime_seconds();
	
	// keep track of warnings
	bool crossedBoundaryButFixedByReducingTimeStep = false;
	bool crossedBoundaryYesButFixedBruteForce = false;
	bool crossedBoundaryYesButFixedUsingEuler = false;
	
	//preliminary error checking
	warningMessage = "";
	if(dt<simulationTime*RELATIVE_EPSILON){
		warningMessage = "Time step too small";
		return false;
	}
	if(simulationTime < dt){
		warningMessage = "Time step exceeds simulation time";
		return false;
	}
	if(maxRecordedPoints < 1){
		warningMessage = "Requested zero recorded points";
		return false;
	}
	
	const double recordingTimeStep = simulationTime/maxRecordedPoints;
	model.reserveSpaceForScaledTimeSeries(maxRecordedPoints);
	bool k1AlreadyCalculated = false;
	
	// get initial state (shape and scale)
	if(!model.getScaledInitialState(t, currentY, currentS)){
		warningMessage = "Failed to get initial state";
		return false;
	}
	
	// record initial state
	double lastRecordedTime = -INFTY_D;
	if(maxRecordedPoints>1){
		model.registerScaledState(t, currentY, currentS);
		lastRecordedTime = t;
	}
	
	//run simulation
	//default time step is dt, but temporarily dt might be reduced to a smaller value
	for(recorded=1, current_dt1=current_dt2=dt, refinements=0, iterations=0; t<endTime; /* increment of t handled in loop */ ){
		// at this stage currentY is guaranteed to be a valid state
		// t should always correspond to currentY
		// t2 should always correspond to Y2
		// current_dt1 is used for moving from currentY-->Y2
		// current_dt2 is used for moving from currentY-->candidateY (= potential next currentY)
		++iterations;
		
		// check runtime-out
		if((runtime_out_seconds>0) && (iterations%100==0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation was aborted prematurely after "+makeString(iterations)+"  because it reached the maximum allowed processing time.";
			return true;
		}
		
		// reduce time step to not exceed end time, if needed
		current_dt1 = min(min(current_dt1, current_dt2), endTime-t);
				
		// get dynamics (k1) at the current state if needed
		// the dynamics will be the rate of change for the "shape" Y and for the "scale" S, i.e. symbolically k1 = (dY/dt, dS/dt)
		if(!k1AlreadyCalculated){
			dynamics = model.getRateOfChangeAtScaledState(t, currentY, currentS, kY1, kS1, jumpY, jumpS);
			if(dynamics==RequestedDynamicsForceJumpToState){
				currentY = jumpY;
				currentS = jumpS;
				t = t + current_dt1;
				goto SRK2_REGISTER_STATE; 
			}
			k1AlreadyCalculated = true;
		}
		
		// get Y2 & S2 (in forward k1-direction)
		Y2  = currentY + kY1*current_dt1;
		S2  = currentS + kS1*current_dt1;
		t2  = t + current_dt1; // t2 should always correspond to Y2
		
		// check if (Y2,S2) crossed the boundary and move (Y2,S2) backward as much as needed
		// (Y2,S2) and current_dt1 will be adjusted if this routine returns true
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentY, currentS, current_dt1, Y2, S2);
		if(crossedBoundary == CrossedBoundaryYesButFixedBruteForce){ crossedBoundaryYesButFixedBruteForce = true; }
		else if(crossedBoundary == CrossedBoundaryYesButFixedByReducingTimeStep){ crossedBoundaryButFixedByReducingTimeStep = true; }
				
		// check if the time step should be reduced (as suggested by the model)
		while((refinements<maxTimeStepRefinements) && model.checkShouldRefineTimeStep(t, currentY, currentS, current_dt1, Y2, S2)){
			current_dt1 /= 2;
			++refinements;
			Y2 = currentY + kY1*current_dt1;
			S2 = currentS + kS1*current_dt1;
			t2 = t + current_dt1; // t2 should always correspond to (Y2,S2)
		}		
		
		// get dynamics (k2) at (Y2,S2)
		dynamics = model.getRateOfChangeAtScaledState(t2, Y2, S2, kY2, kS2, jumpY, jumpS);
		if(dynamics==RequestedDynamicsForceJumpToState){
			currentY = jumpY;
			currentS = jumpS;
			t = t2;
			goto SRK2_REGISTER_STATE; 
		}

		// use k1 & k2 to move the currentY forward, potentially after refining the time step multiple times
		kYConsensus = (kY1+kY2)*0.5;
		kSConsensus = (kS1+kS2)*0.5;
		candidateY  = currentY + kYConsensus*current_dt2;
		candidateS  = currentS + kSConsensus*current_dt2;
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentY, currentS, current_dt2, candidateY, candidateS);
		if(crossedBoundary==CrossedBoundaryYesButFixedBruteForce){
			if(refinements<maxTimeStepRefinements){
				++refinements; 
				current_dt2 /= 2;
				crossedBoundaryButFixedByReducingTimeStep = true;
				continue; // repeat the whole iteration with a smaller time step
			}else{
				// use simple Euler scheme for this time step, since kConsensus throws us out of the boundary at even arbitrarily small time steps
				currentY = Y2;
				currentS = S2;
				t = t2;
				crossedBoundaryYesButFixedUsingEuler = true;
				goto SRK2_REGISTER_STATE;
			}
		}
				
		// check if the time step should be reduced (as suggested by the model)
		if((refinements<maxTimeStepRefinements) && model.checkShouldRefineTimeStep(t, currentY, currentS, current_dt2, candidateY, candidateS)){
			++refinements; 
			current_dt2 /= 2; 
			continue; // repeat the last part with a smaller time step
		}
		
		// seems everything worked out as normal, so set the new (currentY, currentS)
		currentY = candidateY;
		currentS = candidateS;
		t = t + current_dt2; // t should always correspond to (currentY, currentS)
		goto SRK2_REGISTER_STATE;
		
		// check and register new state if needed	
		SRK2_REGISTER_STATE:
		// check sanity (some ODEs may explode)		
		if(std::isnan(t) || model.scaledStateIsNaN(currentY,currentS)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation reached NaN.";
			return (recorded>1);
		}
		k1AlreadyCalculated = false;
		if((t-lastRecordedTime > recordingTimeStep) && (recorded<maxRecordedPoints-1)){ // don't record if maxRecordedPoints has been almost exhausted (i.e. 1 remaining), since the final state will be recorded outside of the loop
			model.registerScaledState(t, currentY, currentS);
			++recorded;
			lastRecordedTime = t;
			reporter(recorded, maxRecordedPoints, 1-(endTime-t)/simulationTime);
		}
		
		// initialize counters for next iteration (note: this step may be skipped by a 'continue' statement)
		current_dt1  = dt;
		current_dt2  = dt;
		refinements = 0;
	}
	
	// register final state if needed
	if(t>lastRecordedTime){
		model.registerScaledState(t, currentY, currentS);
	}
	
	// construct warnings message if needed
	if(crossedBoundaryYesButFixedUsingEuler) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally using forward Euler.";
	if(crossedBoundaryYesButFixedBruteForce) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by brute-force adjusting the trajectory.";
	if(crossedBoundaryButFixedByReducingTimeStep) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally reducing the time step.";

	return true;
}





// Two-step Runge-Kutta solver for linear ODEs where the log-scale and a scaled-version of the trajectory should be simulated instead.
// Consider the following linear ODE:
//    dX/dt = A(t)*X(t),
// where the dynamic variable X(t) is a vector or a matrix of size NR x NC, and where A(t) is an explicitly given matrix of size NR x NR.
// Here, the variable X(t) is split into two auxiliary variables, the log-scale S (S=log(mean(X))) and the "shape" Y=X/exp(S).
// Hence, any current state X is represented by a tuple (S,Y), where S is scalar and Y is of similar nature as X.
// A separation between "shape" Y and "scale" S may be especially useful if X(t) decays to zero exponentially, resulting in numerical underflow, or explodes to infinity exponentially, resulting in numerical overflow.
//
// The solver can handle:
//    1. Suggestions by the model for temporarily refined time steps
//    2. Situations where the domain boundary is crossed during an iteration (in which case the time step is temporarily decreased as needed in order to not overshoot)
//
// The shape Y must be of type COORDINATE (either a vector or matrix in row-major format), and the scale S will be of type double.
// The model must be able to provide initial conditions for X, the dynamics for X in the form of A(t) at each time point, as well as storing of the computed trajectory (defined by S and Y).
// Of course, the model may internally store the trajectory X instead of S and Y, but the integrator will always compute and provide S and Y.
//
// Requested times might reverse (i.e. times at which model dynamics are requested might not be strictly increasing), but recorded time series will be strictly forward in time
template<class COORDINATE, class MODEL, class PROGRESS_REPORTER>
bool LinearScaledRungeKutta(const long					NR,								// (INPUT) number of rows in A and X
							const long					NC,								// (INPUT) number of columns in X
							const double				startTime,						// (INPUT) simulation start time
							const double				endTime, 						// (INPUT) simulation end time
							const double 				dt, 							// (INPUT) default integration time step
							MODEL 						&model,							// (INPUT/OUTPUT) object defining model dynamics (via the matrix A), and handling time series storage
							const long					maxRecordedPoints, 				// (INPUT) if small, some intermediate trajectory points are skipped (i.e. not recorded). This might be useful if accurate integration requires a small time step, but would produce a needlesly large time series. If maxRecordedPoints==2, then only the start and end points are recorded. If maxRecordedPoints==1, then only the end point is recorded.
							const long					maxTimeStepRefinements,			// (INPUT) max allowed number of refinements of local time step when encountering invalid states. Only relevant if abortOnInvalidState==false.
							const double				refinement_factor,				// (INPUT) factor by which to refine time steps. Typical values are 2-10.
							const long 					max_exp_order,					// (INPUT) maximum polynomial order for approximating exponentials (short-term propagators). Typical values are 2-5
							const PROGRESS_REPORTER		&reporter,						// (INPUT) callback functor to handle progress reporting during simulation
							const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
							string						&warningMessage){				// (OUTPUT) will be non-empty in case of error, or if non-fatal problems occurred
	COORDINATE currentY, candidateY, Y2, initialX;
	double currentS, candidateS, S2, rescaling;
	std::vector<double> A1, A2, Aconsensus, scratch1, scratch2;
	double t=0, t2, current_dt1, current_dt2, original_dt2;
	long recorded, iterations;
	CrossedBoundary crossedBoundary;
	const double DeltaTime 			= endTime-startTime; // target duration of simulation
	const double start_runtime  	= get_thread_monotonic_walltime_seconds();
	const double min_dt 			= dt/pow(refinement_factor,maxTimeStepRefinements);
	
	// note that we internally measure time in relative time "t", i.e. starting at 0 until DeltaTime
	// this is needed to avoid floating point rounding errors when DeltaT is much smaller than startTime
	// whenever we "speak" to the model (e.g. requesting dynamics or recording a trajectory point), we transform it to real time
	
	
	// keep track of warnings
	bool crossedBoundaryButFixedByReducingTimeStep = false;
	bool crossedBoundaryYesButFixedBruteForce = false;
	bool crossedBoundaryYesButFixedUsingEuler = false;
		
	//preliminary error checking
	warningMessage = "";
	if(dt<DeltaTime*RELATIVE_EPSILON){
		warningMessage = "Time step too small";
		return false;
	}
	if(DeltaTime < dt){
		warningMessage = "Time step exceeds simulation time";
		return false;
	}
	if(maxRecordedPoints < 1){
		warningMessage = "Requested zero recorded points";
		return false;
	}
	
	const double recordingTimeStep = DeltaTime/max(1L,maxRecordedPoints-1);
	model.reserveSpaceForScaledTimeSeries(maxRecordedPoints);
	bool k1AlreadyCalculated = false;
	
	// get initial state (shape and scale)
	if(!model.getInitialState(startTime+t, initialX)){
		warningMessage = "Failed to get initial state";
		return false;
	}
	
	// create scaled version of initial state
	const double initial_mean = vector_mean(initialX);
	if(initial_mean<=0){
		warningMessage = "Initial state as negative or zero scale";
		return false;
	}
	currentY = initialX/initial_mean;
	currentS = log(initial_mean);
	
	// record initial state
	double lastRecordedTime = -INFTY_D;
	if(maxRecordedPoints>1){
		model.registerScaledState(startTime+t, currentY, currentS);
		lastRecordedTime = t;
	}
		
	//run simulation
	//default time step is dt, but temporarily dt might be reduced to a smaller value
	for(recorded=1, current_dt1=current_dt2=dt, iterations=0; t<DeltaTime; /* increment of t handled in loop */ ){
		// at this stage currentY is guaranteed to be a valid state
		// t should always correspond to currentY
		// t2 should always correspond to Y2
		// current_dt1 is used for moving from currentY-->Y2
		// current_dt2 is used for moving from currentY-->candidateY (= potential next currentY)
		++iterations;
		
		// check runtime-out
		if((runtime_out_seconds>0) && (iterations%100==0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
			warningMessage += string(warningMessage=="" ? "" : "; ") + "Simulation was aborted prematurely after "+makeString(iterations)+" iterations because it reached the maximum allowed processing time.";
			return true;
		}
		
		// reduce time step dt1 to not exceed end time nor dt2, if needed
		current_dt1 = min(min(current_dt1,current_dt2), DeltaTime-t);
				
		// get dynamics at the current time t, in the form of the matrix A(t)
		if(!k1AlreadyCalculated){
			model.getLinearDynamics(startTime+t, A1);
			k1AlreadyCalculated = true;
		}
		
		// get Y2 by applying the dynamics A: Y2 = exp(current_dt1*A)*currentY
		// don't update S yet, do so at the end of this iteration
		apply_approximate_matrix_exponential(NR, NC, A1, current_dt1, currentY, max_exp_order, scratch1, scratch2, Y2);
		S2  = currentS; // don't change scale until later
		t2  = t + current_dt1; // t2 should always correspond to Y2

		// check if Y2 crossed the boundary and move Y2 backward as much as needed
		// (Y2,S2) and current_dt1 will be adjusted if this routine returns true
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(startTime+t, currentY, currentS, current_dt1, Y2, S2, true);
		if(crossedBoundary == CrossedBoundaryYesButFixedBruteForce){ crossedBoundaryYesButFixedBruteForce = true; }
		else if(crossedBoundary == CrossedBoundaryYesButFixedByReducingTimeStep){ crossedBoundaryButFixedByReducingTimeStep = true; }
						
		// check if the time step should be reduced (as suggested by the model)
		while((current_dt1>min_dt) && model.checkShouldRefineTimeStep(startTime+t, currentY, currentS, current_dt1, Y2, S2)){
			// recalculate currentY --> Y2 with a smaller time step
			current_dt1 /= refinement_factor;
			apply_approximate_matrix_exponential(NR, NC, A1, current_dt1, currentY, max_exp_order, scratch1, scratch2, Y2);
			S2 = currentS;
			t2 = t + current_dt1; // t2 should always correspond to (Y2,S2)
		}
		
		// stage-2 time-step should not be larger than the stage-1 time-step, but also not below min_dt
		current_dt2 = max(min(min_dt,endTime-t), min(current_dt1,current_dt2));
		
		// get dynamics (matrix A) at t2
		model.getLinearDynamics(startTime+t2, A2);

		// use A1 & A2 to move the currentY forward (t --> t+current_dt2), and check if time step was sufficiently small
		linear_combination(0.5, A1, 0.5, A2, Aconsensus);
		apply_approximate_matrix_exponential(NR, NC, Aconsensus, current_dt2, currentY, max_exp_order, scratch1, scratch2, candidateY);
		candidateS = currentS; // increment scale later on

		// check if we crossed the domain boundary, and correct if needed		
		original_dt2 = current_dt2;
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(startTime+t, currentY, currentS, current_dt2, candidateY, candidateS, true);
		if(crossedBoundary==CrossedBoundaryYesButFixedBruteForce){
			if(current_dt2>min_dt){
				current_dt2 /= refinement_factor;
				crossedBoundaryButFixedByReducingTimeStep = true;
				continue; // repeat the whole iteration with a smaller time step
			}else{
				// use simple Euler scheme for this time step, since kConsensus throws us out of the boundary at even arbitrarily small time steps
				rescaling 	= vector_mean(Y2);
				currentY 	= Y2/rescaling;
				currentS 	= S2+log(rescaling); // since we are rescaling Y2, correct for this rescaling in currentS		
				t 			= t2;
				crossedBoundaryYesButFixedUsingEuler = true;
				goto SRK2_REGISTER_STATE;
			}
		}else if(crossedBoundary == CrossedBoundaryYesButFixedByReducingTimeStep){
			crossedBoundaryButFixedByReducingTimeStep = true;
		}

		// check if the time step should be reduced (as suggested by the model)
		if((current_dt2>min_dt) && model.checkShouldRefineTimeStep(startTime+t, currentY, currentS, current_dt2, candidateY, candidateS)){
			current_dt2 /= refinement_factor;
			continue; // repeat this iteration with a smaller time step, forget about the current candidateY
		}
		
		// seems everything worked out as normal, so set the new (currentY, currentS)
		rescaling = vector_mean(candidateY);
		currentY = candidateY/rescaling;
		currentS = candidateS + log(rescaling); // since we are rescaling currentY, correct the scale for this rescaling
		t = t + current_dt2; // t should always correspond to (currentY, currentS)
		goto SRK2_REGISTER_STATE;
		
		// check and register new state if needed	
		SRK2_REGISTER_STATE:
		// check sanity (some ODEs may explode)		
		if(std::isnan(t) || model.scaledStateIsNaN(currentY,currentS)){
			warningMessage += string(warningMessage=="" ? "" : "; ") + "Simulation reached NaN.";
			return (recorded>1);
		}
		k1AlreadyCalculated = false;
		if((t-lastRecordedTime > recordingTimeStep) && (recorded<maxRecordedPoints-1)){ // don't record if maxRecordedPoints has been almost exhausted (i.e. 1 remaining), since the final state will be recorded outside of the loop
			model.registerScaledState(startTime+t, currentY, currentS);
			++recorded;
			lastRecordedTime = t;
			reporter(recorded, maxRecordedPoints, 1-(DeltaTime-t)/DeltaTime);
		}
		
		// initialize counters for next iteration (note: this step may be skipped by a 'continue' statement)
		// only gradually increase the time step, to avoid repeatedly wasting time in refinements
		current_dt1  = min(refinement_factor*current_dt1,dt);
		current_dt2  = min(refinement_factor*current_dt2,dt);
	}
	
	// register final state if needed
	if(t>lastRecordedTime){
		model.registerScaledState(startTime+t, currentY, currentS);
	}

	// construct warnings message if needed
	if(crossedBoundaryYesButFixedUsingEuler) warningMessage += string(warningMessage=="" ? "" : "; ") + "Simulation crossed domain boundary and was fixed by locally using forward Euler.";
	if(crossedBoundaryYesButFixedBruteForce) warningMessage += string(warningMessage=="" ? "" : "; ") + "Simulation crossed domain boundary and was fixed by brute-force adjusting the trajectory.";
	if(crossedBoundaryButFixedByReducingTimeStep) warningMessage += string(warningMessage=="" ? "" : "; ") + "Simulation crossed domain boundary and was fixed by locally reducing the time step.";
	
	return true;
}






// Rosenbrock-Euler order-2 ODE solver for equations of the form:
//    dX/dt = f(t,X) = A(t)*X(t) + N(t,X(t)),
// where the dynamic variable X(t) is a vector or a matrix of size NR x NC, and where A(t) ("linearity") is an explicitly given matrix of size NR x NR and N(t,X(t)) is the non-linear residual.
// This is of the simplest exponential integrators. Exponential integrators are particularly suited for stiff problems, when the stiffness is caused by the linear dynamics.
// At each iteration, the integration proceeds with the following step:
//	 X(t+eps) = X(t) + [exp(eps*A(t)) - Id] * A(t)^{-1} * f(t,X(t))
// where:
//   f(t,X(t)) = [A(t)*X(t) + N(t,X(t))]
// The exponential * inverse_A term can be approximated as:
//   [exp(eps*A(t)) - Id] * A(t)^{-1} = sum_{k=0}^n (eps/(k+1)) * (eps*A(t))^k/k!
//  Here, we shall call this the Rosenbrock-Euler propagator.
//
// The solver can handle:
//    1. Suggestions by the model for temporarily refined time steps
//    2. Situations where the domain boundary is crossed during an iteration (in which case the time step is temporarily decreased as needed in order to not overshoot)
//
// The model must be able to provide initial conditions for X, the linearity A(t) and the nonlinearity N(t,X(t)) at each time point, as well as storing of the computed trajectory
// For a quick derivation of the Rosenbrock-Euler scheme see: 
//   Chen et al (2018). Exponential Rosenbrock-Euler integrators for elastodynamic simulation. IEEE transactions on visualization and computer graphics. 24:2702-2713
//
// Requested times might reverse (i.e. times at which model dynamics are requested might not be strictly increasing), but recorded time series will be strictly forward in time
template<class COORDINATE, class MODEL, class PROGRESS_REPORTER>
bool RosenbrockEuler(	const long					NR,								// (INPUT) number of rows in A and X
						const long					NC,								// (INPUT) number of columns in X
						const double				startTime,						// (INPUT) simulation start time
						const double				endTime, 						// (INPUT) simulation end time
						const double 				dt, 							// (INPUT) default integration time step
						MODEL 						&model,							// (INPUT/OUTPUT) object defining model dynamics (via the matrix A), and handling time series storage
						const long					maxRecordedPoints, 				// (INPUT) if small, some intermediate trajectory points are skipped (i.e. not recorded). This might be useful if accurate integration requires a small time step, but would produce a needlesly large time series. If maxRecordedPoints==2, then only the start and end points are recorded. If maxRecordedPoints==1, then only the end point is recorded.
						const long					maxTimeStepRefinements,			// (INPUT) max allowed number of refinements of local time step when encountering invalid states. Only relevant if abortOnInvalidState==false.
						const long 					max_exp_order,					// (INPUT) maximum polynomial order for approximating exponentials (short-term propagators). Typical values are 2-5
						const PROGRESS_REPORTER		&reporter,						// (INPUT) callback functor to handle progress reporting during simulation
						const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
						string						&warningMessage){				// (OUTPUT) will be non-empty in case of error, or if non-fatal problems occurred
	COORDINATE currentX, candidateX, X2, nonlinearity, rate;
	std::vector<double> linearity, scratch1, scratch2, default_RE_propagator;
	double t=startTime, t2, current_dt, candidate_dt;
	long recorded;
	CrossedBoundary crossedBoundary;
	const double simulationTime = endTime-startTime; // target duration of simulation
	const double start_runtime  = get_thread_monotonic_walltime_seconds();
	const double min_dt = dt/pow(2.0,maxTimeStepRefinements);
	const bool constant_linearity = model.linearDynamicsAreConstant();
	
	// keep track of warnings
	bool crossedBoundaryButFixedByReducingTimeStep = false;
	bool crossedBoundaryYesButFixedBruteForce = false;
	bool crossedBoundaryYesButFixedUsingEuler = false;
		
	//preliminary error checking
	warningMessage = "";
	if(dt<simulationTime*RELATIVE_EPSILON){
		warningMessage = "Time step too small";
		return false;
	}
	if(simulationTime < dt){
		warningMessage = "Time step exceeds simulation time";
		return false;
	}
	if(maxRecordedPoints < 1){
		warningMessage = "Requested zero recorded points";
		return false;
	}
	
	const double recordingTimeStep = simulationTime/max(1L,maxRecordedPoints-1);
	model.reserveSpaceForTimeSeries(maxRecordedPoints);
	bool k1AlreadyCalculated = false;
	
	// get initial state (shape and scale)
	if(!model.getInitialState(t, currentX)){
		warningMessage = "Failed to get initial state";
		return false;
	}
		
	// record initial state
	double lastRecordedTime = -INFTY_D;
	if(maxRecordedPoints>1){
		model.registerState(t, currentX);
		lastRecordedTime = t;
	}
	
	if(constant_linearity){
		// since linearity is always the same, precompute the default Rosenbrock-Euler propagator (i.e. for the default time step)
		model.getLinearAndNonlinearDynamics(t, currentX, linearity, nonlinearity);
		std::vector<double> identity;
		get_identity_matrix(NR,identity);
		apply_approximate_RosenbrockEuler_exponential(NR, NR, linearity, dt, identity, max_exp_order, scratch1, scratch2, default_RE_propagator);
	}
	
	//run simulation
	//default time step is dt, but temporarily dt might be reduced to a smaller value
	for(recorded=1, current_dt=dt; t<endTime; /* increment of t handled in loop */ ){
		// at this stage currentY is guaranteed to be a valid state
		// t should always correspond to currentY
		
		// check runtime-out
		if((runtime_out_seconds>0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation was aborted prematurely because it reached the maximum allowed processing time.";
			return true;
		}
		
		// reduce time step to not exceed end time, if needed
		current_dt = min(current_dt, endTime-t);
						
		// get dynamics at the current time t, in the form of the linearity (matrix) and nonlinearity
		if(!k1AlreadyCalculated){
			model.getLinearAndNonlinearDynamics(t, currentX, linearity, nonlinearity);
			k1AlreadyCalculated = true;
		}
		
		// get rate of change: rate = linearity*currentX + nonlinearity
		multiply_matrices(NR,NR,NC,linearity,currentX,rate);
		rate += nonlinearity;
		
		// get X2 by applying the Rosenbrock-Euler propagator
		// X2 = currentX + [exp(current_dt*linearity)-Id]*linearity^{-1}*rate
		// t2 should always correspond to X2
		if(constant_linearity && (abs(current_dt-dt)<RELATIVE_EPSILON*dt)){
			multiply_matrices(NR,NR,NC,default_RE_propagator,rate,X2);
		}else{
			apply_approximate_RosenbrockEuler_exponential(NR, NC, linearity, current_dt, rate, max_exp_order, scratch1, scratch2, X2);
		}
		X2 += currentX;
		t2  = t + current_dt; 
						
		// check if the time step should be reduced (as suggested by the model)
		while((current_dt>min_dt) && model.checkShouldRefineTimeStep(t, currentX, current_dt, X2)){
			// recalculate currentX --> X2 with a smaller time step
			current_dt /= 2;
			apply_approximate_RosenbrockEuler_exponential(NR, NC, linearity, current_dt, rate, max_exp_order, scratch1, scratch2, X2);
			X2 += currentX;
			t2  = t + current_dt; // t2 should always correspond to X2
		}
		
		// check if we crossed the domain boundary, and correct if needed
		// candidate_dt should always correspond to candidateX
		candidateX 		= X2; // candidateX and current_dt may be modified by the model below
		candidate_dt 	= current_dt;
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentX, candidate_dt, candidateX);
		if(crossedBoundary==CrossedBoundaryYesButFixedBruteForce){
			if(current_dt>min_dt){
				current_dt /= 2;
				crossedBoundaryButFixedByReducingTimeStep = true;
				continue; // repeat the whole iteration with a smaller time step
			}else{
				// no more time step refinements allowed, and crossing the boundary at even infinitesimal step, so just register candidateX
				currentX 	= candidateX;
				t 			= t2;
				crossedBoundaryYesButFixedUsingEuler = true;
				goto RBE_REGISTER_STATE;
			}
		}else if(crossedBoundary==CrossedBoundaryYesButFixedByReducingTimeStep){
		}
		
		// seems everything worked out as normal, so set the new currentX
		// t should always correspond to currentX
		currentX = candidateX;
		t = t + candidate_dt;
		
		// check and register new state if needed	
		RBE_REGISTER_STATE:
		// check sanity (some ODEs may explode)		
		if(std::isnan(t) || model.stateIsNaN(currentX)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation reached NaN.";
			return (recorded>1);
		}
		k1AlreadyCalculated = false;
		if((t-lastRecordedTime > recordingTimeStep) && (recorded<maxRecordedPoints-1)){ // don't record if maxRecordedPoints has been almost exhausted (i.e. 1 remaining), since the final state will be recorded outside of the loop
			model.registerState(t, currentX);
			++recorded;
			lastRecordedTime = t;
			reporter(recorded, maxRecordedPoints, 1-(endTime-t)/simulationTime);
		}
		
		// initialize counters for next iteration (note: this step may be skipped by a 'continue' statement)
		current_dt  = dt;
	}
	
	// register final state if needed
	if(t>lastRecordedTime){
		model.registerState(t, currentX);
	}

	// construct warnings message if needed
	if(crossedBoundaryYesButFixedUsingEuler) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally using forward Euler.";
	if(crossedBoundaryYesButFixedBruteForce) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by brute-force adjusting the trajectory.";
	if(crossedBoundaryButFixedByReducingTimeStep) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally reducing the time step.";
	
	return true;
}






// Rosenbrock-Euler integrator nested into two-stage Runge-Kutta scheme for solving ODEs of the format:
//    dX/dt = f(t,X) = A(t)*X(t) + N(t,X(t)),
// where the dynamic variable X(t) is a vector or a matrix of size NR x NC, and where A(t) ("linearity") is an explicitly given matrix of size NR x NR and N(t,X(t)) is the non-linear residual.
// This solver uses an exponential integrator for each stage, and includes two stages per iteration. 
// Exponential integrators are particularly suited for stiff problems, when the stiffness is caused by the linear dynamics.
// At each stage, the integration proceeds with the following step:
//	 X(t+eps) = X(t) + [exp(eps*A(t)) - Id] * A(t)^{-1} * f(t,X(t))
// where:
//   f(t,X(t)) = [A(t)*X(t) + N(t,X(t))]
// The exponential * inverse_A term can be approximated as:
//   [exp(eps*A(t)) - Id] * A(t)^{-1} = sum_{k=0}^n (eps/(k+1)) * (eps*A(t))^k/k!
//  Here, we shall call this the Rosenbrock-Euler propagator.
//
// The solver can handle:
//    1. Suggestions by the model for temporarily refined time steps
//    2. Situations where the domain boundary is crossed during an iteration (in which case the time step is temporarily decreased as needed in order to not overshoot)
//
// The model must be able to provide initial conditions for X, the linearity A(t) and the nonlinearity N(t,X(t)) at each time point, as well as storing of the computed trajectory
// For a quick derivation of the Rosenbrock-Euler scheme see: 
//   Chen et al (2018). Exponential Rosenbrock-Euler integrators for elastodynamic simulation. IEEE transactions on visualization and computer graphics. 24:2702-2713
//
// Requested times might reverse (i.e. times at which model dynamics are requested might not be strictly increasing), but recorded time series will be strictly forward in time
template<class COORDINATE, class MODEL, class PROGRESS_REPORTER>
bool RosenbrockEulerRungeKutta2(	const long					NR,								// (INPUT) number of rows in A and X
									const long					NC,								// (INPUT) number of columns in X
									const double				startTime,						// (INPUT) simulation start time
									const double				endTime, 						// (INPUT) simulation end time
									const double 				dt, 							// (INPUT) default integration time step
									MODEL 						&model,							// (INPUT/OUTPUT) object defining model dynamics (via the matrix A), and handling time series storage
									const long					maxRecordedPoints, 				// (INPUT) if small, some intermediate trajectory points are skipped (i.e. not recorded). This might be useful if accurate integration requires a small time step, but would produce a needlesly large time series. If maxRecordedPoints==2, then only the start and end points are recorded. If maxRecordedPoints==1, then only the end point is recorded.
									const long					maxTimeStepRefinements,			// (INPUT) max allowed number of refinements of local time step when encountering invalid states. Only relevant if abortOnInvalidState==false.
									const double				refinement_factor,				// (INPUT) factr by which to refine time steps
									const long 					max_exp_order,					// (INPUT) maximum polynomial order for approximating exponentials (short-term propagators). Typical values are 2-5
									const PROGRESS_REPORTER		&reporter,						// (INPUT) callback functor to handle progress reporting during simulation
									const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
									string						&warningMessage){				// (OUTPUT) will be non-empty in case of error, or if non-fatal problems occurred
	COORDINATE currentX, candidateX, X2, NL1, NL2, rate1, rate2, Rconsensus;
	std::vector<double> A1, A2, Aconsensus, scratch1, scratch2, default_RE_propagator;
	double t=startTime, t2, current_dt1, current_dt2;
	long recorded;
	CrossedBoundary crossedBoundary;
	const double simulationTime 	= endTime-startTime; // target duration of simulation
	const double start_runtime  	= get_thread_monotonic_walltime_seconds();
	const double min_dt 			= dt/pow(refinement_factor,maxTimeStepRefinements);
	const bool constant_linearity 	= model.linearDynamicsAreConstant();
	
	// keep track of warnings
	bool crossedBoundaryButFixedByReducingTimeStep = false;
	bool crossedBoundaryYesButFixedBruteForce = false;
	bool crossedBoundaryYesButFixedUsingEuler = false;
		
	//preliminary error checking
	warningMessage = "";
	if(dt<simulationTime*RELATIVE_EPSILON){
		warningMessage = "Time step too small";
		return false;
	}
	if(simulationTime < dt){
		warningMessage = "Time step exceeds simulation time";
		return false;
	}
	if(maxRecordedPoints < 1){
		warningMessage = "Requested zero recorded points";
		return false;
	}
	
	const double recordingTimeStep = simulationTime/max(1L,maxRecordedPoints-1);
	model.reserveSpaceForTimeSeries(maxRecordedPoints);
	bool k1AlreadyCalculated = false;
	
	// get initial state (shape and scale)
	if(!model.getInitialState(t, currentX)){
		warningMessage = "Failed to get initial state";
		return false;
	}
		
	// record initial state
	double lastRecordedTime = -INFTY_D;
	if(maxRecordedPoints>1){
		model.registerState(t, currentX);
		lastRecordedTime = t;
	}
	
	if(constant_linearity){
		// since linearity is always the same, precompute the default Rosenbrock-Euler propagator (i.e. for the default time step)
		model.getLinearAndNonlinearDynamics(t, currentX, A1, NL1);
		std::vector<double> identity;
		get_identity_matrix(NR,identity);
		apply_approximate_RosenbrockEuler_exponential(NR, NR, A1, dt, identity, max_exp_order, scratch1, scratch2, default_RE_propagator);
	}
	
	//run simulation
	//default time step is dt, but temporarily dt might be reduced to a smaller value
	for(recorded=1, current_dt1=current_dt2=dt; t<endTime; /* increment of t handled in loop */ ){
		// at this stage currentY is guaranteed to be a valid state
		// t should always correspond to currentY
		// t2 should always correspond to X2
		// current_dt1 is used for moving from currentX-->X2, which is an intermediate point for obtaining a second "take" on the local dynamics
		// current_dt2 is used for moving from currentX-->candidateX (= potential next currentX), based on the dynamics averaged between currentX and X2
		
		// check runtime-out
		if((runtime_out_seconds>0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation was aborted prematurely because it reached the maximum allowed processing time.";
			return true;
		}
		
		// reduce time step dt1 to not exceed dt2, nor overshoot past endTime
		current_dt1 = min(min(current_dt1,current_dt2), endTime-t);
		
		// get dynamics at the current time t, in the form of the linearity (matrix) and nonlinearity
		// the rate of change will be: rate = linearity*currentX + nonlinearity
		if(!k1AlreadyCalculated){
			model.getLinearAndNonlinearDynamics(t, currentX, A1, NL1);
			k1AlreadyCalculated = true;
			multiply_matrices(NR,NR,NC,A1,currentX,rate1);
			rate1 += NL1;
		}
		
		// get X2 by applying the Rosenbrock-Euler propagator
		if(constant_linearity && (abs(current_dt1-dt)<RELATIVE_EPSILON*dt)){
			multiply_matrices(NR,NR,NC,default_RE_propagator,rate1,X2);
		}else{
			apply_approximate_RosenbrockEuler_exponential(NR, NC, A1, current_dt1, rate1, max_exp_order, scratch1, scratch2, X2);
		}
		X2 += currentX;
		t2  = t + current_dt1; // t2 should always correspond to X2

		// check if X2 crossed the boundary and move X2 backward as much as needed
		// X2 and current_dt1 may be adjusted if this routine returns something other than CrossedBoundaryNo
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentX, current_dt1, X2);
		if(crossedBoundary == CrossedBoundaryYesButFixedBruteForce){ crossedBoundaryYesButFixedBruteForce = true; }
		else if(crossedBoundary == CrossedBoundaryYesButFixedByReducingTimeStep){ crossedBoundaryButFixedByReducingTimeStep = true; }
						
		// check if the time step should be reduced (as suggested by the model)
		while((current_dt1>min_dt) && model.checkShouldRefineTimeStep(t, currentX, current_dt1, X2)){
			// recalculate currentX --> X2 with a smaller time step
			current_dt1 /= refinement_factor;
			apply_approximate_RosenbrockEuler_exponential(NR, NC, A1, current_dt1, rate1, max_exp_order, scratch1, scratch2, X2);
			X2 += currentX;
			t2  = t + current_dt1; // t2 should always correspond to X2
		}
		
		// stage-2 time-step should not be larger than the stage-1 time-step, but also not below min_dt
		current_dt2 = max(min(min_dt,endTime-t), min(current_dt1,current_dt2));
				
		// get dynamics at t2
		model.getLinearAndNonlinearDynamics(t2, X2, A2, NL2);
		multiply_matrices(NR,NR,NC,A2,X2,rate2);
		rate2 += NL2;

		// use rate1 & rate2 to move the currentX forward, and check if time step was sufficiently small
		linear_combination(0.5, A1, 0.5, A2, Aconsensus);
		linear_combination(0.5, rate1, 0.5, rate2, Rconsensus);
		if(constant_linearity && (abs(current_dt2-dt)<RELATIVE_EPSILON*dt)){
			multiply_matrices(NR,NR,NC,default_RE_propagator,Rconsensus,candidateX);
		}else{
			apply_approximate_RosenbrockEuler_exponential(NR, NC, Aconsensus, current_dt2, Rconsensus, max_exp_order, scratch1, scratch2, candidateX);
		}
		candidateX += currentX;

		// check if we crossed the domain boundary, and correct if needed
		crossedBoundary = model.checkCrossedDomainBoundaryAndFix(t, currentX, current_dt2, candidateX);
		if(crossedBoundary==CrossedBoundaryYesButFixedBruteForce){
			if(current_dt2>min_dt){
				current_dt2 /= refinement_factor;
				crossedBoundaryButFixedByReducingTimeStep = true;
				continue; // repeat the whole iteration with a smaller time step
			}else{
				// use prediction from 1-stage Rosenbrock-Euler scheme for this time step, since kConsensus throws us out of the boundary at even arbitrarily small time steps
				currentX	= X2;
				t 			= t2;
				crossedBoundaryYesButFixedUsingEuler = true;
				goto SRERK2_REGISTER_STATE;
			}
		}

		// check if the time step should be reduced (as suggested by the model)
		if((current_dt2>min_dt) && model.checkShouldRefineTimeStep(t, currentX, current_dt2, candidateX)){
			current_dt2 /= refinement_factor;
			continue; // repeat this iteration with a smaller time step, forget about the current candidateX
		}
		
		// seems everything worked out as normal, so set the new currentX
		currentX = candidateX;
		t = t + current_dt2; // t should always correspond to (currentX, currentS)
		goto SRERK2_REGISTER_STATE;
		
		// check and register new state if needed	
		SRERK2_REGISTER_STATE:
		// check sanity (some ODEs may explode)		
		if(std::isnan(t) || model.stateIsNaN(currentX)){
			warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation reached NaN.";
			return (recorded>1);
		}
		k1AlreadyCalculated = false;
		if((t-lastRecordedTime > recordingTimeStep) && (recorded<maxRecordedPoints-1)){ // don't record if maxRecordedPoints has been almost exhausted (i.e. 1 remaining), since the final state will be recorded outside of the loop
			model.registerState(t, currentX);
			++recorded;
			lastRecordedTime = t;
			reporter(recorded, maxRecordedPoints, 1-(endTime-t)/simulationTime);
		}
		
		// initialize counters for next iteration (note: this step may be skipped by a 'continue' statement)
		current_dt1  = dt;
		current_dt2  = dt;
	}
	
	// register final state if needed
	if(t>lastRecordedTime){
		model.registerState(t, currentX);
	}

	// construct warnings message if needed
	if(crossedBoundaryYesButFixedUsingEuler) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally using 1-stage Rosenbrock-Euler.";
	if(crossedBoundaryYesButFixedBruteForce) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by brute-force adjusting the trajectory.";
	if(crossedBoundaryButFixedByReducingTimeStep) warningMessage += string(warningMessage=="" ? "" : "\n") + "Simulation crossed domain boundary and was fixed by locally reducing the time step.";
	
	return true;
}




// given some continuous scalar univariate function f:x->y, solve the equation f(x)=V within some interval [xmin:xmax] using the bisection method
// this function assumes that sign(f(xmin)-V)!=sign(f(xmax)-V), so that the solution is surely enclosed within the interval
template<class FUNCTOR>
MathError solve_via_bisection(	const FUNCTOR	&f,
								const double	Xmin,				// (INPUT) the left boundary of the interval within which to search. Assumes that sign(f(xmin)-V) != sign(f(xmax)-V)
								const double	Xmax,				// (INPUT) the right boundary of the interval within which to search. Assumes that sign(Y(xmin)-V) != sign(Y(xmax)-V)
								const double	Xstart,				// (INPUT) reasonable first guess
								const double	V,					// (INPUT) the target value, i.e. solve for f(x)=V
								const double	xepsilon,			// (INPUT) absolute error tolerance on an x-scale. If the x-step is within this threshold, the search is halted
								const double	yepsilon,			// (INPUT) absolute error tolerance on a y-scale. If |y-V| is within this threshold, the search is halted
								const long		max_iterations,		// (INPUT) maximum number of iterations allowed, i.e. prior to halting the search
								double			&root){				// (OUTPUT) the root found, i.e. where f becomes zero. Only well-defined if MathErrorNone or MathErrorMaxIterationsReached is returned.
	double a = Xmin, b = Xmax, c = NAN_D;
	double fa, fb, fc;
	root = NAN_D;

	fa = f(a);
	fb = f(b);
	if(abs(fa-V)<=yepsilon){
		root = a;
		return MathErrorNone;
	}
	if(abs(fb-V)<=yepsilon){
		root = b;
		return MathErrorNone;
	}
	if(sgn(fa-V)==sgn(fb-V)) return MathErrorBadBounds;

	long iterations = 0;
	while(iterations<max_iterations){
		c  = (std::isnan(c) ? Xstart : 0.5*(a+b));
		//c  = (std::isnan(c) ? Xstart : (a + (V-fa) * (b-a)/(fb-fa))); // guess next c based on linear interpolation between (a,fa) and (b,fb)
		fc = f(c);
		if((fc==V) || (abs(b-a)<=xepsilon) || (abs(fc-V)<=yepsilon)){
			root = c;
			return MathErrorNone;
		}
		if(sgn(fc-V)==sgn(fa-V)){
			a  = c;
			fa = fc;
		}else{
			b  = c;
			fb = fc;
		}
		++iterations;
	}
	// did not converge, but the last c is the best estimate
	root = c;
	return MathErrorMaxIterationsReached;
}			



// given some continuous scalar univariate function f:x->y, maximize f(x) within some interval [xmin:xmax] using the golden ratio interval elimination method
// This method is able to find optima in the interior as well as on the boundary of the provided interval
// Background on the method: https://en.wikipedia.org/wiki/Golden-section_search
// The search finishes if one of the stopping criteria is fulfilled:
//	  abs_xepsilon
//	  rel_xepsilon
//	  max_iterations
template<class FUNCTOR>
double optimize_via_golden_ratio(	const FUNCTOR	&f,
									const double	Xmin,				// (INPUT) the left boundary of the interval within which to search
									const double	Xmax,				// (INPUT) the right boundary of the interval within which to search
									const bool		minimize,			// (INPUT) whether the function f shall be minimized or maximized
									const double	abs_xepsilon,		// (INPUT) absolute error tolerance on an x-scale. If the x-step is within this threshold, the search is halted
									const double	rel_xepsilon,		// (INPUT) relative error tolerance on an x-scale. If the x-step is within this threshold (relative to the typical scale of the current bracet), the search is halted
									const long		max_iterations){	// (INPUT) maximum number of iterations allowed, i.e. prior to halting the search
	double a = min(Xmin,Xmax), b = max(Xmin,Xmax);
	double h = b-a;
    if((h <= abs_xepsilon) || (h <= rel_xepsilon*0.5*(abs(a)+abs(b)))) return 0.5*(a+b); // initial bracket is already below tolerance

	// prepare Golden ratio constants
	const double invphi  = 0.5*(sqrt(5) - 1); // 1/phi
	const double invphi2 = 0.5*(3 - sqrt(5)); // 1/phi^2

    double c  = a + invphi2 * h;
    double d  = a + invphi * h;
    double fc = f(c);
    double fd = f(d);
	if(std::isnan(fd) || std::isnan(fc)) return NAN_D;

	long iterations=0;
	while(iterations<max_iterations){
		if((minimize && (fc<fd)) || ((!minimize) && (fc>fd))){
			b 	= d;
			d 	= c;
			fd 	= fc;
			h 	= invphi * h;
			c 	= a + invphi2 * h;
			fc 	= f(c);
			if(std::isnan(fc)) return NAN_D;
		}else{
			a 	= c;
			c 	= d;
			fc 	= fd;
			h 	= invphi * h;
			d 	= a + invphi * h;
			fd 	= f(d);
			if(std::isnan(fd)) return NAN_D;
		}
	    if((h <= abs_xepsilon) || (h <= rel_xepsilon*0.5*(abs(b)+abs(c)))){
	    	return (((minimize && (fc<fd)) || ((!minimize) && (fc>fd))) ? 0.5*(a+d) : 0.5*(c+b));
	    }
		++iterations;
	}
	return (((minimize && (fc<fd)) || ((!minimize) && (fc>fd))) ? 0.5*(a+d) : 0.5*(c+b));
}	



class ProgressReporter{
private:
	long reportCount;
	string prefix, suffix;
	bool asPercentage;
	mutable long lastReportedCase;
	mutable double lastReportedFraction;
	bool silent;
public:
	ProgressReporter(const bool silent_){ reportCount = 10; lastReportedCase=-1; lastReportedFraction = 0; silent = silent_; }
	ProgressReporter(long reportCount_){ reportCount = reportCount_; lastReportedCase=-1; lastReportedFraction=0; }
	ProgressReporter(long reportCount_, const string &prefix_, const string &suffix_, bool asPercentage_){ 
		reportCount = reportCount_; prefix = prefix_; suffix = suffix_; asPercentage = asPercentage_; lastReportedCase = -1; lastReportedFraction = 0; 
	}
	
	void setReportCount(long count){ reportCount = count; }
	void setPrefix(const string &prefix_){ prefix = prefix_; }
	void setSuffix(const string &suffix_){ suffix = suffix_; }
	void setAsPercentage(bool asPercentage_){ asPercentage = asPercentage_; }
	void reset(){ lastReportedCase = -1; lastReportedFraction = 0; }
	
	void operator()(long casesFinished, long totalCases, double exactFraction) const{
		if(reportCount<=0) return;
		if(silent) return;
		const double p = (1.0*casesFinished)/totalCases;
		const double last_p = (1.0*lastReportedCase)/totalCases;
		const double dp = 1.0/reportCount;
		if((lastReportedCase>=0) && (floor(p/dp)<=floor(last_p/dp))) return; //don't report this case
		if(floor(p/dp)==0) return;
		lastReportedCase 	 = casesFinished;
		lastReportedFraction = exactFraction;
		const long rounder = pow(10.0,1+log10(max(reportCount,1l)));
		Rcout << prefix;
		if(asPercentage){ Rcout << long(rounder*100.0*exactFraction)/rounder << " %"; }
		else{ Rcout << long(floor(exactFraction*totalCases)) << " out of " << totalCases; }
		Rcout << suffix;
	}
	
	
	void operator()(double rangeFinished, double totalRange) const{
		if(reportCount<=0) return;
		if(silent) return;
		const double fractionFinished = rangeFinished/totalRange;
		const double dfraction = 1.0/reportCount; // targetted fraction step between recordings
		if((floor(fractionFinished/dfraction)<=floor(lastReportedFraction/dfraction))) return; //don't report this case
		if(floor(fractionFinished/dfraction)==0) return;
		lastReportedFraction = fractionFinished;
		const long rounder = pow(10.0,1+log10(max(reportCount,1l))); // auxiliary factor for rounding reported fractions on printout
		Rcout << prefix;
		if(asPercentage){ Rcout << long(rounder*100.0*fractionFinished)/rounder << " %"; }
		else{ Rcout << rangeFinished << " out of " << totalRange; }
		Rcout << suffix;
	}
};






#pragma mark -
#pragma mark Time series analysis
#pragma mark 



//Returns linear regression fit (y=trafo*x + offset) for points passed
//Algorithm tries to minimize \sum_i (y_i-y_{fit})^2
MathError fitLinearRegression(const double pointsX[], const double pointsY[], const long count, double &slope, double &offset){
	if(count == 0){
		slope = offset = NAN_D;
		return MathErrorNoData;
	}else if(count == 1){
		slope = offset = NAN_D;
		return MathErrorUndefined;
	}
	
	double sumX = 0, sumY = 0, sumXY = 0, sumX2 = 0;
	for(long i=0; i<count; ++i){
		sumX  += pointsX[i];
		sumY  += pointsY[i];
		sumXY += pointsX[i]*pointsY[i];
		sumX2 += pointsX[i]*pointsX[i];
	}
	slope = (count*sumXY - sumX*sumY)/(count*sumX2 - SQ(sumX));
	offset = sumY/count - slope*sumX/count;
	return MathErrorNone;
}



MathError fitLinearRegressionNANSensitive(const double pointsX[], const double pointsY[], const long count, double &slope, double &offset){
	double sumX = 0, sumY = 0, sumXY = 0, sumX2 = 0;
	long counted = 0;
	for(long i=0; i<count; ++i){
		if(std::isnan(pointsX[i]) || std::isnan(pointsY[i])) continue;
		++counted;
		sumX  += pointsX[i];
		sumY  += pointsY[i];
		sumXY += pointsX[i]*pointsY[i];
		sumX2 += pointsX[i]*pointsX[i];
	}
	if(counted == 0){
		slope = offset = NAN_D;
		return MathErrorNoData;
	}else if(counted == 1){
		slope = offset = NAN_D;
		return MathErrorUndefined;
	}
	slope = (count*sumXY - sumX*sumY)/(count*sumX2 - SQ(sumX));
	offset = sumY/count - slope*sumX/count;
	return MathErrorNone;
}



template<class TYPE_X, class TYPE_Y>
TYPE_Y interpolate_linear(const TYPE_X &x1, const TYPE_Y &y1, const TYPE_X &x2, const TYPE_Y &y2, const TYPE_X &x){
	if(x1 == x2){
		return 0.5*(y1+y2);
	}else{
		return y1 + (x-x1)*(y2-y1)/(x2-x1);
	}
}




// linearly interpolates an 'old' time series at new time points
// only values from includedNewTimesStart to includedNewTimesEnd (inclusive) will be well-defined; outside values are unpredictable
template<class TYPE, class TIME_ARRAY_TYPE2>
bool interpolateTimeSeriesAtTimes(	const std::vector<double> 	&oldTimes,					// (INPUT)
									const std::vector<TYPE> 	&valuesAtOldTimes,			// (INPUT)
									long						oldStart,
									long						oldEnd,
									const TIME_ARRAY_TYPE2		&newTimes,					// (INPUT)
									long						newStart,
									long						newEnd,
									long						&includedNewTimesStart, 	// (OUTPUT)
									long						&includedNewTimesEnd,	 	// (OUTPUT)
									std::vector<TYPE>			&valuesAtNewTimes){		 	// (OUTPUT)
	valuesAtNewTimes.clear();
	if((newStart<0) || (newStart>newEnd) || (oldStart<0) || (oldStart>oldEnd)) return true;
	if((oldTimes[oldEnd]<newTimes[newStart]) || (newTimes[newEnd]<oldTimes[oldStart])) return true;

	// find first time point to interpolate at
	long tpNew = newStart;
	while((tpNew<=newEnd) && (newTimes[tpNew]<oldTimes[oldStart])) ++tpNew;
	if((tpNew>newEnd) || (newTimes[tpNew]>oldTimes[oldEnd])) return true; // new time domain does not overlap with old time domain
	includedNewTimesStart = tpNew;
	
	// find last time point to interpolate at (mainly for space pre-allocation purposes)
	long tpLastNew = newEnd;
	while((tpLastNew>=tpNew) && (newTimes[tpLastNew]>oldTimes[oldEnd])) --tpLastNew;
	if((tpLastNew<tpNew) || (newTimes[tpLastNew]<oldTimes[oldStart])) return true; // new time domain does not overlap with old time domain
	includedNewTimesEnd = tpLastNew;
	
	valuesAtNewTimes.resize(newTimes.size());
	
	long tpOld = oldStart;
	double time;
	for(; tpNew<=tpLastNew; ++tpNew){
		time = newTimes[tpNew];
		
		// at this point it is guaranteed that   oldTimes[tpOld]<=time   and that oldTimes[oldEnd]>=time
		
		//find left-closest oldTime to time
		while((tpOld+1<=oldEnd) && (oldTimes[tpOld+1]<=time)){ ++tpOld; }
		
		// at this point it is guaranteed that either (tpOld==oldEnd and oldTimes[tpOld]==time), or (tpOld<oldEnd and oldTimes[tpOld]<=time and oldTimes[tpOld+1]>time)
				
		// interpolate old series at focal time
		valuesAtNewTimes[tpNew] = (tpOld==oldEnd ? valuesAtOldTimes[oldEnd] : interpolate_linear(oldTimes[tpOld], valuesAtOldTimes[tpOld], oldTimes[tpOld+1], valuesAtOldTimes[tpOld+1], time));
	}
	return true;
}



// refine time series onto a finer time-grid using linear interpolation between the original points, such that the new (refined) time series satisfies the conditions:
//  1. refined_times[r+1]-refined_times[r] <= max_time_step
//  2. abs(refined_values[r+1]-refined_values[r]) <= max_value_step
//  3. abs(refined_values[r+1]-refined_values[r])/(0.5*(abs(coarse_values[c+1])+abs(coarse_values[c]))) <= max_relative_value_step, where coarse_times[c]:coarse_times[c+1] is the coarse time interval enclosing the refined time interval refined_times[r]:refined_times[r+1]
// and such that the new time series includes the original time points (i.e. is a true super-set of the original time series)
// This function may be useful when a piecewise linear function is non-linearly transformed and then numerically integrated using the trapezoid integration scheme
void refine_time_series_linear(	const std::vector<double> 	&coarse_times,				// (INPUT) 1D array of size NC, listing ascending time points of the original (coarse) time series
								const std::vector<double> 	&coarse_values,				// (INPUT) 1D array of size NC, listing values of the original (coarse) time series
								const double				start_time,					// (INPUT) start time for the refined time series. Must not be greater than coarse_times.back(). To surely include the left-most end of the input time series, you can set end_time=-INFTY_D.
								const double				end_time,					// (INPUT) end time for the refined time series. Must not be smaller than coarse_times[0]. To surely include the right-most end of the input time series, you can set end_time=INFTY_D.
								const double 				max_time_step,				// (INPUT) optional constraint on the time steps of the refined time series; to omit this constraint set this to INFTY_D
								const double 				max_value_step,				// (INPUT) optional constraint on the value steps of the refined time series; to omit this constraint set this to INFTY_D
								const double 				max_relative_value_step,	// (INPUT) optional constraint on the relative value steps of the refined time series; to omit this constraint set this to INFTY_D
								std::vector<double> 		&refined_times,				// (OUTPUT) 1D array of size NR, listing ascending time points of the new (refined) time series
								std::vector<double> 		&refined_values){			// (OUTPUT) 1D array of size NR, listing values of the new (refined) time series
	const long NC = coarse_times.size();
	refined_times.clear();
	refined_values.clear();
	if(start_time>coarse_times.back()) return;
	if(end_time<coarse_times[0]) return;
	const long start_c = (start_time<coarse_times[0] ? 0 : find_next_left_grid_point(coarse_times, start_time, 0)); 		// coarse-grid point immediately below (or equal) to start_time
	const long end_c   = (end_time>coarse_times.back() ? NC-1 : find_next_right_grid_point(coarse_times, end_time, NC-1));	// coarse-grid point immediately above (or equal) to end_time

	// determine size of refined time series
	std::vector<long> dN(NC-1,0); // dN[c] will be the resolution (number of additional time points) of the refine time series within the coarse time interval coarse_times[c]:coarse_times[c+1]. A dN of 0 means the particular coarse time interval does not need to be refined.
	double left_time, right_time, left_value, right_value;
	for(long c=start_c; c<end_c; ++c){
		left_time  	= max(start_time, coarse_times[c]);
		right_time 	= min(end_time, coarse_times[c+1]);
		left_value  = (left_time==coarse_times[c] ? coarse_values[c] : interpolate_linear(coarse_times[c], coarse_values[c], coarse_times[c+1], coarse_values[c+1], left_time));
		right_value = (right_time==coarse_times[c+1] ? coarse_values[c+1] : interpolate_linear(coarse_times[c], coarse_values[c], coarse_times[c+1], coarse_values[c+1], right_time));
		if((right_time-left_time)>max_time_step)	dN[c] = max(dN[c], long(ceil((right_time-left_time)/max_time_step))-1);
		if((right_value-left_value)>max_value_step) 	dN[c] = max(dN[c], long(ceil((right_value-left_value)/max_value_step))-1);
		if(!std::isinf(max_relative_value_step)){
			const double max_step = max_relative_value_step * 0.5*(abs(right_value)+abs(left_value));
			if((right_value-left_value)>max_step) dN[c] = max(dN[c], long(ceil((right_value-left_value)/max_step))-1);			
		}
	}
	const long NR = (end_c-start_c+1) + vector_sum(dN);

	// construct refined time series
	refined_times.resize(NR);
	refined_values.resize(NR);
	double lambda;
	for(long c=start_c, r=0; c<end_c; ++c){
		// refine time interval left_time:right_time
		left_time  = max(start_time, coarse_times[c]);
		right_time = min(end_time, coarse_times[c+1]);
		double refined_time_step = (right_time-left_time)/(1+dN[c]);
		for(long n=0; n<dN[c]+1; ++n){
			refined_times[r]  	= left_time + n*refined_time_step;
			lambda 				= (refined_times[r]-coarse_times[c])/(coarse_times[c+1]-coarse_times[c]);
			refined_values[r]	= coarse_values[c] * (1-lambda) + coarse_values[c+1] * lambda;
			++r;
		}
	}
	refined_times[NR-1] = min(end_time,coarse_times[end_c]);
	if(end_c>0){
		refined_values[NR-1] = interpolate_linear(coarse_times[end_c-1],coarse_values[end_c-1],coarse_times[end_c],coarse_values[end_c],refined_times[NR-1]);
	}else{
		refined_values[NR-1] = coarse_values[end_c];
	}
}



// refine a piecewise polynomial curve onto a finer time-grid, such that the new (refined) spline satisfies the conditions:
//  1. refined_times[r+1]-refined_times[r] <= max_time_step
//  2. abs(refined_values[r+1]-refined_values[r]) <= max_value_step
//  3. abs(refined_values[r+1]-refined_values[r])/(0.5*(abs(coarse_values[c+1])+abs(coarse_values[c]))) <= max_relative_value_step, where coarse_times[c]:coarse_times[c+1] is the coarse time interval enclosing the refined time interval refined_times[r]:refined_times[r+1]
// and such that the new time series includes the original time points (i.e. is a true super-set of the original time series).
// This function may be useful when a piecewise linear function is non-linearly transformed and then numerically integrated using the trapezoid integration scheme
void refine_piecewise_polynomial(	const long		degree,						// (INPUT) polynomial degree of the spline
									const dvector 	&coarse_times,				// (INPUT) 1D array of size NC, listing ascending time points of the original (coarse) spline
									const dvector 	&coarse_coeff,				// (INPUT) 2D array of size NC x (degree+1), in row-major format, listing polynomial coefficients of the original (coarse) spline
									const bool		slideX,						// (INPUT) whether the polynomial is defined as a function of the sliding-time, i.e. in the interval coarse_times[i]:coarse_times[i+1] Y is defined as a polynomial of (X-coarse_times[i]) rather than X. Note that the returned refined piecewise polynomial will also be defined w.r.t. the sliding-time (on the refined time grid)
									const double	start_time,					// (INPUT) start time for the refined time series. Must not be greater than coarse_times.back(). To surely include the left-most end of the input time series, you can set end_time=-INFTY_D.
									const double	end_time,					// (INPUT) end time for the refined time series. Must not be smaller than coarse_times[0]. To surely include the right-most end of the input time series, you can set end_time=INFTY_D.
									const dvector	&max_time_steps,			// (INPUT) optional constraint on the time steps of the refined time series; to omit this constraint set this to INFTY_D. This can either be empty, or a single number or a 1D array of size NC (i.e. specifying the maximum time step for each coarse time interval)
									const double 	max_value_step,				// (INPUT) optional constraint on the value steps of the refined time series; to omit this constraint set this to INFTY_D
									const double 	max_relative_value_step,	// (INPUT) optional constraint on the relative value steps of the refined time series; to omit this constraint set this to INFTY_D
									const long		max_refined_grid_size,		// (INPUT) maximum size of the refined grid. This is a hard bound, to prevent using excessive RAM.
									dvector 		&refined_times,				// (OUTPUT) 1D array of size NR, listing ascending time points of the new (refined) time series
									dvector 		&refined_coeff){			// (OUTPUT) 2D array of size NR x (degree+1), in row-major format, listing polynomial coefficients of the new (refined) spline
	const long NC = coarse_times.size();
	refined_times.clear();
	refined_coeff.clear();
	if(start_time>coarse_times.back()) return;
	if(end_time<coarse_times[0]) return;
	const long start_c = (start_time<coarse_times[0] ? 0 : find_next_left_grid_point(coarse_times, start_time, 0)); 		// coarse-grid point immediately below (or equal) to start_time
	const long end_c   = (end_time>coarse_times.back() ? NC-1 : find_next_right_grid_point(coarse_times, end_time, NC-1));	// coarse-grid point immediately above (or equal) to end_time
	const long baseN   = (end_c-start_c+1);
	
	// determine size of refined time series
	std::vector<long> dN(NC-1,0); // dN[c] will be the resolution (number of additional time points) of the refined time series within the coarse time interval coarse_times[c]:coarse_times[c+1]. A dN of 0 means the particular coarse time interval does not need to be refined.
	double left_time, right_time, min_value, max_value, max_rate, max_time_step, time_shift;
	if(max_refined_grid_size>baseN){
		for(long c=start_c; c<end_c; ++c){
			time_shift		= (slideX ? coarse_times[c] : 0.0);
			left_time  		= max(start_time, coarse_times[c]);
			right_time 		= min(end_time, coarse_times[c+1]);
			min_value		= polynomial_lower_bound(degree, &coarse_coeff[c*(degree+1)+0], left_time-time_shift, right_time-time_shift);
			max_value		= polynomial_upper_bound(degree, &coarse_coeff[c*(degree+1)+0], left_time-time_shift, right_time-time_shift);
			max_rate 		= polynomial_bound_abs_derivative(degree, &coarse_coeff[c*(degree+1)+0], left_time-time_shift, right_time-time_shift);
			max_time_step 	= (max_time_steps.empty() ? INFTY_D : (max_time_steps.size()==1 ? max_time_steps[0] : max_time_steps[c]));
			if((right_time-left_time)>max_time_step)	dN[c] = max(dN[c], long(ceil((right_time-left_time)/max_time_step))-1);
			if((max_value-min_value)>max_value_step)	dN[c] = max(dN[c], long(ceil(max_rate*(right_time-left_time)/max_value_step))-1);
			if(!std::isinf(max_relative_value_step)){
				const double max_step = max_relative_value_step * 0.5*(abs(max_value)+abs(min_value));
				if((max_step>0) && ((max_value-min_value)>max_step)) dN[c] = max(dN[c], long(ceil(max_rate*(right_time-left_time)/max_step))-1);
			}
		}
	}
	long NR = baseN + vector_sum(dN);
	
	if((NR>max_refined_grid_size) && (max_refined_grid_size>baseN)){
		// need to reduce refined grid size. We shall do it uniformly across time
		const double factor = ((max_refined_grid_size-baseN)/double(NR-baseN));
		for(long c=start_c; c<end_c; ++c){
			dN[c] = long(floor(dN[c] * factor));
		}
		NR = baseN + vector_sum(dN);
	}

	// construct refined time series
	refined_times.resize(NR);
	refined_coeff.resize(NR*(degree+1));
	for(long c=start_c, r=0; c<end_c; ++c){
		// refine time interval left_time:right_time
		left_time  = max(start_time, coarse_times[c]);
		right_time = min(end_time, coarse_times[c+1]);
		double refined_time_step = (right_time-left_time)/(1+dN[c]);
		for(long n=0; n<dN[c]+1; ++n){
			refined_times[r] = left_time + n*refined_time_step;
			if(slideX){
				// the polynomial coefficients on the refined grid must be defined w.r.t. the sliding-time, i.e. in each refined-grid cell r, the polynomial will be defined w.r.t. time-refined_times[r]
				get_Xshifted_polynomial(degree,&coarse_coeff[c*(degree+1)],refined_times[r]-coarse_times[c],&refined_coeff[r*(degree+1)]);
			}else{
				// simply copy the original polynomial coefficients from the coarse grid cell onto all enclosed refined-grid cells
				for(long d=0; d<=degree; ++d){ 
					refined_coeff[r*(degree+1)+d] = coarse_coeff[c*(degree+1)+d];
				}
			}
			++r;
		}
	}
	refined_times[NR-1] = min(end_time,coarse_times[end_c]);
	for(long d=0; d<=degree; ++d){ 
		refined_coeff[(NR-1)*(degree+1)+d] = coarse_coeff[end_c*(degree+1)+d];
	}
}


// redefine a piecewise polynomial curve on a new (refined) time-grid, by copying polynomial coefficients onto any added grid points
// strictly speaking the function works even when the new time grid is not a refined version (i.e. a superset) of the old grid, however if the input function was a spline then the result may not satisfy the continuity assumptions of splines anymore
void refine_piecewise_polynomial(	const long		degree,			// (INPUT) polynomial degree of the input time series
									const long		Nold,			// (INPUT) number of grid points in the old time series
									const double	oldTimes[],		// (INPUT) 1D array of size N
									const double	oldCoeff[],		// (INPUT) 2D array of size N x (degree+1), listing polynomial coefficients of the splines on the grid intervals
									const bool		slideX,			// (INPUT) whether the input and output polynomial are defined as a function of the sliding-time rather than absolute time
									const dvector	&newTimes,		// (INPUT) 1D array of size NR, listing new times on which to redefine splines
									dvector			&newCoeff){		// (OUTPUT) 2D array of size NR x (degree+1), listing the polynomial coefficients on the new grid
	newCoeff.resize(newTimes.size()*(degree+1));
	for(long ng=0, og=0; ng<newTimes.size(); ++ng){
		og = find_next_left_grid_point(Nold, oldTimes, newTimes[ng], og); // find old grid point to the immediate left of the new grid point ng
		if(slideX){
			// the polynomial coefficients on the refined grid must be defined w.r.t. the sliding-time, i.e. in each new-grid cell ng, the polynomial will be defined w.r.t. time-newTimes[ng]
			get_Xshifted_polynomial(degree,&oldCoeff[og*(degree+1)],newTimes[ng]-oldTimes[og],&newCoeff[ng*(degree+1)]);
		}else{
			// simply copy the old polynomial's coefficients onto the new grid points
			for(long d=0; d<=degree; ++d){
				newCoeff[ng*(degree+1)+d] = oldCoeff[og*(degree+1)+d];
			}
		}	
	}
}


// wrapper for the function refine_piecewise_polynomial() allowing std::vectors as inputs
void refine_piecewise_polynomial(	const long		degree,
									const dvector 	&oldTimes,		// (INPUT) 1D array of size N
									const dvector 	&oldCoeff,		// (INPUT) 2D array of size N x (degree+1), listing polynomial coefficients of the splines on the grid intervals
									const bool		slideX,			// (INPUT) whether the input and output polynomial are defined as a function of the sliding-time rather than absolute time
									const dvector	&newTimes,		// (INPUT) 1D array of size NR, listing new times on which to redefine splines
									dvector			&newCoeff){		// (OUTPUT) 2D array of size NR x (degree+1), listing the polynomial coefficients on the new grid
	refine_piecewise_polynomial(degree, oldTimes.size(), &oldTimes[0], &oldCoeff[0], slideX, newTimes, newCoeff);
}


// redefine a piecewise polynomial curve on a new (refined) time-grid, by copying polynomial coefficients onto any added grid points
// strictly speaking the function works even when the new time grid is not a refined version (i.e. a superset) of the old grid, however if the input function was a spline then the result may not satisfy the continuity assumptions of splines anymore
void refine_piecewise_polynomial(	const long		degree,
									const dvector 	&oldTimes,		// (INPUT) 1D array of size N
									const dvector 	&oldCoeff,		// (INPUT) 2D array of size N x (degree+1), listing polynomial coefficients of the splines on the grid intervals
									const bool		slideX,			// (INPUT) whether the input and output polynomial are defined as a function of the sliding-time rather than absolute time
									const dvector	&newTimes,		// (INPUT) 1D array of size NR, listing new times on which to redefine splines
									dvector			&newCoeff,		// (OUTPUT) 2D array of size NR x (degree+1), listing the polynomial coefficients on the new grid
									dvector			&newValues){	// (OUTPUT) 1D array of size NR, listing the values on the refined grid
	newCoeff.resize(newTimes.size()*(degree+1));
	newValues.resize(newTimes.size(),0.0);
	for(long ng=0, og=0; ng<newTimes.size(); ++ng){
		og = find_next_left_grid_point(oldTimes, newTimes[ng], og); // find old grid point to the immediate left of the new grid point ng
		if(slideX){
			// the polynomial coefficients on the refined grid must be defined w.r.t. the sliding-time, i.e. in each new-grid cell ng, the polynomial will be defined w.r.t. time-newTimes[ng]
			get_Xshifted_polynomial(degree,&oldCoeff[og*(degree+1)],newTimes[ng]-oldTimes[og],&newCoeff[ng*(degree+1)]);
			newValues[ng] = newCoeff[ng*(degree+1)+0]; // since the new polynomial coefficients are defined w.r.t. a shifted time with offset newTimes[ng], the polynomial's value at newTimes[ng] is simply the 0-th coefficient of the polynomial
		}else{
			// simply copy the old polynomial's coefficients onto the new grid points
			for(long d=0; d<=degree; ++d){
				newCoeff[ng*(degree+1)+d] = oldCoeff[og*(degree+1)+d];
				newValues[ng] += newCoeff[ng*(degree+1)+d]*pow(newTimes[ng],double(d));
			}
		}
	}
}


// fits the curve y = A + B*x + C*x^2 to the data set defined by (x,y) using least squares
// returns false if an error occured, in which case the coefficients will be set to NAN_D.
// epsilon specifies the aimed-for fitting accuracy (in case of iterative refinement), relative to the mean{|y|}. epsilon=1e-6 would typically be reasonable.
// This function is NAN-sensitive, i.e. it avoids NAN in the x[] and y[] data.
bool fitLeastSquares_Quadratic(const std::vector<double> &x, const std::vector<double> &y, double &A, double &B, double &C, double epsilon){
	// calculate problem coefficients
	std::vector<double> b(3,0), sol(3), scratchSpace(3*3), matrix(3*3,0);
	const long N = x.size();
	
	// get scales
	double scaleX = 0, scaleY = 0;
	long counted = 0;
	for(long i=0; i<N; ++i){
		if(std::isnan(x[i]) || std::isnan(y[i])) continue;
		++counted;
		scaleX += abs(x[i]);
		scaleY += abs(y[i]);
	}
	if(counted<3){ A=B=C=NAN_D; return false; }
	scaleX = (counted==0 ? 1.0 : scaleX/counted);
	scaleY = (counted==0 ? 1.0 : scaleY/counted);
	
	for(long i=0; i<N; ++i){
		if(std::isnan(x[i]) || std::isnan(y[i])) continue;
		b[0] += y[i]/scaleY;
		b[1] += (x[i]/scaleX) * (y[i]/scaleY);
		b[2] += SQ(x[i]/scaleX) * (y[i]/scaleY);

		// Vandermonde matrix
		matrix[0*3+0] += 1;
		matrix[0*3+1] += x[i]/scaleX;
		matrix[0*3+2] += SQ(x[i]/scaleX);
		
		matrix[1*3+0] += x[i]/scaleX;
		matrix[1*3+1] += SQ(x[i]/scaleX);
		matrix[1*3+2] += Qube(x[i]/scaleX);
		
		matrix[2*3+0] += SQ(x[i]/scaleX);
		matrix[2*3+1] += Qube(x[i]/scaleX);
		matrix[2*3+2] += QTR(x[i]/scaleX);
	}
	
	if(!LUsolveLinearSystem(	&matrix[0],
							&scratchSpace[0],
							3,
							&b[0],
							epsilon*3,
							5,
							&sol[0])){
		A=B=C=NAN_D;
		return false;
	}
	A = scaleY*sol[0]; 
	B = scaleY*sol[1]/scaleX; 
	C = scaleY*sol[2]/SQ(scaleX); 
	return true;
}






// fits the curve y = A + B*x + C*x^2 + D*x^3 to the data set defined by (x,y) using least squares
// returns false if an error occured, in which case the coefficients will be set to NAN_D.
// epsilon specifies the aimed-for fitting accuracy (in case of iterative refinement), relative to the mean{|y|}. epsilon=1e-6 would typically be reasonable.
// This function is NAN-sensitive, i.e. it avoids NAN in the x[] and y[] data
bool fitLeastSquares_Qubic(const std::vector<double> &x, const std::vector<double> &y, double &A, double &B, double &C, double &D, double epsilon){
	// calculate problem coefficients
	std::vector<double> b(4,0), sol(4), scratchSpace(4*4), matrix(4*4,0);
	const long N = x.size();
	
	// get scales
	double scaleX = 0, scaleY = 0;
	long counted = 0;
	for(long i=0; i<N; ++i){
		if(std::isnan(x[i]) || std::isnan(y[i])) continue;
		++counted;
		scaleX += abs(x[i]);
		scaleY += abs(y[i]);
	}
	if(counted<4){ A=B=C=D=NAN_D; return false; }
	scaleX = (counted==0 ? 1.0 : scaleX/counted);
	scaleY = (counted==0 ? 1.0 : scaleY/counted);
	
	for(long i=0; i<N; ++i){
		if(std::isnan(x[i]) || std::isnan(y[i])) continue;
		b[0] += y[i]/scaleY;
		b[1] += (x[i]/scaleX) * (y[i]/scaleY);
		b[2] += SQ(x[i]/scaleX) * (y[i]/scaleY);
		b[3] += Qube(x[i]/scaleX) * (y[i]/scaleY);
		
		// Vandermonde matrix
		matrix[0*4+0] += 1;
		matrix[0*4+1] += x[i]/scaleX;
		matrix[0*4+2] += SQ(x[i]/scaleX);
		matrix[0*4+3] += Qube(x[i]/scaleX);
		
		matrix[1*4+0] += x[i]/scaleX;
		matrix[1*4+1] += SQ(x[i]/scaleX);
		matrix[1*4+2] += Qube(x[i]/scaleX);
		matrix[1*4+3] += QTR(x[i]/scaleX);
		
		matrix[2*4+0] += SQ(x[i]/scaleX);
		matrix[2*4+1] += Qube(x[i]/scaleX);
		matrix[2*4+2] += QTR(x[i]/scaleX);
		matrix[2*4+3] += (x[i]/scaleX)*QTR(x[i]/scaleX);
		
		matrix[3*4+0] += Qube(x[i]/scaleX);
		matrix[3*4+1] += QTR(x[i]/scaleX);
		matrix[3*4+2] += (x[i]/scaleX)*QTR(x[i]/scaleX);
		matrix[3*4+3] += SQ(x[i]/scaleX)*QTR(x[i]/scaleX);
	}
	
	if(!LUsolveLinearSystem(	&matrix[0],
							&scratchSpace[0],
							4,
							&b[0],
							epsilon*4,
							5,
							&sol[0])){
		A=B=C=D=NAN_D;
		return false;
	}
	A = scaleY*sol[0]; 
	B = scaleY*sol[1]/scaleX; 
	C = scaleY*sol[2]/SQ(scaleX); 
	D = scaleY*sol[3]/Qube(scaleX);
	return true;
}	







// fits the curve y = A + B*x + C*x^2 + D*x^3 + E*x^4 to the data set defined by (x,y) using least squares
// returns false if an error occured, in which case the coefficients will be set to NAN_D.
// epsilon specifies the aimed-for fitting accuracy (in case of iterative refinement), relative to the mean{|y|}. epsilon=1e-6 would typically be reasonable.
// This function is NAN-sensitive, i.e. it avoids NAN in the x[] and y[] data
bool fitLeastSquares_Quartic(const std::vector<double> &x, const std::vector<double> &y, double &A, double &B, double &C, double &D, double &E, double epsilon){
	// calculate problem coefficients
	std::vector<double> b(5,0), sol(5), scratchSpace(5*5), matrix(5*5,0);
	const long N = x.size();
	
	// get scales
	double scaleX = 0, scaleY = 0;
	long counted = 0;
	for(long i=0; i<N; ++i){
		if(std::isnan(x[i]) || std::isnan(y[i])) continue;
		++counted;
		scaleX += abs(x[i]);
		scaleY += abs(y[i]);
	}
	if(counted<5){ A=B=C=D=E=NAN_D; return false; }
	scaleX = (counted==0 ? 1.0 : scaleX/counted);
	scaleY = (counted==0 ? 1.0 : scaleY/counted);
	
	for(long i=0; i<N; ++i){
		if(std::isnan(x[i]) || std::isnan(y[i])) continue;
		b[0] += y[i]/scaleY;
		b[1] += (x[i]/scaleX) * (y[i]/scaleY);
		b[2] += SQ(x[i]/scaleX) * (y[i]/scaleY);
		b[3] += Qube(x[i]/scaleX) * (y[i]/scaleY);
		b[4] += QTR(x[i]/scaleX) * (y[i]/scaleY);
		
		// Vandermonde matrix
		matrix[0*5+0] += 1;
		matrix[0*5+1] += x[i]/scaleX;
		matrix[0*5+2] += SQ(x[i]/scaleX);
		matrix[0*5+3] += Qube(x[i]/scaleX);
		matrix[0*5+4] += QTR(x[i]/scaleX);
		
		matrix[1*5+0] += x[i]/scaleX;
		matrix[1*5+1] += SQ(x[i]/scaleX);
		matrix[1*5+2] += Qube(x[i]/scaleX);
		matrix[1*5+3] += QTR(x[i]/scaleX);
		matrix[1*5+4] += (x[i]/scaleX)*QTR(x[i]/scaleX);
		
		matrix[2*5+0] += SQ(x[i]/scaleX);
		matrix[2*5+1] += Qube(x[i]/scaleX);
		matrix[2*5+2] += QTR(x[i]/scaleX);
		matrix[2*5+3] += (x[i]/scaleX)*QTR(x[i]/scaleX);
		matrix[2*5+4] += SQ(x[i]/scaleX)*QTR(x[i]/scaleX);
		
		matrix[3*5+0] += Qube(x[i]/scaleX);
		matrix[3*5+1] += QTR(x[i]/scaleX);
		matrix[3*5+2] += (x[i]/scaleX)*QTR(x[i]/scaleX);
		matrix[3*5+3] += SQ(x[i]/scaleX)*QTR(x[i]/scaleX);
		matrix[3*5+4] += Qube(x[i]/scaleX)*QTR(x[i]/scaleX);
		
		matrix[4*5+0] += QTR(x[i]/scaleX);
		matrix[4*5+1] += (x[i]/scaleX)*QTR(x[i]/scaleX);
		matrix[4*5+2] += SQ(x[i]/scaleX)*QTR(x[i]/scaleX);
		matrix[4*5+3] += Qube(x[i]/scaleX)*QTR(x[i]/scaleX);
		matrix[4*5+4] += QTR(x[i]/scaleX)*QTR(x[i]/scaleX);
	}
	
	if(!LUsolveLinearSystem(	&matrix[0],
							&scratchSpace[0],
							5,
							&b[0],
							epsilon*5,
							5,
							&sol[0])){
		A=B=C=D=E=NAN_D;
		return false;
	}
	A = scaleY*sol[0]; 
	B = scaleY*sol[1]/scaleX; 
	C = scaleY*sol[2]/SQ(scaleX); 
	D = scaleY*sol[3]/Qube(scaleX);
	E = scaleY*sol[4]/QTR(scaleX);
	return true;
}




// Performs a least residual sum of squares fitting of the real scalar linear function y(x) = slope*x to the data (x[], y[]), by choice of slope.
// Here, x is the predictor and y the response variable.
// returns false on error
template<class REAL_TYPE>
bool fitLeastSquares_linear_real_scalar(const vector<REAL_TYPE> 	&x,		// (input)	Measured covariates
										const vector<REAL_TYPE> 	&y,		// (input)	Measured response variables
										unsigned long				start,	// (input) first considered data index (ignore any entries prior)
										unsigned long				end,	// (input) last considered data index (ignore any entries after)
										REAL_TYPE 					&slope,	// (output) Fitted slope
										REAL_TYPE					&RSS){	// (output) The sum of squared residuals
	REAL_TYPE sumX2(0), sumXY(0), sumY2(0);
	if((x.size()<=end) || (y.size()<=end)) return false;
	if(start>end) return false; //need at least one data point
	unsigned long k;
	for(k=start; k<=end; ++k){
		sumX2 += SQ(x[k]);
		sumY2 += SQ(y[k]);
		sumXY += x[k]*y[k];
	}
	slope 	= sumXY/sumX2;
	RSS		= sumY2 - SQ(sumXY)/sumX2;
	return true;
}




template<class REAL_TYPE>
REAL_TYPE affine_real_scalar_RSS(REAL_TYPE slope, REAL_TYPE intercept, long count, REAL_TYPE meanX, REAL_TYPE meanY, REAL_TYPE meanX2, REAL_TYPE meanY2, REAL_TYPE sumXY){
	return count * SQ(intercept)
			+ 2*slope*intercept*meanX*count 
			- 2*intercept*meanY*count 
			+ SQ(slope)*meanX2*count 
			+ meanY2*count 
			- 2*slope*sumXY;
}


// Performs a least residual sum of squares fitting of the real scalar affine function y(x) = intercept + slope*x to the data (x[], y[]), by choice of intercept & slope.
// Here, x is the covariate and y the response variable.
// returns false on error
template<class REAL_TYPE>
bool fitLeastSquares_affine_real_scalar(const vector<REAL_TYPE> 	&x,			// (INPUT) Measured covariates
										const vector<REAL_TYPE> 	&y,			// (INPUT) Measured response variables
										const long					start,		// (INPUT) first considered data index (ignore any entries prior)
										const long					end,		// (INPUT) last considered data index (ignore any entries after)
										const double				Xshift,		// (INPUT) optional shift for the X-values, so that the fitted model is with respect to X-Xshift rather than X, i.e. Y ~ normal(mean = intercept + slope*(X-Xshift)). Set this to 0 to not perform any X-shift.
										bool						forcePositiveSlope,		// (INPUT)
										bool						forcePositiveIntercept,	// (INPUT)
										REAL_TYPE					&intercept, // (OUTPUT) Fitted intercept
										REAL_TYPE 					&slope,		// (OUTPUT) Fitted slope
										REAL_TYPE					&RSS,		// (OUTPUT) The sum of squared residuals
										long						&count){	// (OUTPUT) the number of data points that were used for fitting (i.e., from start to end, excluding NANs)
	REAL_TYPE meanX(0), meanY(0), meanX2(0), meanY2(0), sumXY(0);
	if((x.size()<=end) || (y.size()<=end)) return false;
	if(start>=end) return false; // need at least two data points
	
	count = 0;
	for(long k=start; k<=end; ++k){
		if(!(std::isnan(x[k]) || std::isnan(y[k]))){
			meanX	+= (x[k]-Xshift);
			meanY	+= y[k];
			meanX2 	+= SQ(x[k]-Xshift);
			meanY2 	+= SQ(y[k]);
			sumXY 	+= (x[k]-Xshift)*y[k];
			++count;
		}
	}
	if(count<2) return false; // need at least two valid data points
	meanX	/= count;
	meanY	/= count;
	meanX2	/= count;
	meanY2	/= count;
	
	slope 		= 	(sumXY/count - meanX*meanY)/(meanX2 - SQ(meanX));
	intercept	= 	(meanX2*meanY - meanX*sumXY/count)/(meanX2 - SQ(meanX));
	
	if(forcePositiveSlope && (!forcePositiveIntercept)){
		if(slope<0){
			slope = 0;
			intercept = meanY;
		}
	}else if((!forcePositiveSlope) && forcePositiveIntercept){
		if(intercept<0){
			intercept = 0;
			slope = sumXY/(count*meanX2);
		}
	}else if(forcePositiveSlope && forcePositiveIntercept){
		if(slope<0 || intercept<0){
			REAL_TYPE RSS00 = affine_real_scalar_RSS(0.0, 0.0, count, meanX, meanY, meanX2, meanY2, sumXY);
			REAL_TYPE RSS01 = affine_real_scalar_RSS(0.0, meanY, count, meanX, meanY, meanX2, meanY2, sumXY);
			REAL_TYPE RSS10 = affine_real_scalar_RSS(sumXY/(count*meanX2), 0.0, count, meanX, meanY, meanX2, meanY2, sumXY);
			if(meanY<0){ 
				intercept = 0;
				slope = ((RSS00<RSS10) || (sumXY<0) ? 0 : sumXY/(count*meanX2));
			}else{
				intercept 	= ((RSS01<RSS10) || (sumXY<0) ? meanY 	: 0);
				slope		= ((RSS01<RSS10) || (sumXY<0) ? 0 		: sumXY/(count*meanX2));
			}
		}
	}
	RSS = affine_real_scalar_RSS(slope, intercept, count, meanX, meanY, meanX2, meanY2, sumXY);

	if(std::isnan(slope) || std::isnan(intercept) || std::isnan(RSS)) return false;
	return true;
}




// Fits the real scalar exponential function y(x) = intercept*exp(x/scale) to the data (x[], y[]), by choice of scale and (possibly) intercept.
// Here, x is the covariate and y the response variable.
// Fitting minimizes sum of squared residuals of log-transformed responses to log-transformed predictions.
// returns false on error
template<class ARRAY_TYPE1, class ARRAY_TYPE2>
bool fit_exp_LeastLogSquares(	const ARRAY_TYPE1	&x,					// (input) Measured covariates
								const ARRAY_TYPE2 	&y,					// (input) Measured response variables.
								long				start,				// (input) first considered data index (ignore any entries prior)
								long				end,				// (input) last considered data index (ignore any entries after)
								bool				intercept_fixed,	// (input) if true, intercept is fixed to the value given. Otherwise, intercept is fitted as well.
								double				&intercept, 		// (input/output) Fitted y-intercept if intercept_fixed==false, otherwise left unchanged.
								double 				&scale,				// (output) Fitted caracteristic scale (=inverse rate)
								double				&RSS){				// (output) The sum of squared residuals (on y-logarithmic scale)
	long k;
	if((end<=start) || (end>=x.size()) || (end>=y.size()) || (end<0) || (start<0)) return false;
	vector<double> X = vector<double>(x.begin()+start,x.begin()+end+1);
	vector<double> Y = vector<double>(y.begin()+start,y.begin()+end+1);
	start = 0; end = X.size()-1;
	for(k=start; k<=end; ++k){
		Y[k] = log(Y[k]);
	}
	if(intercept_fixed){
		double logintercept = log(intercept);
		for(k=start; k<=end; ++k){
			Y[k] -= logintercept;
		}
		if(!fitLeastSquares_linear_real_scalar(X, Y, start, end, scale, RSS)){
			return false;
		}
	}else{
		long count;
		if(!fitLeastSquares_affine_real_scalar(X, Y, start, end, 0, false, false, intercept, scale, RSS, count)){
			return false;
		}
		intercept = exp(intercept);
	}
	scale = 1.0/scale;
	return true;
}



// fit an exponential function y = A*exp(rate*x) locally to a time series, i.e. in a sliding window, via least-squares on a logarithmic scale
// In the n-th sliding window, the underlying statistical model is that log(y) ~ Normal(mean = log(A_n) + rate_n*(x-Xshift_n), variance = S2_n)
// This routine thus returns the fitted coefficients A and B at each time point, fitted to nearby data points
// [[Rcpp::export]]
Rcpp::List fit_exp_LeastLogSquares_moving_window_CPP(	const std::vector<double> 	&X,				// (INPUT) 1D numeric array of size N, listing X-values in ascending order
														const std::vector<double> 	&Y,				// (INPUT) 1D numeric array of size N
														const long					window_size,	// (INPUT) strictly positive integer, specifying the local sliding-window size, i.e. the number of data points to include for each fitting
														const bool					trim_window_at_bounds){	// (INPUT) whether to trim the sliding window when hitting the data's X-bounds. If false, then the sliding window always has the specified size, but may not always be centered around the point of evaluation, and toward the left & right bound the fitted params will be constant.
	const long N = X.size();
	dvector logY(N);
	for(long n=0; n<N; ++n) logY[n] = (Y[n]>0 ? log(Y[n]) : NAN_D);
	const long left_radius  = floor(window_size/2);
	
	dvector A(N,NAN_D), rate(N,NAN_D), S2(N,NAN_D), predicted_logY(N,NAN_D), Xshift(N,NAN_D);
	lvector count(N,0), starts(N,0), ends(N,0);
	double RSS;
	long start, end, next_valid;
	for(long n=0; n<N; ++n){
		if(trim_window_at_bounds){
			start = max(0l, n-left_radius);
			end   = min(N-1, n+window_size-1-left_radius);
		}else{
			if(n<left_radius){
				// too close to the left edge, so will need to adjust the right part of the window
				start = 0;
				end   = min(N-1,start + window_size - 1);
			}else{
				end   = min(N-1,n + window_size - left_radius - 1);
				start = max(0l, end-window_size+1);
			}
		}
		starts[n] = start;
		ends[n]   = end;
		
		// find the next valid point to use as Xshift
		next_valid = find_first_non_nan(X, start);
		if(next_valid<0) continue; // did not find a valid time point to use as shift
		Xshift[n] = X[next_valid];
		
		// fit statistical model log(Y) ~ normal(mean = log(A[n]) + rate[n]*(X-Xshift[n]))
		if(fitLeastSquares_affine_real_scalar(X, logY, start, end, Xshift[n], false, false, A[n], rate[n], RSS, count[n])){
			predicted_logY[n] 	= A[n] + rate[n]*(X[n]-Xshift[n]);
			A[n] 				= exp(A[n]);
			S2[n] 				= RSS/count[n];
		}
	}
	return Rcpp::List::create(	Rcpp::Named("A")				= A,
								Rcpp::Named("rate")				= rate,
								Rcpp::Named("Xshift")			= Xshift,
								Rcpp::Named("Npoints")			= count,
								Rcpp::Named("window_starts")	= starts,
								Rcpp::Named("window_ends")		= ends,
								Rcpp::Named("log_variance") 	= S2, // maximum-likelihood estimate of the error-variance on a log-scale, i.e. VAR(data_logY - expected_logY)
								Rcpp::Named("predicted_logY") 	= predicted_logY);
}




// Functor for fitting an exponential growth model to a time series, whereby Y_n ~ Poisson(mean = A_n*scalings_n*exp(rate_n*(X-Xshift_n)))
// This functor returns a function, whose root (point of zero) corresponds to the maximum-likelihood estimate of the exponential growth rate parameter
// Hence, ML estimation corresponds to finding the root of this functor
class ExpPoissonMLrootant{
public:
	dvector X; 	// 1D numerical vector of length NX, listing the "times"
	dvector Y; 	// 1D numerical vector of length NX, listing the "values", i.e. observed counts
	dvector scalings; // 1D numerical vector of length NX, listing optional scalings for each time point
	long start, end; // first and last data point to use from the time series
	double Xshift; // optional "time-shift" to apply to the time series, i.e. start counting time at Xshift.
	
	// evaluate the functor for some exponential rate
	double operator()(const double rate) const{
		double sumY=0, sumXY=0, sumE=0, sumXE=0, E;
		for(long i=start; i<=end; ++i){
			if((!std::isnan(X[i])) && (!std::isnan(Y[i])) && (!std::isnan(scalings[i]))){
				E		= exp(rate * (X[i]-Xshift));
				sumY 	+= Y[i];
				sumXY 	+= X[i] * Y[i];
				sumE	+= scalings[i] * E;
				sumXE	+= scalings[i] * X[i] * E;
			}
		}
		return (sumXE*sumY/sumE - sumXY);
	}	
	
	// get the ML estimate for A, assuming a specific rate
	double get_A_for_specific_rate(const double rate) const{
		double sumY=0, sumE=0;
		for(long i=start; i<=end; ++i){
			if((!std::isnan(X[i])) && (!std::isnan(Y[i])) && (!std::isnan(scalings[i]))){
				sumY 	+= Y[i];
				sumE	+= scalings[i] * exp(rate * (X[i]-Xshift));
			}
		}
		return (sumY/sumE);
	}
};



// fit an exponential function y = A*exp(rate*x) locally to a time series, i.e. in a sliding window, via maximum likelihood and based on a Poisson model
// In the n-th sliding window, the underlying statistical model is that Y ~ Poisson(mean = A_n*scalings_n*exp(rate_n*(X-Xshift_n))), where Xshift_n is a time-shift chosen for each time window
// This routine thus returns the fitted coefficients A_n and rate_n at each time point, fitted to nearby data points
// Since the routine fits a Poisson model, it is assumed that the time series' values are integers (counts of something).
// [[Rcpp::export]]
Rcpp::List fit_exp_Poisson_moving_window_CPP(	const std::vector<double> 	&X,				// (INPUT) 1D numeric array of size N, listing X-values in ascending order. May contain NAN_D.
												const std::vector<double> 	&Y,				// (INPUT) 1D numeric array of size N. Normally these should be integers. May contain NAN_D.
												const std::vector<double>	&scalings,		// (INPUT) 1D numeric array of size N, specifying optional scalings for the Y, i.e. such that Y_k ~ Poisson(mean = A_n * scalings[n] * exp(rate_n*(X-Xshift_n))). Set these to 1 to omit any scaling. These should not contain NANs.
												const long					window_size,	// (INPUT) strictly positive integer, specifying the local sliding-window size, i.e. the number of data points to include for each fitting
												const bool					trim_window_at_bounds){	// (INPUT) whether to trim the sliding window when hitting the data's X-bounds. If false, then the sliding window always has the specified size, but may not always be centered around the point of evaluation, and toward the left & right bound the fitted params will be constant.
	const long N = X.size();
	const long left_radius  = floor(window_size/2);
	dvector logScaledY(N);
	for(long n=0; n<N; ++n) logScaledY[n] = ((Y[n]>0) && (scalings[n]>0) ? log(Y[n]/scalings[n]) : NAN_D);
	
	// setup functor to find roots of for maximum-likelihood estimation of the rate
	ExpPoissonMLrootant rootant;
	rootant.X = X;
	rootant.Y = Y;
	rootant.scalings = scalings;

	dvector A(N,NAN_D), rate(N,NAN_D), predicted_Y(N,NAN_D), Xshift(N,NAN_D);
	lvector count(N,0), Npositives(N,0), starts(N,0), ends(N,0);
	double guess_rate, guess_A, RSS;
	long start, end, next_valid, dummyL;
	MathError fit_error;
	for(long n=0; n<N; ++n){
		if(trim_window_at_bounds){
			start = max(0l, n-left_radius);
			end   = min(N-1, n+window_size-1-left_radius);
		}else{
			if(n<left_radius){
				// too close to the left edge, so will need to adjust the right part of the window
				start = 0;
				end   = min(N-1,start + window_size - 1);
			}else{
				end   = min(N-1,n + window_size - left_radius - 1);
				start = max(0l, end-window_size+1);
			}
		}
		starts[n] = start;
		ends[n]   = end;
		
		// find the next valid point to use as Xshift
		next_valid = find_first_non_nan(X, start);
		if(next_valid<0) continue; // did not find a valid time point to use as shift
		Xshift[n] = X[next_valid];
		
		// get typical X & Y scales within this time window and count number of valid data points
		double Xscale = 0, Yscale = 0;
		for(long i=start; i<=end; ++i){
			if((!std::isnan(X[i])) && (!std::isnan(Y[i])) && (!std::isnan(scalings[i]))){
				if(Y[i]!=0) ++Npositives[n];
				++count[n];
				Xscale += X[i]-Xshift[n];
				Yscale += Y[i];
			}
		}
		Xscale /= count[n];
		Yscale /= count[n];
		if(count[n]<2){
			// not enough valid data points in this time window, so fitting is not possible
			continue;
		}else if(Yscale==0){
			// all Y values are zero in this time window, so won't be able to get ML estimates
			A[n] 			= 0;
			rate[n] 		= NAN_D;
			predicted_Y[n] 	= 0;
			continue;
		}

		// generate a first guess for the exponential rate, using linear regression on a logarithmic axis
		if(!fitLeastSquares_affine_real_scalar(X, logScaledY, start, end, Xshift[n], false, false, guess_A, guess_rate, RSS, dummyL)){
			// linear regression failed, so create a heuristic first guess
			double smallest = -1, largest = -1;
			for(long i=start; i<=end; ++i){
				if((!std::isnan(logScaledY[i])) && (!std::isnan(X[i])) && ((smallest<0) || (logScaledY[i]<logScaledY[smallest]))) smallest = i;
				if((!std::isnan(logScaledY[i])) && (!std::isnan(X[i])) && ((largest<0) || (logScaledY[i]>logScaledY[largest]))) largest = i;
			}
			if((smallest>=0) && (largest>=0) && (X[smallest]!=X[largest])){
				guess_rate = (logScaledY[largest]-logScaledY[smallest])/(X[largest]-X[smallest]);
			}else{
				guess_rate = 0;
			}
		}
				
		// fit exponential rate, based on statistical model Y ~ Poisson(mean = A[n]*exp(rate[n]*(X-Xshift[n])))
		rootant.Xshift = Xshift[n];
		rootant.start = start;
		rootant.end = end;
		fit_error = solve_via_bisection(rootant,
										min(-1000*1/Xscale,-1000*abs(guess_rate)),	// rate_min
										max(+1000*1/Xscale,+1000*abs(guess_rate)),	// rate_max
										guess_rate,
										0,					// target value
										0.0001 * 1/(X[end]-X[start]),	// rate_epsilon
										0.0001 * Yscale,			// f tollerance
										10000,	// max iterations
										rate[n]);
		if(fit_error==MathErrorNone){
			A[n] = rootant.get_A_for_specific_rate(rate[n]);
			predicted_Y[n] = A[n] * scalings[n] * exp(rate[n]*(X[n]-rootant.Xshift));
			if(A[n]==0) rate[n] = NAN_D; // if A=0, then the rate becomes non-identifiable
		}
	}
	return Rcpp::List::create(	Rcpp::Named("A")				= A,
								Rcpp::Named("rate")				= rate,
								Rcpp::Named("Xshift")			= Xshift,
								Rcpp::Named("Npoints")			= count,
								Rcpp::Named("Npositives")		= Npositives,
								Rcpp::Named("window_starts")	= starts,
								Rcpp::Named("window_ends")		= ends,
								Rcpp::Named("predicted_Y") 		= predicted_Y);
}




// smoothen time series using Savitzky-Golay filtering (unweighted local least-squares polynomial fitting)
// for details see: http://www.mathscratchs.com/help/curvefit/smoothing-data.html
// Note: If the sliding window does not cover enough time points, a lower-order filter may be applied (locally)
// This function supports non-evenly spaced data, which means fitting a polynomial at each time point, making it computationally expensive
// The time series is assumed to be sorted in time
// This function is NAN-sensitive, i.e. NANs in data[] are avoided.
template<class TIME_ARRAY, class VALUE_ARRAY>
bool smoothenTimeSeriesSavitzkyGolay(	const TIME_ARRAY	&times,				// time points in ascending order
										const VALUE_ARRAY	&data, 	
										double				windowTimeSpan,		// span of fitting window in time units. Ignored if <=0.
										long				windowIndexSpan,	// span of fitting window in terms of the number of data points included (will be rounded up to nearest odd number). Can be used as an alternative to windowTimeSpan. Ignored if <=1.
										int					order,
										const bool			allow_less_data_at_edges, // only relevant if windowIndexSpan is used
										vector<double> 		&smoothenedData){
	if(times.size() != data.size()) return false;
	if((order<0) || (order>4)) return false;
	smoothenedData.clear();
	const long N = times.size();
	if(N<1) return true;	// nothing to do
	double coeff_A, coeff_B, coeff_C, coeff_D, coeff_E;
	long localOrder, distinctTimePoints;
	vector<double> localTimes, localData;
	if(windowTimeSpan>0) windowTimeSpan = max(0.0, min(windowTimeSpan, times[N-1]-times[0]));
	else if(windowIndexSpan>1) windowIndexSpan = min(windowIndexSpan,N-1);

	smoothenedData.resize(N);
	for(long n=0, nl,nr,M; n<N; ++n){
		const double t = times[n];
		// determine data points to include
		if(windowIndexSpan>1){
			nl = n-windowIndexSpan/2;
			nr = n+windowIndexSpan/2;
			if(!allow_less_data_at_edges){
				if(nl<0) nr += (0-nl); // shift to the right since we're at the left edge
				else if(nr>=N) nl -= (nr-N+1); // shift to the left since we're at the right edge
			}
		}else{
			const double tl = t-windowTimeSpan/2;
			const double tr = t+windowTimeSpan/2;
			if(times[0]>=tl){
				nl=0;
			}else{
				for(nl=n; times[nl-1]>=tl; --nl){}
			}
			if(times[N-1]<=tr){
				nr=N-1;
			}else{
				for(nr=n; times[nr+1]<=tr; ++nr){}
			}
		}
		nl = max(0l,nl);
		nr = min((long)(N-1),nr);
		M  = nr-nl+1; // number of local data points covered by window
		
		// count distinct time points covered by window
		distinctTimePoints = 1;
		for(long i=nl+1; i<=nr; ++i){ if(times[i]>times[i-1]) ++distinctTimePoints; }
				
		if(M==1){
			smoothenedData[n] = data[n]; // only one data point. 
		}else if((distinctTimePoints==1) || (order==0)){
			// all local data are at the same time point (or polynomial order==0), so take arithmetic average
			double sum = 0; 
			long counted = 0;
			for(long i=nl; i<=nr; ++i){ if(!std::isnan(data[i])){ ++counted; sum += data[i]; } }
			smoothenedData[n] = (counted>0 ? sum/counted : NAN_D);
		
		}else{			
			// prepare local subset of data for polynomial fitting
			localTimes.resize(M);
			localData.resize(M);
			for(long i=nl; i<=nr; ++i){
				localTimes[i-nl] = times[i];
				localData[i-nl]	 = data[i];
			}
		
			// fit polynomial using least squares and evaluate at centre point
			localOrder = (int)min(distinctTimePoints-1,(long)order); // if too few data points available, use lower order
			switch(localOrder){
			case 0:
				smoothenedData[n] = vector_mean(&localData[0], M);
				break;
			case 1: 
				fitLinearRegressionNANSensitive(&localTimes[0], &localData[0], M, coeff_B, coeff_A); 
				smoothenedData[n] = coeff_A + t*coeff_B;
				break;			
			case 2:
				fitLeastSquares_Quadratic(localTimes, localData, coeff_A, coeff_B, coeff_C, 1e-6);	
				smoothenedData[n] = coeff_A + t*coeff_B + SQ(t)*coeff_C;
				break;			
			case 3:
				fitLeastSquares_Qubic(localTimes, localData, coeff_A, coeff_B, coeff_C, coeff_D, 1e-6);	
				smoothenedData[n] = coeff_A + t*coeff_B + SQ(t)*coeff_C + Qube(t)*coeff_D;
				break;	
			case 4:
				fitLeastSquares_Quartic(localTimes, localData, coeff_A, coeff_B, coeff_C, coeff_D, coeff_E, 1e-6);	
				smoothenedData[n] = coeff_A + t*coeff_B + SQ(t)*coeff_C + Qube(t)*coeff_D + QTR(t)*coeff_E;
				break;
			}
		}
	}
	return true;												
}


// Rcpp wrapper for the homonymous function above
// [[Rcpp::export]]
Rcpp::List smoothenTimeSeriesSavitzkyGolay_CPP(	const std::vector<double>	&times,				// time points in ascending order
												const std::vector<double>	&data, 	
												double				windowTimeSpan,		// span of fitting window in time units. Ignored if <=0.
												long				windowIndexSpan,	// span of fitting window in terms of the number of data points included (will be rounded up to nearest odd number). Can be used as an alternative to windowTimeSpan. Ignored if <=1.
												int					order){
	std::vector<double> smoothened_data;
	const bool success = smoothenTimeSeriesSavitzkyGolay(times, data, windowTimeSpan, windowIndexSpan, order, true, smoothened_data);
	return Rcpp::List::create(	Rcpp::Named("success") 			= success,
								Rcpp::Named("smoothened_data") 	= smoothened_data);
}



// Use Savitzky-Golay filter to estimate nth-derivative of a time series
// Elaboration: At each data point, SG filter fits a polynomial to local (nearby) data points using least squares
//				The coefficients of that polynomial are then used to estimate the derivative at that points
//	Note: 	This method DOES NOT ensure that the antiderivative of the estimated derivative (i.e. its integral) equals the original signal or the SG-smoothened signal
//			If you need consistency between the integrated derivative and the smoothened signal, consider SG-smoothening the signal and then taking the derivative using some finite differences scheme
// Currently only 1st and 2nd derivatives are supported (but it shouldn't be hard to extend this function to higher orders)
// Only fitting polynomials of orders 2-4 are supported
// Time points are assumed to be sorted in ascending order
// If the sliding window contains too few data points, the order of the fitted polynomials may be locally reduced
// Note: Time points for which the derivative cannot be estimated (e.g. too few time points available in sliding window) will have derivate==NAN_D
// This function is NAN-sensitive, i.e. NANs in data[] are avoided.
// The set of time points included for each fitting is either specified via windowTimeSpan or windowIndexSpan
template<class TIME_ARRAY, class VALUE_ARRAY>
bool derivativeOfTimeSeries_SavitzkyGolay(	const TIME_ARRAY	&times,				// time points in ascending order
											const VALUE_ARRAY	&data, 	
											double				windowTimeSpan,		// span of fitting window in time units. Ignored if <=0.
											long				windowIndexSpan,	// span of fitting window in terms of the number of data points included (will be rounded up to nearest odd number). Can be used as an alternative to windowTimeSpan. Ignored if <=1.
											int					polynomialOrder,
											int					derivativeOrder,
											vector<double> 		&derivative){
	if(times.size() != data.size()) return false;
	if((polynomialOrder<0) || (polynomialOrder>4) || (derivativeOrder<1) || (derivativeOrder>2) || (polynomialOrder<=derivativeOrder)) return false;
	derivative.clear();
	const long N = times.size();
	if(N<1) return true;	// nothing to do
	double coeff_A, coeff_B, coeff_C, coeff_D, coeff_E;
	long M, localPolynomialOrder, distinctTimePoints;
	vector<double> localTimes, localData;
	if(windowTimeSpan>0) windowTimeSpan = max(0.0, min(windowTimeSpan, times[N-1]-times[0]));
	else if(windowIndexSpan>1) windowIndexSpan = min(windowIndexSpan,N-1);

	derivative.resize(N);
	for(long n=0, nl,nr; n<N; ++n){
		const double t = times[n];
		// determine data points to include
		if(windowIndexSpan>1){
			nl = n-windowIndexSpan/2;
			nr = n+windowIndexSpan/2;
			if(nl<0) nr += (0-nl); // shift to the right since we're at the left edge
			else if(nr>=N) nl -= (nr-N+1); // shift to the left since we're at the right edge
		}else{
			const double tl = t-windowTimeSpan/2;
			const double tr = t+windowTimeSpan/2;
			if(times[0]>=tl){
				nl=0;
			}else{
				for(nl=n; times[nl-1]>=tl; --nl){}
			}
			if(times[N-1]<=tr){
				nr=N-1;
			}else{
				for(nr=n; times[nr+1]<=tr; ++nr){}
			}
		}
		nl = max(0l,nl);
		nr = min((long)(N-1),nr);
		M  = nr-nl+1; // number of local data points covered by window
						
		if(M<derivativeOrder+2){
			// definitely too few data points for estimating derivative 
			derivative[n] = NAN_D;
		
		}else{
			// count distinct time points covered by window
			distinctTimePoints = 1;
			for(long i=nl+1; i<=nr; ++i){ if(times[i]>times[i-1]) ++distinctTimePoints; }
			
			// use lower polynomial order if too few distinct time points
			localPolynomialOrder = (int)min(distinctTimePoints-1,(long)polynomialOrder);
			
			if(localPolynomialOrder<derivativeOrder+1){
				// polynomial order is too low for this derivative order
				derivative[n] = NAN_D;
			}else{
				// prepare local subset of data for polynomial fitting
				localTimes.resize(M);
				localData.resize(M);
				for(long i=nl; i<=nr; ++i){
					localTimes[i-nl] = times[i];
					localData[i-nl]	 = data[i];
				}

				// fit polynomial
				coeff_A = coeff_B = coeff_C = coeff_D = coeff_E = 0;
				switch(localPolynomialOrder){
				case 1: 
					fitLinearRegressionNANSensitive(&localTimes[0], &localData[0], M, coeff_B, coeff_A); 
					break;			
				case 2:
					fitLeastSquares_Quadratic(localTimes, localData, coeff_A, coeff_B, coeff_C, 1e-6);	
					break;			
				case 3:
					fitLeastSquares_Qubic(localTimes, localData, coeff_A, coeff_B, coeff_C, coeff_D, 1e-6);	
					break;	
				case 4:
					fitLeastSquares_Quartic(localTimes, localData, coeff_A, coeff_B, coeff_C, coeff_D, coeff_E, 1e-6);	
					break;	
				}
				
				// calculate derivative using polynomial coefficients
				switch(derivativeOrder){
				case 1: derivative[n] = coeff_B + 2*coeff_C*t + 3*coeff_D*SQ(t) + 4*coeff_E*Qube(t); break;
				case 2: derivative[n] = 2*coeff_C + 6*coeff_D*t + 12*coeff_E*SQ(t); break;
				}
			}
		}
	}
	return true;														
}





// given some density curve on an X-interval, define a non-uniform X-grid on that interval so that the density of grid points reflects the requested density
// this can be used for example to define an age grid, with the grid density reflecting the number of lineages in a timetree at any given age, e.g. for fitting purposes
// the density curve is specified as a piecewise linear function. The density must be non-negative, and strictly positive almost everywhere.
// [[Rcpp::export]]
NumericVector get_inhomogeneous_grid_1D_CPP(const double 				Xstart, 
											const double 				Xend, 
											const long 					Ngrid, 
											const std::vector<double> 	&densityX, 
											const std::vector<double> 	&densityY,
											const double				xepsilon){	// absolute allowed error tolerance on x-scale
	// calculate antiderivative of density, such that A(Xstart) = 0
	dvector Acoeff, A;
	const long Adegree = 2;
	get_antiderivative(densityX, Xstart, densityY, A, Acoeff);
	
	// determine value of A at Xend
	const long g_end  = max(0l,find_next_left_grid_point(densityX, Xend, -1));
	const double Aend = polynomial_value(Adegree,&Acoeff[g_end*(Adegree+1)],Xend);
	
	// normalize antiderivative A such that Aend = Ngrid-1
	Acoeff *= (Ngrid-1.0)/Aend;
	A *= (Ngrid-1.0)/Aend;
		
	// place the n-th grid point (0-based) at the x-value where A(x)=n
	dvector Xgrid(Ngrid);
	Xgrid[0] = Xstart;
	Xgrid[Ngrid-1] = Xend;
	for(long n=1, ga=0, gb=Ngrid-1; n<Ngrid-1; ++n){
		Xgrid[n] = solve_piecewise_polynomial_bisection(densityX,Acoeff,Adegree,false,Xgrid[n-1],Xend,double(n),true,xepsilon,1e-6,1000000l,ga,gb);
	}
	
	return Rcpp::wrap(Xgrid);
}



// given two time-grids X & Y (listing times in ascending order), merge them into a single super-time-grid Z, i.e. such that Z is the union of X and Y
// this function also returns the indices of the input grid points in the resulting super-grid
void merge_time_grids(	const dvector 	&X,		// (INPUT) input time-grid X, listing times in ascending order
						const dvector 	&Y,		// (INPUT) input time-grid Y, listing times in ascending order
						dvector			&Z,		// (OUTPUT) merged output time-grid, listing times in ascending order
						lvector			&XinZ,	// (OUTPUT) integer vector of the same length as X, listing the locations of the X points in Z
						lvector			&YinZ){	// (OUTPUT) integer vector of the same length as Y, listing the locations of the Y points in Z
	const long NX = X.size();
	const long NY = Y.size();
	Z.clear(); Z.reserve(NX+NY);
	XinZ.resize(NX);
	YinZ.resize(NY);
	if(NX==0){
		Z = Y;
		for(long n=0; n<NY; ++n) YinZ[n] = n;
		return;
	}else if(NY==0){
		Z = X;
		for(long n=0; n<NX; ++n) XinZ[n] = n;
		return;		
	}
	
	long next_x = 0, next_y = 0;
	while((next_x<NX) || (next_y<NY)){
		if((next_x>=NX) || ((next_y<NY) && (Y[next_y]<=X[next_x]))){
			// add next Y-element to Z, if increment is non-zero
			if(Z.empty() || (Z.back()!=Y[next_y])){
				Z.push_back(Y[next_y]);			
			}
			YinZ[next_y] = Z.size()-1;
			++next_y;
		}else if((next_y>=NY) || ((next_x<NX) && (X[next_x]<=Y[next_y]))){
			// add next X-element to Z, if increment is non-zero
			if(Z.empty() || (Z.back()!=X[next_x])){
				Z.push_back(X[next_x]);
			}
			XinZ[next_x] = Z.size()-1;
			++next_x;
		}
	}
}



// given two time-grids X & Y (listing times in ascending order), merge them into a single super-time-grid Z, i.e. such that Z is the union of X and Y
// this function allows restricting the resulting grid to some interval [minZ,maxZ]; note that Z will only include points originally in X & Y, so it's not guaranteed that minZ or maxZ will be included in Z
void merge_time_grids(	const dvector 	&X,		// (INPUT) input time-grid X, listing times in ascending order
						const dvector 	&Y,		// (INPUT) input time-grid Y, listing times in ascending order
						const long		minZ,	// (INPUT) smallest value to include from either grid
						const long		maxZ,	// (INPUT) largest value to include from either grid
						dvector			&Z){	// (OUTPUT) merged output time-grid, listing times in ascending order
	Z.clear();	
	long next_x = find_next_right_grid_point(X, minZ, -1);
	long next_y = find_next_right_grid_point(Y, minZ, -1);
	const long last_x = find_next_left_grid_point(X, maxZ, -1);
	const long last_y = find_next_left_grid_point(Y, maxZ, -1);
	if(((next_x<0) || (last_x<0)) && ((next_y<0) || (last_y<0))){
		// neither X nor Y overlap with [minZ,maxZ], so return empty grid
		return;
	}else if((next_x<0) || (last_x<0)){
		// X does not overlap with the requested interval [minZ,maxZ], so just keep the proper subset from Y
		Z.insert(Z.end(), Y.begin()+next_y, Y.begin()+last_y+1);
		return;
	}else if((next_y<0) || (last_y<0)){
		// Y does not overlap with the requested interval [minZ,maxZ], so just keep the proper subset from X
		Z.insert(Z.end(), X.begin()+next_x, X.begin()+last_x+1);
		return;
	}
	Z.reserve((last_x-next_x+1)+(last_y-next_y+1));
	while((next_x<=last_x) || (next_y<=last_y)){
		if((next_x>last_x) || ((next_y<=last_y) && (Y[next_y]<=X[next_x]))){
			// add next Y-element to Z, if increment is non-zero
			if(Z.empty() || (Z.back()!=Y[next_y])){
				Z.push_back(Y[next_y]);			
			}
			++next_y;
		}else if((next_y>last_y) || ((next_x<=last_x) && (X[next_x]<=Y[next_y]))){
			// add next X-element to Z, if increment is non-zero
			if(Z.empty() || (Z.back()!=X[next_x])){
				Z.push_back(X[next_x]);
			}
			++next_x;
		}
	}
}



// remove values from a numeric series (i.e., set to NaN) as needed to ensure that the remaining series is monotonically increasing or decreasing
// [[Rcpp::export]]
Rcpp::List monotonize_series_by_pruning_CPP(std::vector<double> values,	// 1D array of size N, listing the scalar time series values. May include NaN.
											const bool 			increasing,
											const bool			prefer_later_data){
	const long N = values.size();
	long Nremoved = 0;
	if(prefer_later_data){
		double last_valid_value = values[N-1];
		for(long n=N-2; n>=0; --n){
			if(std::isnan(last_valid_value)){
				last_valid_value = values[n];
			}else if(!std::isnan(values[n])){
				if((increasing && (values[n]>last_valid_value)) || ((!increasing) && (values[n]<last_valid_value))){
					// value[n] violates the monotonicity compared to the later data, so remove
					values[n] = NAN_D;
					++Nremoved;
				}else{
					last_valid_value = values[n];
				}
			}
		}
	}else{
		double last_valid_value = values[0];
		for(long n=1; n<N; ++n){
			if(std::isnan(last_valid_value)){
				last_valid_value = values[n];
			}else if(!std::isnan(values[n])){
				if((increasing && (values[n]<last_valid_value)) || ((!increasing) && (values[n]>last_valid_value))){
					// value[n] violates the monotonicity compared to the earlier data, so remove
					values[n] = NAN_D;
					++Nremoved;
				}else{
					last_valid_value = values[n];
				}
			}
		}
	}
	return Rcpp::List::create(	Rcpp::Named("values") 	= Rcpp::wrap(values),
								Rcpp::Named("Nremoved") = Nremoved);
}
					


// modify values of a time series (via linear interpolation) as needed to ensure that the modified time series is monotonically increasing or decreasing
// NaN input values are also replaced via interpolation
// problematic values at edges where interpolation is not available, will be replaced by NaN
// [[Rcpp::export]]
Rcpp::List monotonize_series_via_interpolation_CPP(	const std::vector<double> 	&times,		// 1D array of size N, listing times in ascending order.
													std::vector<double> 		values,	// 1D array of size N, listing the scalar time series values at times[]. May include NaN.
													const bool 					increasing,
													const bool					prefer_later_data){
	const long N 					= values.size();
	double max_violation 			= 0;
	double max_relative_violation 	= 0;
	long Nmodified 					= 0;
	double max_interpolation_gap	= 0; // maximum time interval over which we ended up interpolating
	long last_valid 				= -1;
	lvector invalids; // keep track of encountered invalid points, to be interpolated whenever the opportunity arises (i.e. an enclosing valid bracket is found)
	if(prefer_later_data){
		for(long n=N-1; n>=0; --n){
			if(std::isnan(values[n])){
				invalids.push_back(n);
				++Nmodified;
			}else if(last_valid<0){
				last_valid = n;
			}else{
				if((increasing && (values[n]>values[last_valid])) || ((!increasing) && (values[n]<values[last_valid]))){
					// value[n] violates the monotonicity compared to the later data
					max_violation = max(max_violation, abs(values[n]-values[last_valid]));
					max_relative_violation = max(max_relative_violation, abs(values[n]-values[last_valid])/abs(0.5*(values[last_valid]+values[n])));
					invalids.push_back(n);
					values[n] = NAN_D; // set to NaN for now; if interpolation becomes possible later on, this will be set to the interpolation
					++Nmodified;
				}else{
					// replace all gathered invalids[] via linear interpolation between last_valid and n
					max_interpolation_gap = max(max_interpolation_gap, abs(times[n]-times[last_valid]));
					for(long k=0; k<invalids.size(); ++k){
						values[invalids[k]] = interpolate_linear(times[n], values[n], times[last_valid], values[last_valid], times[invalids[k]]);
					}
					invalids.clear();
					last_valid = n;
				}
			}
		}
	}else{
		for(long n=0; n<N; ++n){
			if(std::isnan(values[n])){
				invalids.push_back(n);
				++Nmodified;
			}else if(last_valid<0){
				last_valid = n;
			}else{
				if((increasing && (values[n]<values[last_valid])) || ((!increasing) && (values[n]>values[last_valid]))){
					// value[n] violates the monotonicity compared to the earlier data
					max_violation = max(max_violation, abs(values[last_valid]-values[n]));
					max_relative_violation = max(max_relative_violation, abs(values[n]-values[last_valid])/abs(0.5*(values[last_valid]+values[n])));
					invalids.push_back(n);
					values[n] = NAN_D; // set to NaN for now; if interpolation becomes possible later on, this will be set to the interpolation
					++Nmodified;
				}else{
					// replace all gathered invalids[] via linear interpolation between last_valid and n
					max_interpolation_gap = max(max_interpolation_gap, abs(times[n]-times[last_valid]));
					for(long k=0; k<invalids.size(); ++k){
						values[invalids[k]] = interpolate_linear(times[n], values[n], times[last_valid], values[last_valid], times[invalids[k]]);
					}
					invalids.clear();
					last_valid = n;
				}
			}
		}
	}
	return Rcpp::List::create(	Rcpp::Named("values") 	 				= Rcpp::wrap(values),
								Rcpp::Named("max_violation") 			= max_violation,
								Rcpp::Named("max_relative_violation") 	= max_relative_violation,
								Rcpp::Named("max_interpolation_gap") 	= max_interpolation_gap,
								Rcpp::Named("Nmodified") 				= Nmodified);
}
			

#pragma mark -
#pragma mark Class: LinearInterpolationFunctor
#pragma mark -


template<class TYPE>
inline void convex_combination(const TYPE &y1, const TYPE &y2, const double lambda, TYPE &y){
	y = (1-lambda)*y1 + lambda*y2;
}


// specialize for vector-valued y
template<>
inline void convex_combination(const std::vector<double> &y1, const std::vector<double> &y2, const double lambda, std::vector<double> &y){
	const long N = y1.size();
	y.resize(N);
	for(long i=0; i<N; ++i){
		y[i] = (1-lambda)*y1[i] + lambda*y2[i];
	}
}




//Class for defining piecewiese linear function using a regular or irregular reference grid
//In the case of a regular grid, value retrieval is of time complexity O(1)
//In the case of an irregular grid, value retrieval is fastest when subsequent requested points are close to each other, because the search for the enclosing grid cell starts from the previous requested grid cell
template<class VALUE_TYPE>
class LinearInterpolationFunctor{
private:
	std::vector<double> 	referencePoints; // store reference points, in the case of an irregular grid. Will be empty iff the grid was regular during setup.
	std::vector<VALUE_TYPE>	referenceValues;
	
	double		domain_min, domain_max;
	double		domainStep;
	double		lengthScale;
	
	bool		periodic;
	VALUE_TYPE	outlier_value_left, outlier_value_right;

	// keep track of the referencePoint matched to the last requested time point
	// This helps find the next reference point faster in the case of non-regular grids, 
	//   assuming that adjacent requests are for times in close proximity (as is the case in ODE solvers)
	// last_requested_reference will correspond to the referencePoint at or immediately to the left (i.e. below) the last requested time
	mutable long last_requested_reference;
	
	void 	set_to_regular_grid_values(	long 				referenceCount,
										const double		domain_min, 
										const double		domain_max, 
										VALUE_TYPE 			referenceValues[],
										bool				periodic,
										const VALUE_TYPE	&outlier_value_left,	// value to use for extending time series to the left if needed, if periodic==false. Irrelevant if periodic==true.
										const VALUE_TYPE	&outlier_value_right);	// value to use for extending time series to the right if needed, if periodic==false. Irrelevant if periodic==true.
public:

	LinearInterpolationFunctor(): domain_min(0), domain_max(0), domainStep(0), lengthScale(1), periodic(false), last_requested_reference(-1) {}

	// constructor setting the functor to some constant value
	LinearInterpolationFunctor(const VALUE_TYPE &value);
	
	// define piecewise linear function, on a regular or irregular gird
	// domain is assumed to span from lowest to highest referencePoints
	// If coordinate is periodic, the last reference point is identified with the first one (its reference value is taken to be (referenceValues[0]+referenceValues[last])/2)
	// Optionally, the function can be internally re-mapped onto a regular grid, for more efficient later value retrieval
	// In the case of an irregular grid (preInterpolateOnRegularGrid=false), later value retrieval is fastest if sequentially requested points are close to each other, because the search starts from the last requested point
	//   Hence, for example, if an ODE solver requests values at non-decreasing points, then value retrieval will be of time-complexity O(N/R) where N is the number of stored reference points and R is the total number of requested points
	LinearInterpolationFunctor(	const std::vector<double> 		&referencePoints, // domain grid points, in ascending order
								const std::vector<VALUE_TYPE> 	&referenceValues,
								bool							periodic,	// if true, time series are extended periodically if needed (e.g. if the reference domain is too small). Otherwise they are extended with zeros.
								const VALUE_TYPE				&outlier_value_left,
								const VALUE_TYPE				&outlier_value_right,
								bool							preInterpolateOnRegularGrid, // if true, then the function is internally pre-interpolated on a regular domain grid. In that case, value retrieval later on will be more efficient.
								double							regularGridStep);	// regular reference-grid step to use, if preInterpolateOnRegularGrid==true. If <=0, then the regular grid step is chosen as the mean step in the (irregular) input grid.
	
	//define piecewise linear function on a regular grid
	//referenceValues[i] should correspond to domain_min + i * intervalWidth(domain)/(referenceCount-1)
	//in particular, referenceValue[referenceCount-1] should correspond to domain_max
	//if coordinate is periodic, the last reference point is identified with the first one (its reference value is taken to be (referenceValues[0]+referenceValues[last])/2)
	//since the domain grid is assumed to be regular, interpolations defined this way are much for efficient
	LinearInterpolationFunctor(	long 				referenceCount, 
								const double		domain_min, 
								const double		domain_max, 
								VALUE_TYPE 			referenceValues[],
								bool				periodic,
								const VALUE_TYPE	&outlier_value_left,
								const VALUE_TYPE	&outlier_value_right);
							
	VALUE_TYPE operator()(double x) const;
	void getValue(double x, VALUE_TYPE &y) const; // equivalent to operator(), but avoiding copy operators
	
	bool isNULL() const{ return referenceValues.empty(); }
	bool isPeriodic() const{ return periodic; }
	void getDomain(double &domain_min_, double &domain_max_) const{ domain_min_ = domain_min; domain_max_ = domain_max; }
	long getReferenceCount() const{ return referenceValues.size(); }
	
	void setTypicalLengthScale(double _lengthScale){ lengthScale = _lengthScale; } 
	double typicalLengthScale() const{ return lengthScale; }
	double getMaxAbs() const{ return vector_max_abs(referenceValues); }
	
	const std::vector<double> &getReferencePoints() const{ return referencePoints; }
	const std::vector<double> &getReferenceValues() const{ return referenceValues; }
	
	inline VALUE_TYPE getFirstReferenceValue() const{ return referenceValues[0]; }
	inline VALUE_TYPE getLastReferenceValue() const{ return referenceValues.back(); }
	inline const VALUE_TYPE* getFirstReferenceValuePointer() const{ return &referenceValues[0]; }
	inline const VALUE_TYPE* getLastReferenceValuePointer() const{ return &referenceValues[referenceValues.size()-1]; }
	
	// obtain a copy of this functor in the form of an Rcpp::List
	Rcpp::List get_copy_as_RcppList() const{
		return Rcpp::List::create(	Rcpp::Named("referencePoints")		= referencePoints,
									Rcpp::Named("referenceValues") 		= referenceValues,
									Rcpp::Named("domain_min") 			= domain_min,
									Rcpp::Named("domain_max") 			= domain_max,
									Rcpp::Named("domainStep") 			= domainStep,
									Rcpp::Named("lengthScale") 			= lengthScale,
									Rcpp::Named("periodic") 			= periodic,
									Rcpp::Named("outlier_value_left") 	= outlier_value_left,
									Rcpp::Named("outlier_value_right") 	= outlier_value_right);
	}

	// instantiate this functor from an Rcpp::List
	void set_to_RcppList(const Rcpp::List &L){
		referencePoints		= Rcpp::as<std::vector<double> >(L["referencePoints"]);
		referenceValues 	= Rcpp::as<std::vector<VALUE_TYPE> >(L["referenceValues"]);
		domain_min 			= Rcpp::as<double>(L["domain_min"]);
		domain_max 			= Rcpp::as<double>(L["domain_max"]);
		domainStep 			= Rcpp::as<double>(L["domainStep"]);
		lengthScale 		= Rcpp::as<double>(L["lengthScale"]);
		periodic 			= Rcpp::as<bool>(L["periodic"]);
		outlier_value_left	= Rcpp::as<VALUE_TYPE>(L["outlier_value_left"]);
		outlier_value_right = Rcpp::as<VALUE_TYPE>(L["outlier_value_right"]);
	}
};



template<class VALUE_TYPE>
LinearInterpolationFunctor<VALUE_TYPE>::LinearInterpolationFunctor(const VALUE_TYPE &value){
	referenceValues.clear();
	referencePoints.clear();
	outlier_value_left  = value;
	outlier_value_right = value;
	domain_min 	= 0;
	domain_max 	= 0;
	domainStep 	= 0;
	lengthScale = 1;
	periodic	= false;
	last_requested_reference = -1;
}



template<class VALUE_TYPE>
LinearInterpolationFunctor<VALUE_TYPE>::LinearInterpolationFunctor(	const std::vector<double> 		&referencePoints_, 
														const std::vector<VALUE_TYPE> 	&referenceValues_,
														bool							periodic_,
														const VALUE_TYPE				&outlier_value_left_,
														const VALUE_TYPE				&outlier_value_right_,
														bool							preInterpolateOnRegularGrid,
														double							regularGridStep){
	periodic 			= periodic_;
	outlier_value_left 	= outlier_value_left_;
	outlier_value_right	= outlier_value_right_;
	referencePoints.clear(); 
	referenceValues.clear();
	last_requested_reference = -1;
	if(referencePoints_.empty()) return;

	if(preInterpolateOnRegularGrid && (referencePoints_.size()>1)){
		// pre-interpolate on regular domain grid, then setup interpolator functor
		const double domain_min_ 	= referencePoints_.front();
		const double domain_span	= (referencePoints_.back()-referencePoints_.front());
		regularGridStep 			= (regularGridStep<=0 ? domain_span/(referencePoints_.size()-1) : regularGridStep);
		const long NR 				= max(2l, long(1 + (referencePoints_.back() - referencePoints_.front())/regularGridStep));
		std::vector<double> regular_reference_points(NR);
		std::vector<VALUE_TYPE> regular_reference_values;
		for(long i=0; i<NR; ++i) regular_reference_points[i] = domain_min_ + i*regularGridStep;
		regular_reference_points[NR-1] = referencePoints_.back(); // make sure last regular_reference_point is equal to the original last _referencePoint
		long includedNewTimesStart=0, includedNewTimesEnd=0;
		interpolateTimeSeriesAtTimes<VALUE_TYPE,vector<double> >(	referencePoints_,
																	referenceValues_,
																	0,
																	referencePoints_.size()-1,
																	regular_reference_points,
																	0,
																	NR-1,
																	includedNewTimesStart,
																	includedNewTimesEnd,
																	regular_reference_values);
		set_to_regular_grid_values(	1+includedNewTimesEnd-includedNewTimesStart, 
									regular_reference_points[includedNewTimesStart], 
									regular_reference_points[includedNewTimesEnd], 
									&regular_reference_values[includedNewTimesStart], 
									periodic_,
									outlier_value_left_,
									outlier_value_right_);
		
	}else{
		// use provided irregular grid
		referencePoints = referencePoints_;
		referenceValues = referenceValues_;
		const long referenceCount = referencePoints.size();
		
		// figure out domain
		domain_min	= referencePoints.front();
		domain_max	= referencePoints.back();
		lengthScale	= domain_max - domain_min;
		
		// force periodic boundaries if necessary
		if(periodic){
			referenceValues[0] = referenceValues[referenceCount-1] = 0.5*(referenceValues[0] + referenceValues[referenceCount-1]);
		}
	}
}


template<class VALUE_TYPE>
void LinearInterpolationFunctor<VALUE_TYPE>::set_to_regular_grid_values(	long 				referenceCount, 
																	const double 		domain_min_, 
																	const double 		domain_max_, 
																	VALUE_TYPE 			*referenceValues_,
																	bool				periodic_,
																	const VALUE_TYPE	&outlier_value_left_,
																	const VALUE_TYPE	&outlier_value_right_){

	periodic			= periodic_;
	domain_min			= domain_min_;
	domain_max			= domain_max_;
	lengthScale			= (domain_max-domain_min);
	domainStep			= (domain_max-domain_min)/max(1.0, referenceCount-1.0);
	outlier_value_left 	= outlier_value_left_;
	outlier_value_right	= outlier_value_right_;
	referencePoints.clear(); 
	referenceValues.clear();
	last_requested_reference = -1;
	if(referenceCount == 0) return;
		
	referenceValues.resize(referenceCount);
	for(long i=0; i<referenceCount; ++i){
		referenceValues[i] = referenceValues_[i];
	}
		
	// enforce periodic boundary values if necessary
	if(periodic){
		referenceValues[0] = referenceValues[referenceCount-1] = (referenceValues[0] + referenceValues[referenceCount-1])/2.0;
	}
}


template<class VALUE_TYPE>
LinearInterpolationFunctor<VALUE_TYPE>::LinearInterpolationFunctor(	long 				referenceCount, 
														const double 		domain_min_, 
														const double 		domain_max_, 
														VALUE_TYPE 			referenceValues_[],
														bool				periodic_,
														const VALUE_TYPE	&outlier_value_left_,
														const VALUE_TYPE	&outlier_value_right_){
	set_to_regular_grid_values(referenceCount, domain_min_, domain_max_, referenceValues_, periodic_, outlier_value_left_, outlier_value_right_);
}



template<class VALUE_TYPE>
void LinearInterpolationFunctor<VALUE_TYPE>::getValue(double x, VALUE_TYPE &y) const{
	if(referenceValues.empty()){
		//nothing available
		y = outlier_value_left;
		return;
	}
	const long referenceCount = referenceValues.size();
	if(periodic){
		x = moduloInterval(x, domain_min, domain_max);
	}else if(x < domain_min){
		// requested x is outside (left) of the reference domain
		y = outlier_value_left;
		last_requested_reference = 0;
		return;
	}else if(x > domain_max){
		// requested x is outside (right) of the reference domain
		y = outlier_value_right;
		last_requested_reference = referenceCount-1;
		return;
	}
	if(referenceCount == 1){
		y = referenceValues[0];
		last_requested_reference = 0;
		return;
	}
	
	if(referencePoints.empty()){
		long j = floor((x - domain_min)/domainStep); //left border of relevant domain-interval
		j = min(j, referenceCount - 1); //avoid out-of-bounds errors caused by numerical inaccuracies
		last_requested_reference = j;
		if(j == referenceCount - 1){
			y = referenceValues[referenceCount - 1];
		}else{
			// linear interpolation between referenceValues
			const double lambda = (x - (domain_min + j*domainStep))/domainStep;
			convex_combination(referenceValues[j], referenceValues[j+1], lambda, y);
		} 
	}else{
		long j;
		if(last_requested_reference<0) last_requested_reference = 0;
		if(referencePoints[last_requested_reference]<=x){
			// search for reference point, towards the right
			for(j=last_requested_reference; j<referenceCount-1; ++j){
				if(referencePoints[j+1] > x) break;
			}
		}else{
			// search for reference point, towards the left
			for(j=last_requested_reference; j>=0; --j){
				if(referencePoints[j] <= x) break;
			}
		}
		if(j>=referenceCount-1){
			y = referenceValues[referenceCount - 1];
			last_requested_reference = referenceCount - 1;
		}else if(j<=0){
			y = referenceValues[0];
			last_requested_reference = 0;			
		}else{
			// linear interpolation between grid points
			const double lambda = (x - referencePoints[j])/(referencePoints[j+1] - referencePoints[j]);
			convex_combination(referenceValues[j], referenceValues[j+1], lambda, y);
			last_requested_reference = j;
		}
	}
}


template<class VALUE_TYPE>
VALUE_TYPE LinearInterpolationFunctor<VALUE_TYPE>::operator()(double x) const{
	VALUE_TYPE V;
	getValue(x,V);
	return V;
}





/* UNFINISHED

#pragma mark -
#pragma mark Class: PiecewisePolynomial
#pragma mark -



//Class for defining piecewiese polynomial function on a 1D grid
//In the case of a regular grid, value retrieval is of time complexity O(1)
//In the case of an irregular grid, value retrieval is fastest when subsequent requested points are close to each other, because the search for the enclosing grid cell starts from the previous requested grid cell
template<class VALUE_TYPE>
class PiecewisePolynomial{
protected:
	long 					order;	// order of the piecewise polynomials. For example, piecewise linear functions have order 1.
	std::vector<double> 	Xgrid;	// 1D array of size NG, listing x-values in ascending order
	std::vector<VALUE_TYPE>	Ycoeff;	// 2D array of size NG x (order+1) in row-major format, listing polynomial coefficients in each Xgrid interval. Ycoeff[g*(order+1) + p] will be the p-th polynomial coefficient applicable within the x-interval Xgrid[g]:Xgrid[g+1]
	
	// keep track of the Xgrid interval matched to the last requested x
	// This helps find the next grid point faster in the case of non-regular grids, 
	//   assuming that adjacent requests are for x values in close proximity (as is the case in ODE solvers)
	// last_grid_point will correspond to the Xgrid point at or immediately to the left (i.e. below) the last requested x
	mutable long last_grid_cell;
	
	// evaluate the piecewise polynomial function at some x, based on polynomial coefficients specified at some grid cell g.
	void evaluate_at_grid_cell const(	const double 	x,			// (INPUT) requested x-value at which to calculate y
										const double 	g,			// (INPUT) integer within 0,..,NG-1, specifying the grid cell from which to take polynomial coefficients
										VALUE_TYPE		&y) const{	// (OUTPUT) y-value calculated at the requested x
		y = Ycoeff[g*(order+1)+0];
		for(long p=1; p<=order; ++p){
			y += Ycoeff[g*(order+1)+p] * pow(x,p);
		}
	}
	
	// find nearest Xgrid point immediately below (or equal to) x
	// if x<Xgrid[0], then -1 is returned
	long find_grid_cell const(const double x, long last_g){
		const long NG = Xgrid.size();
		if(NG==0) return -1;
		if(x<Xgrid[0]) return -1;
		long g;
		if(last_g<0) last_g = 0;
		if(Xgrid[last_g]<=x){
			// search for grid point, towards the right
			for(g=last_g; g<NG-1; ++g){
				if(Xgrid[g+1] > x) break;
			}
		}else{
			// search for grid point, towards the left
			for(g=last_g; g>=0; --g){
				if(Xgrid[g] <= x) break;
			}
		}
		return g;
	}
	
public:

	PiecewisePolynomial(): order(0), last_grid_cell(-1) {}

	// constructor setting the functor to some constant value
	PiecewisePolynomial(const VALUE_TYPE &value){
		order = 0;
		Xgrid.assign(1,0);
		Ycoeff.assign(1,value);
		last_grid_cell = -1;
	}
	
	// calculate the antiderivative of the function over x, i.e. A(x):=int_Xstart^x Y(u) du
	// the result will be a piecewise polynomial of higher order
	void get_antiderivative(const double Xstart, PiecewisePolynomial<VALUE_TYPE> &A) const{
		const long NG = Xgrid.size();
		if(NG==0){
			A = PiecewisePolynomial<VALUE_TYPE>();
			return;
		}
		A.order = order+1;
		A.Xgrid = Xgrid;
		A.Ycoeff.resize(NG*(order+2));
		A.last_grid_cell = -1;
		cumulativeA = 0;
		double alpha;
		for(long g=1; g<NG; ++g){
			Acoeff[(i-1)*(order+2) + 0] = cumulativeA;
			for(long p=0; p<=order; ++p){
				// term of order p in Y gives rise to a term of order (p+1) in the antiderivative, as well as part of order 0
				alpha = Ycoeff[(g-1)*(order+1) + p]/(p+1);
				A.Ycoeff[(g-1)*(order+2) + (p+1)]   = alpha;
				A.Ycoeff[(g-1)*(order+2) + 0] 		-= alpha * pow(Xgrid[g-1],p+1);
				cumulativeA += alpha*(pow(Xgrid[g],p+1) - pow(Xgrid[g-1],p+1));
			}
		}
		// set the coefficient at the last grid point equal to the coefficients at the second-last grid point
		for(long p=0; p<=order+1; ++p) A.Ycoeff[(NG-1)*(order+2)+p] = A.Ycoeff[(NG-2)*(order+2)+p];
		
		// shift antiderivative if needed, i.e. if Xstart is different from the convention used above (where A=0 at Xgrid[0]).
		if(Xstart!=Xgrid[0]){
			VALUE_TYPE Astart;
			const long g_start = max(0,A.find_grid_cell(x,0));
			evaluate_at_grid_cell(Xstart,g_start,Astart);
			for(long g=0; g<NG; ++g) A.Ycoeff[g*(order+2)+0] -= Astart;
		}
	}
	
	
		
	// calculate value at some requested x
	// this function is equivalent to operator(), but avoids excessive copying
	void getValue(double x, VALUE_TYPE &y) const{
		const long NG = Xgrid.size();
		if(NG==0){
			//nothing available, or a constant functor
			y = NAN_D;
			return;
		}
		
		// find Xgrid point applicable to the requested x
		// normally this will be the Xgrid point immediately below (or equal to) x, unless x<Xgrid[0]
		const long g = max(0,find_grid_cell(x, last_grid_cell)); // may return -1 if x is below Xgrid[0], so cap at 0 if needed
		evaluate_at_grid_cell(x,g,y);
		last_grid_cell = g;
	}

	
	VALUE_TYPE operator()(double x) const {
		VALUE_TYPE V;
		getValue(x,V);
		return V;
	}

	long get_order() const{ return order; }
}


*/




#pragma mark -
#pragma mark Stochastic processes
#pragma mark 



// generate a sample from an OU process, conditional upon a previous sample
inline double get_next_OU_sample(	double mean,
									double decay_rate,
									double stationary_std,
									double dt,
									double previous){
	// use transition probability density (Gaussian, known analytically)
	const double std 		 = stationary_std * sqrt(1-exp(-2*dt*decay_rate));
	const double expectation = mean + (previous-mean) * exp(-dt*decay_rate);
	return expectation + std*random_standard_normal();
	// Alternative: Use correlation structure, and the fact that OU is Gaussian
	//const double rho = ((1.0/decay_rate)<dt*STRANDOM_EPSILON ? 0 : exp(-dt*decay_rate));
	//return mean*(1-rho) + rho*previous + sqrt(1-rho*rho)*random_standard_normal()*stationary_std;								
}


// [[Rcpp::export]]
Rcpp::List get_Ornstein_Uhlenbeck_time_series_CPP(	const std::vector<double> 	&times, 			// (INPUT) 1D array of size NT, listing times in ascending order
													const double				start_value,		// (INPUT) value at times[0]. If NAN_D, it will be chosen automatically
													const double				stationary_mean,	// (INPUT) mean of the stationary distribution
													const double				stationary_std,		// (INPUT) standard deviation of the stationary distribution
													const double 				decay_rate){		// (INPUT) exponential decay rate (or equilibration rate), in units 1/edge_length
	const long NT = times.size();
	dvector values(NT);
	if(NT>0){
		if(std::isnan(start_value)){
			values[0] = stationary_mean + stationary_std*random_standard_normal();
		}else{
			values[0] = start_value;
		}
	}
	for(long t=1; t<NT; ++t){
		values[t] = get_next_OU_sample(stationary_mean, decay_rate, stationary_std, times[t]-times[t-1], values[t-1]);
	}
	return Rcpp::List::create(Rcpp::Named("values") = Rcpp::wrap(values));
}


// generate a sample from a Brownian motion process, conditional upon a previous sample
inline double get_next_BM_sample(	double	diffusivity,
									double	dt,
									double 	previous){
	return previous + sqrt(2 * diffusivity * dt) * random_standard_normal();									
}


// generate a sample from a bounded Brownian motion process (constrained in an interval via reflection), conditional upon a previous sample
// in practice, this is done by first simulating an unbounded BM, and then reflecting at the boundaries as much as needed
double get_next_bounded_BM_sample(	double	diffusivity,
									double	min_state,
									double	max_state,
									double	dt,
									double 	previous){
	// first perform calculations assuming min_state=0, then add it at the end
	double step  = max_state - min_state;
	if(step<=0) return min_state;
	double state = previous + sqrt(2 * diffusivity * dt) * random_standard_normal();
	state -= min_state; // temporarily shift axis so that lower reflection point is at origin
	state = abs(state); // reflection at the origin
	long wrap_count = floor(state/step);
	if(wrap_count%2==0){
		state = state - (wrap_count*step);
	}else if(wrap_count%2==1){
		state = step - (state - (wrap_count*step));
	}
	state += min_state; // reverse temporary axis shift
	return state;
}


// generate a time series according to a bounded Brownian Motion, i.e. a diffusing particle constrained to a finite interval with reflecting boundaries
// [[Rcpp::export]]
Rcpp::List get_bounded_BM_time_series_CPP(	const std::vector<double> 	&times, 		// (INPUT) 1D array of size NT, listing times in ascending order
											const double				start_value,	// (INPUT) value at times[0]. If NAN_D, it will be chosen automatically
											const double				diffusivity,	// (INPUT) the diffusion coefficient of the Brownian Motion, in units value^2/time
											const std::vector<double>	&lower,			// (INPUT) 1D numeric array of length NT. specifying the lower bound at each time point, i.e. the lower reflection point. Can also be a single scalar.
											const std::vector<double> 	&upper){		// (INPUT) 1D numeric array of length NT. specifying the upper bound at each time point, i.e. the upper reflection point. Can also be a single scalar.
	const long NT = times.size();
	const bool scalar_lower = (lower.size()==1);
	const bool scalar_upper = (upper.size()==1);
	dvector values(NT);
	if(NT>0){
		if(std::isnan(start_value)){
			values[0] = R::runif(lower[0],upper[0]);
		}else{
			values[0] = start_value;
		}
	}
	for(long t=1; t<NT; ++t){
		values[t] = get_next_bounded_BM_sample(diffusivity, (scalar_lower ? lower[0] : lower[t]), (scalar_upper ? upper[0] : upper[t]), times[t]-times[t-1], values[t-1]);
	}
	return Rcpp::List::create(Rcpp::Named("values") = Rcpp::wrap(values));
}



// get next discrete state of a continuous-time Markov chain, that has evolved along a specific interval dt
// the returned_state may also be the same as the previous_state (i.e., a transition need not necessarily occur)
long get_next_Mk_state(	const matrix_exponentiator 	&transition_matrix_exponentiator,
						std::vector<double>			&scratch_exp,					// (SCRATCH) scratch space used to store the temporary exponential
						const double 				dt,
						const long 					previous_state){
	const long Nstates = transition_matrix_exponentiator.NR;
	transition_matrix_exponentiator.get_exponential(dt, scratch_exp);
	// use row of exponentiated transition matrix corresponding to the previous state, as probability vector for the next state
	const long next_state = random_int_from_distribution<double>(&scratch_exp[previous_state*Nstates+0], Nstates);
	return next_state;
}



// get next discrete state of a continuous-time Markov chain, based on the transition rate matrix, and conditionally upon a single transition having occurred
// for efficiency, the caller guarantees that total_transition_rate equals the sum of transition rates from old_state
long get_next_Mk_state(	const long 					Nstates,					// (INPUT) number of discrete states
						const std::vector<double>	&transition_matrix,			// (INPUT) transition rate matrix. 2D array of size Nstates x Nstates, in row-major format, such that transition_matrix[r*Nstates+c] is the transition rate r-->c
						const double				total_transition_rate,		// (INPUT) equal to sum_{c != old_state} transition_matrix[old_state*Nstates+c]
						const long					old_state){					// (INPUT) old state, from which a single transition is to be simulated
	double p = R::runif(0.0,1.0);
	for(long c=0; c<Nstates; ++c){
		if(c==old_state) continue; // statying at the old state is not allowed, since we're conditioning this on a transition having occurred
		if(p<=transition_matrix[old_state*Nstates+c]/total_transition_rate) return c;
		p -= transition_matrix[old_state*Nstates+c]/total_transition_rate;
	}
	return Nstates-1;
}



// given some Poissonian rate over time, draw random event times t1<=t2<=t3<=...
// the rate is given as a piecewise polynomial function of time
void get_Poisson_event_times(	const PiecewisePolynomial<double>	&rate,			// (INPUT) the rate of the Poisson process, defined as a piecewise polynomial function (e.g. piecewise linear or spline). Make sure the rate is defined for sufficiently high times, or its extrapolation properties are suitable for calculating a rate at sufficiently large times
								const long 							Nevents,		// (INPUT) number of event times to generate
								const double						time0,			// (INPUT) the time point from which to start counting, i.e. the process starts at this time
								std::vector<double>					&event_times){	// (OUTPUT) 1D array of size Nevents, listing the randomly generated event times, in ascending order
	// integrate the rate over time, R(t):=int_{time0}^t rate(s) ds
	PiecewisePolynomial<double> R;
	rate.get_antiderivative(time0, R);
	const long NRG = R.X.size();
	dvector Rvalues(NRG);
	for(long g=0; g<NRG; ++g){
		Rvalues[g] = R.value_at_grid_point(g);
	}
	
	// generate scaled event times according to a rate-1 Poisson process
	event_times.resize(Nevents);
	for(long e=0; e<Nevents; ++e){
		event_times[e] = (e==0 ? 0.0 : event_times[e-1]) + random_exponential_distribution(1.0);
	}
	
	// scale event times back to original time units
	// true event time = R^{-1}(scaled event time)
	const double yepsilon = 1e-6;
	for(long e=0, g=0; e<Nevents; ++e){
		g = find_next_left_grid_point(Rvalues, event_times[e], g); // find furthest grid point g where Rvalues[g]<=event_times[e]. Note that since R(time0)=0 and R is monotonically increasing and event_times[e]>=0, such a grid point exists
		event_times[e] = time0 + solve_polynomial_via_bisection(&R.coeff[g*(R.degree+1)], 
																R.degree, 
																(R.slideX ? R.X[g] : 0.0), // Xshift
																R.X[g],  // xmin
																(g+1<NRG ? R.X[g+1] : INFTY_D), // xmax 
																event_times[e], // target value
																0, 			// xepsilon, 
																yepsilon,
																100000l); 	// max_iterations
	}
}


// Rcpp wrapper for the function get_Poisson_event_times()
// [[Rcpp::export]]
Rcpp::List get_Poisson_event_times_CPP(	const std::vector<double> 	&time_grid,		// (INPUT) 1D array of size NG, listing times in ascending order, on which the Poissonian rate is defined. Beyond this time grid, the rate is extrapolated as a constant
										const std::vector<double>	&rates,			// (INPUT) 1D array of size NG, listing strictly positive Poissonian rates on the time_grid[]
										const long					splines_degree,	// (INPUT) integer (0,1,2 or 3) specifying the polynomial degree of the Poissonian rate between grid points. E.g. splines_degree==1 means piecewise linear.
										const long					Nevents,		// (INPUT) number of random events to generate
										const double				time0){			// (INPUT) time at which the process begins. Typically this is zero.
	// translate rate time series into a piecewise polynomial
	PiecewisePolynomial<double> rate_polynomial;
	rate_polynomial.set_to_spline(time_grid, rates, splines_degree, true, ExtrapolationTypeConst, ExtrapolationTypeConst, rates[0], rates.back());

	// draw event times
	dvector event_times;
	get_Poisson_event_times(rate_polynomial, Nevents, time0, event_times);
	
	return Rcpp::List::create(	Rcpp::Named("success")		= true,
								Rcpp::Named("event_times") 	= event_times);
	
}



#pragma mark -
#pragma mark Deterministic diversity dynamics
#pragma mark



// Class for representing simulated diversity at any time point, including cumulative speciation & extinction events
class TreeStateHistory{
public:
	double diversity;				// current number of extant species
	double coalescent_diversity;	// number of species in coalescent final tree
	double Nbirths;					// cumulative number of speciation events since the beginning of a simulation
	double Ndeaths;					// cumulative number of extinction events since the beginning of a simulation
	double Pextinction;				// extinction probability of a size-1 clade until max time. Only makes sense for simulations performed in reverse.
	double Pmissing;				// probability of a size-1 clade missing at max time (either extinction or not discovered). Only makes sense for simulations performed in reverse.
	TreeStateHistory(){ diversity = coalescent_diversity = Nbirths = Ndeaths = Pextinction = 0; }
	TreeStateHistory(const double start_diversity){ diversity = coalescent_diversity = start_diversity; Nbirths = Ndeaths = 0; Pextinction = 0; Pmissing = 0; }
	bool isnan() const{ return (std::isnan(diversity) || std::isnan(coalescent_diversity) || std::isnan(Nbirths) || std::isnan(Ndeaths) || std::isnan(Pextinction) || std::isnan(Pmissing)); }
};


inline TreeStateHistory operator*(TreeStateHistory x, double scalar){
	x.diversity 			*= scalar;
	x.coalescent_diversity 	*= scalar;
	x.Nbirths 				*= scalar;
	x.Ndeaths 				*= scalar;
	x.Pextinction 			*= scalar;
	x.Pmissing	 			*= scalar;
	return x;
}


TreeStateHistory& operator*=(TreeStateHistory &x, double scalar) {
	x.diversity 			*= scalar;
	x.coalescent_diversity	*= scalar;
	x.Nbirths 				*= scalar;
	x.Ndeaths 				*= scalar;
	x.Pextinction			*= scalar;
	x.Pmissing				*= scalar;
	return x;
}


inline TreeStateHistory operator+(TreeStateHistory x, const TreeStateHistory &y){
	x.diversity 			+= y.diversity;
	x.coalescent_diversity	+= y.coalescent_diversity;
	x.Nbirths 				+= y.Nbirths;
	x.Ndeaths 				+= y.Ndeaths;
	x.Pextinction			+= y.Pextinction;
	x.Pmissing				+= y.Pmissing;
	return x;
}


inline TreeStateHistory &operator+=(TreeStateHistory &x, const TreeStateHistory &y){
	x.diversity 			+= y.diversity;
	x.coalescent_diversity	+= y.coalescent_diversity;
	x.Nbirths 				+= y.Nbirths;
	x.Ndeaths 				+= y.Ndeaths;
	x.Pextinction			+= y.Pextinction;
	x.Pmissing				+= y.Pmissing;
	return x;
}


inline TreeStateHistory operator-(TreeStateHistory x, const TreeStateHistory &y){
	x.diversity 			-= y.diversity;
	x.coalescent_diversity 	-= y.coalescent_diversity;
	x.Nbirths 				-= y.Nbirths;
	x.Ndeaths 				-= y.Ndeaths;
	x.Pextinction			-= y.Pextinction;
	x.Pmissing				-= y.Pmissing;
	return x;
}


inline TreeStateHistory &operator-=(TreeStateHistory &x, const TreeStateHistory &y){
	x.diversity 			-= y.diversity;
	x.coalescent_diversity	-= y.coalescent_diversity;
	x.Nbirths 				-= y.Nbirths;
	x.Ndeaths 				-= y.Ndeaths;
	x.Pextinction			-= y.Pextinction;
	x.Pmissing				-= y.Pmissing;
	return x;
}


inline TreeStateHistory operator/(TreeStateHistory x, const TreeStateHistory &y){
	x.diversity 			/= y.diversity;
	x.coalescent_diversity	/= y.coalescent_diversity;
	x.Nbirths 				/= y.Nbirths;
	x.Ndeaths 				/= y.Ndeaths;
	x.Pextinction			/= y.Pextinction;
	x.Pmissing				/= y.Pmissing;
	return x;
}


inline TreeStateHistory operator/(TreeStateHistory x, double scalar){
	x.diversity 			/= scalar;
	x.coalescent_diversity	/= scalar;
	x.Nbirths 				/= scalar;
	x.Ndeaths 				/= scalar;
	x.Pextinction			/= scalar;
	x.Pmissing				/= scalar;
	return x;
}


inline TreeStateHistory &operator/=(TreeStateHistory &x, double scalar){
	x.diversity 			/= scalar;
	x.coalescent_diversity	/= scalar;
	x.Nbirths 				/= scalar;
	x.Ndeaths 				/= scalar;
	x.Pextinction			/= scalar;
	x.Pmissing				/= scalar;
	return x;
}



// class for species speciation-extinction model for tree generation
// includes integration of cumulative birth & death events
// can be integrated in reverse (time counted backwards) as well as in forward direction
// When integrated in reverse, the extinction probability Pextinction(t,T) and the probability of missing Pmissing are also simulated
// In any case, the simulated diversity is the true (total) extant diversity at any time. To make the simulated time series coalescent (i.e. only including extant lineages), use calculate_coalescent_diversities() afterwards.
class TreeSpeciationExtinctionModel{
	double 	min_valid_diversity;
	bool 	reverse;
	double 	reflection_time;
	bool	has_probabilities;
public:
	std::vector<TreeStateHistory> trajectory;
	std::vector<double> times;
	double initial_diversity;
	
	// cladogenic model parameters; must be set by whoever creates this class instance
	double rarefaction;
	double birth_rate_intercept, birth_rate_factor, birth_rate_exponent;
	double death_rate_intercept, death_rate_factor, death_rate_exponent;
	long Nsplits;		
	LinearInterpolationFunctor<double> added_birth_rate_pc, added_death_rate_pc; // optional added per-capita birth- and death-rates, for example to model mass extinction events

	TreeSpeciationExtinctionModel(){ 
		min_valid_diversity = 0; 
		reverse = false; 
		rarefaction = 1; 
		has_probabilities = false; 
		added_birth_rate_pc = LinearInterpolationFunctor<double>(0); 
		added_death_rate_pc = LinearInterpolationFunctor<double>(0); 
	}
	
	// use parameters from another model instance
	void adopt_parameters(const TreeSpeciationExtinctionModel &model2){
		birth_rate_intercept 	= model2.birth_rate_intercept;
		birth_rate_factor 		= model2.birth_rate_factor;
		birth_rate_exponent 	= model2.birth_rate_exponent;
		death_rate_intercept 	= model2.death_rate_intercept;
		death_rate_factor 		= model2.death_rate_factor;
		death_rate_exponent 	= model2.death_rate_exponent;
		Nsplits 				= model2.Nsplits;
		min_valid_diversity 	= model2.min_valid_diversity;
		reflection_time 		= model2.reflection_time;
		reverse	 				= model2.reverse;
		added_birth_rate_pc		= model2.added_birth_rate_pc;
		added_death_rate_pc		= model2.added_death_rate_pc;
	}
		
	// model is warned that a time series of size count will need to be stored
	void reserveSpaceForTimeSeries(long Ntimes){ 
		trajectory.clear();
		trajectory.reserve(Ntimes); 
		times.clear();
		times.reserve(Ntimes); 
	}
	
	// reverse the dynamics of the model, i.e. return vector field -V(_reflection_time-t) whenever V(t) is requested.
	// this should be called prior to simulating
	void set_reverse(double _reflection_time){
		reflection_time 	= _reflection_time;
		reverse 			= true;
		has_probabilities 	= true;
	}
	
	// reverse the time course of a simulated trajectory
	// this should be called after simulating
	void reverse_trajectory(const double final_time){
		TreeStateHistory state;
		const long NMT = times.size();
		double time;
		for(long t=0, s; t<NMT/2; ++t){
			s 				= NMT-t-1;
			state 			= trajectory[t];
			trajectory[t] 	= trajectory[s];
			trajectory[s] 	= state;
			time 			= times[t];
			times[t] 		= times[s];
			times[s] 		= time;
		}
		// correct direction of accumulation of time, Nbirths & Ndeaths
		const double max_Nbirths = trajectory[0].Nbirths;
		const double max_Ndeaths = trajectory[0].Ndeaths;
		for(long t=0; t<NMT; ++t){
			times[t] 				= final_time - times[t];
			trajectory[t].Nbirths 	= max_Nbirths - trajectory[t].Nbirths;
			trajectory[t].Ndeaths 	= max_Ndeaths - trajectory[t].Ndeaths;
		}
	}
	
	// calculate probabilities fo extinction and missing, by integrating in reverse
	// this should be done after the end of a simulation, since the ODE's initial condition and rate of change depends on the final trajectory of diversity
	// assumes that times[] are in ascending order
	void calculate_probabilities(){
		if(has_probabilities) return;
		const long NMT = times.size();
		trajectory[NMT-1].Pextinction = 0; 					// probability of extinction begins at 0
		trajectory[NMT-1].Pmissing = (1.0-rarefaction); 	// probability of missing begins at (1-rarefacton_fraction)
		for(long t=NMT-2; t>=0; --t){
			const double birth_rate_pc = get_speciation_rate_at_state(times[t+1],trajectory[t+1].diversity)/trajectory[t+1].diversity;
			const double death_rate_pc = get_extinction_rate_at_state(times[t+1],trajectory[t+1].diversity)/trajectory[t+1].diversity;
			const double dt = times[t+1]-times[t];
			trajectory[t].Pextinction = trajectory[t+1].Pextinction + dt*(death_rate_pc - trajectory[t+1].Pextinction*(birth_rate_pc+death_rate_pc) + pow(trajectory[t+1].Pextinction,Nsplits)*birth_rate_pc);
			trajectory[t].Pmissing	  = trajectory[t+1].Pmissing + dt*(death_rate_pc - trajectory[t+1].Pmissing*(birth_rate_pc+death_rate_pc) + pow(trajectory[t+1].Pmissing,Nsplits)*birth_rate_pc);
		}
		has_probabilities = true;
	}
	
	
	void get_coalescent_trajectory(	double 							resolution, 	// (INPUT)
									double 							rarefaction, 	// (INPUT)
									std::vector<TreeStateHistory> 	&coalescent) const{	// (OUTPUT)
		const long NMT = times.size();
		const double final_time = times.back();
		coalescent = trajectory;
		coalescent[NMT-1].Pextinction = 0; // probability of extinction begins at 0
		coalescent[NMT-1].Pmissing = (resolution<=0 ? (1.0-rarefaction) : 0.0);
		long resolution_jump = NMT-1; // will be updated (decreased) when we actually find the resolution jump
		double total_diversity_at_resolution_age, coalescent_diversity_at_resolution_age;
		for(long t=NMT-2; t>=0; --t){
			const double birth_rate_pc = get_speciation_rate_at_state(times[t+1],coalescent[t+1].diversity)/coalescent[t+1].diversity;
			const double death_rate_pc = get_extinction_rate_at_state(times[t+1],coalescent[t+1].diversity)/coalescent[t+1].diversity;
			double dt = times[t+1]-times[t];
			coalescent[t].Pextinction = coalescent[t+1].Pextinction + dt*(death_rate_pc - coalescent[t+1].Pextinction*(birth_rate_pc+death_rate_pc) + pow(coalescent[t+1].Pextinction,Nsplits)*birth_rate_pc);
			coalescent[t].Pmissing	  = coalescent[t+1].Pmissing + dt*(death_rate_pc - coalescent[t+1].Pmissing*(birth_rate_pc+death_rate_pc) + pow(coalescent[t+1].Pmissing,Nsplits)*birth_rate_pc);
			const double last_age = final_time-times[t+1];
			const double new_age  = final_time-times[t];
			if((new_age>=resolution) && (last_age<resolution)){
				// we just jumped over rarefaction_age, so apply rarefaction to Pmissing
				resolution_jump = t+1;
				total_diversity_at_resolution_age = trajectory[t+1].diversity + (trajectory[t].diversity-trajectory[t+1].diversity)*(resolution-last_age)/(new_age-last_age);
				double Pmissing_at_resolution_age = coalescent[t+1].Pmissing + (coalescent[t].Pmissing-coalescent[t+1].Pmissing)*(resolution-last_age)/(new_age-last_age); // linearly interpolate to get Pmissing at resolution age
				Pmissing_at_resolution_age = 1 - rarefaction*(1-Pmissing_at_resolution_age); // apply rarefaction at resolution age
				coalescent_diversity_at_resolution_age = total_diversity_at_resolution_age * (1-Pmissing_at_resolution_age);
				dt = new_age-resolution;
				coalescent[t].Pmissing = Pmissing_at_resolution_age + dt*(death_rate_pc - Pmissing_at_resolution_age*(birth_rate_pc+death_rate_pc) + pow(Pmissing_at_resolution_age,Nsplits)*birth_rate_pc);
			}
		}
		// flatten coalescent diversity for ages < resolution_age
		for(long t=NMT-2; t>=0; --t){
			if((final_time-times[t])<resolution){
				coalescent[t].coalescent_diversity = coalescent_diversity_at_resolution_age;
				coalescent[t].Pmissing = coalescent_diversity_at_resolution_age/coalescent[t].diversity;
			}else{
				coalescent[t].coalescent_diversity = coalescent[t].diversity * (1-coalescent[t].Pmissing);
			}
		}
	}
	
	
	// calculate coalescent diversities time series, based on the diversities & Pmissing
	void calculate_coalescent_diversities(){
		calculate_probabilities();  // make sure probabilities have been calculated
		for(long t=0; t<times.size(); ++t){
			trajectory[t].coalescent_diversity = trajectory[t].diversity * (1-trajectory[t].Pmissing);
		}
	}

	bool getInitialState(double time, TreeStateHistory &state) const{ 
		state = TreeStateHistory(initial_diversity); 
		if(reverse){
			state.Pextinction = 0;
			state.Pmissing = (1.0-rarefaction);
		}else{
			// not implemented for forward integration; Pextinction & Pmissing can be calculated afterwards (in reverse) using calculate_probabilities()
			state.Pextinction = 0;
			state.Pmissing = 0;
		}
		return true; 
	}

	// record time series point
	void registerState(double time, const TreeStateHistory &state){
		trajectory.push_back(state); 
		times.push_back(time); 
		if(reverse){
			trajectory.back().Pextinction = min(1.0,max(0.0,trajectory.back().Pextinction));
			trajectory.back().Pmissing = min(1.0,max(0.0,trajectory.back().Pmissing));
		}
	}

	double get_speciation_rate_at_state(double time, const double diversity) const{
		const double lambda = birth_rate_intercept + birth_rate_factor*pow(diversity,birth_rate_exponent) + diversity*added_birth_rate_pc(time);
		return lambda;
	}
	
	double get_extinction_rate_at_state(double time, const double diversity) const{
		return death_rate_intercept + death_rate_factor*pow(diversity,death_rate_exponent) + diversity*added_death_rate_pc(time);
	}

	RequestedDynamics getRateOfChangeAtState(double time, const TreeStateHistory &current_state, TreeStateHistory &rate_of_change, TreeStateHistory &jump_state){
		if(reverse) time = reflection_time-time;
		const double N 	= current_state.diversity;
		const double Pe = current_state.Pextinction;
		const double Pm = current_state.Pmissing;
		const double B 	= get_speciation_rate_at_state(time,current_state.diversity);
		const double D 	= get_extinction_rate_at_state(time,current_state.diversity);
		rate_of_change.Nbirths 		= B;
		rate_of_change.Ndeaths 		= D;
		rate_of_change.diversity 	= (Nsplits-1)*B - D;
		if(reverse){
			rate_of_change.Pextinction	= D/N - Pe*(B+D)/N + pow(Pe,Nsplits)*B/N;
			rate_of_change.Pmissing		= D/N - Pm*(B+D)/N + pow(Pm,Nsplits)*B/N;
		}else{
			rate_of_change.Pextinction = 0; // not implemented for forward integration
			rate_of_change.Pmissing = 0; // not implemented for forward integration			
		}
		if(reverse) rate_of_change.diversity = -rate_of_change.diversity;
		return RequestedDynamicsRateOfChange; 
	}

	// returns true if for some reason the time step should be refined, e.g. if the domain boundary is crossed
	bool checkShouldRefineTimeStep(double time, const TreeStateHistory &current_state, double dt, const TreeStateHistory &candidate_state) const{ 
		if(reverse){
			return (candidate_state.diversity<min_valid_diversity) || (candidate_state.Pextinction<0) || (candidate_state.Pextinction>1) || (candidate_state.Pmissing<0) || (candidate_state.Pmissing>1); 
		}else{
			return (candidate_state.diversity<min_valid_diversity);
		}
	}
	
	// check if candidate_state is outside of the domain boundaries.
	// In that case, tries to correct the candidate_state to be the "last" valid state on the linear trajectory
	// If this is not possible (e.g. because previous_state was already on the boundary), the problematic components are brute-force adjusted to be within the domain. In this case, the routine returns CrossedBoundaryYesButFixedBruteForce.
	CrossedBoundary checkCrossedDomainBoundaryAndFix(	double					previous_time,
														const TreeStateHistory	&previous_state,		// previous state (assumed to be valid!)
														double 					&dt,					// (INPUT/OUTPUT) will be adjusted (reduced) if candidate_state crossed the boundary. The modified value is guaranteed to be within (0,dt]
														TreeStateHistory		&candidate_state,		// (INPUT/OUTPUT) if candidate_state is outside of domain boundaries, then this may become the "last" state (on the linear trajectory from previous_state to candidate_state) within the domain (if possible).
														const bool				intermediate) const{	// (INPUT) is the candidate point an intermediate point (i.e. as used in Runge-Kutta schemes), or a final point (i.e., the final prediction for the candidate time)
		if(candidate_state.diversity>=min_valid_diversity){
			return CrossedBoundaryNo;
		}else if(previous_state.diversity>min_valid_diversity){
			double lambda = 1;
			if(candidate_state.diversity<min_valid_diversity) lambda = min(lambda,(previous_state.diversity-min_valid_diversity)/(previous_state.diversity-candidate_state.diversity));
			candidate_state = previous_state*(1-lambda) + candidate_state*lambda;
			dt *= lambda;
			return CrossedBoundaryYesButFixedByReducingTimeStep;
		}else{
			candidate_state.diversity 	= max(min_valid_diversity,candidate_state.diversity);
			candidate_state.Pextinction	= min(1.0,max(0.0,candidate_state.Pextinction));
			candidate_state.Pmissing 	= min(1.0,max(0.0,candidate_state.Pmissing));
			return CrossedBoundaryYesButFixedBruteForce;
		}
		/*
		if((candidate_state.diversity>=min_valid_diversity) && ((!reverse) || ((candidate_state.Pextinction>=0) && (candidate_state.Pextinction<=1)))){
			return CrossedBoundaryNo;
		}else if((previous_state.diversity>min_valid_diversity) && ((!reverse) || ((previous_state.Pextinction>=0) && (previous_state.Pextinction<=1)))){
			double lambda = 1;
			if(candidate_state.diversity<min_valid_diversity) lambda = min(lambda,(previous_state.diversity-min_valid_diversity)/(previous_state.diversity-candidate_state.diversity));
			if(reverse && (candidate_state.Pextinction<0)) lambda = min(lambda,(previous_state.Pextinction-0)/(previous_state.Pextinction-candidate_state.Pextinction));
			if(reverse && (candidate_state.Pextinction>1)) lambda = min(lambda,(1-previous_state.Pextinction)/(candidate_state.Pextinction-previous_state.Pextinction));
			candidate_state = previous_state*(1-lambda) + candidate_state*lambda;
			dt *= lambda;
			return CrossedBoundaryYesButFixedByReducingTimeStep;
		}else{
			candidate_state.diversity 	= max(min_valid_diversity,candidate_state.diversity);
			candidate_state.Pextinction 	= min(1.0,max(0.0,candidate_state.Pextinction));
			return CrossedBoundaryYesButFixedBruteForce;
		}
		*/
	}
	
	bool stateIsNaN(const TreeStateHistory &state) const{
		return state.isnan();
	}
	
	// estimate the maximum rate of change of diversity, for diversities within the interval [0, max_diversity]
	// may be useful for choosing the time step for simulations
	double estimate_max_rate_of_change(const double max_time, const double max_diversity, bool per_capita) const{
		double zenith_diversity;
		const double B3 = (per_capita ? birth_rate_exponent-1 : birth_rate_exponent); 	// adjust exponent for per-capita rates
		const double D3 = (per_capita ? death_rate_exponent-1 : death_rate_exponent);	// adjust exponent for per-capita rates
		if(death_rate_exponent==birth_rate_exponent) zenith_diversity = max_diversity;
		else if(((death_rate_factor==0) || (D3==0)) && (D3>B3)) zenith_diversity = max_diversity; // zenith is not defined
		else if(((birth_rate_factor==0) || (B3==0)) && (B3>D3)) zenith_diversity = max_diversity; // zenith is not defined
		else zenith_diversity = max(0.0,min(max_diversity, pow(((Nsplits-1)*birth_rate_factor*B3)/(death_rate_factor*D3), 1.0/(D3-B3))));
		const double R1 = abs((Nsplits-1)*get_speciation_rate_at_state(max_time,max_diversity) - get_extinction_rate_at_state(max_time,max_diversity))/(per_capita ? max_diversity : 1.0);
		const double R2 = abs((Nsplits-1)*get_speciation_rate_at_state(max_time,zenith_diversity) - get_extinction_rate_at_state(max_time,zenith_diversity))/(per_capita ? zenith_diversity : 1.0);
		const double R3 = abs((Nsplits-1)*get_speciation_rate_at_state(max_time,0) - get_extinction_rate_at_state(max_time,0));
		return max(R1, max(R2, R3));
	}
};



// simulate the deterministic trajectory of a speciation-extinction model for tree growth, calculating the diversities at specific times
// Optionally, the model can be integrated backwards in time, so that the "initial condition" (start_diversity) refers to the final true diversity of the tree.
// Typically, if 'coalescent' is true then 'reverse' should be true, unless the true diversity (i.e. of extant clades) is known for the past in the coalescent tree.
// [[Rcpp::export]]
Rcpp::List simulate_deterministic_diversity_growth_CPP(	const double 		birth_rate_intercept,	// (INPUT) intercept of Poissonian rate at which new tips are added to the tree
														const double 		birth_rate_factor,		// (INPUT) power-law factor of Poissonian rate at which new tips are added to the tree
														const double 		birth_rate_exponent,	// (INPUT) power-law exponent of Poissonian rate at which new tips are added to the tree
														const double 		death_rate_intercept,	// (INPUT) intercept of Poissonian rate at which extant tips are removed from the tree
														const double 		death_rate_factor,		// (INPUT) power-law factor of Poissonian rate at which extant tips are removed from the tree
														const double 		death_rate_exponent,	// (INPUT) power-law exponent of Poissonian rate at which extant tips are removed from the tree
														const double		resolution,				// (INPUT) optional resoluton at which to collapse final tree (prior to any rarefaction). Set this to 0 for no collapsing. Note that collapsing affects the last part of the coalescent diversity curve.
														const double		rarefaction,			// (INPUT) optional rarefaction fraction to apply, i.e. fraction of tips remaining after rarefaction. Rarefaction is assumed to occur after collapsing at the given resolution (i.e. the collapsed tree is rarefied). Set this to 1 for no rarefaction. Note that rarefaction affects the full coalescent diversity curve in a non-uniform manner.
														const long			Nsplits,				// (INPUT) number of children to create at each diversification event. Must be at least 2. For a bifurcating tree this should be set to 2. If >2, then the tree will be multifurcating.
														const std::vector<double>	&additional_rates_times,	// (INPUT) optional 1D array of size NAR, listing time points (in ascending order) for custom additional per-capita birth and/or death rates. Can be empty.
														const std::vector<double>	&additional_birth_rates_pc,	// (INPUT) optional 1D array of size NAR, listing custom per-capita birth rates (additive to the power law). Can be empty.
														const std::vector<double>	&additional_death_rates_pc,	// (INPUT) optional 1D array of size NAR, listing custom per-capita birth rates (additive to the power law). Can be empty.
														const bool					additional_periodic,		// (INPUT) if true, additional pc birth & death rates are extended periodically if needed. Otherwise they are extended with zeros.
														const std::vector<double>	&times,					// (INPUT) times at which to calculate deterministic diversities, in ascending order
														const double		start_time,				// (INPUT) Time at which to start simulation. This should be the beginning of the tree (<=times[0]).
														const double		final_time,				// (INPUT) Time at which to end simulation. This should be the ending time of the tree (>= times.back()).
														const double		start_diversity,		// (INPUT) diversity of extant clades at start_time. If reverse==true, this is the visible diversity after rarefaction. Only relevant if reverse=false.
														const double		final_diversity,		// (INPUT) total extant diversity at final_time, without any rarefaction or collapsing at some resolution. Only relevant if reverse==true.
														const bool			reverse,				// (OUTPUT) if true, then the tree model is integrated in backward time direction. In that case, start_diversity is interpreted as the true diversity at times.back()
														const bool			include_coalescent,			// (INPUT) if true, the coalescent diversities are also returned. These are the diversities that would remain in a coalescent tree (i.e. only including ancestors of extant tips)
														const bool			include_probabilities,		// (INPUT) if true, then Prepresentation (for each time point) will also be returned. This only makes sense for reverse integrations
														const bool			include_birth_rates,		// (INPUT) if true, the speciation rates corresponding to times[] will also be returned
														const bool			include_death_rates,		// (INPUT) if true, the extinction rates corresponding to times[] will also be returned
														const bool			include_Nevents,			// (INPUT) if true, then the total predicted birth events (starting from times[0] and up to each time point) will also be calculated and returned
														const double		runtime_out_seconds){		// (INPUT) max allowed runtime in seconds. If <=0, this option is ignored.
	const long NT = times.size();
	const bool has_added_birth_rates = ((additional_rates_times.size()>0) && (additional_birth_rates_pc.size()>0));
	const bool has_added_death_rates = ((additional_rates_times.size()>0) && (additional_death_rates_pc.size()>0));
	
	// prepare returned data structures, fill later
	std::vector<double> birth_rates, death_rates, Nbirths, Ndeaths;
	std::vector<double> coalescent_diversities, total_diversities, Psurvival, Prepresentation;
	
	if((birth_rate_intercept==0) && (birth_rate_exponent==1) && (death_rate_intercept==0) && (death_rate_exponent==1) && (Nsplits==2) && (!has_added_birth_rates) && (!has_added_death_rates)){
		// special case: constant per-capita speciation & extinction rates
		// use known analytical solutions
		total_diversities.resize(NT);
		if(include_coalescent) coalescent_diversities.resize(NT);
		if(include_probabilities){
			Psurvival.resize(NT);
			Prepresentation.resize(NT);
		}
		if(include_birth_rates) birth_rates.resize(NT);
		if(include_death_rates) death_rates.resize(NT);
		if(include_Nevents){ 
			Nbirths.resize(NT); 
			Ndeaths.resize(NT); 
		}
		const double diversification_rate = birth_rate_factor-death_rate_factor;
		double Pmissing;
		double Pmissing_at_resolution_age, total_diversity_at_resolution_age, coalescent_diversity_at_resolution_age;
		// calculate total & coalescent diversity at resolution_age, as well as Pmissing at resolution_age.
		if(resolution>0){
			if(birth_rate_factor==death_rate_factor){
				Pmissing_at_resolution_age = 1 - 1/(1+resolution*birth_rate_factor);
			}else{
				Pmissing_at_resolution_age = 1 - diversification_rate/(birth_rate_factor - death_rate_factor*exp(-resolution*diversification_rate));
			}
			if(reverse){
				total_diversity_at_resolution_age = final_diversity * exp(-resolution*diversification_rate);
			}else{
				total_diversity_at_resolution_age = start_diversity * exp((final_time-resolution-start_time)*diversification_rate);
			}
			coalescent_diversity_at_resolution_age = total_diversity_at_resolution_age * rarefaction * (1-Pmissing_at_resolution_age);
			
		}
		// calculate total & coalescent diversity and other variables for all time points
		for(long t=0; t<NT; ++t){
			const double age = final_time - times[t];
			if(reverse){
				total_diversities[t] = final_diversity * exp(-age*diversification_rate);
			}else{
				total_diversities[t] = start_diversity * exp((times[t]-start_time)*diversification_rate);
			}
			if(age>=resolution){
				const double effective_rarefaction = rarefaction * (1-Pmissing_at_resolution_age);
				const double effective_age = age-resolution;
				if(birth_rate_factor==death_rate_factor){
					Pmissing = 1 - effective_rarefaction/(1+effective_rarefaction*effective_age*birth_rate_factor);
				}else{
					Pmissing = 1 - effective_rarefaction*diversification_rate/(effective_rarefaction*birth_rate_factor + ((1-effective_rarefaction)*birth_rate_factor - death_rate_factor)*exp(-effective_age*diversification_rate));
				}
			}else{
				Pmissing = 1.0 - coalescent_diversity_at_resolution_age/total_diversities[t]; // effective Pmissing, based on coalescent and total diversity after collapsing and rarefaction
			}
			if(include_probabilities){
				if(birth_rate_factor==death_rate_factor){
					Psurvival[t] = 1/(1+age*birth_rate_factor);
				}else{
					Psurvival[t] = diversification_rate/(birth_rate_factor - death_rate_factor*exp(-age*diversification_rate));
				}
				Prepresentation[t] = 1-Pmissing;
			}
			if(include_birth_rates) birth_rates[t] = birth_rate_factor * total_diversities[t];
			if(include_death_rates) death_rates[t] = death_rate_factor * total_diversities[t];
			if(include_Nevents){
				Nbirths[t] = total_diversities[0]*(birth_rate_factor/diversification_rate) * (exp(diversification_rate*(times[t]-times[0])) - 1);
				Ndeaths[t] = total_diversities[0]*(death_rate_factor/diversification_rate) * (exp(diversification_rate*(times[t]-times[0])) - 1);
			}
			if(include_coalescent) coalescent_diversities[t] = total_diversities[t] * (1-Pmissing);
		}
		
	}else{
		// general case: integrate ODEs using Runge-Kutta
		
		// prepare data structures for storing raw simulation results (will be interpolated onto requested time points afterwards)
		std::vector<double> model_times;
		std::vector<TreeStateHistory> model_trajectory;
		long NMT;
		
		// initialize model for tree growth
		TreeSpeciationExtinctionModel model;
		if(has_added_birth_rates) model.added_birth_rate_pc = LinearInterpolationFunctor<double>(additional_rates_times, additional_birth_rates_pc, additional_periodic,0.0,0.0,true,0);
		if(has_added_death_rates) model.added_death_rate_pc = LinearInterpolationFunctor<double>(additional_rates_times, additional_death_rates_pc, additional_periodic,0.0,0.0,true,0);
		model.birth_rate_intercept 	= birth_rate_intercept;
		model.birth_rate_factor 	= birth_rate_factor;
		model.birth_rate_exponent 	= birth_rate_exponent;
		model.death_rate_intercept 	= death_rate_intercept;
		model.death_rate_factor 	= death_rate_factor;
		model.death_rate_exponent 	= death_rate_exponent;
		model.Nsplits				= Nsplits;
		model.initial_diversity		= (reverse ? final_diversity : start_diversity);
		if(reverse) model.set_reverse(final_time); // time-reverse the dynamics (vector field) of the model, in order to then simulate backwards in time using the conventional Runge-Kitta integrator
		string warningMessage;
		const double default_dt = min((final_time-times[0])/NT, (reverse ? 1e-1*final_diversity/model.estimate_max_rate_of_change(final_time, final_diversity,false) : (final_time-times[0])/NT)); // guess appropriate simulation time step
		
		if((resolution>0) && reverse){
			// run simulation in 2 chunks, one for ages<resolution and then again for ages>resolution
			model.rarefaction = 1; // rarefaction occurs after collapsing tree at resolution, so simulate first part as if there was no rarefaction
			bool success = RungeKutta2<TreeStateHistory,TreeSpeciationExtinctionModel,ProgressReporter>
										(0,
										resolution,
										min(resolution/100, 1e-1*final_diversity/model.estimate_max_rate_of_change(final_time, final_diversity,false)),
										model,
										3, // number of points to record
										2,
										2, // refinement_factor
										ProgressReporter(true),
										runtime_out_seconds,
										warningMessage);
			if((!success) || (model.times.back()<resolution)) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Could not integrate backwards until resolution age: "+warningMessage ); // simulation failed
			// extract some information about the resolution_age (last time point of simulation)
			const long NMTr = model.times.size();
			const double total_diversity_at_resolution_age 		= model.trajectory[NMTr-1].diversity;
			const double Pmissing_at_resolution_age 			= model.trajectory[NMTr-1].Pmissing;
			const double coalescent_diversity_at_resolution_age = total_diversity_at_resolution_age * rarefaction * (1-Pmissing_at_resolution_age);
			
			// integrate further back in time, starting at resolution_age
			TreeSpeciationExtinctionModel model2;
			model2.adopt_parameters(model);
			model2.initial_diversity = total_diversity_at_resolution_age;
			model2.rarefaction = rarefaction*(1-Pmissing_at_resolution_age); // effective rarefaction at resolution age (i.e. after collapsing and rarefaction)
			success = RungeKutta2<TreeStateHistory,TreeSpeciationExtinctionModel,ProgressReporter>
										(resolution,
										default_dt + (final_time-times[0]),
										default_dt,
										model2,
										10*times.size(),
										2,
										2, // refinement_factor
										ProgressReporter(true),
										runtime_out_seconds,
										warningMessage);
			if(!success) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Could not integrate backwards past resolution age: "+warningMessage ); // simulation failed

			// reverse time course of simulated trajectories
			model.reverse_trajectory(final_time); 
			model2.reverse_trajectory(final_time);
			
			// until resolution_age, coalescent diversities and Pmissing are special, due to collapsing of the tree at the given resolution
			for(long t=0; t<NMTr; ++t){
				model.trajectory[t].coalescent_diversity = coalescent_diversity_at_resolution_age;
				model.trajectory[t].Pmissing = coalescent_diversity_at_resolution_age/model.trajectory[t].diversity;
			}

			if(include_probabilities) model2.calculate_probabilities(); // make sure probabilities have been calculated (may not be the case if simulation was not in reverse). This must come after reversing the trajectory.
			if(include_coalescent) model2.calculate_coalescent_diversities(); // make sure coalescent diversities have been calculated for the trajectory (may not be the case if simulation was not in reverse). This must come after reversing the trajectory.
			
			// adjust Nbirths & Ndeaths
			if(include_Nevents){
				for(long t=0; t<NMTr; ++t){
					model.trajectory[t].Nbirths += model2.trajectory.back().Nbirths;
					model.trajectory[t].Ndeaths += model2.trajectory.back().Ndeaths;
				}
			}

			// extract and concatenate trajectories from the two simulations
			NMT = NMTr+model2.times.size();
			model_times.reserve(NMT);
			model_times.insert(model_times.end(), model2.times.begin(), model2.times.end());
			model_times.insert(model_times.end(), model.times.begin(), model.times.end());
			model_trajectory.reserve(NMT);
			model_trajectory.insert(model_trajectory.end(), model2.trajectory.begin(), model2.trajectory.end());
			model_trajectory.insert(model_trajectory.end(), model.trajectory.begin(), model.trajectory.end());


		}else{
			// run simulation for all ages
			// note that simulation a priori calculates total extant diversities over time, we make the diversities coalescent afterwards
			model.rarefaction = rarefaction;
			const bool success = RungeKutta2<TreeStateHistory,TreeSpeciationExtinctionModel,ProgressReporter>
										((reverse ? 0 : start_time),
										default_dt+(reverse ? final_time-times[0] : final_time),
										default_dt,
										model,
										10*times.size(),
										2,
										2, // refinement_factor
										ProgressReporter(true),
										runtime_out_seconds,
										warningMessage);
			if(!success) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = warningMessage ); // simulation failed

			if(reverse) model.reverse_trajectory(final_time); // reverse time course of simulated trajectory
			
			if((!reverse) && (resolution>0)){
				model.get_coalescent_trajectory(resolution, rarefaction, model_trajectory);
			}else{
				if(include_probabilities) model.calculate_probabilities(); // make sure probabilities have been calculated (may not be the case if simulation was not in reverse). This must come after reversing the trajectory.
				if(include_coalescent) model.calculate_coalescent_diversities(); // make sure coalescent diversities have been calculated for the trajectory (may not be the case if simulation was not in reverse). This must come after reversing the trajectory.
				model_trajectory = model.trajectory;
			}
			model_times = model.times;
			NMT = model_times.size();
		}
		
		// extract total diversities (extinct+extant) from simulated trajectory
		long includedNewTimesStart, includedNewTimesEnd;
		std::vector<double> model_total_diversities(NMT);
		for(long t=0; t<NMT; ++t) model_total_diversities[t] = model_trajectory[t].diversity;
		interpolateTimeSeriesAtTimes(model_times, model_total_diversities, 0, NMT-1, times, 0, NT-1, includedNewTimesStart, includedNewTimesEnd, total_diversities);

		if(include_coalescent){
			// extract coalescent diversities from simulated trajectory
			std::vector<double> model_coalescent_diversities(NMT);
			for(long t=0; t<NMT; ++t) model_coalescent_diversities[t] = model_trajectory[t].coalescent_diversity;
			interpolateTimeSeriesAtTimes(model_times, model_coalescent_diversities, 0, NMT-1, times, 0, NT-1, includedNewTimesStart, includedNewTimesEnd, coalescent_diversities);
		}
	
		if(include_probabilities){
			// extract probabilities of lineage survival & representation from simulated trajectory
			std::vector<double> model_Prepresentation(NMT);
			std::vector<double> model_Psurvival(NMT);
			for(long t=0; t<NMT; ++t){
				model_Psurvival[t] = 1-model_trajectory[t].Pextinction;
				model_Prepresentation[t] = 1-model_trajectory[t].Pmissing;
			}
			interpolateTimeSeriesAtTimes(model_times, model_Prepresentation, 0, NMT-1, times, 0, NT-1, includedNewTimesStart, includedNewTimesEnd, Prepresentation);
			interpolateTimeSeriesAtTimes(model_times, model_Psurvival, 0, NMT-1, times, 0, NT-1, includedNewTimesStart, includedNewTimesEnd, Psurvival);
		}

		// extract cumulative speciation & extinction events, if needed
		if(include_Nevents){
			// extract from model trajectory
			std::vector<double> model_Nbirths(NMT), model_Ndeaths(NMT);
			for(long t=0; t<NMT; ++t){
				model_Nbirths[t] = model_trajectory[t].Nbirths;
				model_Ndeaths[t] = model_trajectory[t].Ndeaths;
			}
			interpolateTimeSeriesAtTimes(model_times, model_Nbirths, 0, NMT-1, times, 0, NT-1, includedNewTimesStart, includedNewTimesEnd, Nbirths);
			interpolateTimeSeriesAtTimes(model_times, model_Ndeaths, 0, NMT-1, times, 0, NT-1, includedNewTimesStart, includedNewTimesEnd, Ndeaths);
		}

		// calculate speciation rates for requested times, if needed
		if(include_birth_rates){
			birth_rates.resize(NT);
			for(long t=0; t<NT; ++t) birth_rates[t] = model.get_speciation_rate_at_state(times[t],total_diversities[t]);
		}
		// calculate extinction rates for requested times, if needed
		if(include_death_rates){
			death_rates.resize(NT);
			for(long t=0; t<NT; ++t) death_rates[t] = model.get_extinction_rate_at_state(times[t],total_diversities[t]);
		}
	}
	

	 		
	return Rcpp::List::create(	Rcpp::Named("success")					= true,
								Rcpp::Named("coalescent_diversities") 	= Rcpp::wrap(coalescent_diversities),	// coalescent diversity at each time point (if requested)
								Rcpp::Named("total_diversities")		= Rcpp::wrap(total_diversities),	// total (extant+extinct) diversity at each time point
								Rcpp::Named("Psurvival")				= Rcpp::wrap(Psurvival),
								Rcpp::Named("Prepresentation")			= Rcpp::wrap(Prepresentation),
								Rcpp::Named("birth_rates")				= Rcpp::wrap(birth_rates),
								Rcpp::Named("death_rates")				= Rcpp::wrap(death_rates),
								Rcpp::Named("Nbirths")					= Nbirths,
								Rcpp::Named("Ndeaths")					= Ndeaths);
}





// Estimate past diversity, birth & death rates, based on a time series of coalescent diversities
// This reconstruction is non-parametric, i.e. no particular model is assumed (except for knowledge or constancy of the per-capita birth rate)
// Note: This function is currently only implemented for bifurcating trees. 
// It's possible to extend to multifurcating trees (Nsplits>2), but it requires solving (e.g. numerically) a higher-order polynomial at each age.
// Input:
//	A time series of coalescent diversities, i.e. as would be visible in a coalescent phylogenetic tree.
// 	Corresponding assumed per-capita birth rates. Alternatively, these can be assumed to be constant and be estimated directly from the coalescent_diversity time series.
// Output:
//  The estimated true past diversities (N(t))
//  The estimated corresponding death (extinction) rates (delta(t))
//  The probability of a size-1 clade surviving from each time point to the present (P(t))
//  The estimated total number of speciation & extinction events, over the considered age interval
// [[Rcpp::export]]
Rcpp::List reconstruct_past_diversity_from_coalescent_CPP(	const std::vector<double>	&times,						// (INPUT) 1D numeric array of size NT, listing times in ascending order. The last time point corresponds to the "present" (age=0)
															const std::vector<double>	&raw_coalescent_diversities,// (INPUT) 1D numeric array of size NT, listing coalescent diversities as visible after rarefaction. Should be unsmoothened.
															const std::vector<double> 	&birth_rates_pc,			// (INPUT) 1D numeric array of size NT, listing known or assumed per-capita birth rates. Can also be of size 1, in which case the same per-capita birth rate is assumed throughout. Can also be empty, in which case a constant per-capita birth rate is assumed and estimated from the last slope of the coalescent_diversity curve.
															const double				rarefaction,				// (INPUT) optional rarefaction fraction assumed to have occurred at the very end (fraction of kept tips). Set to 1.0 for no rarefaction.
															const double 				max_age,					// (INPUT) max age (distance from last time point) to consider for integrating total births & deaths. If <=0, all times are considered.
															const long					smoothing_span,				// (INPUT) Integer. Optional number of time points for smoothening the diversities time series via Savitzky-Golay-filter. If <=2, no smoothing is done. Smoothening the coalescent diversity can reduce the noise in the non-parametric reconstruction. 
															const long					smoothing_order){			// (INPUT) Integer. Optional polynomial order of the smoothing model.
	const long NT 			= times.size();
	const double max_time 	= times[NT-1];
	const double Nsplits	= 2; // currently not implemented for other Nsplits
	
	// smoothen time series if needed
	// using a Savitzky-Golay-filter of 2nd-order (1st order if smoothen==3)
	const bool smoothen = (smoothing_span>2);
	std::vector<double> smooth_coalescent_diversities(NT);
	if(smoothen){
		if(!smoothenTimeSeriesSavitzkyGolay(times, raw_coalescent_diversities, 0.0, smoothing_span, min(smoothing_span-2,smoothing_order), true, smooth_coalescent_diversities)){
			return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Smoothing of coalescent diversity time series failed");
		}
		smooth_coalescent_diversities[NT-1] = raw_coalescent_diversities[NT-1];
	}
	const std::vector<double> &coalescent_diversities = (smoothen ? smooth_coalescent_diversities : raw_coalescent_diversities);
	
	// determine latest (most recent) per-capita birth rate if needed
	double last_birth_rate_pc;
	const bool constant_birth_rate_pc = (birth_rates_pc.size()<=1);
	if(birth_rates_pc.size()==0){
		// estimate from slope of raw_coalescent_diversities
		// the raw (unsmoothened) data is preferred for estimation of birth_rate_pc, so as to preserve the underlying (by assumption) exponential structure
		last_birth_rate_pc = log(raw_coalescent_diversities[NT-1]/raw_coalescent_diversities[NT-2])/(times[NT-1] - times[NT-2]);
		// correct for the effects of rarefaction
		const double zeta = 1-rarefaction;
		if(rarefaction<1) last_birth_rate_pc *= rarefaction/(Nsplits-1-Nsplits*zeta+pow(zeta,Nsplits));
	}else{
		// either birth_rates_oc are provided for all time points, or as a single constant number
		last_birth_rate_pc = birth_rates_pc[birth_rates_pc.size()-1];
	}
	if(last_birth_rate_pc<0) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Latest per-capita birth rate is negative or zero");
	
	// determine first considered time point
	long first_time_point;
	if(max_age>0){
		first_time_point = NT-1;
		for(long t=0; t<NT; ++t){
			if((max_time-times[t])<=max_age){
				first_time_point = t;
				break;
			}
		}
	}else{
		first_time_point = 0;
	}
			
	// calculate survival chances & true diversities over time
	std::vector<double> Prepresentation(NT), total_diversities(NT), nu(NT), pulled_total_diversities(NT);
	for(long t=NT-1, tl, tr; t>=0; --t){
		tl = (t==0 ? 0 : t-1);
		tr = (t==0 ? 1 : t); // use left-sided derivative, to avoid the discontinuity at age=0
		const double birth_rate_pc = (constant_birth_rate_pc ? last_birth_rate_pc : birth_rates_pc[t]);
		nu[t] = -log(coalescent_diversities[tr]/coalescent_diversities[tl])/(times[tr]-times[tl]); // (1/Nc)*dNc/dtau = dlog(Nc)/dtau, where Nc: coalescent_diversity
		Prepresentation[t] = -nu[t]/birth_rate_pc;
		if((Prepresentation[t]<=1e-8) && (coalescent_diversities[t]>0)){
			// diversity not defined (Prepresentation seems spuriously low)
			total_diversities[t] = NAN_D;
			Prepresentation[t] 	= max(0.0, Prepresentation[t]);
		}else if(Prepresentation[t]>1){
			// diversity not defined (Prepresentation > 1)
			Prepresentation[t]  = 1;
			total_diversities[t] = NAN_D;			
		}else{
			total_diversities[t] = coalescent_diversities[t]/Prepresentation[t];
		}
		pulled_total_diversities[t] = - last_birth_rate_pc*coalescent_diversities[t]/nu[t];
	}
	total_diversities[NT-1] = coalescent_diversities[NT-1]/rarefaction; // coalescent diversity last time point is assumed to be equal to true diversity multiplied by rarefaction fraction
	
	// calculate birth & death rates & pulled diversification rate
	std::vector<double> death_rates(NT), death_rates_pc(NT), birth_rates(NT), pulled_diversification_rates(NT), pulled_extinction_rates(NT);
	for(long t=NT-1, tl, tr; t>=0; --t){
		tl = (t==0 ? 0 : t-1);
		tr = (t==0 ? 1 : t); // use left-sided derivative, to avoid the discontinuity at age=0
		const double birth_rate_pc = (constant_birth_rate_pc ? last_birth_rate_pc : birth_rates_pc[t]);
		birth_rates[t] 		= birth_rate_pc*total_diversities[t];
		death_rates[t] 		= -(total_diversities[tr] - total_diversities[tl])/(times[tr] - times[tl]) + birth_rates[t];
		death_rates_pc[t] 	= death_rates[t]/(0.5*(total_diversities[tr] + total_diversities[tl]));
		pulled_diversification_rates[t] = (log(-coalescent_diversities[tr]/nu[tr]) - log(-coalescent_diversities[tl]/nu[tl]))/(times[tr] - times[tl]);
		pulled_extinction_rates[t]		= last_birth_rate_pc - pulled_diversification_rates[t];
	}
	
	// calculate Psurvival (based on estimate per-capita birth & death rates)
	std::vector<double> Psurvival(NT);
	Psurvival[NT-1] = 1;
	for(long t=NT-2; t>=0; --t){
		const double dt = times[t+1]-times[t];
		const double birth_rate_pc = (constant_birth_rate_pc ? last_birth_rate_pc : birth_rates_pc[t]);
		// 2-step Runge-Kutta
		double rate1 = - (death_rates_pc[t] - (1-Psurvival[t+1])*(birth_rate_pc+death_rates_pc[t]) + pow((1-Psurvival[t+1]),Nsplits)*birth_rate_pc);
		double temp_Psurvival = Psurvival[t+1] + dt*rate1;
		double rate2 = - (death_rates_pc[t] - (1-temp_Psurvival)*(birth_rate_pc+death_rates_pc[t]) + pow((1-temp_Psurvival),Nsplits)*birth_rate_pc);
		Psurvival[t] = max(0.0, min(1.0, Psurvival[t+1] + dt*0.5*(rate1+rate2)));
	}
	
	
	// calculate total number of births & deaths
	const double total_births = integrate1D(times,birth_rates,first_time_point,NT-1,true);
	const double total_deaths = integrate1D(times,death_rates,first_time_point,NT-1,true);
	
	return Rcpp::List::create(	Rcpp::Named("success")						= true,
								Rcpp::Named("total_diversities")			= Rcpp::wrap(total_diversities),
								Rcpp::Named("birth_rates") 					= Rcpp::wrap(birth_rates),
								Rcpp::Named("death_rates") 					= Rcpp::wrap(death_rates),
								Rcpp::Named("Prepresentation")				= Rcpp::wrap(Prepresentation),
								Rcpp::Named("Psurvival")					= Rcpp::wrap(Psurvival),
								Rcpp::Named("total_births")					= total_births,
								Rcpp::Named("total_deaths")					= total_deaths,
								Rcpp::Named("last_birth_rate_pc")			= last_birth_rate_pc,
								Rcpp::Named("last_death_rate_pc")			= death_rates[NT-1]/total_diversities[NT-1],
								Rcpp::Named("pulled_diversification_rates")	= Rcpp::wrap(pulled_diversification_rates),
								Rcpp::Named("pulled_extinction_rates")		= Rcpp::wrap(pulled_extinction_rates),
								Rcpp::Named("pulled_total_diversities")		= Rcpp::wrap(pulled_total_diversities));
}






// Estimate past diversity, birth & death rates, based on a time series of coalescent diversities
// This reconstruction is non-parametric, i.e. no particular model is assumed (except for knowledge or constancy of the per-capita birth rate)
// Similar to reconstruct_past_diversity_from_biased_coalescent_CPP, but this one allows specification of an arbitrary discovery_fraction for each time, as opposed to assuming random unbiased sampling at the end
// Note: This function is currently only implemented for bifurcating trees. 
// Input:
//	A time series of coalescent diversities, i.e. as would be visible in a coalescent phylogenetic tree.
// 	Corresponding assumed per-capita birth rates. Alternatively, these can be assumed to be constant and be estimated directly from the coalescent_diversity time series.
//  A time series of discovery_fractions, at each age specifying the fraction of lineages at that age, with extant discovered descendants
// Output:
//  The estimated true past diversities (N(t))
//  The estimated corresponding death (extinction) rates (delta(t))
//  The probability of a size-1 clade surviving from each time point to the present (P(t))
//  The estimated total number of speciation & extinction events, over the considered age interval
// [[Rcpp::export]]
Rcpp::List reconstruct_past_diversity_from_biased_coalescent_CPP(	const std::vector<double>	&times,						// (INPUT) 1D numeric array of size NT, listing times in ascending order. The last time point corresponds to the "present" (age=0)
																	const std::vector<double>	&raw_coalescent_diversities,// (INPUT) 1D numeric array of size NT, listing coalescent diversities as visible after rarefaction. Should be unsmoothened.
																	const std::vector<double> 	&birth_rates_pc,			// (INPUT) 1D numeric array of size NT, listing known or assumed per-capita birth rates. Can also be of size 1, in which case the same per-capita birth rate is assumed throughout. Can also be empty, in which case a constant per-capita birth rate is assumed and estimated from the last slope of the coalescent_diversity curve.
																	const std::vector<double> 	&discovery_fractions,		// (INPUT) 1D numeric array of size NT, listing discovery fractions for each age and synchronized with times[]. For example, discovery_fractions.back() corresponds to the fraction of discovered extant species.
																	const std::vector<double> 	&discovery_fraction_slopes,	// (INPUT) 1D numeric array of size NT, listing the 1st derivative of the discovery_fractions (w.r.t. time) at times[]
																	const double 				max_age,					// (INPUT) max age (distance from last time point) to consider for integrating total births & deaths. If <=0, all times are considered.
																	const long					smoothing_span,				// (INPUT) Integer. Optional number of time points for smoothening the diversities time series via Savitzky-Golay-filter. If <=2, no smoothing is done. Smoothening the coalescent diversity can reduce the noise in the non-parametric reconstruction. 
																	const long					smoothing_order){			// (INPUT) Integer. Optional polynomial order of the smoothing model.
	const long NT 			= times.size();
	const double max_time 	= times[NT-1];
	
	// smoothen time series if needed
	// using a Savitzky-Golay-filter of 2nd-order (1st order if smoothen==3)
	const bool smoothen = (smoothing_span>2);
	std::vector<double> smooth_coalescent_diversities(NT);
	if(smoothen){
		if(!smoothenTimeSeriesSavitzkyGolay(times, raw_coalescent_diversities, 0.0, smoothing_span, min(smoothing_span-2,smoothing_order), true, smooth_coalescent_diversities)){
			return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Smoothing of coalescent diversity time series failed");
		}
		smooth_coalescent_diversities[NT-1] = raw_coalescent_diversities[NT-1];
	}
	const std::vector<double> &coalescent_diversities = (smoothen ? smooth_coalescent_diversities : raw_coalescent_diversities);
	
	// determine latest (most recent) per-capita birth rate if needed
	double last_birth_rate_pc;
	const bool constant_birth_rate_pc = (birth_rates_pc.size()<=1);
	if(birth_rates_pc.size()==0){
		// estimate from slope of raw_coalescent_diversities
		// the raw (unsmoothened) data is preferred for estimation of birth_rate_pc, so as to preserve the underlying (by assumption) exponential structure
		last_birth_rate_pc = log(raw_coalescent_diversities[NT-1]/raw_coalescent_diversities[NT-2])/(times[NT-1] - times[NT-2]);
		// correct for the effects of incomplete discovery
		last_birth_rate_pc = max(0.0, last_birth_rate_pc*discovery_fractions[NT-1] - discovery_fraction_slopes[NT-1]);
	}else{
		// either birth_rates_oc are provided for all time points, or as a single constant number
		last_birth_rate_pc = birth_rates_pc[birth_rates_pc.size()-1];
	}
	if(last_birth_rate_pc<0) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Latest per-capita birth rate is negative or zero");
	
	// determine first considered time point
	long first_time_point;
	if(max_age>0){
		first_time_point = NT-1;
		for(long t=0; t<NT; ++t){
			if((max_time-times[t])<=max_age){
				first_time_point = t;
				break;
			}
		}
	}else{
		first_time_point = 0;
	}
			
	// calculate survival chances & true diversities over time
	std::vector<double> survival_chances(NT), true_diversities(NT), nu(NT);
	for(long t=NT-1, tl, tr; t>=0; --t){
		tl = (t==0 ? 0 : t-1);
		tr = (t==0 ? 1 : t); // use left-sided derivative, to avoid the discontinuity at age=0
		const double birth_rate_pc = (constant_birth_rate_pc ? last_birth_rate_pc : birth_rates_pc[t]);
		nu[t] = -log(coalescent_diversities[tr]/coalescent_diversities[tl])/(times[tr]-times[tl]); // (1/Nc)*dNc/dtau = dlog(Nc)/dtau, where Nc: coalescent_diversity
		survival_chances[t] = -(discovery_fractions[t]*nu[t] + discovery_fraction_slopes[t])/birth_rate_pc;
		if((survival_chances[t]<=1e-2/coalescent_diversities[NT-1]) && (coalescent_diversities[t]>0)){
			// diversity not defined (survival seems spuriously low)
			true_diversities[t] = NAN_D;
		}else if(survival_chances[t]>1){
			// diversity not defined (survival chance > 1)
			true_diversities[t] = NAN_D;			
		}else{
			true_diversities[t] = coalescent_diversities[t]/(survival_chances[t] * discovery_fractions[t]);
		}
	}
	true_diversities[NT-1] = coalescent_diversities[NT-1]/discovery_fractions[NT-1]; // coalescent diversity last time point is assumed to be equal to true diversity multiplied by discovery fraction (since survival_chance = 1)
	
	// calculate birth & death rates & per-capita (exponential) pulled growth rate
	std::vector<double> death_rates(NT), birth_rates(NT), pulled_diversification_rates(NT);
	for(long t=NT-1, tl, tr; t>=0; --t){
		tl = (t==0 ? 0 : t-1);
		tr = (t==0 ? 1 : t); // use left-sided derivative, to avoid the discontinuity at age=0
		const double birth_rate_pc = (constant_birth_rate_pc ? last_birth_rate_pc : birth_rates_pc[t]);
		birth_rates[t] = birth_rate_pc*true_diversities[t];
		death_rates[t] = max(0.0, -(true_diversities[tr] - true_diversities[tl])/(times[tr] - times[tl]) + birth_rates[t]); // death rate must be non-negative
		pulled_diversification_rates[t] = (log(-coalescent_diversities[tr]/(nu[tr]*SQ(discovery_fractions[tr]) - discovery_fractions[tr]*discovery_fraction_slopes[tr])) - log(-coalescent_diversities[tl]/(nu[tl]*SQ(discovery_fractions[tl]) - discovery_fractions[tl]*discovery_fraction_slopes[tl])))/(times[tr] - times[tl]);
	}
	
	// calculate total number of births & deaths
	const double total_births = integrate1D(times,birth_rates,first_time_point,NT-1,true);
	const double total_deaths = integrate1D(times,death_rates,first_time_point,NT-1,true);
	
	return Rcpp::List::create(	Rcpp::Named("success")						= true,
								Rcpp::Named("true_diversities")				= Rcpp::wrap(true_diversities),
								Rcpp::Named("birth_rates") 					= Rcpp::wrap(birth_rates),
								Rcpp::Named("death_rates") 					= Rcpp::wrap(death_rates),
								Rcpp::Named("Psurvival")					= Rcpp::wrap(survival_chances),
								Rcpp::Named("total_births")					= total_births,
								Rcpp::Named("total_deaths")					= total_deaths,
								Rcpp::Named("last_birth_rate_pc")			= last_birth_rate_pc,
								Rcpp::Named("last_death_rate_pc")			= death_rates[NT-1]/true_diversities[NT-1],
								Rcpp::Named("pulled_diversification_rates")	= Rcpp::wrap(pulled_diversification_rates));
}



// Estimate past diversity, birth & death rates, based on a time series of true diversities
// Input:
//	A time series of diversities, i.e. as would be visible in a non-coalescent phylogenetic tree (including extinct species).
// 	Corresponding assumed per-capita birth rates.
// Output:
//  The estimated corresponding death (extinction) rates (delta(t))
//  The probability of a size-1 clade surviving from each time point to the present (P(t))
//  The estimated total number of speciation & extinction events, over the considered age interval
// [[Rcpp::export]]
Rcpp::List reconstruct_past_diversifications_CPP(	const std::vector<double>	&times,				// (INPUT) 1D numeric array of size NT, listing times in ascending order. The last time point corresponds to the "present" (age=0)
													const std::vector<double>	&raw_diversities,	// (INPUT) 1D numeric array of size NT, listing true diversities. Should be unsmoothened.
													const std::vector<double> 	&birth_rates_pc,	// (INPUT) 1D numeric array of size NT, listing known or assumed per-capita birth rates. Can also be of size 1, in which case the same per-capita birth rate is assumed throughout.
													const double				rarefaction,		// (INPUT) optional rarefaction fraction, to apply when calculating survival chances
													const long					Nsplits,			// (INPUT)
													const double 				max_age,			// (INPUT) max age (distance from last time point) to consider for integrating total births & deaths. If <=0, all times are considered.
													const long					smoothing_span,		// (INPUT) Integer. Optional number of time points for smoothening the diversities time series via Savitzky-Golay-filter. If <=2, no smoothing is done. Smoothening the coalescent diversity can reduce the noise in the non-parametric reconstruction. 
													const long					smoothing_order){	// (INPUT) Integer. Optional polynomial order of the smoothing model.
	const long NT 			= times.size();
	const double max_time 	= times[NT-1];
	const bool const_birth_rate_pc = (birth_rates_pc.size()==1);
	
	// smoothen time series if needed
	// using a Savitzky-Golay-filter of 2nd-order (1st order if smoothen==3)
	const bool smoothen = (smoothing_span>2);
	std::vector<double> smooth_diversities(NT);
	if(smoothen){
		if(!smoothenTimeSeriesSavitzkyGolay(times, raw_diversities, 0.0, smoothing_span, min(smoothing_span-2,smoothing_order), true, smooth_diversities)){
			return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Smoothing of coalescent diversity time series failed");
		}
		smooth_diversities[NT-1] = raw_diversities[NT-1];
	}
	const std::vector<double> &true_diversities = (smoothen ? smooth_diversities : raw_diversities);
			
	// determine first considered time point
	long first_time_point;
	if(max_age>0){
		first_time_point = NT-1;
		for(long t=0; t<NT; ++t){
			if((max_time-times[t])<=max_age){
				first_time_point = t;
				break;
			}
		}
	}else{
		first_time_point = 0;
	}
			
	// calculate birth & death rates & per-capita (exponential) growth rates
	std::vector<double> birth_rates(NT), death_rates(NT), diversification_rates(NT);
	for(long t=NT-1, tl, tr; t>=0; --t){
		tl = (t==0 ? 0 : t-1);
		tr = (t==NT-1 ? t : t+1);
		const double birth_rate_pc = (const_birth_rate_pc ? birth_rates_pc[0] : birth_rates_pc[t]);
		birth_rates[t] = birth_rate_pc*true_diversities[t];
		death_rates[t] = (true_diversities[tr] - true_diversities[tl])/(times[tr] - times[tl]) + birth_rates[t];
		diversification_rates[t] = (log(true_diversities[tr])-log(true_diversities[tl]))/(times[tr] - times[tl]);
	}
	
	// integrate backwards to calculate probability of survival & representation
	// probability of representation at age=0 begins at (1-rarefaction), probability of survival at age=0 begins at 1
	std::vector<double> Psurvival(NT), Prepresentation(NT);
	Prepresentation[NT-1] = rarefaction;
	Psurvival[NT-1] = 1;
	for(long t=NT-2; t>=0; --t){
		const double dt = times[t+1]-times[t];
		const double birth_rate_pc = (const_birth_rate_pc ? birth_rates_pc[0] : birth_rates_pc[t]);
		const double death_rate_pc = 0.5*(death_rates[t+1]/true_diversities[t+1] + death_rates[t]/true_diversities[t]);
		Psurvival[t] = Psurvival[t+1] - dt*(death_rate_pc - (1-Psurvival[t+1])*(birth_rate_pc+death_rate_pc) + pow((1-Psurvival[t+1]),Nsplits)*birth_rate_pc);
		Prepresentation[t] = Prepresentation[t+1] - dt*(death_rate_pc - (1-Prepresentation[t+1])*(birth_rate_pc+death_rate_pc) + pow((1-Prepresentation[t+1]),Nsplits)*birth_rate_pc);
	}
	
	// use Prepresentation & Psurvival to calculate coalescent diversity and Pdiscovery
	std::vector<double> coalescent_diversities(NT);
	std::vector<double> Pdiscovery(NT);
	for(long t=0; t<NT; ++t){
		coalescent_diversities[t] = true_diversities[t]*Prepresentation[t];	
		Pdiscovery[t] = Prepresentation[t]/Psurvival[t];
	}

	
	// calculate total number of births & deaths
	const double total_births = integrate1D(times,birth_rates,first_time_point,NT-1,true);
	const double total_deaths = integrate1D(times,death_rates,first_time_point,NT-1,true);
	
	return Rcpp::List::create(	Rcpp::Named("success")					= true,
								Rcpp::Named("birth_rates") 				= Rcpp::wrap(birth_rates),
								Rcpp::Named("death_rates") 				= Rcpp::wrap(death_rates),
								Rcpp::Named("Psurvival")				= Rcpp::wrap(Psurvival),		// probability of a lineage surviving until today
								Rcpp::Named("Pdiscovery")				= Rcpp::wrap(Pdiscovery),		// probability of an extant lineage being discovered
								Rcpp::Named("Prepresentation")			= Rcpp::wrap(Prepresentation),	// probability of a lineage surviving and being discovered (=Psurvival*Pdiscovery)
								Rcpp::Named("coalescent_diversities")	= Rcpp::wrap(coalescent_diversities),
								Rcpp::Named("total_births")				= total_births,
								Rcpp::Named("total_deaths")				= total_deaths,
								Rcpp::Named("diversification_rates")	= Rcpp::wrap(diversification_rates));
}



// Based on the timings of birth (speciation) & death (extinction) events, calculate the diversity-over-time curve
// diversity at time t = start_diversity + (Nsplits-1)*total_birth_counts_until[t] - total_death_counts_until[t]
// [[Rcpp::export]]
Rcpp::List get_diversities_from_birth_and_death_events_CPP(	const std::vector<double> &times,
															const std::vector<double> &birth_times,		// 1D array of size NT, listing birth times in ascending order
															const std::vector<double> &death_times,		// 1D array of size NT, listing death times in ascending order
															const double 		start_diversity,	// (INPUT) diversity prior to any recorded birth & death events. Will typically be 0 or 1.
															const double		Nsplits){			// (INPUT) number of new species emerging from each speciation event
	const long NT = times.size();
	const long NB = birth_times.size();
	const long ND = death_times.size();
	
	// bin birth events into slots
	// births_per_time[t] = number of births that occurred between times[t-1] and times[t]
	std::vector<double> births_per_time(NT,0);
	long time_slot = 0;
	for(long b=0; b<NB; ++b){
		const double birth_time = birth_times[b];
		while((times[time_slot]<birth_time) && (time_slot<NT)){ ++time_slot; }
		if(time_slot<NT) ++births_per_time[time_slot];
	}
	
	// bin death events into slots
	// deaths_per_time[t] = number of deaths that occurred between times[t-1] and times[t]
	std::vector<double> deaths_per_time(NT,0);
	time_slot = 0;
	for(long d=0; d<ND; ++d){
		const double death_time = death_times[d];
		while((times[time_slot]<death_time) && (time_slot<NT)){ ++time_slot; }
		if(time_slot<NT) ++deaths_per_time[time_slot];
	}
	
	// count cumulative births & deaths at each time slot
	double diversity = start_diversity;
	std::vector<double> diversities(NT);
	for(long t=0; t<NT; ++t){
		diversity += (Nsplits-1)*births_per_time[t] - deaths_per_time[t];
		diversities[t] = diversity;
	}
	
	return Rcpp::List::create(Rcpp::Named("diversities") = diversities);
}






// ##########################################################


#pragma mark -
#pragma mark Building auxiliary data structures
#pragma mark 



// retrieve the parent clade of each clade
// returns a 1D array of size (Ntips+Nnodes), with elements in (Ntips):(Ntips+Nnodes-1)
// for the root, the parent will be set to -1 (i.e. not available)
template<class ARRAY_TYPE>
void get_parent_per_clade(	const long			Ntips,
							const long 			Nnodes,
							const long			Nedges,
							const ARRAY_TYPE 	&tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
							std::vector<long>	&clade2parent){		// (OUTPUT) 1D array of size (Nnodes+Ntips), normally with values in 0:(Nnodes+Ntipes-1). Clades with no parent will have have value -1.
	clade2parent.assign(Ntips+Nnodes, -1);
	for(long edge=0; edge<Nedges; ++edge){
		clade2parent[tree_edge[edge*2+1]] = tree_edge[edge*2+0];
	}
}


// given a mapping clade-->parent (with the root's parent being <0), determine the root of a tree by traversing upwards
// this function is more efficient than just searching for the clade without parent
long get_root_from_clade2parent(const long 				first_guess,		// (INPUT) first guess for what the root should be. The better the guess (i.e. the closer to the root), the faster the true root will be found. A good first guess is usually Ntips;
								const std::vector<long> &clade2parent){		// (INPUT) 1D array of size (Nnodes+Ntips), normally with values in 0:(Nnodes+Ntipes-1). Clades with no parent will have have value -1.
	long clade = first_guess;
	while(clade2parent[clade]>=0){
		clade = clade2parent[clade];
	}
	return clade;
}


// given a mapping clade-->incoming_edge (with the root's incoming edge being <0), determine the root of a tree by traversing upwards
// this function is more efficient than just searching for the clade without parent
template<class ARRAY_TYPE>
long get_root_from_incoming_edge_per_clade(	const long 				first_guess,				// (INPUT) first guess for what the root should be. The better the guess (i.e. the closer to the root), the faster the true root will be found. A good first guess is usually Ntips;
											const ARRAY_TYPE		&tree_edge,					// (INPUT) 2D array of size Nedges x 2, in row-major format
											const std::vector<long> &incoming_edge_per_clade){	// (INPUT) 1D array of size (Nnodes+Ntips), normally with values in 0:(Nedges-1). Clades with no incoming edge will have have value -1.
	long clade = first_guess;
	while(incoming_edge_per_clade[clade]>=0){
		clade = tree_edge[2*incoming_edge_per_clade[clade]+0];
	}
	return clade;
}



// given a mapping clade-->inout edges, determine the root of a tree by traversing upwards
// this function is more efficient than just searching for the clade without parent
template<class ARRAY_TYPE>
long get_root_from_clade2inout_edges(	const long 				first_guess,				// (INPUT) first guess for what the root clade should be. The better the guess (i.e. the closer to the root), the faster the true root will be found. A good first guess is usually Ntips;
										const ARRAY_TYPE		&tree_edge,					// (INPUT) 2D array of size Nedges x 2, in row-major format
										const std::vector<long>	&clade2first_inout_edge,	// (INPUT) 1D array of size Nclades, with values in 0:(2*Nedges-1), mapping clades to their first incoming or outgoing edge.
										const std::vector<long>	&clade2last_inout_edge,		// (INPUT) 1D array of size Nclades, with values in 0:(2*Nedges-1), mapping clades to their last incoming or outgoing edge.
										const std::vector<long>	&inout_edges){				// (INPUT) 1D array of size 2*Nedges, with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in clade2first_inout_edge[] and clade2last_inout_edge[]) to original edge indices.

	long clade = first_guess;
	bool found_parent;
	do{
		found_parent = false;
		for(long e=clade2first_inout_edge[clade], edge; e<=clade2last_inout_edge[clade]; ++e){
			edge  = inout_edges[e];
			if(tree_edge[2*edge+1]==clade){
				// found incoming edge, so transition to parent
				clade = tree_edge[2*edge+0];
				found_parent = true;
				break;
			}
		}
	}while(found_parent);
	return clade;
}






// find the incoming edge per clade
// returns a 1D array of size (Ntips+Nnodes), with elements in 0,..,Nedges-1
// for the root, the incoming edge will be set to -1 (i.e. not available)
template<class ARRAY_TYPE>
void get_incoming_edge_per_clade(	const long			Ntips,
									const long 			Nnodes,
									const long			Nedges,
									const ARRAY_TYPE	&tree_edge,					// (INPUT) 2D array of size Nedges x 2, in row-major format
									std::vector<long>	&incoming_edge_per_clade){	// (OUTPUT) 1D array of size (Nnodes+Ntips), with values in 0,..,Nedges-1. 
	incoming_edge_per_clade.assign(Ntips+Nnodes,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
}


template<class ARRAY_TYPE>
void get_incoming_edge_per_tip(	const long			Ntips,
								const long			Nedges,
								const ARRAY_TYPE	&tree_edge,					// (INPUT) 2D array of size Nedges x 2, in row-major format
								std::vector<long>	&incoming_edge_per_tip){	// (OUTPUT) 1D array of size Ntips, with values in 0,..,Nedges-1. 
	incoming_edge_per_tip.assign(Ntips,-1);
	for(long edge=0, child; edge<Nedges; ++edge){
		child = tree_edge[edge*2+1];
		if(child<Ntips) incoming_edge_per_tip[child] = edge;
	}
}


// for a given phylogenetic tree, create list of incoming edges for each clade
// normally each clade has either 0 or 1 incoming edges
// the tree need not be rooted, and may include multifurcations and monofurcations
// [[Rcpp::export]]
std::vector<std::vector<long> > get_incoming_edges_per_clade_CPP(	const long				Ntips,
																	const long 				Nnodes,
																	const long				Nedges,
																	const std::vector<long>	&tree_edge){	// (INPUT) 2D array of size Nedges x 2, in row-major format
	std::vector<std::vector<long> > incoming_edges_per_clade(Ntips+Nnodes);
	for(long edge=0, child; edge<Nedges; ++edge){
		child = tree_edge[edge*2+1];
		incoming_edges_per_clade[child].push_back(edge);
	}
	return incoming_edges_per_clade;
}




// find the incoming edge per clade
// returns a 1D array of size (Ntips+Nnodes), with elements in 0,..,Nedges-1
// for the root, the incoming edge will be set to -1 (i.e. not available)
template<class ARRAY_TYPE>
void get_child_count_per_node(	const long			Ntips,
								const long 			Nnodes,
								const long			Nedges,
								const ARRAY_TYPE	&tree_edge,					// (INPUT) 2D array of size Nedges x 2, in row-major format
								std::vector<long>	&child_count_per_node){		// (OUTPUT) 1D array of size Nnodes
	child_count_per_node.assign(Nnodes,0);
	for(long edge=0; edge<Nedges; ++edge){
		if(tree_edge[edge*2+0]<Ntips) continue;
		child_count_per_node[tree_edge[edge*2+0]-Ntips] += 1;
	}
}




// determine which nodes are basal (i.e. all their children are tips)
template<class ARRAY_TYPE>
void determine_basal_nodes(	const long			Ntips,
							const long 			Nnodes,
							const long			Nedges,
							const ARRAY_TYPE	&tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format
							std::vector<char>	&node_is_basal){	// (OUTPUT) 1D array of size Nnodes
	node_is_basal.assign(Nnodes,1);
	for(long edge=0; edge<Nedges; ++edge){
		if(tree_edge[edge*2+0]<Ntips) continue;
		if(tree_edge[edge*2+1]>=Ntips) node_is_basal[tree_edge[edge*2+0]-Ntips] = 0;
	}
}





// determine root of a tree
// Assuming that the tree is connected (when edge directions are ignored), this function will return -1 if the tree is not properly rooted
// Hence, this function can also be used to check if the tree is properly rooted (provided that it is connected)
template<class ARRAY_TYPE>
long get_root_clade(const long			Ntips,
					const long 			Nnodes,
					const long			Nedges,
					const ARRAY_TYPE	&tree_edge){			// (INPUT) 2D array (in row-major format) of size Nedges x 2
	const long Nclades = Ntips+Nnodes;
	std::vector<long> Nparents_per_clade(Nclades,0);
	for(long edge=0; edge<Nedges; ++edge){
		Nparents_per_clade[tree_edge[edge*2+1]] += 1;
	}
	long root = -1;
	for(long c=0; c<Nclades; ++c){
		if(Nparents_per_clade[c]>1) return -1; // found a clade with multiple parents, which cannot be in a rooted tree
		if(Nparents_per_clade[c]==0){
			// found clade with no parents, so this may be root
			if(root>=0) return -1; // multiple roots found, which cannot be
			root = c;
		}
	}
	return root;
}



// Find the root of a tree (= node with no parent)
// This is an Rcpp wrapper for get_root_clade()
// [[Rcpp::export]]
long get_root_clade_CPP(const long			Ntips,
						const long 			Nnodes,
						const long			Nedges,
						const std::vector<long>	&tree_edge){	// (INPUT) 2D array (in row-major format) of size Nedges x 2
	return get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
}






// returns a mapping from nodes to their child clades (Nclades = Ntips+Nnodes)
// the tree can be multifurcating, and can also include nodes with a single child
// root must specify the root index in the tree (typically root = Ntips)
// Returned values:
//	node2first_child[p] will be an index pointing node p (p=0:(Nnodes-1)) to its first child in children[]
//	node2last_child[p] will be an index pointing node p (p=0:(Nnodes-1)) to its last child in children[]
// 	children[] will be a list of clade indices (i.e. in 0:(Nclades-1)), such that children[node2first_child[p]],...,children[node2last_child[p]] is the set of children of node p
void get_children_per_node(	const long			Ntips,
							const long 			Nnodes,
							const long			Nedges,
							const long 			root, 				// (INPUT) index of root node, i.e. an integer in 0:(Ntips+Nnodes-1)
							const std::vector<long> &tree_edge, 		// (INPUT) 2D array (in row-major format) of size Nedges x 2
							std::vector<long>	&node2first_child,	// (OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1).
							std::vector<long>	&node2last_child,	// (OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1).
							std::vector<long>	&children){			// (OUTPUT) 1D array of size Nedges, with values in 0:(Nclades-1).
	// Terminology in this function:
	// 	'node' runs from 0:(Nnodes-1)
	// 	'tip' runs from 0:(Ntips-1)
	// 	'parent' and 'child' runs from 0:(Ntips+Nnodes-1)
	// 	'edge' runs from 0:(Nedges-1)
	// Recall that:
	// 	tree_edge[] is of size Nedge x 2 (flattened in row-major-format), with entries in 0:(Ntips+Nnodes-1)
	children.resize(Nedges);
	node2first_child.resize(Nnodes);
	node2last_child.resize(Nnodes);
	long node;

	// determine number of children per parent
	// child_count_per_node[n] will be the number of direct children of node n (n=0:(Nnodes-1))
	std::vector<long> child_count_per_node(Nnodes, 0);
	for(long e=0; e<Nedges; ++e){
		child_count_per_node[tree_edge[e*2+0] - Ntips] += 1;
	}
	
	// collect children per parent
	node2first_child[0] = 0;
	node2last_child[0]  = node2first_child[0]+child_count_per_node[0] - 1;
	if(Nnodes>1){
		for(long n=1; n<Nnodes; ++n){
			node2first_child[n] = node2last_child[n-1]+1;
			node2last_child[n]  = node2first_child[n]+child_count_per_node[n] - 1;
		}
	}
	for(long e=0; e<Nedges; ++e){
		node = tree_edge[e*2+0] - Ntips;
		children[node2first_child[node]+child_count_per_node[node]-1] = tree_edge[e*2+1];
		child_count_per_node[node] -= 1;
	}		
}



// Calculate lookup tables mapping nodes to their outgoing (children) edges
// Requirements:
//    The tree can be multifurcating, and can also include nodes with a single child
//    The tree can be rooted or unrooted (all information on edge direction is taken from the tree_edge[] table)
// Returned values:
//	  node2first_edge[p] will be an index pointing node p (p=0:(Nnodes-1)) to edges[]
//	  node2last_edge[p] will be an index pointing node p (p=0:(Nnodes-1)) to edges[]
// 	  edges[] will be a list of edge indices (i.e. in 0:(Nedges-1)), such that edges[node2first_edge[p]],...,edges[node2last_edge[p]] is the set of edges leaving node p
template<class ARRAY_TYPE>
void get_node2edge_mappings(const long			Ntips,
							const long 			Nnodes,
							const long			Nedges,
							const ARRAY_TYPE	&tree_edge, 		// (INPUT) 2D array (in row-major format) of size Nedges x 2
							std::vector<long>	&node2first_edge,	// (OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their first outgoing edge.
							std::vector<long>	&node2last_edge,	// (OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their last outgoing edge.
							std::vector<long>	&edges){			// (OUTPUT) 1D array of size Nedges, with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in node2first_edge[] and node2last_edge[]) to original edge indices.
	// Terminology in this function:
	// 	'node' runs from 0:(Nnodes-1)
	// 	'tip' runs from 0:(Ntips-1)
	// 	'parent' and 'child' runs from 0:(Ntips+Nnodes-1)
	// 	'edge' runs from 0:(Nedges-1)
	// Recall that:
	// 	tree_edge[] is of size Nedge x 2 (flattened in row-major-format), with entries in 0:(Ntips+Nnodes-1)

	edges.resize(Nedges);
	node2first_edge.resize(Nnodes);
	node2last_edge.resize(Nnodes);

	// determine number of children/edges per parent
	// child_count_per_node[n] will be the number of direct children of node n (n=0:(Nnodes-1))
	std::vector<long> child_count_per_node(Nnodes, 0);
	for(long e=0; e<Nedges; ++e){
		child_count_per_node[tree_edge[e*2+0] - Ntips] += 1;
	}
	// collect children per parent
	node2first_edge[0] = 0;
	node2last_edge[0]  = node2first_edge[0]+child_count_per_node[0] - 1;
	if(Nnodes>1){
		for(long n=1; n<Nnodes; ++n){
			node2first_edge[n] = node2last_edge[n-1]+1;
			node2last_edge[n]  = node2first_edge[n]+child_count_per_node[n] - 1;
		}
	}
	for(long e=0, node; e<Nedges; ++e){
		node = tree_edge[e*2+0] - Ntips;
		edges[node2first_edge[node]+child_count_per_node[node]-1] = e;
		child_count_per_node[node] -= 1;
	}
}




// Returns a list of all nodes (and optionally tips) of a tree, such that each node appears prior to its children, and such that nodes closer to the root (in terms of branching counts) appear first
// Also returns a list mapping nodes to their outgoing (children) edges (e.g. as listed in tree_edge)
// Nodes and tips are explored in a breadth-first-search order, from root to tips.
// the tree can be multifurcating, and can also include nodes with a single child
// root must specify the root index in the tree (typically root = Ntips)
// Returned values:
//	queue: A 1D array of integers in 0:(Ntips+Nnodes-1) if include_tips==true, or (Ntips):(Ntips+Nnodes-1) if include_tips==false
//	node2first_edge[p] will be an index pointing node p (p=0:(Nnodes-1)) to edges[]
//	node2last_edge[p] will be an index pointing node p (p=0:(Nnodes-1)) to edges[]
// 	edges[] will be a list of edge indices (i.e. in 0:(Nedges-1)), such that edges[node2first_edge[p]],...,edges[node2last_edge[p]] is the set of edges leaving node p
template<class ARRAY_TYPE>
void get_tree_traversal_root_to_tips(	const long			Ntips,
										const long 			Nnodes,
										const long			Nedges,
										const long 			root, 							// (INPUT) index of root node, i.e. an integer in 0:(Ntips+Nnodes-1)
										const ARRAY_TYPE	&tree_edge, 					// (INPUT) 2D array (in row-major format) of size Nedges x 2
										const bool			include_tips,					// (INPUT) if true, then tips are included in the returned queue[]. This does not affect the returned arrays node2first_edge[], node2last_edge[], edges[].
										const bool			precalculated_edge_mappings,	// (INPUT) if true, then the edge mapping tables node2first_edge[], node2last_edge[] and edges[] are taken as is. Otherwise, they are calculated from scratch.
										std::vector<long>	&queue,							// (OUTPUT) 1D array of size Nnodes if include_tips==false, or size (Ntips+Nnodes) if include_tips==true.
										std::vector<long>	&node2first_edge,				// (INPUT/OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their first outgoing edge. Either pre-calculated, or to be calculated by this function.
										std::vector<long>	&node2last_edge,				// (INPUT/OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their last outgoing edge. Either pre-calculated, or to be calculated by this function.
										std::vector<long>	&edges,							// (INPUT/OUTPUT) 1D array of size Nedges, with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in node2first_edge[] and node2last_edge[]) to original edge indices. Either pre-calculated, or to be calculated by this function.
										const bool			verbose,
										const string		&verbose_prefix){
	// get node-->edge mappings if needed
	if(!precalculated_edge_mappings){
		get_node2edge_mappings(	Ntips,
								Nnodes,
								Nedges,
								tree_edge,
								node2first_edge,
								node2last_edge,
								edges);
	}
	
	// fill queue from root to tips
	long child,node;
	queue.clear();
	queue.reserve(include_tips ? Ntips+Nnodes : Nnodes);
	queue.push_back(root);
	long queue_pointer = 0;
	while(queue_pointer<queue.size()){
		node = queue[queue_pointer] - Ntips;
		queue_pointer += 1;
		if(node<0) continue; // queue[queue_pointer] was actually a tip, not an internal node
		if(node2first_edge[node]>node2last_edge[node]){
			// this should not happen (every node needs to have at least one child)
			if(verbose) Rcout << verbose_prefix << "WARNING: Node " << node << " has no children\n";
			continue;
		}
		for(long ei=node2first_edge[node]; ei<=node2last_edge[node]; ++ei){
			child = tree_edge[edges[ei]*2+1];
			if((!include_tips) && (child<Ntips)) continue; // this child is a tip, so skip as requested
			// append child to queue
			queue.push_back(child);
		}
	}
}


// R wrapper for the function get_tree_traversal_root_to_tips(..) from above
// [[Rcpp::export]]
Rcpp::List get_tree_traversal_root_to_tips_CPP(	const long			Ntips,
												const long 			Nnodes,
												const long			Nedges,
												const std::vector<long>	&tree_edge, 		// (INPUT) 2D array (in row-major format) of size Nedges x 2
												const bool			include_tips){		// (INPUT) if true, then tips are included in the returned queue[]. This does not affect the returned arrays node2first_edge[], node2last_edge[], edges[].
	// determine root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
	
	// get tree traversal
	std::vector<long> queue, node2first_edge, node2last_edge, edges;
	get_tree_traversal_root_to_tips(Ntips,
									Nnodes,
									Nedges,
									root,
									tree_edge,
									include_tips,
									false,
									queue,
									node2first_edge,
									node2last_edge,
									edges,
									false,
									"");
	return Rcpp::List::create(	Rcpp::Named("queue") 			= Rcpp::wrap(queue),
								Rcpp::Named("node2first_edge") 	= Rcpp::wrap(node2first_edge),
								Rcpp::Named("node2last_edge")	= Rcpp::wrap(node2last_edge),
								Rcpp::Named("edges")			= Rcpp::wrap(edges));

}



template<class ARRAY_TYPE>
void get_tree_traversal_depth_first_search(	const long			Ntips,
											const long 			Nnodes,
											const long			Nedges,
											const long 			root, 							// (INPUT) index of root node, i.e. an integer in 0:(Ntips+Nnodes-1)
											const ARRAY_TYPE	&tree_edge, 					// (INPUT) 2D array (in row-major format) of size Nedges x 2
											const bool			include_tips,					// (INPUT) if true, then tips are included in the returned queue[]. This does not affect the returned arrays node2first_edge[], node2last_edge[], edges[].
											const bool			precalculated_edge_mappings,	// (INPUT) if true, then the edge mapping tables node2first_edge[], node2last_edge[] and edges[] are taken as is. Otherwise, they are calculated from scratch.
											std::vector<long>	&queue,							// (OUTPUT) 1D array of size Nnodes if include_tips==false, or size (Ntips+Nnodes) if include_tips==true.
											std::vector<long>	&node2first_edge,				// (INPUT/OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their first outgoing edge. Either pre-calculated, or to be calculated by this function.
											std::vector<long>	&node2last_edge,				// (INPUT/OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their last outgoing edge. Either pre-calculated, or to be calculated by this function.
											std::vector<long>	&edges){						// (INPUT/OUTPUT) 1D array of size Nedges, with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in node2first_edge[] and node2last_edge[]) to original edge indices. Either pre-calculated, or to be calculated by this function.

	// get node-->edge mappings if needed
	if(!precalculated_edge_mappings){
		get_node2edge_mappings(	Ntips,
								Nnodes,
								Nedges,
								tree_edge,
								node2first_edge,
								node2last_edge,
								edges);
	}
	
	// fill stack in depth-first-search direction
	// use a scratch_stack for traversing nodes
	long node,clade;
	std::vector<long> scratch_stack;
	scratch_stack.reserve(floor(2*log(Ntips)/log(2.0))); // rough estimate of typical tree depth x 2. scratch_stack may be resized along the way if needed.
	scratch_stack.push_back(root);
	queue.clear();
	queue.reserve(include_tips ? Ntips+Nnodes : Nnodes);
	while(scratch_stack.size()>0){
		clade = scratch_stack.back();
		scratch_stack.pop_back();
		if(include_tips || (clade>=Ntips)) queue.push_back(clade);
		if(clade>=Ntips){
			node = clade - Ntips;
			for(long ei=node2first_edge[node]; ei<=node2last_edge[node]; ++ei){
				scratch_stack.push_back(tree_edge[edges[ei]*2+1]);
			}
		}
	}
}



// given a phylogenetic tree, create a list of outgoing edges for each clade
// the tree need not be rooted, and may include monofurcations and multifurcations
// [[Rcpp::export]]
std::vector<std::vector<long> > get_outgoing_edges_per_clade_CPP(	const long				Ntips,
																	const long 				Nnodes,
																	const long				Nedges,
																	const std::vector<long>	&tree_edge){ // (INPUT) 2D array (in row-major format) of size Nedges x 2
	const long Nclades = Ntips+Nnodes;
	std::vector<std::vector<long> > edges_per_clade(Nclades);

	// determine number of edges per clade
	// edge_count_per_clade[n] will be the number of direct children of node n (n=0:(Nnodes-1))
	std::vector<long> edge_count_per_clade(Nclades, 0);
	for(long e=0; e<Nedges; ++e){
		edge_count_per_clade[tree_edge[e*2+0]] += 1;
	}
	// collect outgoing edges per clade
	for(long clade=0; clade<Nclades; ++clade){
		edges_per_clade.reserve(edge_count_per_clade[clade]);
	}
	for(long edge=0, parent; edge<Nedges; ++edge){
		parent = tree_edge[edge*2+0];
		edges_per_clade[parent].push_back(edge);
	}
	
	return edges_per_clade;
}



// Given a phylogenetic tree, create lookup tables listing incoming & outgoing edges for each clade (tip & node)
// Requirements:
//    The tree can include monofurcations and/or multifurcations
//    The tree need not be rooted
template<class ARRAY_TYPE>
void get_inout_edges_per_clade(	const long			Ntips,
								const long 			Nnodes,
								const long			Nedges,
								const ARRAY_TYPE	&tree_edge, 		// (INPUT) 2D array (in row-major format) of size Nedges x 2
								std::vector<long>	&clade2first_edge,	// (OUTPUT) 1D array of size Nclades, with values in 0:(Nedges-1), mapping clades to their first incoming or outgoing edge.
								std::vector<long>	&clade2last_edge,	// (OUTPUT) 1D array of size Nclades, with values in 0:(Nedges-1), mapping clades to their last incoming or outgoing edge.
								std::vector<long>	&edges){			// (OUTPUT) 1D array of size 2*Nedges, with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in clade2first_edge[] and clade2last_edge[]) to original edge indices. Note that each edge is listed twice in this array (once as an outgoing and once as an incoming edge).
	const long Nclades = Ntips+Nnodes;
	edges.resize(2*Nedges);
	clade2first_edge.resize(Nclades);
	clade2last_edge.resize(Nclades);

	// determine number of edges per clade
	// edge_count_per_clade[n] will be the number of direct children or parents of node n (n=0:(Nnodes-1))
	std::vector<long> edge_count_per_clade(Nclades, 0);
	for(long e=0; e<Nedges; ++e){
		edge_count_per_clade[tree_edge[e*2+0]] += 1;	// count edge towards parent
		edge_count_per_clade[tree_edge[e*2+1]] += 1;	// count edge towards child
	}
	// collect edges per clade
	clade2first_edge[0] = 0;
	clade2last_edge[0]  = clade2first_edge[0]+edge_count_per_clade[0] - 1;
	if(Nclades>1){
		for(long c=1; c<Nclades; ++c){
			clade2first_edge[c] = clade2last_edge[c-1]+1;
			clade2last_edge[c]  = clade2first_edge[c]+edge_count_per_clade[c] - 1;
		}
	}
	for(long e=0, clade; e<Nedges; ++e){
		// add edge to parent
		clade = tree_edge[e*2+0];
		edges[clade2first_edge[clade]+edge_count_per_clade[clade]-1] = e;
		edge_count_per_clade[clade] -= 1;

		// add edge to child
		clade = tree_edge[e*2+1];
		edges[clade2first_edge[clade]+edge_count_per_clade[clade]-1] = e;
		edge_count_per_clade[clade] -= 1;
	}
}



// Given a phylogenetic tree, create lookup tables listing incoming & outgoing edges for each node
// Requirements:
//    The tree can include monofurcations and/or multifurcations
//    The tree need not be rooted
template<class ARRAY_TYPE>
void get_inout_edges_per_node(	const long			Ntips,
								const long 			Nnodes,
								const long			Nedges,
								const ARRAY_TYPE	&tree_edge, 		// (INPUT) 2D array (in row-major format) of size Nedges x 2
								std::vector<long>	&node2first_edge,	// (OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their first incoming or outgoing edge.
								std::vector<long>	&node2last_edge,	// (OUTPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their last incoming or outgoing edge.
								std::vector<long>	&edges){			// (OUTPUT) 1D array of size 2*Nedges, with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in node2first_edge[] and node2last_edge[]) to original edge indices. Note that some edges will be listed twice in this array (once as an outgoing and once as an incoming edge).
	edges.resize(2*Nedges-Ntips);
	node2first_edge.resize(Nnodes);
	node2last_edge.resize(Nnodes);

	// determine number of edges per node
	// edge_count_per_node[n] will be the number of direct children or parents of node n (n=0:(Nnodes-1))
	std::vector<long> edge_count_per_node(Nnodes, 0);
	for(long e=0; e<Nedges; ++e){
		if(tree_edge[e*2+0]>=Ntips) edge_count_per_node[tree_edge[e*2+0]-Ntips] += 1;	// count edge towards parent
		if(tree_edge[e*2+1]>=Ntips) edge_count_per_node[tree_edge[e*2+1]-Ntips] += 1;	// count edge towards child
	}
	// collect edges per clade
	node2first_edge[0] = 0;
	node2last_edge[0]  = node2first_edge[0]+edge_count_per_node[0] - 1;
	if(Nnodes>1){
		for(long n=1; n<Nnodes; ++n){
			node2first_edge[n] = node2last_edge[n-1]+1;
			node2last_edge[n]  = node2first_edge[n]+edge_count_per_node[n] - 1;
		}
	}
	for(long e=0, node; e<Nedges; ++e){
		// add edge to parent
		node = tree_edge[e*2+0]-Ntips;
		if(node>=0){
			edges[node2first_edge[node]+edge_count_per_node[node]-1] = e;
			edge_count_per_node[node] -= 1;
		}

		// add edge to child
		node = tree_edge[e*2+1]-Ntips;
		if(node>=0){
			edges[node2first_edge[node]+edge_count_per_node[node]-1] = e;
			edge_count_per_node[node] -= 1;
		}
	}
}



// calculate the path from the root to each of the tree's tips
// [[Rcpp::export]]
std::vector<std::vector<long> > get_paths_root_to_tips_CPP(	const long 				Ntips,
															const long 				Nnodes,
															const long 				Nedges,
															const std::vector<long> &tree_edge){
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	std::vector<std::vector<long> > paths_to_tips(Ntips);
	for(long tip=0, clade; tip<Ntips; ++tip){
		paths_to_tips[tip].reserve(floor(2*log(Ntips)/log(2.0))); // rough estimate of typical tree depth x 2
		paths_to_tips[tip].push_back(tip);
		clade = tip;
		while(clade!=root){
			clade = clade2parent[clade];
			paths_to_tips[tip].push_back(clade);
		}
		// reverse direction (make root-->tip)
		reverse_array(paths_to_tips[tip]);
	}
	
	return paths_to_tips;
}


// determine edges adjacent to each edge (i.e. attached to each edge's child & parent)
// returns a list of size Nedges, each entry of which is a list of edge indices adjacent to the focal edge (the upstream adjacent edge is listed first)
// [[Rcpp::export]]
std::vector<std::vector<long> > get_adjacent_edges_per_edge_CPP(const long 				Ntips,
																const long 				Nnodes,
																const long 				Nedges,
																const std::vector<long> &tree_edge){
	// get incoming edge for each clade
	std::vector<long> incoming_edge_per_clade;
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);
	
	std::vector<vector<long> > adjacent_edges_per_edge(Nedges);
	// record all upstream adjacents first
	for(long edge=0, parent, upstream_edge; edge<Nedges; ++edge){
		parent = tree_edge[edge*2+0];
		upstream_edge = incoming_edge_per_clade[parent];
		if(upstream_edge<0) continue;
		adjacent_edges_per_edge[edge].push_back(upstream_edge);
	}
	// record all downstream adjacents
	for(long edge=0, parent, upstream_edge; edge<Nedges; ++edge){
		parent = tree_edge[edge*2+0];
		upstream_edge = incoming_edge_per_clade[parent];
		if(upstream_edge<0) continue;
		adjacent_edges_per_edge[upstream_edge].push_back(edge);
	}
	
	return adjacent_edges_per_edge;
}	



class tree_traversal{
public:
	bool includes_tips;
	long Ntips, Nnodes, Nedges;
	std::vector<long> queue;
	std::vector<long> node2first_edge, node2last_edge;
	std::vector<long> edge_mapping;
	
	// Constructor/initializer
	template<class ARRAY_TYPE>
	tree_traversal(	const long			Ntips_,
					const long 			Nnodes_,
					const long			Nedges_,
					const long 			root, 							// (INPUT) index of root node, i.e. an integer in 0:(Ntips+Nnodes-1)
					const ARRAY_TYPE	&tree_edge, 					// (INPUT) 2D array (in row-major format) of size Nedges x 2
					const bool			include_tips,					// (INPUT) if true, then tips are included in the returned queue[]. This does not affect the returned arrays node2first_edge[], node2last_edge[], edges[].
					const bool			precalculated_edge_mappings){	// (INPUT) if true, then the edge mapping tables node2first_edge[], node2last_edge[] and edges[] are taken as is. Otherwise, they are calculated from scratch.
		includes_tips = include_tips;
		Ntips  = Ntips_;
		Nnodes = Nnodes_;
		Nedges = Nedges_;
		get_tree_traversal_root_to_tips(Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										include_tips,
										precalculated_edge_mappings,
										queue,
										node2first_edge,
										node2last_edge,
										edge_mapping,
										false,
										"");
	}
};






#pragma mark -
#pragma mark Tree statistics
#pragma mark 




// Calculate the sum of incoming and outgoing edge lengths for each tip and node in a tree.
// Hence, this function returns a numeric vector of length Nclades.
// This is needed e.g. for constructing the modified graph Laplacian of a tree (Lewitus and Morlon, 2016, Systematic Biology. 65:495-507)
// Requirements:
//   The tree does not need to be rooted
//   The tree can include multifurcations as well as monofurcations
// [[Rcpp::export]]
NumericVector get_sum_of_edge_lengths_per_clade_CPP(const long					Ntips,
													const long 					Nnodes,
													const long					Nedges,
													const std::vector<long> 	&tree_edge, 	// (INPUT) 2D array (in row-major format) of size Nedges x 2
													const std::vector<double> 	&edge_length){ 	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
	const long Nclades = Ntips + Nnodes;
	const bool has_edge_lengths = (!edge_length.empty());
	dvector edge_length_sums(Nclades,0);
	for(long e=0, parent, child; e<Nedges; ++e){
		parent = tree_edge[2*e+0];
		child  = tree_edge[2*e+1];
		edge_length_sums[parent] += (has_edge_lengths ? edge_length[e] : 1.0);
		edge_length_sums[child]  += (has_edge_lengths ? edge_length[e] : 1.0);
	}
	return Rcpp::wrap(edge_length_sums);
}


// Count the number of tips descending from each node
// The tree must be rooted; the root should be the unique node with no parent
void get_total_tip_count_per_node(	const long			Ntips,
									const long 			Nnodes,
									const long			Nedges,
									const std::vector<long> &tree_edge, 			// (INPUT) 2D array (in row-major format) of size Nedges x 2
									std::vector<long>	&node2total_tip_count){	// (OUTPUT) array of size Nnodes, with each entry being the total number of tips descending from the node
	long clade;

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal route (root --> tips)
	// traversal_queue[] will be of size Nclades, and will have entries in 0:(Nclades-1)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");

	// calculate number of descending tips per node, traversing tips-->root (excluding the root)
	node2total_tip_count.assign(Nnodes,0);
	for(long q=traversal_queue.size()-1; q>=1; --q){
		clade = traversal_queue[q];
		node2total_tip_count[clade2parent[clade]-Ntips] += (clade<Ntips ? 1 : node2total_tip_count[clade-Ntips]);
	}
}


// Count the number of tips descending from each node
// This is an Rcpp wrapper function for get_total_tip_count_per_node(..)
// Requirements:
//   The tree must be rooted; the root should be the unique node with no parent
//   The tree can include multifurcations as well as monofurcations
// [[Rcpp::export]]
IntegerVector get_total_tip_count_per_node_CPP(	const long			Ntips,
												const long 			Nnodes,
												const long			Nedges,
												const std::vector<long> &tree_edge){	// (INPUT) 2D array (in row-major format) of size Nedges x 2
	std::vector<long> node2total_tip_count;
	get_total_tip_count_per_node(	Ntips,
									Nnodes,
									Nedges,
									tree_edge, 
									node2total_tip_count);
	return Rcpp::wrap(node2total_tip_count);
}





// For each node, calculate the mean phylogenetic distance to its descending tips
// Requirements:
//   The tree must be rooted; the root should be the unique node with no parent
//   The tree can include multifurcations as well as monofurcations
// [[Rcpp::export]]
NumericVector get_mean_depth_per_node_CPP(	const long					Ntips,
											const long 					Nnodes,
											const long					Nedges,
											const std::vector<long> 	&tree_edge, 	// (INPUT) 2D array (in row-major format) of size Nedges x 2
											const std::vector<double> 	&edge_length){ 	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// get incoming edge for each clade
	std::vector<long> incoming_edge_per_clade;
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal route (tips --> root)
	// traversal_queue[] will be of size Nclades, and will have entries in 0:(Nclades-1)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true, 	// include tips
										false, 	// no precalculated edge mappings
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");

	// calculate number of descending tips per node & mean distance to tips, traversing tips-->root (excluding the root)
	std::vector<long> node2tip_count(Nnodes,0);
	std::vector<double> node2tip_depth(Nnodes,0);
	for(long q=traversal_queue.size()-1, clade, parent, cnode; q>=1; --q){
		clade	= traversal_queue[q];
		cnode	= clade-Ntips;
		parent 	= clade2parent[clade];
		node2tip_count[parent-Ntips] += (clade<Ntips ? 1 : node2tip_count[cnode]);
		node2tip_depth[parent-Ntips] += (clade<Ntips ? 0 : node2tip_depth[cnode]) + (clade<Ntips ? 1 : node2tip_count[cnode]) * (edge_length.size()==0 ? 1 : edge_length[incoming_edge_per_clade[clade]]);
	}
	for(long node=0; node<Nnodes; ++node){
		node2tip_depth[node] /= node2tip_count[node];
	}
	return Rcpp::wrap(node2tip_depth);
}



// For each node, calculate the number of immediate children
// Requirements:
//   The tree can include multifurcations as well as monofurcations
// [[Rcpp::export]]
NumericVector get_child_count_per_node_CPP(	const long				Ntips,
											const long 				Nnodes,
											const long				Nedges,
											const std::vector<long> &tree_edge){	// (INPUT) 2D array (in row-major format) of size Nedges x 2
	std::vector<long> node2child_count(Nnodes,0);
	for(long e=0; e<Nedges; ++e){
		node2child_count[tree_edge[e*2+0] - Ntips] += 1;
	}
	return Rcpp::wrap(node2child_count);
}




// Calculate sum of all branch lengths, for each subtree in a tree
// This is equivalent to the 'phylogenetic diversity' measure introduced by Faith (1992).
// References: 
//    Faith (1992). Conservation evaluation and phylogenetic diversity. Biological Conservation 61:1-10.
//    Mark Vellend et al. Measuring phylogenetic biodiversity. Table 14.2 (presence/absence based phylogenetic diversity)
void get_cumulative_edge_lengths_per_node(	const long			Ntips,
											const long 			Nnodes,
											const long			Nedges,
											const long 			root, 				// (INPUT) index of root node, i.e. an integer in 0:(Ntips+Nnodes-1)
											const std::vector<long> &tree_edge, 		// (INPUT) 2D array (in row-major format) of size Nedges x 2
											const std::vector<double> &edge_length, 		// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
											std::vector<double>	&node2CBL){			// (OUTPUT) array of size Nnodes, with each entry being the cumulative branch length (phylogenetic diversity) of the subtree rooted in that node.
	const long Nclades = Ntips+Nnodes;
	long clade;

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}

	// get tree traversal route (tips --> root)
	// traversal_queue[] will be of size Nclades, and will have entries in 0:(Nclades-1)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");
	reverse_array(traversal_queue); // make tips-->roots

	// calculate phylogenetic diversities (cumulative branch lengths), traversing tips-->root
	node2CBL.assign(Nnodes,0);
	for(long q=0; q<traversal_queue.size(); ++q){
		clade  = traversal_queue[q];
		if(clade==root) continue;
		node2CBL[clade2parent[clade]-Ntips] += (clade<Ntips ? 0.0 : node2CBL[clade-Ntips]) + (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]);
	}
}



// Calculate the maximum & minimum distance of any tip to the root
// [[Rcpp::export]]
Rcpp::List get_min_max_tip_distance_from_root_CPP(	const long 			Ntips,
													const long 			Nnodes,
													const long 			Nedges,
													const std::vector<long> &tree_edge,		// (INPUT) 2D array of size Nedges x 2 in row-major format
													const std::vector<double> &edge_length){	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
	const long Nclades = Ntips + Nnodes;
	long parent, clade;
										
	// determine root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, node2first_edge, node2last_edge, edge_mapping;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										node2first_edge,
										node2last_edge,
										edge_mapping,
										false,
										"");
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
										
	// calculate distance of each node to its nearest descending tip
	// (traverse tips --> root)
	std::vector<double> min_tip_distance_per_node(Nnodes,INFTY_D), max_tip_distance_per_node(Nnodes,0);
	double min_distance, max_distance;
	for(long q=traversal_queue.size()-1; q>=0; --q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent = tree_edge[incoming_edge_per_clade[clade]*2 + 0];
		min_distance = (clade<Ntips ? 0.0 : min_tip_distance_per_node[clade-Ntips]) + (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]);
		max_distance = (clade<Ntips ? 0.0 : max_tip_distance_per_node[clade-Ntips]) + (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]);
		min_tip_distance_per_node[parent-Ntips]	= min(min_tip_distance_per_node[parent-Ntips], min_distance);
		max_tip_distance_per_node[parent-Ntips]	= max(max_tip_distance_per_node[parent-Ntips], max_distance);
	}

	return Rcpp::List::create(	Rcpp::Named("min_distance")	= Rcpp::wrap(min_tip_distance_per_node[root-Ntips]),
								Rcpp::Named("max_distance")	= Rcpp::wrap(max_tip_distance_per_node[root-Ntips]));
}



// calculate distance from root, for each clade (tips+nodes)
// distance from root = cumulative branch length from root to the clade
template<class ARRAY_TYPE_INT, class ARRAY_TYPE_D>
void get_distances_from_root(	const long 				Ntips,
								const long 				Nnodes,
								const long 				Nedges,
								const ARRAY_TYPE_INT 	&tree_edge,		// (INPUT) 2D array of size Nedges x 2 in row-major format
								const ARRAY_TYPE_D		&edge_length, 	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
								const tree_traversal	&traversal,		// (INPUT) tree traversal data structure (root-->tips). Can (but does not need to) include tips.
								dvector					&distances){	// (OUTPUT) 1D array of size Nclades, listing the phylogenetic distance of each clade from the root
	const long Nclades = Ntips + Nnodes;
	const bool got_edge_lengths = (edge_length.size()>0);				
	const long root = traversal.queue[0];

	distances.resize(Nclades);
	distances[root] = 0;
	for(long q=0, clade; q<traversal.queue.size(); ++q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue;
		for(long e=traversal.node2first_edge[clade-Ntips], edge; e<=traversal.node2last_edge[clade-Ntips]; ++e){
			edge  = traversal.edge_mapping[e];
			distances[tree_edge[2*edge+1]] = distances[clade] + (got_edge_lengths ? edge_length[edge] : 1.0);
		}
	}
}



// calculate distance from root, for each clade (tips+nodes)
// distance from root = cumulative branch length from root to the clade
template<class ARRAY_TYPE_INT, class ARRAY_TYPE_D>
void get_distances_from_root(	const long 				Ntips,
								const long 				Nnodes,
								const long 				Nedges,
								const ARRAY_TYPE_INT 	&tree_edge,		// (INPUT) 2D array of size Nedges x 2 in row-major format
								const ARRAY_TYPE_D		&edge_length, 	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
								std::vector<double>		&distances){	// (OUTPUT) 1D array of size Nclades, listing the phylogenetic distance of each clade from the root

	// find root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);

	// get tree traversal (root-->tips), including tips
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);

	// calculate distances from root
	get_distances_from_root(Ntips, Nnodes, Nedges, tree_edge, edge_length, traversal, distances);
}


// calculate the age (time before present) of each clade (tips+nodes) in a timetree
// The tree does not need to be ultrametric, but the youngest tip is assumed to have been sampled at present-day.
void get_clade_ages(const long 					Ntips,
					const long 					Nnodes,
					const long 					Nedges,
					const std::vector<long> 	&tree_edge,		// (INPUT) 2D array of size Nedges x 2 in row-major format
					const std::vector<double>	&edge_length, 	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
					const tree_traversal		&traversal,		// (INPUT) tree traversal data structure (root-->tips). Can (but does not need to) include tips.
					dvector						&ages){			// (OUTPUT) 1D array of size Nclades, listing the age of each clade
	const long Nclades = Ntips + Nnodes;
	const bool got_edge_lengths = (edge_length.size()>0);				
	const long root = traversal.queue[0];

	// first calculate distances from root (use ages[] as convenient temporary storage)
	ages.resize(Nclades);
	ages[root] = 0;
	for(long q=0, clade; q<traversal.queue.size(); ++q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue;
		for(long e=traversal.node2first_edge[clade-Ntips], edge; e<=traversal.node2last_edge[clade-Ntips]; ++e){
			edge  = traversal.edge_mapping[e];
			ages[tree_edge[2*edge+1]] = ages[clade] + (got_edge_lengths ? edge_length[edge] : 1.0);
		}
	}
	
	// convert distances-from-root to ages
	const double root_age = vector_max(ages);
	for(long clade=0; clade<Nclades; ++clade){
		ages[clade] = root_age - ages[clade];
	}
}



// calculate distance from root, for each clade (tips+nodes)
// distance from root = cumulative branch length from root to the clade
// This is an Rcpp wrapper for the function get_distances_from_root(..)
// [[Rcpp::export]]
NumericVector get_distances_from_root_CPP(	const long 					Ntips,
											const long 					Nnodes,
											const long 					Nedges,
											const std::vector<long> 	&tree_edge,			// (INPUT) 2D array of size Nedges x 2 in row-major format
											const std::vector<double> 	&edge_length){ 		// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
	std::vector<double> distances;
	get_distances_from_root(Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							edge_length,
							distances);
	return Rcpp::wrap(distances);
}









// For each clade (tip & node) in a tree, find the closest tip (in terms of cumulative branch length).
// Optionally, the search can be restricted to descending tips.
// Optionally, the search can also be restricted to a subset of target tips.
// If you want distances in terms of branch counts (instead of cumulative branch lengths), simply provide an empty edge_length[].
// Requirements:
//   The input tree must be rooted (root will be determined automatically, as the node that has no incoming edge)
//   The input tree can be multifurcating and/or monofurcating
// [[Rcpp::export]]
Rcpp::List get_closest_tip_per_clade_CPP(	const long 			Ntips,
											const long 			Nnodes,
											const long 			Nedges,
											const std::vector<long> &tree_edge,				// 2D array of size Nedges x 2 in row-major format
											const std::vector<double> &edge_length, 			// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
											const std::vector<long>	&onlyToTips,			// 1D array listing target tips to restrict search to, or an empty std::vector (consider all tips as targets)
											bool				only_descending_tips,	// if true, then for each clade only descending tips are considered for nearest-distance. If false, some clades may have non-descending tips assigned as nearest tips.
											bool 				verbose,
											const std::string	&verbose_prefix){
	const long Nclades = Ntips + Nnodes;
	long parent, clade, tip, incoming_edge;
	double candidate_distance;

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue_root2tips, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue_root2tips,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);

	// Step 1: calculate nearest descending tip per clade (traverse tips --> root)
	std::vector<long> nearest_descending_tip_per_clade(Nclades,-1);
	std::vector<double> distance_to_nearest_descending_tip_per_clade(Nclades,INFTY_D);
	if(onlyToTips.size()==0){
		// consider all tips as potential targets
		for(long tip=0; tip<Ntips; ++tip){
			nearest_descending_tip_per_clade[tip] = tip;
			distance_to_nearest_descending_tip_per_clade[tip] = 0;
		}
	}else{
		// only consider provided tips as targets
		for(long t=0; t<onlyToTips.size(); ++t){
			tip = onlyToTips[t];
			nearest_descending_tip_per_clade[tip] = tip;
			distance_to_nearest_descending_tip_per_clade[tip] = 0;
		}
	}
	for(long q=traversal_queue_root2tips.size()-1; q>=0; --q){
		clade = traversal_queue_root2tips[q];
		if(clade==root) continue;
		if(nearest_descending_tip_per_clade[clade]<0) continue; // no descending tip available from this clade
		parent			= clade2parent[clade];
		incoming_edge	= incoming_edge_per_clade[clade];
		// propagate information about nearest descending tip, to parent (if better than already saved for the parent)
		candidate_distance = (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge]) + distance_to_nearest_descending_tip_per_clade[clade];
		if(candidate_distance<distance_to_nearest_descending_tip_per_clade[parent]){
			distance_to_nearest_descending_tip_per_clade[parent] = candidate_distance;
			nearest_descending_tip_per_clade[parent] = nearest_descending_tip_per_clade[clade];
		}
	}
	
	if(only_descending_tips){
		// only descending tips allowed, so we're finished
		return Rcpp::List::create(	Rcpp::Named("nearest_tips") 		= Rcpp::wrap(nearest_descending_tip_per_clade),
									Rcpp::Named("nearest_distances") 	= Rcpp::wrap(distance_to_nearest_descending_tip_per_clade));
	}
	
	// Step 2: calculate nearest tip per clade, regardless of whether descending or not (traverse root --> tips)
	std::vector<long> nearest_tip_per_clade(Nclades);
	std::vector<double> distance_to_nearest_tip_per_clade(Nclades);
	nearest_tip_per_clade[root] = nearest_descending_tip_per_clade[root];
	distance_to_nearest_tip_per_clade[root] = distance_to_nearest_descending_tip_per_clade[root];
	for(long q=0; q<traversal_queue_root2tips.size(); ++q){
		clade = traversal_queue_root2tips[q];
		if(clade==root) continue;
		parent				= clade2parent[clade];
		incoming_edge 		= incoming_edge_per_clade[clade];
		candidate_distance 	= (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge]) + distance_to_nearest_tip_per_clade[parent];
		if(candidate_distance<distance_to_nearest_descending_tip_per_clade[clade]){
			// it's shorter to go upwards, than downwards
			distance_to_nearest_tip_per_clade[clade] = candidate_distance;
			nearest_tip_per_clade[clade] = nearest_tip_per_clade[parent];
		}else{
			// nearest descending tip is also nearest tip
			distance_to_nearest_tip_per_clade[clade] = distance_to_nearest_descending_tip_per_clade[clade];
			nearest_tip_per_clade[clade] = nearest_descending_tip_per_clade[clade];
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("nearest_tips") 		= Rcpp::wrap(nearest_tip_per_clade),
								Rcpp::Named("nearest_distances") 	= Rcpp::wrap(distance_to_nearest_tip_per_clade));
}






// For each clade (tip & node) in a rooted tree, find the most distant tip (in terms of cumulative branch length).
// Optionally, the search can be restricted to descending tips.
// Optionally, the search can also be restricted to a subset of target tips.
// If you want distances in terms of branch counts (instead of cumulative branch lengths), simply provide an empty edge_length[].
// Requirements:
//   The input tree must be rooted (root will be determined automatically, as the node that has no incoming edge)
//   The input tree can be multifurcating and/or monofurcating
// [[Rcpp::export]]
Rcpp::List get_farthest_tip_per_clade_CPP(	const long 			Ntips,
											const long 			Nnodes,
											const long 			Nedges,
											const std::vector<long> &tree_edge,				// 2D array of size Nedges x 2 in row-major format
											const std::vector<double> &edge_length, 			// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
											const std::vector<long>	&onlyToTips,			// 1D array listing target tips to restrict search to, or an empty std::vector (consider all tips as targets)
											bool				only_descending_tips,	// if true, then for each clade only descending tips are considered for farthest-distance. If false, some clades may have non-descending tips assigned as farthest tips.
											bool 				verbose,
											const std::string	&verbose_prefix){
	const long Nclades = Ntips + Nnodes;
	long parent, clade, tip, incoming_edge;
	double candidate_distance;
	const bool unit_edge_lengths = (edge_length.size()==0);

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue_root2tips, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue_root2tips,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);

	// Step 1: calculate farthest descending tip per clade (traverse tips --> root)
	std::vector<long> farthest_descending_tip_per_clade(Nclades,-1);
	std::vector<double> distance_to_farthest_descending_tip_per_clade(Nclades,NAN_D);
	if(onlyToTips.size()==0){
		// consider all tips as potential targets
		for(long tip=0; tip<Ntips; ++tip){
			farthest_descending_tip_per_clade[tip] = tip;
			distance_to_farthest_descending_tip_per_clade[tip] = 0;
		}
	}else{
		// only consider provided tips as targets
		for(long t=0; t<onlyToTips.size(); ++t){
			tip = onlyToTips[t];
			farthest_descending_tip_per_clade[tip] = tip;
			distance_to_farthest_descending_tip_per_clade[tip] = 0;
		}
	}
	for(long q=traversal_queue_root2tips.size()-1; q>=0; --q){
		clade = traversal_queue_root2tips[q];
		if(clade==root) continue;
		if(farthest_descending_tip_per_clade[clade]<0) continue; // no descending tip available from this clade
		parent			= clade2parent[clade];
		incoming_edge	= incoming_edge_per_clade[clade];
		// propagate information about farthest descending tip, to parent (if more distant than already saved for the parent)
		candidate_distance = (unit_edge_lengths ? 1.0 : edge_length[incoming_edge]) + distance_to_farthest_descending_tip_per_clade[clade];
		if((farthest_descending_tip_per_clade[parent]<0) || (candidate_distance>distance_to_farthest_descending_tip_per_clade[parent])){
			distance_to_farthest_descending_tip_per_clade[parent] = candidate_distance;
			farthest_descending_tip_per_clade[parent] = farthest_descending_tip_per_clade[clade];
		}
	}
	
	if(only_descending_tips){
		// only descending tips allowed, so we're finished
		return Rcpp::List::create(	Rcpp::Named("farthest_descending_tips") 	 = Rcpp::wrap(farthest_descending_tip_per_clade),
									Rcpp::Named("farthest_descending_distances") = Rcpp::wrap(distance_to_farthest_descending_tip_per_clade));
	}
	

	// Step 2: calculate farthest upstream tip per clade (traverse root --> tips)
	// Note that for the root we keep this undefined (tip index -1)
	std::vector<long> farthest_upstream_tip_per_clade(Nclades,-1);
	std::vector<double> distance_to_farthest_upstream_tip_per_clade(Nclades,NAN_D);
	for(long q=1; q<traversal_queue_root2tips.size(); ++q){
		clade	= traversal_queue_root2tips[q];
		parent	= clade2parent[clade];
		incoming_edge = incoming_edge_per_clade[clade];
		for(long e=traversal_node2first_edge[parent-Ntips], edge, child; e<=traversal_node2last_edge[parent-Ntips]; ++e){
			edge = traversal_edges[e];
			if(edge==incoming_edge) continue;
			child = tree_edge[2*edge+1];
			if(farthest_descending_tip_per_clade[child]<0) continue;
			candidate_distance = (unit_edge_lengths ? 1.0+1.0 : edge_length[edge]+edge_length[incoming_edge]) + distance_to_farthest_descending_tip_per_clade[child];
			if((farthest_upstream_tip_per_clade[clade]<0) || (candidate_distance>distance_to_farthest_upstream_tip_per_clade[clade])){
				farthest_upstream_tip_per_clade[clade] = farthest_descending_tip_per_clade[child];
				distance_to_farthest_upstream_tip_per_clade[clade] = candidate_distance;
			}
		}
		// check if going further up than the parent leads to an even farther target tip
		if(farthest_upstream_tip_per_clade[parent]>=0){
			candidate_distance = (unit_edge_lengths ? 1.0 : edge_length[incoming_edge]) + distance_to_farthest_upstream_tip_per_clade[parent];
			if((farthest_upstream_tip_per_clade[clade]<0) || (candidate_distance>distance_to_farthest_upstream_tip_per_clade[clade])){
				farthest_upstream_tip_per_clade[clade] = farthest_upstream_tip_per_clade[parent];
				distance_to_farthest_upstream_tip_per_clade[clade] = candidate_distance;
			}
		}
	}

	// Step 3: calculate farthest tip per clade, regardless of whether descending or not (traverse root --> tips)
	std::vector<long> farthest_tip_per_clade(Nclades,-1);
	std::vector<double> distance_to_farthest_tip_per_clade(Nclades,0);
	for(long q=0; q<traversal_queue_root2tips.size(); ++q){
		clade = traversal_queue_root2tips[q];
		if((farthest_upstream_tip_per_clade[clade]<0) || (distance_to_farthest_descending_tip_per_clade[clade]>distance_to_farthest_upstream_tip_per_clade[clade])){
			// farthest tip for this clade is downstream
			farthest_tip_per_clade[clade] = farthest_descending_tip_per_clade[clade];
			distance_to_farthest_tip_per_clade[clade] = distance_to_farthest_descending_tip_per_clade[clade];
		}else if((farthest_descending_tip_per_clade[clade]<0) || (distance_to_farthest_descending_tip_per_clade[clade]<distance_to_farthest_upstream_tip_per_clade[clade])){
			// farthest tip for this clade is upstream
			farthest_tip_per_clade[clade] = farthest_upstream_tip_per_clade[clade];
			distance_to_farthest_tip_per_clade[clade] = distance_to_farthest_upstream_tip_per_clade[clade];
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("farthest_descending_tips") 	 	= Rcpp::wrap(farthest_descending_tip_per_clade),
								Rcpp::Named("farthest_descending_distances") 	= Rcpp::wrap(distance_to_farthest_descending_tip_per_clade),
								Rcpp::Named("farthest_upstream_tips") 			= Rcpp::wrap(farthest_upstream_tip_per_clade),
								Rcpp::Named("farthest_upstream_distances") 		= Rcpp::wrap(distance_to_farthest_upstream_tip_per_clade),
								Rcpp::Named("farthest_tips") 					= Rcpp::wrap(farthest_tip_per_clade),
								Rcpp::Named("farthest_distances") 				= Rcpp::wrap(distance_to_farthest_tip_per_clade));
}




// For each edge in a rooted tree, find the most distant tip (in terms of cumulative branch length) in either direction ("upstream" and "downstream").
// Optionally, the search can also be restricted to a subset of target tips.
// If you want distances in terms of branch counts (instead of cumulative branch lengths), simply provide an empty edge_length[].
// Requirements:
//   The input tree must be rooted (root will be determined automatically, as the node that has no incoming edge)
//   The input tree can be multifurcating and/or monofurcating
// [[Rcpp::export]]
Rcpp::List get_farthest_tips_per_edge_CPP(	const long 					Ntips,
											const long 					Nnodes,
											const long 					Nedges,
											const std::vector<long> 	&tree_edge,		// 2D array of size Nedges x 2 in row-major format
											const std::vector<double> 	&edge_length, 	// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
											const std::vector<long>		&onlyToTips){	// 1D array listing target tips to restrict search to, or an empty std::vector (consider all tips as targets)
	const long Nclades = Ntips + Nnodes;
	long parent, clade, node, tip, incoming_edge;
	double candidate_distance;
	const bool unit_edge_lengths = (edge_length.size()==0);

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips), including tips
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);

	// Step 1: calculate farthest descending tip per clade (traverse tips --> root), and thus per edge
	lvector farthest_descending_tip_per_clade(Nclades,-1);
	dvector distance_to_farthest_descending_tip_per_clade(Nclades,NAN_D);
	if(onlyToTips.size()==0){
		// consider all tips as potential targets
		for(long tip=0; tip<Ntips; ++tip){
			farthest_descending_tip_per_clade[tip] = tip;
			distance_to_farthest_descending_tip_per_clade[tip] = 0;
		}
	}else{
		// only consider provided tips as targets
		for(long t=0; t<onlyToTips.size(); ++t){
			tip = onlyToTips[t];
			farthest_descending_tip_per_clade[tip] = tip;
			distance_to_farthest_descending_tip_per_clade[tip] = 0;
		}
	}
	for(long q=traversal.queue.size()-1; q>=0; --q){
		clade = traversal.queue[q];
		if(clade==root) continue;
		if(farthest_descending_tip_per_clade[clade]<0) continue; // no descending tip available from this clade
		parent			= clade2parent[clade];
		incoming_edge	= incoming_edge_per_clade[clade];
		// propagate information about farthest descending tip, to parent (if more distant than already saved for the parent)
		candidate_distance = (unit_edge_lengths ? 1.0 : edge_length[incoming_edge]) + distance_to_farthest_descending_tip_per_clade[clade];
		if((farthest_descending_tip_per_clade[parent]<0) || (candidate_distance>distance_to_farthest_descending_tip_per_clade[parent])){
			distance_to_farthest_descending_tip_per_clade[parent] = candidate_distance;
			farthest_descending_tip_per_clade[parent] = farthest_descending_tip_per_clade[clade];
		}
	}
	lvector farthest_descending_tip_per_edge(Nedges,-1);
	dvector distance_to_farthest_descending_tip_per_edge(Nedges,NAN_D);
	for(long edge=0, child; edge<Nedges; ++edge){
		child = tree_edge[2*edge+1];
		farthest_descending_tip_per_edge[edge] 				= farthest_descending_tip_per_clade[child];
		distance_to_farthest_descending_tip_per_edge[edge] 	= distance_to_farthest_descending_tip_per_clade[child];
	}
	
	// Step 2: calculate farthest upstream tip per edge (traverse edges root --> tips)
	lvector farthest_upstream_tip_per_edge(Nedges,-1);
	dvector distance_to_farthest_upstream_tip_per_edge(Nedges,NAN_D);
	for(long q=0; q<traversal.queue.size(); ++q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue;
		node = clade-Ntips;
		for(long e=traversal.node2first_edge[node], edge; e<=traversal.node2last_edge[node]; ++e){
			edge = traversal.edge_mapping[e];
			// check incoming edge
			if(clade!=root){
				incoming_edge = incoming_edge_per_clade[clade];
				if(farthest_upstream_tip_per_edge[incoming_edge]>=0){
					farthest_upstream_tip_per_edge[edge] = farthest_upstream_tip_per_edge[incoming_edge];
					distance_to_farthest_upstream_tip_per_edge[edge] = (unit_edge_lengths ? 1.0 : edge_length[incoming_edge]) + distance_to_farthest_upstream_tip_per_edge[incoming_edge];
				}
			}
			// check all sister edges in an attempt to find more distant upstream tips
			for(long se=traversal.node2first_edge[node], sister_edge; se<=traversal.node2last_edge[node]; ++se){
				sister_edge = traversal.edge_mapping[se];
				if(sister_edge==edge) continue;
				if(farthest_descending_tip_per_edge[sister_edge]<0) continue; // no valid tip descending from sister_edge
				candidate_distance = (unit_edge_lengths ? 1.0 : edge_length[sister_edge]) + distance_to_farthest_descending_tip_per_edge[sister_edge];
				if((farthest_upstream_tip_per_edge[edge]<0) || (candidate_distance>distance_to_farthest_upstream_tip_per_edge[edge])){
					distance_to_farthest_upstream_tip_per_edge[edge] = candidate_distance;
					farthest_upstream_tip_per_edge[edge] = farthest_descending_tip_per_edge[sister_edge];
				}
			}
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("farthest_descending_tips") 	 	= Rcpp::wrap(farthest_descending_tip_per_edge),
								Rcpp::Named("farthest_descending_distances") 	= Rcpp::wrap(distance_to_farthest_descending_tip_per_edge),
								Rcpp::Named("farthest_upstream_tips") 			= Rcpp::wrap(farthest_upstream_tip_per_edge),
								Rcpp::Named("farthest_upstream_distances") 		= Rcpp::wrap(distance_to_farthest_upstream_tip_per_edge));
}



// Calculate phylogenetic distance matrix between all pairs of focal_clades
// Distance = cumulative branch length of both clades back to their most recent common ancestor (aka "patristic distance")
// This function is slightly different from get_distances_between_clades_CPP(), in that here the distances between all possible clade pairs are returned.
// The time complexity is O(Nfocals*Nfocals*Nanc + Ntips), where Nanc is the average number of ancestors per tip.
// Requirements:
//   The input tree must be rooted (root will be determined automatically, as the node that has no incoming edge)
//   The input tree can include multifurcations and monofurcations
// Attention: 0-based indexing is used for input and output variables, so make sure to shift indices in R before calling this function
// [[Rcpp::export]]
NumericMatrix get_distance_matrix_between_clades_CPP(	const long 					Ntips,
														const long 					Nnodes,
														const long 					Nedges,
														const std::vector<long> 	&tree_edge,			// 2D array of size Nedges x 2 in row-major format
														const std::vector<double> 	&edge_length, 		// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
														const std::vector<long> 	&focal_clades,		// 1D array of size Nfocals, containing values in 0:(Nclades-1). These will correspond to the rows & columns of the returned distance matrix.
														bool 						verbose,
														const std::string 			&verbose_prefix){
	const long Nclades = Ntips + Nnodes;
	const long Nfocals = focal_clades.size();
	long parent, clade;

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);
															
	// calculate number of ancestors and distance from root for each clade
	// (traverse root --> tips)
	std::vector<long> ancestor_count_per_clade(Nclades);
	std::vector<double> distance_from_root_per_clade(Nclades);
	ancestor_count_per_clade[root] 		= 0;
	distance_from_root_per_clade[root] 	= 0;
	for(long q=0; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		ancestor_count_per_clade[clade] 	= 1 + ancestor_count_per_clade[parent];
		distance_from_root_per_clade[clade]	=  (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]) + distance_from_root_per_clade[parent];
	}
	const long total_ancestor_count = vector_sum(ancestor_count_per_clade);
	
	// calculate ancestry for each clade in a long list ancestors[]
	// (traverse root --> tips)
	std::vector<long> clade2first_ancestor(Nclades); // for each clade c, ancestors[clade2first_ancestor[c]..clade2last_ancestor[c]] will be the list of ancestor clades leading to the clade c
	std::vector<long> clade2last_ancestor(Nclades);
	clade2first_ancestor[0] = 0;
	clade2last_ancestor[0] = clade2first_ancestor[0] + ancestor_count_per_clade[0] - 1;
	for(long c=1; c<Nclades; ++c){
		clade2first_ancestor[c] = clade2last_ancestor[c-1] + 1;
		clade2last_ancestor[c]  = clade2first_ancestor[c] + ancestor_count_per_clade[c] - 1;
	}
	std::vector<long> ancestors(total_ancestor_count);
	for(long q=0; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		// step 1: copy the parent's ancestry to the child's ancestry
		for(long a=clade2first_ancestor[parent]; a<=clade2last_ancestor[parent]; ++a){
			ancestors[clade2first_ancestor[clade]+(a-clade2first_ancestor[parent])] = ancestors[a];
		}
		// step 2: append the parent to the clade's ancestry
		ancestors[clade2last_ancestor[clade]] = parent;
	}
		
	// calculate most-recent-common-ancestor and phylogenetic distance for each focal clade pair
	long cladeA, cladeB, mrca;
	NumericMatrix distances(Nfocals,Nfocals);
	for(long i=0; i<Nfocals; ++i){
		cladeA = focal_clades[i];
		distances(i,i) = 0;
		for(long j=i+1; j<Nfocals; ++j){
			cladeB = focal_clades[j];
			// step 1: determine most recent common ancestor
			// check for trivial case
			if(cladeA==cladeB){
				mrca = cladeA;
			}else{
				// follow ancestry of both clades in synchrony, until they diverge
				// note that the first ancestor of every clade will be the root
				long a,b;
				for(a=clade2first_ancestor[cladeA], b=clade2first_ancestor[cladeB]; (a<=clade2last_ancestor[cladeA]) && (b<=clade2last_ancestor[cladeB]); ++a, ++b){
					if(ancestors[a]!=ancestors[b]) break;
					else mrca = ancestors[a];
				}
				// check special case where one clade is descendant of the other (this would lead to a "premature" stop of the above loop)
				if((a<=clade2last_ancestor[cladeA]) && (ancestors[a]==cladeB)){
					mrca = cladeB;
				}else if((b<=clade2last_ancestor[cladeB]) && (ancestors[b]==cladeA)){
					mrca = cladeA;
				}
			}
			// step 2: calculate distance
			distances(i,j) = distance_from_root_per_clade[cladeA] + distance_from_root_per_clade[cladeB] - 2*distance_from_root_per_clade[mrca];
			distances(j,i) = distances(i,j);
		}
		if((i%100)==0) Rcpp::checkUserInterrupt();
	}
	
	return(distances);
}
												


// Calculate phylogenetic distance for pairs of clades (cladesA[] vs cladesB[])
// Distance = cumulative branch length of both clades back to their most recent common ancestor (aka "patristic distance")
// There's some initial overhead involved with this function, but for large number of clade pairs this becomes more efficient
// Time complexity is O(Ntips + Npairs*log(Ntips)).
// Returns a NumericVector of size Npairs, with each entry being the distance between the two clades
// This function is slightly different from get_distance_matrix_between_clades_CPP(), in that here only distances between specific clade pairs are returned.
// Requirements:
//   The input tree must be rooted (root will be determined automatically, as the node that has no incoming edge)
//   The input tree can be multifurcating and/or monofurcating
// Attention: 0-based indexing is used for input and output variables, so make sure to shift indices in R before and after calling this function

// [[Rcpp::export]]
NumericVector get_distances_between_clades_CPP(	const long 					Ntips,
												const long 					Nnodes,
												const long 					Nedges,
												const std::vector<long> 	&tree_edge,			// 2D array of size Nedges x 2 in row-major format
												const std::vector<double> 	&edge_length, 		// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
												const std::vector<long> 	&cladesA,			// 1D array of size Npairs, containing values in 0:(Nclades-1)
												const std::vector<long>		&cladesB,			// 1D array of size Npairs, containing values in 0:(Nclades-1)
												bool 						verbose,
												const std::string 			&verbose_prefix){
	const long Npairs = cladesA.size();
	const long Nclades = Ntips + Nnodes;
	long parent, clade;

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);
															
	// calculate number of ancestors and distance from root for each clade
	// (traverse root --> tips)
	std::vector<long> ancestor_count_per_clade(Nclades);
	std::vector<double> distance_from_root_per_clade(Nclades);
	ancestor_count_per_clade[root] 		= 0;
	distance_from_root_per_clade[root] 	= 0;
	for(long q=0; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		ancestor_count_per_clade[clade] 	= 1 + ancestor_count_per_clade[parent];
		distance_from_root_per_clade[clade]	=  (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]) + distance_from_root_per_clade[parent];
	}
	const long total_ancestor_count = vector_sum(ancestor_count_per_clade);
	
	// calculate ancestry for each clade in a long list ancestors[]
	// (traverse root --> tips)
	std::vector<long> clade2first_ancestor(Nclades); // for each clade c, ancestors[clade2first_ancestor[c]..clade2last_ancestor[c]] will be the list of ancestor clades leading to the clade c
	std::vector<long> clade2last_ancestor(Nclades);
	clade2first_ancestor[0] = 0;
	clade2last_ancestor[0] = clade2first_ancestor[0] + ancestor_count_per_clade[0] - 1;
	for(long c=1; c<Nclades; ++c){
		clade2first_ancestor[c] = clade2last_ancestor[c-1] + 1;
		clade2last_ancestor[c]  = clade2first_ancestor[c] + ancestor_count_per_clade[c] - 1;
	}
	std::vector<long> ancestors(total_ancestor_count);
	for(long q=0; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		// step 1: copy the parent's ancestry to the child's ancestry
		for(long a=clade2first_ancestor[parent]; a<=clade2last_ancestor[parent]; ++a){
			ancestors[clade2first_ancestor[clade]+(a-clade2first_ancestor[parent])] = ancestors[a];
		}
		// step 2: append the parent to the clade's ancestry
		ancestors[clade2last_ancestor[clade]] = parent;
	}
	
	// calculate most-recent-common-ancestor for each clade pair
	std::vector<long> mrca_per_pair(Npairs);
	long cladeA, cladeB;
	for(long p=0; p<Npairs; ++p){
		cladeA = cladesA[p];
		cladeB = cladesB[p];
		// check for trivial case
		if(cladeA==cladeB){
			mrca_per_pair[p] = cladeA;
			continue;
		}
		// follow ancestry of both clades in synchrony, until they diverge
		// note that the first ancestor of every clade will be the root
		long a, b, mrca=-1;
		for(a=clade2first_ancestor[cladeA], b=clade2first_ancestor[cladeB]; (a<=clade2last_ancestor[cladeA]) && (b<=clade2last_ancestor[cladeB]); ++a, ++b){
			if(ancestors[a]!=ancestors[b]) break;
			else mrca = ancestors[a];
		}
		// check special case where one clade is descendant of the other (this would lead to a "premature" stop of the above loop)
		if((a<=clade2last_ancestor[cladeA]) && (ancestors[a]==cladeB)){
			mrca = cladeB;
		}else if((b<=clade2last_ancestor[cladeB]) && (ancestors[b]==cladeA)){
			mrca = cladeA;
		}
		mrca_per_pair[p] = mrca;
		if((p%10000)==0) Rcpp::checkUserInterrupt();
	}
	
	// calculate pairwise distances
	dvector distances(Npairs);
	for(long p=0; p<Npairs; ++p){
		distances[p] = distance_from_root_per_clade[cladesA[p]] + distance_from_root_per_clade[cladesB[p]] - 2*distance_from_root_per_clade[mrca_per_pair[p]];
		if((p%10000)==0) Rcpp::checkUserInterrupt();
	}
	return Rcpp::wrap(distances);
}



// Given a phylogenetic tree, find the two tips that are furthest apart from each other
// [[Rcpp::export]]
Rcpp::List get_farthest_tip_pair_CPP(	const long 					Ntips,
										const long 					Nnodes,
										const long 					Nedges,
										const std::vector<long>		&tree_edge,			// (INPUT) 2D array of size Nedges x 2, flattened in row-major format
										const std::vector<double> 	&edge_length){ 		// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
	const long Nclades = Ntips + Nnodes;
	const bool has_edge_lengths = (edge_length.size()>0);

	// determine parent clade for each clade
	lvector clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);	

	// get incoming edge for each clade
	lvector clade2incoming_edge;
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2incoming_edge);

	// get tree traversal (root-->tips), including tips
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);
	
	// determine farthest tip descending from each node, traversing tips-->root
	// also simultaneously keep track of furthest tip pair found
	lvector clade2farthest_tip(Nclades,-1);
	dvector clade2max_tip_distance(Nclades,0);
	long farthest_tip1=0, farthest_tip2=0;
	double max_tip_distance=0;
	for(long tip=0; tip<Ntips; ++tip){
		clade2farthest_tip[tip] = tip;
	}
	double candidate_distance;
	for(long q=traversal.queue.size()-1, clade, parent, edge; q>0; --q){
		clade  	= traversal.queue[q];
		parent	= clade2parent[clade];
		edge	= clade2incoming_edge[clade];
		candidate_distance = (has_edge_lengths ? edge_length[edge] : 1.0) + clade2max_tip_distance[clade];
		if(clade2farthest_tip[parent]<0){
			// no other child recorded its data for this parent yet
			clade2max_tip_distance[parent] = candidate_distance;
			clade2farthest_tip[parent] = clade2farthest_tip[clade];
		}else{
			// some other child of this parent recorded its data already, so check if it should be replaced, and also check if we can use that previously recorded data to define a more distant tip pair than found so far
			if(clade2max_tip_distance[parent]+candidate_distance>=max_tip_distance){
				// found a more distant tip pair
				max_tip_distance = clade2max_tip_distance[parent]+candidate_distance;
				farthest_tip1	 = clade2farthest_tip[parent];
				farthest_tip2	 = clade2farthest_tip[clade];
			}
			if(candidate_distance>=clade2max_tip_distance[parent]){
				clade2max_tip_distance[parent] = candidate_distance;
				clade2farthest_tip[parent] = clade2farthest_tip[clade];
			}
		}
	}

	return Rcpp::List::create(	Rcpp::Named("farthest_tip1") 	= farthest_tip1,
								Rcpp::Named("farthest_tip2") 	= farthest_tip2,
								Rcpp::Named("max_tip_distance")	= max_tip_distance);
}


// Given a state for each clade (tip or node), and a list of clade pairs (A[], B[]), count the number of state transitions along the shortest path connecting every clade-pair
// [[Rcpp::export]]
Rcpp::List count_transitions_between_clades_CPP(const long 				Ntips,
												const long 				Nnodes,
												const long 				Nedges,
												const std::vector<long>	&tree_edge,			// (INPUT) 2D array of size Nedges x 2, flattened in row-major format
												const std::vector<long>	&clade_states, 		// (INPUT) 1D array of size Nclades, specifying the state at each tip and node
												const std::vector<long>	&cladesA,			// (INPUT) 1D array of size NP, with values in 0,..,Nclades-1, specifying the first clade of each clade pair
												const std::vector<long>	&cladesB){			// (INPUT) 1D array of size NP, with values in 0,..,Nclades-1, specifying the second clade of each clade pair
	const long Nclades = Ntips + Nnodes;
	const long Npairs  = cladesA.size();
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// determine number of transitions from each clade all the way back to the root
	lvector pair2Ntransitions(Npairs,0L);
	lvector clade2Ntransitions(Nclades,-1L);
	for(long p=0, cladeA, cladeB, parent; p<Npairs; ++p){
		cladeA = cladesA[p];
		cladeB = cladesB[p];
		if(cladeA==cladeB) continue; // skip this trivial pair
		clade2Ntransitions[cladeA] = 0;
		// follow cladeA all the way to the root
		parent = clade2parent[cladeA];
		while(parent>=0){
			clade2Ntransitions[parent] = clade2Ntransitions[cladeA] + (clade_states[parent]==clade_states[cladeA] ? 0 : 1);
			cladeA = parent;
			parent = clade2parent[cladeA];
		}
		// follow cladeB all the way to the root, until meeting the rootward path taken from cladeA
		if(clade2Ntransitions[cladeB]>=0){
			pair2Ntransitions[p] = clade2Ntransitions[cladeB];
		}else{
			double Nransitions = 0;
			parent = clade2parent[cladeB];
			while(parent>=0){
				if(clade2Ntransitions[parent]>=0){
					// this parent was already encountered on the path from cladeA to the root
					pair2Ntransitions[p] = clade2Ntransitions[parent] + Nransitions + (clade_states[parent]==clade_states[cladeB] ? 0 : 1);
					break;
				}else{
					Nransitions += (clade_states[parent]==clade_states[cladeB] ? 0 : 1);
					cladeB = parent;
					parent = clade2parent[cladeB];
				}
			}
		}
		// reset the entries in clade2Ntransitions[]
		cladeA = cladesA[p];
		while(cladeA>=0){
			clade2Ntransitions[cladeA] = -1;
			cladeA = clade2parent[cladeA];
		}
		if((p%100)==0) Rcpp::checkUserInterrupt();
	}
	return Rcpp::List::create(Rcpp::Named("pair2Ntransitions") = Rcpp::wrap(pair2Ntransitions));
}



// Count the number of extant lineages in a timetree at various time points, with time points chosen on a regular grid
// The tree need not be ultrametric, although in general this function only makes sense for ultrametric trees (e.g. where edge lengths are time intervals)
// [[Rcpp::export]]
Rcpp::List count_clades_at_regular_times_CPP(	const long 			Ntips,
												const long 			Nnodes,
												const long 			Nedges,
												const std::vector<long>		&tree_edge,		// (INPUT) 2D array of size Nedges x 2, flattened in row-major format
												const std::vector<double>	&edge_length,	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
												const long			Ntimes,				// (INPUT) number of time points
												double				min_time,			// (INPUT) minimum time (distance from root) to consider. If negative, will be set to the minimum possible.
												double				max_time,			// (INPUT) maximum time (distance from root) to consider. If Infinite, will be set to the maximum possible.
												const bool			ultrametric,		// whether the input tree is guaranteed to be ultrametric (even in the presence of numerical inaccuracies)
												const long			degree,				// (INPUT) degree n of the LTT curve: LTT_n(t) will be the number of lineages in the tree at time t that have at least n descending tips in the tree. Typically degree=1, which corresponds to the classical LTT curve.
												const bool			include_slopes){	// (INPUT) if true, slopes of the clades_per_time_point curve are also returned	
	const long Nclades = Ntips + Nnodes;
	const bool got_edge_lengths = (edge_length.size()>0);
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);	

	// get tree traversal (root-->tips), including tips
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);

	// calculate distances from root (traverse root-->tips)
	std::vector<double> clade_times(Nclades);
	clade_times[root] = 0;
	for(long q=0, clade; q<traversal.queue.size(); ++q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue;
		for(long e=traversal.node2first_edge[clade-Ntips], edge; e<=traversal.node2last_edge[clade-Ntips]; ++e){
			edge  = traversal.edge_mapping[e];
			clade_times[tree_edge[2*edge+1]] = clade_times[clade] + (got_edge_lengths ? edge_length[edge] : 1.0);
		}
	}
	const double max_clade_time = vector_max(clade_times);

	// calculate the number of tips descending from each clade (traverse tips-->root)
	// only do this calculation if degree>1
	lvector clade2tips;
	if(degree>1){
		clade2tips.assign(Nclades,0);
		for(long tip=0; tip<Ntips; ++tip){ clade2tips[tip] = 1; }
		for(long q=traversal.queue.size()-1, clade; q>=0; --q){
			clade = traversal.queue[q];
			if(clade==root) continue;
			clade2tips[clade2parent[clade]] += clade2tips[clade];
		}
	}
	
	// determine distance bins
	if(ultrametric){
		// assume that the oldest tip is still at "present day", i.e., any non-zero tip age is solely due to numerical inaccuracies
		max_time = min(max_time, array_min(clade_times,0,Ntips-1));
	}else{
		max_time = min(max_time, array_max(clade_times));
	}
	min_time = max(0.0, min_time);
	const double time_step = (max_time-min_time)/(Ntimes-1);
	std::vector<double> time_points(Ntimes);
	for(long t=0; t<Ntimes; ++t){
		time_points[t] = min_time + time_step*t;
	}
	time_points[Ntimes-1] = max_time; // avoid numerical rounding errors
	const double time_epsilon = (log(Ntips)/log(2))*max_clade_time*1e-8; // "typical" rounding error when calculating distances of tips from the root
	
	// calculate number of lineages within each time point
	std::vector<long> lineages(Ntimes,0);
	for(long edge=0, child, parent; edge<Nedges; ++edge){
		parent = tree_edge[edge*2+0];
		child  = tree_edge[edge*2+1];
		if((degree>1) && (clade2tips[child]<degree)) continue; // number of lineages descending from this lineage is below the threshold, so skip this edge
		// determine time points covered by this edge
		const long last_time_point = ((clade_times[child]>max_time-(child<Ntips ? time_epsilon : 0.0)) ? Ntimes-1 : min(Ntimes-1,long(floor((clade_times[child]-min_time)/time_step))));
		if(last_time_point<0) continue; // edge is outside of considered time span
		const long first_time_point = (parent<0 ? last_time_point : max(0L,long(ceil((clade_times[parent]-min_time)/time_step))));
		if(first_time_point>Ntimes-1) continue; // edge is outside of considered time span
		// count edge towards lineages in the covered time points
		if(first_time_point==last_time_point){ 
			++lineages[first_time_point];
		}else{
			for(long t=first_time_point; t<=last_time_point; ++t) ++lineages[t];
		}
	}
		
	// calculate slopes (symmetric difference coefficient)
	// use one-sided differences at the edges
	std::vector<double> slopes(include_slopes ? Ntimes : 0);
	std::vector<double> relative_slopes(include_slopes ? Ntimes : 0);
	if(include_slopes && (Ntimes>=2)){
		for(long t=0, left, right; t<Ntimes; ++t){
			left  				= max(t-1,0L);
			right 				= min(t+1,Ntimes-1);
			const double dt		= (time_points[right]-time_points[left]);
			slopes[t] 			= (lineages[right]-lineages[left])/dt;
			const double CC		= ((left<t && t<right) ? (lineages[left]+lineages[t]+lineages[right])/3.0 : (lineages[left]+lineages[right])/2.0);
			//relative_slopes[t] 	= (CC==0 ? NAN_D : slopes[t]/CC);
			relative_slopes[t] 	= ((lineages[right]==0) || (lineages[left]==0) ?  slopes[t]/CC : (log(lineages[right])-log(lineages[left]))/dt);
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("time_points") 		= Rcpp::wrap(time_points),
								Rcpp::Named("lineages") 		= Rcpp::wrap(lineages),
								Rcpp::Named("slopes") 			= Rcpp::wrap(slopes),
								Rcpp::Named("relative_slopes") 	= Rcpp::wrap(relative_slopes));
}



// Count number of extant lineages in a timetree, at arbitrary time points
// [[Rcpp::export]]
IntegerVector count_clades_at_times_CPP(const long					Ntips,
										const long 					Nnodes,
										const long					Nedges,
										const std::vector<long> 	&tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
										const std::vector<double>	&edge_length,		// (INPUT) 1D array of size Nedges, or empty
										const std::vector<double>	&times,				// (INPUT) 1D array of size Ntimes
										const long					degree){			// (INPUT) degree n of the LTT curve: LTT_n(t) will be the number of lineages in the tree at time t that have at least n descending tips in the tree. Typically degree=1, which corresponds to the classical LTT curve.
	const long Nclades = Ntips + Nnodes;
	const bool got_edge_lengths = (edge_length.size()>0);
	if(times.empty()) return(Rcpp::wrap(std::vector<long>()));

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);	
	
	// get tree traversal (root-->tips), including tips
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);
	
	// calculate distances from root (traverse root-->tips)
	std::vector<double> distances_from_root(Nclades);
	distances_from_root[root] = 0;
	for(long q=0, clade; q<traversal.queue.size(); ++q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue;
		for(long e=traversal.node2first_edge[clade-Ntips], edge; e<=traversal.node2last_edge[clade-Ntips]; ++e){
			edge  = traversal.edge_mapping[e];
			distances_from_root[tree_edge[2*edge+1]] = distances_from_root[clade] + (got_edge_lengths ? edge_length[edge] : 1.0);
		}
	}
	const double max_distance_from_root = vector_max(distances_from_root);
	
	// calculate the number of tips descending from each clade (traverse tips-->root)
	// only do this calculation if degree>1
	lvector clade2tips;
	if(degree>1){
		clade2tips.assign(Nclades,0);
		for(long tip=0; tip<Ntips; ++tip){ clade2tips[tip] = 1; }
		for(long q=traversal.queue.size()-1, clade; q>=0; --q){
			clade = traversal.queue[q];
			if(clade==root) continue;
			clade2tips[clade2parent[clade]] += clade2tips[clade];
		}
	}
	
	const long Ntimes = times.size();
	double epsilon = (log(Ntips)/log(2.0)) * max_distance_from_root * 1e-10; // "typical" rounding error when calculating distances from the root at the tips
	std::vector<long> lineages(Ntimes,0);
	for(long t=0; t<Ntimes; ++t){
		if(times[t]==0){
			lineages[t] = ((degree<=1) || (clade2tips[root]>=degree) ? 1l : 0l); // at most one lineage at root
			continue;
		}
		for(long clade=0; clade<Nclades; ++clade){
			if(clade==root) continue;
			if((distances_from_root[clade]>=times[t]-(clade<Ntips ? epsilon : 0.0)) && (distances_from_root[clade2parent[clade]]<=times[t]) && ((degree<=1) || (clade2tips[clade]>=degree))){
				lineages[t] += 1;
			}
		}
	}
	return(Rcpp::wrap(lineages));
}


// Count number of extant lineages at arbitrary time points, based on birth and death events
// [[Rcpp::export]]
NumericVector get_LTT_BD_CPP(	const std::vector<double>	&birth_times,	// (INPUT) 1D array (arbitrary length) listing birth times in ascending order
								const std::vector<double>	&death_times,	// (INPUT) 1D array (arbitrary length) listing death times in ascending order
								const std::vector<double>	&time_grid){	// (INPUT) 1D array of size NG, listing requested times in ascending order
	const long NG = time_grid.size();
	const long NB = birth_times.size();
	const long ND = death_times.size();
	
	// count birth & death events towards their next-youngest time-grid intervals
	dvector lineages(NG,0L);
	for(long b=0, g=-1; b<NB; ++b){
		g = find_next_right_grid_point(time_grid, birth_times[b], g);
		if(g<0) break; // all remaining birth events are beyond the scope of the time_grid
		lineages[g] += 1;
	}
	for(long d=0, g=-1; d<ND; ++d){
		g = find_next_right_grid_point(time_grid, death_times[d], g);
		if(g<0) break; // all remaining death events are beyond the scope of the time_grid
		lineages[g] -= 1;
	}
	
	// count cumulative effects of birth & death events on lineage counts over time
	for(long g=1; g<NG; ++g){
		lineages[g] += lineages[g-1];
	}

	return(Rcpp::wrap(lineages));
}



// returns true if the tree includes nodes that have more than 2 children
// [[Rcpp::export]]
bool tree_has_multifurcations_CPP(	const long				Ntips,
									const long 				Nnodes,
									const long				Nedges,
									const std::vector<long>	&tree_edge){	// (INPUT) 2D array (in row-major format) of size Nedges x 2
	std::vector<long> child_count_per_node(Nnodes,0);
	for(long edge=0; edge<Nedges; ++edge) ++child_count_per_node[tree_edge[2*edge+0]-Ntips];
	for(long node=0; node<Nnodes; ++node){
		if(child_count_per_node[node]>2) return true;
	}
	return false;
}


template<class ARRAY_TYPE>
void count_monofurcations_and_multifurcations(	const long			Ntips,
												const long 			Nnodes,
												const long			Nedges,
												const ARRAY_TYPE	&tree_edge,				// (INPUT) 2D array (in row-major format) of size Nedges x 2
												long				&Nmonofurcations,		// (OUTPUT) number of monofurcating nodes
												long				&Nbifurcations,			// (OUTPUT) number of bifurcating nodes
												long				&Nmultifurcations){		// (OUTPUT) number of multifurcating nodes
	std::vector<long> child_count_per_node(Nnodes,0);
	for(long edge=0; edge<Nedges; ++edge) ++child_count_per_node[tree_edge[2*edge+0]-Ntips];
	Nmonofurcations = Nbifurcations = Nmultifurcations = 0;
	for(long node=0; node<Nnodes; ++node){
		if(child_count_per_node[node]==1) ++Nmonofurcations;
		else if(child_count_per_node[node]==2) ++Nbifurcations;
		else ++Nmultifurcations;
	}
}




// Extract speciation events and extinction events (with associated waiting times) from tree
//   Speciation event = non-root node
//   Extinction event = non-crown tip, i.e. tip that is not at maximum distance from the root
// The returned info may be used for fitting birth-death tree-generation models
// [[Rcpp::export]]
Rcpp::List get_speciation_extinction_events_CPP(const long					Ntips,
												const long 					Nnodes,
												const long					Nedges,
												const std::vector<long>		&tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
												const std::vector<double>	&edge_length,		// (INPUT) 1D array of size Nedges, or empty
												const double				min_age,			// (INPUT) min phylogenetic distance from the tree crown, to be considered. If <=0, this constraint is ignored.
												const double				max_age,			// (INPUT) max phylogenetic distance from the tree crown, to be considered. If <=0, this constraint is ignored.
												const std::vector<long>		&only_clades,		// (INPUT) optional list of clade indices to consider. Can also be empty, in which case no filtering is done.
												const std::vector<long>		&omit_clades){		// (INPUT) optional list of clade indices to omit. Can also be empty.
	const long Nclades = Ntips + Nnodes;
	long clade, parent;
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get incoming edge for each clade
	std::vector<long> incoming_edge_per_clade;
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);

	// get tree traversal route (root --> tips)											
	std::vector<long> queue_root2tips, node2first_edge, node2last_edge, edge_mapping;
	get_tree_traversal_root_to_tips(Ntips,
									Nnodes,
									Nedges,
									root,
									tree_edge,
									true,
									false,
									queue_root2tips,
									node2first_edge,
									node2last_edge,
									edge_mapping,
									false,
									"");
								
	// calculate distance from root for each clade (traverse root --> tips)
	std::vector<double> distances_from_root(Nclades);
	distances_from_root[root] = 0;
	for(long q=0; q<queue_root2tips.size(); ++q){
		clade = queue_root2tips[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		distances_from_root[clade] = (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]) + distances_from_root[parent];
	}
	
	// sort clades in chronological order (increasing distance from root)
	std::vector<long> chronological_clade_order(Nclades);
	qsortIndices(distances_from_root, chronological_clade_order);
	const double max_distance_from_root = distances_from_root[chronological_clade_order.back()];
	
	// count number of extant species at the time of each clade
	std::vector<long> diversity_at_clade(Nclades);
	long current_diversity = 1;
	for(long c=0; c<Nclades; ++c){
		clade = chronological_clade_order[c];
		diversity_at_clade[clade] = current_diversity;
		if(clade<Ntips) --current_diversity; // a tip marks an extinction event
		else current_diversity += node2last_edge[clade-Ntips] - node2first_edge[clade-Ntips]; // a node marks a speciation event, generating N-1 new species (where N is the number of children)
	}
	
	// figure out which clades to consider
	std::vector<bool> include_clade(Nclades,(only_clades.size()==0));
	for(long c=0; c<only_clades.size(); ++c) include_clade[only_clades[c]] = true;
	for(long c=0; c<omit_clades.size(); ++c) include_clade[omit_clades[c]] = false;
	if(max_age>0){
		// only consider clades above a certain distance from the root
		for(long c=0; c<Nclades; ++c){
			clade = chronological_clade_order[c];
			include_clade[clade] = include_clade[clade] && ((max_distance_from_root-distances_from_root[clade])<=max_age);
		}
	}
	if(min_age>0){
		// only consider clades below a certain distance from the root
		for(long c=0; c<Nclades; ++c){
			clade = chronological_clade_order[c];
			include_clade[clade] = include_clade[clade] && ((max_distance_from_root-distances_from_root[clade])>=min_age);
		}
	}
	
	// figure out number of speciation events
	long Nspeciations = 0;
	long Npoints = 0;
	for(long c=0; c<Nclades; ++c){
		clade = chronological_clade_order[c];
		if(include_clade[clade]) ++Npoints;
		if((clade>=Ntips) && include_clade[clade]) ++Nspeciations;
	}
	if(Nspeciations>0) --Nspeciations;  // the first event should be discarded, because it has unknown waiting time
	
	// extract speciation events
	// speciation_waiting_times[event] = waiting time to speciation event (counted from previous considered speciation event)
	// speciation_diversities[event] = number of extant clades during the speciation event
	std::vector<double> speciation_waiting_times(Nspeciations);
	std::vector<long> 	speciation_diversities(Nspeciations);
	std::vector<long> 	speciation_clades(Nspeciations);
	std::vector<double> speciation_times(Nspeciations);
	double previous_time = -1; // negative means undefined
	for(long c=0, event=0; c<Nclades; ++c){
		clade = chronological_clade_order[c];
		if(!include_clade[clade]) continue; // omit this clade
		if(clade<Ntips) continue; // not a speciation event
		if(previous_time<0){
			// this is the first speciation event encountered, so just record its time but don't include in returned events
			previous_time = distances_from_root[clade];
			continue;
		}
		speciation_diversities[event]	= diversity_at_clade[clade];
		speciation_clades[event]		= clade;
		speciation_times[event]			= distances_from_root[clade];
		speciation_waiting_times[event]	= speciation_times[event] - previous_time;
		previous_time 					= speciation_times[event];
		++event;
	}
	
	// extract extinction events (non-crown tips mark extinctions)
	std::vector<long> extinction_tips;
	previous_time = -1;
	for(long c=0; c<Nclades; ++c){
		clade = chronological_clade_order[c];
		if((clade>=Ntips) || (distances_from_root[clade]>=(1.0-RELATIVE_EPSILON)*max_distance_from_root)) continue; // non-crown tips correspond to extinction events
		if(previous_time<0){
			// this is the first extinction event encountered, so just record its time but don't include in returned events
			previous_time = distances_from_root[clade];
			continue;
		}
		extinction_tips.push_back(clade);
	}
	const long Nextinctions = extinction_tips.size();
	std::vector<double> extinction_waiting_times(Nextinctions);
	std::vector<long> 	extinction_diversities(Nextinctions);
	std::vector<double> extinction_times(Nextinctions);
	for(long event=0; event<Nextinctions; ++event){
		clade 							= extinction_tips[event];
		extinction_diversities[event]	= diversity_at_clade[clade];
		extinction_times[event]			= distances_from_root[clade];
		extinction_waiting_times[event]	= extinction_times[event] - previous_time;
		previous_time 					= extinction_times[event];
	}
	
	// all considered points (tips & nodes) in chronological order
	std::vector<double> times(Npoints);
	std::vector<long>   diversities(Npoints), clades(Npoints);
	for(long c=0, point=0; c<Nclades; ++c){
		clade = chronological_clade_order[c];
		if(!include_clade[clade]) continue;
		times[point] 		= distances_from_root[clade];
		diversities[point]	= diversity_at_clade[clade];
		clades[point]		= clade;
		++point;
	}
		
	return Rcpp::List::create(	Rcpp::Named("Nspeciations")				= Nspeciations,							// number of speciation events included
								Rcpp::Named("speciation_waiting_times")	= Rcpp::wrap(speciation_waiting_times),	// waiting time until each speciation event (distance from the previous speciation event)
								Rcpp::Named("speciation_diversities") 	= Rcpp::wrap(speciation_diversities), 	// number of clades just prior to the speciation
								Rcpp::Named("speciation_times")			= Rcpp::wrap(speciation_times),			// time at the speciation event
								Rcpp::Named("speciation_clades")		= Rcpp::wrap(speciation_clades),		// clade marking each speciation event
								Rcpp::Named("Nextinctions")				= Nextinctions,							// number of extinction events included
								Rcpp::Named("extinction_waiting_times")	= Rcpp::wrap(extinction_waiting_times),	// waiting time until each extinction event (distance from the previous extinction event)
								Rcpp::Named("extinction_diversities") 	= Rcpp::wrap(extinction_diversities), 	// number of clades just prior to the extinction
								Rcpp::Named("extinction_times")			= Rcpp::wrap(extinction_times),			// time at the extinction event
								Rcpp::Named("extinction_tips")			= Rcpp::wrap(extinction_tips),			// tip marking each extinction event
								Rcpp::Named("max_distance_from_root")	= max_distance_from_root,				// maximum distance of any tip to the root. Note that this may be outside of the span of times[]
								Rcpp::Named("times")					= Rcpp::wrap(times),					// time at each clade, in chronological order (root is first)
								Rcpp::Named("clades")					= Rcpp::wrap(clades),					// clade associated with each time point
								Rcpp::Named("diversities")				= Rcpp::wrap(diversities));				// number of clades just prior to the splitting or extinction of each clade, in chronological order (root is first)
}




// Calculate relative evolutionary divergences (RED) of nodes, similarly to the PhyloRank v0.0.27 package [Parks et al. 2018]
// The RED of a node is a measure of its relative placement between the root and its descending tips
// Hence, the RED is always between 0 and 1, with the root having an RED of 0 and all tips having an RED of 1. REDs for tips are not returned here, since they will always be 1.
// Requirements:
//   The tree must be rooted; the root should be the unique node with no parent
//   The tree can include multifurcations as well as monofurcations
void get_relative_evolutionary_divergences(	const long				Ntips,
											const long 				Nnodes,
											const long				Nedges,
											const std::vector<long>		&tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
											const std::vector<double>		&edge_length,		// (INPUT) 1D array of size Nedges, or empty in which case each edge is interpreted as having length 1)
											std::vector<double>		&REDs){				// (OUTPUT) 1D array of size Nnodes, listing the RED for each node	
	const bool unit_edge_lengths = (edge_length.size()==0);

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get incoming edge for each clade
	std::vector<long> incoming_edge_per_clade;
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);

	// get tree traversal route (root --> tips), including tips										
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);
									
	// get mean distance of each node to its descending tips (traversing tips-->root, exclude root)
	std::vector<long> node2tip_count(Nnodes,0);
	std::vector<double> node2tip_depth(Nnodes,0);
	for(long q=traversal.queue.size()-1, clade, pnode, cnode; q>=1; --q){
		clade	= traversal.queue[q];
		cnode	= clade-Ntips;
		pnode 	= clade2parent[clade] - Ntips;
		node2tip_count[pnode] += (clade<Ntips ? 1 : node2tip_count[cnode]);
		node2tip_depth[pnode] += (clade<Ntips ? 0 : node2tip_depth[cnode]) + (clade<Ntips ? 1 : node2tip_count[cnode]) * (unit_edge_lengths ? 1 : edge_length[incoming_edge_per_clade[clade]]);
	}
	for(long node=0; node<Nnodes; ++node){
		node2tip_depth[node] /= node2tip_count[node];
	}
		
	// calculate RED for each node (traverse root --> tips)
	REDs.resize(Nnodes);
	REDs[root-Ntips] = 0;
	for(long q=1, clade, pnode, cnode; q<traversal.queue.size(); ++q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue; // skip tips
		pnode 	= clade2parent[clade] - Ntips;
		cnode	= clade - Ntips;
		const double mean_distance_to_tips  = node2tip_depth[cnode];
		const double incoming_edge_length 	= (unit_edge_lengths ? 1.0 : edge_length[incoming_edge_per_clade[clade]]);
		if((mean_distance_to_tips + incoming_edge_length)==0){
			REDs[cnode] = REDs[pnode];
		}else{
			REDs[cnode] = min(1.0, REDs[pnode] + (incoming_edge_length/(incoming_edge_length+mean_distance_to_tips)) * (1.0-REDs[pnode]));
		}
	}
}


// Rcpp wrapper for the homonymous base function
// Returns relative evolutionary divergence (RED) values for each node in the tree [Parks et al. 2018]
// The tree must be rooted, but may include monofurcations and multifurcations
// [[Rcpp::export]]
NumericVector get_relative_evolutionary_divergences_CPP(const long					Ntips,
														const long 					Nnodes,
														const long					Nedges,
														const std::vector<long>		&tree_edge,		// (INPUT) 2D array (in row-major format) of size Nedges x 2
														const std::vector<double>	&edge_length){	// (INPUT) 1D array of size Nedges, or empty in which case each edge is interpreted as having length 1)
	std::vector<double> REDs;
	get_relative_evolutionary_divergences(Ntips, Nnodes, Nedges, tree_edge, edge_length, REDs);
	
	return Rcpp::wrap(REDs);
}


// Date (make ultrametric) a phylogenetic tree based on relative evolutionary divergences (RED)
// The RED of a node measures its relative placement between the root and its descending tips.
// For each edge, the RED difference between child & parent is used to set the new length of that edge (times some scaling factor to reproduce the anchor age).
// The tree must be rooted, but may include monofurcations and multifurcations
// [[Rcpp::export]]
Rcpp::List date_tree_via_RED_CPP(	const long				Ntips,
									const long 				Nnodes,
									const long				Nedges,
									const std::vector<long>		&tree_edge,		// (INPUT) 2D array (in row-major format) of size Nedges x 2
									const std::vector<double>		&edge_length,	// (INPUT) 1D array of size Nedges, or empty in which case each edge is interpreted as having length 1)
									const long				anchor_node,	// (INPUT) Index of node to be used as anchor, an integer between 0,..,Nnodes-1. Can also be negative, in which case the root is used as anchor.
									const long				anchor_age){	// (INPUT) Age of the anchor node, a positive real number. If anchor_node<0, then this specifies the age of the root.
	// get node REDs
	std::vector<double> node_REDs;
	get_relative_evolutionary_divergences(Ntips, Nnodes, Nedges, tree_edge, edge_length, node_REDs);
	
	// find scaling factor (time_units/RED_units)
	const double anchor_RED = (anchor_node<0 ? 0 : node_REDs[anchor_node]);
	if(anchor_RED==1) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Anchor is essentially a tip (its relative evolutionary divergence is 1).");

	const double scaling = anchor_age/(1-anchor_RED);
	std::vector<double> edge_times(Nedges);
	for(long edge=0, parent, child; edge<Nedges; ++edge){
		parent = tree_edge[2*edge+0];
		child  = tree_edge[2*edge+1];
		edge_times[edge] = scaling*max(0.0,((child<Ntips ? 1.0 : node_REDs[child-Ntips]) - node_REDs[parent-Ntips]));
	}	

	return Rcpp::List::create(	Rcpp::Named("edge_times") 	= edge_times, 	// the new edge lengths in time units, such that the 
								Rcpp::Named("node_REDs") 	= node_REDs,
								Rcpp::Named("success") 		= true);
}




// Given a phylogenetic tree in standard representation, calculate an alternative representation of the tree structure by enumerating clades and their properties
// Main output:
//   clades[]: 2D matrix of size Nclades x (1+Nsplits), in row-major format, with the following columns:
//     Column 1: Parent clade index
//     Columns 2-(Nsplits+1): Child clade indices, where Nsplits is the maximum number of splits at a node (e.g. Nsplits=2 for bifurcating trees)
//   lengths[]: 1D array of size Nclades, listing incoming edge lengths for each clade (will be negative for the root)
//   Rows in clades[] and entries in length[] can be ordered according to clade indices, or according to post-order traversal.
// Negative values indicate missing values (e.g., no parent for the root, no children for tips)
// This function is loosely analogous (but not identical) to the function phybase::read.tree.nodes
// Requirements:
//    The tree can include monofurcations and multifurcations
//    The tree must be rooted if postorder==true
// [[Rcpp::export]]
Rcpp::List tree_to_clade_list_CPP( 	const long					Ntips,
									const long 					Nnodes,
									const long					Nedges,
									const std::vector<long>		&tree_edge,		// (INPUT) 2D array (in row-major format) of size Nedges x 2
									const std::vector<double>	&edge_length,	// (INPUT) 1D array of size Nedges, or empty (in which case each edge is interpreted as having length 1)
									const bool					postorder){		// (INPUT) if true, nodes are reindexed in the returned lists in postorder traversal, i.e. so that the last node is the root. This is the convention used by phybase::read.tree.nodes. If false, nodes are indexed as in the original tree
	const long Nclades = Ntips + Nnodes;
	const bool no_edge_lengths = (edge_length.size()==0);		
	std::vector<long> old2new_clade(Nclades); // mapping from original clade indices to new clade indices in the returned lists
	if(postorder){
		// index tips as in the original tree, but re-index nodes into postorder traversal, i.e. so that the last listed node is the root
		const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge); // find root
		if(root<0) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error")="Tree is properly rooted");

		// get tree traversal route (root --> tips), excluding tips										
		//tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, false, false);
		std::vector<long> queue, node2first_edge, node2last_edge, edge_mapping;
		get_tree_traversal_depth_first_search(	Ntips,
												Nnodes,
												Nedges,
												root,
												tree_edge,
												false, // don't include tips
												false,
												queue,
												node2first_edge,
												node2last_edge,
												edge_mapping);
		
		// calculate re-indexing mapping for nodes
		// keep tip indices unchanged, i.e. only re-index nodes
		for(long tip=0; tip<Ntips; ++tip){
			old2new_clade[tip] = tip;
		}
		for(long q=0; q<Nnodes; ++q){
			old2new_clade[queue[q]] = Ntips + (Nnodes-1-q);
		}
				
	}else{
		// index all clades as in the original tree
		for(long clade=0; clade<Nclades; ++clade){
			old2new_clade[clade] = clade;
		}
	}
	
	// count the max number of children for any node
	std::vector<long> node2child_count(Nnodes,0);
	for(long e=0; e<Nedges; ++e){
		node2child_count[tree_edge[e*2+0] - Ntips] += 1;
	}
	const long Nsplits = vector_max(node2child_count);

	// populate clades[] and lengths[], with clades potentially reassigned indices according to old2new_clade
	std::vector<long> clades(Nclades*(Nsplits+1),-1), node2next_child(Nnodes,0);
	std::vector<double> lengths(Nclades,-1);
	for(long edge=0, parent, child; edge<Nedges; ++edge){
		parent = old2new_clade[tree_edge[edge*2+0]];
		child  = old2new_clade[tree_edge[edge*2+1]];
		lengths[child] 		= (no_edge_lengths ? 1.0 : edge_length[edge]);
		clades[child*3+0] 	= parent;
		clades[parent*(Nsplits+1)+1+node2next_child[parent-Ntips]] = child;
		++node2next_child[parent-Ntips];
	}

	return Rcpp::List::create(	Rcpp::Named("success")		 = true,
								Rcpp::Named("Nsplits")		 = Nsplits,
								Rcpp::Named("clades")		 = Rcpp::wrap(clades),
								Rcpp::Named("lengths") 		 = Rcpp::wrap(lengths),
								Rcpp::Named("old2new_clade") = Rcpp::wrap(old2new_clade));
}



// Extract binary constraints in FastTree alignment format from a phylogenetic tree
// Every internal bifurcating node with more than 2 descending tips will constitute an additional constraint
// Requirements:
//    The tree can include monofurcations and multifurcations
//    The tree must be rooted if postorder==true
// [[Rcpp::export]]
Rcpp::List extract_fasttree_constraints_CPP( const long				Ntips,
											const long 				Nnodes,
											const long				Nedges,
											const std::vector<long>		&tree_edge){		// (INPUT) 2D array (in row-major format) of size Nedges x 2
	const long Nclades = Ntips+Nnodes;
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// enumerate the children of each node
	std::vector<long> clade2child_index(Nclades,-1);
	std::vector<long> node2child_count(Nnodes,0);
	for(long clade=0, parent; clade<Nclades; ++clade){
		parent = clade2parent[clade];
		if(parent>=0){
			clade2child_index[clade] = node2child_count[parent-Ntips];
			++node2child_count[parent-Ntips];
		}
	}
	
	// get tree traversal route (root --> tips), including tips			
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);

	// calculate number of descending tips per node, traversing tips-->root (excluding the root)
	std::vector<long> node2tip_count(Nnodes,0);
	for(long q=traversal.queue.size()-1, clade; q>=1; --q){
		clade = traversal.queue[q];
		node2tip_count[clade2parent[clade]-Ntips] += (clade<Ntips ? 1 : node2tip_count[clade-Ntips]);
	}
	
	// identify each bifurcating node with >=3 descending tips with a constraint
	// some nodes may not correspond to any constraint
	std::vector<long> node2constraint(Nnodes,-1); // 1D array of size Nnodes, mapping node index to constraint index. May include -1 for nodes not corresponding to any constraint.
	long Nconstraints = 0;
	for(long node=0; node<Nnodes; ++node){
		if((node2child_count[node]==2) && (node2tip_count[node]>=3)){
			node2constraint[node] = Nconstraints;
			++Nconstraints;
		}
	}
	
	// create reverse mapping, i.e. constraint --> node
	lvector constraint2node(Nconstraints,0);
	for(long node=0; node<Nnodes; ++node){
		if(node2constraint[node]>=0){
			constraint2node[node2constraint[node]] = node;
		}
	}
	
	// calculate FastTree constraints
	// for each tip and each ancestral node of that tip, determine which side of the node the tip is on (left/0 or right/1)
	// use "-" for non-ancestral nodes
	std::vector<char> constraints(Ntips*Nconstraints,'-'); // 2D array of size Ntips x Nconstraints, in row-major format
	for(long tip=0, clade, parent, pnode; tip<Ntips; ++tip){
		clade = tip;
		while(clade!=root){
			parent 	= clade2parent[clade];
			pnode 	= parent-Ntips;
			if(node2constraint[pnode]>=0){
				// this node is guaranteed to be bifurcating, so we are either coming from the "left" or "right" (there's no 3rd option)
				constraints[tip*Nconstraints + node2constraint[pnode]] = (clade2child_index[clade]==0 ? '0' : '1');
			}
			clade = parent;
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("Nconstraints")		= Nconstraints,
								Rcpp::Named("node2constraint")	= node2constraint,
								Rcpp::Named("constraint2node")	= constraint2node,
								Rcpp::Named("constraints")		= constraints);
}



// calculate the gamma-statistic for a rooted ultrametric tree
// the tree may include multifurcations and monofurcations
// Pybus and Harvey (2000). Testing macro-evolutionary models using incomplete molecular phylogenies. Proceedings of the Royal Society of London. Series B: Biological Sciences. 267:2267-2272. DOI:10.1098/rspb.2000.1278
// [[Rcpp::export]]
double get_gamma_statistic_CPP(	const long					Ntips,
								const long 					Nnodes,
								const long					Nedges,
								const std::vector<long>		&tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
								const std::vector<double>	&edge_length){		// (INPUT) 1D array of size Nedges, or empty
	const bool got_edge_lengths = (edge_length.size()>0);
	const long Nclades = Ntips + Nnodes;
	// find root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
	
	// get tree traversal (root-->tips), including tips
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);

	// calculate distances from root (traverse root-->tips)
	std::vector<double> clade_times(Nclades);
	clade_times[root] = 0;
	for(long q=0, clade; q<traversal.queue.size(); ++q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue;
		for(long e=traversal.node2first_edge[clade-Ntips], edge; e<=traversal.node2last_edge[clade-Ntips]; ++e){
			edge  = traversal.edge_mapping[e];
			clade_times[tree_edge[2*edge+1]] = clade_times[clade] + (got_edge_lengths ? edge_length[edge] : 1.0);
		}
	}
	const double max_clade_time = vector_max(clade_times);
	
	// calculate node ages
	dvector node_ages(Nnodes);
	for(long node=0; node<Nnodes; ++node){
		node_ages[node] = max_clade_time - clade_times[node+Ntips];
	}

	// sort nodes in decreasing age (i.e., root-->tips)
	// each node corresponds to a branching event, so nodes are sorted in chronological order of branching events
	lvector node_order;
	qsortIndices_reverse(node_ages, node_order);

	// determine number of lineages (LTT) shortly after each branching event
	// this code assumes that the tree is ultrametric, i.e. none of the represented lineages went extinct
	lvector	event2LTT(Nnodes,0l);
	for(long event=0, node; event<Nnodes; ++event){
		node = node_order[event];
		event2LTT[event] = (event==0 ? 1l : event2LTT[event-1]) + (traversal.node2last_edge[node]-traversal.node2first_edge[node]);
	}
	
	// calculate T[n] := \sum_{k=1}^n k*g_k, where g_k is the distance of event k to the next event
	dvector event2T(Nnodes,0l);
	double dist;
	for(long event=0, node; event<Nnodes; ++event){
		node = node_order[event];
		dist = node_ages[node_order[event]] - (event==Nnodes-1 ? 0.0 : node_ages[node_order[event+1]]);
		event2T[event] = (event==0 ? 0.0 : event2T[event-1]) + event2LTT[event]*dist;
	}
	
	// calculate gamma-statistic
	double gamma = 0;
	for(long event=0; event<Nnodes-1; ++event){
		gamma += event2T[event];
	}
	gamma = ((gamma/(Nnodes-1.0)) - 0.5*event2T.back())/(event2T.back()/sqrt(12.0*(Nnodes-1)));
	return gamma;
}



// Given a rooted tree, calculate the Colless Index of tree imbalance
// For bifurcating trees the Colless Index will be between 0 (perfectly balanced tree) and 1 (least balanced possible)
// For multifurcating trees, however, the Colless Index may be above 1.
// [[Rcpp::export]]
double get_Colless_Imbalance_CPP(	const long					Ntips,
									const long 					Nnodes,
									const long					Nedges,
									const std::vector<long>		&tree_edge,	// (INPUT) 2D array (in row-major format) of size Nedges x 2
									const bool					normalized){// (INPUT) whether to normalize the Colless index based on the maximum expected value for bifurcating trees

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal route (root --> tips)
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);

	// calculate number of descending tips per node, traversing tips-->root (excluding the root)
	dvector node2tip_count(Nnodes,0);
	for(long q=traversal.queue.size()-1, clade; q>=1; --q){
		clade = traversal.queue[q];
		node2tip_count[clade2parent[clade]-Ntips] += (clade<Ntips ? 1 : node2tip_count[clade-Ntips]);
	}
	
	// calculate Colless Index
	double colless = 0;
	for(long q=traversal.queue.size()-1, parent_node, child1, child2; q>=0; --q){
		parent_node = traversal.queue[q] - Ntips;
		if(parent_node<0) continue;
		for(long e1=traversal.node2first_edge[parent_node]; e1<=traversal.node2last_edge[parent_node]; ++e1){
			child1  = tree_edge[2*traversal.edge_mapping[e1] + 1];
			for(long e2=e1+1; e2<=traversal.node2last_edge[parent_node]; ++e2){
				child2  = tree_edge[2*traversal.edge_mapping[e2] + 1];
				colless += abs((child1<Ntips ? 1l : node2tip_count[child1-Ntips]) - (child2<Ntips ? 1l : node2tip_count[child2-Ntips]));
			}
		}
	}
	if(normalized) colless /= 0.5 * (Ntips-1) * (Ntips-2); // normalize by the maximum possible value if the tree was bifurcating
	return colless;
}



#pragma mark -
#pragma mark Generating & manipulating trees
#pragma mark



// extract a subset of a tree
// the subset of clades (noder & tips) to keep is specified explicitly; edges are kept iff both their parent and child clade is kept
// If a node is kept but none of its descendants, then that node is turned into a tip
// This function assumes that the clades to be kept define a proper tree (and not a forest); no expansion is done.
// this function guarantees that the extracted tree follows the "phylo" conventions on indexing tips, nodes and the root
template<class INT_ARRAY>
void get_arbitrary_subtree(	const long				Ntips,				// (INPUT)
							const long 				Nnodes,				// (INPUT)
							const long				Nedges,				// (INPUT)
							const INT_ARRAY			&tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
							const std::vector<char>	&keep_clade,		// (INPUT) 1D array of size Nclades, specifying whether a clade is to be kept or not
							long					&Ntips_new,			// (OUTPUT) Number of tips kept
							long					&Nnodes_new,		// (OUTPUT) Number of nodes kept
							long					&Nedges_new,		// (OUTPUT) Number of edges kept
							long					&new_root,			// (OUTPUT) new root clade index. Is actually guaranteed to be = Ntips_new.
							std::vector<long>		&new2old_clade,		// (OUTPUT) 1D array of size Nclades_new, mapping new clade indices to old indices
							std::vector<long>		&old2new_clade,		// (OUTPUT) 1D array of size Nclades, mapping old clade indices to new indices (or -1 if a clade is not kept)
							std::vector<long>		&new2old_edge,		// (OUTPUT) 1D array of size Nedges_new, mapping new edge indices to old indices
							std::vector<long>		&new_tree_edge){	// (OUTPUT) 2D array of size Nedges_new x 2, in row-major format, storing the edges of the extracted tree
	const long Nclades = Ntips+Nnodes;

	// determine number of tips & nodes & edges to keep
	Nedges_new = Ntips_new = Nnodes_new = 0;
	std::vector<bool> clade_becomes_tip(Nclades, true);
	for(long edge=0, parent, child; edge<Nedges; ++edge){
		parent 	= tree_edge[edge*2+0];
		child 	= tree_edge[edge*2+1];
		Nedges_new += (keep_clade[child] && keep_clade[parent] ? 1 : 0);
		if(keep_clade[child]) clade_becomes_tip[parent] = false;
	}
	for(long clade=0; clade<Nclades; ++clade){
		if(keep_clade[clade]){
			if(clade_becomes_tip[clade]) ++Ntips_new;
			else ++Nnodes_new;
		}
	}
	const long Nclades_new = Ntips_new + Nnodes_new;

	// create new2old clade mappings
	new_tree_edge.resize(2*Nedges_new);
	new2old_clade.resize(Nclades_new);
	new2old_edge.resize(Nedges_new);
	old2new_clade.assign(Nclades,-1);
	long next_tip_new  = 0;
	long next_node_new = 0;
	for(long clade=0, clade_new; clade<Nclades; ++clade){
		if(keep_clade[clade]){
			if(clade_becomes_tip[clade]) clade_new = next_tip_new++;
			else clade_new = Ntips_new + (next_node_new++);
			new2old_clade[clade_new] = clade;
			old2new_clade[clade] = clade_new;
		}
	}
	
	// determine new root (= new clade with no incoming kept edge)
	new_root = -1;
	std::vector<bool> new_clade_is_root(Nclades_new,true);
	for(long edge=0, parent, child; edge<Nedges; ++edge){
		parent 	= tree_edge[edge*2+0];
		child 	= tree_edge[edge*2+1];
		if(keep_clade[parent] && keep_clade[child]) new_clade_is_root[old2new_clade[child]] = false;
	}
	for(long clade_new=0; clade_new<Nclades_new; ++clade_new){
		if(new_clade_is_root[clade_new]){
			new_root = clade_new;
			break;
		}
	}
		
	// enforce common convention that new_root=Ntips_new 
	// (swap indices with clade previously mapped to Ntips_new)
	long new_root_in_old_tree				= new2old_clade[new_root];
	old2new_clade[new_root_in_old_tree] 	= Ntips_new;
	old2new_clade[new2old_clade[Ntips_new]] = new_root;
	new2old_clade[new_root] 				= new2old_clade[Ntips_new];
	new2old_clade[Ntips_new] 				= new_root_in_old_tree;
	new_root 								= Ntips_new;
	
	// create new2old edge mappings, and new edges
	long next_edge_new = 0;
	for(long edge=0, parent, child; edge<Nedges; ++edge){
		parent 	= tree_edge[edge*2+0];
		child 	= tree_edge[edge*2+1];
		if(keep_clade[child] && keep_clade[parent]){
			new2old_edge[next_edge_new] = edge;
			new_tree_edge[next_edge_new*2+0] = old2new_clade[parent];
			new_tree_edge[next_edge_new*2+1] = old2new_clade[child];
			++next_edge_new;
		}
	}
}



// Re-index clades in a tree such that tips are indexed 0,..,Ntips-1 and nodes are indexed Ntips,..,Nclades-1
// The tree is a-priori only defined based on tree_edge, but no convention of tip & node indexing is assumed
// The number of tips & nodes is inferred based on the tree topology (tree_edge)
// Optionally: The root of the tree (if existent) is ensured to have the index Ntips
// Note: This function does not change the tree topology nor does it reorder edges, it just assigns new indices to tips & nodes
void reindex_clades(	const long				Nclades,			// (INPUT) number of clades (tips or nodes) in the tree
						const long				Nedges,				// (INPUT) number of edges in the tree
						const std::vector<long>	&tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format
						const bool				root_convention,	// (INPUT) If true, the root of the tree (if existent) is ensured to obtain the index = Ntips
						long					&Ntips,				// (OUTPUT) the inferred number of tips in the tree
						long					&Nnodes,			// (OUTPUT) the inferred number of nodes in the tree
						std::vector<long>		&old2new_clade){	// (OUTPUT) 1D array of size Nclades, mapping old-->new clade indices
	// determine tips & nodes
	std::vector<bool> clade_is_tip(Nclades,true);
	for(long edge=0; edge<Nedges; ++edge){
		clade_is_tip[tree_edge[edge*2+0]] = false;
	}
	Ntips = Nnodes = 0;
	for(long clade=0; clade<Nclades; ++clade){
		if(clade_is_tip[clade]) ++Ntips;
		else ++Nnodes;
	}
		
	// re-index clades
	old2new_clade.resize(Nclades);
	long next_tip=0, next_node=0;
	for(long clade=0; clade<Nclades; ++clade){
		if(clade_is_tip[clade]) old2new_clade[clade] = (next_tip++);
		else old2new_clade[clade] = Ntips+(next_node++);
	}
		
	// make sure root is indexed = Ntips, if requested
	if(root_convention){
		std::vector<bool> clade_is_root(Nclades,true);
		for(long edge=0; edge<Nedges; ++edge){
			clade_is_root[tree_edge[edge*2+1]] = false;
		}
		long root = -1, occupier=-1;
		for(long clade=0; clade<Nclades; ++clade){
			if(clade_is_root[clade]) root = clade;
			if(old2new_clade[clade]==Ntips) occupier = clade; // clade currently re-indexed to Ntips
		}
		if(root>=0){
			// tree has a root, so correct index
			long temp_root_new 		= old2new_clade[root];
			old2new_clade[root] 	= Ntips;
			old2new_clade[occupier] = temp_root_new;
		}
	}
}



// sort tree edges such that they are in preorder (root-->tips) or postorder (tips-->root) traversal
// Note that both combinations (depth_first_search=TRUE,root_to_tips=FALSE) and (depth_first_search=FALSE,root_to_tips=FALSE) will yield a valid "postorder" traversal, needed for a multitude of phylogenetics algorithms
template<class ARRAY_TYPE>
void sort_tree_edges_root_to_tips(	const long			Ntips,
									const long			Nnodes,
									const long			Nedges,
									const bool			depth_first_search,	// (INPUT) if false, then breadth-first-seacrh is used instead
									const bool			root_to_tips,		// (INPUT) if false, then edges will be sorted tips-->root
									const ARRAY_TYPE	&tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, listing edges
									std::vector<long>	&new2old_edge){		// (OUTPUT) 1D array of size Nedges, mapping new-->old edge index
	// get incoming edge for each clade
	std::vector<long> incoming_edge_per_clade;
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);
		
	const long root = get_root_from_incoming_edge_per_clade(Ntips, tree_edge, incoming_edge_per_clade);

	// get node-->edge mappings
	std::vector<long> node2first_edge, node2last_edge, edges;
	get_node2edge_mappings(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							node2first_edge,
							node2last_edge,
							edges);
	
	long child,node,clade;
	long next_new_edge = 0;
	new2old_edge.resize(Nedges);
	if(depth_first_search){
		// renumber edges in depth-first-search direction
		// use a scratch_stack for traversing nodes
		std::vector<long> scratch_stack;
		scratch_stack.reserve(floor(2*log(Ntips)/log(2.0))); // rough estimate of typical tree depth x 2. scratch_stack may be resized along the way if needed.
		scratch_stack.push_back(root);
		while(scratch_stack.size()>0){
			clade = scratch_stack.back();
			scratch_stack.pop_back();
			if(incoming_edge_per_clade[clade]>=0) new2old_edge[next_new_edge++] = incoming_edge_per_clade[clade];
			if(clade>=Ntips){
				node = clade-Ntips;
				for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
					scratch_stack.push_back(tree_edge[edges[e]*2+1]);
				}
			}
		}
		
	}else{
		// renumber edges in breadth-first-search direction
		// use a scratch_queue for traversing nodes
		std::vector<long> scratch_queue;
		scratch_queue.reserve(Nnodes);
		scratch_queue.push_back(root);
		long queue_pointer = 0;
		while(queue_pointer<scratch_queue.size()){
			node = scratch_queue[queue_pointer] - Ntips;
			queue_pointer += 1;
			for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
				new2old_edge[next_new_edge++] = edges[e];
				child = tree_edge[edges[e]*2+1];
				if(child>=Ntips) scratch_queue.push_back(child);	// add child to queue for further exploration in the next iteration
			}
		}
	}
	
	if(!root_to_tips) reverse_array(new2old_edge);
}




// sort tree edges such that parent nodes are in preorder (root-->tips) traversal
// Returns an integer array new2old_edge mapping new-->old edge indices
// This is an Rcpp wrapper for sort_tree_edges_root_to_tips(..)
// [[Rcpp::export]]
IntegerVector sort_tree_edges_root_to_tips_CPP(	const long				Ntips,
												const long 				Nnodes,
												const long				Nedges,
												const bool				depth_first_search,	// (INPUT) if false, then breadth-first-seacrh is used instead.
												const bool				root_to_tips,		// (INPUT) if false, then edges will be sorted tips-->root
												const std::vector<long>	&tree_edge){		// (INPUT) 2D array of size Nedges x 2, in row-major format, listing edges.
	std::vector<long> new2old_edge;
	sort_tree_edges_root_to_tips(	Ntips,
									Nnodes,
									Nedges,
									depth_first_search,
									root_to_tips,
									tree_edge,
									new2old_edge);
	return Rcpp::wrap(new2old_edge);
}







// Adjust edge directions so that a specific node becomes the root of the tree (i.e. has no parent and all other tips & nodes descend from it)
// Note that node & tip indices remain unchanged, which may break the common convention that node #1 is the root.
// This function operates in-situ
template<class ARRAY_TYPE>
void root_tree_at_node(	const long 	Ntips,
						const long 	Nnodes,
						const long 	Nedges,
						ARRAY_TYPE 	&tree_edge,		// (INPUT/OUTPUT) 2D array of size Nedges x 2 in row-major format, with values in 0,..,(Nclades-1). Will be modified in-situ
						const long	new_root_node){	// (INPUT) index of node to be turned into root. Must be within 1,..,Nnodes.
	const long Nclades = Ntips+Nnodes;
	
	// create lookup tables listing all edges (regardless of direction) per clade
	std::vector<long> clade2first_edge, clade2last_edge, edges;
	get_inout_edges_per_clade(	Ntips,
								Nnodes,
								Nedges,
								tree_edge,
								clade2first_edge,
								clade2last_edge,
								edges);
						
	// determine correct edge direction and keep track of which edge is incoming for each clade
	// use a scratch_stack for traversing nodes in depth-first-search order
	const long root = new_root_node+Ntips;
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	long child,clade;
	std::vector<long> scratch_stack;
	scratch_stack.reserve(floor(2*log(Ntips)/log(2.0))); // rough estimate of typical tree depth x 2. scratch_stack may be resized along the way if needed.
	scratch_stack.push_back(root);
	while(scratch_stack.size()>0){
		clade = scratch_stack.back();
		scratch_stack.pop_back();
		for(long e=clade2first_edge[clade], edge; e<=clade2last_edge[clade]; ++e){
			edge = edges[e];
			if(edge==incoming_edge_per_clade[clade]) continue; // skip incoming edge
			// at this point it is guaranteed that edge should be outgoing from clade, so adjust direction if needed
			if(tree_edge[edge*2+0]!=clade){
				// edge is in wrong direction, so reverse
				tree_edge[edge*2+1] = tree_edge[edge*2+0];
				tree_edge[edge*2+0] = clade;
			}
			// place child on stack (if a node) and note its incoming edge
			child = tree_edge[edge*2+1];
			incoming_edge_per_clade[child] = edge;
			if(child>=Ntips) scratch_stack.push_back(child);
		}
	}
}


// Adjust edge directions so that a specific node becomes the root of the tree (i.e. has no parent and all other tips & nodes descend from it)
// This function returns a 2D array of size Nedges x 2, in row-major format, listing the new tree edges.
// This function is an Rcpp wrapper for the function root_tree_at_node(..)
// [[Rcpp::export]]
IntegerVector root_tree_at_node_CPP(const long 				Ntips,
									const long 				Nnodes,
									const long 				Nedges,
									const std::vector<long> &tree_edge,		// (INPUT) 2D array of size Nedges x 2 in row-major format, with values in 0,..,(Nclades-1)
									const long				new_root_node){	// (INPUT) index of node to be turned into root. Must be within 1,..,Nnodes.
	lvector new_tree_edge = tree_edge;
	root_tree_at_node(	Ntips,
						Nnodes,
						Nedges,
						new_tree_edge,
						new_root_node);
	return Rcpp::wrap(new_tree_edge);
}
	





// collapse monofurcations in a rooted phylogenetic tree
// i.e. whenever a node only has one child, then combine its 2 adjacent edges into one and remove that node
// note that the tips are always kept
template<class ARRAY_INT, class ARRAY_DOUBLE>
void get_tree_with_collapsed_monofurcations(const long 				Ntips,
											const long 				Nnodes,
											const long 				Nedges,
											const long 				root,
											const bool				force_keep_root,	// (INPUT) if true, the root is always kept even if it only has one child
											const lvector			&force_keep_nodes,	// (INPUT) optional 1D vector, listing node indices (in 0,..,Nnodes-1) to keep even if they are monofurcations
											const ARRAY_INT			&tree_edge,			// (INPUT) 2D array of size Nedges x 2 (in row-major format)
											const ARRAY_DOUBLE 		&edge_length, 		// (INPUT) 1D array of size Nedges, or an empty array (all branches have length 1)
											std::vector<long>		&new_tree_edge,		// (OUTPUT) 2D matrix (in row major format)
											std::vector<double>		&new_edge_length,	// (OUTPUT)
											std::vector<long>		&new2old_node,		// (OUTPUT)
											long					&new_root,			// (OUTPUT) index of the root in the collapsed tree. In newer implementations, this is actually guaranteed to be = Ntips (as is common convention).
											double					&root_shift){		// (OUTPUT) phylogenetic distance of new root from the old root (will be 0 if the old root is kept)
	const long Nclades = Ntips+Nnodes;
	long node, edge, incoming_edge, child, parent;

	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										false,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");
										
	// determine which nodes to keep
	long Nnodes_kept = 0;
	std::vector<bool> keep_node(Nnodes);
	for(long node=0; node<Nnodes; ++node){
		keep_node[node] = ((Ntips+node)==root && force_keep_root) || (traversal_node2first_edge[node]<traversal_node2last_edge[node]); // only keep node if it has at least 2 edges leaving from it
		if(keep_node[node]) ++Nnodes_kept;
	}
	for(long fn=0, node; fn<force_keep_nodes.size(); ++fn){
		node = force_keep_nodes[fn];
		if(!keep_node[node]){
			keep_node[node] = true;
			++Nnodes_kept;
		}
	}
		
	// calculate the mappings new node <--> old node
	// traverse root-->tips, so that the old root is mapped to the lowest new node index (0)
	new2old_node.clear();
	new2old_node.reserve(Nnodes_kept);
	std::vector<long> old2new_node(Nnodes,-1);
	long Nedges_kept = Nedges;
	for(long q=0; q<traversal_queue.size(); ++q){
		node = traversal_queue[q]-Ntips;
		if(keep_node[node]){
			new2old_node.push_back(node);
			old2new_node[node] = new2old_node.size()-1;
		}else{ 
			Nedges_kept -= (traversal_node2last_edge[node]-traversal_node2first_edge[node]+1); // edges leaving this node will not be kept
		}
	}
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
	
	// collapse edges (traverse root --> tips)
	// note that at first, tree_edge[,] will list old clade indices (will be renamed later)
	new_tree_edge.resize(Nedges_kept*2);
	new_edge_length.resize(Nedges_kept);
	std::vector<long> new_incoming_edge_per_clade = incoming_edge_per_clade; // make a copy, then modify in-situ
	long next_new_edge = 0;
	std::vector<long> old2new_edge(Nedges,-1);
	for(long q=0; q<traversal_queue.size(); ++q){
		parent = traversal_queue[q];
		node = parent-Ntips;
		for(long ei=traversal_node2first_edge[node]; ei<=traversal_node2last_edge[node]; ++ei){
			edge  = traversal_edges[ei];
			child = tree_edge[edge*2+1];
			if(keep_node[node]){
				new_tree_edge[next_new_edge*2+0] = parent;
				new_tree_edge[next_new_edge*2+1] = child;
				new_edge_length[next_new_edge] = (edge_length.size()==0 ? 1 : edge_length[edge]);
				old2new_edge[edge] = next_new_edge;
				++next_new_edge;
			}else{
				// merge this departing edge with the incoming edge
				// note that this should only occur once per node (we're not removing nodes with more than 1 departing edges)
				incoming_edge = new_incoming_edge_per_clade[parent];
				if(incoming_edge>=0){
					new_tree_edge[old2new_edge[incoming_edge]*2+1] 	= child;
					new_edge_length[old2new_edge[incoming_edge]] 	+= (edge_length.size()==0 ? 1 : edge_length[edge]); // append this edge's length to the incoming edge
				}
				new_incoming_edge_per_clade[child] = incoming_edge; // update incoming edge of child, since this child was assigned as the destination of incoming_edge. If incoming_edge was -1 (i.e. clade was root), then child becomes the de-facto new root
			}
		}
	}

	// rename nodes in new_tree_edge[]
	for(long new_edge=0; new_edge<Nedges_kept; ++new_edge){
		parent = new_tree_edge[new_edge*2+0];
		child  = new_tree_edge[new_edge*2+1];
		new_tree_edge[new_edge*2+0] = Ntips + old2new_node[parent-Ntips];
		if(child>=Ntips) new_tree_edge[new_edge*2+1] = Ntips + old2new_node[child-Ntips];
	}
	
	// determine new root (new node with no parent)
	root_shift = 0;
	if(force_keep_root){
		new_root = Ntips + old2new_node[root-Ntips];
	}else{
		new_root = get_root_clade(Ntips, Nnodes_kept, Nedges_kept, new_tree_edge);
		// traverse from new to old root and calculate cumulative distance
		// use old tree structure
		long clade = Ntips+new2old_node[new_root-Ntips];
		while(incoming_edge_per_clade[clade]>=0){
			edge = incoming_edge_per_clade[clade];
			root_shift += (edge_length.size()==0 ? 1 : edge_length[edge]);
			clade = tree_edge[edge*2 + 0];
		}
	}
}


// Rcpp wrapper for get_tree_with_collapsed_monofurcations()
// [[Rcpp::export]]
Rcpp::List get_tree_with_collapsed_monofurcations_CPP(	const long 					Ntips,
														const long 					Nnodes,
														const long 					Nedges,
														const std::vector<long>		&tree_edge,			// (INPUT) 2D array of size Nedges x 2 (in row-major format)
														const std::vector<double> 	&edge_length, 		// (INPUT) 1D array of size Nedges, or an empty array (all branches have length 1)
														const bool					force_keep_root,	// (INPUT) if true, the root is always kept even if it only has one child
														const std::vector<long>		&force_keep_nodes){	// (INPUT) optional list of nodes to keep, even if they are monofurcations
	// find root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);

	std::vector<long> new_tree_edge, new2old_node;
	std::vector<double> new_edge_length;
	long new_root;
	double root_shift;
	get_tree_with_collapsed_monofurcations(	Ntips,
											Nnodes,
											Nedges,
											root,
											force_keep_root,
											force_keep_nodes,
											tree_edge,	
											edge_length,
											new_tree_edge,
											new_edge_length,
											new2old_node,
											new_root,
											root_shift);

	return Rcpp::List::create(	Rcpp::Named("Nnodes_new")		= long(new2old_node.size()), // explicitly cast to long, otherwise Rcpp does not know how to wrap it (on Windows)
								Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new_edge_length") 	= Rcpp::wrap(new_edge_length), // if the original edge_length[] was empty, then new_edge_length[e] will be the number of combined edges making up the new edge e
								Rcpp::Named("new2old_node") 	= Rcpp::wrap(new2old_node),
								Rcpp::Named("new_root") 		= new_root,
								Rcpp::Named("root_shift") 		= root_shift);
}






// Extract subtree with subset of tips and/or nodes
// Requirements:
//   tree can include multifucations and monofurcations
//   tree must be rooted (root will be determined automatically, as the node without a parent)
// [[Rcpp::export]]
Rcpp::List get_subtree_with_specific_clades_CPP(const long 			Ntips,
												const long 			Nnodes,
												const long 			Nedges,
												const std::vector<long> &tree_edge,			// 2D array of size Nedges x 2 in row-major format
												const std::vector<double> &edge_length, 		// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
												const std::vector<long> &clades_to_keep,	// 1D array with values in 0,..,(Nclades-1)
												const bool			collapse_monofurcations,	// if true, nodes that are left with only one child (after pruning) will be removed (and the adjacent edges will be combined into a single edge)
												const bool			force_keep_root,	// if true, then the root is kept even if collapse_monofurcations==true and the root is monofurcating.
												bool				keep_all_children_of_explicit_clades_to_keep,	// if true, then for each clade in clades_to_keep[], all children are also kept. Otherwise only one child is kept.
												bool				keep_all_tips_of_explicit_clades_to_keep){		// if true, then for each clade in clades_to_keep[], all descending tips are also kept. Otherwise (typically) only one tip is kept per child.
	const long Nclades = Ntips+Nnodes;
	long clade, parent, child, node;
	if(keep_all_tips_of_explicit_clades_to_keep) keep_all_children_of_explicit_clades_to_keep = false;

	// get parent of each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);	
											
	// get tree traversal route (tips --> root)											
	std::vector<long> traversal_queue_root2tips, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(Ntips,
									Nnodes,
									Nedges,
									root,
									tree_edge,
									true,
									false,
									traversal_queue_root2tips,
									traversal_node2first_edge,
									traversal_node2last_edge,
									traversal_edges,
									false,
									"");
			
	// determine which clades to keep
	std::vector<bool> keep_clade_explicitly(Nclades,false);
	std::vector<bool> keep_clade_ancestrally(Nclades,false);
	std::vector<bool> keep_clade_descending(Nclades,false);
	for(long c=0; c<clades_to_keep.size(); ++c){
		clade = clades_to_keep[c];
		keep_clade_explicitly[clade]  = true;
		keep_clade_ancestrally[clade] = true;
		keep_clade_descending[clade]  = true;
	}
	// step 1: propagate status of to-be-kept upwards (traverse tips-->root)
	// so that all ancestors of a clade to be kept are also kept
	for(long q=traversal_queue_root2tips.size()-1; q>=0; --q){
		clade = traversal_queue_root2tips[q];
		if(keep_clade_ancestrally[clade] && (clade!=root)){
			// clade is to be kept, so propagate upwards
			keep_clade_ancestrally[clade2parent[clade]] = true;
		}
	}
	// step 2: propagate status of to-be-kept downwards (traverse root-->tips)
	// so that each clade to be kept leads to at least one (or multiple) tips
	for(long q=0; q<traversal_queue_root2tips.size(); ++q){
		clade = traversal_queue_root2tips[q];
		if(keep_clade_descending[clade] && (clade>=Ntips)){
			node = clade - Ntips;
			const bool keep_all_children = (keep_all_tips_of_explicit_clades_to_keep || (keep_clade_explicitly[clade] && keep_all_children_of_explicit_clades_to_keep));
			if((!keep_all_children) && (!keep_clade_explicitly[clade]) && keep_clade_ancestrally[clade]) continue; // clade has already been included due to one of its descendants, so no need to include more of its children
			for(long e=traversal_node2first_edge[node]; e<=(keep_all_children ? traversal_node2last_edge[node] : traversal_node2first_edge[node]); ++e){
				child = tree_edge[traversal_edges[e]*2+1];
				keep_clade_descending[child] = true;
			}
		}
	}
	// step 3: merge various reasons of clade inclusion
	std::vector<bool> keep_clade(Nclades,false);
	long Nedges_kept = 0;
	for(long c=0; c<Nclades; ++c){
		keep_clade[c] = (keep_clade_explicitly[c] || keep_clade_ancestrally[c] || keep_clade_descending[c]);
		if(keep_clade[c] && (c!=root)) ++Nedges_kept; // for each kept clade, there corresponds exactly one kept edge (unless the clade is root)
	}

		
	// translate old clade indices to new (map to -1 if not included)
	long Nclades_kept = 0;
	long Ntips_kept   = 0;
	std::vector<long> old2new_clade(Nclades,-1);
	for(long c=0; c<Nclades; ++c){
		if(keep_clade[c]){
			old2new_clade[c] = (Nclades_kept++);
			if(c<Ntips) Ntips_kept += 1;
		}
	}
	long new_root 	 = old2new_clade[root];
	long Nnodes_kept = Nclades_kept-Ntips_kept;
	
	// calculate the reverse mapping (new --> old clade index)
	std::vector<long> new2old_clade(Nclades_kept,-1);
	for(long c=0; c<Nclades; ++c){
		if(keep_clade[c]) new2old_clade[old2new_clade[c]] = c;
	}
	
	// enforce common convention that new_root=Ntips_kept (swap indices with clade previously mapped to Ntips_kept)
	old2new_clade[root] 						= Ntips_kept;
	old2new_clade[new2old_clade[Ntips_kept]] 	= new_root;
	new2old_clade[new_root] 					= new2old_clade[Ntips_kept];
	new2old_clade[Ntips_kept] 					= root;
	new_root 									= Ntips_kept;
	
	// extract subset of kept edges
	lvector new_tree_edge(Nedges_kept*2);
	dvector new_edge_length(Nedges_kept);
	long next_new_edge = 0;
	for(long edge=0; edge<Nedges; ++edge){
		parent = tree_edge[edge*2+0];
		child  = tree_edge[edge*2+1];
		if(keep_clade[parent] && keep_clade[child]){
			new_tree_edge[next_new_edge*2+0] = old2new_clade[parent];
			new_tree_edge[next_new_edge*2+1] = old2new_clade[child];
			new_edge_length[next_new_edge]   = (edge_length.size()==0 ? 1 : edge_length[edge]);
			++next_new_edge;
		}
	}
	
	
	// collapse monofurcations if needed
	double root_shift = 0;
	if(collapse_monofurcations){
		long newer_root;
		std::vector<long> newer_tree_edge, newer2new_node;
		std::vector<double> newer_edge_length;
		get_tree_with_collapsed_monofurcations(	Ntips_kept, 
												(Nclades_kept-Ntips_kept), 
												Nedges_kept,
												new_root,
												force_keep_root,
												lvector(),
												new_tree_edge,
												new_edge_length,
												newer_tree_edge,
												newer_edge_length,
												newer2new_node,
												newer_root,
												root_shift);
		new_tree_edge 	= newer_tree_edge;
		new_edge_length = newer_edge_length;
		Nnodes_kept		= newer2new_node.size();
		Nclades_kept 	= Ntips_kept+Nnodes_kept;
		std::vector<long> newer2old_node(Nnodes_kept);
		for(long n=0; n<Nnodes_kept; ++n){
			newer2old_node[n] = new2old_clade[Ntips_kept+newer2new_node[n]]-Ntips;
		}
		new2old_clade.resize(Nclades_kept);
		for(long n=0; n<Nnodes_kept; ++n){
			new2old_clade[Ntips_kept+n] = Ntips + newer2old_node[n];
		}
		new_root = newer_root;
	}
	
	return Rcpp::List::create(	Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new_edge_length") 	= Rcpp::wrap(new_edge_length), // if the original edge_length[] was empty, then new_edge_length[e] will be the number of combined edges making up the new edge e
								Rcpp::Named("new2old_clade") 	= Rcpp::wrap(new2old_clade),
								Rcpp::Named("new_root") 		= new_root,
								Rcpp::Named("Ntips_kept") 		= Ntips_kept,
								Rcpp::Named("Nnodes_kept") 		= Nnodes_kept,
								Rcpp::Named("Nedges_kept") 		= Nedges_kept,
								Rcpp::Named("root_shift") 		= root_shift);
}



// extract the subtree descending from a specific node
// using a pre-computed edge_mapping for efficiency
// this function guarantees that new_root will be equal to Ntips_kept
void get_subtree_at_node(	const long 			Ntips,
							const long 			Nnodes,
							const long 			Nedges,
							const std::vector<long> &tree_edge,			// (INPUT) 2D array of size Nedges x 2 in row-major format
							const long			focal_node,			// (INPUT) node at which to extract subtree. This node will become the root of the extracted subtree. Must be within 0,..,Nnodes-1
							const lvector 		&node2first_edge, 	// (INPUT) 1D array of size Nnodes, mapping nodes to its first entry in edge_mapping[], as calculated by get_node2edge_mappings()
							const lvector 		&node2last_edge, 	// (INPUT) 1D array of size Nnodes, mapping nodes to its last entry in edge_mapping[], as calculated by get_node2edge_mappings()
							const lvector 		&edge_mapping,		// (INPUT) 1D array of size Nedges, listing edges ordered by node, as calculated by get_node2edge_mappings()
							lvector				&scratch_nodes,		// (SCRATCH)
							lvector				&scratch_new_nodes,	// (SCRATCH)
							lvector				&new_tree_edge,		// (OUTPUT)
							lvector				&new2old_clade,		// (OUTPUT)
							lvector				&new2old_edge,		// (OUTPUT)
							long				&new_root,			// (OUTPUT)
							long				&Ntips_kept,		// (OUTPUT)
							long				&Nnodes_kept,		// (OUTPUT)
							long				&Nedges_kept){		// (OUTPUT)
	// determine size of subtree (count descendants of focal_node)
	// use a scratch_nodes for traversing nodes in a depth-first-search manner
	long child,node,edge;
	scratch_nodes.clear();
	scratch_nodes.reserve(Nnodes);
	scratch_nodes.push_back(focal_node);
	Ntips_kept  = 0;
	Nnodes_kept = 1;
	Nedges_kept = 0;
	while(scratch_nodes.size()>0){
		node = scratch_nodes.back();
		scratch_nodes.pop_back();
		for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
			edge  = edge_mapping[e];
			child = tree_edge[edge*2+1];
			if(child>=Ntips) scratch_nodes.push_back(child-Ntips); // add child to stack for further exploration in the next iteration
			// increment counters
			++Nedges_kept;
			if(child<Ntips){
				++Ntips_kept;
			}else{
				++Nnodes_kept;
			}
		}
	}
	const long Nclades_kept = Ntips_kept + Nnodes_kept;
	
	// re-traverse subtree (focal_node-->tips) and extract subtree & generate new2old mappings
	// scratch_nodes[] and scratch_new_nodes[] will be synchronized, containing old & new node indices respectively while traversing
	new2old_clade.resize(Nclades_kept);
	new2old_edge.resize(Nedges_kept);
	new_tree_edge.resize(2*Nedges_kept);
	//   add first new_node (=focal_node) as root of the new tree
	scratch_nodes.clear();
	scratch_new_nodes.clear();
	scratch_new_nodes.reserve(Nnodes);
	scratch_nodes.push_back(focal_node);
	scratch_new_nodes.push_back(0);
	new2old_clade[Ntips_kept] = focal_node+Ntips;
	long new_edge=-1, new_tip=-1, new_node=0, new_child, new_parent;
	while(scratch_nodes.size()>0){
		node 		= scratch_nodes.back();
		new_parent 	= Ntips_kept + scratch_new_nodes.back();
		scratch_nodes.pop_back();
		scratch_new_nodes.pop_back();
		for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
			edge  = edge_mapping[e];
			child = tree_edge[edge*2+1];
			if(child<Ntips){
				++new_tip;
				new_child = new_tip;
			}else{
				++new_node;
				scratch_nodes.push_back(child-Ntips); // add child node to stack for further exploration in the next iteration
				scratch_new_nodes.push_back(new_node);
				new_child = Ntips_kept+new_node;
			}
			new2old_clade[new_child] = child;
			// add new edge new_parent --> new_child
			++new_edge;
			new2old_edge[new_edge] = edge;
			new_tree_edge[2*new_edge+0] = new_parent;
			new_tree_edge[2*new_edge+1] = new_child;
		}
	}
	
	new_root = Ntips_kept;
}
							





// extract the subtree descending from a specific node
// The new root is guaranteed to be the node with index = 0.
// [[Rcpp::export]]
Rcpp::List get_subtree_at_node_CPP(	const long 			Ntips,
									const long 			Nnodes,
									const long 			Nedges,
									const std::vector<long> &tree_edge,		// (INPUT) 2D array of size Nedges x 2 in row-major format
									const long			new_root_node){	// (INPUT) node at which to extract subtree. This node will become the root of the extracted subtree. Must be within 0,..,Nnodes-1	
	// get edge mapping for easier traversal
	std::vector<long> node2first_edge, node2last_edge, edge_mapping;
	get_node2edge_mappings(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							node2first_edge,
							node2last_edge,
							edge_mapping);

	// Extract subtree, using edge mapping
	lvector new_tree_edge, new2old_clade, new2old_edge, scratch_stack1, scratch_stack2;
	long new_root, Ntips_kept, Nnodes_kept, Nedges_kept;
	get_subtree_at_node(Ntips,
						Nnodes,
						Nedges,
						tree_edge,
						new_root_node,
						node2first_edge,
						node2last_edge,
						edge_mapping,
						scratch_stack1,
						scratch_stack2,
						new_tree_edge,
						new2old_clade,
						new2old_edge,
						new_root,
						Ntips_kept,
						Nnodes_kept,
						Nedges_kept);
	
	return Rcpp::List::create(	Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new2old_clade") 	= Rcpp::wrap(new2old_clade),
								Rcpp::Named("new2old_edge") 	= Rcpp::wrap(new2old_edge),
								Rcpp::Named("new_root") 		= new_root,
								Rcpp::Named("Ntips_kept") 		= Ntips_kept,
								Rcpp::Named("Nnodes_kept") 		= Nnodes_kept,
								Rcpp::Named("Nedges_kept") 		= Nedges_kept);
}



// extract subtrees descending from specific nodes
// The new roots are guaranteed to be the node with index = 0
// This function is optimized for extracting a large number of nodes from large trees
// [[Rcpp::export]]
Rcpp::List get_subtrees_at_nodes_CPP(	const long 				Ntips,
										const long 				Nnodes,
										const long 				Nedges,
										const std::vector<long> &tree_edge,			// (INPUT) 2D array of size Nedges x 2 in row-major format
										const std::vector<long>	&new_root_nodes){	// (INPUT) 1D array listing focal nodes whose subtrees to extract. Each of these nodes will become the root of an extracted subtree. Each entry must be within 0,..,Nnodes-1	
	const long NS = new_root_nodes.size();
	
	// determine parent edge for each node, if available
	lvector node2parent_edge(Nnodes,-1);
	for(long edge=0, child; edge<Nedges; ++edge){
		child = tree_edge[2*edge+1];
		if(child>=Ntips) node2parent_edge[child-Ntips] = edge;
	}
	
	// get edge mapping for easier traversal
	std::vector<long> node2first_edge, node2last_edge, edge_mapping;
	get_node2edge_mappings(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							node2first_edge,
							node2last_edge,
							edge_mapping);

	// Extract subtrees, using precomputed edge mapping
	std::vector<lvector> new_tree_edge(NS), new2old_clade(NS), new2old_edge(NS);
	lvector scratch_stack1, scratch_stack2;
	lvector new_root(NS), Ntips_kept(NS), Nnodes_kept(NS), Nedges_kept(NS), stem_edges(NS);
	for(long n=0, node; n<NS; ++n){
		node = new_root_nodes[n];
		stem_edges[n] = node2parent_edge[node];
		get_subtree_at_node(Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							node,
							node2first_edge,
							node2last_edge,
							edge_mapping,
							scratch_stack1,
							scratch_stack2,
							new_tree_edge[n],
							new2old_clade[n],
							new2old_edge[n],
							new_root[n],
							Ntips_kept[n],
							Nnodes_kept[n],
							Nedges_kept[n]);
	}
	
	return Rcpp::List::create(	Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new2old_clade") 	= Rcpp::wrap(new2old_clade),
								Rcpp::Named("new2old_edge") 	= Rcpp::wrap(new2old_edge),
								Rcpp::Named("stem_edges") 		= Rcpp::wrap(stem_edges),
								Rcpp::Named("new_root") 		= Rcpp::wrap(new_root),
								Rcpp::Named("Ntips_kept") 		= Rcpp::wrap(Ntips_kept),
								Rcpp::Named("Nnodes_kept") 		= Rcpp::wrap(Nnodes_kept),
								Rcpp::Named("Nedges_kept") 		= Rcpp::wrap(Nedges_kept));
}







// Extract subtree with a specific subset of tips (and including all of their ancestors)
// If the original edge_length[] was empty, then new_edge_length[e] will be the number of combined edges making up the new edge e
// This function guarantees that the new root of the subtree will have index = Ntips_kept.
// Requirements:
//   tree can include multifucations and monofurcations
//   tree must be rooted (root will be determined automatically, as the node without a parent)
template<class ARRAY_INT,class ARRAY_DOUBLE>
void get_subtree_with_specific_tips(const long 			Ntips,
									const long 			Nnodes,
									const long 			Nedges,
									const ARRAY_INT 	&tree_edge,						// (INPUT) 2D array of size Nedges x 2 in row-major format
									const ARRAY_DOUBLE 	&edge_length, 					// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
									const ARRAY_INT		&tips_to_keep,					// (INPUT) 1D array with values in 0,..,(Ntips-1)
									bool				collapse_monofurcations,		// (INPUT) if true, nodes that are left with only one child (after pruning) will be removed (and the adjacent edges will be combined into a single edge)
									bool				force_keep_root,				// (INPUT) alwasy keep root, even if collapse_monofurcations==true and root is monofurcating
									const lvector		&force_keep_nodes,				// (INPUT) optional 1D vector, listing node indices (in 0,..,Nnodes-1) to keep even if all descending tips have been removed (turn nodes to tips) or if they become monofurcations. Note that nodes not explicitly listed here may or may not be removed.
									std::vector<long>	&new_tree_edge,					// (OUTPUT) 2D array of size Nedges x 2, in row-major format
									std::vector<double>	&new_edge_length,				// (OUTPUT) 1D array of size Nedges
									std::vector<long>	&new2old_clade,					// (OUTPUT) 1D array of size Nclades_kept
									long				&new_root,						// (OUTPUT) root index in the new tree. In newer implementations this is actually guaranteed to be Ntips_kept+1.
									long				&old_stem_edge,					// (OUTPUT) edge index in the old tree, leading into the subtree's root. May be <0, if not applicable (e.g., if the subtree's root is the same as the supertree's root)
									long				&Ntips_new,						// (OUTPUT)
									long				&Nnodes_new,					// (OUTPUT)
									long				&Nedges_new,					// (OUTPUT)
									double				&root_shift){					// (OUTPUT)
	const long Nclades = Ntips+Nnodes;
	long clade, parent, child;
	
	// get parent & incoming edge of each clade
	lvector clade2parent(Nclades,-1), clade2incoming_edge(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		clade2parent[tree_edge[edge*2+1]] = tree_edge[edge*2+0];
		clade2incoming_edge[tree_edge[edge*2+1]] = edge;
	}

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
								
	// get tree traversal route
	std::vector<long> traversal_queue_root2tips, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(Ntips,
									Nnodes,
									Nedges,
									root,
									tree_edge,
									true,
									false,
									traversal_queue_root2tips,
									traversal_node2first_edge,
									traversal_node2last_edge,
									traversal_edges,
									false,
									"");
		
	// determine which clades to keep
	Nedges_new = 0;
	std::vector<bool> keep_clade(Nclades,false);
	std::vector<bool> clade_has_kept_descendant(Nclades,false); // keep track of which clades are ancestors of other kept clades (and thus will not become tips)
	for(long t=0; t<tips_to_keep.size(); ++t){
		keep_clade[tips_to_keep[t]] = true;
	}
	for(long fn=0; fn<force_keep_nodes.size(); ++fn){
		keep_clade[Ntips+force_keep_nodes[fn]] = true;
	}
	// propagate status of to-be-kept upwards (traverse tips-->root)
	// so that all ancestors of a clade to be kept are also kept
	for(long q=traversal_queue_root2tips.size()-1; q>=0; --q){
		clade = traversal_queue_root2tips[q];
		if(keep_clade[clade] && (clade!=root)){
			// clade is to be kept, so propagate upwards
			keep_clade[clade2parent[clade]] = true;
			clade_has_kept_descendant[clade2parent[clade]] = true;
			++Nedges_new;
		}
	}
		
	// translate old clade indices to new (map to -1 if not included)
	// make sure that all new tips are indexed before nodes
	long Nclades_new = 0;
	Ntips_new = 0;
	lvector old2new_clade(Nclades,-1);
	for(long tip=0; tip<Ntips; ++tip){
		if(keep_clade[tip]){
			old2new_clade[tip] = (Nclades_new++);
			++Ntips_new;
		}
	}
	for(long clade=Ntips; clade<Nclades; ++clade){
		if(keep_clade[clade] && (!clade_has_kept_descendant[clade])){
			// this node is kept without having a kept descendant, so turn into a tip
			old2new_clade[clade] = (Nclades_new++);
			++Ntips_new;
		}
	}
	for(long clade=Ntips; clade<Nclades; ++clade){
		if(keep_clade[clade] && (old2new_clade[clade]<0)){
			// this clade is kept and has not yet been assigned a new index
			old2new_clade[clade] = (Nclades_new++);
		}
	}
	new_root 	= old2new_clade[root];
	Nnodes_new 	= Nclades_new - Ntips_new;
	
	// calculate the reverse mapping (new --> old clade index)
	new2old_clade.assign(Nclades_new,-1);
	for(long c=0; c<Nclades; ++c){
		if(keep_clade[c]) new2old_clade[old2new_clade[c]] = c;
	}
	
	// enforce common convention that new_root = Ntips_new (swap indices with clade previously mapped to Ntips_new)
	old2new_clade[root] 						= Ntips_new;
	old2new_clade[new2old_clade[Ntips_new]] 	= new_root;
	new2old_clade[new_root] 					= new2old_clade[Ntips_new];
	new2old_clade[Ntips_new] 					= root;
	new_root 									= Ntips_new;
	
	// extract subset of kept edges
	new_tree_edge.resize(Nedges_new*2);
	new_edge_length.resize(Nedges_new);
	long next_new_edge = 0;
	for(long edge=0; edge<Nedges; ++edge){
		parent = tree_edge[edge*2+0];
		child  = tree_edge[edge*2+1];
		if(keep_clade[parent] && keep_clade[child]){
			new_tree_edge[next_new_edge*2+0] = old2new_clade[parent];
			new_tree_edge[next_new_edge*2+1] = old2new_clade[child];
			new_edge_length[next_new_edge] = (edge_length.size()==0 ? 1 : edge_length[edge]);
			++next_new_edge;
		}
	}
	
	// update list of force_keep_nodes (update indices and remove nodes that became tips)
	lvector force_keep_nodes_new;
	force_keep_nodes_new.reserve(force_keep_nodes.size());
	for(long fn=0, new_clade; fn<force_keep_nodes.size(); ++fn){
		new_clade = old2new_clade[Ntips+force_keep_nodes[fn]];
		if(new_clade>=Ntips_new) force_keep_nodes_new.push_back(new_clade-Ntips_new);
	}
		
	// collapse monofurcations if needed
	root_shift = 0;
	if(collapse_monofurcations){
		long newer_root;
		std::vector<long> newer_tree_edge, newer2new_node;
		std::vector<double> newer_edge_length;
		get_tree_with_collapsed_monofurcations(	Ntips_new, 
												Nnodes_new, 
												Nedges_new,
												new_root,
												force_keep_root,
												force_keep_nodes_new,
												new_tree_edge,
												new_edge_length,
												newer_tree_edge,
												newer_edge_length,
												newer2new_node,
												newer_root,
												root_shift);
		new_tree_edge 	= newer_tree_edge;
		new_edge_length = newer_edge_length;
		Nnodes_new		= newer2new_node.size();
		Nclades_new 	= Ntips_new+Nnodes_new;
		Nedges_new		= newer_edge_length.size();
		lvector newer2old_node(Nnodes_new);
		for(long n=0; n<Nnodes_new; ++n){
			newer2old_node[n] = new2old_clade[Ntips_new+newer2new_node[n]]-Ntips;
		}
		new2old_clade.resize(Nclades_new);
		for(long n=0; n<Nnodes_new; ++n){
			new2old_clade[Ntips_new+n] = Ntips + newer2old_node[n];
		}
		new_root = newer_root;
	}
	
	// determine edge (old indexing) leading into the subtree's root
	old_stem_edge = clade2incoming_edge[new2old_clade[new_root]];
}




// Extract subtree with a specific subset of tips (and including all of their ancestors)
// This is an Rcpp wrapper for get_subtree_with_specific_tips()
// Requirements:
//   tree can include multifucations and monofurcations
//   tree must be rooted (root will be determined automatically, as the node without a parent)
// [[Rcpp::export]]
Rcpp::List get_subtree_with_specific_tips_CPP(	const long 					Ntips,
												const long 					Nnodes,
												const long 					Nedges,
												const std::vector<long> 	&tree_edge,			// (INPUT) 2D array of size Nedges x 2 in row-major format
												const std::vector<double> 	&edge_length, 		// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
												const std::vector<long> 	&tips_to_keep,		// (INPUT) 1D array with values in 0,..,(Ntips-1)
												bool						collapse_monofurcations,	// (INPUT) if true, nodes that are left with only one child (after pruning) will be removed (and the adjacent edges will be combined into a single edge)
												bool						force_keep_root){			// (INPUT) alwasy keep root, even if collapse_monofurcations==true and root is monofurcating

	std::vector<long> new_tree_edge, new2old_clade;
	std::vector<double> new_edge_length;
	long new_root, Ntips_kept, Nnodes_kept, Nedges_kept, old_stem_edge;
	double root_shift;
	get_subtree_with_specific_tips(	Ntips,
									Nnodes,
									Nedges,
									tree_edge,
									edge_length,
									tips_to_keep,
									collapse_monofurcations,
									force_keep_root,
									lvector(),
									new_tree_edge,
									new_edge_length,
									new2old_clade,
									new_root,
									old_stem_edge,
									Ntips_kept,
									Nnodes_kept,
									Nedges_kept,
									root_shift);
	
	return Rcpp::List::create(	Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new_edge_length") 	= Rcpp::wrap(new_edge_length),
								Rcpp::Named("new2old_clade") 	= Rcpp::wrap(new2old_clade),
								Rcpp::Named("new_root") 		= new_root,
								Rcpp::Named("old_stem_edge")	= old_stem_edge,
								Rcpp::Named("Ntips_kept") 		= Ntips_kept,
								Rcpp::Named("Nnodes_kept") 		= Nnodes_kept,
								Rcpp::Named("Nedges_kept") 		= Nedges_kept,
								Rcpp::Named("root_shift")		= root_shift);
}




// returns a random tip in the tree, by randomly traversing the tree root-->tips.
// each child of a node is equally probable to be traversed (when passed through that node)
long get_tip_by_random_uniform_traversal(	const long			Ntips,
											const long			root,				// (INPUT) root index. Typically root=Ntips.
											std::vector<long>	&node2first_child,	// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Should be in the format returned by get_children_per_node(..).
											std::vector<long>	&node2last_child,	// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Should be in the format returned by get_children_per_node(..).
											std::vector<long>	&children){			// (INPUT) 1D array of size Nedges, with values in 0:(Nclades-1). Should be in the format returned by get_children_per_node(..).
	long clade = root;
	while(clade>=Ntips){
		clade = children[uniformIntWithin(node2first_child[clade-Ntips],node2last_child[clade-Ntips])];
	}
	return clade;
}



// returns a random tip in the tree, by randomly traversing the tree root-->tips.
// The probability of a child of a node is given externally (via clade2probability[]). Note t
long get_tip_by_random_traversal(	const long					Ntips,
									const long					root,					// (INPUT) root index. Typically root=Ntips.
									const std::vector<long>		&node2first_child,		// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Should be in the format returned by get_children_per_node(..).
									const std::vector<long>		&node2last_child,		// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Should be in the format returned by get_children_per_node(..).
									const std::vector<long>		&children,				// (INPUT) 1D array of size Nedges, with values in 0:(Nclades-1). Should be in the format returned by get_children_per_node(..).
									const std::vector<double>	&clade2weight){			// (INPUT) 1D array of non-negative numbers, of size Nclades, giving probability weights to each clade. Weights are normalized among children of each node, during traversal.
	double p, total_weight;
	long node;
	long clade = root;
	while(clade>=Ntips){
		node = clade-Ntips;
		// determine normalization factor for weights
		total_weight = 0;
		for(long c=node2first_child[node]; c<=node2last_child[node]; ++c){
			total_weight += clade2weight[children[c]];
		}
		// choose random child
		if(total_weight==0){
			// this should not happen, but if it does continue anyway by choosing a random child
			clade = children[uniformIntWithin(node2first_child[node],node2last_child[node])];
			continue;
		}
		clade = children[node2last_child[node]]; // leftover child (if random p<1)
		// p = double(rand())/RAND_MAX; // random number within [0,1]. Note that R package builders discourage the use of rand(), so use R::runif() instead
		p = R::runif(0.0,1.0);
		for(long c=node2first_child[node]; c<=node2last_child[node]; ++c){
			p -= clade2weight[children[c]]/total_weight;
			if((p<=0) && (clade2weight[children[c]]>0)){
				clade = children[c];
				break;
			}
		}
	}
	return clade;
}





// ###############################################################################################
// Ancestral state reconstruction of discrete characters
// using fixed-rate continuous-time Markov models and the rerooting method by Yang et al. (1995).




// Set an internal node as a new root for a phylogenetic tree, by redefining edge directions
// The number of tips & nodes & edges (and their indices) will remain the same, but some edges will change direction i.f. tree_edge[e,0] will be swapped with tree_edge[e,1].
// The original root may become a singleton node (i.e. with only one outgoing edge) and the new root may become trifurcating (i.e. with 3 outgoing edges)
// Requirements:
//	 The original tree can include mono- and multi-furcations
//   The original tree must be rooted.
template<class ARRAY_TYPE>
void reroot_tree_at_node(	const long 	Ntips,
							const long 	Nnodes,
							const long 	Nedges,
							const long	old_root,		// (INPUT) integer in Nnodes,..,Ntips+Nnodes-1.
							const long 	new_root,		// (INPUT) integer in Nnodes,..,Ntips+Nnodes-1. Node to be set as new root.	
							ARRAY_TYPE	&tree_edge){	// (INPUT/OUTPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1). After rerooting, this array will be modified to reflect the new edge directions (i.e. tree_edge[e,0] will be swapped with tree_edge[e,1] for some e).			
	// get incoming edge for each clade
	std::vector<long> incoming_edge_per_clade;
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);
	
	// traverse from new_root towards old_root
	long clade = new_root;
	long parent, edge;
	while(clade!=old_root){
		edge	= incoming_edge_per_clade[clade];
		parent 	= tree_edge[edge*2+0];
		// swap parent & child in this edge
		tree_edge[edge*2+0] = clade;
		tree_edge[edge*2+1] = parent;
		clade	= parent;
	}
}



// Set an internal node as a new root for a phylogenetic tree, by redefining edge directions
// Similar to the function above, but here incoming_edge_per_clade[] is also updated
template<class ARRAY_TYPE>
void reroot_tree_at_node(	const long 			Ntips,
							const long 			Nnodes,
							const long 			Nedges,
							const long			old_root,					// (INPUT) integer in Ntips,..,Ntips+Nnodes-1.
							const long 			new_root,					// (INPUT) integer in Ntips,..,Ntips+Nnodes-1. Node to be set as new root.	
							ARRAY_TYPE			&tree_edge,					// (INPUT/OUTPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1). After rerooting, this array will be modified to reflect the new edge directions (i.e. tree_edge[e,0] will be swapped with tree_edge[e,1] for some e).
							std::vector<long>	&incoming_edge_per_clade){	// (INPUT/OUTPUT) 1D array of size Nclades, with elements in 0,..,Nedges-1. Will be updated after the rerooting.
	
	// traverse from new_root towards old_root
	long clade = new_root;
	long parent, edge, previous_edge=-1;
	while(clade!=old_root){
		edge	= incoming_edge_per_clade[clade];
		parent 	= tree_edge[edge*2+0];
		if(previous_edge>=0) incoming_edge_per_clade[clade] = previous_edge;
		// swap parent & child in this edge
		tree_edge[edge*2+0] = clade;
		tree_edge[edge*2+1] = parent;
		previous_edge = edge; // keep a record of this edge for the next iteration, so that we can update the incoming edge after retrieving some old information
		clade = parent;
	}
	if(old_root!=new_root){
		incoming_edge_per_clade[old_root] = previous_edge;
		incoming_edge_per_clade[new_root] = -1;
	}
}



// Unroot a phylogenetic tree, by eliminating the root node if it is bifurcating (thus connecting its two child edges into a single edge)
// Note that noting is done if the tree is not rooted, or if the root has more than 2 outgoing edges.
// The number of tips and the tip indices are kept unchanged, and the number of nodes is reduced by at most 1
// Requirements:
//	 The original tree can include mono- and multi-furcations
//   The original tree may or may not be rooted.
//	 The original tree must have at least 3 tips.
// [[Rcpp::export]]
Rcpp::List eliminate_bifurcating_root_CPP(const long 					Ntips,
									const long 					Nnodes,
									const long 					Nedges,
									const std::vector<long>		&tree_edge,		// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
									const std::vector<double> 	&edge_length){ 	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
	const long Nclades = Ntips + Nnodes;
	
	// determine root (unique node with no incoming edge)
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
	if(root<0) return Rcpp::List::create(Rcpp::Named("changed") = false); // tree is already unrooted
	
	// find all edges descending from the root
	lvector root_edges;
	for(long edge=0; edge<Nedges; ++edge){
		if(tree_edge[2*edge+0] == root) root_edges.push_back(edge);
	}
	if(root_edges.size()!=2) return Rcpp::List::create(Rcpp::Named("changed") = false); // root is not bifurcating
	
	// keeep track of the two root children, numbered such that upon removal of the root child2 will descend frm child1
	// make sure that child1 is not a tip (since the original tree has at least 3 tips by assumption, it is guaranteed that one of the two root children is a node)
	long root_child1 = tree_edge[2*root_edges[0]+1];
	long root_child2 = tree_edge[2*root_edges[1]+1];
	if(root_child1<Ntips)  std::swap(root_child1, root_child2);
	
	// renumber all clades > root by subtracting 1
	const long Nclades_new = Nclades-1;
	lvector new2old_clade(Nclades_new,-1), old2new_clade(Nclades,-1);
	for(long clade=0; clade<root; ++clade){
		new2old_clade[clade] = clade;
		old2new_clade[clade] = clade;
	}
	for(long clade=root+1; clade<Nclades; ++clade){
		new2old_clade[clade-1] = clade;
		old2new_clade[clade]   = clade-1;
	}
	
	// to satisfy some conventions of the ape package, make sure that the node gaining an extra child (due to edge merging) is the first node in the new tree
	const long semiroot = root_child1;
	const long swapped  = new2old_clade[Ntips];
	if(swapped!=semiroot){
		new2old_clade[Ntips] = semiroot;
		new2old_clade[old2new_clade[semiroot]] = swapped;

		long temp = old2new_clade[semiroot];
		old2new_clade[semiroot] = Ntips;
		old2new_clade[swapped]  = temp;		
	}
	
	// redefine all edges, in terms of the new clade indices and with the two root edges merged
	const long Nedges_new = Nedges-1;
	lvector new_tree_edge;
	new_tree_edge.reserve(Nedges_new*2);
	dvector new_edge_length;
	new_edge_length.reserve(Nedges_new);
	for(long edge=0; edge<Nedges; ++edge){
		if(tree_edge[2*edge+0]==root) continue; // skip root edges for now
		new_tree_edge.push_back(old2new_clade[tree_edge[2*edge+0]]);
		new_tree_edge.push_back(old2new_clade[tree_edge[2*edge+1]]);
		new_edge_length.push_back(edge_length.empty() ? 1.0 : edge_length[edge]);
	}
	
	// add merged root edges as one final edge, pointing from root_child1 to root_child2
	new_tree_edge.push_back(old2new_clade[root_child1]);
	new_tree_edge.push_back(old2new_clade[root_child2]);
	new_edge_length.push_back(edge_length.empty() ? 2.0 : (edge_length[root_edges[0]]+edge_length[root_edges[1]]));


	return Rcpp::List::create(	Rcpp::Named("changed") 			= true,
								Rcpp::Named("Nnodes") 			= Nnodes-1,
								Rcpp::Named("Nclades") 			= Nclades_new,
								Rcpp::Named("Nedges") 			= Nedges_new,
								Rcpp::Named("old2new_clade")	= old2new_clade,
								Rcpp::Named("new2old_clade")	= new2old_clade,
								Rcpp::Named("new_tree_edge") 	= new_tree_edge,
								Rcpp::Named("new_edge_length")	= new_edge_length);
}




// Given a set of target tips in an unrooted tree, find the node (or node) in the tree such that when that node is made root, the target tips form a monophyletic group whose MRCA is a child of that node (if as_MRCA==false) or whose MRCA is that node (if as_MRCA==true).
// This node can also be defined as follows (if as_MRCA==false): It is the single node, for which exactly one connected edge satisfies "all tips on the other side are targets", and all other connected edges satisfy "all tips on the other side are non-targets".
// Returns -1 on failure, otherwise it will return a clade index
// [[Rcpp::export]]
long find_root_for_monophyletic_clade_CPP(	const long				Ntips,
											const long 				Nnodes,
											const long				Nedges,
											std::vector<long>		tree_edge,		// (INPUT) 2D array (in row-major format) of size Nedges x 2
											const bool				is_rooted,		// (INPUT) if true, the input tree is guaranteed to already be rooted. Otherwise, it will be temporarily rooted internally at some arbitrary node.
											const std::vector<long>	&target_tips,	// (INPUT) 1D array of tip indices, listing target tips to be made monophyletic
											const bool				as_MRCA){		// (INPUT) if true, the MRCA of the target tips is returned, otherwise the parent of the MRCA is returned
	const long Nclades = Ntips+Nnodes;
	long clade, node, parent;
	if(target_tips.empty()) return -1;
			
	// temporarily root tree if needed (for purposes of traversal)
	// all tip/node/edge indices remain the same
	if(!is_rooted){
		root_tree_at_node(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,	// will be modified in-situ
							1);
	}
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
		
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal route (root --> tips)		
	std::vector<long> traversal_queue_root2tips, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true, 	// include tips
										false, 	// no precalcuated edge mapping
										traversal_queue_root2tips,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");
	
	// Step 1: for each clade, determine the number of children with descending (or being) targets and descending (or being) non-targets (traverse tips --> root)
	std::vector<long> Nchildren_with_descending_targets(Nnodes, 0), Nchildren_with_descending_nontargets(Nnodes, 0), Nchildren_per_node(Nnodes,0), Nnonmonofurcating_children_with_descending_targets(Nnodes,0);
	std::vector<bool> tip_is_target(Ntips,false);
	for(long t=0; t<target_tips.size(); ++t){
		tip_is_target[target_tips[t]] = true;
	}
	for(long q=traversal_queue_root2tips.size()-1, cnode, pnode; q>=1; --q){
		clade 	= traversal_queue_root2tips[q];
		parent 	= clade2parent[clade];
		cnode 	= clade-Ntips;
		pnode 	= parent-Ntips;
		Nchildren_per_node[pnode] += 1;
		// propagate information about descending targets & non-targets to parent
		if(clade<Ntips){
			Nchildren_with_descending_targets[pnode] 	+= (tip_is_target[clade] ? 1 : 0);
			Nchildren_with_descending_nontargets[pnode] += (tip_is_target[clade] ? 0 : 1);
			Nnonmonofurcating_children_with_descending_targets[pnode] += (tip_is_target[clade] ? 1 : 0);
		}else{
			Nchildren_with_descending_targets[pnode] 	+= (Nchildren_with_descending_targets[cnode]>0 ? 1 : 0);
			Nchildren_with_descending_nontargets[pnode] += (Nchildren_with_descending_nontargets[cnode]>0 ? 1 : 0);
			if(traversal_node2last_edge[cnode]-traversal_node2first_edge[cnode]>0) Nnonmonofurcating_children_with_descending_targets[pnode] += (Nchildren_with_descending_targets[cnode]>0 ? 1 : 0);
		}
	}	
	
	// Step 2: determine which clades have upstream targets (traverse root --> tips)
	std::vector<bool> clade_has_upstream_targets(Nclades, false), clade_has_upstream_nontargets(Nclades, false);
	for(long q=1, cnode, pnode; q<traversal_queue_root2tips.size(); ++q){
		clade  	= traversal_queue_root2tips[q];
		parent 	= clade2parent[clade];
		cnode 	= clade-Ntips;
		pnode 	= parent-Ntips;
		if(clade_has_upstream_targets[parent]) clade_has_upstream_targets[clade] = true;
		else if((clade<Ntips ? (tip_is_target[clade] ? 1 : 0) : (Nchildren_with_descending_targets[cnode]>0 ? 1 : 0)) < Nchildren_with_descending_targets[pnode]) clade_has_upstream_targets[clade] = true;
		if(clade_has_upstream_nontargets[parent]) clade_has_upstream_nontargets[clade] = true;
		else if((clade<Ntips ? (tip_is_target[clade] ? 0 : 1) : (Nchildren_with_descending_nontargets[cnode]>0 ? 1 : 0)) < Nchildren_with_descending_nontargets[pnode]) clade_has_upstream_nontargets[clade] = true;
	}
	
	if(as_MRCA){
		// Step 3: Find clade for which at most one inout edge has non-targets on the other side, and such that that edge has only non-targets
		// Monofurcations need to be accounted for in special ways, i.e. make sure the new root is the MRCA of the target tips and not further upstream connected via monofurcations
		for(clade=0; clade<Nclades; ++clade){
			node = clade-Ntips;
			const long Nedges_with_targets = (clade_has_upstream_targets[clade] ? 1 : 0) + (clade<Ntips ? 0 : Nchildren_with_descending_targets[node]);
			const long Nedges_with_nontargets = (clade_has_upstream_nontargets[clade] ? 1 : 0) + (clade<Ntips ? 0 : Nchildren_with_descending_nontargets[node]);
			const long Nedges_total = (clade<Ntips ? 0 : Nchildren_per_node[node])+(clade==root ? 0 : 1);
			if(Nedges_with_nontargets>1) continue; // clade has more than one inout edges with non-targets on the other side
			if(Nedges_with_nontargets==0) return clade;
			if((Nedges_with_nontargets==1) && (Nedges_with_targets==Nedges_total-1) && (Nedges_total>2)) return clade; // if clade only has 2 edges, then if it were made root it would be a monofurcation, in which case the proper MRCA would actually be further downstream
		}
	}else{
		// Step 3: Find clade that has exactly one inout edge, on the other side of which are all targets and only targets
		// Monofurcations need to be accounted for in special ways, i.e. make sure the MRCA of the target tips is a direct (not indirect) descendant of the new root
		for(clade=0; clade<Nclades; ++clade){
			node 	= clade-Ntips;
			parent 	= clade2parent[clade];
			if((clade<Ntips) && (tip_is_target[clade])) continue;
			if(clade_has_upstream_targets[clade] && clade_has_upstream_nontargets[clade]) continue; // there's targets as well as non-targets upstream
			if((clade>=Ntips) && ((Nchildren_with_descending_targets[node]+Nchildren_with_descending_nontargets[node])>Nchildren_per_node[node])) continue; // some children include targets as well as non-targets
			const bool monofurcating_parent = (parent<0 ? false : (parent==root ? (traversal_node2last_edge[parent-Ntips]-traversal_node2first_edge[parent-Ntips]==1) : (traversal_node2last_edge[parent-Ntips]-traversal_node2first_edge[parent-Ntips]==0))); // would the parent node become a monofurcation if clade was made root?
			if(clade<Ntips){
				if(clade_has_upstream_targets[clade] && (!monofurcating_parent)) return clade; // all targets are upstream, but make sure that immediately upstream node is not a monofurcation
			}else{
				if(clade_has_upstream_targets[clade] && (Nchildren_with_descending_targets[node]==0) && (!monofurcating_parent)) return clade; // all targets are upstream, but make sure that immediately upstream node is not a monofurcation
				if((!clade_has_upstream_targets[clade]) && (Nchildren_with_descending_targets[node]==1) && (Nnonmonofurcating_children_with_descending_targets[node]==1)) return clade; // targets are on the other side of exactly one descending edge that does not lead to a monofurcation
			}
		}
	}
	return -1;
}








// Given a set of target tips in an unrooted tree, find the "separator" edge in the tree such that all (or most) target tips are on the one side of the edge, and all (or most) non-target tips are on the other side.
// Specifically, for any edge e (with some arbitrary direction) let N_e^u & N_e^d be the number of targets upstream & downstream, respectively, and let M_e^u & M_e^d be the number of non-targets upstream and downstream, respectively.
// Define E(e) := (N_e^u>N_e^d ? M_e^u+N_e^d : M_e^d+N_e^u). Then the "separator" edge is the edge e that minimizes E(e).
// This function can be used to determine the root of an unrooted tree that would make a set of target tips monophyletic or at least "almost monophyletic" (i.e. with minimal deviation from monophyly).
// This function returns an edge index (in 0,..,Nedges-1), or -1 in the case of failure.
// [[Rcpp::export]]
Rcpp::List find_edge_splitting_tree_CPP(const long				Ntips,
										const long 				Nnodes,
										const long				Nedges,
										std::vector<long>		tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
										const bool				is_rooted,			// (INPUT) if true, the input tree is guaranteed to already be rooted. Otherwise, it will be temporarily rooted internally at some arbitrary node.
										const std::vector<long>	&target_tips,		// (INPUT) 1D array of tip indices, listing target tips by which to separate the tree. Can contain duplicates (will be ignored)
										const bool				include_misplaced){	// (INPUT) if true, then the misplaced tips (corresponding to the "optimal" splitting) are also returned as lists. This requires some extra computation.
	const long Nclades = Ntips+Nnodes;
	long clade, child, parent;
	if(target_tips.empty()) return Rcpp::List::create(Rcpp::Named("edge") = -1);
			
	// temporarily root tree if needed (for purposes of traversal)
	// all tip/node/edge indices remain the same
	if(!is_rooted){
		root_tree_at_node(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,	// will be modified in-situ
							1);
	}
	
	// prepare auxiliary lookup tables
	std::vector<long> clade2parent, incoming_edge_per_clade;;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);
		
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal route (root --> tips)		
	std::vector<long> traversal_queue_root2tips, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true, 	// include tips
										false, 	// no precalculated edge mapping
										traversal_queue_root2tips,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");
	
	// Step 1: for each clade, determine the number of downstream targets and non-targets (traverse tips-->root)
	std::vector<long> Ntargets_downstream_per_clade(Nclades,0), Nnontargets_downstream_per_clade(Nclades,0);
	for(long tip=0; tip<Ntips; ++tip){
		Nnontargets_downstream_per_clade[tip] = 1;
	}
	for(long t=0, tip; t<target_tips.size(); ++t){
		tip = target_tips[t];
		Ntargets_downstream_per_clade[tip] 		= 1;
		Nnontargets_downstream_per_clade[tip] 	= 0;
	}
	for(long q=traversal_queue_root2tips.size()-1; q>=1; --q){
		clade 	= traversal_queue_root2tips[q];
		parent	= clade2parent[clade];
		Ntargets_downstream_per_clade[parent] 	 += Ntargets_downstream_per_clade[clade];
		Nnontargets_downstream_per_clade[parent] += Nnontargets_downstream_per_clade[clade];
	}
	
	// Step 2: for each clade, determine the number of upstream targets and non-targets (traverse root-->tips)
	std::vector<long> Ntargets_upstream_per_clade(Nclades,0), Nnontargets_upstream_per_clade(Nclades,0);
	for(long q=1; q<traversal_queue_root2tips.size(); ++q){
		clade  	= traversal_queue_root2tips[q];
		parent 	= clade2parent[clade];
		Ntargets_upstream_per_clade[clade] 		= Ntargets_upstream_per_clade[parent] + (Ntargets_downstream_per_clade[parent]-Ntargets_downstream_per_clade[clade]);
		Nnontargets_upstream_per_clade[clade] 	= Nnontargets_upstream_per_clade[parent] + (Nnontargets_downstream_per_clade[parent]-Nnontargets_downstream_per_clade[clade]);
	}
	
	// Step 3: find edge with minimal error in target monophyly (i.e. most targets on one side and most non-targets on the other side)
	long best_edge = -1, best_Ntargets_upstream, best_Ntargets_downstream, best_Nmisplaced_targets, best_Nmisplaced_nontargets;
	for(long edge=0; edge<Nedges; ++edge){
		child 	= tree_edge[edge*2+1];
		const long Ntargets_upstream 		= Ntargets_upstream_per_clade[child];
		const long Ntargets_downstream 		= Ntargets_downstream_per_clade[child];
		const long Nnontargets_upstream 	= Nnontargets_upstream_per_clade[child];
		const long Nnontargets_downstream 	= Nnontargets_downstream_per_clade[child];
		const long Nmisplaced_targets		= (Ntargets_upstream>Ntargets_downstream ? Ntargets_downstream : Ntargets_upstream);
		const long Nmisplaced_nontargets	= (Ntargets_upstream>Ntargets_downstream ? Nnontargets_upstream : Nnontargets_downstream);
		if((best_edge<0) || (Nmisplaced_targets+Nmisplaced_nontargets<best_Nmisplaced_targets+best_Nmisplaced_nontargets)){
			best_edge 	= edge;
			best_Nmisplaced_targets		= Nmisplaced_targets;
			best_Nmisplaced_nontargets	= Nmisplaced_nontargets;
			best_Ntargets_upstream 		= Ntargets_upstream;
			best_Ntargets_downstream 	= Ntargets_downstream;
		}
	}
	
	// determine misplaced tips if requested
	std::vector<long> misplaced_targets, misplaced_nontargets;
	if(include_misplaced){
		misplaced_targets.reserve(best_Nmisplaced_targets);
		misplaced_nontargets.reserve(best_Nmisplaced_nontargets);
		
		// determine which clades (especially tips) descend from best_edge (traverse root-->tips)
		std::vector<bool> descends_from_best_edge(Nclades,false);
		descends_from_best_edge[tree_edge[2*best_edge+1]] = true;
		for(long q=1; q<traversal_queue_root2tips.size(); ++q){
			clade 	= traversal_queue_root2tips[q];
			parent 	= clade2parent[clade];
			if(!descends_from_best_edge[clade]) descends_from_best_edge[clade] = descends_from_best_edge[parent];
		}
		
		// collect misplaced target & non-target tips
		const bool targets_should_be_upstream = (best_Ntargets_upstream>best_Ntargets_downstream);
		for(long tip=0; tip<Ntips; ++tip){
			if(descends_from_best_edge[tip] && (Ntargets_downstream_per_clade[tip]==1) && targets_should_be_upstream) misplaced_targets.push_back(tip);					// misplaced downstream target
			else if(descends_from_best_edge[tip] && (Nnontargets_downstream_per_clade[tip]==1) && (!targets_should_be_upstream)) misplaced_nontargets.push_back(tip);	// misplaced downstream non-target
			else if((!descends_from_best_edge[tip]) && (Ntargets_downstream_per_clade[tip]==1) && (!targets_should_be_upstream)) misplaced_targets.push_back(tip);		// misplaced upstream target
			else if((!descends_from_best_edge[tip]) && (Nnontargets_downstream_per_clade[tip]==1) && targets_should_be_upstream) misplaced_nontargets.push_back(tip); 	// misplaced upstream non-target
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("edge") 					= best_edge,
								Rcpp::Named("Nmisplaced_targets")		= best_Nmisplaced_targets,
								Rcpp::Named("Nmisplaced_nontargets")	= best_Nmisplaced_nontargets,
								Rcpp::Named("Ntargets_upstream") 		= best_Ntargets_upstream,
								Rcpp::Named("Ntargets_downstream") 		= best_Ntargets_downstream,
								Rcpp::Named("misplaced_targets") 		= misplaced_targets,
								Rcpp::Named("misplaced_nontargets") 	= misplaced_nontargets);
}




// Collapse tree nodes (and their descending subtrees) into tips, whenever all descending tips have a distance from a node below a certain threshold (or whenever the sum of edges descending from the node is below the threshold, see option criterion)
// If shorten==true:
//   Collapsed nodes will be turned into tips, while retaining the length of their incoming edge (thus the tree is shortened)
// If shorten==false:
//   Whenever all tips descending from some node have a distance from the node below a certain threshold, remove all tips and make the node into a tip, extending its incoming edge by length L, where L was the longest distance to any of the tips
//   In other words, from each subtree below a certain diversity threshold, replace it with a single tip extending to the maximum depth of the subtree
// This function can be used to get the "coarse structure" of a tree
// This function guarantees that the new_root will have index = Ntips_new
// [[Rcpp::export]]
Rcpp::List collapse_tree_at_resolution_CPP(	const long			Ntips,
											const long 			Nnodes,
											const long			Nedges,
											const std::vector<long>	&tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
											const std::vector<double> &edge_length, 		// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
											const double		resolution,			// (INPUT) maximum (inclusive) phylogenetic distance of descending tips from the node to be collapsed
											const bool			shorten,			// (INPUT) if true, then collapsed nodes are replaced with a tip at the same location, hence potentially shortening the tree
											const std::string	&criterion){		// (INPUT) criterion by which to collapse nodes. 'sum_tip_paths': interpret resolution as a max allowed sum of tip distances from the collapsed node. 'max_tip_depth': resolution refers to the max tip distance from the collapsed node. 'max_tip_pair_dist': resolution refers to the max distance between any pair of tips descending from the collapsed node.
	const long Nclades = Ntips + Nnodes;
	long clade, parent, edge, node;
	const bool unit_edge_lengths = (edge_length.size()==0);
	const bool sum_tip_paths = (criterion=="sum_tip_paths");
	const bool max_tip_depth = (criterion=="max_tip_depth");
	const bool tip_pair_dist = (criterion=="max_tip_pair_dist");
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}

	// find root using the mapping clade-->incoming_edge
	const long root = get_root_from_incoming_edge_per_clade(Ntips, tree_edge, incoming_edge_per_clade);
		
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, node2first_edge, node2last_edge, edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,	// include tips
										false,	// edge mapping is not precalculated
										traversal_queue,
										node2first_edge,
										node2last_edge,
										edges,
										false,
										"");
										
	// Step 1: calculate distance to furthest descending tip for each node (traverse tips-->root)
	std::vector<double> clade2max_tip_depth(Nclades,0);
	std::vector<double> clade2sum_tip_paths((sum_tip_paths ? Nclades : 0),0);
	std::vector<double> clade2max_tip_pair_dist((tip_pair_dist ? Nclades : 0),0);
	std::vector<double> clade2farthest_tip(Nclades,-1);
	for(long tip=0; tip<Ntips; ++tip){
		clade2farthest_tip[tip] = tip;
	}
	for(long q=traversal_queue.size()-1, clade; q>=1; --q){
		clade  = traversal_queue[q];
		edge   = incoming_edge_per_clade[clade];
		parent = tree_edge[edge*2+0];
		const double candidate_distance = clade2max_tip_depth[clade] + (unit_edge_lengths ? 1.0 : edge_length[edge]);
		if(tip_pair_dist){
			// keep track of max distance between any pair of descending tips
			// this must come before updating clade2max_tip_depth[parent]
			if(clade2farthest_tip[parent]>=0){
				clade2max_tip_pair_dist[parent] = max(clade2max_tip_pair_dist[parent], max(clade2max_tip_depth[parent]+candidate_distance, clade2max_tip_pair_dist[clade]));
			}else{
				clade2max_tip_pair_dist[parent] = clade2max_tip_pair_dist[clade];
			}
		}
		if((clade2farthest_tip[parent]<0) || (clade2max_tip_depth[parent]<candidate_distance)){
			// keep track of farthest descending tip
			clade2max_tip_depth[parent] = candidate_distance;
			clade2farthest_tip[parent]	= clade2farthest_tip[clade];
		}
		if(sum_tip_paths){
			// keep track of sum of paths to descending tips
			clade2sum_tip_paths[parent] += clade2sum_tip_paths[clade] + (unit_edge_lengths ? 1.0 : edge_length[edge]);
		}
	}
	if(tip_pair_dist){
		// make sure nodes with just 
	}
	const std::vector<double> &clade2criterion = (sum_tip_paths ? clade2sum_tip_paths : (max_tip_depth ? clade2max_tip_depth : clade2max_tip_pair_dist));
	
	// Step 2: count how many tips/nodes/edges to extract
	// traverse root-->tips in depth-first-search order, stoppping at nodes below the threshold
	// use a scratch_stack for traversing nodes
	long Ntips_new  = 0;
	long Nnodes_new = 0;
	long Nedges_new = 0;
	std::vector<long> scratch_stack;
	scratch_stack.reserve(floor(2*log(Ntips)/log(2.0))); // rough estimate of typical tree depth x 2. scratch_stack may be resized along the way if needed.
	scratch_stack.push_back(root);
	while(scratch_stack.size()>0){
		clade = scratch_stack.back();
		node  = clade - Ntips;
		scratch_stack.pop_back();
		if((clade2criterion[clade]<=resolution) || (clade<Ntips)){
			// this is a tip, or a node that should be collapsed into a new tip
			++Ntips_new;
			continue; // don't expand children
		}else{
			++Nnodes_new;
		}
		// traverse outgoing edges and children
		for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
			edge  = edges[e];
			scratch_stack.push_back(tree_edge[edge*2+1]); // add child to stack for further exploration in the next iteration
			++Nedges_new;
		}
	}
	const long Nclades_new 	= Ntips_new + Nnodes_new;
	const long Ncollapsed 	= Nnodes - Nnodes_new; // number of collapsed nodes
	
	// Step 3: Traverse again root-->tips (depth-first-search) and create mappings old-->new
	std::vector<long> old2new_clade(Nclades,-1), old2new_edge(Nedges,-1), collapsed_nodes, farthest_tips;
	collapsed_nodes.reserve(Ncollapsed);
	farthest_tips.reserve(Ncollapsed);
	long next_new_tip  = 0;
	long next_new_node = 0;
	long next_new_edge = 0;
	scratch_stack.clear();
	scratch_stack.push_back(root);
	while(scratch_stack.size()>0){
		clade = scratch_stack.back();
		node  = clade - Ntips;
		scratch_stack.pop_back();
		if((clade2criterion[clade]<=resolution) || (clade<Ntips)){
			// this is a tip, or a node that should be collapsed into a new tip
			old2new_clade[clade] = (next_new_tip++);
			if(clade>=Ntips){
				// keep record of collapsed node
				collapsed_nodes.push_back(clade-Ntips);
				farthest_tips.push_back(clade2farthest_tip[clade]);
			}
			continue; // don't expand children
		}else{
			old2new_clade[clade] = Ntips_new + (next_new_node++);
		}
		// traverse outgoing edges and children
		for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
			edge  = edges[e];
			scratch_stack.push_back(tree_edge[edge*2+1]); // add child to stack for further exploration in the next iteration
			old2new_edge[edge] = (next_new_edge++);
		}
	}
	
	// Step 4: Extract collapsed tree and generate new2old mappings
	std::vector<long> new_tree_edge(2*Nedges_new), new2old_clade(Nclades_new), new2old_edge(Nedges_new);
	std::vector<double> new_edge_length(shorten ? 0 : Nedges_new);
	for(long edge=0, new_edge, child; edge<Nedges; ++edge){
		new_edge = old2new_edge[edge];
		if(new_edge<0) continue; // this edge is not to be kept
		child = tree_edge[edge*2+1];
		new2old_edge[new_edge] = edge;
		new_tree_edge[new_edge*2+0] = old2new_clade[tree_edge[edge*2+0]];
		new_tree_edge[new_edge*2+1] = old2new_clade[child];
		if(!shorten){
			new_edge_length[new_edge] = (unit_edge_lengths ? 1.0 : edge_length[edge]);
			if((child>=Ntips) && (old2new_clade[child]<Ntips_new)) new_edge_length[new_edge] += clade2max_tip_depth[child]; // this child is a collapsed node, so extend incoming edge to maximum depth of any old descendant
		}
	}
	for(long clade=0, new_clade; clade<Nclades; ++clade){
		new_clade = old2new_clade[clade];
		if(new_clade>=0) new2old_clade[new_clade] = clade;
	}
	
	// determine new root & root shift
	// traverse from new to old root and calculate cumulative distance
	// use old tree structure
	const long new_root = old2new_clade[root];
	clade = root;
	long root_shift = 0;
	while(incoming_edge_per_clade[clade]>=0){
		edge = incoming_edge_per_clade[clade];
		root_shift += (unit_edge_lengths ? 1.0 : edge_length[edge]);
		clade = tree_edge[edge*2 + 0];
	}

	return Rcpp::List::create(	Rcpp::Named("new_tree_edge") 		= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new_edge_length") 		= Rcpp::wrap(new_edge_length), // only relevant if (shorten==false)
								Rcpp::Named("new2old_clade") 		= Rcpp::wrap(new2old_clade),
								Rcpp::Named("new2old_edge") 		= Rcpp::wrap(new2old_edge),
								Rcpp::Named("old2new_clade") 		= Rcpp::wrap(old2new_clade),
								Rcpp::Named("collapsed_nodes") 		= Rcpp::wrap(collapsed_nodes),
								Rcpp::Named("farthest_tips") 		= Rcpp::wrap(farthest_tips), // will be synchronized with collapsed_nodes[]
								Rcpp::Named("new_root") 			= new_root, // in newer implementations this is actually guaranteed to be = Ntips_new
								Rcpp::Named("Ntips_new") 			= Ntips_new,
								Rcpp::Named("Nnodes_new") 			= Nnodes_new,
								Rcpp::Named("Nedges_new") 			= Nedges_new,
								Rcpp::Named("root_shift") 			= root_shift);
}






// Trim a phylogenetic tree by cutting off tips and nodes, so that all remaining tips have height<=max_height.
// Note that some edge lengths will be modified (edges cut will be shortened)
// If initially all tips had height>=max_height, then the trimmed tree will be ultrametric
// [[Rcpp::export]]
Rcpp::List trim_tree_at_height_CPP(	const long					Ntips,
									const long 					Nnodes,
									const long					Nedges,
									const std::vector<long>		&tree_edge,					// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
									const std::vector<double> 	&edge_length, 				// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
									const double				max_distance_from_root){	// (INPUT) phylogenetic distance from the root, at which to trim the tree
	const long Nclades = Ntips + Nnodes;
	long parent, child, edge, node;
	const bool unit_edge_lengths = (edge_length.size()==0);
	
	// find root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
		
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, node2first_edge, node2last_edge, edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,	// include tips
										false,	// edge mapping is not precalculated
										traversal_queue,
										node2first_edge,
										node2last_edge,
										edges,
										false,
										"");
																				
	// Step 1: Determine which clades to keep (traverse root-->tips)
	// (include calculation of distances_from_root)
	std::vector<char> keep_clade(Nclades);
	std::vector<double> distances_from_root(Nclades);
	distances_from_root[root] = 0;
	keep_clade[root] = true;
	for(long q=0; q<traversal_queue.size(); ++q){
		parent = traversal_queue[q];
		if(parent<Ntips) continue;
		node = parent - Ntips;
		for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
			edge 	= edges[e];
			child 	= tree_edge[edge*2+1];
			distances_from_root[child] = (unit_edge_lengths ? 1.0 : edge_length[edge]) + distances_from_root[parent];
			keep_clade[child] = ((distances_from_root[parent]<max_distance_from_root) || (distances_from_root[child]<=max_distance_from_root)); // keep a child iff the child is not above the max_distance_from_root, or the parent is strictly below max_distance_from_root
		}
	}
	
	// Step 2: Extract subtree
	// edges to be kept is determined based on clades to be kept
	long Ntips_new, Nnodes_new, Nedges_new, new_root;
	std::vector<long> new2old_clade, old2new_clade, new2old_edge, new_tree_edge;
	get_arbitrary_subtree(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							keep_clade,
							Ntips_new,
							Nnodes_new,
							Nedges_new,
							new_root,
							new2old_clade,
							old2new_clade,
							new2old_edge,
							new_tree_edge);
							
	// figure out which nodes became tips
	long Ntips_ex_nodes = 0;
	for(long tip_new=0; tip_new<Ntips_new; ++tip_new){
		if(new2old_clade[tip_new]>=Ntips) ++Ntips_ex_nodes;
	}
	std::vector<long> new_tips_ex_nodes(Ntips_ex_nodes);
	for(long tip_new=0, counter=0; tip_new<Ntips_new; ++tip_new){
		if(new2old_clade[tip_new]>=Ntips) new_tips_ex_nodes[counter++] = tip_new;
	}
							
	// Step 3: Calculate edge lengths for extracted subtree
	// (trim terminal edges if needed)
	dvector new_edge_length(Nedges_new);
	lvector new_edges_trimmed;
	new_edges_trimmed.reserve(Nedges_new);
	long Nedges_trimmed = 0;
	for(long edge_new=0, edge; edge_new<Nedges_new; ++edge_new){
		edge 	= new2old_edge[edge_new];
		parent 	= tree_edge[edge*2+0];
		child 	= tree_edge[edge*2+1];
		const double old_length = (unit_edge_lengths ? 1.0 : edge_length[edge]);
		if(old_length>max_distance_from_root-distances_from_root[parent]){
			new_edge_length[edge_new] = max_distance_from_root-distances_from_root[parent];
			new_edges_trimmed.push_back(edge_new);
			++Nedges_trimmed;
		}else{
			new_edge_length[edge_new] = old_length;
		}
	}
					
	return Rcpp::List::create(	Rcpp::Named("Ntips_new")		= Ntips_new,
								Rcpp::Named("Nnodes_new")		= Nnodes_new,
								Rcpp::Named("Nedges_new")		= Nedges_new,		// number of edges kept
								Rcpp::Named("Nedges_trimmed")	= Nedges_trimmed,	// number of kept edges that were trimmed (i.e. reduced in length)
								Rcpp::Named("new_root")			= new_root,			// new root clade index. This is actually guaranteed to be = Ntips_new.
								Rcpp::Named("new2old_clade")	= Rcpp::wrap(new2old_clade),
								Rcpp::Named("new2old_edge")		= Rcpp::wrap(new2old_edge),
								Rcpp::Named("new_tree_edge")	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new_edge_length")	= Rcpp::wrap(new_edge_length),
								Rcpp::Named("new_edges_trimmed")= Rcpp::wrap(new_edges_trimmed), // new edges that used to be longer and were trimmed
								Rcpp::Named("new_tips_ex_nodes")= Rcpp::wrap(new_tips_ex_nodes)); // new tips that used to be nodes
}





// Split a phylogenetic tree at a specific height (distance from the root), yielding multiple subtrees rooted at the split_height
// [[Rcpp::export]]
Rcpp::List split_tree_at_height_CPP(const long			Ntips,
									const long 			Nnodes,
									const long			Nedges,
									const std::vector<long>	&tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
									const std::vector<double> &edge_length, 		// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
									const double		root_edge,			// (INPUT) length of trailing edge leading into root (usually 0)
									const double		split_height){		// (INPUT) phylogenetic distance from the root, at which to split the tree. If zero, the original tree is returned. If >tree_span, an error is returned.
	const long Nclades = Ntips + Nnodes;
	long parent, child, edge, node;
	const bool unit_edge_lengths = (edge_length.size()==0);
	
	// find root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
	
	// Step 1: Determine splitting points and assign edges & clades to subtrees
	std::vector<long> edge2subtree(Nedges,-1), clade2subtree(Nclades,-1);
	std::vector<double> distances_from_root(Nclades,0);
	std::vector<char> split_edge(Nedges);
	long Nsubtrees	= 0;
	distances_from_root[root] = 0;
	if(split_height<=0){
		// there will only be one subtree, namely the original tree
		// no edge shall be split
		Nsubtrees = 1;
		split_edge.assign(Nedges,0);
		edge2subtree.assign(Nedges,0);
		clade2subtree.assign(Nclades,0);
	}else{
		// get tree traversal route (root --> tips), not including tips
		tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, false, false);
		// iterate through the tree (root --> tips)														
		for(long q=0; q<traversal.queue.size(); ++q){
			parent = traversal.queue[q];
			if(parent<Ntips) continue;
			node = parent - Ntips;
			for(long e=traversal.node2first_edge[node]; e<=traversal.node2last_edge[node]; ++e){
				edge 	= traversal.edge_mapping[e];
				child 	= tree_edge[edge*2+1];
				distances_from_root[child] = (unit_edge_lengths ? 1.0 : edge_length[edge]) + distances_from_root[parent];
				split_edge[edge] = ((distances_from_root[parent]<split_height) && (distances_from_root[child]>=split_height));
				if(split_edge[edge]){
					// designate a new subtree, starting at this new splitting point
					++Nsubtrees;
					edge2subtree[edge] 	 = Nsubtrees-1;
					clade2subtree[child] = Nsubtrees-1;
				}else{
					edge2subtree[edge] 	 = clade2subtree[parent];
					clade2subtree[child] = clade2subtree[parent];		
				}
			}
		}
	}
	
	if(Nsubtrees==0){
		return Rcpp::List::create(Rcpp::Named("success") = true, Rcpp::Named("Nsubtrees") = Nsubtrees);
	}
	
	// Step 2: Count Ntips, Nedges, Nnodes per subtree
	// Also record subtree roots (in terms of old clade indices)
	std::vector<long> NStips(Nsubtrees,0), NSnodes(Nsubtrees,0), NSclades(Nsubtrees,0), NSedges(Nsubtrees,0);
	std::vector<long> subtree2root_clade(Nsubtrees);
	for(long edge=0, subtree; edge<Nedges; ++edge){
		subtree = edge2subtree[edge];
		if(split_edge[edge]){
			// edge is split, so actually will not belong to any subtree
			// designate child as the subtree's root
			subtree2root_clade[subtree] = tree_edge[2*edge+1];
		}else if(subtree>=0){
			++NSedges[subtree];
		}
	}
	if(split_height<=0) subtree2root_clade[0] = root; // deal with special case where none of the edges were split, and the whole tree is the sole subtree
	for(long clade=0, subtree; clade<Nclades; ++clade){
		subtree = clade2subtree[clade];
		if(subtree>=0){
			if(clade<Ntips){
				++NStips[subtree];
			}else{
				++NSnodes[subtree];
			}
			++NSclades[subtree];
		}
	}
	
	// Step 3: Calculate clade & edge offsets per subtree (e.g. for storing all subtree info into a single linear array)
	std::vector<long> clade_offsets(Nsubtrees,0), edge_offsets(Nsubtrees,0);
	for(long subtree=1; subtree<Nsubtrees; ++subtree){
		clade_offsets[subtree] = clade_offsets[subtree-1] + NSclades[subtree-1];
		edge_offsets[subtree]  = edge_offsets[subtree-1] + NSedges[subtree-1];
	}
	
	// Step 4: Map subtree clade indices <--> old clade indices
	// Since we traverse old clades in ascending order, and subtree tips are old tips, it is guaranteed that the tips of each subtree will be indexed before the subtree's nodes
	std::vector<long> new2old_clade(clade_offsets.back()+NSclades.back()), old2new_clade(Nclades,-1);
	std::vector<long> subtree2next_clade_index(Nsubtrees,0); // auxilliary, keeping track of next new clade index to be assigned for each subtree
	for(long clade=0, subtree; clade<Nclades; ++clade){
		subtree = clade2subtree[clade];
		if(subtree>=0){
			new2old_clade[clade_offsets[subtree] + subtree2next_clade_index[subtree]] = clade;
			old2new_clade[clade] = subtree2next_clade_index[subtree];
			++subtree2next_clade_index[subtree];
		}
	}
	
	// Step 5: Make sure the root of each subtree is indexed = NStips[i]
	for(long subtree=0, temp, current_root, new_root; subtree<Nsubtrees; ++subtree){
		if(NStips[subtree]<=1) continue; // pathological tree, no need to adjust the root index
		current_root = old2new_clade[subtree2root_clade[subtree]];
		new_root = NStips[subtree];
		temp = new2old_clade[clade_offsets[subtree]+current_root];
		new2old_clade[clade_offsets[subtree]+current_root] = new2old_clade[clade_offsets[subtree]+new_root];
		new2old_clade[clade_offsets[subtree]+new_root] = temp;
		// update old2new_clade[] to be consistent with the modified new2old_clade[]
		old2new_clade[new2old_clade[clade_offsets[subtree]+current_root]] = current_root;
		old2new_clade[new2old_clade[clade_offsets[subtree]+new_root]] = new_root;
	}
	
	// Step 6: Define edges of subtrees, mapping to new clade indices of each subtree
	std::vector<long> new2old_edge(edge_offsets.back()+NSedges.back());
	std::vector<long> subtree2next_edge_index(Nsubtrees,0); // auxilliary, keeping track of next new clade index to be assigned for each subtree
	std::vector<long> subtree_edges(2*(edge_offsets.back()+NSedges.back()));
	std::vector<long> new_roots(Nsubtrees);
	std::vector<double> new_root_edges(Nsubtrees);
	for(long edge=0, child, subtree; edge<Nedges; ++edge){
		subtree = edge2subtree[edge];
		if(split_edge[edge]){
			// this edge is actually becoming the incoming tail of the root, not an actual 2-node-connecting edge
			child					= tree_edge[2*edge+1];
			new_roots[subtree]  	= old2new_clade[child];
			new_root_edges[subtree] = distances_from_root[child]-split_height;
		}else if(subtree>=0){
			new2old_edge[edge_offsets[subtree] + subtree2next_edge_index[subtree]] = edge;
			subtree_edges[2*(edge_offsets[subtree]+subtree2next_edge_index[subtree]) + 0] = old2new_clade[tree_edge[2*edge+0]];
			subtree_edges[2*(edge_offsets[subtree]+subtree2next_edge_index[subtree]) + 1] = old2new_clade[tree_edge[2*edge+1]];			
			++subtree2next_edge_index[subtree];
		}
	}
	if(split_height<=0){
		// special case: explicitly set root info, since no edge was actually split
		new_roots[0] = root;
		new_root_edges[0] = root_edge;
	}
				
	return Rcpp::List::create(	Rcpp::Named("success")			= true,
								Rcpp::Named("Nsubtrees")		= Nsubtrees,
								Rcpp::Named("NStips")			= NStips,
								Rcpp::Named("NSnodes")			= NSnodes,
								Rcpp::Named("NSedges")			= NSedges,
								Rcpp::Named("new_roots")		= new_roots,		// new root clade indices for each subtree. 
								Rcpp::Named("root_edges")		= new_root_edges,	// length of tailing edges leading into root, for each subtree.
								Rcpp::Named("subtree_edges")	= subtree_edges,
								Rcpp::Named("new2old_clade")	= Rcpp::wrap(new2old_clade),
								Rcpp::Named("new2old_edge")		= Rcpp::wrap(new2old_edge),
								Rcpp::Named("clade2subtree")	= Rcpp::wrap(clade2subtree));
}







// extend terminal edges (edges leading to tips) so that each tip has the same fixed distance (new_height) from the root
// if a tip already extends beyond the specified new_height, its incoming edge remains unchanged
// this is a quick-and-dirty way to make the tree ultrametric
// [[Rcpp::export]]
Rcpp::List extend_tree_to_height_CPP(	const long			Ntips,
										const long 			Nnodes,
										const long			Nedges,
										const std::vector<long>	&tree_edge,		// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
										const std::vector<double> &edge_length, 	// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
										double				new_height){	// (INPUT) phylogenetic distance from the root, to which all tips are to be extended. If negative, this is set to the max_distance_to_root of the input tree		
	// get incoming edge per tip (terminal edges)
	std::vector<long> incoming_edge_per_tip;
	get_incoming_edge_per_tip(Ntips, Nedges, tree_edge, incoming_edge_per_tip);
	
	// calculate all current distances from root
	std::vector<double> distances_from_root;
	get_distances_from_root(Ntips, Nnodes, Nedges, tree_edge, edge_length, distances_from_root);
	if(new_height<0) new_height = array_max(distances_from_root);
	
	// extend terminal edges to new_height
	double max_extension = 0;
	std::vector<double> new_edge_length(Nedges);
	if(edge_length.size()==0){
		new_edge_length.assign(Nedges,1);
	}else{
		for(long edge=0; edge<Nedges; ++edge) new_edge_length[edge] = edge_length[edge];
	}
	for(long tip=0; tip<Ntips; ++tip){
		const double extension = new_height - distances_from_root[tip];
		if(extension>0) new_edge_length[incoming_edge_per_tip[tip]] += extension;
		max_extension = max(max_extension, extension);
	}

	return Rcpp::List::create(	Rcpp::Named("new_edge_length")	= Rcpp::wrap(new_edge_length),
								Rcpp::Named("max_extension")	= Rcpp::wrap(max_extension)); // max length that was added to any edge
}


// Eliminate multifurcations in a tree by replacing them with multiple descending bifurcations
// Tip indices remain the same, but edge indices and the total number of nodes/edges may increase (if the tree includes multifurcations)
// Old nodes retain their index, and new nodes will have indices Nnodes,...,(Nnew_nodes-1)
// New nodes will always descend from the old multifurcating nodes, that is, for every multifurcating old node that is split into bifurcations, the newly added nodes will descend from the old node (that kept its original index)
// The tree need not be rooted
void multifurcations_to_bifurcations(	const long					Ntips,
										const long 					Nnodes,
										const long					Nedges,
										const std::vector<long>		&tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
										const std::vector<double> 	&edge_length,		// (INPUT) 1D array of size Nedges listing edge lengths, or an empty vector (all edges have length 1)
										const double				dummy_edge_length,	// (INPUT) length to be used for new auxiliary edges (connecting old to new nodes) when splitting multifurcations. This will typically be zero, or a small number if zero-length edges are not desired.
										long						&Nnew_nodes,		// (OUTPUT) number of nodes in the new tree
										long						&Nnew_edges,		// (OUTPUT) number of edges in the new tree
										std::vector<long>			&new_tree_edge,		// (OUTPUT) 2D array of size Nnew_edges x 2, in row-major format, with elements in 0,..,(Nclades-1)
										std::vector<double>			&new_edge_length,	// (OUTPUT) 1D array of size Nnew_edges, listing the lengths of edges in the new tree
										std::vector<long>			&old2new_edge){		// (OUTPUT) 1D array of size Nedges, with values in 0,..,(Nnew_edges-1) mapping old edges to new edges
	long edge, child, Nchildren;
	
	// get edge mappings
	std::vector<long> node2first_edge, node2last_edge, edge_mapping;
	get_node2edge_mappings(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							node2first_edge,
							node2last_edge,
							edge_mapping);
							
	// determine number of nodes/edges in the new tree (based on the number & size of multifurcations)
	Nnew_edges = Nedges;
	for(long node=0; node<Nnodes; ++node){
		Nchildren = node2last_edge[node] - node2first_edge[node] + 1;
		if(Nchildren>2) Nnew_edges += (Nchildren-2);
	}
	Nnew_nodes = Nnodes + (Nnew_edges - Nedges);
	
	if(Nnew_edges==Nedges){
		// return tree without modification
		new_tree_edge 	= tree_edge;
		new_edge_length	= (edge_length.size()==0 ? std::vector<double>(Nedges,1) : edge_length);
		old2new_edge.resize(Nedges);
		for(edge=0; edge<Nedges; ++edge) old2new_edge[edge] = edge;
		return;
	}
							
	// iterate through nodes and expand any multifurcations
	new_tree_edge.clear();
	new_edge_length.clear();
	new_tree_edge.reserve(2*Nnew_edges);
	new_edge_length.reserve(Nnew_edges);
	old2new_edge.resize(Nedges);
	long next_new_node = Nnodes;
	for(long node=0, clade, next_parent; node<Nnodes; ++node){
		clade = Ntips + node;
		Nchildren = node2last_edge[node] - node2first_edge[node] + 1;
		if(Nchildren>0){
			// the first child is always preserved
			edge = edge_mapping[node2first_edge[node]];
			new_tree_edge.push_back(clade);
			new_tree_edge.push_back(tree_edge[2*edge+1]);
			new_edge_length.push_back(edge_length.size()==0 ? 1.0 : edge_length[edge]);
			old2new_edge[edge] = new_edge_length.size()-1;
		}
		if(Nchildren<=2){
			// node does not multifurcate, so also preserve 2nd child (if available) and move on to the next node
			if(Nchildren>1){
				edge = edge_mapping[node2first_edge[node]+1];
				new_tree_edge.push_back(clade);
				new_tree_edge.push_back(tree_edge[2*edge+1]);
				new_edge_length.push_back(edge_length.size()==0 ? 1.0 : edge_length[edge]);
				old2new_edge[edge] = new_edge_length.size()-1;
			}
			continue;
		}
		// for all children but the first and last create a new node
		next_parent = clade;
		for(long e=1+node2first_edge[node]; e<node2last_edge[node]; ++e){
			edge  = edge_mapping[e];
			child = tree_edge[2*edge+1];
			// create new edge next_parent --> next_new_node
			new_tree_edge.push_back(next_parent);
			new_tree_edge.push_back(Ntips+next_new_node);
			new_edge_length.push_back(dummy_edge_length);
			// add child to the new node
			new_tree_edge.push_back(Ntips+next_new_node);
			new_tree_edge.push_back(child);
			new_edge_length.push_back(edge_length.size()==0 ? 1.0 : edge_length[edge]);
			old2new_edge[edge] = new_edge_length.size()-1;
			next_parent = Ntips + next_new_node;
			++next_new_node;
		}
		// add last child to the last new node (next_parent)
		edge  = edge_mapping[node2last_edge[node]];
		child = tree_edge[2*edge+1];
		new_tree_edge.push_back(next_parent);
		new_tree_edge.push_back(child);
		new_edge_length.push_back(edge_length.size()==0 ? 1.0 : edge_length[edge]);
		old2new_edge[edge] = new_edge_length.size()-1;
	}
}



// Replace multifurcations with multiple bifurcations
// The tree need not be rooted
// Rcpp wrapper function
// [[Rcpp::export]]
Rcpp::List multifurcations_to_bifurcations_CPP(	const long			Ntips,
												const long 			Nnodes,
												const long			Nedges,
												const std::vector<long>	&tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
												const std::vector<double> &edge_length,		// (INPUT) 1D array of size Nedges listing edge lengths, or an empty vector (all edges have length 1)
												const double		dummy_edge_length){	// (INPUT) length to be used for new auxiliary edges (connecting old to new nodes) when splitting multifurcations. This will typically be zero, or a small number if zero-length edges are not desired.
	long Nnew_nodes, Nnew_edges;
	std::vector<long> new_tree_edge, old2new_edge;
	std::vector<double> new_edge_length;
	multifurcations_to_bifurcations(Ntips,
									Nnodes,
									Nedges,
									tree_edge,
									edge_length,
									dummy_edge_length,
									Nnew_nodes,
									Nnew_edges,
									new_tree_edge,
									new_edge_length,
									old2new_edge);
											
	return Rcpp::List::create(	Rcpp::Named("Nnew_nodes") 		= Nnew_nodes,
								Rcpp::Named("Nnew_edges") 		= Nnew_edges,
								Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new_edge_length") 	= Rcpp::wrap(new_edge_length),
								Rcpp::Named("old2new_edge") 	= Rcpp::wrap(old2new_edge));
}


// Merge some nodes into their parent nodes (or with their children nodes), thus generating multifurcations
// The input tree must be rooted
// Terminology:
//   An "absorbing ancestor" is a node that is not upward-merged but has children nodes merged into it.
// [[Rcpp::export]]
Rcpp::List merge_nodes_to_multifurcations_CPP(	const long					Ntips,
												const long 					Nnodes,
												const long					Nedges,
												const std::vector<long>		&tree_edge,				// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
												const std::vector<double>	&edge_length,			// (INPUT) 1D array of size Nedges listing edge lengths, or an empty vector (all edges have length 1)
												const std::vector<long>		&nodes_to_merge,		// (INPUT) 1D array, listing the indices of nodes to be merged upwards (i.e., absorbed into their parents)
												const bool					merge_with_parents,		// (INPUT) whether the nodes listed in nodes_to_merge[] should be merged with their parents. If false, the specified nodes will be merged with their children (whenever these are not tips).
												const bool					keep_ancestral_ages){	// (INPUT) whether the generated multifurcations should have the same age as the absorbing ancestor. If false, then the age of a multifurcation will be the average of the absorbing ancestor's age and the ages of its merged child nodes, constrained from below by the ages of non-merged descendants.
	const bool has_edge_lengths	= (edge_length.size()>0);
	const long Nclades = Ntips + Nnodes;
	
	// determine parent clade and incoming edge for each clade
	lvector clade2parent(Nclades,-1), clade2inedge(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		clade2parent[tree_edge[edge*2+1]] = tree_edge[edge*2+0];
		clade2inedge[tree_edge[edge*2+1]] = edge;
	}
		
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal (root-->tips), including tips
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);
	
	// tag nodes to be merged upwards and determine how many nodes & edges will remain after merging
	// every node merging reduces the number of nodes and the number of edges by 1
	std::vector<bool> merge_node(Nnodes, false);
	long Nnodes_new=Nnodes, Nedges_new=Nedges; // will be updated (decreased) below
	if(merge_with_parents){
		// nodes_to_merge[] should be merged upwards into their parents
		for(long n=0, node; n<nodes_to_merge.size(); ++n){
			node = nodes_to_merge[n];
			if(clade2parent[node+Ntips]>=0){
				merge_node[node] = true; // make sure you don't tag the root
				--Nnodes_new;
				--Nedges_new;
			}
		}
	}else{
		// the child-nodes of nodes_to_merge[] should be merged into their parents
		for(long n=0, e, pnode, child; n<nodes_to_merge.size(); ++n){
			pnode = nodes_to_merge[n];
			for(e=traversal.node2first_edge[pnode]; e<=traversal.node2last_edge[pnode]; ++e){
				child = tree_edge[2*traversal.edge_mapping[e]+1];
				if(child>=Ntips){
					// this child is a node and should be merged into pnode
					merge_node[child-Ntips] = true;
					--Nnodes_new;
					--Nedges_new;
				}
			}
		}
	}
	const long new_root = Ntips;
	
	// determine time shifts for generated multifurcations (relative to the absorbing ancestor), if needed
	dvector absorbing_node_time_shifts(Nnodes,0); // only entries for absorbing ancestors (non-upward-merged nodes with upward-merged children) will be relevant
	if(!keep_ancestral_ages){
		// determine maximum allowed time shifts based on non-merged nodes (whose ages are assumed solid), traverse tips-->root
		dvector clade_times;
		get_distances_from_root(Ntips, Nnodes, Nedges, tree_edge, edge_length, traversal, clade_times);
		dvector max_allowed_ancestor_times(Nclades,INFTY_D);
		for(long q=traversal.queue.size()-1, clade, parent; q>=1; --q){
			clade = traversal.queue[q];
			if((clade<Ntips) || (!merge_node[clade-Ntips])){
				// this clade will not be merged upwards, so consider its time solid. Hence, no ancestor can have a later time than this clade
				max_allowed_ancestor_times[clade] = clade_times[clade];
			}
			parent = clade2parent[clade];
			if(parent>=0) max_allowed_ancestor_times[parent] = min(max_allowed_ancestor_times[parent], max_allowed_ancestor_times[clade]);
		}
		double sum_merged_child_edges, max_allowed_time;
		for(long node=0, clade, Nmerged_children, child, e, edge; node<Nnodes; ++node){
			if(merge_node[node]) continue; // no need to determine age of upward-merged node, since this node will be lost anyway
			clade					= node+Ntips;
			Nmerged_children 		= 0;
			sum_merged_child_edges 	= 0;
			max_allowed_time		= INFTY_D;
			for(e=traversal.node2first_edge[node]; e<=traversal.node2last_edge[node]; ++e){
				edge  = traversal.edge_mapping[e];
				child = tree_edge[2*edge+1];
				max_allowed_time = min(max_allowed_time, max_allowed_ancestor_times[child]);
				if((child>=Ntips) && merge_node[child-Ntips]){
					// this child will be merged upwards
					++Nmerged_children;
					sum_merged_child_edges += (has_edge_lengths ? edge_length[edge] : 1.0);
				}
			}
			if(Nmerged_children>0){
				absorbing_node_time_shifts[node] = min(max_allowed_time-clade_times[clade], (0+sum_merged_child_edges)/(Nmerged_children+1)); // shift this node so that its new age is the average between all participating nodes (focal node & its merged children), but prevent shifting past the maximum allowed time
			}
		}
	}
	
	// traverse nodes root-->tips and merge tagged nodes upwards along the way
	// whenever we merge a node upwards, we are modifying its parent's edge list, so we need to do this from root-->tips
	// work with existing node & edge indices, eliminate resulting "ghosts" later
	lvector modified_tree_edge = tree_edge;
	dvector modified_edge_length;
	if(has_edge_lengths){
		modified_edge_length = edge_length;
	}else{
		modified_edge_length.assign(Nedges,1.0);
	}
	for(long q=0, clade, node, absorbing_ancestor, parent, child, e, edge, inedge; q<traversal.queue.size(); ++q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue; // tips are never merged upwards
		node  	= clade-Ntips;
		inedge  = clade2inedge[clade];
		if((!merge_node[node]) && (absorbing_node_time_shifts[node]!=0)){
			// this is an absorbing ancestor that will be shifted in time, so adjust the length of its incoming edge, as well as its outgoing non-merged-child edges (so that the ages of unmerged children remain unchanged)
			if(inedge>=0) modified_edge_length[inedge] += absorbing_node_time_shifts[node];
			for(e=traversal.node2first_edge[node]; e<=traversal.node2last_edge[node]; ++e){
				edge  = traversal.edge_mapping[e];
				child = tree_edge[2*edge+1];
				if((child<Ntips) || (!merge_node[child-Ntips])){
					modified_edge_length[edge] -= absorbing_node_time_shifts[node];
				}
			}
		}else if(merge_node[node]){
			// merge this node upwards into its parent, eventually eliminating its incoming edge
			parent = clade2parent[clade];
			absorbing_ancestor = modified_tree_edge[2*inedge+0]; // this may be an ancestor further back than the node's original parent, if the node's parent was also merged upwards
			for(e=traversal.node2first_edge[node]; e<=traversal.node2last_edge[node]; ++e){
				edge = traversal.edge_mapping[e];
				modified_tree_edge[2*edge+0] = absorbing_ancestor;
				modified_edge_length[edge] += modified_edge_length[inedge] - (parent==absorbing_ancestor ? absorbing_node_time_shifts[absorbing_ancestor-Ntips] : 0.0);
			}
		}
	}
	
	// re-index remaining nodes & edges (eliminate "ghosts")
	lvector old2new_node(Nnodes,-1);
	lvector new2old_node(Nnodes_new);
	long next_new_node=0;
	for(long old_node=0; old_node<Nnodes; ++old_node){
		if(!merge_node[old_node]){
			old2new_node[old_node] = next_new_node;
			new2old_node[next_new_node] = old_node;
			++next_new_node;
		}
	}
	lvector new_tree_edge(2*Nedges_new);
	dvector new_edge_length(Nedges_new);
	long next_new_edge=0;
	for(long edge=0, child; edge<Nedges; ++edge){
		child = modified_tree_edge[2*edge+1];
		if((child<Ntips) || (!merge_node[child-Ntips])){
			// this edge does not point into an upward-merged node, so keep it
			// make sure to use the new node indices
			new_tree_edge[2*next_new_edge+1] = (child<Ntips ? child : Ntips+old2new_node[child-Ntips]);
			new_tree_edge[2*next_new_edge+0] = Ntips+old2new_node[modified_tree_edge[2*edge+0]-Ntips];
			new_edge_length[next_new_edge] 	 = modified_edge_length[edge];
			++next_new_edge;
		}
	}

	return Rcpp::List::create(	Rcpp::Named("Nnodes_new") 		= Nnodes_new,
								Rcpp::Named("Nedges_new") 		= Nedges_new,
								Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new_edge_length") 	= Rcpp::wrap(new_edge_length),
								Rcpp::Named("new_root") 		= Rcpp::wrap(new_root), // this is actually guaranteed to be Ntips
								Rcpp::Named("new2old_node") 	= Rcpp::wrap(new2old_node),
								Rcpp::Named("old2new_node") 	= Rcpp::wrap(old2new_node));
}


// Modify the times of specific tips & nodes, by adding or subtracting specific values
// Here, "time" refers to time since the root
// Excessive shifting can result in negative edge lengths, which are corrected in a variety of alternative ways (see option negative_edge_lengths). 
// However, to avoid changing the overall span of the tree (root age and tip times), you should not shift a clade beyond the boundaries of the tree (i.e., resulting in a negative time or a time beyond its descending tips).
// The input tree must be rooted
// The input tree does not need to be ultrametric, but edge lengths are interpreted as time
// [[Rcpp::export]]
Rcpp::List shift_clade_times_CPP(const long					Ntips,
								const long 					Nnodes,
								const long					Nedges,
								const std::vector<long>		&tree_edge,				// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
								const std::vector<double>	&edge_length,			// (INPUT) 1D array of size Nedges listing edge lengths
								const std::vector<long>		&clades_to_shift,		// (INPUT) 1D array, listing the indices of clades whose ages should be shifted
								const std::vector<double>	&time_shifts,			// (INPUT) 1D array, of the same length as clades_to_shift, listing the increments to be applied to the clade times. E.g. a positive value means that the clade's time is increased (the clade is moved towards the present)
								const bool					shift_descendants,		// (INPUT) if true, then the subclade descending from a shifted clade will move along with it, thus shifting the time of all descendants by the same amount. If false, the descending tips & nodes retain their original time (unless negative edges are created, see option negative_edge_lengths)
								const std::string			&negative_edge_lengths){// (INPUT) whether and how to fix negative edge lengths resulting from excessive shifting. Must be either "error", "allow" (allow negative edge lengths), "move_all_descendants", "move_all_ancestors", "move_child" (only move children to younger ages as needed, traversing the tree root-->tips) or "move_parent" (only move parents to older ages as needed, traversing the tree tips-->root). Note that "move_child" could result in tips moving, if an ancestral node is shifted too much towards younger ages. Similarly, "move_parent" could result in the root moving towards an older age if some descendant was shifted too far towards the root.
	const long Nclades = Ntips + Nnodes;
	
	// determine parent clade and incoming edge for each clade
	lvector clade2parent(Nclades,-1), clade2inedge(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		clade2parent[tree_edge[edge*2+1]] = tree_edge[edge*2+0];
		clade2inedge[tree_edge[edge*2+1]] = edge;
	}
		
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal (root-->tips), including tips
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);

	// shift clades
	double time_shift;
	dvector new_edge_length = edge_length;
	for(long s=0, clade, inedge, node, e; s<clades_to_shift.size(); ++s){
		clade 		= clades_to_shift[s];
		node		= clade-Ntips;
		time_shift	= time_shifts[s];
		inedge 		= clade2inedge[clade];
		// adjust edge lengths
		new_edge_length[inedge] += time_shift;
		if((clade>=Ntips) && (!shift_descendants)){
			// also adjust outgoing edge lengths, so that descendant tips & nodes retain their original times
			for(e=traversal.node2first_edge[node]; e<=traversal.node2last_edge[node]; ++e){
				new_edge_length[traversal.edge_mapping[e]] -= time_shift;
			}
		}
	}
	
	// check and correct negative edge lengths if needed
	if(negative_edge_lengths=="error"){
		for(long edge=0; edge<Nedges; ++edge){
			if(new_edge_length[edge]<0) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = stringprintf("Age shifts resulted in negative edge lengths, e.g. at edge %d (length %g)",edge,new_edge_length[edge]));
		}
	}else if(negative_edge_lengths=="move_child"){
		// move root-->tips, moving nodes & tips away from the root as needed
		double error;
		for(long q=1, clade, node, inedge, e; q<traversal.queue.size(); ++q){
			clade 	= traversal.queue[q];
			inedge	= clade2inedge[clade];
			if(new_edge_length[inedge]>=0) continue; // nothing to fix here
			error = abs(new_edge_length[inedge]);
			new_edge_length[inedge] = 0;
			if(clade<Ntips) continue; // clade is a tip, so no need to adjust outgoing edges
			node = clade - Ntips;
			// shorten outgoing edges, so that descendant times remain the same
			for(e=traversal.node2first_edge[node]; e<=traversal.node2last_edge[node]; ++e){
				new_edge_length[traversal.edge_mapping[e]] -= error;
			}
		}
	}else if(negative_edge_lengths=="move_parent"){
		// move tips-->root, moving nodes & tips towards the root as needed
		double error;
		for(long q=traversal.queue.size()-1, clade, parent, pnode, inedge, e, edge; q>=1; --q){
			clade 	= traversal.queue[q];
			inedge	= clade2inedge[clade];
			if(new_edge_length[inedge]>=0) continue; // nothing to fix here
			parent = clade2parent[clade];
			error  = abs(new_edge_length[inedge]);
			new_edge_length[inedge] = 0;
			// extend sister edges, so that sister clades remain "parallel" to this subclade and the parent moves backward on its own
			pnode = parent-Ntips;
			for(e=traversal.node2first_edge[pnode]; e<=traversal.node2last_edge[pnode]; ++e){
				edge = traversal.edge_mapping[e];
				if(edge!=inedge) new_edge_length[edge] += error;
			}
			if(parent==root) continue; // parent is root, so no need to adjust its incoming edge
			new_edge_length[clade2inedge[parent]] -= error; // shorten parent's incoming edge, so that descendant times remain the same
		}
	}else if(negative_edge_lengths=="move_all_descendants"){
		// move root-->tips, moving descending subclades away from the root as needed
		for(long q=1, clade, inedge; q<traversal.queue.size(); ++q){
			clade 	= traversal.queue[q];
			inedge	= clade2inedge[clade];
			if(new_edge_length[inedge]>=0) continue; // nothing to fix here
			new_edge_length[inedge] = 0;
		}
	}else if(negative_edge_lengths=="move_all_ancestors"){
		// move tips-->root, moving all ancestors towards the root as needed
		double error;
		for(long q=traversal.queue.size()-1, clade, inedge, e, edge; q>=1; --q){
			clade 	= traversal.queue[q];
			inedge	= clade2inedge[clade];
			if(new_edge_length[inedge]>=0) continue; // nothing to fix here
			error = abs(new_edge_length[inedge]);
			new_edge_length[inedge] = 0;
			// extend sister edges of all ancestors, so that sister clades remain "parallel" to this subclade and the ancestors move backward on their own
			while(clade2parent[clade]>=0){
				inedge 	= clade2inedge[clade];
				clade 	= clade2parent[clade];
				for(e=traversal.node2first_edge[clade-Ntips]; e<=traversal.node2last_edge[clade-Ntips]; ++e){
					edge = traversal.edge_mapping[e];
					if(edge!=inedge) new_edge_length[edge] += error;
				}
			}
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("success") 			= true,
								Rcpp::Named("new_edge_length") 	= Rcpp::wrap(new_edge_length));
}




// Eliminate short edges by merging affected nodes/tips into multifurcations
// The tree must be rooted
// [[Rcpp::export]]
Rcpp::List merge_short_edges_CPP(	const long					Ntips,
									const long 					Nnodes,
									const long					Nedges,
									const std::vector<long>		&tree_edge,				// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
									const std::vector<double> 	&edge_length,			// (INPUT) 1D array of size Nedges listing edge lengths, or an empty vector (all edges have length 1)
									const double				edge_length_epsilon,	// (INPUT) non-negative number. Max edge length for an edge to be considered zero-length. Typically 0 or some small positive number.
									const bool					force_keep_tips){		// (INPUT) if true, all tips are kept even if their incoming edges are short. Note that tip indices may still change.
	const long Nclades = Ntips + Nnodes;
	const bool unit_edge_lengths = (edge_length.size()==0);

	// get edge mappings
	std::vector<long> node2first_edge, node2last_edge, edge_mapping;
	get_node2edge_mappings(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							node2first_edge,
							node2last_edge,
							edge_mapping);
							
	// get root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
							
	// get a lower bound for the number of edges & clades in the new tree
	// note that some short edges may be kept if they are extended by ancestral short edges
	long Nnew_edges_min = 0;
	for(long edge=0, child; edge<Nedges; ++edge){
		child = tree_edge[edge*2+1];
		const double length = (unit_edge_lengths ? 1.0 : edge_length[edge]);
		if((length>edge_length_epsilon) || (force_keep_tips && (child<Ntips))) ++Nnew_edges_min;
	}
	
	if(Nnew_edges_min==Nedges){
		// return tree without modification
		std::vector<long> new2old_clade(Nclades), new2old_edge(Nedges);
		for(long clade=0; clade<Nclades; ++clade) new2old_clade[clade] = clade;
		for(long edge=0; edge<Nedges; ++edge) new2old_edge[edge] = edge;
		return Rcpp::List::create(	Rcpp::Named("Nnew_tips") 		= Ntips,
									Rcpp::Named("Nnew_nodes") 		= Nnodes,
									Rcpp::Named("Nnew_edges") 		= Nedges,
									Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(tree_edge),
									Rcpp::Named("new_edge_length") 	= Rcpp::wrap(unit_edge_lengths ? std::vector<double>(Nedges,1.0) : edge_length),
									Rcpp::Named("new2old_clade") 	= Rcpp::wrap(new2old_clade),
									Rcpp::Named("new2old_edge") 	= Rcpp::wrap(new2old_edge),
									Rcpp::Named("root")				= root);
	}
	
	// iterate through edges (root-->tips) and merge any short edges
	// use a scratch_stack to traverse root-->tips via depth-first-search
	std::vector<long> new2old_clade, new2old_edge;
	std::vector<long> 	new_tree_edge; // first populate with old node/tip indices, then update
	std::vector<double> new_edge_length;
	new2old_clade.reserve(Nnew_edges_min+1);
	new2old_edge.reserve(Nnew_edges_min);
	new_tree_edge.reserve(Nnew_edges_min*2);
	new_edge_length.reserve(Nnew_edges_min);
	std::vector<long> 	current_tree_edge = tree_edge; 	// parents will be updated here, as short edges are being eliminated (parents of sub-edges are moved closer to the root). Hence, edges must be traversed root-->tip.
	std::vector<double> current_edge_length; 			// edge lengths will be updated here, as short edges are being eliminated and their lengths appended to sub-edges
	if(!unit_edge_lengths) current_edge_length = edge_length;
	else current_edge_length.assign(Nedges,1);

	new2old_clade.push_back(root); // always keep root
	std::vector<long> scratch_stack; // will list nodes whose edges still need to be checked
	scratch_stack.reserve(floor(2*log(Ntips)/log(2.0))); // rough estimate of typical tree depth x 2. scratch_stack may be resized along the way if needed.
	scratch_stack.push_back(root); // start traversal at root
	while(scratch_stack.size()>0){
		const long clade = scratch_stack.back();
		scratch_stack.pop_back();
		const long node = clade - Ntips;
		for(long e=node2first_edge[node], edge, parent, child; e<=node2last_edge[node]; ++e){
			edge 	= edge_mapping[e];
			parent	= current_tree_edge[edge*2+0]; // may be different from clade, if this edge was extended due to elimination of its parent edge
			child 	= current_tree_edge[edge*2+1];
			if(child>=Ntips) scratch_stack.push_back(child); // add child node to stack for further exploration later on
	
			const double length = current_edge_length[edge];
			if((length>edge_length_epsilon) || (force_keep_tips && (child<Ntips))){
				// keep this edge and its child
				new2old_clade.push_back(child);
				const long next_new_edge = new2old_edge.size();
				new2old_edge.push_back(edge);
				new_tree_edge.resize(new_tree_edge.size()+2);
				new_tree_edge[next_new_edge*2+0] = parent;
				new_tree_edge[next_new_edge*2+1] = child;
				new_edge_length.push_back(length);
			}else{
				// eliminate this short edge by merging child into parent
				// this eliminates the child tip/node
				// the child's children become the parent's children
				if(child<Ntips){
					// child is a tip, so just eliminate this tip (i.e. do nothing)
				}else{
					// child is a node, so attach its children to the parent node
					const long cnode = child-Ntips;
					for(long se=node2first_edge[cnode], sub_edge; se<=node2last_edge[cnode]; ++se){
						sub_edge = edge_mapping[se];
						current_tree_edge[sub_edge*2+0] = parent;
						current_edge_length[sub_edge] 	+= length;
					}
				}
			
			}
		}
	}
	const long Nnew_edges  = new2old_edge.size();
	const long Nnew_clades = 1+Nnew_edges;
	
	// update clade indices in new_tree_edge
	std::vector<long> old2new_clade(Nclades,-1);
	for(long new_clade=0; new_clade<Nnew_clades; ++new_clade){
		old2new_clade[new2old_clade[new_clade]] = new_clade;
	}
	for(long new_edge=0; new_edge<Nnew_edges; ++new_edge){
		new_tree_edge[new_edge*2+0] = old2new_clade[new_tree_edge[new_edge*2+0]];
		new_tree_edge[new_edge*2+1] = old2new_clade[new_tree_edge[new_edge*2+1]];
	}
	
	// correct new clade indices so that tips are indexed 0,..,Nnew_tips-1 and nodes are indexed Nnew_tips,..,Nnew_clades-1
	long Nnew_tips, Nnew_nodes;
	std::vector<long> new2newer_clade;
	reindex_clades(	Nnew_clades,
					Nnew_edges,
					new_tree_edge,
					true, // ensure root is re-indexed to Nnew_tips
					Nnew_tips,
					Nnew_nodes,
					new2newer_clade);
	for(long clade=0; clade<Nclades; ++clade){
		if(old2new_clade[clade]>=0){
			old2new_clade[clade] = new2newer_clade[old2new_clade[clade]];
			new2old_clade[old2new_clade[clade]] = clade;
		}
	}
	for(long new_edge=0; new_edge<Nnew_edges; ++new_edge){
		new_tree_edge[new_edge*2+0] = new2newer_clade[new_tree_edge[new_edge*2+0]];
		new_tree_edge[new_edge*2+1] = new2newer_clade[new_tree_edge[new_edge*2+1]];
	}

	return Rcpp::List::create(	Rcpp::Named("Nnew_tips") 		= Nnew_tips,
								Rcpp::Named("Nnew_nodes") 		= Nnew_nodes,
								Rcpp::Named("Nnew_edges") 		= Nnew_edges,
								Rcpp::Named("new_tree_edge") 	= Rcpp::wrap(new_tree_edge),
								Rcpp::Named("new_edge_length") 	= Rcpp::wrap(new_edge_length),
								Rcpp::Named("new2old_clade") 	= Rcpp::wrap(new2old_clade),
								Rcpp::Named("new2old_edge") 	= Rcpp::wrap(new2old_edge),
								Rcpp::Named("root")				= Nnew_tips);
}


// Pick random subsets of tips from a tree, by traversing from root-->tips and at each node choosing randomly between children
// The size of each random subset is Nrandoms, the number of independent subsets is Nsubsets
// [[Rcpp::export]]
std::vector<long> pick_random_tips_CPP(	const long			Ntips,
										const long 			Nnodes,
										const long			Nedges,
										const std::vector<long> &tree_edge, 			// (INPUT) 2D array (in row-major format) of size Nedges x 2, or an empty std::vector (no tree available).
										const long			Nrandoms,				// (INPUT) number of random tips to pick at each experiment (i.e. in each independent subset)
										const long			Nsubsets,				// (INPUT) number of times the experiment should be repeated, i.e. each time drawing Nrandoms tips anew.
										const bool			with_replacement){		// (INPUT) pick tips with replacement. If false, then children with no descending tips left to pick from, are excluded from traversal; all other children of a node remain equally probable.
	
	const long Nclades = Ntips+Nnodes;
	long tip, clade;
	if((!with_replacement) && (Nrandoms>Ntips)) return std::vector<long>(); // this should not happen
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
		
	// prepare access structures for random tip selection
	std::vector<long> node2first_child, node2last_child, children, node2total_tip_count;
	get_children_per_node(	Ntips,
							Nnodes,
							Nedges,
							root,
							tree_edge,
							node2first_child,
							node2last_child,
							children);
	if(!with_replacement){
		get_total_tip_count_per_node(	Ntips,
										Nnodes,
										Nedges,
										tree_edge,
										node2total_tip_count);
	}
	std::vector<long> 	tips_remaining_per_node;
	std::vector<double> clade2weight;
	
	// pick Nrepeats random tip subsets of size Nrandoms
	std::vector<long> random_tips(Nsubsets*Nrandoms);
	for(long s=0; s<Nsubsets; ++s){
		if(!with_replacement){
			// re-initialize counters for this repeat
			tips_remaining_per_node = node2total_tip_count;
			clade2weight.resize(Nclades);
			for(clade=0; clade<Nclades; ++clade){
				clade2weight[clade] = (clade<Ntips ? 1.0 : (tips_remaining_per_node[clade-Ntips]>0 ? 1 : 0));
			}
		}
		for(long t=0; t<Nrandoms; ++t){
			if(with_replacement){
				tip = get_tip_by_random_uniform_traversal(	Ntips,
															root,
															node2first_child,
															node2last_child,
															children);
			}else{
				tip = get_tip_by_random_traversal(	Ntips,
													root,
													node2first_child,
													node2last_child,
													children,
													clade2weight);
				clade2weight[tip] = 0; // prevent re-inclusion of this tip in the future (i.e. don't replace)
				// propagate information upwards
				clade = tip;
				while(clade!=root){
					clade = clade2parent[clade];
					tips_remaining_per_node[clade-Ntips] -= 1;
					if(tips_remaining_per_node[clade-Ntips]<=0){
						// no more tips to draw from this clade, so set weight to zero
						clade2weight[clade] = 0.0;
					}
				}					
			}
			random_tips[s*Nrandoms + t] = tip;
		}
		// abort if the user has interrupted the calling R program
		Rcpp::checkUserInterrupt();
	}
	return random_tips;
}







// assign tips & nodes of a tree to groups, such that each group is monophyletic (a "taxon") represented by exactly one of given representative tips
// this is the "reverse" operation of picking one representative from each taxon, for a given partitioning of tips into taxa
// The tree must be rooted; the root should be the unique node with no parent
void assign_clades_to_taxa(	const long				Ntips,
							const long 				Nnodes,
							const long				Nedges,
							const std::vector<long> 	&tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
							const std::vector<long>	&representatives,	// (INPUT) 1D array of size NR, each element of which is the index of a tip representing a distinct taxon
							std::vector<long>		&clade2taxon){		// (OUTPUT) 1D array of size Nclades, mapping each tip & node of the tree to one of NR taxa. In particular, tip2taxon[representatives[r]] = r for all r=1,..,NR. Nodes with more than 1 descending representatives (and thus not part of a specific taxon) will have value -1. If NR==0, then all clades will be assigned to value -1. Also clades with ambiguous taxon assignment (as can occur due to multifurcations) will have value -1
	const long Nclades = Ntips+Nnodes;
	const long NR = representatives.size();
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal route (root --> tips)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,		// include tips in traversal
										false,		// edge mappings are not yet computed
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");
	
	// traverse tips-->root and keep track of which nodes have a descending representative
	clade2taxon.assign(Nclades,-1);
	std::vector<long> clade2Nrep_tips(Nclades,0);
	std::vector<long> clade2Nrep_children(Nclades,0);
	for(long r=0; r<NR; ++r){
		clade2taxon[representatives[r]] = r;
		clade2Nrep_tips[representatives[r]] = 1;
		clade2Nrep_children[representatives[r]] = 1;
	}
	for(long q=traversal_queue.size()-1, parent, clade; q>=1; --q){
		clade  = traversal_queue[q];
		parent = clade2parent[clade];
		if((clade2taxon[clade]>=0) && (clade2taxon[parent]<0)) clade2taxon[parent] = clade2taxon[clade];
		clade2Nrep_tips[parent] += clade2Nrep_tips[clade];
		if(clade2Nrep_children[clade]>0) clade2Nrep_children[parent] += 1;
	}
	
	
	// traverse root-->tips and assign taxa & status to clades
	// status = -1 means multiple descending representatives
	// status = 0 means an unambiguous taxon assignment
	// status = 1 means that the taxon assignment - although performed - was ambiguous
	for(long q=0, clade; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade2Nrep_tips[clade]>1){
			// node contains multiple descending representatives, so no taxon can be assigned
			clade2taxon[clade] = -1;
		}else if(clade2Nrep_tips[clade]==1){
			// node contains exactly one descending representative, so keep whatever taxon was assigned
			continue;
		}else if(clade==root){
			// clade is root and contains no descending representative. This case is pathological, and only occurs if NR==0.
			continue;
		}else{
			// this non-root clade contains no descending representatives, so need to assign taxon based on parent
			clade2taxon[clade] = clade2taxon[clade2parent[clade]];
		}
	}
}




// Rcpp wrapper for the homonymous function above
// [[Rcpp::export]]
Rcpp::List assign_clades_to_taxa_CPP(	const long				Ntips,
										const long 				Nnodes,
										const long				Nedges,
										const std::vector<long> 	&tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
										const std::vector<long>	&representatives){	// (INPUT) 1D array of size NR, each element of which is the index of a tip representing a distinct taxon
	std::vector<long> clade2taxon;
	assign_clades_to_taxa(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							representatives,
							clade2taxon);
	return Rcpp::List::create(Rcpp::Named("clade2taxon") = clade2taxon);
}





// pick one tip descending from a clade
// if multiple tips descend from a clade, then which tip is returned may be arbitrary
// returns false on failure
bool aux_get_one_descending_tip(const long				Ntips,			// (INPUT) number of tips in the tree
								const long 				Nnodes,			// (INPUT) number of internal nodes in the tree
								const long				Nedges,			// (INPUT) number of edges in the tree
								const lvector			&tree_edge,		// (INPUT) 2D array (in row-major format) of size Nedges x 2
								const tree_traversal	&traversal, 	// (INPUT) tree traversal struct, for traversing tree root-->tips
								const lvector			&tips_per_clade,// (INPUT) 1D array of size Nclades, listing the number of available tips per clade for picking. tips_per_clade[c] may be less than the actual number of descending tips, e.g. if some tips are deemed "non-available". tips_per_clade[] must be self-consistent, i.e. tips_per_clade[c] must be the sum of tips_per_clade[cc] across all child-clades cc of c.
								const long				start_clade,	// (INPUT) start clade, i.e. from which to pick a descending tip. If the clade is a tip, it is returned itself (if it is "available").
								long					&tip){			// (OUTPUT) the returned descending tip.
	long clade = start_clade;
	bool moved;
	while(clade>=Ntips){
		moved = false;
		for(long e=traversal.node2first_edge[clade-Ntips], edge, child; e<=traversal.node2last_edge[clade-Ntips]; ++e){
			edge  = traversal.edge_mapping[e];
			child = tree_edge[2*edge + 1];
			if(tips_per_clade[child]>0){
				// found a child with available tips, so move down there
				clade = child;
				moved = true;
				break;
			}
		}
		if(!moved) return false; // reached a dead-end without finding a tip
	}
	if(tips_per_clade[clade]==1){
		tip = clade;
		return true;
	}else{
		return false;
	}
}


// pick two tips descending from a node
// if more than 2 tips descend from a node, then which tips are returned may be arbitrary
// Returns false on error (e.g. if the node does not have >=2 descending tips)
bool aux_get_two_descending_tips(	const long				Ntips,			// (INPUT) number of tips in the tree
									const long 				Nnodes,			// (INPUT) number of internal nodes in the tree
									const long				Nedges,			// (INPUT) number of edges in the tree
									const lvector			&tree_edge,		// (INPUT) 2D array (in row-major format) of size Nedges x 2
									const tree_traversal	&traversal, 	// (INPUT) tree traversal struct, for traversing tree root-->tips
									const lvector			&tips_per_clade,// (INPUT) 1D array of size Nclades, listing the number of available tips per clade for picking. tips_per_clade[c] may be less than the actual number of descending tips, e.g. if some tips are deemed "non-available". tips_per_clade[] must be self-consistent, i.e. tips_per_clade[c] must be the sum of tips_per_clade[cc] across all child-clades cc of c.
									const long				start_clade,	// (INPUT) start clade, i.e. from which to pick the descending tips
									long					&tip1,			// (OUTPUT) the 1st descending tip
									long					&tip2){			// (OUTPUT) the 2nd descending tip
	long clade = start_clade;
	long node, tip;
	tip1 = -1;
	tip2 = -1;
	// move root-->tips until we find a bifurcation (or multifurcation), then pick one tip from each of the first two descending lineages
	while(clade>=Ntips){
		node = clade - Ntips;
		for(long e=traversal.node2first_edge[node], child; e<=traversal.node2last_edge[node]; ++e){
			child = tree_edge[2*traversal.edge_mapping[e]+1];
			if(tips_per_clade[child]>=2){
				// this child has at least two available tips
				if(tip1<0){
					// move down that path to get both tips
					clade = child;
					break;
				}else{
					// only need one more tip, so just get it from this child
					if(!aux_get_one_descending_tip(Ntips, Nnodes, Nedges, tree_edge, traversal, tips_per_clade, child, tip)) return false; // something went wrong
					tip2 = tip;
					return true;
				}
			}else if(tips_per_clade[child]==1){
				// this node has exactly one available tip, so get it
				if(!aux_get_one_descending_tip(Ntips, Nnodes, Nedges, tree_edge, traversal, tips_per_clade, child, tip)) return false; // something went wrong
				if(tip1<0){
					tip1 = tip;
				}else if(tip2<0){
					tip2 = tip;
					return true;
				}
			}
		}
	}
	// at this point we have reached a single tip
	if(tip1<0){
		// we need at least 2 tips, so we have failed
		return false;
	}else if(tips_per_clade[clade]>=0){
		// we only need one more tip, and this one is available
		tip2 = clade;
		return true;
	}else{
		// we needed one more tip, but this one is not available, so we have failed
		return false;
	}
}

// Split tree into pairs of sister-tips, such that the paths within distinct pairs do not overlap
// The algorithm proceeds by repeatedly picking nodes with 2 descending tips until no such node is left (at which point the algorithm stops)
// If the input tree only contains monofurcations and bifurcations (recommended), it is guaranteed that at most one unpaired tip will be left (i.e., if Ntips was odd)
// [[Rcpp::export]]
Rcpp::List extract_independent_sister_tips_CPP(	const long				Ntips,
												const long 				Nnodes,
												const long				Nedges,
												const std::vector<long>	&tree_edge){	// (INPUT) 2D array (in row-major format) of size Nedges x 2
	const long Nclades = Ntips + Nnodes;
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal route (root --> tips), including tips			
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, true, false);
		
	// count number of tips descending from each node
	lvector tips_per_clade(Nclades,0);
	for(long tip=0; tip<Ntips; ++tip) tips_per_clade[tip] = 1;
	for(long q=traversal.queue.size()-1, clade, parent; q>=0; --q){
		clade = traversal.queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		tips_per_clade[parent] += tips_per_clade[clade];
	}
	
	// create list of nodes with 2 tips
	lvector dnodes; // list of nodes with exactly 2 descending and not yet paired tips
	dnodes.reserve(Nnodes);
	lvector node2dnode_index(Nnodes,-1); // node2dnode_index[n] will be the location of the n-th node in the dnodes[] array
	for(long node=0; node<Nnodes; ++node){
		if(tips_per_clade[node+Ntips]==2){
			node2dnode_index[node] = dnodes.size();
			dnodes.push_back(node);
		}
	}
	
	// iterate through the list of dnodes to extract sister-species pairs
	std::vector<long> tip_pairs; // 2D array of size NP x 2, in row-major format, listing the tip pairs	
	tip_pairs.reserve(ceil(Ntips/2));
	long tip1, tip2, dn, node, clade;
	while(!dnodes.empty()){
		dn    = dnodes.size()-1;
		node  = dnodes[dn];
		clade = node+Ntips;
		// find tips descending from this node
		if(!aux_get_two_descending_tips(Ntips, Nnodes, Nedges, tree_edge, traversal, tips_per_clade, clade, tip1, tip2)) Rcout << "ERROR: Something went wrong. Maybe a bug?" << endl;
		tip_pairs.push_back(tip1);
		tip_pairs.push_back(tip2);
		// update the (unpaired) tip counts of all ancestral nodes starting at tip1 and tip2 (traverse tips --> root)
		// also remove or add nodes from the dnodes[] list as appropriate
		clade = clade2parent[tip1];
		while(clade>=0){
			node = clade-Ntips;
			tips_per_clade[clade] -= 1;
			if((node2dnode_index[node]>=0) && (tips_per_clade[clade]!=2)){
				// remove from dnodes list
				remove_item_from_mapped_list(dnodes, node2dnode_index, node);
			}else if((node2dnode_index[node]<0) && (tips_per_clade[clade]==2)){
				// add to dnodes list
				node2dnode_index[node] = dnodes.size();
				dnodes.push_back(node);
			}
			clade = clade2parent[clade];
		}
		clade = clade2parent[tip2];
		while(clade>=0){
			node = clade-Ntips;
			tips_per_clade[clade] -= 1;
			if((node2dnode_index[node]>=0) && (tips_per_clade[clade]!=2)){
				// remove from dnodes list
				remove_item_from_mapped_list(dnodes, node2dnode_index, node);
			}else if((node2dnode_index[node]<0) && (tips_per_clade[clade]==2)){
				// add to dnodes list
				node2dnode_index[node] = dnodes.size();
				dnodes.push_back(node);
			}
			clade = clade2parent[clade];
		}
	}
	return Rcpp::List::create(Rcpp::Named("tip_pairs") = tip_pairs);
}


// Join two rooted phylogenetic trees, by placing the root edge of one tree (tree2) onto a new branching point on some target edge of another tree (tree 1)
// Hence, tree2 ends up becoming a subtree (a monophyletic group) of the final joined tree
// Note that tree1 will generally not be a monophyletic group in the joined tree, unless the target_edge1 is the root edge of tree1 (i.e. target_edge1<0 in the current notation)
// [[Rcpp::export]]
Rcpp::List join_rooted_trees_CPP(	const long					Ntips1,
									const long					Nnodes1,
									const long					Nedges1,					// (INPUT) number of edges in tree1
									const std::vector<long>		&tree_edge1,				// (INPUT) 2D array (in row-major format) of size Nedges1 x 2
									const std::vector<double>	&edge_length1,				// (INPUT) 1D array of size Nedges1, or empty
									const long					Ntips2,
									const long					Nnodes2,
									const long					Nedges2,					// (INPUT) number of edges in tree1
									const std::vector<long>		&tree_edge2,				// (INPUT) 2D array (in row-major format) of size Nedges2 x 2
									const std::vector<double>	&edge_length2,				// (INPUT) 1D array of size Nedges2, or empty
									const long					target_edge1,				// (INPUT) edge index in tree1 onto which tree2 is to be joined. If <0, then this refers to the edge leading into the root of tree1.
									const double				target_edge_length1,		// (INPUT) length of the edge segment in tree1 from the joining-point to the next child node, i.e. how far from the child of target_edge1 should the joining occur.
									const double				root_edge_length2){			// (INPUT) length of the edge leading into the root of tree2, i.e. the distance from the joining point to the root of tree2
	const long Nclades1 = Ntips1 + Nnodes1;
	const long Nclades2 = Ntips2 + Nnodes2;
	const bool has_edge_lengths1 = (!edge_length1.empty());
	const bool has_edge_lengths2 = (!edge_length2.empty());
	long root;
	lvector tree_edge;
	tree_edge.reserve(2*(Nedges1+Nedges2+2));
	dvector edge_length;
	edge_length.reserve(Nedges1+Nedges2+2);
	//lvector clade1_to_clade(Nclades1), clade2_to_clade(Nclades2);
	
	// determine roots of trees
	const long root1 = get_root_clade(Ntips1, Nnodes1, Nedges1, tree_edge1);
	const long root2 = get_root_clade(Ntips2, Nnodes2, Nedges2, tree_edge2);

	// merge edge lists of the two trees, making sure to re-number clades of tree2
	// shift clade indices of tree2 by Nclades1, and shift edge indices of tree2 by Nedges1, worry about indexing conventions later
	tree_edge 	= tree_edge1;
	if(has_edge_lengths1){
		edge_length = edge_length1;
	}else{
		edge_length.assign(Nedges1,1.0);
	}
	for(long edge2=0; edge2<Nedges2; ++edge2){
		tree_edge.push_back(Nclades1+tree_edge2[2*edge2+0]);
		tree_edge.push_back(Nclades1+tree_edge2[2*edge2+1]);
		if(has_edge_lengths2){
			edge_length.push_back(edge_length2[edge2]);				
		}else{
			edge_length.push_back(1.0);
		}
	}
	
	if(target_edge1<0){
		// join both trees at their root edges, creating a new root node
				
		// add edges connecting the two roots with the joining point (which becomes a new node)
		root = Nclades1+Nclades2;
		tree_edge.push_back(root);
		tree_edge.push_back(root1);
		edge_length.push_back(target_edge_length1);
		tree_edge.push_back(root);
		tree_edge.push_back(Nclades1+root2);
		edge_length.push_back(root_edge_length2);
	}else{
		// join tree2 onto some edge of tree1, creating a new node somewhere along target_edge1
		
		// split target_edge1 to place tree2, connecting the newly formed node to root2
		root 				= root1;
		const long joint 	= Nclades1+Nclades2;
		const long parent1 	= tree_edge1[2*target_edge1+0];
		//  create edge: joint --> root2
		tree_edge.push_back(joint);
		tree_edge.push_back(Nclades1+root2);
		edge_length.push_back(root_edge_length2);
		//  create edge: parent1 --> joint
		tree_edge.push_back(parent1);
		tree_edge.push_back(joint);
		edge_length.push_back(max(0.0,(has_edge_lengths1 ? edge_length1[target_edge1] : 1.0)-target_edge_length1));
		//  modify target_edge1 (originally parent1 --> child1) to become: joint --> child1
		tree_edge[2*target_edge1+0] = joint;
		edge_length[target_edge1] = target_edge_length1;
	}
	const long Nedges  = tree_edge.size()/2;
	const long Nclades = Nclades1+Nclades2+1;
	
	// re-index tips & nodes to be consistent with "phylo" convention
	long Ntips, Nnodes;
	lvector old2new_clade;
	reindex_clades(	Nclades,
					Nedges,
					tree_edge,
					true,	// obey root_convention
					Ntips,
					Nnodes,
					old2new_clade);
	for(long edge=0; edge<Nedges; ++edge){
		tree_edge[2*edge+0] = old2new_clade[tree_edge[2*edge+0]];
		tree_edge[2*edge+1] = old2new_clade[tree_edge[2*edge+1]];
	}
	lvector clade1_to_clade(Nclades1), clade2_to_clade(Nclades2);
	for(long clade1=0; clade1<Nclades1; ++clade1){
		clade1_to_clade[clade1] = old2new_clade[clade1];
	}
	for(long clade2=0; clade2<Nclades2; ++clade2){
		clade2_to_clade[clade2] = old2new_clade[clade2+Nclades1];
	}
	
	return Rcpp::List::create(	Rcpp::Named("Ntips")  			= Ntips,
								Rcpp::Named("Nnodes") 			= Nnodes,
								Rcpp::Named("Nedges") 			= Nedges,
								Rcpp::Named("tree_edge") 		= tree_edge,
								Rcpp::Named("edge_length") 		= edge_length,
								Rcpp::Named("root")		 		= old2new_clade[root], // this is actually guaranteed to always be equal to Ntips
								Rcpp::Named("clade1_to_clade") 	= clade1_to_clade,
								Rcpp::Named("clade2_to_clade") 	= clade2_to_clade);
}



#pragma mark -
#pragma mark Comparing trees
#pragma mark 


// Congruify trees (match nodes) as described by [Eastman et al (2013). Congruification: support for time scaling large phylogenetic trees. Methods in Ecology and Evolution. 4:688-691]
// This function essentially finds nodes in the "target" tree (T) that are equivalent ("concordant") branching points to nodes in the "reference" tree (R)
// In case multiple T-nodes are concordant to the same R-node, preference is given to the one closest to the tips. The same holds for R-nodes.
// This may be useful if R is dated (time-calibrated) and T is to be dated using information on branching times in R.
// [[Rcpp::export]]
Rcpp::List congruify_trees_CPP(	const long				RNtips,
								const long 				RNnodes,
								const long				RNedges,
								const std::vector<long> &Rtree_edge,
								const long				TNtips,
								const long 				TNnodes,
								const long				TNedges,
								const std::vector<long> &Ttree_edge,
								const std::vector<long>	&mapping){		// 2D array of size NM x 2, in row-major format, mapping a subset of T-tips to a subset of R-tips (mapping[m,0]-->mapping[m,1]). The mapping need not be one-to-one, but T-->R must be a valid mapping. In particular, every T-tip can appear at most once in the mapping.
	const long NM = mapping.size()/2;
	const long RNclades = RNtips + RNnodes;
	const long TNclades = TNtips + TNnodes;
	
	// determine parent clade for each clade
	std::vector<long> Rclade2parent, Tclade2parent;
	get_parent_per_clade(RNtips, RNnodes, RNedges, Rtree_edge, Rclade2parent);
	get_parent_per_clade(TNtips, TNnodes, TNedges, Ttree_edge, Tclade2parent);

	// find root using the mapping clade2parent
	const long Rroot = get_root_from_clade2parent(RNtips, Rclade2parent);
	const long Troot = get_root_from_clade2parent(TNtips, Tclade2parent);

	// get tree traversal route (root --> tips)
	std::vector<long> Rtraversal_queue, Rtraversal_node2first_edge, Rtraversal_node2last_edge, Rtraversal_edges;
	get_tree_traversal_root_to_tips(	RNtips,
										RNnodes,
										RNedges,
										Rroot,
										Rtree_edge,
										true,		// include tips in traversal
										false,		// edge mappings are not yet computed
										Rtraversal_queue,
										Rtraversal_node2first_edge,
										Rtraversal_node2last_edge,
										Rtraversal_edges,
										false,
										"");
	std::vector<long> Ttraversal_queue, Ttraversal_node2first_edge, Ttraversal_node2last_edge, Ttraversal_edges;
	get_tree_traversal_root_to_tips(	TNtips,
										TNnodes,
										TNedges,
										Troot,
										Ttree_edge,
										true,		// include tips in traversal
										false,		// edge mappings are not yet computed
										Ttraversal_queue,
										Ttraversal_node2first_edge,
										Ttraversal_node2last_edge,
										Ttraversal_edges,
										false,
										"");
	
	// create mapping from T-tips to focal tips
	// focals = domain(mapping)
	std::vector<long> Ttip2focal(TNtips,-1); // Ttip2Focal[r] maps the T-tip r to the focal tip index. Can be -1, if T-tip r is not included in the focals (i.e. not in image(mapping))
	long next_focal=0;
	for(long m=0, Ttip; m<NM; ++m){
		Ttip = mapping[2*m+0];
		if(Ttip2focal[Ttip]<0){
			Ttip2focal[Ttip] = next_focal;
			++next_focal;
		}
	}
	const long Nfocals = next_focal;
	
	
	// create membership tables (2D array of size Nclades x Nfocals, in row-major format)
	// memberships[c,f] specifies whether clade c includes (has a descendant) the focal tip f
	std::vector<bool> Rmemberships(RNclades*Nfocals,false), Tmemberships(TNclades*Nfocals,false);
	// set membership of tips included in mapping
	for(long m=0, Rtip, Ttip, Ftip; m<NM; ++m){
		Ttip = mapping[2*m+0];
		Rtip = mapping[2*m+1];
		Ftip = Ttip2focal[Ttip];
		Rmemberships[Nfocals*Rtip+Ftip] = true;
		Tmemberships[Nfocals*Ttip+Ftip] = true;
	}
	// propagate memberships (inclusion of focals) upwards (tips-->root)
	for(long q=Rtraversal_queue.size()-1, parent, clade; q>=1; --q){
		clade  = Rtraversal_queue[q];
		parent = Rclade2parent[clade];
		for(long f=0; f<Nfocals; ++f){
			Rmemberships[Nfocals*parent+f] = (Rmemberships[Nfocals*parent+f] || Rmemberships[Nfocals*clade+f]);
		}
	}
	for(long q=Ttraversal_queue.size()-1, parent, clade; q>=1; --q){
		clade  = Ttraversal_queue[q];
		parent = Tclade2parent[clade];
		for(long f=0; f<Nfocals; ++f){
			Tmemberships[Nfocals*parent+f] = (Tmemberships[Nfocals*parent+f] || Tmemberships[Nfocals*clade+f]);
		}
	}
	
	
	// find equivalent R & T nodes based on membership tables
	// traverse T-tree tips-->roots, so that Tclades closer to the tips are found first (in case of multiple matches)
	std::vector<long> mapped_Tnodes, mapped_Rnodes;
	for(long q=Ttraversal_queue.size()-1, Tclade; q>=0; --q){
		Tclade = Ttraversal_queue[q];
		if(Tclade<TNtips) continue;
		
		// find equivalent node in R-tree, if possible
		// start searching at root, moving towards tips
		// at each branching point, at most one of the children will be a valid next step
		long Rclade = Rroot;
		while(Rclade>=RNtips){
			// requirement "T_in_R": at this point it is guaranteed that Rmemberships[Rclade,:] includes all focal tips that are included in Tmemberships[Tclade,:]
			// At most one of the children of Rclade will still satisfy "T_in_R", so go to that one
			// If a child violates "T_in_R", then all of its descendants also violate "T_in_R", so there is no point in continuing in children violating "T_in_R"
			long Rnode = Rclade-RNtips;
			long next_Rclade = -1;
			for(long e=Rtraversal_node2first_edge[Rnode], Rchild; e<=Rtraversal_node2last_edge[Rnode]; ++e){
				Rchild = Rtree_edge[2*Rtraversal_edges[e]+1];
				bool OK = true;
				for(long f=0; f<Nfocals; ++f){
					if(Tmemberships[Nfocals*Tclade+f] && (!Rmemberships[Nfocals*Rchild+f])){
						OK = false;
						break;
					}
				}
				if(OK){
					next_Rclade = Rchild;
					break;
				}
			}
			if(next_Rclade<0){
				// none of the children of Rclade satisfy requirement "T_in_R"
				// so if there exists an equivalent R-clade to Tclade, it must be Rclade, so check
				bool OK = true;
				for(long f=0; f<Nfocals; ++f){
					if(Tmemberships[Nfocals*Tclade+f] != Rmemberships[Nfocals*Rclade+f]){
						OK = false;
						break;
					}
				}
				if(OK){
					// found equivalent clade
					mapped_Tnodes.push_back(Tclade-TNtips);
					mapped_Rnodes.push_back(Rclade-RNtips);
				}
				break; // move to next Tclade, regardless of success
			}else{
				// move one step closer to tips
				Rclade = next_Rclade;
			}
		}
		if(q%100==0) Rcpp::checkUserInterrupt(); // abort if the user has interrupted the calling R program
	}

	// remove duplicate mapped Tclades
	std::vector<long> mapped_Tnodes_deduplicated, mapped_Rnodes_deduplicated;
	std::vector<bool> Rincluded(RNnodes);
	mapped_Tnodes_deduplicated.reserve(mapped_Tnodes.size());
	mapped_Rnodes_deduplicated.reserve(mapped_Rnodes.size());
	for(long m=0, Tnode, Rnode; m<mapped_Tnodes.size(); ++m){
		Tnode = mapped_Tnodes[m];
		Rnode = mapped_Rnodes[m];
		if(!Rincluded[Rnode]){
			mapped_Tnodes_deduplicated.push_back(Tnode);
			mapped_Rnodes_deduplicated.push_back(mapped_Rnodes[m]);
			Rincluded[Rnode] = true;
		}
	}

	return Rcpp::List::create(	Rcpp::Named("mapped_Tnodes") = mapped_Tnodes_deduplicated,
								Rcpp::Named("mapped_Rnodes") = mapped_Rnodes_deduplicated);
}




// Match nodes from one tree to another, assuming that the tree topologies are the same (but indexed differently) and that both have the same root
// This may be useful if nodes and/or tips were re-indexed, and the only way to match old to new nodes is based on topology (e.g. node names are missing)
// This function returns an error if the trees don't have equivalent topologies, so it can also be used as a simple equivalence test
// If you are dealing with different trees, consider using congruify_trees_CPP(..)
// [[Rcpp::export]]
Rcpp::List match_tree_nodes_CPP(const long				Ntips,
								const long 				Nnodes,
								const long				Nedges,
								const std::vector<long> 	&tree_edgeA,
								const std::vector<long> 	&tree_edgeB,
								const std::vector<long>		&tipsA2B){		// 1D array of size Ntips, mapping A-tip indices to B-tip indices (tipsA2B[a] is the B-tip corresponding to a-th A-tip)
	const long Nclades = Ntips + Nnodes;
	
	// determine parent clade for each clade
	std::vector<long> clade2parentA, clade2parentB;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edgeA, clade2parentA);
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edgeB, clade2parentB);

	// find root using the mapping clade2parent
	const long rootA = get_root_from_clade2parent(Ntips, clade2parentA);
	const long rootB = get_root_from_clade2parent(Ntips, clade2parentB);

	// get tree traversal route (root --> tips)
	std::vector<long> traversal_queueA, traversal_node2first_edgeA, traversal_node2last_edgeA, traversal_edgesA;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										rootA,
										tree_edgeA,
										true,		// include tips in traversal
										false,		// edge mappings are not yet computed
										traversal_queueA,
										traversal_node2first_edgeA,
										traversal_node2last_edgeA,
										traversal_edgesA,
										false,
										"");

	// map clades A-->B
	// traverse tips-->root and propagate information from child to parent
	std::vector<long> cladesA2B(Nclades,-1);
	std::vector<bool> matchedB(Nnodes,false);
	long Nmatched = 0;
	for(long tip=0; tip<Ntips; ++tip) cladesA2B[tip] = tipsA2B[tip];
	for(long q=traversal_queueA.size()-1, cladeA, parentA, cladeB, parentB; q>=1; --q){
		cladeA 	= traversal_queueA[q];
		parentA = clade2parentA[cladeA];
		if(cladesA2B[parentA]>=0) continue; // parentA already mapped, so skip
		// assuming cladeA is already mapped to some B-clade, map parentA to the B-clade's parent.
		cladeB = cladesA2B[cladeA];
		if(cladeB==rootB){
			// something went wrong (non-rootA mapped to rootB. This can only happen if the two trees are rooted differently
			return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Tree roots don't match");
		}else{
			parentB = clade2parentB[cladeB];
			cladesA2B[parentA] = parentB;
			++Nmatched;
			if(matchedB[parentB-Ntips]) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Some nodes in tree B were matched more than once");
			matchedB[parentB-Ntips] = true;
		}
	}
	if(Nmatched<Nnodes) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Some nodes in tree A could not be matched");
		
	// extract mapped nodes
	std::vector<long> nodesA2B(Nnodes);
	for(long node=0; node<Nnodes; ++node) nodesA2B[node] = cladesA2B[Ntips+node]-Ntips;

	return Rcpp::List::create(	Rcpp::Named("success") 	= true,
								Rcpp::Named("nodesA2B") = nodesA2B,
								Rcpp::Named("rootA")	= rootA,
								Rcpp::Named("rootB")	= rootB);
}





// Calculate Robinson-Foulds distance between two rooted trees, ad described by Day (1985).
// The trees must share the same tips, but may exhibit different topologies
// This metric quantifies the disagreement in topologies, but does not take into account edge lengths
// The distance between the trees depends on their rooting, i.e. this is not an appropriate metric for comparing unrooted trees (note that this differs from the original RF metric, which is for unrooted trees)
// [William Day (1985). Optimal algorithms for comparing trees with labeled leaves. Journal of Classification. 2:7-28]
// [[Rcpp::export]]
Rcpp::List get_Robinson_Foulds_distance_CPP(const long				Ntips,
											const long 				NnodesA,
											const long				NedgesA,
											const std::vector<long> &tree_edgeA,
											const long 				NnodesB,
											const long				NedgesB,
											const std::vector<long> &tree_edgeB,
											const std::vector<long>	&tipsA2B){		// 1D array of size Ntips, mapping A-tip indices to B-tip indices (tipsA2B[a] is the B-tip corresponding to a-th A-tip)
	const long NcladesA = Ntips + NnodesA;
	const long NcladesB = Ntips + NnodesB;
	
	// determine parent clade for each clade
	std::vector<long> clade2parentA, clade2parentB;
	get_parent_per_clade(Ntips, NnodesA, NedgesA, tree_edgeA, clade2parentA);
	get_parent_per_clade(Ntips, NnodesB, NedgesB, tree_edgeB, clade2parentB);

	// find root using the mapping clade2parent
	const long rootA = get_root_from_clade2parent(Ntips, clade2parentA);
	const long rootB = get_root_from_clade2parent(Ntips, clade2parentB);

	// get tree traversal route (root --> tips) in depth-first-search mode, for each tree (DFS is important, to ensure a certain traversal order of tips and nodes)
	std::vector<long> traversal_queueA, traversal_node2first_edgeA, traversal_node2last_edgeA, traversal_edgesA;
	get_tree_traversal_depth_first_search(	Ntips,
											NnodesA,
											NedgesA,
											rootA,
											tree_edgeA,
											true,		// include tips in traversal
											false,		// edge mappings are not yet computed
											traversal_queueA,
											traversal_node2first_edgeA,
											traversal_node2last_edgeA,
											traversal_edgesA);
	std::vector<long> traversal_queueB, traversal_node2first_edgeB, traversal_node2last_edgeB, traversal_edgesB;
	get_tree_traversal_depth_first_search(	Ntips,
											NnodesB,
											NedgesB,
											rootB,
											tree_edgeB,
											true,		// include tips in traversal
											false,		// edge mappings are not yet computed
											traversal_queueB,
											traversal_node2first_edgeB,
											traversal_node2last_edgeB,
											traversal_edgesB);
										
	
	// create mappings of tipsA-->focal_tips and tipsB-->focal_tips (where focal tips are just a re-indexing of tips, so that they are in the same order as reached by the DFS tree traversal of treeA)
	// requirement: A2F[a] = B2F[tipsA2B[a]]
	std::vector<long> tipsA2F(Ntips,-1), tipsB2F(Ntips,-1);
	long next_focal = 0;
	for(long q=traversal_queueA.size()-1, clade; q>=0; --q){
		clade = traversal_queueA[q];
		if(clade<Ntips) tipsA2F[clade] = (next_focal++);
	}
	for(long tipA=0; tipA<Ntips; ++tipA){
		tipsB2F[tipsA2B[tipA]] = tipsA2F[tipA];
	}
	

	//count the number of tips descending from each clade (traverse tips-->root)
	std::vector<long> cladeA2tip_counts(NcladesA,0), cladeB2tip_counts(NcladesB,0);
	for(long tip=0; tip<Ntips; ++tip){
		cladeA2tip_counts[tip] = 1;
		cladeB2tip_counts[tip] = 1;
	}
	for(long q=traversal_queueA.size()-1, cladeA; q>=1; --q){
		cladeA = traversal_queueA[q];
		cladeA2tip_counts[clade2parentA[cladeA]] += cladeA2tip_counts[cladeA];
	}
	for(long q=traversal_queueB.size()-1, cladeB; q>=1; --q){
		cladeB = traversal_queueB[q];
		cladeB2tip_counts[clade2parentB[cladeB]] += cladeB2tip_counts[cladeB];
	}
		
	// create membership tables (list of focal tips descending from each node)
	// A memberships will be sorted (i.e. each membershipsA[a][] will be a list of ascending focal tip indices)
	// this is achieved because we're traversing the tree in reverse-depth-first-search, and A tips are mapped to focal tips in ascending order, and node traversal is consistent with tip order
	std::vector<std::vector<long> > membershipsA(NnodesA), membershipsB(NnodesB);
	for(long node=0; node<NnodesA; ++node) membershipsA[node].reserve(cladeA2tip_counts[node+Ntips]); // preallocate space
	for(long node=0; node<NnodesB; ++node) membershipsB[node].reserve(cladeB2tip_counts[node+Ntips]); // preallocate space
	for(long q=traversal_queueA.size()-1, clade, cnode, pnode; q>=1; --q){
		clade = traversal_queueA[q];
		pnode = clade2parentA[clade] - Ntips;
		if(clade<Ntips){
			membershipsA[pnode].push_back(tipsA2F[clade]);
		}else{
			cnode = clade-Ntips;
			membershipsA[pnode].insert(membershipsA[pnode].end(), membershipsA[cnode].begin(), membershipsA[cnode].end());
		}
		if(q%100==0) Rcpp::checkUserInterrupt(); // abort if the user has interrupted the calling R program
	}
	for(long q=traversal_queueB.size()-1, clade, cnode, pnode; q>=1; --q){
		clade = traversal_queueB[q];
		pnode = clade2parentB[clade] - Ntips;
		if(clade<Ntips){
			membershipsB[pnode].push_back(tipsB2F[clade]);
		}else{
			cnode = clade-Ntips;
			membershipsB[pnode].insert(membershipsB[pnode].end(), membershipsB[cnode].begin(), membershipsB[cnode].end());
		}
		if(q%100==0) Rcpp::checkUserInterrupt(); // abort if the user has interrupted the calling R program
	}

	// also sort B memberships (A memberships are already sorted)
	for(long node=0; node<NnodesB; ++node){
		std::sort(membershipsB[node].begin(), membershipsB[node].end());
	}
	
	
	// find equivalent nodes between the two trees, by traversing tips-->roots
	long Nmatches=0;
	std::vector<long> cladeA2B(NcladesA); // cladeA2B[a] points to a clade in treeB that is fully contained (but not necessarily equal) to A-clade a
	std::vector<bool> matchedB(NcladesB,false); // keep track of B-clades that were matched before, to avoid duplicate matching
	for(long tipA=0; tipA<Ntips; ++tipA) cladeA2B[tipA] = tipsA2B[tipA];
	for(long q=traversal_queueA.size()-1, cladeA, nodeA; q>=0; --q){
		cladeA = traversal_queueA[q];
		if(cladeA<Ntips) continue;
		nodeA = cladeA - Ntips;
		// at this point, each child of cladeA is mapped to a cladeB which it fully contains (i.e. all of whose tips also descends the child)
		// check if any of the mapped cladeBs can be moved upstream and still be contained in cladeA
		for(long e=traversal_node2first_edgeA[nodeA], childA, cladeB, nodeB; e<=traversal_node2last_edgeA[nodeA]; ++e){
			childA = tree_edgeA[2*traversal_edgesA[e]+1];
			cladeB = cladeA2B[childA];
			cladeA2B[cladeA] = cladeB; // since 
			bool OK = true;
			while(OK && (cladeB!=rootB)){
				cladeB 	= clade2parentB[cladeB];
				nodeB	= cladeB-Ntips;
				// membershipsA[nodeA][] and membershipsB[nodeB][] and guaranteed to be sorted in ascending order
				for(long fb=0, fa=-1, f; fb<membershipsB[nodeB].size(); ++fb){
					f  = membershipsB[nodeB][fb];
					fa = find_in_ascending_list(membershipsA[nodeA],f,fa+1);
					if(fa<0){
						OK = false;
						break;
					}
				}
				if(OK) cladeA2B[cladeA] = cladeB; // still OK, so update matched clade
			}
			cladeB = cladeA2B[cladeA];
			if(cladeA2tip_counts[cladeA]==cladeB2tip_counts[cladeB]){
				// found fully matching B-clade (since cladeA2B[cladeA] is contained in cladeA, and actually has the same size)
				if(!matchedB[cladeB]){
					++Nmatches; // found equivalent B-clade that wasn't matched before
					matchedB[cladeB] = true;
				}
				break;
			}
		}
	}

	return Rcpp::List::create(Rcpp::Named("Nmatches") = Nmatches);
}


#pragma mark -
#pragma mark Tree dating
#pragma mark 


// given a phylogenetic tree and a vector in [0,1]^Nnodes ("relative ages"), map relative ages (R) to absolute ages (A = distance from present) for each node
// The mapping proceeds from root to tips
// The mapping is defined as follows:
//		For any clade, let C1,C2,..,Cn be the sequence of clades from the root to that clade
//		Let L[C_n] := min(max_abs_node_ages[C_n], A[C_{n-1}])
//		Then A[C_n] = L[C_n] + R[C_n]*(min_abs_node_ages[C_n]-L[C_n])
// Tips are implicitly assumed to have age 0.
// This mapping may be useful for fitting absolute node ages while searching within classical box-constraints
// [[Rcpp::export]]
std::vector<double> relative_to_absolute_node_ages_CPP(	const long 					Ntips,
														const long 					Nnodes,
														const long 					Nedges,
														const std::vector<long>		&tree_edge,				// (INPUT) 2D array of size Nedges x 2, in row-major format
														const std::vector<long>		&traversal_queue,		// (INPUT) traversal queue from root-->tips (listing clade indices), not including tips
														const std::vector<double>	&relative_node_ages,	// (INPUT) 1D array of size Nnodes, listing relative ages of each node. Values should be within [0,1].
														const std::vector<double>	&min_abs_node_ages,		// (INPUT) 1D array of size Nnodes
														const std::vector<double>	&max_abs_node_ages){	// (INPUT) 1D array of size Nnodes
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	const long root_node = traversal_queue[0] - Ntips;
	
	std::vector<double> abs_node_ages(Nnodes);
	abs_node_ages[root_node] = max_abs_node_ages[root_node] + relative_node_ages[root_node]*(min_abs_node_ages[root_node] - min_abs_node_ages[root_node]);
	for(long q=1, clade, node, pnode; q<traversal_queue.size(); ++q){
		clade 	= traversal_queue[q];
		node 	= clade-Ntips;
		pnode 	= clade2parent[clade]-Ntips;
		const double L = min(max_abs_node_ages[node], abs_node_ages[pnode]);
		abs_node_ages[node] = L + relative_node_ages[node]*(min_abs_node_ages[node] - L);
	}
	return abs_node_ages;
}


// [[Rcpp::export]]
std::vector<double> propagate_min_ages_upstream_CPP(const long 					Ntips,
													const long 					Nnodes,
													const long 					Nedges,
													const std::vector<long>		&tree_edge,
													const std::vector<long>		&traversal_queue,		// (INPUT) traversal queue from root-->tips (listing clade indices), not including tips
													const std::vector<long>		&anchor_nodes,
													const std::vector<long>		&anchor_min_ages){
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	std::vector<double> min_node_ages(Nnodes,0);
	for(long a=0; a<anchor_nodes.size(); ++a){
		min_node_ages[anchor_nodes[a]] = anchor_min_ages[a];
	}
	
	// propagate min_ages upstream
	for(long q=traversal_queue.size()-1, clade, node, pnode; q>=1; --q){
		clade	= traversal_queue[q];
		node	= clade - Ntips;
		pnode 	= clade2parent[clade] - Ntips;
		min_node_ages[pnode] = max(min_node_ages[pnode], min_node_ages[node]);
	}
	
	return min_node_ages;
}


// [[Rcpp::export]]
std::vector<double> propagate_max_ages_downstream_CPP(	const long 					Ntips,
														const long 					Nnodes,
														const long 					Nedges,
														const std::vector<long>		&tree_edge,
														const std::vector<long>		&traversal_queue,		// (INPUT) traversal queue from root-->tips (listing clade indices), not including tips
														const std::vector<long>		&anchor_nodes,
														const std::vector<long>		&anchor_max_ages){
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	std::vector<double> max_node_ages(Nnodes,INFTY_D);
	for(long a=0; a<anchor_nodes.size(); ++a){
		max_node_ages[anchor_nodes[a]] = anchor_max_ages[a];
	}
	
	// propagate max_ages downstream
	for(long q=0, clade, node, pnode; q<traversal_queue.size(); ++q){
		clade	= traversal_queue[q];
		node	= clade - Ntips;
		pnode 	= clade2parent[clade] - Ntips;
		max_node_ages[pnode] = min(max_node_ages[pnode], max_node_ages[node]);
	}
	
	return max_node_ages;
}




#pragma mark -
#pragma mark Plotting trees
#pragma mark 


// calculate the geometric placement of tips & nodes for plotting a tree as a phylogram
// The root is placed on the left end, tips are placed on the right end, edges extend horizontally left to right
// tips y-coordinates of all clades will be within 0 and Ntips
// [[Rcpp::export]]
Rcpp::List get_phylogram_geometry_CPP(	const long			Ntips,
										const long 			Nnodes,
										const long			Nedges,
										std::vector<long>	tree_edge,				// (INPUT) 2D array (in row-major format) of size Nedges x 2
										const std::vector<double>	&edge_length){	// (INPUT) 1D array of size Nedges, or empty
	const long Nclades = Ntips + Nnodes;
										
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,	// include tips
										false,	// edge mappings are not yet calculated
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
										
	// calculate distance from root for each clade
	// (traverse root --> tips, excluding the root)
	std::vector<double> distances_from_root(Nclades);
	distances_from_root[root] = 0;
	for(long q=1, clade, parent; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		parent = clade2parent[clade];
		distances_from_root[clade] = (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]) + distances_from_root[parent];
	}

	// calculate number of descending tips per node, traversing tips-->root (excluding the root)
	std::vector<long> node2total_tip_count(Nnodes,0);
	for(long q=traversal_queue.size()-1, clade; q>=1; --q){
		clade = traversal_queue[q];
		node2total_tip_count[clade2parent[clade]-Ntips] += (clade<Ntips ? 1 : node2total_tip_count[clade-Ntips]);
	}
	
	// calculate y-intervals of clades (traverse root-->tips)
	std::vector<double> clade_min_y(Nclades), clade_max_y(Nclades);
	clade_min_y[root] = 0;
	clade_max_y[root] = Ntips;
	for(long q=0, clade, node; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade<Ntips) continue;
		node = clade - Ntips;
		double cumulative_fraction = 0, fraction;
		for(long e=traversal_node2first_edge[node], child; e<=traversal_node2last_edge[node]; ++e){
			child 				= tree_edge[2*traversal_edges[e]+1];
			fraction			= (child<Ntips ? 1l : node2total_tip_count[child-Ntips])/double(node2total_tip_count[node]);
			clade_min_y[child] 	= clade_min_y[clade] + cumulative_fraction*(clade_max_y[clade]-clade_min_y[clade]);
			clade_max_y[child] 	= clade_min_y[clade] + (cumulative_fraction+fraction)*(clade_max_y[clade]-clade_min_y[clade]);
			cumulative_fraction += fraction;
		}
	}
	
	// calculate y-coordinates of clades (centers of y-intervals)
	std::vector<double> clade_y(Nclades);
	for(long clade=0; clade<Nclades; ++clade){
		clade_y[clade] = 0.5*(clade_min_y[clade] + clade_max_y[clade]);
	}
										
	return Rcpp::List::create(	Rcpp::Named("clade_x") 	= Rcpp::wrap(distances_from_root),
								Rcpp::Named("clade_y") 	= Rcpp::wrap(clade_y),
								Rcpp::Named("min_x")	= 0.0,
								Rcpp::Named("max_x")	= array_max(distances_from_root),
								Rcpp::Named("min_y")	= 0.5,
								Rcpp::Named("max_y")	= Ntips-0.5,
								Rcpp::Named("root_y")	= clade_y[root]);
}



#pragma mark -
#pragma mark Writing/reading trees
#pragma mark 


// convert a tree to a string in Newick (parenthetic) format
// If the tree is not rooted, it is first rooted 
// [[Rcpp::export]]
std::string tree_to_Newick_string_CPP(	const long					Ntips,
										const long 					Nnodes,
										const long					Nedges,
										std::vector<long>			tree_edge,				// (INPUT) 2D array (in row-major format) of size Nedges x 2
										const std::vector<double>	&edge_length,	// (INPUT) 1D array of size Nedges, or empty
										const StringVector			&tip_labels,		// (INPUT) 1D array of size Ntips, or empty
										const StringVector			&node_labels,		// (INPUT) 1D array of size Nnodes, or empty
										const StringVector			&edge_labels,		// (INPUT) 1D array of size Nedges, or empty
										const std::vector<long>		&edge_numbers,		// (INPUT) 1D array of size Nedges, or empty
										const long					digits,				// (INPUT) number of digits used for printing edge lengths
										const double				root_edge_length,	// (INPUT) optional edge length leading into the root. Not really an edge of the tree. Ignored if negative.
										const int					quoting){			// (INPUT) whether to enclose tip & node names in single quotes. 0:never, 1:always single quoting, 2:always double quoting, -1:quote only if needed, and prefer single quotes if possible, -2:quote only if needed, and prefer double quotes if possible
	const bool has_tip_labels  	= (tip_labels.size()>0);
	const bool has_node_labels  = (node_labels.size()>0);
	const bool has_edge_lengths	= (edge_length.size()>0);
	const bool has_edge_labels	= (edge_labels.size()>0);
	const bool has_edge_numbers	= (edge_numbers.size()>0);
	const long Nclades = Ntips + Nnodes;
	long child,node,clade;
	ostringstream output;
	output << std::setprecision(digits);

	// get incoming edge for each clade
	std::vector<long> incoming_edge_per_clade;
	get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);

	// find root based on incoming_edge_per_clade
	// make sure there is exactly one root
	long root = -1;
	for(long clade=Ntips; clade<Nclades; ++clade){
		if(incoming_edge_per_clade[clade]<0){
			if(root>=0){
				// already encountered a root, so abort and re-root properly
				root = -1;
				break;
			}else{
				root = clade;
			}
		}
	}
	
	if(root<0){
		// tree is not rooted, so root at some node
		root = Ntips;
		root_tree_at_node(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,	// will be modified in-situ
							root);
							
		// re-calculate incoming edge for each clade
		get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, tree_edge, incoming_edge_per_clade);
	}

	// get edge mappings (for efficient tree traversal)
	std::vector<long> node2first_edge, node2last_edge, edge_mapping;
	get_node2edge_mappings(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							node2first_edge,
							node2last_edge,
							edge_mapping);
	
	// create output queue in depth-first-search direction
	// also count how many brackets to close at each clade
	// use a scratch_stack for traversing nodes
	// note that the returned tree will actually be reverse version (i.e. tips-->root, in reversed debth-first-search)
	std::vector<long> scratch_stack, queue;
	std::vector<long> Nbrackets_to_close(Nclades,0);
	std::vector<bool> is_first_child(Nclades,false);
	scratch_stack.reserve(floor(2*log(Ntips)/log(2.0))); // rough estimate of typical tree depth x 2. scratch_stack may be resized along the way if needed.
	scratch_stack.push_back(root);
	queue.reserve(Nclades);
	is_first_child[root] = true;
	while(scratch_stack.size()>0){
		clade = scratch_stack.back();
		scratch_stack.pop_back();
		queue.push_back(clade);
		if(clade>=Ntips){
			node = clade - Ntips;
			for(long e=node2last_edge[node]; e>=node2first_edge[node]; --e){
				child = tree_edge[edge_mapping[e]*2+1];
				scratch_stack.push_back(child); // add child to stack for further exploration in the next iteration
				if(e==node2last_edge[node]){
					// delegate braket closing for this clade to last descending tip
					Nbrackets_to_close[child] = 1 + Nbrackets_to_close[clade];
					Nbrackets_to_close[clade] = 0;
				}
				is_first_child[child] = (e==node2first_edge[node]);
			}
		}
	}
		
	// traverse output queue in reverse direction
	string quote = (quoting==0 ? "" : (quoting==1 ? "'" : (quoting==2 ? "\"" : "")));
	string name;
	for(long q=queue.size()-1; q>=0; --q){
		clade = queue[q];
		for(long b=0; b<Nbrackets_to_close[clade]; ++b) output << "(";
		name = "";
		if(clade<Ntips){
			if(has_tip_labels) name = tip_labels[clade];
		}else{
			output << ")";
			if(has_node_labels) name = node_labels[clade - Ntips];
		}
		if(name!=""){
			if(quoting==-1){
				// quote only if needed, and prefer single quotes if possible
				quote = ((name.find_first_of('\'')!=string::npos) ? "\"" : ((name.find_first_of("():,\"")!=string::npos) ? "'" : ""));
			}else if(quoting==-2){
				// quote only if needed, and prefer double quotes if possible
				quote = ((name.find_first_of('"')!=string::npos) ? "'" : ((name.find_first_of("():,'")!=string::npos) ? "\"" : ""));			
			}
			output << quote << name << quote;
		}
		if(has_edge_lengths && (clade!=root)) output << ":" << edge_length[incoming_edge_per_clade[clade]];
		if(has_edge_numbers && (clade!=root)) output << "{" << edge_numbers[incoming_edge_per_clade[clade]] << "}";
		if(has_edge_labels && (clade!=root)){
			name = edge_labels[incoming_edge_per_clade[clade]];
			if(quoting==-1){
				// quote only if needed, and prefer single quotes if possible
				quote = ((name.find_first_of('\'')!=string::npos) ? "\"" : ((name.find_first_of("():,\"")!=string::npos) ? "'" : ""));
			}else if(quoting==-2){
				// quote only if needed, and prefer double quotes if possible
				quote = ((name.find_first_of('"')!=string::npos) ? "'" : ((name.find_first_of("():,'")!=string::npos) ? "\"" : ""));			
			}
			output << "[" << quote << name << quote << "]";
		}
		if(!is_first_child[clade]) output << ",";
	}
	if(has_edge_lengths && (root_edge_length>=0)) output << ":" << root_edge_length;
	output << ";";

	return(output.str());
}


// parse a string consisting one any subset of the following parts:
//   edge_length (scalar number)
//	 [edge_name] (edge name in square brackets)
//	 [edge_number] (edge number in curly brackets)
bool aux_Newick_parse_edge_info(	const std::string 	&input,
									const bool			interpret_quotes,	// (INPUT) don't split within quotes. If false, quotes are read verbatim just like any other character
									const bool			look_for_name,		// (INPUT) look for an edge name inside square brackets. If false, square brackets are interpreted verbatim like any other character.
									const bool			look_for_number,	// (INPUT) look for an edge number inside curly braces. If false, curly braces are interpreted verbatim like any other character.
									const long 			start,				// (INPUT) first character in input to consider
									const long 			end,				// (INPUT) last character in input to consider
									double				&edge_length,		// (OUTPUT) edge length. Will be NAN_D if not available
									string				&edge_name,			// (OUTPUT) edge label. Will be empty ("") if not available
									long				&edge_number,		// (OUTPUT) edge number. Will be -1 if not available
									string				&error){			// (OUTPUT) error description in case of failure
	if(end<start){
		// no information available
		edge_name 	= "";
		edge_number = -1;
		edge_length	= NAN_D;
		return true;
	}else if((!look_for_name) && (!look_for_number)){
		// classical Newick format, edge info only contains edge_length
		edge_name 	= "";
		edge_number = -1;
		edge_length	= string2Double(input.substr(start,end-start+1));
		return true;
	}
	long length_end = -1, name_start=-1, name_end=-1, number_start=-1, number_end=-1;
	bool single_quote_open = false, double_quote_open = false;
	for(long i=start; i<=end; ++i){
		if(interpret_quotes){
			if((!single_quote_open) && (input[i]=='"')){
				double_quote_open = !double_quote_open;
				continue;
			}else if((!double_quote_open) && (input[i]=='\'')){
				single_quote_open = !single_quote_open;
				continue;
			}
		}
		if((!single_quote_open) && (!double_quote_open)){
			if(look_for_name && (input[i]=='[')){
				if(name_start>=0){
					error = "Redundant opening square bracket [";
					return false;
				}
				name_start = i;
				if(length_end<0) length_end= i-1;
			}else if(look_for_name && (input[i]==']')){
				name_end = i;
				if(name_start<0){
					error = "Unexpected closing square bracket ]";
					return false;
				}
				if(length_end<0) length_end= i-1;
			}else if(look_for_number && (input[i]=='{')){
				if(number_start>=0){
					error = "Redundant opening curly brace {";
					return false;
				}
				number_start = i;
				if(length_end<0) length_end= i-1;
			}else if(look_for_number && (input[i]=='}')){
				number_end = i;
				if(number_start<0){
					error = "Unexpected closing curly brace }";
					return false;
				}
				if(length_end<0) length_end= i-1;
			}
		}
	}
	if(length_end<0) length_end = end;
	
	// basic error checking
	if(((name_start==-1) && (name_end>=0)) || ((name_start>=0) && (name_end==-1))){
		error = "Unbalanced square brackets";
		return false;
	}else if(name_end<name_start){
		error = "Redundant or misplaced square bracket [";
		return false;
	}else if(name_start>=end){
		error = "Unexpected square bracket [ at end of edge info";
		return false;
	}
	if(((number_start<0) && (number_end>=0)) || ((number_start>=0) && (number_end<0))){
		error = "Unbalanced curly braces";
		return false;
	}else if(number_end<number_start){
		error = "Redundant or misplaced curly brace {";
		return false;
	}else if(number_start>=end){
		error = "Unexpected curly brace { at end of edge info";
		return false;
	}
	
	// extract pieces based on delimiting positions
	if(length_end>=start){
		edge_length	= string2Double(input.substr(start,length_end-start+1));
	}else{
		edge_length = NAN_D;
	}
	if((name_start<=name_end) && (name_end>=start)){
		// edge_name will be enclosed in square brackets, so omit those
		edge_name = input.substr(name_start+1,name_end-name_start-1);
	}else{
		edge_name = "";
	}
	if((number_start<=number_end) && (number_end>=start)){
		// edge_number will be enclosed in curly braces, so omit those
		edge_number = string2Long(input.substr(number_start+1,number_end-number_start-1));
	}else{
		edge_number = -1;
	}

	// trim flanking quotes from edge_name if needed
	if(interpret_quotes){
		const long L = edge_name.length();
		if((L>0) && (((edge_name[0]=='"') && (edge_name[L-1]=='"')) || ((edge_name[0]=='\'') && (edge_name[L-1]=='\'')))){
			edge_name = edge_name.substr(1,L-2);
		}
	}
	return true;
}



// auxiliary routine for parsing a single edge in a Newick string
// returns false on failure
bool aux_Newick_extract_next_edge(	const std::string 	&input,
									const bool			interpret_quotes,	// (INPUT) don't split within quotes. If false, quotes are read verbatim just like any other character
									const bool			look_for_name,		// (INPUT) look for an edge name inside square brackets. If false, square brackets are interpreted verbatim like any other character.
									const bool			look_for_number,	// (INPUT) look for an edge number inside curly braces. If false, curly braces are interpreted verbatim like any other character.
									long 				&pointer,			// (INPUT/OUTPUT) will move towards the left
									string 				&child_name,		// (OUTPUT) child name. Will be empty ("") if not available
									double				&edge_length,		// (OUTPUT) edge length. Will be NAN_D if not available
									string				&edge_name,			// (OUTPUT) edge label. Will be empty ("") if not available
									long				&edge_number,		// (OUTPUT) edge number. Will be negative if not available
									string				&error){			// (OUTPUT) error description in case of failure
	long left = -1, split=-1;
	bool single_quote_open = false, double_quote_open = false, square_bracket_open = false;
	for(long i=pointer; i>=0; --i){
		if(interpret_quotes){
			if((!single_quote_open) && (input[i]=='"')){
				double_quote_open = !double_quote_open;
				continue;
			}else if((!double_quote_open) && (input[i]=='\'')){
				single_quote_open = !single_quote_open;
				continue;
			}
		}
		if(look_for_name && (!square_bracket_open) && (input[i]==']') && (!double_quote_open) && (!single_quote_open)){
			square_bracket_open = true;
			continue;
		}else if(look_for_name && square_bracket_open && (input[i]=='[') && (!double_quote_open) && (!single_quote_open)){
			square_bracket_open = false;
			continue;
		}
		if((!single_quote_open) && (!double_quote_open) && (!square_bracket_open)){
			if(input[i]==':'){
				split = i;
			}else if((input[i]=='(') || (input[i]==')') || (input[i]==',')){
				left = i;
				break;
			}
		}
	}
	if(left<0){
		error = "Missing terminal character '(', ')' or ','";
		return false;
	}else if(single_quote_open){
		error = "Imbalanced single quotes";
		return false;	
	}else if(double_quote_open){
		error = "Imbalanced single quotes";
		return false;	
	}else if(square_bracket_open){
		error = "Imbalanced square brackets";
		return false;	
	}
	if(left==pointer){
		// no child name nor edge information available
		child_name 	= "";
		edge_length = NAN_D;
		edge_name 	= "";
		edge_number	= -1;
		return true;
	}
	if(split<0){
		// no edge information available, interpret whole specifier as child_name
		child_name	= input.substr(left+1,pointer-left);
		edge_length	= NAN_D;
		edge_name	= "";
		edge_number	= -1;
	}else{
		child_name = input.substr(left+1,split-left-1);
		if(!aux_Newick_parse_edge_info(input, interpret_quotes, look_for_name, look_for_number, split+1, pointer, edge_length, edge_name, edge_number, error)){
			error = "Misspectified edge info near '"+input.substr(left+1,pointer-left)+"': "+error;
			return false;
		}
	}
	// trim flanking quotes from child_name if needed
	if(interpret_quotes){
		const long L = child_name.length();
		if((L>0) && (((child_name[0]=='"') && (child_name[L-1]=='"')) || ((child_name[0]=='\'') && (child_name[L-1]=='\'')))){
			child_name = child_name.substr(1,L-2);
		}
	}
	pointer = left;
	return true;
}



// read a phylogenetic tree from a Newick-formatted string
// Note: The Newick string is read from right to left and thus in depth-first-search root-->tips order. Hence the returned edges are listed in depth-first-search root-->tips order ("cladewise" in the terminology of ape).
// [[Rcpp::export]]
Rcpp::List read_Newick_string_CPP(	std::string	input,
									const bool	underscores_as_blanks,
									const bool	interpret_quotes,
									const bool	look_for_edge_names,
									const bool	look_for_edge_numbers){
	// remove any newline characters
	input.erase(std::remove(input.begin(), input.end(), '\n'), input.end());
	
	// trim any whitespace
	input = trim_whitespace(input);
	
	// replace underscores with blanks if needed
	if(underscores_as_blanks){
		std::replace(input.begin(), input.end(), '_', ' ');
	}
	
	// estimate number of tips, nodes & edges for pre-allocation purposes
	const long estimated_Nclades  = count_occurrences(input, ',', interpret_quotes) + count_occurrences(input, ')', interpret_quotes);
	const long estimated_Nedges = estimated_Nclades - 1;
	if(estimated_Nclades==0) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Input string does not appear to be a valid Newick tree");
	
	// pre-allocate space
	std::vector<std::string> clade_names, edge_names;
	std::vector<double> edge_lengths;
	std::vector<long> tree_edge, edge_numbers;
	clade_names.reserve(estimated_Nclades);
	edge_lengths.reserve(estimated_Nedges);
	edge_names.reserve(estimated_Nedges);
	edge_numbers.reserve(estimated_Nedges);
	tree_edge.reserve(2*estimated_Nedges);
	
	// prepare auxiliary data structures
	std::vector<long> clade_stack; // keep track of which node we are currently in. clade_stack[n+1] is a child of clade_stack[n]
	long pointer = input.length()-1;
	if(input[pointer]==';') --pointer;
	std::string error, child_name, edge_name;
	double edge_length, root_edge;
	long edge_number;
	
	// read input left<--right
	while(pointer>=0){
		if(clade_stack.empty() && (!clade_names.empty())){
			return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Tree appears to have multiple roots: Reached top tree level prior to reaching left-end of input string, at position "+makeString(pointer+1));
		}
		if(!aux_Newick_extract_next_edge(input, interpret_quotes, look_for_edge_names, look_for_edge_numbers, pointer, child_name, edge_length, edge_name, edge_number, error)){
			return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Invalid child specifier to the left of position "+makeString(pointer+1)+": "+error);
		}
		clade_names.push_back(child_name);
		if(clade_stack.empty()){
			// clade is root
			root_edge = edge_length;
		}else{
			edge_lengths.push_back(edge_length);
			edge_names.push_back(edge_name);
			edge_numbers.push_back(edge_number);
			tree_edge.push_back(clade_stack.back());
			tree_edge.push_back(clade_names.size()-1);
		}
		if(input[pointer]==')'){
			// moving one level deeper, into a new child
			clade_stack.push_back(clade_names.size()-1);
			--pointer;
		}else if(input[pointer]=='('){
			// finished at this level, moving up to parents
			while((pointer>=0) && ((input[pointer]=='(') || std::isspace(input[pointer]))){
				if(input[pointer]=='('){
					if(clade_stack.empty()){
						return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Unbalanced parentheses, found redundant closing '(' at position "+makeString(pointer+1));
					}
					clade_stack.pop_back();
				}
				--pointer;
			}
			if((pointer>=0) && (input[pointer]==',')) --pointer;
			else if((pointer>=0) && (input[pointer]==')')) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Unexpected opening paranthesis ')' at position "+makeString(pointer+1));
		}else{
			// more clades to be extracted at this level
			--pointer;
		}
		if(clade_names.size()%1000==0) Rcpp::checkUserInterrupt(); // abort if the user has interrupted the calling R program
	}
	
	// nothing left to parse, so check if we came back to level 0
	if(!clade_stack.empty()) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Unbalanced parentheses, missing "+makeString(clade_stack.size())+" closing parentheses '(' on the left end");
		
	// re-index clades (tips & nodes) consistent with the phylo format
	const long Nclades = clade_names.size();
	const long Nedges  = edge_lengths.size();
	std::vector<long> old2new_clade;
	long Ntips, Nnodes;
	reindex_clades(	Nclades,
					Nedges,
					tree_edge,
					true,
					Ntips,
					Nnodes,
					old2new_clade);
	const long root = Ntips;
	for(long edge=0; edge<Nedges; ++edge){
		tree_edge[2*edge+0] = old2new_clade[tree_edge[2*edge+0]];
		tree_edge[2*edge+1] = old2new_clade[tree_edge[2*edge+1]];
	}
	vector<string> tip_names(Ntips), node_names(Nnodes);
	for(long clade=0, new_clade; clade<Nclades; ++clade){
		new_clade = old2new_clade[clade];
		if(new_clade<Ntips)  tip_names[new_clade] = clade_names[clade];
		if(new_clade>=Ntips) node_names[new_clade-Ntips] = clade_names[clade];
	}
	
	return Rcpp::List::create(	Rcpp::Named("Ntips") 		= Ntips,
								Rcpp::Named("Nnodes") 		= Nnodes,
								Rcpp::Named("Nedges") 		= Nedges,
								Rcpp::Named("tip_names") 	= Rcpp::wrap(tip_names),
								Rcpp::Named("node_names") 	= Rcpp::wrap(node_names),
								Rcpp::Named("tree_edge")	= Rcpp::wrap(tree_edge),
								Rcpp::Named("edge_lengths")	= Rcpp::wrap(edge_lengths),
								Rcpp::Named("edge_names")	= Rcpp::wrap(edge_names),
								Rcpp::Named("edge_numbers")	= Rcpp::wrap(edge_numbers),
								Rcpp::Named("root")			= root,
								Rcpp::Named("root_edge")	= root_edge, // length of dummy "edge" (lacking a parent) leading into root
								Rcpp::Named("success")		= true);
}






#pragma mark -
#pragma mark Statistics of trait distribution
#pragma mark 




// Auxiliary function to get_trait_depth_consenTRAIT_CPP()
// Assumes a pre-calculated traversal root (root-->tips)
void aux_get_trait_depth_consenTRAIT(	const long 					Ntips,
										const long 					Nnodes,
										const long 					Nedges,
										const long					root,				// integer in Ntips:(Nclades-1)
										const std::vector<long> 	&tree_edge,			// 2D array of size Nedges x 2, flattened in row-major format
										const std::vector<double> 	&edge_length, 		// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
										const std::vector<long>		&state_per_tip,		// 1D std::vector of integer states of size Ntips. <=0 means absence, >0 means presence.
										const double				threshold_fraction,	// minimum fraction of tips in a clade that must share the trait, in order for the clade to be counted towards tau_D. In the original paper by Martiny et al (2013), this was 0.9.
										const bool					count_singletons,	// if true, then singleton tips (i.e. having the trait but not counted towards any positive clade) are included by assuming a non-discovered sister tip having the trait with probability 0.5, and diverging just after the parent node [Martiny et al 2013].
										const bool					weighted,			// if true, then positive clades (i.e. counted towards tauD) are weighted according to the number of positive tips
										const std::vector<long>		&traversal_queue,				// (INPUT) 1D array of size Nclades, with values in 0:(Nclades-1). Traversal queue root-->tips. Generated using the function get_tree_traversal_root_to_tips(include_tips=true).
										const std::vector<long>		&traversal_node2first_edge,		// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
										const std::vector<long>		&traversal_node2last_edge,		// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
										const std::vector<long>		&traversal_edges,				// (INPUT) 1D array of size Nedges, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
										const std::vector<long> 	&clade2parent,					// (INPUT) 1D array of size Nclades, with values in 0:(Nclades-1).
										const std::vector<long>		&incoming_edge_per_clade,		// (INPUT) 1D array of size Nclades, with values in 0:(Nedges-1).
										const std::vector<double>	&node2max_tip_distance,			// (INPUT) 1D array of size Nnodes, specifying the max phylogenetic distance for each node to its tips.
										const double				singleton_threshold,			// (INPUT) phylogenetic distance threshold for counting a node as a single tip if its max distance to its tips is equal to or below this threshold. For example, if this is 0, then nodes whose descendants are all identical, will be considered as singletons.
										const bool					include_list_of_positives,		// (INPUT) if true, a list of the clades found to be positive is returned.
										std::vector<long>			&tips_per_clade,				// (OUTPUT) 1D array of size Nclades, listing the number of tips descending from each clade
										std::vector<long>			&positives_per_clade,			// (OUTPUT) 1D array of size Nclades, listing the number of positive tips descending from each clade.
										std::vector<double> 		&mean_depth_per_clade,			// (OUTPUT) 1D array of size Nclades, listing the mean phylogenetic depth of each clade
										double						&mean_depth,					// (OUTPUT) mean clade depth at which trait is conserved. This is the original tau_D introduced by Martiny et al. (2013).
										double						&var_depth,						// (OUTPUT) variance of clade depth at which trait is conserved.
										double						&min_depth,						// (OUTPUT) minimum clade depth at which trait is conserved.
										double						&max_depth,						// (OUTPUT) maximum clade depth at which trait is conserved.
										long						&Npositives,					// (OUTPUT) number of positive clades counted towards the tauD statistic
										std::vector<long>			&positive_clades,				// (OUTPUT) clade indices that were found to be positive in the trait. Optional output (see option include_list_of_positives). Will include only those clades that are also included in the statistics; in particular, singletons are only included if count_singletons==true.
										bool 						verbose,
										const std::string			&verbose_prefix){
	const long Nclades = Ntips + Nnodes;
	long clade, parent, child, node, incoming_edge;
	positive_clades.clear();

	// count number of tips with the trait ("positives"), for each clade
	// also count cumulative phylogenetic distances to tips, for each clade
	tips_per_clade.assign(Nclades, 0);
	positives_per_clade.assign(Nclades, 0);
	mean_depth_per_clade.assign(Nclades, 0); // initially this will be the cumulative depth_per_clade, normalization is done at the end
	for(long tip=0; tip<Ntips; ++tip){
		tips_per_clade[tip] = 1;
		positives_per_clade[tip] = (state_per_tip[tip]<=0 ? 0 : 1);
	}
	// traverse tips-->root
	for(long q=traversal_queue.size()-1; q>=0; --q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent 			= clade2parent[clade];
		incoming_edge 	= incoming_edge_per_clade[clade];
		positives_per_clade[parent] 	+= positives_per_clade[clade];
		tips_per_clade[parent] 			+= tips_per_clade[clade];
		mean_depth_per_clade[parent] 	+= mean_depth_per_clade[clade] + tips_per_clade[clade] * (edge_length.size()==0 ? 1 : edge_length[incoming_edge]);
	}
	for(clade=0; clade<Nclades; ++clade){
		mean_depth_per_clade[clade] /= tips_per_clade[clade];
	}
	
	// traverse through all nodes for which "almost all" tips share the trait (i.e. "positive" nodes), and calculate their mean depth (--> tau_D statistic)
	// traverse root-->tips, and whenever a node is counted as positive, also mark its sub-clades as counted (without actually counting them)
	double sum_depths 		= 0;
	double sum_sqr_depths 	= 0;
	double total_weight		= 0;
	min_depth 				= NAN_D;
	max_depth 				= NAN_D;
	Npositives		 		= 0;
	std::vector<long> clade_counted(Nclades, false);
	for(long q=0; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		const double fraction_positives = positives_per_clade[clade]/double(tips_per_clade[clade]);
		if((!clade_counted[clade]) && (fraction_positives>=threshold_fraction)){
			if((clade<Ntips) || (node2max_tip_distance[clade-Ntips]<=singleton_threshold)){
				// clade is a singleton, so treat in a special way (or omit)
				// singleton tip sensu Martiny et al (2013): tip having the trait, but not counted towards any "positive" node (because too few of it's neighbors have the trait)
				// according to Martiny, such cases may occur due to undersampling of the tree.
				// this is a modified version of Martiny's: Some nodes may also be counted as singletons, if their descendants are very similar to each other
				if(count_singletons){
					++Npositives;
					const double temp_depth = 0.5*(edge_length.size()==0 ? 1 : edge_length[incoming_edge_per_clade[clade]]);
					total_weight	+= 1;
					sum_depths 		+= temp_depth;
					sum_sqr_depths 	+= SQR(temp_depth);
					if(std::isnan(min_depth) || (min_depth>temp_depth)) min_depth = temp_depth;
					if(std::isnan(max_depth) || (max_depth<temp_depth)) max_depth = temp_depth;
					clade_counted[clade] = true;
					positive_clades.push_back(clade);
				}
			}else{
				// clade is a node with phylogenetic diversity above the singleton threshold
				++Npositives;
				const double temp_depth = mean_depth_per_clade[clade];
				const double weight = (weighted ? positives_per_clade[clade] : 1);
				total_weight	+= weight;
				sum_depths 		+= weight * temp_depth;
				sum_sqr_depths 	+= weight * SQR(temp_depth);
				if(std::isnan(min_depth) || (min_depth>temp_depth)) min_depth = temp_depth;
				if(std::isnan(max_depth) || (max_depth<temp_depth)) max_depth = temp_depth;
				clade_counted[clade] = true;
				positive_clades.push_back(clade);
			}
		}
		if(clade_counted[clade] && (clade>=Ntips)){
			// clade was counted (either because it's positive, or due to an ancestral positive node)
			// so mark its children as counted as well (the state of being counted will propagate all the way to the tips)
			node = clade - Ntips;
			for(long e=traversal_node2first_edge[node]; e<=traversal_node2last_edge[node]; ++e){
				child = tree_edge[traversal_edges[e]*2+1];
				clade_counted[child] = true;
			}
		}
	}
	
	mean_depth = (total_weight==0 ? NAN_D : sum_depths/total_weight);
	var_depth  = (total_weight==0 ? NAN_D : (Npositives==1 ? 0.0 : (sum_sqr_depths/total_weight - SQR(mean_depth))));
}




// Calculate phylogenetic depth of a binary trait (presence/absence) on a tree, using the consenTRAIT metric.
// Reference: Martiny et al (2013). Phylogenetic conservatism of functional traits in microorganisms. ISME Journal. 7:830-838
// consenTRAIT: Consensus Analysis of Phylogenetic Trait Distribution
//
// Input: A phylogenetic tree, and the states of a binary trait on all tips of the tree.
// Output: Mean depth at which the trait varies ("trait depth").
// P-value is probability that random tauD (with randomly reshuffled tip states) would lead to an equal or greater tauD than observed.
// The time complexity of this routine is O(Nedges).
//
// Requirements:
//   Tree must be rooted (root will be determined automatically based on the edges)
//   Tree may include monofurcations or multifurcations.
// [[Rcpp::export]]
Rcpp::List get_trait_depth_consenTRAIT_CPP(	const long 					Ntips,
											const long 					Nnodes,
											const long 					Nedges,
											const std::vector<long> 	&tree_edge,				// 2D array of size Nedges x 2, flattened in row-major format
											const std::vector<double> 	&edge_length, 			// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
											const std::vector<long>		&state_per_tip,			// 1D std::vector of integer states of size Ntips. <=0 means absence, >0 means presence.
											const double				threshold_fraction,		// minimum fraction of tips in a clade that must share the trait, in order for the clade to be counted towards tau_D. In the original paper by Martiny et al (2013), this was 0.9.
											const bool					count_singletons,		// if true, then singleton tips (i.e. having the trait but not counted towards any positive clade) are included by assuming a non-discovered sister tip having the trait with probability 0.5, and diverging just after the parent node [Martiny et al 2013]. If false, leftover singletons will be ignored. This may be useful if you suspect many of them to be false positives.
											const bool					weighted,				// if true, then positive clades (i.e. counted towards tauD) are weighted according to the number of positive tips
											const double				singleton_threshold,	// phylogenetic distance threshold (max distance from node to its tips) for counting a node as a single tip if its diversity is equal to or below this threshold. For example, if this is 0, then nodes whose descendants are all identical, will be considered as singletons.
											const long					Npermutations,			// number of random permutations (re-assignments of states) for estimating P-values
											bool 						verbose,
											const std::string 			&verbose_prefix){
	const long Nclades = Ntips + Nnodes;
	long clade, parent;
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}	
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);
										
	// calculate max phylogenetic distance for each node to any of its tips, traversing tips-->root
	// needed to identify nodes that are essentially tips (i.e. all of its descendants are closely related)
	std::vector<double> node2max_tip_distance(Nnodes,0);
	for(long q=traversal_queue.size()-1; q>=0; --q){
		clade  = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		node2max_tip_distance[parent-Ntips] = max(node2max_tip_distance[parent-Ntips], (clade<Ntips ? 0.0 : node2max_tip_distance[clade-Ntips]) + (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]));
	}
				
	// get observed tauD
	std::vector<long> current_state_per_tip = state_per_tip;
	std::vector<long> positive_clades, positives_per_clade, tips_per_clade;
	std::vector<double> mean_depth_per_clade;
	double tauD, varD, minD, maxD;
	long Npositives;
	aux_get_trait_depth_consenTRAIT(Ntips,
									Nnodes,
									Nedges,
									root,
									tree_edge,
									edge_length,
									current_state_per_tip,
									threshold_fraction,
									count_singletons,
									weighted,
									traversal_queue,
									traversal_node2first_edge,
									traversal_node2last_edge,
									traversal_edges,
									clade2parent,
									incoming_edge_per_clade,
									node2max_tip_distance,
									singleton_threshold,
									true,
									tips_per_clade,
									positives_per_clade,
									mean_depth_per_clade,
									tauD,
									varD,
									minD,
									maxD,
									Npositives,
									positive_clades,
									verbose,
									verbose_prefix);
														
	// estimate P-value of observed tauD, by randomly shuffling tip states
	double Pvalue = NAN_D;
	double mean_random_tauD = NAN_D;
	if(Npermutations>0){
		long count_deeper 				= 0;
		double sum_random_tauD 			= 0;
		long count_valid_permutations 	= 0;
		double random_tauD, random_varD, random_minD, random_maxD;
		long random_Npositives;
		lvector positive_clades, dummy_positive_clades, dummy_positives_per_clade, dummy_tips_per_clade;
		dvector dummy_mean_depth_per_clade;
		for(long p=0; p<Npermutations; ++p){
			random_shuffle_insitu(current_state_per_tip, 0, Ntips-1); // randomly shuffle tip states
			aux_get_trait_depth_consenTRAIT(Ntips,
											Nnodes,
											Nedges,
											root,
											tree_edge,
											edge_length,
											current_state_per_tip,
											threshold_fraction,
											count_singletons,
											weighted,
											traversal_queue,
											traversal_node2first_edge,
											traversal_node2last_edge,
											traversal_edges,
											clade2parent,
											incoming_edge_per_clade,
											node2max_tip_distance,
											singleton_threshold,
											false,
											dummy_tips_per_clade,
											dummy_positives_per_clade,
											dummy_mean_depth_per_clade,
											random_tauD,
											random_varD,
											random_minD,
											random_maxD,
											random_Npositives,
											dummy_positive_clades,
											false,
											verbose_prefix);
			if(!std::isnan(random_tauD)){
				++count_valid_permutations;
				if(random_tauD>=tauD) ++count_deeper;
				sum_random_tauD += random_tauD;
			}
		}
		Pvalue = (count_valid_permutations>0 ? count_deeper/double(count_valid_permutations) : NAN_D);
		mean_random_tauD = (count_valid_permutations>0 ? sum_random_tauD/count_valid_permutations : NAN_D);
	}

	return Rcpp::List::create(	Rcpp::Named("tauD") 					= tauD,
								Rcpp::Named("varD") 					= varD,
								Rcpp::Named("minD") 					= minD,
								Rcpp::Named("maxD") 					= maxD,
								Rcpp::Named("Npositives") 				= Npositives,
								Rcpp::Named("tips_per_clade")			= Rcpp::wrap(tips_per_clade),
								Rcpp::Named("positive_clades")			= Rcpp::wrap(positive_clades),
								Rcpp::Named("positives_per_clade")		= Rcpp::wrap(positives_per_clade),
								Rcpp::Named("mean_depth_per_clade")		= Rcpp::wrap(mean_depth_per_clade),
								Rcpp::Named("Pvalue") 					= Pvalue,
								Rcpp::Named("mean_random_tauD")			= mean_random_tauD);
}








// Auxiliary function to get_discrete_trait_depth_CPP()
// Assumes a pre-calculated traversal root (root-->tips)
void aux_get_discrete_trait_depths(	const long 					Ntips,
									const long 					Nnodes,
									const long 					Nedges,
									const long 					Nstates,
									const long					root,				// integer in Ntips:(Nclades-1)
									const std::vector<long> 	&tree_edge,			// 2D array of size Nedges x 2, flattened in row-major format
									const std::vector<double> 	&edge_length, 		// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
									const std::vector<long>		&state_per_tip,		// 1D std::vector of integer states of size Ntips, with values in 0,..,Nstates
									const double				threshold_fraction,	// minimum fraction of tips in a clade that must share a specific state, in order for the clade to be considered to have this "uniform state". In the original paper by Martiny et al (2013), this was 0.9. It should be strictly greater than 0.5.
									const bool					count_singletons,
									const bool					weighted,						// if true, then uniform clades are weighted according to the number of tips in the dominant state
									const std::vector<long>		&traversal_queue,				// (INPUT) 1D array of size Nclades, with values in 0:(Nclades-1). Traversal queue root-->tips. Generated using the function get_tree_traversal_root_to_tips(include_tips=true).
									const std::vector<long>		&traversal_node2first_edge,		// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
									const std::vector<long>		&traversal_node2last_edge,		// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
									const std::vector<long>		&traversal_edges,				// (INPUT) 1D array of size Nedges, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
									const std::vector<long> 	&clade2parent,					// (INPUT) 1D array of size Nclades, with values in 0:(Nclades-1).
									const std::vector<long>		&incoming_edge_per_clade,		// (INPUT) 1D array of size Nclades, with values in 0:(Nedges-1).
									const std::vector<double>	&node2max_tip_distance,			// (INPUT) 1D array of size Nnodes, specifying the max phylogenetic distance for each node to its tips.
									const double				singleton_threshold,			// (INPUT) phylogenetic distance threshold for counting a node as a single tip if its max distance to its tips is equal to or below this threshold. For example, if this is 0, then nodes whose descendants are all identical, will be considered as singletons.
									std::vector<long>			&dominant_state_per_clade,		// (OUTPUT) 1D array of size Nclades, listing the dominant (most frequent) state for each clade
									double						&mean_depth,					// (OUTPUT) mean phylogenetic depth of all maximal uniform clades, regardless of its dominant state
									double						&var_depth,						// (OUTPUT) variance of phylogenetic depths of all maximal uniform clades
									double						&min_depth,						// (OUTPUT) minimum phylogenetic depth of all maximal uniform clades
									double						&max_depth,						// (OUTPUT) maximum phylogenetic depth of all maximal uniform clades
									long						&Nmax_uniform,					// (OUTPUT) number of maximal uniform clades, regardless of state
									std::vector<double>			&mean_depth_per_state,			// (OUTPUT) mean depth of maximal uniform clades, separately for each state.
									std::vector<double>			&var_depth_per_state,			// (OUTPUT) variance of depth of maximal uniform clades.
									std::vector<double>			&min_depth_per_state,			// (OUTPUT) minimum phylogenetic depth of all maximal uniform clades
									std::vector<double>			&max_depth_per_state,			// (OUTPUT) maximum phylogenetic depth of all maximal uniform clades
									std::vector<long>			&Nmax_uniform_per_state,		// (OUTPUT) number of maximal uniform clades, separately for each possible state
									bool 						verbose,
									const std::string			&verbose_prefix){
	const long Nclades = Ntips + Nnodes;
	long clade, parent, child, node, incoming_edge;

	// count number of tips with in each state and for each clade
	// also count cumulative phylogenetic distances to tips, for each clade
	lvector tips_per_clade(Nclades, 0);
	lvector tips_per_clade_per_state(Nclades*Nstates, 0); // 2D array of size Nclades x Nstates, in row-major format
	dvector mean_depth_per_clade(Nclades, 0); // initially this will be the cumulative depth_per_clade, normalization is done at the end
	for(long tip=0; tip<Ntips; ++tip){
		tips_per_clade[tip] = 1;
		tips_per_clade_per_state[tip*Nstates+state_per_tip[tip]] = 1;
	}
	// traverse tips-->root
	for(long q=traversal_queue.size()-1; q>=0; --q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent 			= clade2parent[clade];
		incoming_edge 	= incoming_edge_per_clade[clade];
		for(long state=0; state<Nstates; ++state){
			tips_per_clade_per_state[parent*Nstates+state] += tips_per_clade_per_state[clade*Nstates+state];
		}
		tips_per_clade[parent] 			+= tips_per_clade[clade];
		mean_depth_per_clade[parent] 	+= mean_depth_per_clade[clade] + tips_per_clade[clade] * (edge_length.size()==0 ? 1 : edge_length[incoming_edge]);
	}
	for(clade=0; clade<Nclades; ++clade){
		mean_depth_per_clade[clade] /= tips_per_clade[clade];
	}
	
	// determine dominant state for each clade
	dominant_state_per_clade.assign(Nclades,-1);
	for(clade=0; clade<Nclades; ++clade){
		dominant_state_per_clade[clade] = array_argmax(Nstates,&tips_per_clade_per_state[clade*Nstates]);
	}
	
	// traverse through all nodes with a uniform state, i.e. for which "almost all" tips share same state, and calculate their mean depth (--> tau_D statistic)
	// traverse root-->tips, and whenever a node is counted as a uniform node, also mark its sub-clades as counted (without actually counting them)
	double total_weight 	= 0;
	double sum_depths 		= 0;
	double sum_sqr_depths 	= 0;
	Nmax_uniform				= 0;
	min_depth 				= NAN_D;
	max_depth 				= NAN_D;
	min_depth_per_state.assign(Nstates,NAN_D);
	max_depth_per_state.assign(Nstates,NAN_D);
	dvector total_weight_per_state(Nstates,0);
	dvector sum_depths_per_state(Nstates,0);
	dvector sum_sqr_depths_per_state(Nstates,0);
	lvector clade_counted(Nclades, false);
	Nmax_uniform_per_state.assign(Nstates, 0);
	for(long q=0, dominant_state; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		dominant_state = dominant_state_per_clade[clade];
		const double fraction_dominant = tips_per_clade_per_state[clade*Nstates+dominant_state]/double(tips_per_clade[clade]);
		if((!clade_counted[clade]) && (fraction_dominant>=threshold_fraction)){
			if((clade<Ntips) || (node2max_tip_distance[clade-Ntips]<=singleton_threshold)){
				// clade is a singleton, so treat in a special way (or omit)
				// singleton tip sensu Martiny et al (2013): tip having the trait, but not considered to be a uniform node (because too few of it's neighbors have the state)
				// according to Martiny, such cases may occur due to undersampling of the tree.
				// this is a modified version of Martiny's: Some nodes may also be counted as singletons, if their descendants are very similar to each other
				if(count_singletons){
					clade_counted[clade] = true;
					const double temp_depth = 0.5*(edge_length.size()==0 ? 1 : edge_length[incoming_edge_per_clade[clade]]);

					++Nmax_uniform;
					total_weight	+= 1;
					sum_depths 		+= temp_depth;
					sum_sqr_depths 	+= SQR(temp_depth);
					if(std::isnan(min_depth) || (min_depth>temp_depth)) min_depth = temp_depth;
					if(std::isnan(max_depth) || (max_depth<temp_depth)) max_depth = temp_depth;

					++Nmax_uniform_per_state[dominant_state];
					total_weight_per_state[dominant_state]	 += 1;
					sum_depths_per_state[dominant_state] 	 += temp_depth;
					sum_sqr_depths_per_state[dominant_state] += SQR(temp_depth);
					if(std::isnan(min_depth_per_state[dominant_state]) || (min_depth_per_state[dominant_state]>temp_depth)) min_depth_per_state[dominant_state] = temp_depth;
					if(std::isnan(max_depth_per_state[dominant_state]) || (max_depth_per_state[dominant_state]<temp_depth)) max_depth_per_state[dominant_state] = temp_depth;
				}
			}else{
				// clade is a node with phylogenetic diversity above the singleton threshold
				clade_counted[clade] = true;
				const double temp_depth = mean_depth_per_clade[clade];
				const double weight 	= (weighted ? tips_per_clade_per_state[clade*Nstates+dominant_state] : 1);

				++Nmax_uniform;
				total_weight	+= weight;
				sum_depths 		+= weight * temp_depth;
				sum_sqr_depths 	+= weight * SQR(temp_depth);
				if(std::isnan(min_depth) || (min_depth>temp_depth)) min_depth = temp_depth;
				if(std::isnan(max_depth) || (max_depth<temp_depth)) max_depth = temp_depth;

				++Nmax_uniform_per_state[dominant_state];
				total_weight_per_state[dominant_state]	 += weight;
				sum_depths_per_state[dominant_state] 	 += weight * temp_depth;
				sum_sqr_depths_per_state[dominant_state] += weight * SQR(temp_depth);
				if(std::isnan(min_depth_per_state[dominant_state]) || (min_depth_per_state[dominant_state]>temp_depth)) min_depth_per_state[dominant_state] = temp_depth;
				if(std::isnan(max_depth_per_state[dominant_state]) || (max_depth_per_state[dominant_state]<temp_depth)) max_depth_per_state[dominant_state] = temp_depth;
			}
		}
		if(clade_counted[clade] && (clade>=Ntips)){
			// clade was counted (either because it's a uniform clade, or due to an ancestral uniform clade)
			// so mark its children as counted as well (the state of being counted will propagate all the way to the tips)
			node = clade - Ntips;
			for(long e=traversal_node2first_edge[node]; e<=traversal_node2last_edge[node]; ++e){
				child = tree_edge[traversal_edges[e]*2+1];
				clade_counted[child] = true;
			}
		}
	}
	
	mean_depth = (total_weight==0 ? NAN_D : sum_depths/total_weight);
	var_depth  = (total_weight==0 ? NAN_D : (Nmax_uniform==1 ? 0.0 : (sum_sqr_depths/total_weight - SQR(mean_depth))));
	mean_depth_per_state.assign(Nstates,NAN_D);
	var_depth_per_state.assign(Nstates,NAN_D);
	for(long state=0; state<Nstates; ++state){
		mean_depth_per_state[state] = (total_weight_per_state[state]==0 ? NAN_D : sum_depths_per_state[state]/total_weight_per_state[state]);
		var_depth_per_state[state]  = (total_weight_per_state[state]==0 ? NAN_D : (Nmax_uniform_per_state[state]==1 ? 0.0 : (sum_sqr_depths_per_state[state]/total_weight_per_state[state] - SQR(mean_depth_per_state[state]))));		
	}
}



// Calculate phylogenetic depth of a discrete trait on a tree, using a variant of the consenTRAIT metric.
// A clade is said to be "uniform" in a state s if that state is dominant in the clade and its frequency exceeds the threshold_fraction
// A clade is "maximal uniform" if it is uniform and is not descending from another uniform clade.
// For every possible state s, the clades being maximal uniform in state s are determined and their phylogenetic depth is averaged, to obtain a state-specific tauD_s
// A universal tauD is determined by calculating the average phylogenetic depth of all maximal uniform clades.
// This approach is a modification of the consenTRAIT metric originally described for binary (presence/absence) traits
//
// Reference: Martiny et al (2013). Phylogenetic conservatism of functional traits in microorganisms. ISME Journal. 7:830-838
// consenTRAIT: Consensus Analysis of Phylogenetic Trait Distribution
//
// Input: A phylogenetic tree, and the states of a discrete trait on all tips of the tree, as integers from 0 to Nstates-1
// Output: Mean depth at which the trait varies ("trait depth").
// P-value is probability that random tauD (with randomly reshuffled tip states) would lead to an equal or greater tauD than observed.
// The time complexity of this routine is O(Nedges * Nstates).
//
// Requirements:
//   Tree must be rooted (root will be determined automatically based on the edges)
//   Tree may include monofurcations or multifurcations.
// [[Rcpp::export]]
Rcpp::List get_discrete_trait_depth_CPP(const long 					Ntips,
										const long 					Nnodes,
										const long 					Nedges,
										const long 					Nstates,
										const std::vector<long> 	&tree_edge,				// 2D array of size Nedges x 2, flattened in row-major format
										const std::vector<double> 	&edge_length, 			// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
										const std::vector<long>		&state_per_tip,			// 1D std::vector of integer states of size Ntips, with values in 0,..,Nstates-1
										const double				threshold_fraction,		// minimum fraction of tips in a clade with identical state, in order for the clade to be considered as having the particular "uniform state". In the original paper by Martiny et al (2013), this was 0.9.
										const bool					count_singletons,		// if true, then singleton tips (i.e. having a state but not counted towards any parent clade) are considered to have that uniform state by assuming a non-discovered sister tip having the same state with probability 0.5, and diverging just after the parent node [Martiny et al 2013]. If false, leftover singletons will be ignored. This may be useful if you suspect many of them to be false positives.
										const bool					weighted,				// if true, then uniform clades are weighted according to the number of tips in the dominant state
										const double				singleton_threshold,	// phylogenetic distance threshold (max distance from node to its tips) for counting a node as a single tip if its diversity is equal to or below this threshold. For example, if this is 0, then nodes whose descendants are all identical, will be considered as singletons.
										const long					Npermutations,			// number of random permutations (state reshufflings) for estimating P-values
										bool 						verbose,
										const std::string 			&verbose_prefix){
	const long Nclades = Ntips + Nnodes;
	long clade, parent;
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}	
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);
										
	// calculate max phylogenetic distance for each node to any of its tips, traversing tips-->root
	// needed to identify nodes that are essentially tips (i.e. all of its descendants are closely related)
	std::vector<double> node2max_tip_distance(Nnodes,0);
	for(long q=traversal_queue.size()-1; q>=0; --q){
		clade  = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		node2max_tip_distance[parent-Ntips] = max(node2max_tip_distance[parent-Ntips], (clade<Ntips ? 0.0 : node2max_tip_distance[clade-Ntips]) + (edge_length.size()==0 ? 1.0 : edge_length[incoming_edge_per_clade[clade]]));
	}
					
	// get observed tauD_s for every state s
	long Nmax_uniform;
	double tauD, varD, minD, maxD;
	lvector current_state_per_tip = state_per_tip;
	lvector dominant_state_per_clade, Nmax_uniform_per_state;
	dvector tauD_per_state, varD_per_state, minD_per_state, maxD_per_state;
	aux_get_discrete_trait_depths(	Ntips,
									Nnodes,
									Nedges,
									Nstates,
									root,
									tree_edge,
									edge_length,
									current_state_per_tip,
									threshold_fraction,
									count_singletons,
									weighted,
									traversal_queue,
									traversal_node2first_edge,
									traversal_node2last_edge,
									traversal_edges,
									clade2parent,
									incoming_edge_per_clade,
									node2max_tip_distance,
									singleton_threshold,
									dominant_state_per_clade,
									tauD,
									varD,
									minD,
									maxD,
									Nmax_uniform,
									tauD_per_state,
									varD_per_state,
									minD_per_state,
									maxD_per_state,
									Nmax_uniform_per_state,
									verbose,
									verbose_prefix);

														
	// estimate P-value of observed tauD, by randomly shuffling tip states
	double Pvalue = NAN_D;
	double mean_random_tauD = NAN_D;
	if(Npermutations>0){
		long count_deeper 				= 0;
		double sum_random_tauD 			= 0;
		long count_valid_permutations 	= 0;
		double random_tauD, random_varD, random_minD, random_maxD;
		long random_Nmax_uniform;
		dvector random_tauD_per_state, random_varD_per_state, random_minD_per_state, random_maxD_per_state, mean_random_tauD_per_state;
		lvector random_dominant_state_per_clade, random_Nmax_uniform_per_state;
		for(long p=0; p<Npermutations; ++p){
			random_shuffle_insitu(current_state_per_tip, 0, Ntips-1); // randomly shuffle tip states
			aux_get_discrete_trait_depths(Ntips,
											Nnodes,
											Nedges,
											Nstates,
											root,
											tree_edge,
											edge_length,
											current_state_per_tip,
											threshold_fraction,
											count_singletons,
											weighted,
											traversal_queue,
											traversal_node2first_edge,
											traversal_node2last_edge,
											traversal_edges,
											clade2parent,
											incoming_edge_per_clade,
											node2max_tip_distance,
											singleton_threshold,
											random_dominant_state_per_clade,
											random_tauD,
											random_varD,
											random_minD,
											random_maxD,
											random_Nmax_uniform,
											random_tauD_per_state,
											random_varD_per_state,
											random_minD_per_state,
											random_maxD_per_state,
											random_Nmax_uniform_per_state,
											false,
											verbose_prefix);
			if(!std::isnan(random_tauD)){
				++count_valid_permutations;
				if(random_tauD>=tauD) ++count_deeper;
				sum_random_tauD += random_tauD;
			}
		}
		if(count_valid_permutations>0){
			Pvalue = count_deeper/double(count_valid_permutations);
			mean_random_tauD = sum_random_tauD/count_valid_permutations;
		}else{
			Pvalue = NAN_D;
			mean_random_tauD = NAN_D;
		}
	}

	return Rcpp::List::create(	Rcpp::Named("tauD") 				= tauD,
								Rcpp::Named("varD") 				= varD,
								Rcpp::Named("minD") 				= minD,
								Rcpp::Named("maxD") 				= maxD,
								Rcpp::Named("Nmax_uniform") 			= Nmax_uniform,
								Rcpp::Named("tauD_per_state") 		= tauD_per_state,
								Rcpp::Named("varD_per_state") 		= varD_per_state,
								Rcpp::Named("minD_per_state") 		= minD_per_state,
								Rcpp::Named("maxD_per_state") 		= maxD_per_state,
								Rcpp::Named("Nmax_uniform_per_state")	= Nmax_uniform_per_state,
								Rcpp::Named("Pvalue") 				= Pvalue,
								Rcpp::Named("mean_random_tauD")		= mean_random_tauD);
}





// Calculate phylogenetic ("spatial") autocorrelation (AC) function of a continuous trait on tips of a tree
// AC(x) = correlation between the trait values of two random tips at distance x from each other
// Distance x = cumulative branch length between the two tips (aka "patristic distance")
// The autocorrelation function is estimated on a discrete distance grid (with x ranging from 0 to max)
//
// Input: 
//   A phylogenetic tree, and the states of a continuous trait on all tips of the tree.
// Output: 
//   The distance grid on which AC was estimated.
//	 Estimated AC(x) for each x on the distance grid
//   The number of tip pairs used to estimate AC(x) for each x on the distance grid
//	The mean absolute deviation between trait values of tips at distance x, for each x on the distance grid
//
// The time complexity of this routine is O(Ntips + Npairs*Nanc), where Nanc is the typical number of ancestors per tip.
// The memory complexity is O(Nedges+Npairs)
//
// Requirements:
//   Tree must be rooted (root will be determined automatically based on the edges)
//   Tree may include monofurcations or multifurcations.
//
// [[Rcpp::export]]
Rcpp::List ACF_continuous_trait_CPP(const long 					Ntips,
									const long 					Nnodes,
									const long 					Nedges,
									const std::vector<long> 	&tree_edge,			// 2D array of size Nedges x 2, flattened in row-major format
									const std::vector<double> 	&edge_length, 		// 1D array of size Nedges, or an empty std::vector (all branches have length 1)
									const std::vector<double>	&state_per_tip,		// 1D std::vector of numeric states of size Ntips.
									long						max_Npairs,			// maximum number of random tip pairs for estimating the correlation function. If <=0, all possible pairs are used.
									const std::vector<double>	&phylodistance_grid,// 1D array of size Nbins, specifying the left-edges of the phylodistance bins in which to estimate the ACF. Must be in strictly ascending order.
									const double				max_phylodistance,	// maximum phylodistance to consider, i.e. the right-edge of the right-most phylodistance bin. Hence, the first bin extends from phylodistance_grid[1] to phylodistance_grid[2], while the last cell extends from phylodistance_grid.back() to max_phylodistance
									const bool					grid_is_uniform,	// whether the provided phylodistance_grid[] is uniform (except perhaps for the last grid cell) 
									bool 						verbose,
									const std::string 			&verbose_prefix){	
	const long Nbins = phylodistance_grid.size();

	// pick tip pairs to include
	lvector tipsA, tipsB;
	long Npairs = Ntips*(Ntips-1)/2+Ntips;
	if(max_Npairs >= Npairs){
		// include every possible tip pair exactly once
		tipsA.resize(Npairs);
		tipsB.resize(Npairs);
		for(long n=0, p=0, m; n<Ntips; ++n){
			for(m=n; m<Ntips; ++m){
				tipsA[p] = n;
				tipsB[p] = m;
				++p;
			}
		}
	}else{
		// pick a random subset of max_Npairs tips
		Npairs = max_Npairs;
		tipsA.resize(Npairs);
		tipsB.resize(Npairs);
		for(long p=0; p<Npairs; ++p){
			tipsA[p] = uniformIntWithin(0,Ntips-1);
			tipsB[p] = uniformIntWithin(0,Ntips-1);
		}
	}
	
	// calculate distance for each tip pair
	const dvector phylodistances = Rcpp::as<std::vector<double> >(
								get_distances_between_clades_CPP(	Ntips,
																	Nnodes,
																	Nedges,
																	tree_edge,
																	edge_length,
																	tipsA,
																	tipsB,
																	verbose,
																	verbose_prefix));
																		
	// calculate correlation within each distance-bin
	// use empirical means & variances for tipsA and tipsB separately, and separately for each bin, to ensure that empirical autocorrelations are always within [-1,1].
	const double min_phylodistance = phylodistance_grid[0];
	const double dx = ((grid_is_uniform && (Nbins>0)) ? phylodistance_grid[1]-phylodistance_grid[0] : NAN_D);
	double max_encountered_phylodistance = 0;
	std::vector<double> meanA_per_bin(Nbins,0);
	std::vector<double> meanB_per_bin(Nbins,0);
	std::vector<long>   pairs_per_bin(Nbins,0);
	for(long p=0, bin=-1; p<Npairs; ++p){
		if((phylodistances[p]<min_phylodistance) || (phylodistances[p]>max_phylodistance)) continue; // this phylodistance is outside of the considered range
		if(grid_is_uniform){
			bin = max(0L,min(Nbins-1,long(floor((phylodistances[p]-min_phylodistance)/dx))));
		}else{
			bin = find_next_left_grid_point(phylodistance_grid, phylodistances[p], bin);
		}
		pairs_per_bin[bin] += 1;
		meanA_per_bin[bin] += state_per_tip[tipsA[p]];
		meanB_per_bin[bin] += state_per_tip[tipsB[p]];
	}
	for(long bin=0; bin<Nbins; ++bin){
		if(pairs_per_bin[bin]==0) continue;
		meanA_per_bin[bin] /= pairs_per_bin[bin];
		meanB_per_bin[bin] /= pairs_per_bin[bin];
	}
	double stateA, stateB;
	std::vector<double> varA_per_bin(Nbins,0);
	std::vector<double> varB_per_bin(Nbins,0);
	std::vector<double> autocorrelations(Nbins,0);
	std::vector<double> mean_abs_deviations(Nbins,0);
	std::vector<double> mean_rel_deviations(Nbins,0);
	for(long p=0, bin=-1; p<Npairs; ++p){
		if((phylodistances[p]<min_phylodistance) || (phylodistances[p]>max_phylodistance)) continue; // this phylodistance is outside of the considered range
		stateA 	= state_per_tip[tipsA[p]];
		stateB 	= state_per_tip[tipsB[p]];
		if(grid_is_uniform){
			bin = max(0L,min(Nbins-1,long(floor((phylodistances[p]-min_phylodistance)/dx))));
		}else{
			bin = find_next_left_grid_point(phylodistance_grid, phylodistances[p], bin);
		}
		varA_per_bin[bin] += SQR(stateA-meanA_per_bin[bin]);
		varB_per_bin[bin] += SQR(stateB-meanB_per_bin[bin]);
		autocorrelations[bin] 	 += (stateA-meanA_per_bin[bin])*(stateB-meanB_per_bin[bin]);
		mean_abs_deviations[bin] += abs(stateA-stateB);
		mean_rel_deviations[bin] += (stateA==stateB ? 0.0 : abs(stateA-stateB)/(0.5*(abs(stateA)+abs(stateB))));
		max_encountered_phylodistance = max(max_encountered_phylodistance, phylodistances[p]);
		if((p%1000)==0) Rcpp::checkUserInterrupt();
	}
	for(long bin=0; bin<Nbins; ++bin){
		if(pairs_per_bin[bin]==0){
			varA_per_bin[bin] = NAN_D;
			varB_per_bin[bin] = NAN_D;
			autocorrelations[bin] 	 = NAN_D;
			mean_abs_deviations[bin] = NAN_D;
			mean_rel_deviations[bin] = NAN_D;
		}else{
			varA_per_bin[bin] /= pairs_per_bin[bin];
			varB_per_bin[bin] /= pairs_per_bin[bin];
			autocorrelations[bin] 	 /= pairs_per_bin[bin] * sqrt(varA_per_bin[bin]*varB_per_bin[bin]);
			mean_abs_deviations[bin] /= pairs_per_bin[bin];
			mean_rel_deviations[bin] /= pairs_per_bin[bin];
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("N_per_grid_point")				= Rcpp::wrap(pairs_per_bin),
								Rcpp::Named("autocorrelations") 			= Rcpp::wrap(autocorrelations),
								Rcpp::Named("mean_rel_deviations") 			= Rcpp::wrap(mean_rel_deviations),
								Rcpp::Named("mean_abs_deviations") 			= Rcpp::wrap(mean_abs_deviations),
								Rcpp::Named("max_encountered_phylodistance")= Rcpp::wrap(max_encountered_phylodistance));
}





// For each node in a tree, calculate the empirical frequencies of states of a discrete trait, based on the states of descending tips
// This may be a very crude reconstruction of ancestral state probabilities (when normalized)
// Returns a 2D integer array of size Nnodes x Nstates, in row-major format
// [[Rcpp::export]]
Rcpp::List get_empirical_state_frequencies_per_node_CPP(	const long			Ntips,
															const long			Nnodes,
															const long			Nedges,
															const long			Nstates,		// (INPUT) number of discrete states for the trait
															const std::vector<long> &tree_edge,		// (INPUT) 2D array (in row-major format) of size Nedges x 2, or an empty std::vector (no tree available). A tree is needed if the tip_distribution relies on a tree structure.
															const std::vector<long>	&tip_states){	// (INPUT) 1D array of size Ntips, listing the discrete state for each tip
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// get tree traversal route (tips --> root)
	// traversal_queue[] will be of size Nclades, and will have entries in 0:(Nclades-1)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,	// include tips
										false,	// edge mapping is not pre-computed
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");
	
	// calculate empirical frequencies per node, traversing tips-->root (excluding the root)
	std::vector<long> frequencies_per_node(Nnodes*Nstates,0); // 2D array in row-major format
	long clade, parent;
	for(long q=traversal_queue.size()-1; q>=1; --q){
		clade	= traversal_queue[q];
		parent 	= clade2parent[clade];
		if(clade<Ntips){
			frequencies_per_node[(parent-Ntips)*Nstates+tip_states[clade]] += 1;
		}else{
			for(long s=0; s<Nstates; ++s) frequencies_per_node[(parent-Ntips)*Nstates + s] += frequencies_per_node[(clade-Ntips)*Nstates + s];
		}
	}
	
	return Rcpp::List::create(Rcpp::Named("frequencies_per_node") = Rcpp::wrap(frequencies_per_node));
}





// [[Rcpp::export]]
Rcpp::List get_trait_richness_collectors_curve_CPP(	const long			Ntips,
													const long 			Nnodes,
													const long			Nedges,
													const long			Ntraits,
													const long 			root, 					// (INPUT) index of root node, i.e. an integer in 0:(Ntips+Nnodes-1)
													const std::vector<long> &tree_edge, 			// (INPUT) 2D array (in row-major format) of size Nedges x 2, or an empty std::vector (no tree available). A tree is needed if the tip_distribution relies on a tree structure.
													const std::vector<double> &edge_length, 			// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1). Only relevant if tip_distribution=="phylogenetic_diversity".
													const std::vector<long>	&tip2first_trait,		// (INPUT) 1D array of size Nnodes, with values in 0:(NCtraits-1).
													const std::vector<long>	&tip2last_trait,		// (INPUT) 1D array of size Nnodes, with values in 0:(NCtraits-1).
													const std::vector<long>	&traits,				// (INPUT) 1D array of size NCtraits, with values in 0:(Ntraits-1)
													const std::vector<long>	&rarefaction_depths,	// (INPUT) 1D array of size Ndepths, defining the different rarefaction depths at which to estimate the collector's curve.
													const long			Nrepeats,				// (INPUT) number of random repeats for calculating the average collector's curve at a given rarefaction depth
													const std::string	&tip_distribution,		// (INPUT) probability distribution for randomly choosing tips. Options are "uniform_tips" (each tip is equally likely), "uniform_children" (traversing root-->tips, with each child of a node being equally likely), “uniform_tips_without_replacement" (choose random tip without replacement), "uniform_children_without_replacement"
													const bool			use_realized_depths){	// (INPUT) if true, the rarefaction_depths are interpreted as centroids of realized rarefaction depth intervals, and the collector's curve is calculated as a function of realized rarefaction depth (rather than imposed rarefaction depth). Only relevant if tip_distribution is with replacement.
	
	const long Nclades = Ntips+Nnodes;
	const long Ndepths = rarefaction_depths.size();
	long tip, clade, count_tips_remaining;
	
	const bool distribution_uniform_tips 				= (tip_distribution=="uniform_tips");
	const bool distribution_uniform_tips_wr 			= (tip_distribution=="uniform_tips_without_replacement");
	const bool distribution_uniform_children 			= (tip_distribution=="uniform_children");
	const bool distribution_uniform_children_wr			= (tip_distribution=="uniform_children_without_replacement");
	const bool distribution_phylogenetic_diversity		= (tip_distribution=="phylogenetic_diversity");
	const bool need_tree 								= (distribution_uniform_children || distribution_uniform_children_wr || distribution_phylogenetic_diversity);

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	if(need_tree) get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
		
	// prepare access structures for random tip selection
	std::vector<long> node2first_child, node2last_child, children;
	std::vector<long> node2total_tip_count;
	std::vector<double> clade2weight, node2diversity;
	if(distribution_uniform_children || distribution_uniform_children_wr || distribution_phylogenetic_diversity){
		get_children_per_node(	Ntips,
								Nnodes,
								Nedges,
								root,
								tree_edge,
								node2first_child,
								node2last_child,
								children);
	}
	if(distribution_uniform_children_wr){
		get_total_tip_count_per_node(	Ntips,
										Nnodes,
										Nedges,
										tree_edge,
										node2total_tip_count);
	}
	if(distribution_phylogenetic_diversity){
		// calculate phylogenetic diversities (cumulative branch lengths) for nodes
		get_cumulative_edge_lengths_per_node(	Ntips,
												Nnodes,
												Nedges,
												root,
												tree_edge,
												edge_length,
												node2diversity);
		// assign weights to children (for random traversal root-->tips) proportional to their phylogenetic diversity (PD)
		// set tip-children (and children with zero PD) to the average weight as their non-tip sister clades.
		clade2weight.resize(Nclades);
		for(long node=0; node<Nnodes; ++node){
			double cumulative_PD 	= 0;
			long tip_count 			= 0;
			const long child_count 	= node2last_child[node] - node2first_child[node] + 1;
			for(long c=node2first_child[node], child; c<=node2last_child[node]; ++c){
				child = children[c];
				if((child<Ntips) || (node2diversity[child-Ntips]==0)){
					tip_count += 1;
				}else{
					cumulative_PD += node2diversity[child-Ntips];
				}
			}
			const double PD_per_tip = ((tip_count<child_count) ? cumulative_PD/(child_count-tip_count) : 1.0);
			for(long c=node2first_child[node], child; c<=node2last_child[node]; ++c){
				child = children[c];
				clade2weight[child] = (((child<Ntips) || (node2diversity[child-Ntips]==0)) ? PD_per_tip : node2diversity[child-Ntips]);
			}			
		}
		clade2weight[root] = node2diversity[root-Ntips];
	}
	std::vector<long> tips_remaining_per_node;
								
	// calculate trait_richness collector's curve
	std::vector<char> 	tip_included;
	std::vector<char> 	trait_included(Ntraits);
	std::vector<double> trait_richness_means(Ndepths,0);
	std::vector<double> trait_richness_stds(Ndepths,0);
	std::vector<double> trait_richness_mins(Ndepths,Ntraits+1);
	std::vector<double> trait_richness_maxs(Ndepths,-1);
	std::vector<long>	Nrepeats_per_depth(Ndepths,0);
	for(long d=0; d<Ndepths; ++d){
		long rarefaction_depth = rarefaction_depths[d];
		if(distribution_uniform_tips_wr || distribution_uniform_children_wr) rarefaction_depth = min(Ntips,rarefaction_depth);
		for(long r=0; r<Nrepeats; ++r){
			// re-initialize some counters for this iteration
			trait_included.assign(Ntraits,false);
			tip_included.assign(Ntips,false);
			if(distribution_uniform_tips_wr){
				count_tips_remaining = Ntips;
			}else if(distribution_uniform_children_wr){
				tips_remaining_per_node = node2total_tip_count;
				clade2weight.resize(Nclades);
				for(clade=0; clade<Nclades; ++clade) clade2weight[clade] = (clade<Ntips ? 1.0 : (tips_remaining_per_node[clade-Ntips]>0 ? 1 : 0));
			}
			// choose multiple random tips and count trait richness
			for(long t=0; t<rarefaction_depth; ++t){
				if(distribution_uniform_tips){
					tip = uniformIntWithin(0,Ntips-1);

				}else if(distribution_uniform_tips_wr){
					// choose random tip from remaining tips (i.e. without replacement)
					long tip_counter = uniformIntWithin(0,count_tips_remaining-1);
					for(tip=0; tip<Ntips; ++tip){
						if(!tip_included[tip]) --tip_counter;
						if(tip_counter<0) break;
					}
					--count_tips_remaining;

				}else if(distribution_uniform_children){
					tip = get_tip_by_random_uniform_traversal(	Ntips,
																root,
																node2first_child,
																node2last_child,
																children);
				}else if(distribution_uniform_children_wr){
					tip = get_tip_by_random_traversal(	Ntips,
														root,
														node2first_child,
														node2last_child,
														children,
														clade2weight);
					clade2weight[tip] = 0; // prevent re-inclusion of this tip in the future (i.e. don't replace)
					// propagate information upwards
					clade = tip;
					while(clade!=root){
						clade = clade2parent[clade];
						tips_remaining_per_node[clade-Ntips] -= 1;
						if(tips_remaining_per_node[clade-Ntips]<=0){
							// no more tips to draw from this clade, so set weight to zero
							clade2weight[clade] = 0.0;
						}
					}
					
				}else if(distribution_phylogenetic_diversity){
					tip = get_tip_by_random_traversal(	Ntips,
														root,
														node2first_child,
														node2last_child,
														children,
														clade2weight);
				}
				tip_included[tip] = true;
				for(long i=tip2first_trait[tip]; i<=tip2last_trait[tip]; ++i){
					trait_included[traits[i]] = true;
				}
			}
			
			// count towards statistics of collector's curve
			const double trait_richness			= vector_sum(trait_included);
			const long effective_d				= (use_realized_depths ? get_nearest_index(rarefaction_depths, vector_sum(tip_included)) : d);
			trait_richness_means[effective_d] 	+= trait_richness;
			trait_richness_stds[effective_d]  	+= SQR(trait_richness);
			trait_richness_mins[effective_d] 	= min(trait_richness_mins[effective_d], trait_richness);
			trait_richness_maxs[effective_d] 	= max(trait_richness_maxs[effective_d], trait_richness);
			Nrepeats_per_depth[effective_d]  	+= 1;			
		}
	}

	for(long d=0; d<Ndepths; ++d){
		if(Nrepeats_per_depth[d]==0){
			trait_richness_means[d] = NAN_D;
			trait_richness_stds[d]  = NAN_D;
			trait_richness_mins[d]  = NAN_D;
			trait_richness_maxs[d]  = NAN_D;
		}else{
			trait_richness_means[d] /= Nrepeats_per_depth[d];
			trait_richness_stds[d]  = sqrt(trait_richness_stds[d]/Nrepeats_per_depth[d] - SQR(trait_richness_means[d]));
		}	
	}
	
	return Rcpp::List::create(	Rcpp::Named("trait_richness_means") = Rcpp::wrap(trait_richness_means),
								Rcpp::Named("trait_richness_stds") 	= Rcpp::wrap(trait_richness_stds),
								Rcpp::Named("trait_richness_mins") 	= Rcpp::wrap(trait_richness_mins),
								Rcpp::Named("trait_richness_maxs") 	= Rcpp::wrap(trait_richness_maxs),
								Rcpp::Named("Nrepeats_per_depth") 	= Rcpp::wrap(Nrepeats_per_depth));
}










// Calculate Phylogenetic Independent Contrasts (PIC) for multiple continuous traits on a tree [Felsenstein 1985, page 10]
// One PIC is returned for each non-monofurcating node (or each bifurcating node, if only_bifurcations=true).
// If only_bifurcations==false, then:
//    If multifurcations are present and, these are internally expanded to bifurcations and an additional PIC is returned for each such bifurcation.
//    Hence, the number of returned PICs is the number of bifurcations in the tree, after multifurcations have been expanded to bifurcations.
// Literature:
//    Felsenstein (1985). Phylogenies and the Comparative Method. The American Naturalist. 125:1-15.
// Requirements:
//   Tree can include multi- and mono-furcations.
//   Tree must be rooted. Root will be determined automatically as the node with no parent.
void get_phylogenetic_independent_contrasts(const long			Ntips,
											const long 			Nnodes,
											const long			Nedges,
											const long			Ntraits,
											const std::vector<long> &tree_edge,				// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
											const std::vector<double> &edge_length,			// (INPUT) 1D array of size Nedges, or an empty std::vector (all edges have length 1)
											const std::vector<double>	&tip_states,			// (INPUT) 2D array of size Ntips x Ntraits, in row-major format, listing numeric states for each trait at each tip
											const bool			only_bifurcations,		// (INPUT) if true, PICs are only calculated for bifurcating nodes in the input tree, and multifurcations are not expanded.
											const bool			scaled,					// (INPUT) if true, then PICs are rescaled by the square-root of their expected variances (=the edge lengths connecting the compared nodes/tips)
											std::vector<double>	&PICs,					// (OUTPUT) 2D array of size Npics x Ntraits, in row-major format, listing PICs for each trait and for each considered node
											std::vector<double>	&distances,				// (OUTPUT) 1D array of size Npics, listing phylogenetic distances corresponding to the PICs. Under a Brownian motion mode, these are proportional to the variance of each PIC
											std::vector<long>	&PIC_nodes,				// (OUTPUT) 1D array of size Npics, listing node indices for which PICs were calculated. Negative values indicate nodes that were not actually in the original tree, but were created temporarily during expansion of multifurcations
											std::vector<double>	&root_state,			// (OUTPUT) 1D array of size Ntraits, holding the root's globally reconstructed state (X_k sensu Felsenstein)
											std::vector<double>	&root_standard_error){	// (OUTPUT) 1D array of size Ntraits, listing the standard errors for the root's state (under a Brownian motion model) [Garland et al. (1999). An introduction to phylogenetically based statistical methods, with a new method for confidence intervals on ancestral values]
	// check if tree has monofurcations & multifurcations
	long Nmonofurcations, Nbifurcations, Nmultifurcations;
	count_monofurcations_and_multifurcations(	Ntips,
												Nnodes,
												Nedges,
												tree_edge,
												Nmonofurcations,
												Nbifurcations,
												Nmultifurcations);
	
	std::vector<long> local_tree_edge;
	std::vector<double> local_edge_length;
	long Nlocal_edges, Nlocal_nodes;
	if((!only_bifurcations) && (Nmultifurcations>0)){
		// Tree has multifurcations, so expand them first to bifurcations
		// Note that the number of monofurcations will remain unchanged, but the number of bifurcations/nodes/edges will increase
		std::vector<long> dummy;
		multifurcations_to_bifurcations(Ntips,
										Nnodes,
										Nedges,
										tree_edge,
										edge_length,
										0,
										Nlocal_nodes,
										Nlocal_edges,
										local_tree_edge,
										local_edge_length,
										dummy);
	}else{
		local_tree_edge 	= tree_edge;
		local_edge_length 	= edge_length;
		Nlocal_nodes 		= Nnodes;
		Nlocal_edges 		= Nedges;
	}
	const long Nlocal_clades = Ntips + Nlocal_nodes;
	const long Npics = (only_bifurcations ? Nbifurcations : (Nlocal_nodes - Nmonofurcations));
	
	// get incoming edge for each clade
	std::vector<long> incoming_edge_per_clade;
	get_incoming_edge_per_clade(Ntips, Nlocal_nodes, Nlocal_edges, local_tree_edge, incoming_edge_per_clade);
	
	// get root
	const long local_root = get_root_from_incoming_edge_per_clade(Ntips, local_tree_edge, incoming_edge_per_clade);

	// prepare tree traversal route (root-->tips) and edge mappings
	std::vector<long> traversal_queue, node2first_edge, node2last_edge, edge_mapping;
	get_tree_traversal_root_to_tips(Ntips,
									Nlocal_nodes,
									Nlocal_edges,
									local_root,
									local_tree_edge,
									false,	// don't include tips
									false,	// edge mappings are not pre-calculated
									traversal_queue,
									node2first_edge,
									node2last_edge,	
									edge_mapping,
									false,
									"");
									
	// prepare incoming edge length per clade (will be modified later on as part of Felsenstein's algorithm)
	std::vector<double> incoming_length_per_clade(Nlocal_clades);
	if(local_edge_length.size()>0){
		for(long clade=0; clade<Nlocal_clades; ++clade){
			if(clade!=local_root) incoming_length_per_clade[clade] = local_edge_length[incoming_edge_per_clade[clade]];
		}
	}else{
		incoming_length_per_clade.assign(incoming_length_per_clade.size(),1);
		incoming_length_per_clade[local_root] = 0;
	}
	
									
	// calculate Felsenstein's PICs in a postorder traversal (tips-->root)
	const double edge_length_epsilon = RELATIVE_EPSILON * array_nonzero_min(local_edge_length); // substitute to be used for zero edge lengths
	std::vector<double> node_states(Nlocal_nodes*Ntraits,0);	// 2D numeric array of size Nlocal_nodes x Ntraits
	PICs.clear(); PICs.reserve(Npics*Ntraits);
	distances.clear(); distances.reserve(Npics);
	PIC_nodes.clear(); PIC_nodes.reserve(Npics);
	long edge1, edge2, child, child1, child2, node, clade, trait;
	double length, total_weight, X1, X2;
	for(long q=traversal_queue.size()-1; q>=0; --q){
		clade	= traversal_queue[q];
		node	= clade - Ntips;
		// calculate Felsenstein's X_k (node_states) and nu_k (incoming_length_per_clade)
		total_weight = 0;
		for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
			child 	= local_tree_edge[2*edge_mapping[e]+1];
			length 	= incoming_length_per_clade[child];
			if(length==0) length = edge_length_epsilon;
			for(trait=0; trait<Ntraits; ++trait){
				node_states[node*Ntraits+trait] += (1.0/length) * (child<Ntips ? tip_states[child*Ntraits+trait] : node_states[(child-Ntips)*Ntraits+trait]);
			}
			total_weight += (1.0/length);
		}
		for(trait=0; trait<Ntraits; ++trait) node_states[node*Ntraits+trait] /= total_weight;
		incoming_length_per_clade[clade] += 1.0/total_weight;
		
		// calculate PICs using Felsenstein's X_i & nu_i (skip over monofurcating nodes)
		// note that monofurcating nodes acquire the same state as their child, and their modified incoming_length is the same as their child plus the length of their incoming edge
		if(1+node2last_edge[node]-node2first_edge[node] != 2) continue; // node is not bifurcating
		edge1		= edge_mapping[node2first_edge[node]];
		edge2		= edge_mapping[node2first_edge[node]+1];
		child1 		= local_tree_edge[2*edge1+1];
		child2 		= local_tree_edge[2*edge2+1];
		for(trait=0; trait<Ntraits; ++trait){
			X1 = (child1<Ntips ? tip_states[child1*Ntraits+trait] : node_states[(child1-Ntips)*Ntraits+trait]);
			X2 = (child2<Ntips ? tip_states[child2*Ntraits+trait] : node_states[(child2-Ntips)*Ntraits+trait]);
			PICs.push_back(X2 - X1);
		}
		distances.push_back(incoming_length_per_clade[child1] + incoming_length_per_clade[child2]);
		PIC_nodes.push_back(node<Nnodes ? node : -1); // keep track which node this PIC corresponds to. -1 means this temporary node did not exist in the original tree
	}
	
	// extract estimated root state & calculate standard error
	// this should come before the scaling further below
	// Standard error formula according to: [Garland et al. (1999). Page 377]
	root_state.resize(Ntraits);
	root_standard_error.assign(Ntraits,0);
	for(trait=0; trait<Ntraits; ++trait){
		root_state[trait] = node_states[(local_root - Ntips)*Ntraits+trait];
		for(long p=0; p<Npics; ++p){
			root_standard_error[trait] += SQ(PICs[p*Ntraits+trait])/distances[p];
		}
		root_standard_error[trait] *= incoming_length_per_clade[local_root] / Npics;
		root_standard_error[trait] = sqrt(root_standard_error[trait]);
	}
	
	// rescale returned PICs if needed
	if(scaled){
		for(long p=0; p<Npics; ++p){
			for(trait=0; trait<Ntraits; ++trait){
				PICs[p*Ntraits+trait] /= sqrt(distances[p]);
			}
		}
	}
}




// Calculate Phylogenetic Independent Contrasts (PIC) for multiple continuous traits on a tree [Felsenstein 1985, page 10]
// This is an Rcpp wrapper for the function get_phylogenetic_independent_contrasts(..)
// [[Rcpp::export]]
Rcpp::List get_phylogenetic_independent_contrasts_CPP(	const long			Ntips,
														const long 			Nnodes,
														const long			Nedges,
														const long			Ntraits,
														const std::vector<long> &tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
														const std::vector<double> &edge_length,		// (INPUT) 1D array of size Nedges, or an empty std::vector (all edges have length 1)
														const std::vector<double>	&tip_states,		// (INPUT) 2D array of size Ntips x Ntraits, in row-major format, listing numeric states for each trait at each tip
														const bool			only_bifurcations,	// (INPUT) if true, PICs are only calculated for bifurcating nodes in the input tree, and multifurcations are not expanded.
														const bool			scaled){			// (INPUT)if true, then PICs are rescaled by the square-root of their expected variances (=the edge lengths connecting the compared nodes/tips)
		std::vector<double> PICs, distances, root_state, root_standard_error;
		std::vector<long> PIC_nodes;
		get_phylogenetic_independent_contrasts(	Ntips,
												Nnodes,
												Nedges,
												Ntraits,
												tree_edge,
												edge_length,
												tip_states,
												only_bifurcations,
												scaled,
												PICs,
												distances,
												PIC_nodes,
												root_state,
												root_standard_error);
		return Rcpp::List::create(	Rcpp::Named("PICs")  				= Rcpp::wrap(PICs),
									Rcpp::Named("distances") 			= Rcpp::wrap(distances),
									Rcpp::Named("nodes") 				= Rcpp::wrap(PIC_nodes),
									Rcpp::Named("root_state") 			= Rcpp::wrap(root_state),
									Rcpp::Named("root_standard_error") 	= Rcpp::wrap(root_standard_error));
}





// Fit a multivariate Brownian motion model for multiple correlated continuous traits evolving under Brownian motion
// Estimates the diffusivity matrix D[i,j], so that exp(-X^T*D^{-1}*X/(4*L))/sqrt(det(2*pi*D)) is the probability density for the multidimensional trait vector X after phylogenetic distance L, if initially located at the origin.
void fit_Brownian_motion_model(	const long			Ntips,
								const long 			Nnodes,
								const long			Nedges,
								const long			Ntraits,
								const std::vector<long> &tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
								const std::vector<double> &edge_length,		// (INPUT) 1D array of size Nedges, or an empty std::vector (all edges have length 1)
								const std::vector<double>	&tip_states,		// (INPUT) 2D array of size Ntips x Ntraits, in row-major format, listing numeric states for each trait at each tip
								std::vector<double>	&diffusivity){		// (OUTPUT) 2D array of size Ntraits x Ntraits, in row-major format, listing the fitted diffusion matrix D
											
	// calculate phylogenetic independent contrasts (PIC)
	// PICs correspond to independent increments of a multivariate Brownian motion process
	std::vector<double> PICs, distances, root_state, root_standard_error;
	std::vector<long> PIC_nodes;
	get_phylogenetic_independent_contrasts(	Ntips,
											Nnodes,
											Nedges,
											Ntraits,
											tree_edge,
											edge_length,
											tip_states,
											false,
											true,	// rescale PICs by phylogenetic distances
											PICs,
											distances,
											PIC_nodes,
											root_state,
											root_standard_error);
											
	// estimate diffusivity matrix based on independent contrasts
	// maximum-likelihood estimator on the intrinsic geometry of positive-definite matrices
	const long Npics = distances.size();
	diffusivity.assign(Ntraits*Ntraits,0); // 2D matrix of size Ntraits x Ntraits, in row-major format
	for(long t1=0; t1<Ntraits; ++t1){
		for(long t2=t1; t2<Ntraits; ++t2){
			for(long p=0; p<Npics; ++p){
				diffusivity[t1*Ntraits+t2] += PICs[p*Ntraits+t1]*PICs[p*Ntraits+t2];
			}
			diffusivity[t1*Ntraits+t2] /= (2*Npics);
			diffusivity[t2*Ntraits+t1]  = diffusivity[t1*Ntraits+t2];
		}
	}
}




// Calculate mean & standard deviation of a numeric trait across all extant clades over time
// This function requires that the trait is known for all tips and nodes of the tree
// [[Rcpp::export]]
Rcpp::List get_trait_stats_at_times_CPP(const long					Ntips,
										const long 					Nnodes,
										const long					Nedges,
										std::vector<long> 			tree_edge,			// (INPUT) 2D array (in row-major format) of size Nedges x 2
										const std::vector<double>	&edge_length,		// (INPUT) 1D array of size Nedges, or empty
										const std::vector<double>	&times,				// (INPUT) 1D array of size Ntimes
										const std::vector<double>	&states,			// (INPUT) 1D array of size Nclades, listing the trait's value on each tip & node
										const bool					return_states){		// (INPUT) if true, then this function also returns a matrix of size Ntimes x Nclades (in row-major format) listing the states encountered at each time point and for each clade relevant at that time point. This matrix will contain NAs, for any (t,c) combination where clade c's incoming edge did not cross time t.
	const long Nclades = Ntips + Nnodes;

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);	
	
	// calculate distances from root
	std::vector<double> distances_from_root(Nclades);
	get_distances_from_root(Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							edge_length,
							distances_from_root);
	
	const long Ntimes = times.size();
	dvector state_means(Ntimes,0), state_stds(Ntimes,0);
	lvector clade_counts(Ntimes,0); // number of clades over which the trait statistics were calculated, at each time point
	dvector states_matrix;
	if(return_states) states_matrix.assign(Ntimes*Nclades, NAN_D);
	for(long t=0; t<Ntimes; ++t){
		for(long clade=0; clade<Nclades; ++clade){
			if(clade==root) continue;
			if((distances_from_root[clade]>=times[t]) && (distances_from_root[clade2parent[clade]]<=times[t]) && (!std::isnan(states[clade]))){
				if(return_states) states_matrix[Nclades*t+clade] = states[clade];
				clade_counts[t] += 1;
				state_means[t]  += states[clade];
				state_stds[t]   += SQ(states[clade]);
			}
		}
		state_means[t] /= clade_counts[t];
		state_stds[t] = sqrt(state_stds[t]/clade_counts[t] - SQ(state_means[t]));
	}
	return Rcpp::List::create(	Rcpp::Named("clade_counts")		= Rcpp::wrap(clade_counts),
								Rcpp::Named("means")			= Rcpp::wrap(state_means),
								Rcpp::Named("stds")				= Rcpp::wrap(state_stds),
								Rcpp::Named("states_matrix")	= Rcpp::wrap(states_matrix));
}



#pragma mark -
#pragma mark Most recent common ancestors
#pragma mark 



// for each clade in a list of clades (nodes or tips), find a minimal set of tips so that the clade is the most recent common ancestor of those tips.
// This routine has an overhead at the beginning, but becomes more efficient when the number of given MRCAs is high.
// Time complexity = O(Ntips+Nnodes) + O(mrcas)
// Requirements:
//   The input tree must be rooted
//   The input tree can be multifurcating and/or monofurcating (except for the given MRCAs, each of which must have at least 2 children)
// Returns:
//   mrca2first_tip[]: 1D array of size Nmrcas, with values being indices in mrca_tips[]
//   mrca2last_tip[]: 1D array of size Nmrcas, with values being indices in mrca_tips[]
//   mrca_tips[]: 1D array with values in 0:(Ntips-1), so that mrca_tips[mrca2first_tip[m]],..,mrca_tips[mrca2last_tip[m]] are tips whose MRCA is clade mrcas[m]

// [[Rcpp::export]]
Rcpp::List get_mrca_defining_tips_CPP(	const long 			Ntips,
										const long 			Nnodes,
										const long 			Nedges,
										const std::vector<long> &tree_edge,		// (INPUT) 2D array of size Nedges x 2 in row-major format, with values in 0:(Nclades-1)
										const std::vector<long>	&mrcas,			// (INPUT) 1D array of size Nmrcas, with values in 0,..,(Nclades-1). For each mrcas[i], a set of tips is returned such that mrcas[i] is the MRCA of those tips
										bool				verbose,
										const std::string	&verbose_prefix){
	/* indexing conventions:
		parent, child: always run within 0,..,Nclades-1
		node: always runs within 0,..,Nnodes-1
		tip: always runs within 0,..,Ntips
	*/
	const long Nmrcas = mrcas.size();
	long clade, node, child;

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (tips --> root)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);
	reverse_array(traversal_queue); // make tips-->roots
										
	// to each node, assign one tip descending from it (traverse tips --> root)
	std::vector<long> node2tip(Nnodes, -1);
	for(long q=0, parent; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		if(clade<Ntips){
			// clade is a tip
			node2tip[parent-Ntips] = clade;
		}else{
			// clade is a node, so propagate its tip upwards
			node2tip[parent-Ntips] = node2tip[clade-Ntips];
		}
	}
	
	// for each MRCA, collect one descending tip per child
	std::vector<long> mrca2first_tip(Nmrcas);
	std::vector<long> mrca2last_tip(Nmrcas);
	std::vector<long> mrca_tips;
	mrca_tips.reserve(2*Nmrcas); // will need 2 tips per mrca
	for(long m=0, mrca; m<Nmrcas; ++m){
		mrca = mrcas[m];
		if(mrca<Ntips){
			// mrca is a tip
			mrca2first_tip[m] = mrca_tips.size();
			mrca2last_tip[m]  = mrca_tips.size();
			mrca_tips.push_back(mrca);
		}else{
			// mrca is a node, so collect one tip per child (for up to 2 children, no more needed)
			node = mrca-Ntips;
			mrca2first_tip[m] = mrca_tips.size();
			for(long ei=traversal_node2first_edge[node]; ei<=min(traversal_node2last_edge[node],traversal_node2first_edge[node]+1); ++ei){
				child = tree_edge[traversal_edges[ei]*2+1];
				if(child<Ntips) mrca_tips.push_back(child);
				else mrca_tips.push_back(node2tip[child-Ntips]);
			}
			mrca2last_tip[m] = mrca_tips.size()-1;
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("mrca2first_tip") 	= Rcpp::wrap(mrca2first_tip),
								Rcpp::Named("mrca2last_tip") 	= Rcpp::wrap(mrca2last_tip),
								Rcpp::Named("mrca_tips")		= Rcpp::wrap(mrca_tips));
}



// determine pairwise ancestry relationships between a subset of focal clades
// i.e. determine all pairs c,a for which focal_clades[a] is an ancestor of focal_clades[c]
// Requirements:
//   The input tree must be rooted
//   The input tree can be multifurcating and/or monofurcating
// Returns a 2D array in row-major format, listing pairs (c,a)

// [[Rcpp::export]]
IntegerVector get_pairwise_ancestries_CPP(	const long 				Ntips,
											const long 				Nnodes,
											const long 				Nedges,
											const long				root,				// (INPUT) integer within Ntips:(Ntips+Nnodes-1)
											const std::vector<long> &tree_edge,			// (INPUT) 2D array of size Nedges x 2 in row-major format, with values in 0:(Nclades-1)
											const std::vector<long>	&focal_clades){		// (INPUT) 1D array of size Nfocals, with values in 0,..,(Nclades-1)

	const long Nclades = Ntips + Nnodes;
	long ancestor, clade;

	// determine parent clade for each clade
	std::vector<long> clade2parent(Ntips+Nnodes, -1);
	for(long edge=0; edge<Nedges; ++edge){
		clade2parent[tree_edge[edge*2+1]] = tree_edge[edge*2+0];
	}
	
	// create mapping all_clades --> focal_clades (use -1 if not mapped)
	std::vector<long> all2focal_clade_index(Nclades,-1);
	for(long fc=0; fc<focal_clades.size(); ++fc){
		all2focal_clade_index[focal_clades[fc]] = fc;
	}

	// traverse upwards from each of the focal clades and keep track of encountered focal ancestors
	std::vector<long> descendant_ancestor_pairs; // list of descendant-ancestor pairs found, in row-major format. I.e. descendant_ancestor_pairs[2*p+0] is the descendant of descendant_ancestor_pairs[2*p+1].
	for(long fc=0; fc<focal_clades.size(); ++fc){
		clade = focal_clades[fc];
		if(clade==root) continue;
		ancestor = clade;
		do{
			ancestor = clade2parent[ancestor];
			if(all2focal_clade_index[ancestor]>=0){
				descendant_ancestor_pairs.push_back(fc);
				descendant_ancestor_pairs.push_back(all2focal_clade_index[ancestor]);
			}
		}while(ancestor!=root);
	}
	
	return  Rcpp::wrap(descendant_ancestor_pairs);
}


// OLD CODE. WORKS FINE, BUT LESS CPU-EFFICIENT THAN ALTERNATIVE BELOW
// Calculate most-recent-common-ancestors (MRCA) for pairs of clades (cladesA[] vs cladesB[])
// IntegerVector get_MRCAs_CPP(const long 				Ntips,
// 							const long 				Nnodes,
// 							const long 				Nedges,
// 							const std::vector<long>	&tree_edge,			// (INPUT) 2D array of size Nedges x 2, flattened in row-major format
// 							const std::vector<long>	&cladesA,			// (INPUT) 1D array of size NP, with values in 0,..,Nclades-1, specifying the first clade of each clade pair
// 							const std::vector<long>	&cladesB){			// (INPUT) 1D array of size NP, with values in 0,..,Nclades-1, specifying the second clade of each clade pair
// 	const long Nclades = Ntips + Nnodes;
// 	const long Npairs  = cladesA.size();
// 	
// 	// determine parent clade for each clade
// 	std::vector<long> clade2parent;
// 	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
// 
// 	// determine number of transitions from each clade all the way back to the root
// 	lvector MRCAs(Npairs);
// 	std::vector<bool> encountered(Nclades,false);
// 	for(long p=0, cladeA, cladeB; p<Npairs; ++p){
// 		cladeA = cladesA[p];
// 		cladeB = cladesB[p];
// 		if(cladeA==cladeB){
// 			MRCAs[p] = cladeA;
// 			continue;
// 		}
// 		// follow cladeA all the way to the root
// 		cladeA = cladesA[p];
// 		while(cladeA>=0){
// 			encountered[cladeA] = true;
// 			cladeA = clade2parent[cladeA];
// 		}
// 		// follow cladeB all the way to the root, until meeting the rootward path taken from cladeA
// 		cladeB = cladesB[p];
// 		while(cladeB>=0){
// 			if(encountered[cladeB]){
// 				// this parent was already encountered on the path from cladeA to the root
// 				MRCAs[p] = cladeB;
// 				break;
// 			}else{
// 				cladeB = clade2parent[cladeB];
// 			}
// 		}
// 		// reset the entries in encountered[]
// 		cladeA = cladesA[p];
// 		while(cladeA>=0){
// 			encountered[cladeA] = false;
// 			cladeA = clade2parent[cladeA];
// 		}
// 		if((p%10)==0) Rcpp::checkUserInterrupt();
// 	}
// 	return Rcpp::wrap(MRCAs);
// }




// Calculate most-recent-common-ancestors (MRCA) for pairs of clades (cladesA[] vs cladesB[])
// there's some initial overhead involved with this function, but for large number of clade pairs this becomes more efficient
// Time complexity is O(Ntips+Nnodes).
// Returns an IntegerVector of size Npairs, with each entry being the clade index of the MRCA of the pair
// If one clade is descendant of the other clade, the latter will be returned as MRCA
// Requirements:
//   The input tree must be rooted (root will be determined automatically, as the node that has no incoming edge)
//   The input tree can be multifurcating and/or monofurcating
// Attention: 0-based indexing is used for input and output variables, so make sure to shift indices in R before and after calling this function
// [[Rcpp::export]]
IntegerVector get_most_recent_common_ancestors_CPP(	const long 			Ntips,
													const long 			Nnodes,
													const long 			Nedges,
													const std::vector<long> &tree_edge,			// 2D array of size Nedges x 2 in row-major format
													const std::vector<long> &cladesA,			// 1D array of size Npairs, containing values in 0:(Nclades-1)
													const std::vector<long>	&cladesB,			// 1D array of size Npairs, containing values in 0:(Nclades-1)
													bool 				verbose,
													const std::string 	&verbose_prefix){
	const long Npairs = cladesA.size();
	const long Nclades = Ntips + Nnodes;
	long parent, clade;

	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);
	
	// determine incoming edge per clade
	std::vector<long> incoming_edge_per_clade(Nclades,-1);
	for(long edge=0; edge<Nedges; ++edge){
		incoming_edge_per_clade[tree_edge[edge*2+1]] = edge;
	}
	
	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);
	
	// get tree traversal route (root --> tips)											
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										true,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);
															
	// calculate number of ancestors and distance from root for each clade
	// (traverse root --> tips)
	std::vector<long> ancestor_count_per_clade(Nclades);
	ancestor_count_per_clade[root] = 0;
	for(long q=0; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		ancestor_count_per_clade[clade] = 1 + ancestor_count_per_clade[parent];
	}
	const long total_ancestor_count = vector_sum(ancestor_count_per_clade);
	
	// calculate ancestry for each clade in a long list ancestors[]
	// (traverse root --> tips)
	std::vector<long> clade2first_ancestor(Nclades); // for each clade c, ancestors[clade2first_ancestor[c]..clade2last_ancestor[c]] will be the list of ancestor clades leading to the clade c
	std::vector<long> clade2last_ancestor(Nclades);
	clade2first_ancestor[0] = 0;
	clade2last_ancestor[0] = clade2first_ancestor[0] + ancestor_count_per_clade[0] - 1;
	for(long c=1; c<Nclades; ++c){
		clade2first_ancestor[c] = clade2last_ancestor[c-1] + 1;
		clade2last_ancestor[c]  = clade2first_ancestor[c] + ancestor_count_per_clade[c] - 1;
	}
	std::vector<long> ancestors(total_ancestor_count);
	for(long q=0; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		if(clade==root) continue;
		parent = clade2parent[clade];
		// step 1: copy the parent's ancestry to the child's ancestry
		for(long a=clade2first_ancestor[parent]; a<=clade2last_ancestor[parent]; ++a){
			ancestors[clade2first_ancestor[clade]+(a-clade2first_ancestor[parent])] = ancestors[a];
		}
		// step 2: append the parent to the clade's ancestry
		ancestors[clade2last_ancestor[clade]] = parent;
	}
	
	// calculate most-recent-common-ancestor for each clade pair
	lvector mrca_per_pair(Npairs);
	long cladeA, cladeB;
	for(long p=0; p<Npairs; ++p){
		cladeA = cladesA[p];
		cladeB = cladesB[p];
		// check for trivial case
		if(cladeA==cladeB){
			mrca_per_pair[p] = cladeA;
			continue;
		}
		// follow ancestry of both clades in synchrony, until they diverge
		// note that the first ancestor of every clade will be the root
		long a,b, mrca=-1;
		for(a=clade2first_ancestor[cladeA], b=clade2first_ancestor[cladeB]; (a<=clade2last_ancestor[cladeA]) && (b<=clade2last_ancestor[cladeB]); ++a, ++b){
			if(ancestors[a]!=ancestors[b]) break;
			else mrca = ancestors[a];
		}
		// check special case where one clade is descendant of the other (this would lead to a "premature" stop of the above loop)
		if((a<=clade2last_ancestor[cladeA]) && (ancestors[a]==cladeB)){
			mrca = cladeB;
		}else if((b<=clade2last_ancestor[cladeB]) && (ancestors[b]==cladeA)){
			mrca = cladeA;
		}
		mrca_per_pair[p] = mrca;
	}
	return(Rcpp::wrap(mrca_per_pair));
}




// Given a set of clades (tips & nodes, "descendants"), find their most recent common ancestor
// [[Rcpp::export]]
long get_most_recent_common_ancestor_CPP(	const long 			Ntips,
											const long 			Nnodes,
											const long 			Nedges,
											const std::vector<long> &tree_edge,			// 2D array of size Nedges x 2 in row-major format
											const std::vector<long> &descendants){		// 1D array of size ND, containing values in 0:(Nclades-1)
	const long Nclades = Ntips + Nnodes;
	const long ND = descendants.size();
	if(ND==0) return 0;
	if(ND==1) return descendants[0];
	
	// determine parent clade for each clade
	std::vector<long> clade2parent;
	get_parent_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2parent);

	// find root using the mapping clade2parent
	const long root = get_root_from_clade2parent(Ntips, clade2parent);

	// traverse upwards from each descendant and count the number of visits to each clade in the tree
	long clade;
	std::vector<long> visits_per_clade(Nclades, 0);
	for(long d=0; d<ND; ++d){
		clade = descendants[d];
		++visits_per_clade[clade];
		while(clade2parent[clade]>=0){
			clade = clade2parent[clade];
			++visits_per_clade[clade];
		}
	}
	
	// traverse from one of the descendants back to the root once more, to find the first clade from which all descendants passed
	clade = descendants[0];
	while(clade2parent[clade]>=0){
		if(visits_per_clade[clade]==ND) return clade; // all descendants passed by this clade, so it's the MRCA
		clade = clade2parent[clade];
	}
	return root;
}



// Given a rooted tree, determine whether a set of tips is monophyletic
// Requirements:
//   The tree must be rooted
//   The tree can include monofurcations and multifurcations
// [[Rcpp::export]]
bool is_monophyletic_tip_set_CPP(	const long 			Ntips,
									const long 			Nnodes,
									const long 			Nedges,
									const std::vector<long> &tree_edge,			// (INPUT) 2D array of size Nedges x 2 in row-major format
									const std::vector<long> &focal_tips){		// (INPUT) 1D array of size Nfocals, listing focal tip indices to test for monophyly
	if(focal_tips.size()<=1) return true;
	// find MRCA of focal tips
	const long mrca = get_most_recent_common_ancestor_CPP(	Ntips,
															Nnodes,
															Nedges,
															tree_edge,
															focal_tips);
	if(mrca<Ntips) return true; // MRCA is a tip, so must be monophyletic
	
	// get edge mappings for tree traversal
	std::vector<long> node2first_edge, node2last_edge, edges;
	get_node2edge_mappings(	Ntips,
							Nnodes,
							Nedges,
							tree_edge,
							node2first_edge,
							node2last_edge,
							edges);

	// traverse from MRCA to tips, to check if any of the reached tips are not in the focal set
	// use a scratch_stack for depth-first-search traversal
	std::vector<bool> tip_is_focal(Ntips, false);
	for(long t=0; t<focal_tips.size(); ++t) tip_is_focal[focal_tips[t]] = true;
	long child,node;
	std::vector<long> scratch_stack;
	scratch_stack.reserve(floor(2*log(Ntips)/log(2.0))); // rough estimate of typical tree depth x 2. scratch_stack may be resized along the way if needed.
	scratch_stack.push_back(mrca);
	while(scratch_stack.size()>0){
		node = scratch_stack.back() - Ntips;
		scratch_stack.pop_back();
		for(long e=node2first_edge[node]; e<=node2last_edge[node]; ++e){
			child = tree_edge[edges[e]*2+1];
			if((child<Ntips) && (!tip_is_focal[child])) return false; // reached a non-focal child
			if(child>=Ntips) scratch_stack.push_back(child); // add child node to stack for further exploration in the next iteration
		}
	}

	return true;
}





#pragma mark -
#pragma mark Ancestral state reconstruction
#pragma mark





// calculate the best (lowest) cost of any transition parent-->child, given a particular parent state and a particular child cost table (extracted from master_cost_table)
// this function assumes that the cost table for the child has already been calculated (hence, you should move tips-->root)
double aux_get_cost_of_parent_state_transitioning_to_one_child(	const long					Nstates,
																const long					parent_state, 
																const long 					edge,
																const double				edge_weight,
																const long 					child,
																const std::vector<double>			transition_costs, 		// (INPUT) 2D array of size Nstates x Nstates (in row-major format)
																const std::vector<double>	&master_cost_table,		// (INPUT) 2D array of size (Ntips+Nnodes) x Nstates (in row-major format)
																std::vector<double>			&scratch_space,			// temporary space for intermediate operations
																std::vector<long>			&master_transitions,				// (INPUT/OUTPUT) 1D array (preferably reserved up to size Nnodes*Nstates*Nstates)
																std::vector<long>			&edge_and_state2first_transition,	// (INPUT/OUTPUT) 1D array of size Nedges*Nstates.
																std::vector<long>			&edge_and_state2last_transition){	// (INPUT/OUTPUT) 1D array of size Nedges*Nstates.
	std::vector<double> &choice_costs = scratch_space;
	choice_costs.resize(Nstates);
	for(long state=0; state<Nstates; ++state){
		choice_costs[state] = transition_costs[parent_state*Nstates + state]*edge_weight + master_cost_table[child*Nstates + state];
	}
	const double best_cost = array_min(choice_costs);
	edge_and_state2first_transition[edge*Nstates + parent_state] = master_transitions.size();
	for(long transition=0; transition<Nstates; ++transition){
		if(abs(choice_costs[transition]-best_cost)<=RELATIVE_EPSILON*best_cost){
			master_transitions.push_back(transition);
		}
	}
	edge_and_state2last_transition[edge*Nstates + parent_state]	= master_transitions.size()-1;	
	return best_cost;
}




// calculate the cost of a particular state in a particular node, best on the best (lowest) costs of transitions to each of the children
// this function assumes that the cost table of each child has already been calculated (hence, you should move tips-->root)
double aux_get_cost_of_parent_state_transitioning_to_all_children(	const long					Nstates,
																	const long 					node, 							// (INPUT) integer in 0:(Nnodes-1)
																	const long 					parent_state,					// (INPUT) integer in 0:(Nstates-1)
																	const double				branch_length_exponent,			// (INPUT) non-negative number
																	const std::vector<double>			transition_costs,	 			// (INPUT) 2D array of size Nstates x Nstates (in row-major format)
																	const std::vector<double>	&master_cost_table,				// (INPUT) 2D array of size (Ntips+Nnodes) x Nstates (in row-major format)
																	const std::vector<long>			&tree_edge,						// (INPUT) 2D array of size Nedges x 2 (in row-major format), in similar format as tree$edge in R "phylo" trees.
																	const std::vector<double>			&edge_length,				// (INPUT) 1D array of size Nedges
																	const std::vector<long>		&traversal_node2first_edge,		// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
																	const std::vector<long>		&traversal_node2last_edge,		// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
																	const std::vector<long>		&traversal_edges,				// (INPUT) 1D array of size Nedges, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
																	std::vector<double>			&scratch_space,						// temporary space for intermediate operations
																	std::vector<long>			&master_transitions,				// (INPUT/OUTPUT) 1D array (preferably reserved up to size Nnodes*Nstates*Nstates)
																	std::vector<long>			&edge_and_state2first_transition,	// (INPUT/OUTPUT) 1D array of size Nedges*Nstates.
																	std::vector<long>			&edge_and_state2last_transition){	// (INPUT/OUTPUT) 1D array of size Nedges*Nstates.
	double S = 0;
	double edge_weight;
	long edge, child;
	for(long ei=traversal_node2first_edge[node]; ei<=traversal_node2last_edge[node]; ++ei){
		edge  = traversal_edges[ei];
		child = tree_edge[edge*2+1];
		edge_weight = (branch_length_exponent==0 ? 1.0 : 1/pow((edge_length.size()==0 ? 1.0 : edge_length[edge]),branch_length_exponent));
		S += aux_get_cost_of_parent_state_transitioning_to_one_child(	Nstates,
																		parent_state, 
																		edge, 
																		edge_weight, 
																		child,
																		transition_costs,
																		master_cost_table,
																		scratch_space,
																		master_transitions,
																		edge_and_state2first_transition,
																		edge_and_state2last_transition);
	}
	return S;
}
	
	

// Weighted maximum parsimony ansestral state reconstruction for discrete traits.
// Modification of Sankoff algorithm for reconstructing discrete ancestral states (Weighted Small Parsimony Problem)
// Sankoff's algorithm allows the inclusion of a cost matrix: transition_costs[i,j] is the cost of transitioning i-->j (ignoring edge length)
// The modification of this function is that optionally, edge lengths can be used to weight the transition costs:
// 	Longer edges imply smaller transition costs between states
// 	Specifically, the cost of transitioning is transition_cost[i,j]/(edge_length^branch_length_exponent)
//	where branch_length_exponent can be e.g. 0 (Sankoff's original algorithm), 1 (linear weighting) or 0.5 (square-root weighting, corresponding to a Brownian motion)
// Requirements:
//	Tree can be multifurcating, and can also include nodes with a single child
//	If (branch_length_exponent!=0) then: All branches must have length > 0
// For a description of the original Sankoff algorithm, see: 
//	http://telliott99.blogspot.ca/2010/03/fitch-and-sankoff-algorithms-for.html
//	(page 11) https://cs.brown.edu/courses/csci1950-z/slides/CSCI1950ZFall09_Lecture2.pdf
// The function returns a (non-flattened) NumericMatrix of size Nnodes x Nstates.
//
// Attention: Be carefull to use the C++ style indexing (0-based) when passing index-variables or index arrays to this function.
// For example, root must be a 0-based index, and tree_edge[] must have values in 0:(Ntips+Nnodes-1) instead of 1:(Ntips+Nnodes)
// [[Rcpp::export]]
Rcpp::List WMPR_ASR_CPP(const long					Ntips,
						const long 					Nnodes,
						const long					Nedges,
						const long					Nstates, 				// (INPUT) number of possible states
						const std::vector<long>		&tree_edge, 			// (INPUT) 2D array of size Nedges x 2 (in row-major format), in similar format as tree$edge in R "phylo" trees. This array holds the topology of the tree (apart from branch lengths).
						const std::vector<double>	&edge_length,			// (INPUT) 1D array of size Nedges, synchronized with the rows of tree_edge[,], i.e. with edge_length[e] being the length of edge e. Can also be an empty vector (all edges have length 1.0).
						const std::vector<long>		&tip_states, 			// (INPUT) 1D array of size Ntips, with values being in 0:(Nstates-1)
						const std::vector<double>	&transition_costs,	 	// (INPUT) 2D array of size Nstates x Nstates (in row-major format), with transition_costs[i,j] being the cost of transition i-->j. Normally transition_cost[i,i]=0 for all i. Some transitions may be vorbitten, in which case the transition cost should be set to infinity (INFTY_D).
						const double 				branch_length_exponent, // (INPUT) exponent for weighting transition costs by branch length. To ignore branch lengths (i.e. to obtain the non-weighted MPR algorithm), set this to 0.
						bool						weight_posteriors_by_scenario_counts,	// (INPUT) if true, then the posterior_probability of a state (in a specific node) is proportional to the number of scenarios in which the trait is at that state
						bool						verbose,
						const std::string			&verbose_prefix){
	// Terminology in this function:
	// 	'node' runs from 0:(Nnodes-1)
	// 	'tip' runs from 0:(Ntips-1)
	// 	'parent' and 'child' runs from 0:(Ntips+Nnodes-1)
	// 	'edge' runs from 0:(Nedges-1)
	// 	'state' runs from 0:(Nstates-1)
	const long Nclades = Ntips+Nnodes;
	long node, state, parent, transition, child, edge;
	std::vector<double> scratch_space;

	// determine root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);
		
	// create tree-access structures and determine order in which to traverse tree
	std::vector<long> traversal_queue_root2tips, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										false,
										false,
										traversal_queue_root2tips,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										verbose,
										verbose_prefix);
	
	// get traversal route tips --> root
	std::vector<long> traversal_queue_tips2root = traversal_queue_root2tips;
	reverse_array(traversal_queue_tips2root); 
	
	
	// master_cost_table[,] should be a 2D numeric array of size (Ntips+Nnodes) x Nstates (in row-major-format)
	// the row master_cost_table[r,] is the cost table of tip or node r
	std::vector<double> master_cost_table(Nclades*Nstates, 0.0);
	
	// fill costs for tips (this is easy, since states are known)
	for(long tip=0; tip<Ntips; ++tip){
		for(state=0; state<Nstates; ++state){
			master_cost_table[tip*Nstates + state] = INFTY_D;
		}
		master_cost_table[tip*Nstates + tip_states[tip]] = 0.0;
	}

	// edge_and_state2first_transition[,] and edge_and_state2last_transition[,] contain indices mapping to master_transitions[]
	// for any edge e connecting parent node n to some child, and assuming n is at state s, the integers
	//	master_transitions[edge_and_state2first_transition[e,s]:edge_and_state2last_transition[e,s]]
	// are within in 1:Nstates, and are the optimal states to which node n switched during edge e.
	// Typically there will be only one "best transition" (i.e. edge_and_state2first_transition[e,s]==edge_and_state2last_transition[e,s]), 
	// but in case of multiple MPR solutions some edges may have multiple alternative best transitions (i.e. edge_and_state2first_transition[e,s] > edge_and_state2last_transition[e,s]

	// pre-allocate space at upper bound of possible need
	std::vector<long> master_transitions;
	master_transitions.reserve(Nnodes*Nstates*Nstates);
	std::vector<long> edge_and_state2first_transition(Nedges*Nstates);
	std::vector<long> edge_and_state2last_transition(Nedges*Nstates);
	
	
	// traverse tree (tips-->root) and build master_cost_table
	for(long parent_i=0; parent_i<Nnodes; ++parent_i){
		parent = traversal_queue_tips2root[parent_i];
		// calculate the cost associated with any state in this particular node
		for(state=0; state<Nstates; ++state){
			master_cost_table[parent*Nstates + state] = aux_get_cost_of_parent_state_transitioning_to_all_children(	Nstates,
																													(parent - Ntips),
																													state,
																													branch_length_exponent,
																													transition_costs,
																													master_cost_table,
																													tree_edge,
																													edge_length,
																													traversal_node2first_edge,
																													traversal_node2last_edge,
																													traversal_edges,
																													scratch_space,
																													master_transitions,
																													edge_and_state2first_transition,
																													edge_and_state2last_transition);
		}
		if(parent_i % 100 == 0) Rcpp::checkUserInterrupt();
	}
	
		
	// count number of scenarios (MPR solutions) implying each state in each node (based on lowest cost in the root, and then the associated transitions to the children)
	// scenario_count_per_node_and_state[n,s] will be the number of MPR solutions ("scenarios") in which node n is at state s
	// scenario_count_per_node_and_state[,] will be filled in the order root-->tips
	// See pages 18-19 in: https://cs.brown.edu/courses/csci1950-z/slides/CSCI1950ZFall09_Lecture2.pdf
	// Note: This should be floating point, not int, because in the latter case you risk integer overflow and thus the spontaneous introduction of negative values! This cost me 1 day of bug-hunting!
	std::vector<double> scenario_count_per_node_and_state(Nnodes*Nstates, 0.0); // scenario_count_per_node_and_state[Nnodes x Nstates] in row-major format
	
	const double best_root_cost = array_min(master_cost_table, root*Nstates, (root*Nstates+Nstates-1));
	for(state=0; state<Nstates; ++state){
		if(abs(master_cost_table[root*Nstates+state]-best_root_cost)<=RELATIVE_EPSILON*best_root_cost){
			scenario_count_per_node_and_state[(root-Ntips)*Nstates + state] = 1;
		}
	}
	
	for(long q=0; q<traversal_queue_root2tips.size(); ++q){
		parent 	= traversal_queue_root2tips[q];
		node	= parent-Ntips;
		for(long ei=traversal_node2first_edge[node]; ei<=traversal_node2last_edge[node]; ++ei){
			edge  = traversal_edges[ei];
			child = tree_edge[edge*2+1];
			if(child<Ntips) continue;
			for(state=0; state<Nstates; ++state){
				if(scenario_count_per_node_and_state[node*Nstates+state]>0){
					// examine all optimal transitions parent --> child, when parent is at this particular state
					for(long transition_i=edge_and_state2first_transition[edge*Nstates+state]; transition_i<=edge_and_state2last_transition[edge*Nstates+state]; ++transition_i){
						transition = master_transitions[transition_i];
						// increment scenario_count for the corresponding state in this particular child
						scenario_count_per_node_and_state[(child-Ntips)*Nstates + transition] += scenario_count_per_node_and_state[node*Nstates + state];
					}
				}
			}
		}
		if(q % 100 == 0) Rcpp::checkUserInterrupt();
	}

		
	// For a given tree, there may be multiple alternative scenarios (MPR solutions) for the ancestral states
	// based on the scenario count per node and per state, define posterior_probabilities for nodes
	double mass;
	NumericMatrix posterior_probabilities(Nnodes,Nstates); // this will be a 2D array of size Nnodes x Nstates. Note that in this case we're not flattening, for convenience, because we're returning this to R and there we like to have a non-flattened 2D matrix. 
	for(node=0; node<Nnodes; ++node){
		mass = 0;
		if(weight_posteriors_by_scenario_counts){
			// weight states proportional to the number of scenarios
			for(state=0, mass=0; state<Nstates; ++state) mass += scenario_count_per_node_and_state[node*Nstates + state];
			if(mass==0){
				//if(verbose) Rcout << verbose_prefix << "WARNING: Node " << node << " (clade " << (node+Ntips) << ") has max-parsimony mass = 0 (i.e. no predicted state). It's posterior probabilities will all be set to NaN\n";
				for(state=0; state<Nstates; ++state) posterior_probabilities(node,state) = NAN_D;
			}else{
				for(state=0; state<Nstates; ++state) posterior_probabilities(node,state) = scenario_count_per_node_and_state[node*Nstates + state]/mass;
			}
		}else{
			// all states with non-zero scenario count are weighted equally
			for(state=0, mass=0; state<Nstates; ++state) mass += (scenario_count_per_node_and_state[node*Nstates + state]>0 ? 1.0 : 0.0);
			for(state=0; state<Nstates; ++state) posterior_probabilities(node,state) = (scenario_count_per_node_and_state[node*Nstates + state]>0 ? 1.0 : 0.0)/mass;			
		}
		if(node % 100 == 0) Rcpp::checkUserInterrupt();
	}

	return Rcpp::List::create(	Rcpp::Named("posterior_probabilities") 	= Rcpp::wrap(posterior_probabilities),
								Rcpp::Named("scenario_counts") 			= Rcpp::wrap(scenario_count_per_node_and_state),
								Rcpp::Named("best_root_cost")			= best_root_cost);
}



// calculate log-likelihood and posterior probability at the root, for a fixed-rates continuous-time Markov model for discrete character evolution
// A major computational bottleneck is the exponentiation of the transition matrix along each edge, i.e. exp(edge_length*Q), where Q[r,c] is the transition rate r-->c.
// Exponentiation of the transition matrix can happen in one of 2 ways:
//    1. By providing all pre-calculated exponentials, via expQ_per_edge. This uses more RAM, but is much faster than the other methods below.
//	  2. By providing a transition_exponentiator object. Only relevant if expQ_per_edge[] is empty.
// Returns false on failure
bool aux_ASR_with_fixed_rates_Markov_model(	const long					Ntips,
											const long 					Nnodes,
											const long					Nedges,
											const long					Nstates,
											const long					root,								// (INPUT) root of the tree, an integer in Ntips,..,Nnodes+Ntips-1
											const std::vector<long>		&tree_edge,							// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
											const std::vector<double> 	&edge_length, 						// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
											const std::vector<double>	&prior_probabilities_per_tip, 		// (INPUT) 2D array of size Ntips x Nstates, in row-major format, listing prior probability distributions for each tip
											const std::string			&root_prior_type,					// (INPUT) either "custom" (in which case root_prior[] must be provided) or "likelihoods" (the computed state-likelihoods at the root (D[s]) are used as prior probability distribution) or "max_likelihood" (the root state with greatest computed likelihood is assumed as state of the root). 
											const std::vector<double>	&root_prior,						// (INPUT) 1D array of size Nstates, listing prior probability distribution for root. Used to combine the root's clade likelihoods (D) into an overall log-likelihood of the model. Only relevant if root_prior_type=="custom", otherwise this can be an empty vector.
											const matrix_exponentiator	&transition_exponentiator,			// (INPUT) initialized exponentiator object for transition matrix. Either transition_exponentiator or expQ_per_edge must be provided.
											const std::vector<double>	&expQ_per_edge,						// (INPUT) 3D array of size Nedges x Nstates x Nstates, in layer-row-major format, listing the exponentiated transition matrix along each edge. Either transition_exponentiator or expQ_per_edge must be provided.
											const std::vector<long>		&traversal_queue,					// (INPUT) 1D array of size Nnodes, with values in Ntips:(Nclades-1). Traversal queue root-->tips (not including tips). Generated using the function get_tree_traversal_root_to_tips(include_tips=false).
											const std::vector<long>		&traversal_node2first_edge,			// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
											const std::vector<long>		&traversal_node2last_edge,			// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
											const std::vector<long>		&traversal_edges,					// (INPUT) 1D array of size Nedges, with values in 0:(Nedges-1). Generated using the function get_tree_traversal_root_to_tips().
											const double				runtime_out_seconds,				// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
											std::vector<double>			&logposteriors,						// (OUTPUT) 1D array of size Nnodes x Nstates, listing the logarithms of the posterior likelihoods at each node. This is used both as internal scratch space as well as to return the final posteriors.
											double						&loglikelihood,						// (OUTPUT) log-likelihood for the full tree
											std::string					&error){							// (OUTPUT) error message in the case of failure
	long clade, edge, child, node;
	const bool use_precalculated_expQ = (expQ_per_edge.size()>0);
	const double max_edge_length 	  = (edge_length.size()==0 ? 1.0 : array_max(edge_length));
	const double start_runtime 		  = (runtime_out_seconds>0 ? get_thread_monotonic_walltime_seconds() : 0.0);
	error = "";
							
	// calculate probability distribution on each node (traverse tips-->root)
	logposteriors.assign(Nnodes*Nstates,0.0);
	std::vector<double> Y, logY, expQ;
	double const *expQ_pointer;
	double LL_shift = 0; // log of the rescaling factor of the likelihood, or additive corection to the loglikelihood
	for(long q=traversal_queue.size()-1; q>=0; --q){
		clade  = traversal_queue[q];
		node   = clade-Ntips;
		// set probability distribution of clade to the element-wise product of its children's probability distributions
		for(long e=traversal_node2first_edge[node]; e<=traversal_node2last_edge[node]; ++e){
			edge  = traversal_edges[e];
			child = tree_edge[2*edge+1];
			if(use_precalculated_expQ){
				expQ_pointer = &expQ_per_edge[edge*Nstates*Nstates];
			}else{
				// calculate exponential of transition matrix along edge
				transition_exponentiator.get_exponential((edge_length.size()==0 ? 1.0 : edge_length[edge])/max_edge_length, expQ);
				expQ_pointer = &expQ[0];
			}
									
			// use exponentiated transition matrix to propagate likelihoods ("posteriors") from child to parent
			// posteriors[parent] = product_{child in children} exp(edge_length * Q) * posteriors[child]
			// this corresponds to solving the Kolmogorov backward equation along each child-edge in backward-time direction, with initial condition the likelihoods at the child
			if(child<Ntips){
				multiply_matrix_with_vector(Nstates, Nstates, expQ_pointer, &prior_probabilities_per_tip[child*Nstates], Y);
				for(long s=0; s<Nstates; ++s) logposteriors[node*Nstates+s] += log(Y[s]); // factor Y into the posterior likelihoods of this node
			}else{
				multiply_matrix_with_log_vector(Nstates, Nstates, expQ_pointer, &logposteriors[(child-Ntips)*Nstates], logY);
				for(long s=0; s<Nstates; ++s) logposteriors[node*Nstates+s] += logY[s]; // factor Y into the posterior likelihoods of this node
			}

			// rescale (normalize) clade's posterior likelihoods
			// this is necessary due to scaling issues for very large trees; the non-normalized posterior likelihoods tend to 0 for older nodes
			// note that since the Mk propagator (exp(t*Q)) is linear, rescaling just rescales the overall likelihood (this is corrected for below)
			double S = 0;
			for(long s=0; s<Nstates; ++s) S += exp(logposteriors[node*Nstates+s]);
			for(long s=0; s<Nstates; ++s) logposteriors[node*Nstates+s] -= log(S);
		
			// incorporate rescaling factor (used to normalize this node's posterior) into tree's loglikelihood
			// if we weren't rescaling each node's posterior, this would not be necessary
			LL_shift += log(S);
		}
		if(q % 100 == 0){
			Rcpp::checkUserInterrupt();
			if((runtime_out_seconds>0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
				error = "Timed out";
				return false;
			}
		}
	}
	
	// use root's posterior (combined with it's prior) to calculate the model's loglikelihood
	if(root_prior_type=="custom"){
		double L = 0, S = 0;
		for(long s=0; s<Nstates; ++s){
			L += exp(logposteriors[(root-Ntips)*Nstates+s])*root_prior[s];
			S += root_prior[s];
		}
		loglikelihood = log(L/S) + LL_shift;
	}else if(root_prior_type=="max_likelihood"){
		double max_log_prob = -INFTY_D;
		for(long s=0; s<Nstates; ++s) max_log_prob = max(max_log_prob, logposteriors[(root-Ntips)*Nstates+s]);
		loglikelihood = max_log_prob + LL_shift;
	}else if(root_prior_type=="likelihoods"){
		double L = 0, S = 0;
		for(long s=0; s<Nstates; ++s){
			L += SQ(exp(logposteriors[(root-Ntips)*Nstates+s]));
			S += exp(logposteriors[(root-Ntips)*Nstates+s]);
		}
		loglikelihood = log(L/S) + LL_shift;
	}
	return true;
}







// re-root the tree and update the posterior probabilities at each node (since each node's posterior is determined by the posteriors of its children)
// since rerooting at a new node does not change the tip/node/edge indices, nor the total number of inout edges per clade, we just need to update a few tree-access data structures and update the posteriors for the affected nodes.
// Note: This algorithm cannot easily be generalized to rerooting at tips, because this would change the total number of tips & nodes and hence mess up the indexing of tips {0,..,Ntips-1} and nodes {Ntips,...,Ntips+Nnodes-1}.
// Exponentiation of the transition matrix can happen in one of 3 ways:
//    1. By providing all pre-calculated exponentials, via expQ_per_edge. This uses more RAM, but is much faster than the other methods below.
//	  2. By providing a transition_exponentiator object.
// The above options are checked and utilized in the listed order. Whenever the associated arrays of an option are empty, the next option is checked.
void aux_reroot_and_update_ASR_with_fixed_rates_Markov_model(	const long					Ntips,
																const long 					Nnodes,
																const long					Nedges,
																const long					Nstates,
																const long					old_root,							// (INPUT) old (current) root of the tree, an integer in Ntips,..,Nnodes+Ntips-1
																const long					new_root,							// (INPUT) new root for the tree, an integer in Ntips,..,Nnodes+Ntips-1
																const std::vector<double> 	&edge_length, 						// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
																const double				max_edge_length,					// (INPUT) 
																const std::vector<long>		&clade2first_inout_edge,			// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping clades to their first incoming or outgoing edge.
																const std::vector<long>		&clade2last_inout_edge,				// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping clades to their last incoming or outgoing edge.
																const std::vector<long>		&inout_edges,						// (INPUT) 1D array of size 2*Nedges, with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in clade2first_inout_edge[] and clade2last_inout_edge[]) to original edge indices.
																const std::vector<double>	&prior_probabilities_per_tip, 		// (INPUT) 2D array of size Ntips x Nstates, in row-major format, listing prior probability distributions for each tip
																const matrix_exponentiator	&transition_exponentiator,			// (INPUT) initialized exponentiator object for transition matrix
																const std::vector<double>	&expQ_per_edge,						// (INPUT) Optional 3D array of size Nedges x Nstates x Nstates, in layer-row-major format, listing the exponentiated transition matrix along each edge. Only relevant if use_precalculated_expQ==true. Can be empty, if exp(Q) is to be calculated using eigendecomposition or polynomials.
																std::vector<long> 			&current_tree_edge,					// (INPUT/OUTPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1). Will be updated after the rerooting.		
																std::vector<long>			&current_incoming_edge_per_clade,	// (INPUT/OUTPUT) 1D array of size Nclades, with elements in 0,..,Nedges-1. Will be updated after the rerooting.
																std::vector<double>			&posteriors){						// (INPUT/OUTPUT) 1D array of size Nnodes x Nstates, listing the posterior probabilities at each node. Should be pre-computed for the current tree, and will be updated after the rerooting.
	const bool use_precalculated_expQ = (expQ_per_edge.size()>0);
	if(new_root==old_root) return; // nothing to do
	
	// reroot (this changes edge directions, but keeps tip/node/edge indices the same)
	reroot_tree_at_node(Ntips, Nnodes, Nedges, old_root, new_root, current_tree_edge, current_incoming_edge_per_clade);
	
	// update posteriors of nodes that have been traversed by the rerooting
	std::vector<double> Y, expQ;
	double const *expQ_pointer; // will point to the location of the exponentiated transition matrix (which may be different for each edge)
	long clade = old_root;
	while(true){
		// re-calculate posterior for the currently focal clade (based on the posteriors of its children)
		const long node = clade-Ntips;
		for(long s=0; s<Nstates; ++s) posteriors[node*Nstates+s] = 1.0; // initialize posteriors for this node, repopulate below based on children
		for(long e=clade2first_inout_edge[clade], edge, child; e<=clade2last_inout_edge[clade]; ++e){
			edge  = inout_edges[e];
			if(current_tree_edge[2*edge+0]!=clade) continue; // this edge is not outgoing from this clade (in the rerooted tree)
			child = current_tree_edge[2*edge+1];
			if(use_precalculated_expQ){
				expQ_pointer = &expQ_per_edge[edge*Nstates*Nstates];
			}else{
				// calculate exponential of transition matrix along edge
				transition_exponentiator.get_exponential((edge_length.size()==0 ? 1.0 : edge_length[edge])/max_edge_length, expQ);
				expQ_pointer = &expQ[0];
			}
			// use exponentiated transition matrix to propagate probabilities from children to parent
			// probabilities[parent] = product_{child in children} exp(edge_length * Q) * probabilities[child]
 			if(child<Ntips) multiply_matrix_with_vector(Nstates, Nstates, expQ_pointer, &prior_probabilities_per_tip[child*Nstates], Y);
 			else multiply_matrix_with_vector(Nstates, Nstates, expQ_pointer, &posteriors[(child-Ntips)*Nstates], Y);
			for(long s=0; s<Nstates; ++s) posteriors[node*Nstates+s] *= max(0.0,Y[s]); // factor Y into the posterior of this node. Avoid negative values from rounding errors
		}
			
		// rescale (normalize) clade's probability distribution
		double S = 0;
		for(long s=0; s<Nstates; ++s) S += posteriors[node*Nstates+s];
		for(long s=0; s<Nstates; ++s) posteriors[node*Nstates+s] /= S;
		
		// move on to parent
		if(clade!=new_root) clade = current_tree_edge[current_incoming_edge_per_clade[clade]*2+0];
		else break;
	}
}




// calculate the loglikelihood of a fixed-rates Markov model for discrete character evolution on a phylogenetic tree, provided a fixed transition matrix
// Optionally, the marginal ancestral likelihoods can be computed for all nodes, using the rerooting algorithm by [Yang et al. (1995). Genetics 141:1641-1650]
// Calculating marginal ancestral likelihoods substantially increases computation time, so don't request this if you only care about the loglikelihood (e.g. for fitting purposes)
// Optionally, the eigendecomposition of the transition matrix (eigenvalues & eigenvectors) can be provided to potentially speed up exponentiations. If provided, this is used blindly for all exponentiations.
// If an eigendecomposition for the transition_matrix is not provided, then a Taylor series (matrix polynomials) approximation is used instead. This includes preconditioning steps and seems to scratch quite well, and is similarly fast as using an eigendecomposition.
// Requirements:
//   Tree can include multi- and mono-furcations.
//   Tree must be rooted. Root will be determined automatically as the node with no parent.
// [[Rcpp::export]]
Rcpp::List ASR_with_fixed_rates_Markov_model_CPP(	const long					Ntips,
													const long 					Nnodes,
													const long					Nedges,
													const long					Nstates,
													const std::vector<long> 	&tree_edge,						// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
													const std::vector<double> 	&edge_length, 					// (INPUT) 1D array of size Nedges, or an empty std::vector (all edges have length 1)
													const std::vector<double> 	&transition_matrix,				// (INPUT) 2D array of size Nstates x Nstates, in row-major format. Transition-rate matrix Q in row-major format, i.e. Q[r*Nstates + c] is (r,c)-th-element of Q and equal to the transition rate r-->c.
													const ComplexVector			&eigenvalues,					// (INPUT) Optional 1D vector of size Nstates, listing the eigenvalues of the transition_matrix (corresponding to some diagonalization). Can also be an empty vector if eigendecomposition not available.
													const ComplexVector			&EVmatrix,						// (INPUT) Optional 2D array of size Nstates x Nstates, in row-major format, whose columns are the eigenvectors of the transition_matrix. Can also be an empty vector if eigendecomposition not available.
													const ComplexVector			&inverse_EVmatrix,				// (INPUT) Optional 2D array of size Nstates x Nstates, in row-major format, the inverse of EVmatrix. Can also be an empty vector if eigendecomposition not available.
													const std::vector<double>	&prior_probabilities_per_tip, 	// (INPUT) 2D array of size Ntips x Nstates, in row-major format, listing prior probability distributions for each tip
													const std::string			&root_prior_type,				// (INPUT) either "custom" (in which case root_prior[] must be provided) or "likelihoods" (the computed state-likelihoods at the root (D[s]) are used as prior probability distribution) or "max_likelihood" (the root state with greatest computed likelihood is assumed as state of the root). 
													const std::vector<double>	&root_prior,					// (INPUT) 1D array of size Nstates, listing prior probability distribution for root. Used to combine the root's clade likelihoods (D) into an overall log-likelihood of the model. Only relevant if root_prior_type=="custom", otherwise this can be an empty vector.
													bool						include_ancestral_likelihoods,	// (INPUT) whether to include ancestral likelihoods in return variables
													bool						reroot,							// (INPUT) if true, then the marginal ancestral states estimates (conditional scaled likelihoods as if each node was a root) of all nodes are also returned as an array of size Nnodes x Nstates. Only use if needed, since it's computationally expensive.
													const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
													const double				exponentiation_accuracy,		// (INPUT) maximum allowed error when exponentiating the transition matrix via polynomials, in terms of the Hilbert-Schmidt L2 norm. Only relevant if exponentiation is done using the polynomials.
													const long					max_polynomials,				// (INPUT) maximum possible number of polynomials to use for exponentiating the transition_matrix via polynomials, regardless of the pursued accuracy epsilon. Used as safety vault, but may break the guaranteed accuracy. A value ~100 is usually enough.
													const bool					store_exponentials){			// (INPUT) if True, then exponentials are pre-calculated and stored for the calculation of ancestral_likelihoods. This may save time because each exponential is only calculated once, but will use up more memory since all exponentials are stored. Only relevant if reroot==TRUE, otherwise exponentials are never stored.
	const double start_runtime 			= (runtime_out_seconds>0 ? get_thread_monotonic_walltime_seconds() : 0.0);
	const long Nclades 					= Ntips + Nnodes;
	const bool use_precalculated_expQ 	= (reroot && store_exponentials);
	const bool has_eigendecomposition	= (eigenvalues.size()>0 && EVmatrix.size()>0 && inverse_EVmatrix.size()>0);
	const double max_edge_length 		= (edge_length.size()==0 ? 1.0 : array_max(edge_length));
	const long root 					= get_root_clade(Ntips, Nnodes, Nedges, tree_edge);

	// transform some of the R vectors to C++ vectors
	std::vector<cdouble> eigenvaluesCPP, EVmatrixCPP, inverse_EVmatrixCPP;
	cast_ComplexVector_to_CPP(eigenvalues, eigenvaluesCPP);
	cast_ComplexVector_to_CPP(EVmatrix, EVmatrixCPP);
	cast_ComplexVector_to_CPP(inverse_EVmatrix, inverse_EVmatrixCPP);

	// prepare tree traversal route (root-->tips)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(Ntips,
									Nnodes,
									Nedges,
									root,
									tree_edge,
									false,	// don't include tips
									false,	// edge mappings are not pre-calculated
									traversal_queue,
									traversal_node2first_edge,
									traversal_node2last_edge,	
									traversal_edges,
									false,
									"");
	
	// check for abortion signal or timeout
	Rcpp::checkUserInterrupt();
	if((runtime_out_seconds>0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
		return Rcpp::List::create(	Rcpp::Named("success") = false, Rcpp::Named("error") = "Aborted prematurely, because we reached the maximum allowed processing time");
	}

	// prepare data structure for exponentiations of transition matrix, if needed
	matrix_exponentiator transition_exponentiator;
	if(has_eigendecomposition){
		// prepare exponentiator using eigen-decomposition
		transition_exponentiator.initialize(Nstates, eigenvaluesCPP, EVmatrixCPP, inverse_EVmatrixCPP, max_edge_length);
	}else{
		// prepare exponentiator using matrix polynomials
		const long min_polynomials = min_polynomials_for_positive_exponential_of_irreducible_matrix(Nstates, transition_matrix);
		transition_exponentiator.initialize(Nstates, transition_matrix, max_edge_length, 1e-4, min_polynomials, max_polynomials, true);
	}
											
	// pre-calculate exponentials of transition_matrix along edges if needed
	// This allows for a faster rerooting algorithm because it avoids repeated calculations of the same exponential, but needs more RAM if Nedges is large
	std::vector<double> expQ_per_edge;
	if(use_precalculated_expQ){
		std::vector<double> scratch_expQ;
		expQ_per_edge.resize(Nedges*Nstates*Nstates);
		for(long edge=0; edge<Nedges; ++edge){
			transition_exponentiator.get_exponential((edge_length.size()==0 ? 1.0 : edge_length[edge])/max_edge_length, scratch_expQ);
			for(long r=0; r<Nstates; ++r){
				for(long c=0; c<Nstates; ++c){
					expQ_per_edge[edge*Nstates*Nstates + r*Nstates + c] = scratch_expQ[r*Nstates + c];
				}
			}
			// check for abortion signal or timeout
			if(edge % 1000 == 0){
				Rcpp::checkUserInterrupt();
				if((runtime_out_seconds>0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
					return Rcpp::List::create(	Rcpp::Named("success") = false, Rcpp::Named("error") = "Aborted prematurely, because we reached the maximum allowed processing time");
				}
			}
		}
	}
			
	// calculate loglikelihood & posteriors for all nodes (this is not the same as the ancestral likelihoods)
	dvector logposteriors;
	double loglikelihood;
	string error;
	if(!aux_ASR_with_fixed_rates_Markov_model(	Ntips,
												Nnodes,
												Nedges,
												Nstates,
												root,
												tree_edge,
												edge_length,
												prior_probabilities_per_tip,
												root_prior_type,
												root_prior,
												transition_exponentiator,
												expQ_per_edge,
												traversal_queue,
												traversal_node2first_edge,
												traversal_node2last_edge,
												traversal_edges,
												(runtime_out_seconds>0 ? max(0.01*runtime_out_seconds, runtime_out_seconds-(get_thread_monotonic_walltime_seconds()-start_runtime)) : -1.0),
												logposteriors,
												loglikelihood,
												error)){
		return Rcpp::List::create(	Rcpp::Named("success") = false, Rcpp::Named("error") = "Calculation of likelihood failed: "+error);
	}
	
	// calculate state likelihoods at the nodes if requested
	dvector posteriors;
	if(include_ancestral_likelihoods){
		posteriors.resize(logposteriors.size());
		for(long n=0; n<posteriors.size(); ++n) posteriors[n] = exp(logposteriors[n]);
	}
	
	if(!include_ancestral_likelihoods){
		return Rcpp::List::create(	Rcpp::Named("success") = true, 
									Rcpp::Named("loglikelihood") = loglikelihood);
	}else if(!reroot){
		return Rcpp::List::create(	Rcpp::Named("success") 				 = true,
									Rcpp::Named("loglikelihood") 		 = loglikelihood,
									Rcpp::Named("ancestral_likelihoods") = Rcpp::wrap(posteriors));
	}

	// calculate marginal ancestral states (posterior probabilities) at each node, as if that node was the root [Yang et al. 1995]
	// note that the original edge mappings (e.g. traversal_node2first_edge[]) will no longer be consistent with current_tree_edge after rerooting
	// Notation: current_(..) refers to tree-access structures that are updated at each rerooting. They will be consistent with each other, but not necessarily with the original tree structure.
	std::vector<double> ancestral_likelihoods(Nnodes*Nstates);
	// prepare some data structures, which will be updated everytime we reroot
	long current_root = root;
	std::vector<long> current_tree_edge(tree_edge.begin(), tree_edge.end());
	std::vector<long> clade2first_inout_edge, clade2last_inout_edge, inout_edges; // will be invariant to rerootings
	get_inout_edges_per_clade(Ntips, Nnodes, Nedges, tree_edge, clade2first_inout_edge, clade2last_inout_edge, inout_edges);
	std::vector<long> current_incoming_edge_per_clade(Nclades,-1); // will change with each rerooting
	for(long edge=0; edge<Nedges; ++edge) current_incoming_edge_per_clade[current_tree_edge[edge*2+1]] = edge;
	// calculate depth-first-search traversal route
	// this minimizes the distance between successive old_root-new_root pairs and thus the computation required to update the posteriors on each rerooting
	// similarly, depth_first_search_queue[] will still list successive nodes, but not necessarily in downstream order
	std::vector<long> depth_first_search_queue;
	get_tree_traversal_depth_first_search(	Ntips,
											Nnodes,
											Nedges,
											root,
											tree_edge,
											false,	// don't include tips
											true, 	// edge mappings are pre-calculated
											depth_first_search_queue,
											traversal_node2first_edge,
											traversal_node2last_edge,
											traversal_edges);
	
	// reroot at each node, updating the tree-access structures and updating the posteriors at each node
	// edge directions will change, but tip/node/edge indices remain the same
	for(long q=0, new_root; q<depth_first_search_queue.size(); ++q){
		new_root = depth_first_search_queue[q];
		aux_reroot_and_update_ASR_with_fixed_rates_Markov_model(Ntips,
																Nnodes,
																Nedges,
																Nstates,
																current_root,
																new_root,
																edge_length,
																max_edge_length,
																clade2first_inout_edge,
																clade2last_inout_edge,
																inout_edges,
																prior_probabilities_per_tip,
																transition_exponentiator,
																expQ_per_edge,
																current_tree_edge,
																current_incoming_edge_per_clade,
																posteriors);
		// the posteriors of the root are equal to its ancestral_likelihoods, so extract those
		for(long s=0; s<Nstates; ++s) ancestral_likelihoods[(new_root-Ntips)*Nstates+s] = posteriors[(new_root-Ntips)*Nstates+s];
		current_root = new_root;
		// abort if the user has interrupted the calling R program
		if(q % 100 == 0){
			Rcpp::checkUserInterrupt();
			if((runtime_out_seconds>0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)){
				return Rcpp::List::create(	Rcpp::Named("success") = false, Rcpp::Named("error") = "Aborted prematurely during rerooting, because we reached the maximum allowed processing time", Rcpp::Named("loglikelihood") = loglikelihood);
			}
		}
	}
	
	return Rcpp::List::create(	Rcpp::Named("loglikelihood") = loglikelihood,
								Rcpp::Named("ancestral_likelihoods") = Rcpp::wrap(ancestral_likelihoods));
}




// calculate the loglikelihood of a fixed-rates continuous-time Markov model for discrete character evolution on a phylogenetic tree (Mk model)
// This is a simplification of the function ASR_with_fixed_rates_Markov_model_CPP(), optimized for likelihood calculations rather than ancestral state reconstruction
// Requirements:
//   Tree can include multi- and mono-furcations.
//   Tree must be rooted. Root will be determined automatically as the node with no parent.
// [[Rcpp::export]]
Rcpp::List Mk_loglikelihood_CPP(const long					Ntips,
								const long 					Nnodes,
								const long					Nedges,
								const long					Nstates,
								const std::vector<long> 	&tree_edge,						// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
								const std::vector<double> 	&edge_length, 					// (INPUT) 1D array of size Nedges, or an empty std::vector (all edges have length 1)
								const std::vector<double> 	&transition_matrix,				// (INPUT) 2D array of size Nstates x Nstates, in row-major format. Transition-rate matrix Q in row-major format, i.e. Q[r*Nstates + c] is (r,c)-th-element of Q and equal to the transition rate r-->c.
								const std::vector<double>	&prior_probabilities_per_tip, 	// (INPUT) 2D array of size Ntips x Nstates, in row-major format, listing prior probability distributions for each tip
								const std::string			&root_prior_type,				// (INPUT) either "custom" (in which case root_prior[] must be provided) or "likelihoods" (the computed state-likelihoods at the root (D[s]) are used as prior probability distribution) or "max_likelihood" (the root state with greatest computed likelihood is assumed as state of the root). 
								const std::vector<double>	&root_prior,					// (INPUT) 1D array of size Nstates, listing prior probability distribution for root. Used to combine the root's clade likelihoods (D) into an overall log-likelihood of the model. Only relevant if root_prior_type=="custom", otherwise this can be an empty vector.
								const double				oldest_age,						// (INPUT) oldest age to consider. Typically this will be <=root_age. If negative, this is set to the root age.
								const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
								const double				exponentiation_accuracy,		// (INPUT) maximum allowed error when exponentiating the transition matrix via polynomials, in terms of the Hilbert-Schmidt L2 norm. Only relevant if exponentiation is done using the polynomials.
								const long					max_polynomials){				// (INPUT) maximum possible number of polynomials to use for exponentiating the transition_matrix via polynomials, regardless of the pursued accuracy epsilon. Used as safety vault, but may break the guaranteed accuracy. A value ~100 is usually enough.
	const double start_runtime   = (runtime_out_seconds>0 ? get_thread_monotonic_walltime_seconds() : 0.0);
	const double max_edge_length = (edge_length.size()==0 ? 1.0 : array_max(edge_length));
	
	// determine tree root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);

	// get tree traversal route (root --> tips), not including tips			
	tree_traversal traversal(Ntips, Nnodes, Nedges, root, tree_edge, false, false);

	// calculate clade ages	
	dvector clade_ages;
	get_clade_ages(Ntips, Nnodes, Nedges, tree_edge, edge_length, traversal, clade_ages);

	// determine sub-trees depending on oldest_age
	lvector subroots;
	if(oldest_age<0){
		// set the oldest_age to the root
		subroots.push_back(root);
	}else{
		subroots.reserve(Ntips);
		for(long edge=0, child, parent; edge<Nedges; ++edge){
			parent = tree_edge[2*edge+0];
			child  = tree_edge[2*edge+1];
			if((clade_ages[child]<=oldest_age) && (clade_ages[parent]>oldest_age) && (child>=Ntips)){
				// cut off a subtree at this edge, with the edge's child being the root of that subtree
				// don't include tips as subroots, since on their own they contain no information for fitting the Mk model
				subroots.push_back(child);
			}
		}
		if(subroots.empty()) subroots.push_back(root); // oldest_age was placed before the tree's root
	}
	const long Nsubtrees = subroots.size();
	
	// prepare transition matrix exponentiator functor
	matrix_exponentiator transition_exponentiator;
	const long min_polynomials = min_polynomials_for_positive_exponential_of_irreducible_matrix(Nstates, transition_matrix);
	transition_exponentiator.initialize(Nstates, transition_matrix, max_edge_length, 1e-4, min_polynomials, max_polynomials, true);
							
	// calculate state likelihoods on each node (traverse tips-->root)
	// To avoid numerical overflow due to extremely small posterior densities, we represent the latter as a scale & log-transformed density variation
	// 	 Hence, the true posterior density at a given node is exp(posterior_scales[node,])*exp(logposteriors[node,]) 
	//   Since the Mk propagator (exp(t*Q)) is linear, we can propagate posteriors along edges conveniently even using this representation, i.e. with the overall scale factored out
	dvector logposteriors(Nnodes*Nstates,0.0), posterior_scales(Nnodes,0);
	dvector Y, logY, expQ;
	for(long q=traversal.queue.size()-1, clade, node; q>=0; --q){
		clade = traversal.queue[q];
		if(clade<Ntips) continue; // skip tips
		if((oldest_age>0) && (clade_ages[clade]>oldest_age)) continue;
		node = clade - Ntips;
		// set probability distribution of clade to the element-wise product of its children's probability distributions
		for(long e=traversal.node2first_edge[node], edge, child; e<=traversal.node2last_edge[node]; ++e){
			edge  = traversal.edge_mapping[e];
			child = tree_edge[2*edge+1];

			// calculate exponential of transition matrix along edge
			transition_exponentiator.get_exponential((edge_length.size()==0 ? 1.0 : edge_length[edge])/max_edge_length, expQ);
									
			// use exponentiated transition matrix to propagate likelihoods ("posteriors") from child to parent
			// posteriors[parent] = product_{child in children} exp(edge_length * Q) * posteriors[child]
			// this corresponds to solving the Kolmogorov backward equation along each child-edge in backward-time direction, with initial condition the likelihoods at the child
			if(child<Ntips){
				multiply_matrix_with_vector(Nstates, Nstates, &expQ[0], &prior_probabilities_per_tip[child*Nstates], Y);
				for(long s=0; s<Nstates; ++s) logposteriors[node*Nstates+s] += log(Y[s]); // factor Y into the posterior likelihoods of this node
			}else{
				multiply_matrix_with_log_vector(Nstates, Nstates, &expQ[0], &logposteriors[(child-Ntips)*Nstates], logY);
				for(long s=0; s<Nstates; ++s) logposteriors[node*Nstates+s] += logY[s]; // factor Y into the posterior likelihoods of this node
			}

			// rescale (normalize) clade's posterior likelihoods and keep track of the true scales
			double S = 0;
			for(long s=0; s<Nstates; ++s) S += exp(logposteriors[node*Nstates+s]);
			for(long s=0; s<Nstates; ++s) logposteriors[node*Nstates+s] -= log(S);
			posterior_scales[node] += (child<Ntips ? 0.0 : posterior_scales[child-Ntips]) + log(S); // count contribution of this scaling factor (used to normalize this node's posterior) to this node's posterior_scales			
		}
		if(q % 100 == 0){
			Rcpp::checkUserInterrupt();
			if((runtime_out_seconds>0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Timed out");
		}
	}
	
	
	// finalize model loglikelihood based on the calculated posteriors at each subtree root
	dvector subroot_logposteriors;
	double loglikelihood = 0, subroot_posterior_scale;
	for(long n=0, subroot; n<Nsubtrees; ++n){
		subroot = subroots[n];
		
		// get logposteriors[] at this subroot
		if(subroot<Ntips){
			// subroot is a tip, so use tip's prior
			extract_row(prior_probabilities_per_tip, Nstates, subroot, subroot_logposteriors);
			subroot_posterior_scale = 0;
		}else{
			extract_row(logposteriors, Nstates, subroot-Ntips, subroot_logposteriors);
			subroot_posterior_scale = posterior_scales[subroot-Ntips];
		}
				
		// calculate contribution of subroot to the overall loglikelihood
		if(root_prior_type=="custom"){
			double L = 0, S = 0;
			for(long s=0; s<Nstates; ++s){
				L += exp(subroot_logposteriors[s])*root_prior[s];
				S += root_prior[s];
			}
			loglikelihood += log(L/S) + subroot_posterior_scale;
		}else if(root_prior_type=="max_likelihood"){
			double max_log_prob = -INFTY_D;
			for(long s=0; s<Nstates; ++s) max_log_prob = max(max_log_prob, subroot_logposteriors[s]);
			loglikelihood += max_log_prob + subroot_posterior_scale;
		}else if(root_prior_type=="likelihoods"){
			double L = 0, S = 0;
			for(long s=0; s<Nstates; ++s){
				L += SQ(exp(subroot_logposteriors[s]));
				S += exp(subroot_logposteriors[s]);
			}
			loglikelihood += log(L/S) + subroot_posterior_scale;
		}
	}

	return Rcpp::List::create(	Rcpp::Named("success") 			= true, 
								Rcpp::Named("loglikelihood")	= loglikelihood);
}






// calculate the loglikelihood of a fixed-rates time-reversible continuous-time Markov model for discrete character evolution (Mk model) based on a set of independent contrasts
// Time reversibility means that the transition-rates matrix is symmetric.
// [[Rcpp::export]]
Rcpp::List TR_Mk_loglikelihood_ICs_CPP(	const long					Nstates,
										const std::vector<double> 	&phylodistances,				// (INPUT) 1D array of size NP, listing phylogenetic distances of the independent contrasts, i.e. phylodistances[p] is the phylogenetic (patristic) distance between the two tips of the p-th independent contrast
										const std::vector<double> 	&transitions, 					// (INPUT) 2D array of size NP x 2, in row-major format, listing the states of the tips in each independent contrast. I.e. the tips in the p-th independent contrast have states transitions[p,0] & transitions[p,1]
										const std::vector<double> 	&transition_matrix,				// (INPUT) 2D array of size Nstates x Nstates, in row-major format. Transition-rate matrix Q in row-major format, i.e. Q[r*Nstates + c] is (r,c)-th-element of Q and equal to the transition rate r-->c.
										const double				runtime_out_seconds,			// (INPUT) max allowed runtime in seconds. If <=0, this is ignored.
										const double				exponentiation_accuracy,		// (INPUT) maximum allowed error when exponentiating the transition matrix via polynomials, in terms of the Hilbert-Schmidt L2 norm. Only relevant if exponentiation is done using the polynomials.
										const long					max_polynomials){				// (INPUT) maximum possible number of polynomials to use for exponentiating the transition_matrix via polynomials, regardless of the pursued accuracy epsilon. Used as safety vault, but may break the guaranteed accuracy. A value ~100 is usually enough.
	const double start_runtime   	= (runtime_out_seconds>0 ? get_thread_monotonic_walltime_seconds() : 0.0);
	const double max_phylodistance 	= array_max(phylodistances);
	const long NP 					= phylodistances.size();
		
	// prepare transition matrix exponentiator functor
	matrix_exponentiator transition_exponentiator;
	const long min_polynomials = min_polynomials_for_positive_exponential_of_irreducible_matrix(Nstates, transition_matrix);
	transition_exponentiator.initialize(Nstates, transition_matrix, max_phylodistance, 1e-4, min_polynomials, max_polynomials, true);
	
	double LL = 0;
	dvector expQ;
	for(long p=0; p<NP; ++p){
		transition_exponentiator.get_exponential(phylodistances[p]/max_phylodistance, expQ);
		LL += log(expQ[transitions[p*2+0]*Nstates+transitions[p*2+1]]); // add contribution of this independent contrast: probability of state transitions[p,0] transitioning to transitions[p,1] and subsequent sampling of the destination tip
		if(p % 100 == 0){
			Rcpp::checkUserInterrupt();
			if((runtime_out_seconds>0) && (get_thread_monotonic_walltime_seconds()-start_runtime>=runtime_out_seconds)) return Rcpp::List::create(Rcpp::Named("success") = false, Rcpp::Named("error") = "Timed out");
		}
	}

	return Rcpp::List::create(	Rcpp::Named("success") 			= true, 
								Rcpp::Named("loglikelihood")	= LL);
}



/* OLD CODE. WORKS, BUT LESS EFFICIENTLY.

// calculate the loglikelihood of a fixed-rates Markov model for discrete character evolution on a phylogenetic tree, provided a fixed transition matrix
// Optionally, the marginal ancestral states (likelihoods) can be computed for all nodes, using the rerooting algorithm by [Yang et al. (1995). Genetics 141:1641-1650]
// Reconstructing marginal ancestral states substantially increases computation time, so don't request this if you only care about the loglikelihood (e.g. for fitting purposes)
// Requirements:
//   Tree must be rooted. Root will be determined automatically as the node with no parent.
// [[Rcpp::export]]
Rcpp::List ASR_with_fixed_rates_Markov_model_CPP(	const long			Ntips,
													const long 			Nnodes,
													const long			Nedges,
													const long			Nstates,
													const std::vector<long> &tree_edge,						// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
													const std::vector<double> &edge_length, 					// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
													const std::vector<double> &transition_matrix,				// (INPUT) 2D array of size Nstates x Nstates, in row-major format. Transition-rate matrix Q in row-major format, i.e. Q[r*Nstates + c] is (r,c)-th-element of Q
													const std::vector<double>	&prior_probabilities_per_tip, 	// (INPUT) 2D array of size Ntips x Nstates, in row-major format, listing prior probability distributions for each tip
													const std::vector<double>	&prior_probabilities_for_root,	// (INPUT) 1D array of size Nstates, listing prior probability distribution for root. Which prior you use for the root is often a matter of choice.
													bool				include_ancestral_likelihoods){	// (INPUT) if true, then the marginal ancestral states (posterior probabilities as if each node was a root) of all nodes are also returned as an array of size Nnodes x Nstates. Only use if needed, since it's computationally expensive.
	const long Nclades = Ntips+Nnodes;
	const bool use_precalculated_expQ = include_ancestral_likelihoods;
	std::vector<long> current_tree_edge(tree_edge.begin(), tree_edge.end());
	const double max_edge_length = (edge_length.size()==0 ? 1.0 : array_max(edge_length));
	long current_root = get_root_clade(Ntips, Nnodes, Nedges, current_tree_edge);
	
	// prepare data structures for exponentiations of transition matrix
	std::vector<double> transition_polynomials, transition_polynomial_norms;
	long Npolynomials;
	const long NPmax = 1000; // max number of matrix polynomials for calculating the exponential
	const double exponentiation_accuracy = 1e-3;
	calculate_matrix_polynomials(Nstates,
								std::vector<double>(transition_matrix.begin(), transition_matrix.end()),
								max_edge_length,
								exponentiation_accuracy,
								NPmin,
								NPmax,
								transition_polynomials,
								transition_polynomial_norms,
								Npolynomials);
								
	// pre-calculate exponentials of transition_matrix along edges if needed
	// This is faster but needs more RAM if Nedges is large
	std::vector<double> expQ_per_edge;
	if(use_precalculated_expQ){
		expQ_per_edge.resize(Nedges*Nstates*Nstates);
		std::vector<double> scratch_expQ;
		for(long edge=0; edge<Nedges; ++edge){
			get_matrix_exponential_using_polynomials(	Nstates,
													Npolynomials,
													transition_polynomials,
													transition_polynomial_norms,
													(edge_length.size()==0 ? 1.0 : edge_length[edge])/max_edge_length,
													exponentiation_accuracy,
													scratch_expQ);
			for(long r=0; r<Nstates; ++r){
				for(long c=0; c<Nstates; ++c){
					expQ_per_edge[edge*Nstates*Nstates + r*Nstates + c] = scratch_expQ[r*Nstates + c];
				}
			}
		}
	}
								
	// calculate loglikelihood
	std::vector<double> posteriors;
	std::vector<long> scratch_traversal_queue, scratch_traversal_node2first_edge, scratch_traversal_node2last_edge, scratch_traversal_edges;
	double loglikelihood;
	aux_ASR_with_fixed_rates_Markov_model(	Ntips,
											Nnodes,
											Nedges,
											Nstates,
											current_root,
											current_tree_edge,
											edge_length,
											prior_probabilities_per_tip,
											prior_probabilities_for_root,
											Npolynomials,
											transition_polynomials,
											transition_polynomial_norms,
											exponentiation_accuracy,
											NPmin,
											expQ_per_edge,
											use_precalculated_expQ,
											scratch_traversal_queue,
											scratch_traversal_node2first_edge,
											scratch_traversal_node2last_edge,
											scratch_traversal_edges,
											posteriors,
											loglikelihood);

	// calculate marginal ancestral states (posterior probabilities) at each node, as if that node was the root [Yang et al. 1995]
	std::vector<double> ancestral_likelihoods(Nnodes*Nstates);
	if(include_ancestral_likelihoods){
		std::vector<long> clade2first_inout_edge, clade2last_inout_edge, inout_edges;
		get_inout_edges_per_clade(Ntips, Nnodes, Nedges, current_tree_edge, clade2first_inout_edge, clade2last_inout_edge, inout_edges);
		double dummy_loglikelihood;
		for(long new_root=Ntips; new_root<Nclades; ++new_root){
			// reroot tree at this node (edge directions will change, but tip/node/edge indices remain the same)
			reroot_tree_at_node(Ntips, Nnodes, Nedges, current_root, new_root, current_tree_edge);
			current_root = new_root;
			// calculate posterior as if this node was the root
			aux_ASR_with_fixed_rates_Markov_model(	Ntips,
													Nnodes,
													Nedges,
													Nstates,
													new_root,
													current_tree_edge,
													edge_length,
													prior_probabilities_per_tip,
													prior_probabilities_for_root,
													Npolynomials,
													transition_polynomials,
													transition_polynomial_norms,
													exponentiation_accuracy,
													NPmin,
													expQ_per_edge,
													use_precalculated_expQ,
													scratch_traversal_queue,
													scratch_traversal_node2first_edge,
													scratch_traversal_node2last_edge,
													scratch_traversal_edges,
													posteriors,
													dummy_loglikelihood);
			for(long s=0; s<Nstates; ++s) ancestral_likelihoods[(new_root-Ntips)*Nstates+s] = posteriors[(new_root-Ntips)*Nstates + s];
		}
	}
	
	if(include_ancestral_likelihoods){
		return Rcpp::List::create(	Rcpp::Named("loglikelihood") = loglikelihood,
									Rcpp::Named("ancestral_likelihoods") = Rcpp::wrap(ancestral_likelihoods));
	}else{
		return Rcpp::List::create(	Rcpp::Named("loglikelihood") = loglikelihood);
	}
}
*/


// forward-project marginal likelihoods of a subset of nodes to their descending tips by applying the transposed exponentiated transition rate matrix
// [[Rcpp::export]]
NumericVector apply_fixed_rate_Markov_model_to_missing_clades_CPP(	const long					Ntips,
																	const long 					Nnodes,
																	const long					Nedges,
																	const long					Nstates,
																	const std::vector<long> 	&tree_edge,				// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
																	const std::vector<double> 	&edge_length, 			// (INPUT) 1D array of size Nedges, or an empty std::vector (all branches have length 1)
																	const std::vector<double> 	&transition_matrix,		// (INPUT) 2D array of size Nstates x Nstates, in row-major format. Transition-rate matrix Q in row-major format, i.e. Q[r*Nstates + c] is (r,c)-th-element of Q and equal to the transition rate r-->c. Sum-per-row should be zero.
																	const double				exponentiation_accuracy,// (INPUT) maximum allowed error when exponentiating the transition matrix, in terms of the Hilbert-Schmidt L2 norm.
																	const long					max_polynomials,		// (INPUT) maximum possible number of polynomials to use for exponentiating the transition_matrix, regardless of the pursued accuracy epsilon. Used as safety vault, but may break the guaranteed accuracy. A value ~100 is usually enough.
																	LogicalVector				likelihoods_known,		// (INPUT) 1D array of size Nclades, indicating whether the likelihoods for a particular clade are known (1) or unknown/to be determined (0).
																	std::vector<double>			likelihoods,			// (INPUT) 2D matrix of size Nclades x Nstates, in row-major format. Likelihoods of each state in each clade (tip & node) of the tree.
																	const bool					unknown_likelihoods_as_priors){	// (INPUT) use unknown likelihoods (i.e. likelihoods[r,:] when likelihoods_known[r]==false) as priors. If false, unknown likelihoods are ignored, and effectively a flat prior is used
	// determine root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);

	// get tree traversal route (tips --> root)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										false,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");
										
	// transpose transition matrix
	dvector transposed_transition_matrix;
	transpose_matrix(Nstates, Nstates, transition_matrix, transposed_transition_matrix);
										
	// prepare data structures for exponentiations of transition matrix
	const double max_edge_length = (edge_length.size()==0 ? 1.0 : array_max(edge_length));
	matrix_exponentiator transition_exponentiator;
	const long min_polynomials = min_polynomials_for_positive_exponential_of_irreducible_matrix(Nstates, transposed_transition_matrix);
	transition_exponentiator.initialize(Nstates, transposed_transition_matrix, max_edge_length, exponentiation_accuracy, min_polynomials, max_polynomials, true);
	
	// calculate unknown likelihoods based on parents with known likelihoods (traverse root --> tips)
	std::vector<double> expQ, Y;
	for(long q=0, node, clade; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		node = clade - Ntips;
		for(long e=traversal_node2first_edge[node], edge, child; e<=traversal_node2last_edge[node]; ++e){
			edge  = traversal_edges[e];
			child = tree_edge[edge*2+1];
			if(likelihoods_known[child]) continue; // skip children with known likelihoods
			// exponentiate transition matrix along this edge
			transition_exponentiator.get_exponential((edge_length.size()==0 ? 1.0 : edge_length[edge])/max_edge_length, expQ);
			// propagate clade's likelihoods to child by multiplying with exponentiated transition matrix
			multiply_matrix_with_vector(Nstates, Nstates, &expQ[0], &likelihoods[clade*Nstates], Y);
			if(unknown_likelihoods_as_priors){
				for(long s=0; s<Nstates; ++s) likelihoods[child*Nstates+s] *= Y[s];
			}else{
				for(long s=0; s<Nstates; ++s) likelihoods[child*Nstates+s] = Y[s];
			}
			likelihoods_known[child] = true;
		}
	}
										
	return Rcpp::wrap(likelihoods);														
}





// apply a set of numerical attributes to all clades in a tree, based on the attributes of their closest ancestor with known attributes
// [[Rcpp::export]]
NumericVector apply_attributes_to_descendants_CPP(	const long				Ntips,
													const long 				Nnodes,
													const long				Nedges,
													const long				Nattributes,
													const std::vector<long> &tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
													std::vector<long>		attributes_known,	// (INPUT) 1D array of size Nclades, indicating whether the attributes for a particular clade are known (1) or unknown/to be set (0).
													std::vector<double>		attributes){		// (INPUT) 2D matrix of size Nclades x Nattributes, in row-major format. Attributes of each state in each clade (tip & node) of the tree. Only the attributes for clades with known attributes are relevant.
	// determine root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);

	// get tree traversal route (tips --> root)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(	Ntips,
										Nnodes,
										Nedges,
										root,
										tree_edge,
										false,
										false,
										traversal_queue,
										traversal_node2first_edge,
										traversal_node2last_edge,
										traversal_edges,
										false,
										"");

	// set unknown attributes based on parents with known attributes (traverse root --> tips)
	for(long q=0, node, clade; q<traversal_queue.size(); ++q){
		clade = traversal_queue[q];
		node = clade - Ntips;
		for(long e=traversal_node2first_edge[node], edge, child; e<=traversal_node2last_edge[node]; ++e){
			edge  = traversal_edges[e];
			child = tree_edge[edge*2+1];
			if(attributes_known[child]) continue; // skip children with known attributes
			// propagate clade's attributes to child
			for(long s=0; s<Nattributes; ++s) attributes[child*Nattributes+s] = attributes[clade*Nattributes+s];
			attributes_known[child] = true;
		}
	}
										
	return Rcpp::wrap(attributes);														
}





// calculate (or update) quadratic parameters for squared-change parsimony [Maddison 1991], for a single focal node (based on its children)
template<class ARRAY_TYPE>
void aux_get_quadratic_parameters_for_squared_change_parsimony(	const long				Ntips,
																const long 				Nnodes,
																const ARRAY_TYPE 		&tree_edge,							// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
																const std::vector<double> 	&edge_length,						// (INPUT) 1D array of size Nedges, or an empty std::vector (all edges have length 1)
																const std::vector<double>		&tip_states,						// (INPUT) 1D array of size Ntips
																const std::vector<long>	&node2first_edge,					// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their first outgoing edge (or incoming/outgoing edge if edge_mapping_inout==true).
																const std::vector<long>	&node2last_edge,					// (INPUT) 1D array of size Nnodes, with values in 0:(Nedges-1), mapping nodes to their last outgoing edge (or incoming/outgoing edge if edge_mapping_inout==true).
																const std::vector<long>	&edge_mapping,						// (INPUT) 1D array of size Nedges (or 2*Nedges if edge_mapping_inout==true), with values in 0:(Nedges-1). Maps internal edge indices (i.e. as listed in node2first_edge[] and node2last_edge[]) to original edge indices.
																const bool				edge_mapping_inout,					// (INPUT) whether the provided edge mapping lists inout edges. If false, then edge mapping must only list outgoing edges.
																const double			edge_length_epsilon,				// (INPUT) replacement length to use for edges with length 0, so that children are weighted much more strongly compared to children with edge_length>0
																const long				focal_node,							// (INPUT) focal node for which to calculate/update the quadratic parameters
																std::vector<double>		&quadratic_parameters_per_node){	// (INPUT/OUTPUT) 2D array of size Nnodes x 3, in row-major format, storing the squared-change parsimony quadratic parametsrs [Maddison 1991]. For focal_node, these parameters will be updated based on its children.
		const long focal_clade = focal_node + Ntips;
		double p1, p2, p3, length;
		long edge, child;
		
		// initialize
		quadratic_parameters_per_node[focal_node*3+0] = 0;
		quadratic_parameters_per_node[focal_node*3+1] = 0;
		quadratic_parameters_per_node[focal_node*3+2] = 0;
		
		// traverse through the focal node's children
		for(long e=node2first_edge[focal_node]; e<=node2last_edge[focal_node]; ++e){
			edge  	= edge_mapping[e];
			if(edge_mapping_inout && (tree_edge[2*edge+0]!=focal_clade)) continue; // this is not an outgoing edge
			child 	= tree_edge[2*edge+1];
			length 	= (edge_length.size()==0 ? 1.0 : edge_length[edge]);
			if(length==0) length = edge_length_epsilon;
			if(child>=Ntips){
				p1 = quadratic_parameters_per_node[(child-Ntips)*3+0];
				p2 = quadratic_parameters_per_node[(child-Ntips)*3+1];
				p3 = quadratic_parameters_per_node[(child-Ntips)*3+2];
			}
			quadratic_parameters_per_node[focal_node*3+0] += (child<Ntips ? 1.0/length						: p1/(length*p1+1));
			quadratic_parameters_per_node[focal_node*3+1] += (child<Ntips ? -2*tip_states[child]/length 	: p2/(length*p1+1));
			quadratic_parameters_per_node[focal_node*3+2] += (child<Ntips ? SQR(tip_states[child])/length 	: p3 - length*SQR(p2)/(4*(length*p1+1)));
		}
}




// Reconstruct of continuous ancestral states via squared change parsimony
// This is related to Phylogenetically Independent Contrasts ASR), with the difference that PIC reconstruction for a node only takes into account the subtree descending from a node.
// Note that the "mean node value" X_i of a node used to calculate the phylogenetic independent contrasts (PICs) by Felsenstein, 
//		is in fact the value that would have been reconstructed at P by (weighted) squared-change parsimony were P's clade is the whole tree; 
//		that is, it minimizes locally (over P's clade) the sum of (weighted) squared changes [as explained by Maddison 1991].
//		For the root, this is also the global optimum.
//		Hence, to obtain ancestral states for non-root nodes, you need to reroot at each node.
//		Maddison (1991) provides an alternative postorder-traversal algorithm to Felsenstein (whose original intention was not ASR), for arriving at the same local estimates for each node (=global estimate for the root) 
//		by keeping track of "quadratic parameters" at each tip/node. The quadratic parameters of each node can be calculated purely based on the  quadratic parameters of its children, and branch lengths can be included for weighting.
//		It turns out that the calculation of quadratic parameters sensu Maddison is more easily generalizable to multifurcations.
// To obtain the classical PIC estimates, set global=false. To obtain Maddison's global estimates (via rerooting) set global=true.
// Literature:
//    Felsenstein (1985). Phylogenies and the Comparative Method. The American Naturalist. 125:1-15.
//    Maddison (1991). Squared-change parsimony reconstructions of ancestral states for continuous-valued characters on a phylogenetic tree. Systematic Zoology. 40:304-314.
// Requirements:
//   Tree can include multi- and mono-furcations.
//   Tree must be rooted. Root will be determined automatically as the node with no parent.
// [[Rcpp::export]]
Rcpp::List ASR_via_squared_change_parsimony_CPP(const long			Ntips,
												const long 			Nnodes,
												const long			Nedges,
												const std::vector<long> &tree_edge,			// (INPUT) 2D array of size Nedges x 2, in row-major format, with elements in 0,..,(Nclades-1)				
												const std::vector<double> &edge_length,		// (INPUT) 1D array of size Nedges, or an empty std::vector (all edges have length 1)
												const std::vector<double>	&tip_states,		// (INPUT) numeric states of the trait at each tip
												bool				global){			// (INPUT) if true, then global squared-change parsimony state estimates are returned (as recommended by Maddison). This requires rerooting at each node and updating the quadratic parameters. If false, then the local estimate of each node (i.e. only considering its descending subtree) are returned. This corresponds to classical PIC.
	long node, clade;
	const double edge_length_epsilon = RELATIVE_EPSILON * array_nonzero_min(edge_length); // substitute to be used for zero edge lengths
	std::vector<double> ancestral_states(Nnodes); // will be populated later on
	
	// determine root
	const long root = get_root_clade(Ntips, Nnodes, Nedges, tree_edge);

	// prepare tree traversal route (root-->tips)
	std::vector<long> traversal_queue, traversal_node2first_edge, traversal_node2last_edge, traversal_edges;
	get_tree_traversal_root_to_tips(Ntips,
									Nnodes,
									Nedges,
									root,
									tree_edge,
									false,	// don't include tips
									false,	// edge mappings are not pre-calculated
									traversal_queue,
									traversal_node2first_edge,
									traversal_node2last_edge,	
									traversal_edges,
									false,
									"");
								
	// calculate quadratic parameters [Maddison 1991] in a postorder traversal (tips-->root)
	std::vector<double> quadratic_parameters_per_node(Nnodes*3); // 2D array of size Nnodes x 3, in row-major format
	for(long q=traversal_queue.size()-1; q>=0; --q){
		aux_get_quadratic_parameters_for_squared_change_parsimony(	Ntips,
																	Nnodes,
																	tree_edge,
																	edge_length,
																	tip_states,
																	traversal_node2first_edge,
																	traversal_node2last_edge,
																	traversal_edges,
																	false, // egde mappings are not inout
																	edge_length_epsilon,
																	traversal_queue[q]-Ntips, // focal node for which to calculate quadratic parameters (based on its children)
																	quadratic_parameters_per_node);
	}
	const double TSS = quadratic_parameters_per_node[(root-Ntips)*3+2] - SQR(quadratic_parameters_per_node[(root-Ntips)*3+1])/(4*quadratic_parameters_per_node[(root-Ntips)*3+0]); // minimized total sum of squared changes over the tree [Maddison 1991, Formula 7]
		
	if(!global){
		// return the local state estimate for each node, i.e. only taking into account its descending subtree
		// this is equivalent to the X_k from Felsenstein's phylogenetic independent contrasts
		// however we used Maddison's quadratic parameters because it's convenient and has been generalized to multifurcations [Maddison 1991]
		for(node=0; node<Nnodes; ++node){
			ancestral_states[node] = -quadratic_parameters_per_node[node*3+1]/(2*quadratic_parameters_per_node[node*3+0]); // [Maddison 1991, page 312]
		}
		
	}else{
		// calculate global estimates: This requires rerooting at each node and updating affected quadratic parameters.
		
		// prepare some data structures for tree traversal, which will be updated everytime we reroot
		// Note that the original edge mappings (e.g. traversal_node2first_edge[]) will no longer be consistent with current_tree_edge after rerooting
		// Notation: current_(..) refers to tree-access structures that are updated at each rerooting. They will be consistent with each other, but not necessarily with the original tree structure.
		long current_root = root;
		std::vector<long> current_tree_edge(tree_edge.begin(), tree_edge.end());
		std::vector<long> current_incoming_edge_per_clade; // will change with each rerooting
		get_incoming_edge_per_clade(Ntips, Nnodes, Nedges, current_tree_edge, current_incoming_edge_per_clade);
		
		// get inout edges for each node
		// these will be invariant to rerootings
		std::vector<long> node2first_inout_edge, node2last_inout_edge, inout_edges;
		get_inout_edges_per_node(Ntips, Nnodes, Nedges, tree_edge, node2first_inout_edge, node2last_inout_edge, inout_edges);

		// calculate depth-first-search traversal route
		// this minimizes the distance between successive old_root-new_root pairs and thus the computation required to update the posteriors on each rerooting
		// similarly, depth_first_search_queue[] will still list successive nodes, but not necessarily in downstream order
		std::vector<long> depth_first_search_queue;
		get_tree_traversal_depth_first_search(	Ntips,
												Nnodes,
												Nedges,
												root,
												tree_edge,
												false,	// don't include tips
							