#' Generate C code for a function and compile it
#' 
#' @param f Named character vector containing the right-hand sides of the ODE
#' @param forcings Character vector with the names of the forcings
#' @param jacobian Logical indicating whether the jacobian is computed and written into the C file
#' @param boundary data.frame with columns name, yini, yend specifying the boundary condition set-up. NULL if not a boundary value problem
#' @param compile Logical. If FALSE, only the C file is written
#' @param nGridpoints Integer, defining the number of grid points between tmin and tmax where the ODE
#' is computed in any case. 
#' @param modelname Character. The C file is generated in the working directory and is named <modelname>.c.
#' If \code{NULL}, a random name starting with ".f" is chosen, i.e. the file is hidden on a UNIX system.
#' @details The function replaces variables by arrays \code{y[i]}, etc. and replaces "^" by pow() 
#' in order to have the correct C syntax. The file name of the C-File is derived from \code{f}. 
#' I.e. \code{funC(abc, ...} will generate a file abc.c in the current directory. 
#' Currently, only explicit ODE specification is supported, i.e. you need to have the right-hand sides of the ODE.
#' In case you have explicitly time-dependent ODEs, please introduce time t as explicit variable, e.g. \code{f <- c(t = "1", x = "a*t - x", ...)}.
#' 
#' @return the name of the generated shared object file together with a number of attributes
#' @examples 
#' # Exponential decay plus constant supply
#' f <- c(x = "-k*x + supply")
#' func <- funC(f, forcings = "supply")
#' @export
funC <- function(f, forcings=NULL, jacobian=FALSE, boundary=NULL, compile = TRUE, nGridpoints = 500, modelname = NULL) {
  
  myattr <- attributes(f)
  if("names"%in%names(myattr)) myattr <- myattr[-which(names(myattr)=="names")]
  
  if(is.null(modelname)) modelname <- paste(c(".f", sample(c(letters, 0:9), 8, TRUE)), collapse="")
  dllname <- modelname
  filename <- paste0(dllname, ".c")
  
  ## If boundary conditions are given, sort for leftbc first
  if(!is.null(boundary)) {
    leftbc <- boundary$name[!is.na(boundary$yini)]
    f <- c(f[names(f)%in%leftbc], f[!names(f)%in%leftbc])
  }
  
  ## Analyze f by parser
  
  variables <- names(f)
  symbols <- getSymbols(f)
  parameters <- symbols[!symbols%in%c(variables, forcings, "time")]
  jac <- NULL
  
  if(jacobian) jac  <- jacobianSymb(f)
  not.zero.jac <- which(jac != "0")
  
  dv <- length(variables)
  dp <- length(parameters)
  if(is.null(forcings)) di <- 0 else di <- length(forcings)
  
  ## Replace powers and symbols to get correct C syntax
  
  f <- replaceOperation("^", "pow", f)
  f <- replaceSymbols(variables, paste0("y[", 1:length(variables)-1, "]"), f)
  
  if(jacobian) {
    jac <- replaceOperation("^", "pow", jac)
    jac <- replaceSymbols(variables, paste0("y[", 1:length(variables)-1, "]"), jac)
  }
  
  
  
  
  
  
  
  ## ------------ write C code -------------
  
  mypath <- system.file(package="R2CdeSolve")
  #splinefile <- paste0("cat ", mypath,"/code/splineCreateEvaluate.c")
  includings <- c("#include <R.h>",
                  "#include <math.h>")
  definitions <- paste0("#define ", c(parameters, paste0("y",0:(dv-1),"_0")), " parms[", 0:(dv+dp-1),"]")
  if(!is.null(forcings)) definitions <- c(definitions, paste0("#define ", forcings, " forc[", 0:(di-1),"]"))
  
  sink(filename)
  cat("/** Code auto-generated by cOde", as.character(packageVersion("cOde")), "**/\n")
  cat(paste(includings, "\n"))
  cat("\n")
  cat(paste("static double parms[", dv+dp,"];\n", sep=""))
  cat(paste("static double forc[", di,"];\n", sep=""))
  cat("\n")
  cat(paste(definitions, "\n"))
  cat("\n")
  cat("void initmod(void (* odeparms)(int *, double *)) {\n")
  cat(paste("\t int N=", dv+dp,";\n",sep=""))
  cat("\t odeparms(&N, parms);\n")
  cat("}\n")
  cat("\n")
  cat("void initforc(void (* odeforcs)(int *, double *)) {\n")
  cat(paste("\t int N=", di,";\n",sep=""))
  cat("\t odeforcs(&N, forc);\n")
  cat("}\n")
  cat("\n")
  
  
  ## Derivative function
  
  cat("/** Derivatives (ODE system) **/\n")
  cat("void derivs (int *n, double *t, double *y, double *ydot, double *RPAR, int *IPAR) {\n")
  cat("\n")
  cat("\t double time = *t;\n")
  cat("\n")
  
  #if(length(reductions)>0) cat(paste("\t double ", reductions, ";\n", sep=""))
  cat(paste("\t ydot[", 0:(dv-1),"] = ", f,";\n", sep=""))
  cat("\n")
  cat("}\n")
  cat("\n")
  
  ## Jacobian of deriv
  if(jacobian) {
    cat("/** Jacobian of the ODE system **/\n")
    cat("void jacobian (int *n, double *t, double *y, double * df, double *RPAR, int *IPAR) {\n")
    cat("\n")
    cat("double time = *t;\n")
    cat("\n")
    cat("\t int i;\n")
    cat("for(i=0; i<(*n) * (*n); i++) df[i] = 0.;\n")
    cat(paste("\t df[", not.zero.jac-1,"] = ", jac[not.zero.jac],";\n", sep=""))
    cat("\n")
    cat("}\n")
    cat("\n")
  }
  
  
  if(!is.null(boundary)) {
    
    ## Check length of boundary conditions
    nbc <- length(which(!is.na(c(boundary$yini, boundary$yend))))
    if(nbc != dv) {
      sink()
      warning("Number of boundary conditions not correct\n")
      return()
    }
    
    boundary <- boundary[match(variables, boundary$name),]
    
    leftbc <- which(!is.na(boundary$yini))
    rightbc <- which(!is.na(boundary$yend))
    myorder <- c(leftbc, rightbc)
    
    ## Boundary Condition (for compatibility with bvpSolve)
    
    cat("/** Boundary Conditions **/\n")
    cat("void gsub(int *i, int *n, double *z, double *g, double *RPAR, int *IPAR) {\n")
    cat("\n")
    cat(paste("\t if (*i==", 1,") *g=z[", myorder[1]-1, "]-y", 0, "_0;\n", sep=""))
    if(dv>1) cat(paste("\t else if (*i==", 2:dv,") *g=z[", myorder[-1]-1, "]-y", 2:dv-1, "_0;\n", sep=""))
    cat("\n")
    cat("}\n")
    cat("\n")
    
    ## Jacobian of Boundary Condition (for compatibility with bvpSolve)
    
    cat("/** Jacobian of the Boundary Conditions **/\n")
    cat("void dgsub(int *i, int *n, double *z, double *dg, double *RPAR, int *IPAR) {\n")
    cat("\n")
    cat("\t int j;\n")
    cat("\t for (j = 0; j< *n; j++) dg[j] = 0;\n")
    
    cat(paste("\t if (*i==", 1,") dg[", myorder[1]-1, "] = 1.;\n", sep=""))
    if(dv>1) cat(paste("\t else if (*i==", 2:dv,") dg[", myorder[-1]-1, "]=1.;\n", sep=""))
    cat("\n")
    cat("}\n")
  }
  
  
  
  sink()
  
  ## ----------- compile C code and load shared object file---------
  
  .so <- .Platform$dynlib.ext
  soExists <- file.exists(paste0(dllname, .so))
  if(compile) 
    system(paste("R CMD SHLIB", filename))
  
  #dyn.load(paste0(dllname, .so))
  
  
  ## ----------- function return -----------
  
  
  f <- dllname
  attributes(f) <- c(attributes(f), myattr)
  
  attr(f, "variables") <- variables
  attr(f, "parameters") <- parameters
  attr(f, "forcings") <- forcings
  attr(f, "jacobian") <- jacobian
  attr(f, "boundary") <- boundary
  attr(f, "nGridpoints") <- nGridpoints
  
  
  class(f) <- c("nospline", class(f))
  
  
  
  return(f)
  
}



#' Dynamically load DLL with automatic unloading of conflicting DLLs
#' 
#' @param func result from funC(), contains the information about the DLL name to be loaded
#' @param cfunction character, denoting the C function name.
#' @details If the C function name is already part of another loaded DLL, the corresponding DLL is
#' unloaded and the desired func DLL is loaded instead.
#' @export
loadDLL <- function(func, cfunction="deriv") {
  
  .so <- .Platform$dynlib.ext
  checkDLL <- try(getNativeSymbolInfo(cfunction), silent=TRUE)
  if(inherits(checkDLL, "try-error")) {
    dyn.load(paste0(func, .so))
  } else if((checkDLL$package)[[1]] != func) {
    #warning("Conflicting DLL was unloaded")
    dyn.unload(paste0((checkDLL$package)[[1]], .so))
    dyn.load(paste0(func, .so))
  }
  
}

#' Generate interpolation spline for the forcings and write into list of matrices
#' 
#' @param func result from funC()
#' @param forcings data.frame with columns name (factor), time (numeric) and value (numeric)
#' @return list of matrices with time points and values assigned to the forcings interface of deSolve
#' @details Splines are generated for each name in forcings and both, function value and first
#' derivative are evaluated at the time points of the data frame.
#' @examples
#' f <- c(x = "-k*x + a - b")
#' func <- funC(f, forcings = c("a", "b"))
#' forcData <- rbind(
#'   data.frame(name = "a", time = c(0, 1, 10), value = c(0, 5, 2)),
#'   data.frame(name = "b", time = c(0, 5, 10), value = c(1, 3, 6)))
#' forc <- setForcings(func, forcData) 
#' @export
setForcings <- function(func, forcings) {
  
  
  inputs <- attr(func, "forcings")
  
  times <- NULL
  values <- NULL
  timespan <- range(forcings$time)
  
  out <- do.call(c, lapply(inputs, function(i) {
    
    t <- forcings[forcings$name == i, "time"]
    x <- forcings[forcings$name == i, "value"]
    
    if(length(t)==1) {
      t <- seq(timespan[1], timespan[2], len=4)
      x <- rep(x, 4)
    }
    
    
    
    mat <- list(cbind(t, x))
    names(mat) <- i
    
    
    return(mat)
    
  }))
  
  
  return(out)
  
}


#' Interface to ode()
#' 
#' @param y named vector of type numeric. Initial values for the integration
#' @param times vector of type numeric. Integration times
#' @param func return value from funC()
#' @param parms named vector of type numeric. 
#' @param ... further arguments going to \code{ode()}
#' @details See deSolve-package for a full description of possible arguments
#' @return matrix with times and states
#' @example inst/examples/example1.R
#' @export
odeC <- function(y, times, func, parms, ...) {
  
  nGridpoints <- attr(func, "nGridpoints")
  times.inner <- seq(min(c(times, 0)), max(times), len=nGridpoints)
  times.inner <- sort(unique(c(times, times.inner)))
  which.times <- match(times, times.inner)
  
  loadDLL(func)
  y <- y[attr(func, "variables")]
  parms <- parms[attr(func, "parameters")]
  parms <- c(parms, rep(0, length(y)))
  if (attr(func, "jacobian")) 
    jacfunc <- "jacobian"
  else jacfunc <- NULL
  if (is.null(attr(func, "forcings"))) 
    initforc <- NULL
  else initforc <- "initforc"
  out <- deSolve::ode(y, times.inner, "derivs", parms, dllname = func, initfunc = "initmod", 
             initforc = initforc, jacfunc = jacfunc, 
             ...)[which.times,]
  return(out)
  
  
}


#' Interface to bvptwp()
#' 
#' 
#' @param yini named vector of type numeric. Initial values to be overwritten.
#' @param x vector of type numeric. Integration times
#' @param func return value from funC() with a boundary argument. 
#' @param yend named vector of type numeric. End values to be overwritten.
#' @param parms named vector of type numeric. The dynamic parameters.
#' @param xguess vector of type numeric, the x values
#' @param yguess matrix with as many rows as variables and columns as x values
#' @param ... further arguments going to \code{bvptwp()}
#' @details See bvpSolve-package for a full description of possible arguments
#' @return matrix with times and states
#' @example inst/examples/example4.R
#' @export
bvptwpC <- function(yini=NULL, x, func, yend=NULL, parms, xguess=NULL, yguess=NULL,  ...) {
  
  loadDLL(func)
  
  dynpar <- parms[attr(func, "parameters")]
  boundary <- attr(func, "boundary")
  leftbc <- boundary$name[!is.na(boundary$yini)]
  rightbc <- boundary$name[!is.na(boundary$yend)]
  
  ## Fill yini/yend with values from func. If yini/yend are given,
  ## set their values.
  bini <- boundary$yini
  names(bini) <- boundary$name
  bini <- bini[!is.na(bini)]
  
  bend <- boundary$yend
  names(bend) <- boundary$name
  bend <- bend[!is.na(bend)]
  
  
  if(!is.null(yini)) bini[names(yini)] <- yini
  if(!is.null(yend)) bend[names(yend)] <- yend
  if(is.null(attr(func, "forcings"))) initforc <- NULL else initforc <- "initforc"
  
  
  
  posbound <- c(rep(min(x), length(bini)), rep(max(x), length(bend)))
  
  
  statepars <- c(bini, bend)
  newparms <- c(dynpar, statepars)
  
  
  out <- bvpSolve::bvptwp(x = x, parms = newparms, xguess = xguess, yguess = yguess, posbound=posbound,
                          func = "derivs", jacfunc = "jacobian", bound = "gsub", jacbound = "dgsub", 
                          initfunc = "initmod", initforc = initforc,
                          dllname = func,
                          ncomp = length(statepars),
                          ...)
  
  colnames(out) <- c("x", attr(func, "variables"))
  
  return(out)
  
  
}

