#' bspline: build and use B-splines for interpolation and regression.
#' 
#' Build and use B-splines for interpolation and regression.
#'   In case of regression, equality constraints as well as monotonicity
#'   requirement can be imposed. Moreover, 
#'   knot positions (not only spline coefficients) can be part of 
#'   optimized parameters too. User is provided with 
#'   functions calculating spline values at arbitrary points. This 
#'   functions can be differentiated to obtain B-splines calculating 
#'   derivatives at any point. B-splines of this package can 
#'   simultaneously operate on a series of curves sharing the same set of 
#'   knots. 'bspline' is written with concern about computing 
#'   performance that's why the basis calculation is implemented in C++.
#'   The rest is implemented in R but without notable impact on computing speed.
#'
#' @section bspline functions: \itemize{
#'  \item{\code{bsc}:}{ basis matrix (implemented in C++)}
#'  \item{\code{bsp}:}{ values of B-spline from its coefficients}
#'  \item{\code{dbsp}:}{ derivative of B-spline}
#'  \item{\code{par2bsp}:}{ build B-spline function from parameters }
#'  \item{\code{bsppar}:}{ retrieve B-spline parameters from its function}
#'  \item{\code{smbsp}:}{ build smoothing B-spline}
#'  \item{\code{fitsmbsp}:}{ build smoothing B-spline with optimized knot positions}
#'  \item{\code{diffn}:}{ finite differences}
#' }
#'
#' @docType package
#' @name bspline
#' @useDynLib bspline, .registration=TRUE
NULL

#' Calculate B-spline values from their coefficients qw and knots xk
#'
#' @param x Numeric vector, abscissa points at which B-splines should be calculated.
#'   They are supposed to be non decreasing.
#' @param xk Numeric vector, knots of the B-splines. They are supposed to be non decreasing.
#' @param qw Numeric vector or matrix, coefficients of B-splines. \code{NROW(qw)}
#'   must be equal to \code{length(xk)-n-1} where \code{n} is the next parameter
#' @param n Integer scalar, polynomial order of B-splines, by default cubic splines
#'   are calculated.
#' @return Numeric matrix (column number depends on qw dimensions), B-spline values on x.
#' @details This function does nothing else than calculate a dot-product between
#'   a B-spline basis matrix calculated by \code{bsc()} and coefficients \code{qw}.
#'   If qw is a matrix, each
#'   column corresponds to a separate set of coefficients.
#'   For x values falling outside of xk range, the B-splines values are set to 0.
#'   To get a function calculating spline values at arbitrary points from \code{xk}
#'   and \code{qw}, cf. \code{par2bsp()}.
#' @seealso {bsc}, {par2bsp}
#' @export
bsp=function(x, xk, qw, n=3L) {
    stopifnot(NROW(qw) == length(xk)-n-1L)
    bsc(x, xk, n=n)%*%qw
}

#' Smoothing B-spline of order n >= 0
#'
#' @param x Numeric vector, abscissa points
#' @param y Numeric vector or matrix or data.frame, ordinate values to be smoothed
#'   (one set per column in case of matrix or data.frame)
#' @param n Integer scalar, polynomial order of B-splines (3 by default)
#' @param xki Numeric vector, strictly internal B-spline knots, i.e. lying strictly
#'   inside of \code{x} bounds. If NULL (by default), they are
#'   estimated with the help of \code{iknots()}. This vector is used as initial approximation
#'   during optimization process. Must be non decreasing if not NULL.
#' @param nki Integer scalar, internal knot number (1 by default). When
#'   nki==0, it corresponds to polynomial regression. If \code{xki}
#'   is not NULL, this parameter is ignored.
#' @param lieq List, equality constraints to respect by the smoothing spline,
#'   one list item per y column. By default (NULL), no constraint is imposed.
#'   Constraints are given as a 2-column matrix \code{(xe, ye)} where
#'   for each xe, an ye value is imposed. If a list item is NULL, no constraint
#'   is imposed on corresponding y column.
#' @param monotone Numeric scalar, if monotone > 0, resulting B-spline
#'   must be increasing;
#'   if monotone < 0, B-spline must be decreasing; if monotone == 0 (default), no
#'   constraint on monotonicity is imposed.
#' @return Function, smoothing B-splines
#'   respecting optional constraints (generated by \code{par2bsp()}).
#' @details
#'   If constraints are set, we use \code{nlsic::lsie_ln()} to solve a
#'   least squares
#'   problem with equality constraints in least norm sens for each y column.
#'   Otherwise, \code{nlsic::ls_ln_svd()} is used for the whole y matrix.
#'   The solution of least squares problem is a vector of B-splines coefficients \code{qw},
#'   one vector per \code{y} column. These vectors are used to define B-spline function
#'   which is returned as the result.\cr\cr
#'   NB. When \code{nki >= length(x)-n-1} (be it from direct setting or calculated
#'   from \code{length(xki)}), it corresponds
#'   to spline interpolation, i.e. the resulting spline will pass
#'   exactly by (x,y) points (well, up to numerical precision).
#' @seealso \code{bsppar} for retrieving parameters of B-spline functions; \code{par2bsp}
#'   for generating B-spline function; \code{iknots} for estimation of knot positions
#' @examples
#'   x=seq(0, 1, length.out=11)
#'   y=sin(pi*x)+rnorm(x, sd=0.1)
#'   # constraint B-spline to be 0 at the interval ends
#'   fsm=smbsp(x, y, nki=1, lieq=list(rbind(c(0, 0), c(1, 0))))
#'   # check parameters of found B-splines
#'   bsppar(fsm)
#'   plot(x, y) # original "measurements"
#'   # fine grained x
#'   xfine=seq(0, 1, length.out=101)
#'   lines(xfine, fsm(xfine)) # fitted B-splines
#'   lines(xfine, sin(pi*xfine), col="blue") # original function
#'   # visualize knot positions
#'   xk=bsppar(fsm)$xk
#'   points(xk, fsm(xk), pch="x", col="red")
#' @importFrom nlsic lsie_ln ls_ln_svd
#' @alias fitsmbsp
#' @alias smbsp
#' @export
smbsp=function(x, y, n=3L, xki=NULL, nki=1L, lieq=NULL, monotone=0) {
    y=as.matrix(y)
    nx=length(x)
    nc=ncol(y)
    stopifnot(nx == nrow(y))

    if (is.null(xki)) {
        # build default xki
        #dxk=diff(range(x))/(nki+1)
        #xki=x[1L]+seq_len(nki)*dxk
        xki=iknots(x, y, nki=nki, n=n)
    } else {
        nki=length(xki)
    }
    #stopifnot(nki > 0)
    if (nki > 0L) {
        stopifnot("xki is not inside of x range"=all((range(xki)-range(x))*c(1,-1) > 0))
        xk=c(rep(x[1L], n+1), xki, rep(x[nx], n+1))
    } else {
        xk=c(rep(x[1L], n+1), rep(x[nx], n+1))
    }
    
    nk=length(xk)
    nw=nk-n-1L
    a=bsc(x, xk, n=n)
    # equality constraints
    if (!is.null(lieq)) {
        stopifnot(length(lieq) == ncol(y))
        # todo: bezn() only on unique x from lieq
        liece=lapply(lieq, function(meq) {
            if (is.null(meq) || length(meq) == 0L) {
                NULL
            } else {
                list(e=bsc(meq[,1L], n=n, xk), ce=meq[,2L])
            }
        })
    }
    if (monotone != 0) {
        # B-spline must be monotonously in- or de-creasing
        i=seq_len(nw-1L)
        u=cbind(diag(x=-1., nrow=nw-1L), double(nw-1L))
        u[cbind(i, i+1)]=1.
        co=double(nw-1)
        if (monotone < 0)
            u=-u
    } else {
        u=NULL
        co=NULL
    }
    qra=qr(a, LAPACK=TRUE)
    if (!is.null(lieq) || monotone != 0) {
        #browser()
        qw=structure(vapply(seq_len(nc), function(ic) {nlsic::lsie_ln(qra, y[,ic], u=u, co=co, e=liece[[ic]]$e, ce=liece[[ic]]$ce)}, double(nw)), dim=c(nw, nc))
    } else {
        qw=nlsic::ls_ln_svd(a, y)
    }
    par2bsp(n, qw, xk)
}

#' Smoothing B-spline with optimized knot positions
#'
#' Optimize smoothing B-spline coefficients (smbsp) and knot positions (fitsmbsp)
#' such that residual squared sum is minimized for all y columns.
#' @details
#' Border and external knots are fixed, only strictly internal knots can move
#' during optimization. The optimization process is constrained to respect a minimal
#' distance between knots as well as to bound them to x range.
#' This is done to avoid knots getting unsorted during iterations and/or going
#' outside of a meaningful range.
#' @param control List, passed through to \code{nlsic()} call
#' @rdname smbsp
#' @alias fitsmbsp
#' @alias smbsp
#' @examples
#'  # fit broken line with linear B-splines
#'  x1=seq(0, 1, length.out=11)
#'  x2=seq(1, 3, length.out=21)
#'  x3=seq(3, 4, len=11)
#'  y1=x1+rnorm(x1, sd=0.1)
#'  y2=-2+3*x2+rnorm(x2, sd=0.1)
#'  y3=4+x3+rnorm(x3, sd=0.1)
#'  x=c(x1, x2[-1], x3[-1])
#'  y=c(y1, y2[-1], y3[-1])
#'  plot(x, y)
#'  if (requireNamespace("numDeriv", quietly=TRUE)) {
#'     f=fitsmbsp(x, y, n=1, nki=2)
#'     lines(x, f(x))
#'  }
#' @importFrom nlsic nlsic lsi_ln
#' @export
fitsmbsp=function(x, y, n=3L, xki=NULL, nki=1L, lieq=NULL, monotone=0, control=list()) {
    np=length(x)
    y=as.matrix(y)
    stopifnot(nrow(y) == np)
    if (is.null(xki)) {
        xki=iknots(x, y, n=n, nki=nki)
    }
    nki=length(xki)
    # inequalities
    # p[1] >= tail(x_1, 1)+epsx; p[i]+epsx <= p[i+1]; p[nki]+epsx <= x_n
    dx=diff(x)
    dx=dx[dx != 0.]
    stopifnot(dx > 0.)
    epsx=min(dx)*0.1
    u=matrix(0., nrow=nki+1L, ncol=nki)
    co=double(nrow(u))
    u[1L,1L]=1.; co[1L]=x[1L]+epsx
    i=seq_len(nki-1L)
    u[cbind(i+1L, i)]=-1.
    u[cbind(i+1L, i+1L)]=1.; co[i+1L]=epsx
    u[nki+1L, nki]=-1.; co[nki+1]=-x[np]-epsx
    fit=nlsic::nlsic(xki, function(xki, cjac)
        list(res=smbsp(x, y, n=n, xki=xki, nki=0, lieq, monotone)(x)-y),
        u=u, co=co, control=control, flsi=nlsic::lsi_ln)
    if (fit$error != 0)
        stop(fit$mes)
    smbsp(x, y, n=n, xki=fit$par, nki=0, lieq, monotone)
}

#' Derivative of B-spline
#'
#' @param f Function, B-spline such as returned by \code{smbsp()} or \code{par2bsp()}
#' @param nderiv Integer scalar >= 0, order of derivative to calculate (1 by default)
#' @return Function calculating requested derivative
#' @examples
#'   x=seq(0., 1., length.out=11L)
#'   y=sin(2*pi*x)
#'   f=smbsp(x, y, nki=2L)
#'   d_f=dbsp(f)
#'   xf=seq(0., 1., length.out=101) # fine grid for plotting
#'   plot(xf, d_f(xf)) # derivative estimated by B-splines
#'   lines(xf, 2.*pi*cos(2*pi*xf), col="blue") # true derivative
#'   xk=bsppar(d_f)$xk
#'   points(xk, d_f(xk), pch="x", col="red") # knot positions
#' @export
dbsp=function(f, nderiv=1L) {
    stopifnot(nderiv >= 0L)
    if (nderiv == 0L)
        return(f)
    e=environment(f)
    qw=e$qw
    xk=e$xk
    n=e$n
    stopifnot(n >= nderiv)
    xkn=xk[c(-1L, -length(xk))]
    qwn=n*arrApply::arrApply(qw, 1L, "diff")/diff(xkn, lag=n)
    dimnames(qwn)=dimnames(qw)
    #browser()
    res=par2bsp(n-1L, qwn, xkn)

    if (nderiv == 1L) {
        res
    } else {
        dbsp(res, nderiv=nderiv-1L)
    }
}

#' Retrieve parameters of B-splines
#'
#' @param f Function, B-splines such that returned by par3bsp(), smbsp(), ...
#' @return List having components: n - polynomial order, qw - coefficients, xk -
#'  knots
#' @export
bsppar=function(f) {
    as.list(environment(f))
}
#' Convert parameters to B-spline function
#'
#' @param n Integer scalar, polynomial order of B-splines
#' @param qw Numeric vector or matrix, coefficients of B-splines, one set per
#'  column in case of matrix
#' @param xk Numeric vector, knots
#' @return Function, calculating B-splines at arbitrary points and having
#'  interface \code{f(x, select)} where \code{x} is a vector of abscissa points.
#'  Parameter \code{select} is passed to
#'  \code{qw[, select, drop=FALSE]} and can be missing. This function will return
#'  a matrix of size \code{length(x) x ncol(qw)} if \code{select} is missing. Elsewhere,
#'  a number of column will depend on \code{select} parameter. Column names in
#'  the result matrix will be inherited from \code{qw}.
#' @export
par2bsp=function(n, qw, xk)
    local({
        n=n
        qw=qw
        xk=xk
        function(x, select) {
            if (base::missing(select)) bsp(x, xk, qw, n=n) else
            bsp(x, xk, qw[, select, drop=FALSE], n=n)
        }
    })
#' Finite differences
#'
#' Calculate dy/dx where x,y are first and second columns of the entry matrix 'm'
#' @param m 2- or more-column numeric matrix
#' @param ndiff Integer scalar, order of finite difference (1 by default)
#' @return Numeric matrix, first column is midpoints of x, the second
#'  and following are dy/dx
#' @importFrom arrApply arrApply
#' @export
diffn=function(m, ndiff=1L) {
    stopifnot(ndiff >= 0L)
    if (ndiff == 0L)
        return(m)
    stopifnot(nrow(m) > ndiff)
    nr=nrow(m)
    d=arrApply::arrApply(m, 1L, "diff")
    res=cbind(m[-nr,1L]+0.5*d[,1L], d[,-1L]/d[,1L])
    if (ndiff > 1L) {
        res=diffn(res, ndiff-1L)
    }
    colnames(res)=colnames(m)
    res
}
#' Estimate internal knot positions equalizing jumps in n-th derivative
#'
#' Normalized total variation of n-th finite differences is calculated for each column in
#' \code{y} then averaged. These averaged values are fitted by a linear spline to
#' find knot positions that equalize the jumps of n-th derivative.\cr
#' NB. This function is used internally in \code{(fit)smbsp()} and a priori
#' has no interest to be called directly by user.
#' @param x Numeric vector
#' @param y Numeric vector or matrix
#' @param nki Integer scalar, number of internal knots to estimate (1 by default)
#' @param n Integer scalar, polynomial order of B-spline (3 by default)
#' @return Numeric vector, estimated knot positions
#' @export
iknots=function(x, y, nki=1L, n=3L) {
    stopifnot(nki >= 0L)
    if (nki == 0L)
        return(double(0L))
    y=as.matrix(y)
    dxy=diffn(cbind(x, y), n+1L)
    dy=dxy[,-1L, drop=FALSE]
    xtv=diffn(cbind(x, x), n)[,1L]
    tv=rbind(0, arrApply::arrApply(abs(dy)*diff(x, lag=n+1), 1, "cumsum"))
    tv=arrApply::arrApply(tv, 2L, "multv", v=1./tv[nrow(tv),])
    tv[!is.finite(tv)]=0.
    tv=rowSums(tv)
    tv=tv/tv[length(tv)]
    ra=range(xtv)
    ftv=smbsp(xtv, tv, xki=xtv[1]+diff(ra)*seq(0, 1, len=12)[2:11], n=1L, lieq=list(cbind(ra,0:1)), monotone=1)
    par=bsppar(ftv)
    if (FALSE) {
        print(c("knots qw=", par$qw))
        print(c("knots dqw=", diff(par$qw)))
    }
    etv=seq(0, 1, length.out=nki+2L)[c(-1L, -(nki+2L))] # equalized tv
    qw=par$qw
    dq=diff(qw)
    dq[dq < 0]=0 # to force monotonicity despite round off errors
    qw=c(0., cumsum(dq))
    qw=qw/qw[length(qw)] # to make qw end up in 1
    ik=findInterval(etv, qw, rightmost.closed=TRUE)
    i=seq_along(etv)
    k=ik[i]
    x1=par$xk[k+1L]
    x2=par$xk[k+2L]
    y1=qw[k]
    y2=qw[k+1L]
    x1+(x2-x1)*(etv-y1)/(y2-y1)
}
