% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/families.R
\name{custom_family}
\alias{custom_family}
\alias{customfamily}
\title{Custom Families in \pkg{brms} Models}
\usage{
custom_family(
  name,
  dpars = "mu",
  links = "identity",
  type = c("real", "int"),
  lb = NA,
  ub = NA,
  vars = NULL,
  loop = TRUE,
  specials = NULL,
  threshold = "flexible",
  log_lik = NULL,
  posterior_predict = NULL,
  posterior_epred = NULL,
  predict = NULL,
  fitted = NULL,
  env = parent.frame()
)
}
\arguments{
\item{name}{Name of the custom family.}

\item{dpars}{Names of the distributional parameters of
the family. One parameter must be named \code{"mu"} and
the main formula of the model will correspond to that
parameter.}

\item{links}{Names of the link functions of the 
distributional parameters.}

\item{type}{Indicates if the response distribution is
continuous (\code{"real"}) or discrete (\code{"int"}). This controls
if the corresponding density function will be named with
\code{<name>_lpdf} or \code{<name>_lpmf}.}

\item{lb}{Vector of lower bounds of the distributional 
parameters. Defaults to \code{NA} that is no lower bound.}

\item{ub}{Vector of upper bounds of the distributional 
parameters. Defaults to \code{NA} that is no upper bound.}

\item{vars}{Names of variables that are part of the likelihood function
without being distributional parameters. That is, \code{vars} can be used
to pass data to the likelihood. Such arguments will be added to the list of
function arguments at the end, after the distributional parameters. See
\code{\link{stanvar}} for details about adding self-defined data to the
generated \pkg{Stan} model. Addition arguments \code{vreal} and \code{vint}
may be used for this purpose as well (see Examples below). See also
\code{\link{brmsformula}} and \code{\link{addition-terms}} for more
details.}

\item{loop}{Logical; Should the likelihood be evaluated via a loop
(\code{TRUE}; the default) over observations in Stan?
If \code{FALSE}, the Stan code will be written in a vectorized
manner over observations if possible.}

\item{specials}{A character vector of special options to enable
for this custom family. Currently for internal use only.}

\item{threshold}{Optional threshold type for custom ordinal families.
Ignored for non-ordinal families.}

\item{log_lik}{Optional function to compute log-likelihood values of
the model in \R. This is only relevant if one wants to ensure 
compatibility with method \code{\link[brms:log_lik.brmsfit]{log_lik}}.}

\item{posterior_predict}{Optional function to compute posterior prediction of
the model in \R. This is only relevant if one wants to ensure compatibility
with method \code{\link[brms:posterior_predict.brmsfit]{posterior_predict}}.}

\item{posterior_epred}{Optional function to compute expected values of the
posterior predictive distribution of the model in \R. This is only relevant
if one wants to ensure compatibility with method
\code{\link[brms:posterior_epred.brmsfit]{posterior_epred}}.}

\item{predict}{Deprecated alias of `posterior_predict`.}

\item{fitted}{Deprecated alias of `posterior_epred`.}

\item{env}{An \code{\link{environment}} in which certain post-processing 
functions related to the custom family can be found, if there were not 
directly passed to \code{custom_family}. This is only
relevant if one wants to ensure compatibility with the methods
\code{\link[brms:log_lik.brmsfit]{log_lik}},
\code{\link[brms:posterior_predict.brmsfit]{posterior_predict}}, or
\code{\link[brms:posterior_epred.brmsfit]{posterior_epred}}.
By default, \code{env} is the environment from which 
\code{custom_family} is called.}
}
\value{
An object of class \code{customfamily} inheriting
  from class \code{\link{brmsfamily}}.
}
\description{
Define custom families (i.e. response distribution) for use in 
\pkg{brms} models. It allows users to benefit from the modeling 
flexibility of \pkg{brms}, while applying their self-defined likelihood
functions. All of the post-processing methods for \code{brmsfit} 
objects can be made compatible with custom families. 
See \code{vignette("brms_customfamilies")} for more details.
For a list of built-in families see \code{\link{brmsfamily}}.
}
\details{
The corresponding probability density or mass \code{Stan} 
  functions need to have the same name as the custom family.
  That is if a family is called \code{myfamily}, then the 
  \pkg{Stan} functions should be called \code{myfamily_lpdf} or
  \code{myfamily_lpmf} depending on whether it defines a 
  continuous or discrete distribution.
}
\examples{
\dontrun{
## demonstrate how to fit a beta-binomial model
## generate some fake data
phi <- 0.7
n <- 300
z <- rnorm(n, sd = 0.2)
ntrials <- sample(1:10, n, replace = TRUE)
eta <- 1 + z
mu <- exp(eta) / (1 + exp(eta))
a <- mu * phi
b <- (1 - mu) * phi
p <- rbeta(n, a, b)
y <- rbinom(n, ntrials, p)
dat <- data.frame(y, z, ntrials)

# define a custom family
beta_binomial2 <- custom_family(
  "beta_binomial2", dpars = c("mu", "phi"),
  links = c("logit", "log"), lb = c(NA, 0),
  type = "int", vars = "vint1[n]"
)

# define the corresponding Stan density function
stan_density <- "
  real beta_binomial2_lpmf(int y, real mu, real phi, int N) {
    return beta_binomial_lpmf(y | N, mu * phi, (1 - mu) * phi);
  }
"
stanvars <- stanvar(scode = stan_density, block = "functions")

# fit the model
fit <- brm(y | vint(ntrials) ~ z, data = dat, 
           family = beta_binomial2, stanvars = stanvars)
summary(fit)


# define a *vectorized* custom family (no loop over observations)
# notice also that 'vint' no longer has an observation index
beta_binomial2_vec <- custom_family(
  "beta_binomial2", dpars = c("mu", "phi"),
  links = c("logit", "log"), lb = c(NA, 0),
  type = "int", vars = "vint1", loop = FALSE
)

# define the corresponding Stan density function
stan_density_vec <- "
  real beta_binomial2_lpmf(int[] y, vector mu, real phi, int[] N) {
    return beta_binomial_lpmf(y | N, mu * phi, (1 - mu) * phi);
  }
"
stanvars_vec <- stanvar(scode = stan_density_vec, block = "functions")

# fit the model
fit_vec <- brm(y | vint(ntrials) ~ z, data = dat, 
           family = beta_binomial2_vec, 
           stanvars = stanvars_vec)
summary(fit_vec)
}

}
\seealso{
\code{\link{brmsfamily}}, \code{\link{brmsformula}},
   \code{\link{stanvar}}
}
