% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_vp.R
\name{calculate_vp}
\alias{calculate_vp}
\title{Calculate a vertical profile (\code{vp}) from a polar volume (\code{pvol}) file}
\usage{
calculate_vp(
  file,
  vpfile = "",
  pvolfile_out = "",
  autoconf = FALSE,
  verbose = FALSE,
  warnings = TRUE,
  mount,
  sd_vvp_threshold,
  rcs = 11,
  dual_pol = TRUE,
  rho_hv = 0.95,
  single_pol = TRUE,
  elev_min = 0,
  elev_max = 90,
  azim_min = 0,
  azim_max = 360,
  range_min = 5000,
  range_max = 35000,
  n_layer = 20,
  h_layer = 200,
  dealias = TRUE,
  nyquist_min = if (dealias) 5 else 25,
  dbz_quantity = "DBZH",
  mistnet = FALSE,
  mistnet_elevations = c(0.5, 1.5, 2.5, 3.5, 4.5),
  local_install,
  local_mistnet
)
}
\arguments{
\item{file}{Character (vector). Either a path to a single radar polar volume
(\code{pvol}) file containing multiple scans/sweeps, or multiple paths to scan
files containing a single scan/sweep. Or a single \code{pvol} object. The file data format should be either 1)
\href{https://github.com/adokter/vol2bird/blob/master/doc/OPERA2014_O4_ODIM_H5-v2.2.pdf}{ODIM}
format, which is the implementation of the OPERA data information model in
the \href{https://support.hdfgroup.org/HDF5/}{HDF5} format, 2) a format
supported by the \href{https://trmm-fc.gsfc.nasa.gov/trmm_gv/software/rsl/}{RSL library} or 3) Vaisala
IRIS (IRIS RAW) format.}

\item{vpfile}{Character. File name. When provided, writes a vertical profile
file (\code{vpfile}) either in the VPTS CSV or ODIM HDF5 format to disk.}

\item{pvolfile_out}{Character. File name. When provided, writes a polar
volume (\code{pvol}) file in the ODIM HDF5 format to disk. Useful for converting
RSL formats to ODIM.}

\item{autoconf}{Logical. When \code{TRUE}, default optimal configuration settings
are selected automatically and other user settings are ignored.}

\item{verbose}{Logical. When \code{TRUE}, vol2bird \code{stdout} is piped to the R
console.}

\item{warnings}{Logical. When \code{TRUE}, vol2bird warnings are piped to the R
console.}

\item{mount}{Character. Directory path of the mount point for the Docker
container (deprecated).}

\item{sd_vvp_threshold}{Numeric. Lower threshold for the radial velocity
standard deviation (profile quantity \code{sd_vvp}) in m/s. Biological signals
with \code{sd_vvp < sd_vvp_threshold} are set to zero. Defaults to 2 m/s for
C-band radars and 1 m/s for S-band radars.}

\item{rcs}{Numeric. Radar cross section per bird to use, in cm^2.}

\item{dual_pol}{Logical. When \code{TRUE}, uses dual-pol mode, in which
meteorological echoes are filtered using the correlation coefficient
threshold \code{rho_hv}.}

\item{rho_hv}{Numeric. Lower threshold in correlation coefficient to use for
filtering meteorological scattering.}

\item{single_pol}{Logical. When \code{TRUE}, uses precipitation filtering in single
polarization mode based on reflectivity and radial velocity quantities.}

\item{elev_min}{Numeric. Minimum elevation angle to include, in degrees.}

\item{elev_max}{Numeric. Maximum elevation angle to include, in degrees.}

\item{azim_min}{Numeric. Minimum azimuth to include, in degrees clockwise
from north.}

\item{azim_max}{Numeric. Maximum azimuth to include, in degrees clockwise
from north.}

\item{range_min}{Numeric. Minimum range to include, in m.}

\item{range_max}{Numeric. Maximum range to include, in m.}

\item{n_layer}{Numeric. Number of altitude layers to use in generated
profile.}

\item{h_layer}{Numeric. Width of altitude layers to use in generated profile,
in m.}

\item{dealias}{Logical. Whether to dealias radial velocities. This should
typically be done when the scans in the polar volume have low Nyquist
velocities (below 25 m/s).}

\item{nyquist_min}{Numeric. Minimum Nyquist velocity of scans to include, in
m/s.}

\item{dbz_quantity}{Name of the available reflectivity factor to use if not
\code{DBZH} (e.g. \code{DBZV}, \code{TH}, \code{TV}).}

\item{mistnet}{Logical. Whether to use the MistNet segmentation model.}

\item{mistnet_elevations}{Numeric vector of length 5. Elevation angles to
feed to the MistNet segmentation model, which expects exactly 5 elevation
scans at 0.5, 1.5, 2.5, 3.5 and 4.5 degrees. Specifying different elevation
angles may compromise segmentation results.}

\item{local_install}{Character. Path to local vol2bird installation (e.g.
\code{your/vol2bird_install_directory/vol2bird/bin/vol2bird.sh}). (deprecated)}

\item{local_mistnet}{Character. Path to local MistNet segmentation model in
PyTorch format (e.g. \verb{/your/path/mistnet_nexrad.pt}).}
}
\value{
A vertical profile object of class \code{vp}. When defined, output files
\code{vpfile} and \code{pvolfile_out} are saved to disk.
}
\description{
Calculates a vertical profile of biological scatterers (\code{vp}) from a polar
volume (\code{pvol}) file using the algorithm
\href{https://github.com/adokter/vol2bird/}{vol2bird} (Dokter et al.
2011 \doi{10.1098/rsif.2010.0116}).
}
\details{
\subsection{Typical use}{

Common arguments set by users are \code{file}, \code{vpfile} and \code{autoconf}.
Turn on \code{autoconf} to automatically select the optimal parameters for a given
radar file. The default for C-band data is to apply rain-filtering in single
polarization mode and dual polarization mode when available. The default for
S-band data is to apply precipitation filtering in dual-polarization mode
only.

Arguments that sometimes require non-default values are: \code{rcs},
\code{sd_vvp_threshold}, \code{range_max}, \code{dual_pol}, \code{dealias}. Other arguments are
typically left at their defaults.
}

\subsection{sd_vvp_threshold}{

For altitude layers with a VVP-retrieved radial velocity standard deviation
value below the threshold \code{sd_vvp_threshold}, the bird density \code{dens} is set
to zero (see vertical profile \code{\link[=summary.vp]{vp}} class). This threshold
might be dependent on radar processing settings. Results from validation
campaigns so far indicate that 2 m/s is the best choice for this parameter
for most C-band weather radars, which is used as the C-band default. For
S-band, the default threshold is 1 m/s.
}

\subsection{rcs}{

The default radar cross section (\code{rcs}) (11 cm^2) corresponds to the average
value found by Dokter et al. (2011) in a calibration campaign of a full
migration autumn season in western Europe at C-band. Its value may depend on
radar wavelength. \code{rcs} will scale approximately \eqn{M^{2/3}} with \code{M} the
bird's mass.
}

\subsection{dual_pol}{

For S-band (radar wavelength ~ 10 cm), currently only \code{dual_pol = TRUE} mode
is recommended.
}

\subsection{azim_min / azim_max}{

\code{azim_min} and \code{azim_max} only affects reflectivity-derived estimates in the
profile (\code{DBZH}, \code{eta}, \code{dens}), not radial-velocity derived estimates (\code{u},
\code{v}, \code{w}, \code{ff}, \code{dd}, \code{sd_vvp}), which are estimated on all azimuths at all
times. \code{azim_min}, \code{azim_max} may be set to exclude an angular sector with
high ground clutter.
}

\subsection{range_min / range_max}{

Using default values of \code{range_min} and \code{range_max} is recommended. Ranges
closer than 5 km tend to be contaminated by ground clutter, while range gates
beyond 35 km become too wide to resolve the default altitude layer width of
200 meter (see \code{\link[=beam_width]{beam_width()}}). \code{range_max} may be extended up to 40 km
(\code{40000}) for volumes with low elevations only, in order to extend coverage
to higher altitudes.
}

\subsection{h_layer}{

The algorithm has been tested and developed for altitude layers with \code{h_layer = 200}m. Smaller widths than 100 m are not recommended as they may cause
instabilities of the volume velocity profiling (VVP) and dealiasing routines,
and effectively lead to pseudo-replicated altitude data, since altitudinal
patterns smaller than the beam width cannot be resolved.
}

\subsection{dealias}{

Dealiasing uses the torus mapping method by Haase et al. (2004).
}

\subsection{Local installation}{

You may point parameter \code{local_mistnet} to a local download of the MistNet segmentation model in
PyTorch format, e.g. \verb{/your/path/mistnet_nexrad.pt}. The MistNet model can
be downloaded at \url{https://s3.amazonaws.com/mistnet/mistnet_nexrad.pt}.
}
}
\examples{
# Locate and read the polar volume example file
pvolfile_source <- system.file("extdata", "volume.h5", package = "bioRad")

# Calculate the profile
if (requireNamespace("vol2birdR", quietly = TRUE)) {
vp <- calculate_vp(pvolfile_source)

# Get summary info
vp
}

}
\references{
Dokter et al. (2011) is the main reference for the profiling algorithm
(vol2bird) underlying this function. When using the \code{mistnet} option, please
also cite Lin et al. (2019). When dealiasing data (\code{dealias}), please also
cite Haase et al. (2004).
\itemize{
\item Dokter AM, Liechti F, Stark H, Delobbe L,Tabary P, Holleman I (2011) Bird
migration flight altitudes studied by a network of operational weather
radars, Journal of the Royal Society Interface 8 (54), pp. 30-43.
\doi{10.1098/rsif.2010.0116}
\item Haase G & Landelius T (2004)
Dealiasing of Doppler radar velocities using a torus mapping. Journal of
Atmospheric and Oceanic Technology 21(10), pp. 1566-1573.
\doi{10.1175/1520-0426(2004)021<1566:DODRVU>2.0.CO;2}
\item Lin T-Y, Winner K, Bernstein G, Mittal A, Dokter AM, Horton KG, Nilsson C,
Van Doren BM, Farnsworth A, La Sorte FA, Maji S, Sheldon D (2019) MistNet:
Measuring historical bird migration in the US using archived weather radar
data and convolutional neural networks. Methods in Ecology and Evolution 10
(11), pp. 1908-22. \doi{10.1111/2041-210X.13280}
}
}
\seealso{
\itemize{
\item \code{\link[=summary.pvol]{summary.pvol()}}
\item \code{\link[=summary.vp]{summary.vp()}}
}
}
