# /*
#  *  bigmemory: an R package for managing massive matrices using C,
#  *  with support for shared memory.
#  *
#  *  Copyright (C) 2009 John W. Emerson and Michael J. Kane
#  *
#  *  This file is part of bigmemory.
#  *
#  *  bigmemory is free software; you can redistribute it and/or modify
#  *  it under the terms of the GNU Lesser General Public License as published
#  *  by the Free Software Foundation; either version 3 of the License, or
#  *  (at your option) any later version.
#  *
#  *  This program is distributed in the hope that it will be useful,
#  *  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  *  GNU Lesser General Public License for more details.
#  *
#  *  You should have received a copy of the GNU Lesser General Public License
#  *  along with this program; if not, a copy is available at
#  *  http://www.r-project.org/Licenses/
#  */


setClass("big.matrix", representation(address='externalptr'))

big.matrix <- function(nrow, ncol, type='integer', init=NULL, dimnames=NULL,
                       separated=FALSE, shared=FALSE, backingfile=NULL, 
                       backingpath=NULL, descriptorfile=NULL, preserve=TRUE,
                       nebytes=0)
{
  if (nrow < 1 | ncol < 1)
    stop('A big.matrix instance must have at least one row and one column')
  
  typeVal=NULL
  if (type == 'integer') typeVal <- 4
  if (type == 'double') typeVal <- 8
  if (type == 'short') typeVal <- 2
  if (type == 'char') typeVal <- 1
  if (is.null(typeVal)) stop('invalid type')
  address <- NULL
  if (is.null(backingfile) & !shared)
  {
    address <- .Call('CCreateMatrix', as.double(nrow), as.double(ncol),
                     as.double(init), as.integer(typeVal),
                     as.logical(separated), as.double(nebytes))
    if (is.null(address))
      stop(paste("Error: Memory could not be allocated for instance of",
		             "type big.matrix", sep=' '))
    x <- new("big.matrix", address=address)
    if (is.null(x))
      stop("Error encountered when creating instance of type big.matrix")
    dimnames(x) <- dimnames
    return(x)
  }
  else
  {
    return(shared.big.matrix(nrow=nrow, ncol=ncol, type=type, init=init, 
           dimnames=dimnames, separated=separated, backingfile=backingfile, 
           backingpath=backingpath, descriptorfile=descriptorfile, 
           preserve=preserve, nebytes=nebytes))
  }
}

is.big.matrix <- function(x) return(class(x) == "big.matrix")

as.big.matrix <- function(x, type=NULL, separated=FALSE, shared=FALSE,
                          backingfile=NULL, backingpath=NULL,
                          descriptorfile=NULL, preserve=TRUE)
{
  if (is.vector(x)) {
    x <- matrix(x, length(x), 1)
    warning("Coercing vector to a single-column matrix.")
  }
  if (!is.matrix(x)) 
    stop('argument is not a matrix; perhaps it is a data frame?')
  if (!is.numeric(x)) {
    warning("Casting matrix to numeric type")
    x <- matrix( as.numeric(x), nrow=nrow(x), dimnames=dimnames(x) )
  }
  if (is.null(type)) type <- typeof(x)
  
  if (type=="integer" | type=="double" | type=="short" | type=="char") 
  {
    if (shared | !is.null(backingfile)) {
      y <- shared.big.matrix(nrow=nrow(x), ncol=ncol(x), type=type, init=NULL, 
                             dimnames=dimnames(x), separated=separated,
                             backingfile=backingfile, backingpath=backingpath,
                             descriptorfile=descriptorfile, preserve=preserve)
    } else {
      y <- big.matrix(nrow=nrow(x), ncol=ncol(x), type=type, init=NULL, 
                      dimnames=dimnames(x), separated=separated)
    }
    y[1:nrow(x),1:ncol(x)] <- x
    junk <- gc() 
  } else stop('bigmemory: that type is not implemented.')
  return(y)
}

colnames.bm <- function(x)
{
  ret <- .Call("GetColumnNamesBM", x@address)
  if (length(ret)==0) return(NULL)
  return(ret)
}

rownames.bm <- function(x)
{
  ret <- .Call("GetRowNamesBM", x@address)
  if (length(ret)==0) return(NULL)
  return(ret)
}

assign('colnames.bm<-', 
  function(x, value) {
    if (!is.shared(x)) {
      if (is.character(value)) {
        if (any(value=="")) {
          value <- NULL
          warning("empty strings prohibited in column names")
        }
      } else {
        if (!is.null(value)) {
          value <- as.character(value)
          warning("column names coerced to character")
        }
      }
      if (!is.null(value) & length(value) != ncol(x))
        stop("length of 'colnames' not equal to array extent.")
      .Call("SetColumnNames", x@address, value)
      return(x)
    } else stop('changing column names of a shared object is prohibited.')
  })

assign('rownames.bm<-',
  function(x,value) {
    if (!is.shared(x)) {
      if (is.character(value)) {
        if (any(value=="")) {
          value = NULL
          warning("empty strings prohibited in row names")
        }
      } else {
        if (!is.null(value)) {
          value <- as.character(value)
          warning("row names coerced to character")
        }
      }
      if (length(value) != nrow(x) & !is.null(value)) 
        stop("length of 'rownames' not equal to array extent.")
      .Call("SetRowNames", x@address, value)
      return(x)
    } else stop('changing row names of a shared object is prohibited.')
  })

setMethod('ncol', signature(x="big.matrix"),
  function(x) {
    return(.Call("CGetNcol", x@address))
  })

setMethod('nrow', signature(x="big.matrix"), 
  function(x) {
    return(.Call("CGetNrow", x@address))
  })

setMethod('dim', signature(x="big.matrix"),
  function(x) return(c(nrow(x), ncol(x))))

GetElements.bm <- function(x, i, j, drop=TRUE)
{
  if (!is.numeric(i) & !is.character(i) & !is.logical(i))
    stop("row indices must be numeric, logical, or character vectors.")
  if (!is.numeric(j) & !is.character(j) & !is.logical(j))
    stop("column indices must be numeric, logical, or character vectors.")
  if (is.character(i))
    if (is.null(rownames(x))) stop("row names do not exist.")
    else i <- mmap(i, rownames(x))
  if (is.character(j))
    if (is.null(colnames(x))) stop("column names do not exist.")
    else j <- mmap(j, colnames(x))
  if (is.logical(i)) {
    if (length(i) != nrow(x))
      stop("row vector length must match the number of rows of the matrix.")
    i <- which(i)
  }
  if (is.logical(j)) {
    if (length(j) != ncol(x))
      stop(paste("column vector length must match the number of",
                 "columns of the matrix."))
    j <- which(j)
  }

  tempi <- .Call("CCleanIndices", as.double(i), as.double(nrow(x)))
  if (is.null(tempi[[1]])) stop("Illegal row index usage in extraction.\n")
  if (tempi[[1]]) i <- tempi[[2]]
  tempj <- .Call("CCleanIndices", as.double(j), as.double(ncol(x)))
  if (is.null(tempj[[1]])) stop("Illegal column index usage in extraction.\n")
  if (tempj[[1]]) j <- tempj[[2]]

  if (is.shared(x) && options()$rlock.enabled) lockcols(x, j, 'r')
  retList <- .Call("GetMatrixElements", x@address, as.double(j), as.double(i))
  if (is.shared(x) && options()$rlock.enabled) unlockcols(x,j)

  dimnames(retList[[1]]) <- list( retList[[2]], retList[[3]] )
  if (drop) {
    if (any(dim(retList[[1]])==1)) {
      if (dim(retList[[1]])[1]!=1 || dim(retList[[1]])[2]!=1) {
        if (dim(retList[[1]])[1]==1) {
          thesenames <- retList[[3]]
        } else thesenames <- retList[[2]]
      } else thesenames <- NULL
      retList[[1]] = as.vector(retList[[1]])
      names(retList[[1]]) <- thesenames
    }
  }
  return(retList[[1]])
}

GetCols.bm <- function(x, j, drop=TRUE)
{
  if (!is.numeric(j) & !is.character(j) & !is.logical(j))
    stop("column indices must be numeric, logical, or character vectors.")
  if (is.character(j))
    if (is.null(colnames(x))) stop("column names do not exist.")
    else j <- mmap(j, colnames(x))
  if (is.logical(j)) {
    if (length(j) != ncol(x))
      stop(paste("column vector length must match the number of",
                 "columns of the matrix."))
    j <- which(j)
  }

  tempj <- .Call("CCleanIndices", as.double(j), as.double(ncol(x)))
  if (is.null(tempj[[1]])) stop("Illegal column index usage in extraction.\n")
  if (tempj[[1]]) j <- tempj[[2]]

  if (is.shared(x) && options()$rlock.enabled) lockcols(x, j, 'r')
  retList <- .Call("GetMatrixCols", x@address, as.double(j))
  if (is.shared(x) && options()$rlock.enabled) unlockcols(x,j)

  dimnames(retList[[1]]) <- list( retList[[2]], retList[[3]] )
  if (drop) {
    if (any(dim(retList[[1]])==1)) {
      if (dim(retList[[1]])[1]!=1 || dim(retList[[1]])[2]!=1) {
        if (dim(retList[[1]])[1]==1) {
          thesenames <- retList[[3]]
        } else thesenames <- retList[[2]]
      } else thesenames <- NULL
      retList[[1]] = as.vector(retList[[1]])
      names(retList[[1]]) <- thesenames
    }
  }
  return(retList[[1]])
}

GetRows.bm <- function(x, i, drop=TRUE)
{
  if (!is.numeric(i) & !is.character(i) & !is.logical(i))
    stop("row indices must be numeric, logical, or character vectors.")
  if (is.character(i))
    if (is.null(rownames(x))) stop("row names do not exist.")
    else i <- mmap(i, rownames(x))
  if (is.logical(i)) {
    if (length(i) != nrow(x))
      stop("row vector length must match the number of rows of the matrix.")
    i <- which(i)
  }

  tempi <- .Call("CCleanIndices", as.double(i), as.double(nrow(x)))
  if (is.null(tempi[[1]])) stop("Illegal row index usage in extraction.\n")
  if (tempi[[1]]) i <- tempi[[2]]

  if (is.shared(x) && options()$rlock.enabled) lockcols(x, 1:ncol(x), 'r')
  retList <- .Call("GetMatrixRows", x@address, as.double(i))
  if (is.shared(x) && options()$rlock.enabled) unlockcols(x, 1:ncol(x))

  dimnames(retList[[1]]) <- list( retList[[2]], retList[[3]] )
  if (drop) {
    if (any(dim(retList[[1]])==1)) {
      if (dim(retList[[1]])[1]!=1 || dim(retList[[1]])[2]!=1) {
        if (dim(retList[[1]])[1]==1) {
          thesenames <- retList[[3]]
        } else thesenames <- retList[[2]]
      } else thesenames <- NULL
      retList[[1]] = as.vector(retList[[1]])
      names(retList[[1]]) <- thesenames
    }
  }
  return(retList[[1]])
}

GetAll.bm <- function(x, drop=TRUE)
{
  # Note here the locks are handled in the signature, because there
  # is no index cleaning to be done.

  retList <- .Call("GetMatrixAll", x@address)

  dimnames(retList[[1]]) <- list( retList[[2]], retList[[3]] )
  if (drop) {
    if (any(dim(retList[[1]])==1)) {
      if (dim(retList[[1]])[1]!=1 || dim(retList[[1]])[2]!=1) {
        if (dim(retList[[1]])[1]==1) {
          thesenames <- retList[[3]]
        } else thesenames <- retList[[2]]
      } else thesenames <- NULL
      retList[[1]] = as.vector(retList[[1]])
      names(retList[[1]]) <- thesenames
    }
  }
  return(retList[[1]])
}

setMethod("[",
  signature(x = "big.matrix", drop = "missing"),
  function(x, i, j) {
    return(GetElements.bm(x, i, j))
  })

setMethod("[",
  signature(x = "big.matrix", drop = "logical"),
  function(x, i, j, drop) {
    return(GetElements.bm(x, i, j, drop))
  })

setMethod("[",
  signature(x = "big.matrix", i="missing", drop = "missing"),
  function(x, j) {
    return(GetCols.bm(x, j))
  })

setMethod("[",
  signature(x = "big.matrix", i="missing", drop = "logical"),
  function(x, j, drop) {
    return(GetCols.bm(x, j, drop))
  })

setMethod("[",
  signature(x = "big.matrix", j="missing", drop = "missing"),
  function(x, i) {
    return(GetRows.bm(x, i))
  })

setMethod("[",
  signature(x = "big.matrix", j="missing", drop = "logical"),
  function(x, i, drop) {
    return(GetRows.bm(x, i, drop))
  })

# Because we don't have any index checking/fixing, we do our locking here.
setMethod("[",
  signature(x = "big.matrix", i="missing", j="missing", drop = "missing"),
  function(x) {
    if (is.shared(x) && options()$rlock.enabled) lockcols(x,1:ncol(x),'r')
    ret <- GetAll.bm(x)
    if (is.shared(x) && options()$rlock.enabled) unlockcols(x,1:ncol(x))
    return(ret)
})

# Because we don't have any index checking/fixing, we do our locking here.
setMethod("[",
  signature(x = "big.matrix", i="missing", j="missing", drop = "logical"),
  function(x, drop) {
    if (is.shared(x) && options()$rlock.enabled) lockcols(x,1:ncol(x),'r')
    ret <- GetAll.bm(x, drop)
    if (is.shared(x) && options()$rlock.enabled) unlockcols(x,1:ncol(x))
    return(ret)
})

SetElements.bm <- function(x, i, j, value)
{
  if (!is.numeric(i) & !is.character(i) & !is.logical(i))
    stop("row indices must be numeric, logical, or character vectors.")
  if (!is.numeric(j) & !is.character(j) & !is.logical(j))
    stop("column indices must be numeric, logical, or character vectors.")
  if (is.character(i))
    if (is.null(rownames(x))) stop("row names do not exist.")
    else i <- mmap(i, rownames(x))
  if (is.character(j))
    if (is.null(colnames(x))) stop("column names do not exist.")
    else j <- mmap(j, colnames(x))
  if (is.logical(i)) {
    if (length(i) != nrow(x))
      stop("row vector length must match the number of rows of the matrix.")
    i <- which(i)
  }
  if (is.logical(j)) {
    if (length(j) != ncol(x))
      stop(paste("column vector length must match the number of",
                 "columns of the matrix."))
    j <- which(j)
  }

  tempi <- .Call("CCleanIndices", as.double(i), as.double(nrow(x)))
  if (is.null(tempi[[1]])) stop("Illegal row index usage in extraction.\n")
  if (tempi[[1]]) i <- tempi[[2]]
  tempj <- .Call("CCleanIndices", as.double(j), as.double(ncol(x)))
  if (is.null(tempj[[1]])) stop("Illegal column index usage in extraction.\n")
  if (tempj[[1]]) j <- tempj[[2]]

  if ( options()$bigmemory.typecast.warning &&
       (((typeof(value) == "double") && (typeof(x) != "double") ||
       (typeof(value) == "integer" &&
        (typeof(x) != "double" && typeof(x) != "integer")))) )
    warning(cat("Assignment will down cast from ", typeof(value), " to ",
                typeof(x), "\nHint: To remove this warning type:  ",
                "options(bigmemory.typecast.warning=FALSE)\n", sep=''))

  # Note: i may be a mwhich statement in which case we _must_ ensure
  # that we disable read locking before it is evaluated or we will
  # have a race condition.  - Jay and Mike.
  if (is.shared(x)) {
    options(rlock.enabled=FALSE)
    lockcols(x, j, 'w')
  }

  totelts <- length(i) * length(j)
  # If we are assigning from a matrix, make sure the dimensions agree.
  if (is.matrix(value)){
    if (ncol(value) != length(j) | nrow(value) != length(i)) {
      if (is.shared(x)) {
        options(rlock.enabled=TRUE)
        unlockcols(x, j)
      }
      stop("Matrix dimensions do not agree with big.matrix instance set size.")
    }
  } else if (length(value) != totelts) {
    # Otherwise, make sure we are assigning the correct number of things
    # (rep if necessary)
    numReps <- totelts / length(value)
    if (numReps != round(numReps)) {
      if (is.shared(x)) {
        options(rlock.enabled=TRUE)
        unlockcols(x, j)
      }
      stop("number of items to replace is not a multiple of replacement length")
    }
  }
  if (typeof(x) != 'double') {
    integerVals = na.omit(as.integer(value))
    if ( sum(integerVals == na.omit(as.integer(value))) !=
         length(integerVals) | is.factor(value)) {
      warning("non-integer (possibly Inf or -Inf) typecast to integer")
    }
  }
  # Note: we pass doubles as doubles, but anything else as integers.
  if (typeof(x) == 'double') {
    .Call("SetMatrixElements", x@address, as.double(j), as.double(i), 
			as.double(value))
  } else {
    .Call("SetMatrixElements", x@address, as.double(j), as.double(i), 
			as.integer(value))
  }
  if (is.shared(x)) {
    unlockcols(x,j)
    options(rlock.enabled=TRUE)
  }
  return(x)
}

SetCols.bm <- function(x, j, value)
{
  if (!is.numeric(j) & !is.character(j) & !is.logical(j))
    stop("column indices must be numeric, logical, or character vectors.")
  if (is.character(j))
    if (is.null(colnames(x))) stop("column names do not exist.")
    else j <- mmap(j, colnames(x))
  if (is.logical(j)) {
    if (length(j) != ncol(x))
      stop(paste("column vector length must match the number of",
                 "columns of the matrix."))
    j <- which(j)
  }

  tempj <- .Call("CCleanIndices", as.double(j), as.double(ncol(x)))
  if (is.null(tempj[[1]])) stop("Illegal column index usage in extraction.\n")
  if (tempj[[1]]) j <- tempj[[2]]

  if ( options()$bigmemory.typecast.warning &&
       ((typeof(value) == "double") && (typeof(x) != "double") ||
       (typeof(value) == "integer" &&
        (typeof(x) != "double" && typeof(x) != "integer"))) )
    warning(cat("Assignment will down cast from ", typeof(value), " to ",
                typeof(x), "\nHint: To remove this warning type:  ",
                "options(bigmemory.typecast.warning=FALSE)\n", sep=''))

  # Note: i may be a mwhich statement in which case we _must_ ensure
  # that we disable read locking before it is evaluated or we will
  # have a race condition.  - Jay and Mike.
  if (is.shared(x)) {
    options(rlock.enabled=FALSE)
    lockcols(x, j, 'w')
  }

  totelts <- nrow(x) * length(j)
  # If we are assigning from a matrix, make sure the dimensions agree.
  if (is.matrix(value)){
    if (ncol(value) != length(j) | nrow(value) != nrow(x)) {
      if (is.shared(x)) {
        options(rlock.enabled=TRUE)
        unlockcols(x, j)
      }
      stop("Matrix dimensions do not agree with big.matrix instance set size.")
    }
  } else if (length(value) != totelts) {
    # Otherwise, make sure we are assigning the correct number of things
    # (rep if necessary)
    numReps <- totelts / length(value)
    if (numReps != round(numReps)) {
      if (is.shared(x)) {
        options(rlock.enabled=TRUE)
        unlockcols(x, j)
      }
      stop("number of items to replace is not a multiple of replacement length")
    }
  }
  if (typeof(x) != 'double') {
    integerVals = na.omit(as.integer(value))
    if ( sum(integerVals == na.omit(as.integer(value))) !=
         length(integerVals) | is.factor(value)) {
      warning("non-integer (possibly Inf or -Inf) typecast to integer")
    }
  }
  # Note: we pass doubles as doubles, but anything else as integers.
  if (typeof(x) == 'double') {
    .Call("SetMatrixCols", x@address, as.double(j), as.double(value))
  } else {
    .Call("SetMatrixCols", x@address, as.double(j), as.integer(value))
  }
  if (is.shared(x)) {
    unlockcols(x,j)
    options(rlock.enabled=TRUE)
  }
  return(x)
}

SetRows.bm <- function(x, i, value) 
{
  if (!is.numeric(i) & !is.character(i) & !is.logical(i))
    stop("row indices must be numeric, logical, or character vectors.")
  if (is.character(i))
    if (is.null(rownames(x))) stop("row names do not exist.")
    else i <- mmap(i, rownames(x))
  if (is.logical(i)) {
    if (length(i) != nrow(x))
      stop("row vector length must match the number of rows of the matrix.")
    i <- which(i)
  }

  tempi <- .Call("CCleanIndices", as.double(i), as.double(nrow(x)))
  if (is.null(tempi[[1]])) stop("Illegal row index usage in extraction.\n")
  if (tempi[[1]]) i <- tempi[[2]]

  if ( options()$bigmemory.typecast.warning &&
       ((typeof(value) == "double") && (typeof(x) != "double") ||
       (typeof(value) == "integer" &&
        (typeof(x) != "double" && typeof(x) != "integer"))) )
    warning(cat("Assignment will down cast from ", typeof(value), " to ",
                typeof(x), "\nHint: To remove this warning type:  ",
                "options(bigmemory.typecast.warning=FALSE)\n", sep=''))

  # Note: i may be a mwhich statement in which case we _must_ ensure
  # that we disable read locking before it is evaluated or we will
  # have a race condition.  - Jay and Mike.
  if (is.shared(x)) {
    options(rlock.enabled=FALSE)
    lockcols(x, 1:ncol(x), 'w')
  }

  totelts <- length(i) * ncol(x)
  # If we are assigning from a matrix, make sure the dimensions agree.
  if (is.matrix(value)){
    if (ncol(value) != ncol(x) | nrow(value) != length(i)) {
      if (is.shared(x)) {
        options(rlock.enabled=TRUE)
        unlockcols(x, 1:ncol(x))
      }
      stop("Matrix dimensions do not agree with big.matrix instance set size.")
    }
  } else if (length(value) != totelts) {
    # Otherwise, make sure we are assigning the correct number of things
    # (rep if necessary)
    numReps <- totelts / length(value)
    if (numReps != round(numReps)) {
      if (is.shared(x)) {
        options(rlock.enabled=TRUE)
        unlockcols(x, 1:ncol(x))
      }
      stop("number of items to replace is not a multiple of replacement length")
    }
  }
  if (typeof(x) != 'double') {
    integerVals = na.omit(as.integer(value))
    if ( sum(integerVals == na.omit(as.integer(value))) !=
         length(integerVals) | is.factor(value)) {
      warning("non-integer (possibly Inf or -Inf) typecast to integer")
    }
  }
  # Note: we pass doubles as doubles, but anything else as integers.
  if (typeof(x) == 'double') {
    .Call("SetMatrixRows", x@address, as.double(i), as.double(value))
  } else {
    .Call("SetMatrixRows", x@address, as.double(i), as.integer(value))
  }
  if (is.shared(x)) {
    unlockcols(x,1:ncol(x))
    options(rlock.enabled=TRUE)
  }
  return(x)
}

SetAll.bm <- function(x, value) 
{
  if ( options()$bigmemory.typecast.warning &&
       ((typeof(value) == "double") && (typeof(x) != "double") ||
       (typeof(value) == "integer" &&
        (typeof(x) != "double" && typeof(x) != "integer"))) )
    warning(cat("Assignment will down cast from ", typeof(value), " to ",
                typeof(x), "\nHint: To remove this warning type:  ",
                "options(bigmemory.typecast.warning=FALSE)\n", sep=''))

  # Note: i may be a mwhich statement in which case we _must_ ensure
  # that we disable read locking before it is evaluated or we will
  # have a race condition.  - Jay and Mike.
  if (is.shared(x)) {
    options(rlock.enabled=FALSE)
    lockcols(x, 1:ncol(x), 'w')
  }

  totelts <- nrow(x) * ncol(x)
  # If we are assigning from a matrix, make sure the dimensions agree.
  if (is.matrix(value)){
    if (ncol(value) != ncol(x) | nrow(value) != nrow(x)) {
      if (is.shared(x)) {
        options(rlock.enabled=TRUE)
        unlockcols(x, 1:ncol(x))
      }
      stop("Matrix dimensions do not agree with big.matrix instance set size.")
    }
  } else if (length(value) != totelts) {
    # Otherwise, make sure we are assigning the correct number of things
    # (rep if necessary)
    numReps <- totelts / length(value)
    if (numReps != round(numReps)) {
      if (is.shared(x)) {
        options(rlock.enabled=TRUE)
        unlockcols(x, 1:ncol(x))
      }
      stop("number of items to replace is not a multiple of replacement length")
    }
  }
  if (typeof(x) != 'double') {
    integerVals = na.omit(as.integer(value))
    if ( sum(integerVals == na.omit(as.integer(value))) !=
         length(integerVals) | is.factor(value)) {
      warning("non-integer (possibly Inf or -Inf) typecast to integer")
    }
  }
  # Note: we pass doubles as doubles, but anything else as integers.
  if (typeof(x) == 'double') {
    .Call("SetMatrixAll", x@address, as.double(value))
  } else {
    .Call("SetMatrixAll", x@address, as.integer(value))
  }
  if (is.shared(x)) {
    unlockcols(x,1:ncol(x))
    options(rlock.enabled=TRUE)
  }
  return(x)
}

setMethod('[<-',
  signature(x = "big.matrix"),
  function(x, i, j, value) {
    return(SetElements.bm(x, i, j, value))
  })

setMethod('[<-',
  signature(x = "big.matrix", i="missing"),
  function(x, j, value) {
    return(SetCols.bm(x, j, value))
  })

setMethod('[<-',
  signature(x = "big.matrix", j="missing"),
  function(x, i, value) {
    return(SetRows.bm(x, i, value))
  })

setMethod('[<-',
  signature(x = "big.matrix", i="missing", j="missing"),
  function(x, value) {
    return(SetAll.bm(x, value))
  })

setMethod('typeof', signature(x="big.matrix"),
  function(x) return(.Call('GetTypeString', x@address)))

setMethod('head', signature(x="big.matrix"),
  function(x, n = 6) {
    n <- min(as.integer(n), nrow(x))
    if (n<1 | n>nrow(x)) stop("n must be between 1 and nrow(x)")
    return(x[1:n,])
  })

setMethod('tail', signature(x="big.matrix"),
  function(x, n = 6) {
    n <- min(as.integer(n), nrow(x))
    if (n<1 | n>nrow(x)) stop("n must be between 1 and nrow(x)")
    return(x[(nrow(x)-n+1):nrow(x),])
  })

setMethod('print', signature(x='big.matrix'), 
  function(x) {
    if (options()$bigmemory.print.warning==TRUE)
    {
      cat("Warning: This is not advised.  Here is the head of the matrix:\n")
      print(head(x))
    }
    else
    {
      # Should change this to a C print function, unfortunately, for proper
      # formatting, this means we would also have to pass the terminal
      # width.
      print(x[,])
    }
  })

setGeneric('nebytes', function(x) standardGeneric('nebytes'))
setMethod('nebytes', signature(x='big.matrix'),
  function(x) {
    return(.Call("GetNumExtraBytes", x@address))
  })

###################################################################
# mwhich()
#
# x big.matrix  
# cols  is.numeric or is.character
# vals  list of scalar or 2-vectors otherwise
# comps could be missing, in which case we'll fill in 'eq' in signature,
#       a list of comparisons matching dim of associated vals component

setGeneric('mwhich', function(x, cols, vals, comps, op = 'AND')
  standardGeneric('mwhich'))

setMethod('mwhich',
  signature(x='big.matrix', op='character'),
  function(x, cols, vals, comps, op) {
    return(mwhich.internal(x, cols, vals, comps, op, 'MWhichBigMatrix'))
  })

setMethod('mwhich',
  signature(x='matrix', op='character'),
  function(x, cols, vals, comps, op)
  {
    if (is.integer(x))
      return(mwhich.internal(x, cols, vals, comps, op, 'MWhichRIntMatrix'))
    if (is.numeric(x))
      return(mwhich.internal(x, cols, vals, comps, op, 'MWhichRNumericMatrix'))
    stop("Unsupported matrix type given to mwhich")
  })

setMethod('mwhich',
  signature(x='big.matrix', op='missing'),
  function(x, cols, vals, comps)
    return(mwhich.internal(x, cols, vals, comps, op='OR', 
                           whichFuncName='MWhichBigMatrix')))

setMethod('mwhich',
  signature(x='matrix', op='missing'),
  function(x, cols, vals, comps)
  {
    if (is.integer(x))
      return(mwhich.internal(x, cols, vals, comps, op='OR', 
                             whichFuncName='MWhichRIntMatrix'))
    if (is.numeric(x))
      return(mwhich.internal(x, cols, vals, comps, op='OR', 
                             whichFuncName='MWhichRNumericMatrix'))
    stop("Unsupported matrix type given to mwhich")
  })

mwhich.internal <- function(x, cols, vals, comps, op, whichFuncName) 
{
  cols <- cleanupcols(cols, ncol(x), colnames(x))
  if (is.shared(x) && options()$rlock.enabled) lockcols(x, cols, 'r')
  if (length(setdiff(cols, 1:ncol(x))) > 0)
    stop('Invalid column(s) in which()')

  # if vals or comps are not lists but are length 1 or 2, make them
  # trivial lists.
  if ( !is.list(vals) & 
       (length(vals)==1 || length(vals)==2) ) {
    vals <- list(vals)
  } else {
    if (!is.list(vals)) stop('vals should be a list')
  }
  if ( !is.list(comps) &
       (length(comps)==1 || length(comps)==2)) {
    comps <- list(comps)
  } else {
    if (!is.list(comps)) stop('comps should be a list')
  }

  # Replicate vals or comps if appropriate.
  if (length(cols)!=length(vals)) {
    if (length(vals)==1) {
      vals <- data.frame(matrix(unlist(vals), length(vals), length(cols)))
    } else stop('length(vals) must be 1 or length(cols)')
  }
  if (length(cols)!=length(comps)) {
    if (length(comps)==1) {
      comps <- data.frame(matrix(unlist(comps), length(comps), length(cols)),
                          stringsAsFactors=FALSE)
    } else stop('length(comps) must be 1 or length(cols)')
  }
  if (length(comps)!=length(vals))
    stop('length of comps must equal length of vals')
  if (any(!unlist(lapply(comps, is.character))) ||
      any(!(unlist(comps) %in% c('eq', 'neq', 'le', 'lt', 'ge', 'gt')))) {
    stop('comps must contain eq, neq, le, lt, ge, or gt')
  }

  testCol <- cols
  opVal <- 0
  if (op == 'OR') opVal <- 1
  minVal <- rep(NA, length(cols))
  maxVal <- rep(NA, length(cols))
  chkmin <- rep(0, length(cols))
  chkmax <- rep(0, length(cols))

  for (i in 1:length(cols)) {

    if (length(vals[[i]])==1) {
      # Here, we have the easy comparisons.
      if (is.na(vals[[i]]) && (comps[[i]]!='eq' && comps[[i]]!='neq'))
        stop('NA comparison limited to eq and neq, not le, lt, gt, or ge')
      if (length(comps[[i]])==1) {
        if (comps[[i]]=='eq' || comps[[i]]=='neq') {
          minVal[i] <- vals[[i]]
          maxVal[i] <- vals[[i]]
        }
        if (comps[[i]]=='neq') {
          chkmin[i] <- -1
          chkmax[i] <- -1            # Not used, but....
        }
        if (comps[[i]]=='ge' || comps[[i]]=='gt') {
          minVal[i] <- vals[[i]]
          maxVal[i] <- Inf
          if (comps[[i]]=='gt') chkmin[i] <- 1
        }
        if (comps[[i]]=='le' || comps[[i]]=='lt') {
          minVal[i] <- -Inf
          maxVal[i] <- vals[[i]]
          if (comps[[i]]=='lt') chkmax[i] <- 1
        }
      } else stop('vals/comps must be componentwise of same dimension')
    } else {
      # Here, we have two vals and two comps
      if (any(is.na(vals[[i]]))) stop('NAs not allowed in interval comparison')
      minVal[i] <- vals[[i]][1]
      maxVal[i] <- vals[[i]][2]
      if (comps[[i]][1]=='gt') chkmin[i] <- 1
      if (comps[[i]][2]=='lt') chkmax[i] <- 1
      if (comps[[i]][1]!='gt' && comps[[i]][1]!='ge')
        stop('invalid comparison of lower bound')
      if (comps[[i]][2]!='lt' && comps[[i]][2]!='le')
        stop('invalid comparison of upper bound')
    }

  } # End of the for loop

  ##### The new C function has new vectors chkmin and chkmax;
  ##### the value 0 indicates comparison with equality,
  ##### the value 1 indicates a strict inequality,
  ##### the value -1 indicates a 'neq' check;
  ##### if is.na checking is required, only the minVal needs to be
  ##### used, with chkmin = 0 being is.na and chkmin = 1 being !is.na.

  ret = NULL
  if (whichFuncName == 'MWhichBigMatrix')
    ret = .Call(whichFuncName, x@address, as.double(testCol), 
                as.double(minVal), as.double(maxVal), 
                as.integer(chkmin), as.integer(chkmax), as.integer(opVal))
  else
    ret = .Call(whichFuncName, x, nrow(x),
                as.double(testCol), 
                as.double(minVal), as.double(maxVal), 
                as.integer(chkmin), as.integer(chkmax), as.integer(opVal))

  if (is.shared(x) && options()$rlock.enabled)
    unlockcols(x, cols)

  return(ret)
}

setMethod('dimnames', signature(x = "big.matrix"),
  function(x) return(list(rownames.bm(x), colnames.bm(x))))

setMethod('dimnames<-', signature(x = "big.matrix", value='list'),
  function(x, value) {
    rownames.bm(x) <- value[[1]]
    colnames.bm(x) <- value[[2]]
    return(x)
  })

hash.mat <- function(x, col)
{
  col <- cleanupcols(col, ncol(x), colnames(x))
  if (colmin(x, col)<1) 
    stop("Error: minimum value in specified column should be 1 or more.")
  return(matrix(.Call('MatrixHashRanges', x@address, as.double(col)),
                      ncol=2, byrow=TRUE))
}

read.big.matrix <- function(fileName, sep=',', header=FALSE, col.names=NULL, 
                            row.names=NULL, has.row.names=FALSE, ignore.row.names=FALSE,
                            type=NA, skip=0, separated=FALSE,
                            shared=FALSE, backingfile=NULL, backingpath=NULL,
                            descriptorfile=NULL, extraCols=NULL) 
{ stop("Error: You must specify a file name.") }

setMethod('read.big.matrix', signature(fileName='character'),
  function(fileName, sep=',', header=FALSE, col.names=NULL,
           row.names=NULL, has.row.names=FALSE, ignore.row.names=FALSE,
           type=NA, skip=0, separated=FALSE, shared=FALSE, 
           backingfile=NULL, backingpath=NULL, descriptorfile=NULL,
           extraCols=NULL)
  {
    if (is.logical(col.names) | is.logical(row.names))
      stop("row.names and col.names, if used, must only be vectors of names (not logicals).")
    if ( (header | is.character(col.names)) & is.numeric(extraCols) )
      stop(paste("When column names are specified, extraCols must be the names",
                 "of the extra columns."))
    if (!header & is.null(col.names) & is.character(extraCols))
      stop(paste("No header and no column names were specified, so extraCols",
           "must be an integer."))
    headerOffset <- as.numeric(header)
    colNames <- NULL
    if (header) {
      colNames <- unlist(strsplit(
        scan(fileName, what='character', skip=skip, nlines=1, sep="\n", 
             quiet=TRUE), split=sep))
      colNames <- gsub("\"", "", colNames, perl=TRUE)
      colNames <- gsub("\'", "", colNames, perl=TRUE)
      if (is.na(colNames[1]) | colNames[1]=="") colNames <- colNames[-1]
      if (is.character(col.names)) {
        warning("Using supplied column names and skipping the header row.\n")
        colNames <- col.names
      } else {
        if (!is.null(col.names))
          stop("Invalid header/col.names usage (col.names must be a vector of names if used).\n")
      }
    } else {
      if (is.character(col.names)) colNames <- col.names
    }

    # Get the first line of data
    firstLineVals <- unlist(strsplit(
      scan(fileName, what='character', skip=(skip+headerOffset), 
           nlines=1, sep="\n", quiet=TRUE), split=sep))
    firstLineVals[firstLineVals=="NA"] <- NA

    # At this point, we assume there are length(colNames) columns of data if
    # available, otherwise, figure it out.
    if (!is.null(colNames)) numCols <- length(colNames)
    else {
      numCols <- length(firstLineVals) - has.row.names 
    }

    if (length(firstLineVals) - has.row.names != numCols)
      stop("Dimension mismatch between header row and first data row.\n")

    rowNames <- NULL
    if (!is.null(row.names)) {
      if (is.character(row.names)) {
        rowNames <- row.names
        ignore.row.names <- TRUE
      } else { stop("Invalid row.names (must be a vector of names if used).\n") }
    }

    if (is.na(type)) {
      type <- 'double'
      if (has.row.names) firstLineVals <- firstLineVals[-1]
      if (sum(na.omit(as.integer(firstLineVals)) ==
              na.omit(as.double(firstLineVals))) ==
          numCols ) 
        type <- 'integer'
      warning(paste("Because type was not specified, we chose", type,
                    "based on the first line of data."))
    }

    lineCount <- .Call("CCountLines", fileName) - skip - headerOffset
    numRows <- lineCount
    createCols <- numCols
    if (is.numeric(extraCols)) createCols <- createCols + extraCols
    if (is.character(extraCols)) {
      createCols <- createCols + length(extraCols)
      colNames <- c(colNames, extraCols)
    }

    bigMat <- big.matrix(nrow=numRows, ncol=createCols, type=type,
                         dimnames=list(rowNames, colNames), init=NULL, 
                         separated=separated, backingfile=backingfile,
                         backingpath=backingpath,
                         descriptorfile=descriptorfile)

    # has.row.names indicates whether or not there are row names;
    # we take ignore.row.names from the user, but pass (essentially)
    # use.row.names (which is !ignore.row.names) to C:
    .Call('ReadMatrix', fileName, bigMat@address, 
          as.integer(skip+headerOffset), as.double(numRows), 
          as.double(numCols), as.character(sep), as.logical(has.row.names),
          as.logical(!ignore.row.names))

    return(bigMat)
  })

write.big.matrix <- function(x, fileName=NA, row.names = FALSE,
                             col.names = FALSE, sep=",")
{ stop("Error: You must specify a bigmatRix and file name.") }

setMethod('write.big.matrix', signature(x='big.matrix',fileName='character'),
function(x, fileName, row.names = FALSE, col.names = FALSE, sep=",")
{
  if (is.character(row.names))
    stop("You must set the row names before writing.\n")
  if (is.character(col.names))
    stop("You must set the column names before writing.\n")
  if (row.names & !.Call("HasRowColNames",x@address)[1]) {
    row.names <- FALSE
    warning("No row names exist, overriding your row.names option.\n")
  }
  if (col.names & !.Call("HasRowColNames",x@address)[2]) {
    col.names <- FALSE
    warning("No column names exist, overriding your col.names option.\n")
  }

  .Call('WriteMatrix', x@address, fileName, as.logical(row.names), 
        as.logical(col.names), sep)
  invisible(NULL)
})

setGeneric('is.shared', function(x) standardGeneric('is.shared'))

setMethod('is.shared', signature(x='big.matrix'),
  function(x)
  {
    return(.Call("IsShared", x@address))
  })

setMethod('is.shared', signature(x='matrix'), function(x) return(FALSE))

setGeneric('is.separated', function(x) standardGeneric('is.separated'))

setMethod('is.separated', signature(x='big.matrix'),
  function(x)
  {
    return(.Call("IsSeparated", x@address))
  })

setMethod('is.separated', signature(x='matrix'), function(x) return(FALSE))

cleanupcols <- function(cols=NULL, nc=NULL, colnames=NULL) {

  if (is.null(cols)) cols <- 1:nc
  else {
    if (!is.numeric(cols) & !is.character(cols) & !is.logical(cols))
      stop("column indices must be numeric, logical, or character vectors.")
    if (is.character(cols))
      if (is.null(colnames)) stop("column names do not exist.")
      else cols <- mmap(cols, colnames)
    if (is.logical(cols)) {
      if (length(cols) != nc)
        stop(paste("column vector length must match the number of",
                   "columns of the matrix."))
      cols <- which(cols)
    }
    tempj <- .Call("CCleanIndices", as.double(cols), as.double(nc))
    if (is.null(tempj[[1]])) stop("Illegal column index usage in extraction.\n")
    if (tempj[[1]]) cols <- tempj[[2]]
  }
  return(cols)
}

deepcopy <- function(x, cols=NULL, type=NULL, separated=NULL, shared=NULL, 
                     backingfile=NULL, backingpath=NULL,
                     descriptorfile=NULL, preserve=TRUE) 
{
  cols <- cleanupcols(cols, ncol(x), colnames(x))
  if (nrow(x) > 2^31-1)
    stop(paste("Too many rows to copy at this point in time;",
               "this may be fixed in the future."))
  if (is.null(type)) type <- typeof(x)
  if (is.null(separated)) separated <- is.separated(x)
  if (is.null(shared)) shared <- is.shared(x)
  y <- big.matrix(nrow=nrow(x), ncol=length(cols), type=type, init=NULL,
		  dimnames = dimnames(x), separated=separated, shared=shared,
		  backingfile=backingfile, backingpath=backingpath,
      descriptorfile=descriptorfile, preserve=preserve)
  for (i in 1:length(cols)) y[,i] <- x[,cols[i]]

  return(y)
}
