% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data-documentation.R
\docType{data}
\name{ROSETTA.centroids}
\alias{ROSETTA.centroids}
\title{Average Hydraulic Parameters from the ROSETTA Model by USDA Soil Texture
Class}
\format{
A data frame: \describe{

\item{texture}{soil texture class, ordered from low to high
available water holding capacity} \item{theta_r}{average saturated
water content} \item{theta_s}{average residual water content}
\item{alpha}{average value, related to the inverse of the air entry
suction, log10-transformed values with units of cm}
\item{npar}{average value, index of pore size distribution,
log10-transformed values with units of 1/cm}

\item{theta_r_sd}{1 standard deviation of \code{theta_r}}
\item{theta_s_sd}{1 standard deviation of \code{theta_s}}
\item{alpha_sd}{1 standard deviation of \code{alpha}}
\item{npar_sd}{1 standard deviation of \code{npar}}

\item{sat}{approximate volumetric water content at which soil
material is saturated} \item{fc}{approximate volumetric water
content at which matrix potential = -33kPa} \item{pwp}{approximate
volumetric water content at which matrix potential = -1500kPa}
\item{awc}{approximate available water holding capacity: VWC(-33kPa)
\itemize{
\item VWC(-1500kPa)} }
}
}
\source{
\href{https://www.ars.usda.gov/pacific-west-area/riverside-ca/agricultural-water-efficiency-and-salinity-research-unit/docs/model/rosetta-class-average-hydraulic-parameters/}{ROSETTA Class Average Hydraulic Parameters}
}
\usage{
data(ROSETTA.centroids)
}
\description{
Average soil hydraulic parameters generated by the first stage predictions
of the ROSETTA model by USDA soil texture class. These data were extracted
from \href{https://www.ars.usda.gov/pacific-west-area/riverside-ca/agricultural-water-efficiency-and-salinity-research-unit/docs/model/rosetta-class-average-hydraulic-parameters/}{ROSETTA documentation} and re-formatted for ease of use.
}
\details{
Theoretical water-retention parameters for uniform soil material of each
texture class have been estimated via van Genuchten model.

See \href{https://ncss-tech.github.io/AQP/soilDB/fetchKSSL-VG-demo.html}{the related tutorial}
}
\examples{


\dontrun{

library(aqp)
library(soilDB)
library(latticeExtra)

data("ROSETTA.centroids")

# iterate over horizons and generate VG model curve
res <- lapply(1:nrow(ROSETTA.centroids), function(i) {
  m <- KSSL_VG_model(VG_params = ROSETTA.centroids[i, ], phi_min = 10^-3, phi_max=10^6)$VG_curve
  # copy generalized hz label
  m$hz <- ROSETTA.centroids$hz[i]
  # copy ID
  m$texture_class <- ROSETTA.centroids$texture[i]
  return(m)
})

# copy over lab sample number as ID
res <- do.call('rbind', res)

# check: OK
str(res)


# visual check: OK
xyplot(
  phi ~ theta | texture_class, data=res,
  type=c('l', 'g'),
  scales=list(alternating=3, x=list(tick.number=10), y=list(log=10, tick.number=10)),
  yscale.components=yscale.components.logpower,
  ylab=expression(Suction~~(kPa)),
  xlab=expression(Volumetric~Water~Content~~(cm^3/cm^3)),
  par.settings = list(superpose.line=list(col='RoyalBlue', lwd=2)),
  strip=strip.custom(bg=grey(0.85)),
  as.table=TRUE
)


}


}
\references{
van Genuchten, M.Th. (1980). "A closed-form equation for
predicting the hydraulic conductivity of unsaturated soils". Soil Science
Society of America Journal. 44 (5): 892-898.

Schaap, M.G., F.J. Leij, and M.Th. van Genuchten. 2001. ROSETTA: A computer
program for estimating soil hydraulic parameters with hierarchical
pedotransfer functions. Journal of Hydrology 251(3–4): 163-176.
}
\keyword{datasets}
