#######################################################
#	apc package
#	Bent Nielsen, 27 June 2014, version 1
#	function to plot data
#######################################################
#	Copyright 2014 Bent Nielsen
#	Nuffield College, OX1 1NF, UK
#	bent.nielsen@nuffield.ox.ac.uk
#
#	This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#######################################################

apc.data.sums	<- 	function(apc.data.list,data.type="r",apc.index=NULL)
#	BN 15 Dec 2013
#	input:	apc.data.list		
#			apc.index			generated by apc.index	
#			data.type			optional. Character.  "r" response only, "d" dose only, "m" mortality rates. 
#	output:				sums.age	sums of data by age
#						sums.per	sums of data by period
#						sums.coh	sums of data by cohort
{ 	#	apc.data.sums
	######################
	#	get index
	if(is.null(apc.index)==TRUE)
		apc.index	<- apc.get.index(apc.data.list)
	######################
	#	set plot array dimensions
	if(data.type %in% c("d","m") & is.null(apc.data.list$dose)==TRUE) 	return(cat("apc.error: Doses are not available \n"))
	if(data.type == "r")	data.matrix	<- apc.data.list$response
	if(data.type == "d") 	data.matrix	<- apc.data.list$dose
	if(data.type == "m") 	data.matrix	<- apc.data.list$response / apc.data.list$dose
	#################
	#	construct trapezoid matrix in age/cohort coordinates
	trap		<- matrix(data=NA,nrow=apc.index$age.max,ncol=apc.index$coh.max)
	trap[apc.index$index.trap]	<- data.matrix[apc.index$index.data]
	sums.age	<- rowSums(trap,na.rm=TRUE)
	sums.coh	<- colSums(trap,na.rm=TRUE)
	#	construct trapezoid matrix in age/period coordinates
	trap.ap		<- matrix(data=NA,nrow=apc.index$age.max,ncol=apc.index$per.max)
	for(row in 1:apc.index$age.max)
	{
		col.lower	<- max(1,apc.index$per.zero+2-row)
		col.upper	<- min(apc.index$coh.max,apc.index$per.zero+1-row+apc.index$per.max)
		per.lower	<- max(1,row-apc.index$per.zero)
		per.upper	<- col.upper-col.lower+per.lower	
		trap.ap[row,per.lower:per.upper]	<- trap[row,col.lower:col.upper]
	}
	#	take sums
	sums.age	<- rowSums(trap,	na.rm=TRUE)
	sums.coh	<- colSums(trap,	na.rm=TRUE)
	sums.per	<- colSums(trap.ap,	na.rm=TRUE)
	#	return				
	return(list(sums.age=sums.age,
				sums.per=sums.per,
				sums.coh=sums.coh))
}	#	apc.data.sums

apc.plot.data.sums	<- function(apc.data.list,data.type="a",apc.index=NULL,type="o",log="",main.outer=NULL,main.sub=NULL)
#	BN 15 Dec 2013
#	input:	apc.data.list	list. Data
#			apc.index		optional list. Generated by apc.index	
#			type			optional plot parameter. Character. "o" if overlaid points and lines. "l" if lines. "p" if points.  Default is "o".
#			log				optional plot parameter. Character. "y" if y-scale is logarithmic, otherwise ""
#			data.type		optional. Character.  "r" response only, "d" dose only, "m" mortality rates, "a" all (default).
#	output:				plots of age sums, cohort sums, period sums
{	#	apc.plot.data.sums
	######################
	#	get index
	if(is.null(apc.index)==TRUE)
		apc.index	<- apc.get.index(apc.data.list)
	######################
	#	get title
	if(is.null(main.outer)==TRUE)
		main.outer	<- "Data sums by age/period/cohort index"			
	######################
	#	set plot array dimensions
	if(data.type %in% c("r","d","m") | is.null(apc.data.list$dose)==TRUE) 	par(mfrow=c(1,3))
	if(data.type == "a" & is.null(apc.data.list$dose)==FALSE) 				par(mfrow=c(3,3))
	if(data.type %in% c("d","m") & is.null(apc.data.list$dose)==TRUE) 		return(cat("apc.plot.data.sums error: Doses are not available \n"))
	par(mar=c(5,5,2,0),oma=c(0,0,1,1))
	######################
	#	plot response
	if(data.type %in% c("r","a"))
	{	sums.response	<- apc.data.sums(apc.data.list,apc.index,data.type="r")
		if(is.null(main.sub)==TRUE)	main	<- "response"
		plot(seq(from=apc.index$age1,by=apc.index$unit,length=apc.index$age.max),sums.response$sums.age,main=main,xlab="age"   ,ylab="sums of data",type=type,log=log)
		plot(seq(from=apc.index$per1,by=apc.index$unit,length=apc.index$per.max),sums.response$sums.per,main=main,xlab="period",ylab="sums of data",type=type,log=log)
		plot(seq(from=apc.index$coh1,by=apc.index$unit,length=apc.index$coh.max),sums.response$sums.coh,main=main,xlab="cohort",ylab="sums of data",type=type,log=log)
	}
	######################
	#	plot dose
	if(data.type %in% c("d","a") & is.null(apc.data.list$dose)==FALSE)
	{
		sums.dose		<- apc.data.sums(apc.data.list,apc.index,data.type="d")
		if(is.null(main.sub)==TRUE)	main	<- "dose"
		plot(seq(from=apc.index$age1,by=apc.index$unit,length=apc.index$age.max),sums.dose$sums.age,main=main,xlab="age"   ,ylab="sums of data",type=type,log=log)
		plot(seq(from=apc.index$per1,by=apc.index$unit,length=apc.index$per.max),sums.dose$sums.per,main=main,xlab="period",ylab="sums of data",type=type,log=log)
		plot(seq(from=apc.index$coh1,by=apc.index$unit,length=apc.index$coh.max),sums.dose$sums.coh,main=main,xlab="cohort",ylab="sums of data",type=type,log=log)
	}
	######################
	#	plot mortality rates
	if(data.type %in% c("m","a") & is.null(apc.data.list$dose)==FALSE)
	{
		sums.dose		<- apc.data.sums(apc.data.list,apc.index,data.type="m")
		if(is.null(main.sub)==TRUE)	main	<- "rates"
		plot(seq(from=apc.index$age1,by=apc.index$unit,length=apc.index$age.max),sums.dose$sums.age,main=main,xlab="age"   ,ylab="sums of data",type=type,log=log)
		plot(seq(from=apc.index$per1,by=apc.index$unit,length=apc.index$per.max),sums.dose$sums.per,main=main,xlab="period",ylab="sums of data",type=type,log=log)
		plot(seq(from=apc.index$coh1,by=apc.index$unit,length=apc.index$coh.max),sums.dose$sums.coh,main=main,xlab="cohort",ylab="sums of data",type=type,log=log)
	}
	title(main.outer,outer=TRUE)
}	#	apc.plot.data.sums

apc.plot.data.sparsity	<- 	function(apc.data.list,data.type="a",apc.index=NULL,sparsity.limits=c(1,2),cex=NULL,pch=15,main.outer=NULL)
#	BN 24 nov 2013
#	input:	apc.data.list		list. Data
#			apc.index			optional. List. Generated by apc.index	
#			sparsity.limits		optional. Vector with two entries giving sparsity thresholds, default c(1,2)
#			data.type				optional. Character.  "r" response only, "d" dose only, "a" all (default).
#	output:	plots indicating where data are sparse
{	#	apc.plot.data.sparsity	
	######################
	#	get index
	if(is.null(apc.index)==TRUE)
		apc.index	<- apc.get.index(apc.data.list)
	######################
	#	get title
	if(is.null(main.outer)==TRUE)
		main.outer	<- paste("Sparsity plots \n","(black <",as.character(sparsity.limits[1]),",grey <",as.character(sparsity.limits[2]),")")			
	######################
	#	get variables
	xlab	<- apc.index$data.xlab	
	ylab	<- apc.index$data.ylab	
	x1		<- apc.index$data.x1		
	y1		<- apc.index$data.y1
	xmax	<- apc.index$data.xmax
	ymax	<- apc.index$data.ymax
	unit	<- apc.index$unit			  
	######################
	#	set limits	
	xlim	<- c(x1,x1 +(xmax-1)*unit)
	ylim	<- c(y1,y1 +(ymax-1)*unit)
	######################
	#	set plot array dimensions
	if(data.type %in% c("r","d") | is.null(apc.data.list$dose)==TRUE) 	par(mfrow=c(1,1))
	if(data.type == "a" & is.null(apc.data.list$dose)==FALSE) 			par(mfrow=c(1,2))
	if(data.type == "d" & is.null(apc.data.list$dose)==TRUE) 			return(cat("apc.error: Doses are not available \n"))
	par(mar=c(5,5,2,0),oma=c(0,0,2,1))
	######################
	#	plot response
	function.sparsity.plot	<- function(data.matrix,main)
	{	if(is.null(cex)==TRUE)
		{	nmax	<- max(xmax,ymax)
						cex	<- 0.5
			if(nmax<20) cex	<- 1
			if(nmax<10) cex	<- 2
			if(nmax<5 ) cex	<- 5
		}
		plot(1,1,pch=NA,xlim=xlim,ylim=ylim,xlab=xlab,ylab=ylab,main=main)
		for(row in 1:xmax)
			for(col in 1:ymax)
			{
				x	<- x1+(row-1)*unit
				y	<- y1+(col-1)*unit
				if(data.matrix[row,col]<sparsity.limits[2])		points(x,y,cex=cex,pch=pch,col=gray(0.66))	
				if(data.matrix[row,col]<sparsity.limits[1]) 	points(x,y,cex=cex,pch=pch,col=gray(0.0))
			}
	}
	if(data.type %in% c("r","a"))
		function.sparsity.plot(apc.data.list$response,"responses")
	#	plot dose
	if(data.type %in% c("d","a") & is.null(apc.data.list$dose)==FALSE)
		function.sparsity.plot(apc.data.list$dose,"doses")	
	title(main.outer,outer=TRUE)
}	#	apc.plot.data.sparsity

apc.plot.data.within	<- 	function(apc.data.list,data.type="r",thin=NULL,apc.index=NULL,ylab=NULL,type="o",log="y",legend=TRUE,lty=1:5,col=1:6,bty="n",main.outer=NULL)
#	BN 27 jun 2014
#	input:	apc.data.list	list. Data
#			apc.index		optional. List. Generated by apc.index	
#			data.type			optional. Character.  "r" response, "d" dose, "m" mortality.  Default is "r".
#			thin			Optional. Numerical.  age/periods/cohorts are grouped in groups of size thin.  Default=1
#			ylab			optional plot parameter. Character: common label on y axes, for instance "log mortality rates". Default ""
#			type			optional plot parameter. Character. "o" if overlaid points and lines. "l" if lines. "p" if points.  Default is "o".
#			log				optional plot parameter. Character: common scale for y axes, Default: "y" for log scale. Use "" if normal scale
#			legend			optional plot parameter. Logical: should legends be drawn. Default: "TRUE"
#			lty				optional plot parameter. Vector of line types.
#							The first element is for the first column, the second element for the second column, etc.,
#							even if lines are not plotted for all columns. Line types will be used cyclically
#							until all plots are drawn.  Default: 1:5
#			col				optional plot parameter. Vector of colors.
#							The first element is for the first column, the second element for the second column, etc.,
#							even if lines are not plotted for all columns. Colors will be used cyclically
#							until all plots are drawn.  Default is 1:6.
#			bty				optional plot parameter. Character: the type of box to be drawn around the legend.
#							The allowed values are "n" and "o".	Default is "n".			
#	output:	6 plots of data: age vs period, cohort vs age, cohort vs period
{ 	#	apc.plot.data.within
	######################
	#	get index
	if(is.null(apc.index)==TRUE)
		apc.index	<- apc.get.index(apc.data.list)
	######################
	#	get title
	if(is.null(main.outer)==TRUE)
		main.outer	<- "plots of data using two indices"
	######################
	#	get thinning value function
	function.thin.value		<- function(m,thin=NULL)
	#	BN 27 Jun 2014
	#	function for getting default thin value for grouping
	{	#	function.thin.value
		l.thin	<- thin
		if(is.null(l.thin)==TRUE)
			{	ncol	<- ncol(m)
							l.thin <- 1
				if(ncol>10) l.thin <- 2
				if(ncol>20) l.thin <- 5
			}
		return(l.thin)
	}	#	function.thin.value
	######################
	#	thinning function
	function.thin.matrix	<- function(m,l.thin=1)
	#	BN 27 Jun 2013
	#	function for grouping columns: it takes row sums within each group
	{ 	#	function.thin.matrix
		if(l.thin==1)	mm <- m			else
		{
			nrow	<- nrow(m)
			ncol	<- ncol(m)
			ngroup	<- ceiling(ncol/l.thin)
			mm		<- matrix(data=NA,nrow=nrow,ncol=ngroup)
			for(group in 1:(ngroup-1))
				mm[,group]	<- as.matrix(rowSums(m[,((group-1)*l.thin+1):(group*l.thin)]))
			mm[,ngroup]	<- as.matrix(rowSums(as.matrix(m[,((ngroup-1)*l.thin+1):ncol])))
			if(ngroup != ncol/l.thin)
				print("apc.warning: maximal index not divisible by thin, so last group smaller than other groups")
		}
		return(mm)	
	}	#	function.thin.matrix
	######################
	function.trapezoid.to.ap	<- function(trapezoid,transpose=FALSE)
	#	BN, 17 Sep 2013
	#	transforms a trapezoid in AC format to a trapezoid in AP format
	{	#	function.trapezoid.to.ap
		######################
		#	get values
		age.max		<- apc.index$age.max
		per.max		<- apc.index$per.max
		coh.max		<- apc.index$coh.max
		per.zero	<- apc.index$per.zero
		nrow		<- age.max
		ncol		<- coh.max
		if(transpose==TRUE)
		{
			trapezoid	<- t(trapezoid)
			nrow		<- coh.max
			ncol		<- age.max
		}	
		#	create ap matrix
		m			<- matrix(data=NA,nrow=nrow,ncol=per.max)
		for(row in 1:nrow)
		{
			col.lower	<- max(1,per.zero+2-row)
			col.upper	<- min(ncol,per.zero+1-row+per.max)
			per.lower	<- max(1,row-per.zero)
			per.upper	<- col.upper-col.lower+per.lower	
			m[row,per.lower:per.upper]	<- trapezoid[row,col.lower:col.upper]
		}
		return(m)
	}	#	function.trapezoid.to.ap
	######################
	#	six plot function
	function.six.plot	<- function(data.matrix.r,data.matrix.d=NULL)
	#	BN 27 Jun 2014
	{	#	function.six.plot
		######################
		#	get values
		age1	<- apc.index$age1
		per1	<- apc.index$per1
		coh1	<- apc.index$coh1
		age.max	<- apc.index$age.max
		per.max	<- apc.index$per.max
		coh.max	<- apc.index$coh.max
		unit	<- apc.index$unit
		######################
		#	set dose matrix to ones
		if(is.null(data.matrix.d)==TRUE)
			data.matrix.d	<- matrix(data=1,nrow=nrow(data.matrix.r),ncol=ncol(data.matrix.r))
		######################
		#	get trapezoids in age/cohort coordinate system
		trap.r	<- matrix(data=NA,nrow=age.max,ncol=coh.max)
		trap.r[apc.index$index.trap]	<- data.matrix.r[apc.index$index.data]
		trap.d	<- matrix(data=NA,nrow=age.max,ncol=coh.max)
		trap.d[apc.index$index.trap]	<- data.matrix.d[apc.index$index.data]
		######################
		#	one plot function
		function.one.plot	<- function(m.r,m.d,x1,x.max,xlab,w.max,l.main)
		#	BN 27 Jun 2014
		{	#	function.one.plot
			x		<- seq(from=x1,length=x.max,by=unit)
			l.thin	<- function.thin.value(m.r,thin)
			m		<- function.thin.matrix(m.r/m.d,l.thin=l.thin)
			matplot(x,m,type=type,pch=20,log=log,lty=lty,col=col,main=l.main,xlab=xlab,ylab=ylab)
			within	<- seq(from=per1,length=ceiling(w.max/l.thin),by=unit*l.thin)
			if(legend==TRUE)	legend(x="topleft",legend=as.character(within),lty=lty,col=col,bty=bty)	
		}	#	function.one.plot
		######################
		#	1	period versus age
		m.r	<- function.trapezoid.to.ap(trap.r)
		m.d	<- function.trapezoid.to.ap(trap.d)
		function.one.plot(m.r,m.d,age1,age.max,"age",per.max,"within period")
		######################
		#	2	cohort versus age
		m.r	<- trap.r
		m.d	<- trap.d		
		function.one.plot(m.r,m.d,age1,age.max,"age",coh.max,"within cohort")
		######################
		#	3 	period versus cohort
		m.r	<- function.trapezoid.to.ap(trap.r,transpose=TRUE)
		m.d	<- function.trapezoid.to.ap(trap.d,transpose=TRUE)
		function.one.plot(m.r,m.d,coh1,coh.max,"coh",per.max,"within period")
		######################
		#	4	age versus period
		m.r	<- t(function.trapezoid.to.ap(trap.r))
		m.d	<- t(function.trapezoid.to.ap(trap.d))
		function.one.plot(m.r,m.d,per1,per.max,"per",age.max,"within age")
		######################
		#	5	age versus cohort
		m.r	<- t(trap.r)
		m.d	<- t(trap.d)
		function.one.plot(m.r,m.d,coh1,coh.max,"coh",age.max,"within age")
		######################
		#	6	cohort versus period
		m.r	<- t(function.trapezoid.to.ap(trap.r,transpose=TRUE))
		m.d	<- t(function.trapezoid.to.ap(trap.d,transpose=TRUE))
		function.one.plot(m.r,m.d,per1,per.max,"per",coh.max,"within cohort")
	}	#	function.six.plot
	######################
	#	set plot array dimensions
	if(data.type %in% c("c","m") & is.null(apc.data.list$dose)==TRUE)	return(cat("apc.error: Doses are not available \n"))
	par(mfrow=c(2,3))
	par(mar=c(5,5,2,0),oma=c(0,0,1,1))
	######################
	if(data.type %in% c("r"))
	{	data.r	<- apc.data.list$response;
		if(is.null(ylab)==TRUE)
		{	if(log=="y")	ylab	<- "log response"
			if(log=="")		ylab	<- "response"
		}	
		function.six.plot(data.r)
	}	
	#	plot dose
	if(data.type %in% c("d"))
	{	data.d	<- apc.data.list$dose;
		if(is.null(ylab)==TRUE)
		{	if(log=="y")	ylab	<- "log dose"
			if(log=="")		ylab	<- "dose"
		}	
		function.six.plot(data.d)
	}	
	#	plot mortality rates
	if(data.type %in% c("m"))
	{	data.r	<- apc.data.list$response;
		data.d	<- apc.data.list$dose;
		if(is.null(ylab)==TRUE)
		{	if(log=="y")	ylab	<- "log mortality rate"
			if(log=="")		ylab	<- "mortality rate"
		}	
		function.six.plot(data.r,data.d)
	}	
	title(main.outer,outer=TRUE)
}	#	apc.plot.data.within

apc.plot.data.all	<- 	function(apc.data.list,log="y")
{	#	apc.plot.data.all
#	BN 25 nov 2013
#	input:	apc.data.list	list. Data
#			log				optional plot parameter. Character: common scale for y axes, Default: "y" for log scale. Use "" if normal scale
#	output:	all descriptive plots.
	apc.index	<- apc.get.index(apc.data.list)
	apc.plot.data.sums(apc.data.list,apc.index=apc.index,log=log)
	dev.new()
	apc.plot.data.sparsity(apc.data.list,apc.index=apc.index)
	dev.new()
	apc.plot.data.within(apc.data.list,data.type="r",apc.index=apc.index,log=log)
	if(is.null(apc.data.list$dose)==FALSE)
	{
		dev.new()
		apc.plot.data.within(apc.data.list,data.type="d",apc.index=apc.index,log=log)
		dev.new()
		apc.plot.data.within(apc.data.list,data.type="m",apc.index=apc.index,log=log)
	}
}	#	apc.plot.data.all