% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_vars_dt.R
\name{impute_dtc_dt}
\alias{impute_dtc_dt}
\title{Impute Partial Date Portion of a \code{--DTC} Variable}
\usage{
impute_dtc_dt(
  dtc,
  highest_imputation = "n",
  date_imputation = "first",
  min_dates = NULL,
  max_dates = NULL,
  preserve = FALSE
)
}
\arguments{
\item{dtc}{The \code{--DTC} date to impute

A character date is expected in a format like \code{yyyy-mm-dd} or
\code{yyyy-mm-ddThh:mm:ss}. Trailing components can be omitted and \code{-} is a
valid "missing" value for any component.

\describe{
\item{Permitted values}{a character date variable}
\item{Default value}{none}
}}

\item{highest_imputation}{Highest imputation level

The \code{highest_imputation} argument controls which components of the \code{--DTC}
value are imputed if they are missing. All components up to the specified
level are imputed.

If a component at a higher level than the highest imputation level is
missing, \code{NA_character_} is returned. For example, for \code{highest_imputation = "D"} \code{"2020"} results in \code{NA_character_} because the month is missing.

If \code{"n"} (none, lowest level) is specified no imputation is performed, i.e.,
if any component is missing, \code{NA_character_} is returned.

If \code{"Y"} (year, highest level) is specified, \code{date_imputation} must be \code{"first"} or \code{"last"}
and \code{min_dates} or \code{max_dates} must be specified respectively. Otherwise,
an error is thrown.

\describe{
\item{Permitted values}{\code{"Y"} (year, highest level), \code{"M"} (month), \code{"D"}
(day), \code{"n"} (none, lowest level)}
\item{Default value}{\code{"n"}}
}}

\item{date_imputation}{The value to impute the day/month when a datepart is
missing.

A character value is expected.
\itemize{
\item If  \code{highest_imputation} is \code{"M"}, month and day can be
specified as \code{"mm-dd"}: e.g. \code{"06-15"} for the 15th of June
\item When  \code{highest_imputation} is \code{"M"} or  \code{"D"}, the following keywords are available:
\code{"first"}, \code{"mid"}, \code{"last"} to impute to the first/mid/last
day/month. If \code{"mid"} is specified, missing components are imputed as the
middle of the possible range:
\itemize{
\item If both month and day are missing, they are imputed as \code{"06-30"}
(middle of the year).
\item If only day is missing, it is imputed as \code{"15"} (middle of the month).
}
}

The year can not be specified; for imputing the year
\code{"first"} or \code{"last"} together with \code{min_dates} or \code{max_dates} argument can
be used (see examples).

\describe{
\item{Permitted values}{\code{"first"}, \code{"mid"}, \code{"last"}, or user-defined}
\item{Default value}{\code{"first"}}
}}

\item{min_dates}{Minimum dates

A list of dates is expected. It is ensured that the imputed date is not
before any of the specified dates, e.g., that the imputed adverse event start
date is not before the first treatment date. Only dates which are in the
range of possible dates of the \code{dtc} value are considered. The possible dates
are defined by the missing parts of the \code{dtc} date (see example below). This
ensures that the non-missing parts of the \code{dtc} date are not changed.
A date or date-time object is expected.
For example

\if{html}{\out{<div class="sourceCode r">}}\preformatted{impute_dtc_dtm(
  "2020-11",
  min_dates = list(
   ymd_hms("2020-12-06T12:12:12"),
   ymd_hms("2020-11-11T11:11:11")
  ),
  highest_imputation = "M"
)
}\if{html}{\out{</div>}}

returns \code{"2020-11-11T11:11:11"} because the possible dates for \code{"2020-11"}
range from \code{"2020-11-01T00:00:00"} to \code{"2020-11-30T23:59:59"}. Therefore
\code{"2020-12-06T12:12:12"} is ignored. Returning \code{"2020-12-06T12:12:12"} would
have changed the month although it is not missing (in the \code{dtc} date).

\describe{
\item{Permitted values}{a list of dates, e.g. \code{list(ymd_hms("2021-07-01T04:03:01"), ymd_hms("2022-05-12T13:57:23"))}}
\item{Default value}{\code{NULL}}
}}

\item{max_dates}{Maximum dates

A list of dates is expected. It is ensured that the imputed date is not after
any of the specified dates, e.g., that the imputed date is not after the data
cut off date. Only dates which are in the range of possible dates are
considered. A date or date-time object is expected.

\describe{
\item{Permitted values}{a list of dates, e.g. \code{list(ymd_hms("2021-07-01T04:03:01"), ymd_hms("2022-05-12T13:57:23"))}}
\item{Default value}{\code{NULL}}
}}

\item{preserve}{Preserve day if month is missing and day is present

For example \code{"2019---07"} would return \verb{"2019-06-07} if \code{preserve = TRUE}
(and \code{date_imputation = "MID"}).

\describe{
\item{Permitted values}{\code{TRUE}, \code{FALSE}}
\item{Default value}{\code{FALSE}}
}}
}
\value{
A character vector
}
\description{
Imputation partial date portion of a \code{--DTC} variable based on user input.
}
\details{
Usually this computation function can not be used with \verb{\%>\%}.
}
\examples{
library(lubridate)

dates <- c(
  "2019-07-18T15:25:40",
  "2019-07-18T15:25",
  "2019-07-18T15",
  "2019-07-18",
  "2019-02",
  "2019",
  "2019",
  "2019---07",
  ""
)

# No date imputation (highest_imputation defaulted to "n")
impute_dtc_dt(dtc = dates)

# Impute to first day/month if date is partial
impute_dtc_dt(
  dtc = dates,
  highest_imputation = "M"
)
# Same as above
impute_dtc_dt(
  dtc = dates,
  highest_imputation = "M",
  date_imputation = "01-01"
)

# Impute to last day/month if date is partial
impute_dtc_dt(
  dtc = dates,
  highest_imputation = "M",
  date_imputation = "last",
)

# Impute to mid day/month if date is partial
impute_dtc_dt(
  dtc = dates,
  highest_imputation = "M",
  date_imputation = "mid"
)

# Impute a date and ensure that the imputed date is not before a list of
# minimum dates
impute_dtc_dt(
  "2020-12",
  min_dates = list(
    ymd("2020-12-06"),
    ymd("2020-11-11")
  ),
  highest_imputation = "M"
)

# Impute completely missing dates (only possible if min_dates or max_dates is specified)
impute_dtc_dt(
  c("2020-12", NA_character_),
  min_dates = list(
    ymd("2020-12-06", "2020-01-01"),
    ymd("2020-11-11", NA)
  ),
  highest_imputation = "Y"
)
}
\seealso{
Date/Time Computation Functions that returns a vector:
\code{\link{compute_age_years}()},
\code{\link{compute_dtf}()},
\code{\link{compute_duration}()},
\code{\link{compute_tmf}()},
\code{\link{convert_date_to_dtm}()},
\code{\link{convert_dtc_to_dt}()},
\code{\link{convert_dtc_to_dtm}()},
\code{\link{convert_xxtpt_to_hours}()},
\code{\link{impute_dtc_dtm}()}
}
\concept{com_date_time}
\keyword{com_date_time}
