% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xOBOcode.r
\name{xOBOcode}
\alias{xOBOcode}
\title{Function to create codes annotating nodes in an igraph object}
\usage{
xOBOcode(g, node.level = "term_distance", node.level.value = 2,
node.label.size = 2, node.label.color = "darkblue",
node.label.alpha = 0.8, node.label.padding = 0, node.label.arrow =
0.01,
node.label.force = 0, node.shape = 19, node.xcoord = NULL,
node.ycoord = NULL, node.color = NULL, node.color.title = NULL,
colormap = "grey-grey", ncolors = 64, zlim = NULL,
node.size.range = 4, title = "", edge.size = 0.5,
edge.color = "black", edge.color.alpha = 0.4, edge.curve = 0.1,
edge.arrow = 2, edge.arrow.gap = 0.02, node.table = "term_name",
node.table.wrap = 50, table.base.size = 7, table.row.space = 2,
table.nrow = 55, table.ncol = NULL, root.code = "RT")
}
\arguments{
\item{g}{an object of class "igraph"}

\item{node.level}{a character specifying which node attribute defining
the node level. By default, it is 'term_distance'}

\item{node.level.value}{a positive integer specifying the level value
as major branches. By default, it is 2}

\item{node.label.size}{a character specifying which node attribute used
for node label size}

\item{node.label.color}{a character specifying which node attribute
used for the node label color}

\item{node.label.alpha}{the 0-1 value specifying transparency of node
labelling}

\item{node.label.padding}{the padding around the labeled node}

\item{node.label.arrow}{the arrow pointing to the labeled node}

\item{node.label.force}{the repelling force between overlapping labels}

\item{node.shape}{an integer specifying node shape}

\item{node.xcoord}{a vector specifying x coordinates. If NULL, it will
be created using igraph::layout_with_kk}

\item{node.ycoord}{a vector specifying y coordinates. If NULL, it will
be created using igraph::layout_with_kk}

\item{node.color}{a character specifying which node attribute used for
node coloring}

\item{node.color.title}{a character specifying the title for node
coloring}

\item{colormap}{short name for the colormap. It can be one of "jet"
(jet colormap), "bwr" (blue-white-red colormap), "gbr" (green-black-red
colormap), "wyr" (white-yellow-red colormap), "br" (black-red
colormap), "yr" (yellow-red colormap), "wb" (white-black colormap),
"rainbow" (rainbow colormap, that is,
red-yellow-green-cyan-blue-magenta), and "ggplot2" (emulating ggplot2
default color palette). Alternatively, any hyphen-separated HTML color
names, e.g. "lightyellow-orange" (by default), "blue-black-yellow",
"royalblue-white-sandybrown", "darkgreen-white-darkviolet". A list of
standard color names can be found in
\url{http://html-color-codes.info/color-names}}

\item{ncolors}{the number of colors specified over the colormap}

\item{zlim}{the minimum and maximum values for which colors should be
plotted}

\item{node.size.range}{the range of actual node size}

\item{title}{a character specifying the title for the plot}

\item{edge.size}{a numeric value specifying the edge size. By default,
it is 0.5}

\item{edge.color}{a character specifying which edge attribute defining
the the edge colors}

\item{edge.color.alpha}{the 0-1 value specifying transparency of edge
colors}

\item{edge.curve}{a numeric value specifying the edge curve. 0 for the
straight line}

\item{edge.arrow}{a numeric value specifying the edge arrow. By
default, it is 2}

\item{edge.arrow.gap}{a gap between the arrow and the node}

\item{node.table}{a character specifying which node attribute for
coding. By default, it is 'term_name'}

\item{node.table.wrap}{a positive integer specifying wrap width of
coded node labelling}

\item{table.base.size}{a positive integer specifying font size in the
table}

\item{table.row.space}{a positive numeric value specifying amplying
horizental space for a row with wrapped text}

\item{table.nrow}{a positive integer specifying the number of rows in
the table}

\item{table.ncol}{NULL or a positive integer specifying the number of
columns per page. If NULL, it will be 3 or less}

\item{root.code}{a character specifying the root code. By default, it
is 'RT'}
}
\value{
a list with 3 components, two ggplot objects (code and table) and an
igraph object (ig appended with node attributes 'node.code' and
'node.table')
}
\description{
\code{xOBOcode} is supposed to create codes annotating nodes in an
igraph object. It returns two ggplot2 objects, one for visualing the
network with nodes lablelled by codes, the other for listing code
meaning in a table
}
\note{
none
}
\examples{
\dontrun{
# Load the library
library(XGR)
RData.location <- "http://galahad.well.ox.ac.uk/bigdata/"

# load REACTOME
# 1a) restricted to Immune System ('R-HSA-168256') or Signal Transduction ('R-HSA-162582')
g <- xRDataLoader(RData.customised='ig.REACTOME',
RData.location=RData.location)
neighs.out <- igraph::neighborhood(g, order=vcount(g),
nodes="R-HSA-168256", mode="out")
vids <- V(g)[unique(unlist(neighs.out))]$name
ig <- igraph::induced.subgraph(g, vids=vids)

# 1b) visualise the graph with nodes coded
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=2,
node.shape=19, node.size.range=4, edge.color.alpha=0.2)
pdf('xOBOcode.pdf', useDingbats=FALSE, width=8, height=8)
print(ls_gp$code + coord_equal(ratio=1))
print(ls_gp$table)
dev.off()

# 1c) visualise the graph with nodes coded and colored by information content (IC)
V(ig)$IC <- -1*log10(V(ig)$nAnno/max(V(ig)$nAnno))
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=2,
node.shape=19, node.size.range=4, node.color='IC',
node.color.title='IC', colormap='white-cyan-darkcyan')

V(ig)$term_anno <- log10(V(ig)$nAnno)
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=2,
node.shape=19, node.size.range=4, node.color='term_anno',
node.color.title='# genes\\n(log10)', colormap='white-cyan-darkcyan',
zlim=c(1,4))


# load EF (annotating GWAS reported genes)
# 2a) restricted to disease ('EFO:0000408') and annotation (>=10)
# 2a) restricted to immune system disease ('EFO:0000540') and annotation (>=10)
g <- xRDataLoader(RData.customised='ig.EF',
RData.location=RData.location)
neighs.out <- igraph::neighborhood(g, order=vcount(g),
nodes="EFO:0000540", mode="out")
nodeClan <- V(g)[unique(unlist(neighs.out))]$name
anno <- xRDataLoader(RData.customised='org.Hs.egEF',
RData.location=RData.location)
vec <- sapply(anno$gs, length)
nodeAnno <- names(vec[vec>=10])
neighs.in <- igraph::neighborhood(g, order=vcount(g), nodes=nodeAnno,
mode="in")
nodeAnno <- V(g)[unique(unlist(neighs.in))]$name
vids <- intersect(nodeClan, nodeAnno)
ig <- igraph::induced.subgraph(g, vids=vids)
V(ig)$anno <- anno$gs[V(ig)$name]
# 2b) visualise the graph with nodes coded
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=4,
node.shape=19, node.size.range=4, edge.color.alpha=0.2)
pdf('xOBOcode.pdf', useDingbats=FALSE, width=8, height=8)
print(ls_gp$code + coord_equal(ratio=1))
print(ls_gp$table)
dev.off()
# 2c) ## GWAS genes for immune system disease ('EFO:0000540')
anno <- xRDataLoader(RData.customised='org.Hs.egEF',
RData.location=RData.location)
genes <- anno$gs[['EFO:0000540']]
# 2d) ## GWAS SNPs for immune system disease ('EFO:0000540')
annotation <- xRDataLoader(RData.customised='GWAS2EF',
RData.location=RData.location)
dag <- xDAGpropagate(g, annotation, path.mode="all_paths",
propagation="min")
snps <- unlist(V(dag)[V(dag)$name=='EFO:0000540']$anno)
# 2e) ## ChEMBL targets for immune system disease ('EFO:0000540')
annotation <- xRDataLoader(RData.customised='Target2EF',
RData.location=RData.location)
dag <- xDAGpropagate(g, annotation, path.mode="all_paths",
propagation="max")
targets <- unlist(V(dag)[V(dag)$name=='EFO:0000540']$anno)


# load GOBP
# 3a) restricted to immune system process ('GO:0002376') and annotation (>=10)
g <- xRDataLoader(RData.customised='ig.GOBP',
RData.location=RData.location)
neighs.out <- igraph::neighborhood(g, order=vcount(g),
nodes="GO:0002376", mode="out")
nodeClan <- V(g)[unique(unlist(neighs.out))]$name
anno <- xRDataLoader(RData.customised='org.Hs.egGOBP',
RData.location=RData.location)
vec <- sapply(anno$gs, length)
nodeAnno <- names(vec[vec>=10])
neighs.in <- igraph::neighborhood(g, order=vcount(g), nodes=nodeAnno,
mode="in")
nodeAnno <- V(g)[unique(unlist(neighs.in))]$name
vids <- intersect(nodeClan, nodeAnno)
ig <- igraph::induced.subgraph(g, vids=vids)
V(ig)$anno <- anno$gs[V(ig)$name]
# 3b) visualise the graph with nodes coded
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=1,
node.shape=19, node.size.range=4, edge.color.alpha=0.2)
pdf('xOBOcode.pdf', useDingbats=FALSE, width=8, height=8)
print(ls_gp$code + coord_equal(ratio=1))
print(ls_gp$table)
dev.off()


# load GOMF
# 4a) restricted to molecular function ('GO:0003674') and annotation (>=50)
g <- xRDataLoader(RData.customised='ig.GOMF',
RData.location=RData.location)
neighs.out <- igraph::neighborhood(g, order=vcount(g),
nodes="GO:0003674", mode="out")
nodeClan <- V(g)[unique(unlist(neighs.out))]$name
anno <- xRDataLoader(RData.customised='org.Hs.egGOMF',
RData.location=RData.location)
vec <- sapply(anno$gs, length)
nodeAnno <- names(vec[vec>=50])
neighs.in <- igraph::neighborhood(g, order=vcount(g), nodes=nodeAnno,
mode="in")
nodeAnno <- V(g)[unique(unlist(neighs.in))]$name
vids <- intersect(nodeClan, nodeAnno)
ig <- igraph::induced.subgraph(g, vids=vids)
V(ig)$anno <- anno$gs[V(ig)$name]
# 4b) visualise the graph with nodes coded
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=1,
node.shape=19, node.size.range=4, edge.color.alpha=0.2)
pdf('xOBOcode.pdf', useDingbats=FALSE, width=8, height=8)
print(ls_gp$code + coord_equal(ratio=1))
print(ls_gp$table)
dev.off()


# load HPPA
# 5a) restricted to Abnormality of the immune system ('HP:0002715') and annotation (>=50)
g <- xRDataLoader(RData.customised='ig.HPPA',
RData.location=RData.location)
neighs.out <- igraph::neighborhood(g, order=vcount(g),
nodes="HP:0002715", mode="out")
nodeClan <- V(g)[unique(unlist(neighs.out))]$name
anno <- xRDataLoader(RData.customised='org.Hs.egHPPA',
RData.location=RData.location)
vec <- sapply(anno$gs, length)
nodeAnno <- names(vec[vec>=50])
neighs.in <- igraph::neighborhood(g, order=vcount(g), nodes=nodeAnno,
mode="in")
nodeAnno <- V(g)[unique(unlist(neighs.in))]$name
vids <- intersect(nodeClan, nodeAnno)
ig <- igraph::induced.subgraph(g, vids=vids)
V(ig)$anno <- anno$gs[V(ig)$name]
# 5b) visualise the graph with nodes coded
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=1,
node.shape=19, node.size.range=4, edge.color.alpha=0.2)
pdf('xOBOcode.pdf', useDingbats=FALSE, width=8, height=8)
print(ls_gp$code + coord_equal(ratio=1))
print(ls_gp$table)
dev.off()


# load DO
# 6a) restricted to immune system disease ('DOID:2914') and annotation (>=10)
g <- xRDataLoader(RData.customised='ig.DO',
RData.location=RData.location)
neighs.out <- igraph::neighborhood(g, order=vcount(g),
nodes="DOID:2914", mode="out")
nodeClan <- V(g)[unique(unlist(neighs.out))]$name
anno <- xRDataLoader(RData.customised='org.Hs.egDO',
RData.location=RData.location)
vec <- sapply(anno$gs, length)
nodeAnno <- names(vec[vec>=10])
neighs.in <- igraph::neighborhood(g, order=vcount(g), nodes=nodeAnno,
mode="in")
nodeAnno <- V(g)[unique(unlist(neighs.in))]$name
vids <- intersect(nodeClan, nodeAnno)
ig <- igraph::induced.subgraph(g, vids=vids)
V(ig)$anno <- anno$gs[V(ig)$name]
# 6b) visualise the graph with nodes coded
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=2,
node.shape=19, node.size.range=4, edge.color.alpha=0.2)
pdf('xOBOcode.pdf', useDingbats=FALSE, width=8, height=8)
print(ls_gp$code + coord_equal(ratio=1))
print(ls_gp$table)
dev.off()


# load MP
# 7a) restricted to immune system phenotype ('MP:0005387') and annotation (>=50)
# 7a) restricted to abnormal immune system physiology ('MP:0001790') and annotation (>=50)
g <- xRDataLoader(RData.customised='ig.MP',
RData.location=RData.location)
neighs.out <- igraph::neighborhood(g, order=vcount(g),
nodes="MP:0001790", mode="out")
nodeClan <- V(g)[unique(unlist(neighs.out))]$name
anno <- xRDataLoader(RData.customised='org.Hs.egMP',
RData.location=RData.location)
vec <- sapply(anno$gs, length)
nodeAnno <- names(vec[vec>=50])
neighs.in <- igraph::neighborhood(g, order=vcount(g), nodes=nodeAnno,
mode="in")
nodeAnno <- V(g)[unique(unlist(neighs.in))]$name
vids <- intersect(nodeClan, nodeAnno)
ig <- igraph::induced.subgraph(g, vids=vids)
V(ig)$anno <- anno$gs[V(ig)$name]
# 7b) visualise the graph with nodes coded
ls_gp <- xOBOcode(g=ig, node.level='term_distance', node.level.value=3,
node.shape=19, node.size.range=4, edge.color.alpha=0.2)
pdf('xOBOcode.pdf', useDingbats=FALSE, width=8, height=8)
print(ls_gp$code + coord_equal(ratio=1))
print(ls_gp$table)
dev.off()
}
}
\seealso{
\code{\link{xGGnetwork}}
}
