% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/galaxy.R
\name{run_galaxy_processing}
\alias{run_galaxy_processing}
\title{run_galaxy_processing - automate running code in galaxy}
\usage{
run_galaxy_processing(
  tool_name,
  code,
  tool_version = "unknown",
  unmangle_parameters = TRUE,
  args = NULL,
  logger = NULL,
  source_files = c(),
  env = NULL,
  do_traceback = FALSE
)
}
\arguments{
\item{tool_name}{\itemize{
\item Mandatory. The name of the tool to run.
}}

\item{code}{\itemize{
\item Mandatory. The code to run the tool
}}

\item{tool_version}{\itemize{
\item The version number of the tool to run.
}}

\item{unmangle_parameters}{\itemize{
\item Whether or not to revert mangling from galaxy
useful if galaxy produces strange command parameters.
Not necessary, but produces more explicit outputs.
}}

\item{args}{\itemize{
\item The result of \link{commandArgs} function, or from
the \link{optparse_parameters} function.
Can be NULL. In this case, uses \link{commandArgs} to get the args.
}}

\item{logger}{\itemize{
\item You can provide a logger to use. If not provided,
a logger will be created with the tool's name.
}}

\item{source_files}{\itemize{
\item You may provide some paths to source before executing
the provided code.
}}

\item{env}{\itemize{
\item You may provide a environment object to execute the code
within.
}}

\item{do_traceback}{\itemize{
\item logical - tells whether to produce a traceback in
case of error.
}}
}
\description{
run_galaxy_processing takes the tool's name, and the code to execute.
It detects galaxy-specific environement variable, and show headers
and footer if we are in a galaxy env.

It will automatically convert command line parameters using
W4MRUtils::parse_args if args is not provided.

Then, it unmangles galaxy parameters (galaxy params / values can be
mangled if they contains special characters)

It creates a logger, and provide access to the \code{logger} and \code{args}
variables from withing the code to execute.

Also, before executing the code, if \code{source_files} is set to some paths,
these paths will be source'd, so the code has access to functions
defined in these scripts.
}
\examples{

write_r_file_with_content <- function(content) {
  "
  This function creates a temp R file. It writes the provided
  content in the R file. Then it returns the path of the script.
  "
  path <- tempfile(fileext = ".R")
  file.create(path)
  writeLines(content, con = path)
  return(path)
}
## let's fake a galaxy env
Sys.setenv(GALAXY_SLOTS = 1)
## let's says the tool has been launched with this command line
log_file <- tempfile()
file.create(log_file)
raw_args <- list(
  "--args",
  "--input", "in.csv",
  "--output", "out.csv",
  "--logs", log_file,
  "--one-float", "3.14",
  "--one-integer", "456",
  "--one-logical", "FALSE",
  "--some-floats", "1.5,2.4,3.3",
  "--some-characters", "test,truc,bidule",
  "--debug", "TRUE",
  "--verbose", "FALSE"
)

##
# example 1
##

my_r_script <- write_r_file_with_content('
  my_processing <- function(args, logger) {
    logger$info("The tool is running")
    logger$infof("Input file: \%s.", args$input)
    logger$info("The tool ended.")
  }
')

W4MRUtils::run_galaxy_processing(
  "Test tool 1",
  my_processing(args, logger),
  source_file = my_r_script,
  args = W4MRUtils::parse_args(args = raw_args)
)


##
# example 2

## let's say we have a R script with this content:
path <- write_r_file_with_content('
  setup_logger <- function(args, logger) {
    if (!is.null(args$verbose)) {
      logger$set_verbose(args$verbose)
    }
    if (!is.null(args$debug)) {
      logger$set_debug(args$debug)
    }
    if (!is.null(args$logs)) {
      logger$add_out_paths(args$logs)
    }
  }
  stop_logger <- function(logger) {
    logger$close_files()
  }
  processing <- function(args, logger) {
    setup_logger(args, logger)
    logger$info("The tool is working...")
    logger$infof("Input: \%s.", args$input)
    logger$info("The tool stoping.")
    stop_logger(logger)
    return(NULL)
  }')


## wrapper script:

args <- W4MRUtils::optparse_parameters(
  input = W4MRUtils::optparse_character(),
  output = W4MRUtils::optparse_character(),
  logs = W4MRUtils::optparse_character(),
  one_float = W4MRUtils::optparse_numeric(),
  one_integer = W4MRUtils::optparse_integer(),
  one_logical = W4MRUtils::optparse_flag(),
  some_floats = W4MRUtils::optparse_list(of = "numeric"),
  some_characters = W4MRUtils::optparse_list(of = "character"),
  debug = W4MRUtils::optparse_flag(),
  verbose = W4MRUtils::optparse_flag(),
  args = raw_args[raw_args != "--args"]
)

W4MRUtils::run_galaxy_processing("A Test tool", args = args, {
  ## processing is from the other R script
  processing(args, logger)
}, source_files = path)


}
\author{
L.Pavot
}
